/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE:		dbstat.c
 *
 * SCCSINFO:		@(#)dbstat.c	1.15 6/6/94
 *
 * AUTHOR:		Ralph R|nnquist, 1989-11-20
 *
 * DESCRIPTION:
 *	This file contains the dbstat program, which queries for presence
 *	of a database monitor without actually connecting to it.
 *
 *	Low-level liblincks routine to establish a conenction between
 *	a workspace (application) and database server.
 *
 * MODIFICATIONS:
 *      1994-02-27 Martin Sjlin. Updated to be able to handle
 *                 internet (dot) addresses in host name. Copied
 *                 and modified code from autostart.c
 *      1994-03-03 Martin Sjlin. Extended the "protocol" used
 *                 for checking for running netserv to read version
 *                 information sent back.
 *
 *	<list mods with name and date>
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#ifdef  HAVE_USLEEP			/* included in config.h ifndef */
#include <sys/time.h>
#endif /* n HAVE_USLEEP */
#include <sys/socket.h>			/* MAXHOSTNAMELEN on SCO */
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <netdb.h>			/* for MAXHOSTNAMELEN */

#include "xconfig.h"
#include "libshared.h"

#ifdef sun
extern struct hostent *gethostbyname();
#endif
/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
void main( /* int argc, char *(*argv)[] */ );

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_tcp.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define Info(text,value) (void)fprintf(stderr,"\t%s: %s\n",text,value)
#define Info2(text,value) (void)fprintf(stderr,"\t%s:\n\t\t%s\n",text,value)

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif /* MAXHOSTNAMELEN */

/* duplicated from liblincks.h.  GOTTA FIND A BETTER WAY! */
#ifndef SUCCESS
#define SUCCESS   0
#endif

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static int getdbstatus P_(( char *host, int *sock ));
static void print_version P_((void));

/*********************************************************************
 * INTERNAL (STATIC) DATA: 
 *********************************************************************/
/* none */

/*  */
/************************************************************************
 * Function: static int getdbstatus(char *host,int *sock)
 *
 * Establishes socket connection to a monitor for the present database
 * and reports on failures etc. Returns 0 if something failed, otherwise 1.
 *
 * This function is essentially a copy of LL_FindMonitorHost in
 * file autostart.c, but specialised for this particular task.
 *
 * Modifications:
 *	<list mods here with name and date>
 *
 */
static int getdbstatus(host, sock)
  char *host;
  int *sock;
{
  static int opt = SO_DEBUG;
  static struct hostent *hp = NULL;
  static t_u32bits addr = 0;
  static char hostname[MAXHOSTNAMELEN];	/* when using internet dotted addr */

  /* Present database configuration information.  */
  (void)fprintf(stderr, "Configuration of database %s is\n", DBDIR);
  Info("Internet port number", LL_GetConfig("TCPIPNO"));
  Info("Host for servers", HOST);
  Info("Database owner", OWNER);
  Info2("Server program directory", BINARIES);
  Info2("Logging directory", LOGDIR);

 /* Use HOST from run-time configuration.  */
  (void)strcpy(host, HOST);

  /* try as a internet dot address first */
  if (tcp_lookup(host, &addr, &hp) != SUCCESS) {
    (void)fprintf(stderr, "Host lookup of %s has failed.\n", host);
    return 0;
  }
  strcpy(hostname, host);

  /* try do a reverse lookup */
  if (!hp) {
    if ((hp = gethostbyaddr((char *)&addr, 
			    sizeof(addr), 
			    AF_INET)) == (struct hostent *)NULL)
      (void)fprintf(stderr,"\tReverse lookup for host %s failed\n", host);
    else {
      (void)fprintf(stderr,"\tOfficial host name is %s\n",hp->h_name);
      strcpy(hostname, hp->h_name);
    }
  } 

  /* Tickle the monitor process to check its existence.  */
  if (hp) {
    if (tickle_rpc(hostname, ((u_long)MONITORPROGNO), ((u_long)TCPIPNO)) 
	!= SUCCESS) {
      (void)fprintf(stderr,"\nNo monitor for %s on %s!\n",DBDIR,HOST);
      return 0;
    } else
      (void)fprintf(stderr, "\tThe monitor is running.\n");  
  } 

  /* Create a socket for communication and export socket handle.  */
  if ((*sock = tcp_connect(tcp_error, host, TCPIPNO)) < 0)
    return 0;
  (void)fprintf(stderr, "\tnetserv responds.\n");

  /*
   * Set socket options to avoid transfer delays. This forces every
   * transfer packet to be available as soon as it comes rather than
   * having the network to wait 100ms for additional bytes.
   */
  if (setsockopt(*sock, IPPROTO_TCP, TCP_NODELAY, (char *)&opt, 
		 sizeof(opt)) < 0) {
    (void) fprintf( stderr, "Warning - TCP_NODELAY not supported.\n");
  }

  /* now, we wait for the first character .. */
  {
     char *buf;
     unsigned char ch;
     struct timeval timev;
     fd_set descf;

     FD_ZERO(&descf);			
     FD_SET(*sock, &descf);

     timev.tv_sec  = 20;		/* 20 sec */
     timev.tv_usec =  0;

     if(select(*sock + 1, (fd_set *) & descf, (fd_set *) NULL, (fd_set *) NULL, 
	       &timev) < 1 || !FD_ISSET(*sock, &descf)) {
       (void)fprintf(stderr, 
		     "\tFailed in select waiting for netserv response\n");
       return 1;
     } 
     else if(read(*sock, &ch, sizeof(ch)) != sizeof(ch)) {
       (void)fprintf(stderr, 
		   "\t(Failed to read from netserv - check if old version)\n");
       return 1;
     }
 
     ch = 'S';
     if(write(*sock, &ch, sizeof(ch)) != sizeof(ch)) {
       (void)fprintf(stderr, 
		     "\tFailed to write message to netserv\n");
       return 1;
     }

     FD_ZERO(&descf);			
     FD_SET(*sock, &descf);
     timev.tv_sec  = 20;		/* 20 sec */
     timev.tv_usec =  0;

     if   (select(*sock + 1, (fd_set *) & descf, (fd_set *) NULL, 
		  (fd_set *) NULL, 
		  &timev) < 1 || !FD_ISSET(*sock, &descf))
	(void)fprintf(stderr, 
		      "\tFailed in select waiting for netserv version info\n");
     else if(read(*sock, &ch, sizeof(ch)) != sizeof(ch)) 
	(void)fprintf(stderr,
		      "\tFailed to read size byte from netserv\n");
     else if((buf = (char *) malloc(1 + (ALLOC_T)((unsigned int) ch)) ) == NULL) 
	(void)fprintf(stderr,
		      "\tFailed to allocate memory\n");
     else if(read(*sock, buf, (IOSIZE_T) ch) != (int) ch) 
	(void)fprintf(stderr,
		      "\tFailed to read version message from netserv\n");
     else {
       buf[(int)ch] = '\0';
       (void) fprintf(stderr, "\t%s\n",buf);
     }
  }
  return 1;
}

/*  */
/**********************************************************************
 * Function: void main(int argc, char *(*argv)[])
 * 
 * Modifications:
 *      <list mods with name and date>
 */
void main(argc, argv)
  int argc;
  char *(*argv)[];
{
  static char host[MAXHOSTNAMELEN];
  static int socket;
  char *db = NULL;

  if (argc < 2) {
    if ((db = (char *)getenv("LINCKSDBDIR")) == (char *)NULL) { 
      print_version();
      (void)fprintf(stderr, "Usage: %s database-dir\n", (*argv)[0]);
      exit(1);
    }
  } else {
    db = (*argv)[1];
  }

  print_version();

  if (configuration(db) == 0) {
    (void)fprintf(stderr, "Couldn't configure %s\n", db);
    exit(1);
  }
  if (getdbstatus(host, &socket) == 1) {
    (void)fprintf(stderr,
		  "The database is available for workspace connection.\n");
    exit(0);
  }
  exit(1);
}

#define VERSION "1.3"
#define DATE "1994-06-01"
#define VERSION_INFO "\n"

/*  */
/**********************************************************************
 * Function: static void print_version()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void print_version()
{
  (void)fprintf(stdout, "\tdbstat version %s (%s)\n%s", VERSION, DATE,
                VERSION_INFO);

}
