/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/* 
 * MODULE NAME: 	wssupport.c
 *
 * SCCSINFO:		@(#)wssupport.c	1.8 5/3/94
 *
 * ORIGINAL AUTHOR(S):  Lin Padgham, Ralph R|nnquist and Peter ]berg,
 *			1982-09-18
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 * Low level work space support functions for liblincks.
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <time.h>

#include "sunlincks.h"
#include "dbcodes.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_wssupport.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_logws.h"
#include "f_dbcomm.h"
#include "f_structsupport.h"
#include "f_sysfuns.h"

/* libc */
#if defined(sun) || defined(__sun__)
extern time_t time P_((time_t *tloc));
#endif

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern molecule *LL_moltbl;
extern char LL_loginuser[];
extern int LL_uid;
extern char *sys_errlist[];
extern int sys_nerr;
extern int errno;
extern int LL_freemolindex;

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
/* Size increments for molecule table */ 
#define MOLTABLEINC 16

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * INTERNAL (STATIC) DATA: 
 *********************************************************************/
/* molecule table */
static int moltblsize = 0;

/*  */
/**********************************************************************
 * Function: errcode LL_newmol()
 * 
 * Creates a new slot in the molecule table and initializes it
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_newmol()
{
  molecule *entry;

  /* Malloc new molecule */
  if (LL_mallocmol() == FAIL) {
    return FAIL;
  }

  /* Set up molecule table entry */
  entry = &LL_moltbl[LL_freemolindex-1];
  entry->attributes = (attrgroup *)NULL;
  entry->links = (linkgroup *)NULL;
  entry->image = (image_desc *)NULL;
  entry->ITix = -1;
  entry->inWS = entry->edited = entry->mark = 0;

  /* Return molecule table index */
  return(LL_freemolindex-1);      
}

/*  */
/**********************************************************************
 * Function: errcode LL_mallocmol()
 * 
 * Creates new molecule table entrys as needed
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_mallocmol()
{
  /* Secure free molecule table space */
  if (LL_freemolindex == moltblsize) {
    if (moltblsize == 0) {
      LL_moltbl=(molecule *)calloc((ALLOC_T)MOLTABLEINC, sizeof(molecule));
      if (!LL_moltbl) {
	LogMess(LL_uid,
	        "LL_mallocmol: malloc failed for LL_moltbl, %s",
	        sys_errlist[(errno > sys_nerr) ? 0 : errno]);
        perror("LL_mallocmol: malloc failed for 'LL_moltbl'\n");
	return(FAIL);
      } 
    moltblsize = MOLTABLEINC;
    }
    else {
      LL_moltbl = (molecule *)realloc((FREEPTR *)LL_moltbl,
                                      (ALLOC_T)(moltblsize + MOLTABLEINC)
				      * sizeof(molecule));
      if (!LL_moltbl) {
        LogMess(LL_uid,
                "LL_mallocmol: realloc failed for LL_moltbl, %s",
                sys_errlist[(errno > sys_nerr) ? 0 : errno]);
        perror("LL_mallocmol: realloc failed for 'LL_moltbl'\n");
	return(FAIL);
      }
    moltblsize += MOLTABLEINC;
    }
  }
  ++LL_freemolindex;
  return(SUCCESS);
}

/*  */
/**********************************************************************
 * Function: errcode LL_VSTag(int index, char **string)
 * 
 * Description: 
 *  mallocs some memory and puts 
 *  a string of form yymmdd.nnn in it,
 *  where nnn is VSNET:Count from specified molecule.
 *  Increments VSNET:Count.
 *  Caller is responsible for free'ing the string memory when finished.
 *
 * Return codes:
 *  SUCCESS
 *  ????
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_VSTag(index, string)
  int index;		/* a vs in the molecule table */
  char **string;
{
  attrval v;
  attribute *counter;
  int *cp;
  struct tm *t, *localtime();
  static char buffer[20];
  time_t secs;
  int count = 2;

  /* Get current time */
  (void)time(&secs);
  t = localtime(&secs);

  /* Get/set VSNET:Count value */
  if (LL_sysgetattr(index, VSNET_COUNT, &counter)) {
    v.attsize = sizeof(count);			/* sizeof(int) */
    v.attvalue = (char *)&count;
    if (!LL_sysputattr(index, VSNET_COUNT, &v))
      LL_moltbl[index].edited = TRUE;
    count = 1;
  } else {
    cp = (int *) (counter->value);
    count = (*cp)++;
    LL_moltbl[index].edited = TRUE;
  }

  /* Put together string */
  /* 20 is big enough for yymmdd.nnnnnnnnnnnnn */
  *string = buffer;
  (void)sprintf(*string, "%02d%02d%02d.%d", t->tm_year,
		t->tm_mon + 1,		/* number months 1..12 */
                t->tm_mday, 
		count);
  return(SUCCESS);
}

/*  */
/*
 *  SYSFUN support functions
 */

/*  */
/**********************************************************************
 * Function: errcode LL_findgrptag(FOUR PARAMETERS)
 *
 * Parameters:
 *	cell *struct_ptr
 *	char *tag
 *	cell **prevgrp
 *	cell **grp
 * 
 * FINDGRPTAG
 * Returns pointer to the given group tag in an attribute or link structure
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_findgrptag(struct_ptr,tag,prevgrp,grp)
 cell *struct_ptr;
 char *tag;
 cell **prevgrp,**grp;
{
  if (tag == (char *) NULL)
    return(ERR_GROUP);
  *prevgrp = (cell *)NULL;
  *grp = struct_ptr;
  while (*grp) {
    if (strcmp((*grp)->tag,tag) == 0) return(SUCCESS);
    *prevgrp = *grp;
    *grp = (*grp)->next;
  }
  return(ERR_GROUP);
}

/*  */
/**********************************************************************
 * Function: errcode LL_findfieldtag(FIVE PARAMETERS)
 *
 * Parameters:
 *	cell **struct_ptr
 *	char *group
 *	char *tag
 *	cell **prev
 *	cell **fieldptr
 * 
 * FINDFIELDTAG
 * Returns pointer to the given field tag in an attribute or link group 
 * structure
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_findfieldtag(struct_ptr,group,tag,prev,fieldptr)
  cell **struct_ptr;
  char *group,*tag;
  cell **prev,**fieldptr;
{
  cell *prevgrp,*grp;
  int ERRCODE;

  if ((ERRCODE = LL_findgrptag(*struct_ptr,group,&prevgrp,&grp)))
    return(ERRCODE);
  if (!tag)
    return ERR_FIELD;
  *struct_ptr = grp;
  *prev = (cell *)NULL;
  for (*fieldptr = (cell *)grp->value;*fieldptr;*fieldptr = (*fieldptr)->next){
    if (strcmp((*fieldptr)->tag,tag) == 0)
      return SUCCESS;
    *prev = *fieldptr;
  }
  return(ERR_FIELD);
}

/*  */
/**********************************************************************
 * Function: errcode LL_checkvs(label *lbl1)
 * 
 * CHECKVS 
 * checks that the first arg of label is a vs molecule
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_checkvs(lbl1)
  label *lbl1;
{
  int ERRCODE;
  linkgroup  *grouptag,*prev;

  if (!LL_moltbl[lbl1->vs].inWS &&
      (ERRCODE = LL_DBRetry(lbl1->vs, VS, LL_DBRetrieve)))
    return(ERRCODE);
  ERRCODE = LL_findgrptag((cell *)LL_moltbl[lbl1->vs].links, VSNET,
			  (cell **)&prev, (cell **)&grouptag);
  if (ERRCODE == ERR_GROUP)
    return(ERR_LABEL);
  return(ERRCODE);
}


/* ^L */
/**********************************************************************
 * Function: errcode LL_set_current(label *lbl)
 *
 * Set lbl to the most recent temporally.  Check SYSTEM_LATEST first,
 * check my transient versions, check front of vsnet, or create a 
 * new instance.  return 0 if all seems well, return 1 if there are
 * ambiguities.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int LL_set_current(lbl)
  label *lbl;
{
  linkgroup *gp;
  linktag *slot;
  linkitem *p;

  if (lbl->inst != INST_UNBOUND) {
    return 0;
  }

  /* find most recent temporally in SYSTEM:Latest */
  if ((LL_sysgetlinkitem(lbl->vs, SYSTEM_LATEST, 1, &p)) == SUCCESS) {
    lbl->inst = p->link.inst;
    return (p->next_item != (linkitem *)NULL);
  }

  /* something fishy going on */
  if (!LL_sysgetlinkgroup(lbl->vs, TH_TRANSIENT,&gp) && (gp->value)) {
    mapfields(slot,gp) {
      mapitems(p,slot)
	if (!p->next_item && LL_moltbl[p->link.inst].inWS) {
	  lbl->inst = p->link.inst;
	  return 0;
	}
    }
  }

  if (!LL_sysgetlinkgroup(lbl->vs, VSNET,&gp) && (gp->value)) {
    lbl->inst = gp->value->value->link.inst;
    return 1;
  }
  lbl->inst = LL_newmol();
  (void)LL_newmolinfo(lbl->inst);
  (void)LL_update_VS(lbl);
  return 0;
}

/*  */
/**********************************************************************
 * Function: errcode LL_bindlbl(label *lbl1, label *lbl2)
 * 
 * BINDLBL
 * Set lbl2 to be a bound label for the node version refered to by lbl1.
 * Sets SYSTEM:Current (as latest or as a given bound label).
 * Retrieves the version (inst) of the bound label lbl2 if not already
 * in workspace.
 *
 * Return codes:
 *	SUCCESS
 *	ERR_xxx ????
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_bindlbl(lbl1,lbl2)
  label *lbl1;
  label *lbl2;
{
  int ret_code, ret_code2;

  if ((ret_code = LL_checkvs(lbl1)))
    return(ret_code);

  /* vs of lbl2 is the vs of lbl1 */
  lbl2->vs = lbl1->vs;
  lbl2->inst = lbl1->inst;

  if (LL_set_current(lbl2))
    ret_code = SUCCESS;

  if (!LL_moltbl[lbl2->inst].inWS)
    if ((ret_code2 = LL_DBRetry(lbl2->inst, OTHER, LL_DBRetrieve)))
      return ret_code2;

  return (ret_code); 
}

/*  */
/**********************************************************************
 * Function: errcode LL_checkbuffer(label *lbl, cell  **p)
 * 
 * CHECKBUFFER
 * Ensures that label points to an instance in edit mode.  If not in edit
 * mode ensure buffer creates a new edit instance and copies in the contents
 * of the given instance.  If new instance is created lbl is changed to
 * point to the new instance and *p is set to NULL
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_checkbuffer(lbl,p) 
  label *lbl; 
  cell  **p;
{ 
  label oldlbl;
  linkitem *ip = (linkitem *)NULL;
  linkgroup  *grouptag, *prev;
  int newinst,ERRCODE;

  if ((LL_moltbl[lbl->inst].edited == TRUE)  &&  (INLATEST(lbl)))
    return(SUCCESS);

  /* 
   * We do not allow two transient versions of the same object 
   * to be created at the same time in the same workspace. 
   * (Could give some problems for the history management.)
   */
  if (!LL_findgrptag((cell *)LL_moltbl[lbl->vs].links, TH_TRANSIENT,
		     (cell **)&prev, (cell **)&grouptag) &&
      (grouptag->value)) {
    (void)printf("LL_checkbuffer: multiple transient versions <%d:%d>\n",
		 lbl->vs, lbl->inst);
    return (ERR_DOUBLE_EDITING);
  }
  *p = (cell *)NULL;
  newinst = LL_newmol();
  LL_copymol(lbl->inst,newinst);
  (void)LL_newmolinfo(newinst);
  oldlbl.vs = lbl->vs;
  oldlbl.inst = lbl->inst;
  lbl->inst = newinst;
  (void)LL_sysgetlinkitem(newinst, SYSTEM_PARENT, 1,&ip);

  if (ip) {
    ip->link.vs = oldlbl.vs;
    ip->link.inst = oldlbl.inst;
    LL_freelinkval(ip->next_item); /* Don't inherit parent's parents. */
    ip->next_item = (linkitem *)NULL;
  }
  else if ((ERRCODE = LL_sysputlinkitem(newinst, SYSTEM_PARENT,1,&oldlbl)))
    return(ERRCODE);

  LL_moltbl[lbl->vs].edited = TRUE;
  return(SUCCESS);
}
 
/*  */
/**********************************************************************
 * Function: errcode INLATEST(label *lbl)
 *
 * INLATEST
 * Returns TRUE if lbl is a member of Latest 
 * 
 * Modifications:
 *      <list mods with name and date>
 */
errcode INLATEST(lbl)
  label *lbl;
{
  linktag *linkptr;
  linkitem *itemptr;
  int ERRCODE;

  if ((ERRCODE = LL_sysgetlink(lbl->vs, SYSTEM_LATEST,&linkptr))) 
    return(ERRCODE);
  mapitems(itemptr,linkptr)
    if (itemptr->link.inst == lbl->inst) return(TRUE);
  return(FALSE);
}

/*  */
/**********************************************************************
 * Function: errcode LL_update_VS(label *lbl)
 * 
 * DESCRIPTION
 *
 * LL_update_VS is called at the end of the initial edit operation on an
 * object at which a new version is created (see LL_checkbuffer). The new
 * version, passed in as lbl->inst, is here put into the transient portion
 * of the history using all SYSTEM_LATEST as predecessors. The SYSTEM_LATEST
 * list is also updated to contain only the new version.
 * This function also issues an EDIT_CODE command to the database so as to
 * allow parallel editing notification (PEN) to occur. This signalling is
 * inhibited by clearing the edited flag for the history node prior to
 * calling this function.
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_update_VS(lbl)
  label *lbl;

{
  linkgroup *gp = (linkgroup *)NULL;
  linktag *lat = (linktag *)NULL;
  linkitem *ip = (linkitem *)NULL;
  char *datetag = (char *)NULL;
  int ix;
  int ret_code;

  if (LL_moltbl[lbl->vs].edited) {
    if ((ret_code = LL_DBRetry(lbl->vs,VSE,LL_DBPreedit))) 
      return ret_code;
  }

  /* Grab a new VSNET:`datetag' */
  if ((ret_code = LL_VSTag(lbl->vs, &datetag)))
    return ret_code;

  LL_moltbl[lbl->vs].edited = TRUE;

  if (LL_sysgetlinkgroup(lbl->vs, VSNET, &gp) == ERR_GROUP) {
    (void)LL_sysputlinkitem(lbl->vs, VSNET,datetag,1,lbl);
    if ((ret_code = LL_sysgetlinkgroup(lbl->vs, VSNET, &gp)))
      return (ret_code);
    LL_freelinkgroupval(gp->value);
    gp->value = (linktag *)NULL;
  }

  /* Build TH_TRANSIENT slot using SYSTEM:Latest. */
  if (!LL_sysgetlink(lbl->vs, SYSTEM_LATEST,&lat)) {
    ix = 1;
    mapitems(ip,lat) {
      ret_code =
	(LL_sysputlinkitem(lbl->vs, TH_TRANSIENT, datetag,ix,&(ip->link)));
      ix += 1;
      if (ret_code)
	return (ret_code);
    }
    LL_freelinkval(lat->value);
    lat->value = (linkitem *)NULL;
  }
  if ((ret_code = LL_sysputlinkitem(lbl->vs, TH_TRANSIENT,datetag,1,lbl)))
    return (ret_code);

  /* Update SYSTEM:Latest by adding lbl. */
  return LL_sysputlinkitem(lbl->vs,SYSTEM_LATEST,1,lbl);
}


/*  */
/**********************************************************************
 * Function: errcode LL_newmolinfo(int mol)
 * 
 * NEWMOLINFO 
 * Add creation information to molecule 
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_newmolinfo(mol)
  int mol;
{
  attrval vp;
  time_t clk;
  int ERRCODE;
  char *timestring;      

  LL_moltbl[mol].inWS=TRUE;
  LL_moltbl[mol].edited=TRUE;

  (void)time(&clk);
  LL_strtoval(LL_loginuser,&vp);
  if ((ERRCODE=LL_sysputattr(mol, SYSTEM_OWNER,&vp)))
    return(ERRCODE);
  free(vp.attvalue);
  
  timestring = ctime(&clk);
  timestring[strlen(timestring)-1] = '\0';	/* Zap trailing newline */
  LL_strtoval(timestring,&vp);
  if ((ERRCODE=LL_sysputattr(mol, SYSTEM_CREATED,&vp)))
    return(ERRCODE);
  free(vp.attvalue);

  return(SUCCESS);
}
