/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	structsupport.c
 *
 * SCCSINFO:		@(#)structsupport.c	1.8 5/3/94
 *
 * ORIGINAL AUTHOR(S):  Ralph R|nnquist and Peter ]berg, 1987-09-18
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *
 * This file contains functions for manipulating (copying and freeing)
 * molecule fields
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "sunlincks.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_structsupport.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_logws.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern int LL_uid;
extern char *sys_errlist[];
extern int sys_nerr;
extern int errno;

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define LOG_ERRSTRING   sys_errlist[(errno >= sys_nerr) ? 0 : errno]
#define LOG_MESS(b)     LogMess(LL_uid,"%s, %s", b, LOG_ERRSTRING); \
			perror(b)

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
/* none */

/*  */
/**********************************************************************
 * Function: void LL_lblcpy(label *lbl_a, label *lbl_b)
 *
 * Copies lbl_b to lbl_a.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void LL_lblcpy(lbl_a, lbl_b)
  label *lbl_a, *lbl_b;
{
  lbl_a->vs   = lbl_b->vs;
  lbl_a->inst = lbl_b->inst;
}

/*  */
/**********************************************************************
 * Function: void LL_copymol(int oldindex, newindex)
 *
 * Copies old molecule table entry to new one
 *
 * Modifications:
 *      <list mods with name and date>
 */
void LL_copymol(oldindex, newindex)
  int oldindex, newindex;
{
  molecule *oldentry, *newentry;
  extern molecule *LL_moltbl;

  oldentry = &LL_moltbl[oldindex];
  newentry = &LL_moltbl[newindex];

  LL_releasemol(newentry);

  if (oldentry->attributes)
    newentry->attributes = LL_copyattributes(oldentry->attributes);

  if (oldentry->links)
    newentry->links = LL_copylinks(oldentry->links);

  if (oldentry->image)
    newentry->image = LL_copyimage(oldentry->image);

  newentry->ITix = -1;
  newentry->inWS = newentry->edited = TRUE;
}


/*  */
/*
 * These functions below perform various forms of copying
 */
/**********************************************************************
 * Function: image_desc *LL_copyimage(image_desc *image)
 *
 * Modifications:
 *      <list mods with name and date>
 */
image_desc *LL_copyimage(image)
  image_desc *image;
{
  image_desc *new;

  if ((new = (image_desc *) malloc(sizeof(image_desc))) == NULL) {
    LOG_MESS("LL_copyimage: malloc failed");
    return NULL;
  }
  new->size = image->size;
  new->value = LL_copyval(image->size, image->value);
  return new;
}

/*  */
/**********************************************************************
 * Function: attrgroup *LL_copyattributes(attrgroup *group)
 *
 * Modifications:
 *      <list mods with name and date>
 */
attrgroup *LL_copyattributes(group)
  attrgroup *group;
{
  if (group)
    return LL_copyattrgroup(group, LL_copyattributes(group->next));
  else
    return NULL;
}


/*  */
/**********************************************************************
 * Function: attrgroup *LL_copyattrgroup(attrgroup *group, attrgroup *next)
 *
 * Modifications:
 *      <list mods with name and date>
 */
attrgroup *LL_copyattrgroup(group, next)
  attrgroup *group, *next;
{
  static attrgroup *new;

  if ((new = (attrgroup *) malloc(sizeof(attrgroup))) == NULL) {
    LOG_MESS("LL_copyattrgroup: malloc failed");
    return NULL;
  }
  new->tag = LL_copystring(group->tag);
  new->value = LL_copyattrs(group->value);
  new->next = next;
  return new;
}


/*  */
/**********************************************************************
 * Function: attribute *LL_copyattrs(attribute *attr)
 *
 * Modifications:
 *      <list mods with name and date>
 */
attribute *LL_copyattrs(attr)
  attribute *attr;
{
  if (attr)
    return LL_copyattribute(attr, LL_copyattrs(attr->next));
  else
    return NULL;
}


/*  */
/**********************************************************************
 * Function: attribute *LL_copyattribute(attribute *attr, attribute *next)
 *
 * Modifications:
 *      <list mods with name and date>
 */
attribute *LL_copyattribute(attr, next)
  attribute *attr, *next;
{
  static attribute *new;

  if ((new = (attribute *) malloc(sizeof(attribute))) == NULL) {
    LOG_MESS("LL_copyattribute: malloc failed");
    return NULL;
  }
  new->tag = LL_copystring(attr->tag);
  new->value = LL_copyval(attr->size, attr->value);
  new->size = attr->size;
  new->next = next;
  return new;
}


/*  */
/**********************************************************************
 * Function: linkgroup *LL_copylinks(linkgroup *group)
 *
 * Modifications:
 *      <list mods with name and date>
 */
linkgroup *LL_copylinks(group)
  linkgroup *group;
{
  if (group)
    return LL_copylinkgroup(group, LL_copylinks(group->next));
  else
    return NULL;
}

/*  */
/**********************************************************************
 * Function: linkgroup *LL_copylinkgroup(linkgroup *group, linkgroup *next)
 *
 * Modifications:
 *      <list mods with name and date>
 */
linkgroup *LL_copylinkgroup(group, next)
  linkgroup *group, *next;
{
  static linkgroup *new;

  if ((new = (linkgroup *) malloc(sizeof(linkgroup))) == NULL) {
    LOG_MESS("LL_copylinkgroup: malloc failed");
    return NULL;
  }
  new->tag = LL_copystring(group->tag);
  new->value = LL_copylinktags(group->value);
  new->next = next;
  return new;
}


/*  */
/**********************************************************************
 * Function: linktag *LL_copylinktags(linktag *tp)
 *
 * Modifications:
 *      <list mods with name and date>
 */
linktag *LL_copylinktags(tp)
  linktag *tp;
{
  if (tp)
    return LL_copylinktag(tp, LL_copylinktags(tp->next));
  else
    return NULL;
}


/*  */
/**********************************************************************
 * Function: linktag *LL_copylinktag(linktag *tp, linktag *next)
 *
 * Modifications:
 *      <list mods with name and date>
 */
linktag *LL_copylinktag(tp, next)
  linktag *tp, *next;
{
  static linktag *px;

  if ((px = (linktag *) malloc(sizeof(linktag))) == NULL) {
    LOG_MESS("LL_copylinktag: malloc failed");
    return NULL;
  }
  px->tag = LL_copystring(tp->tag);
  px->value = LL_copyitems(tp->value, (linkitem *) NULL);
  px->next = next;
  return px;
}


/*  */
/**********************************************************************
 * Function: linkitem *LL_copyitems(linkitem *l, linkitem *next)
 *
 * Modifications:
 *      <list mods with name and date>
 */
linkitem *LL_copyitems(l, next)
  linkitem *l, *next;
{
  if (l == NULL)
    return next;
  else
    return LL_copylinkitem(l, LL_copyitems(l->next_item, next));
}


/*  */
/**********************************************************************
 * Function: linkitem *LL_copylinkitem(linkitem *ip, linkitem *next)
 *
 * Modifications:
 *      <list mods with name and date>
 */
linkitem *LL_copylinkitem(ip, next)
  linkitem *ip, *next;
{
  static linkitem *px;

  if ((px = (linkitem *) malloc(sizeof(linkitem))) == NULL) {
    LOG_MESS("LL_copylinkitem: malloc failed");
    return NULL;
  }
  px->link.vs = ip->link.vs;
  px->link.inst = ip->link.inst;
  px->next_item = next;
  return px;
}


/*  */
/**********************************************************************
 * Function: char *LL_copyval(int size, char *s)
 *
 * Make a malloc-ed copy of memory block.
 *
 * Modifications:
 *      <list mods with name and date>
 */
char *LL_copyval(size, s)	/* Make a malloc-ed copy of memory block. */
  int size;
  char *s;
{
  char *d;

  if ((d = malloc((ALLOC_T)size)) == NULL) {
    LOG_MESS("LL_copyval: malloc failed");
    return NULL;
  }

  (void)memcpy(d,s,size);

  return d;
}

/*  */
/**********************************************************************
 * Function: char *LL_copystring(char *s)
 *
 * Make a malloc-ed copy of zero-terminated string.
 *
 * Modifications:
 *      <list mods with name and date>
 */
char *LL_copystring(s)	/* Make a malloc-ed copy of zero-terminated string. */
  char *s;
{
  char *d;

  if (s == NULL) {
    LOG_MESS("LL_copystring: input NULL");
    return NULL;
  }

  if ((d = (char *)malloc((ALLOC_T)strlen(s) + 1)) == NULL) {
    LOG_MESS("LL_copystring: malloc failed");
    return NULL;
  }
  (void)strcpy(d, s);
  return d;
}

/*  */
/**********************************************************************
 * Function: void LL_strtoval(char *s, attrval *vp)
 *
 * Make an attrval record for a string
 *
 * Modifications:
 *      <list mods with name and date>
 */
void LL_strtoval(s, vp)		/* Make an attrval record for a string */
  char *s;
  attrval *vp;
{
  vp->attsize = strlen(s) + 1;
  vp->attvalue = LL_copystring(s);
}

/*  */
/**********************************************************************
 * Function: void LL_releasemol(molecule *mol)
 *
 * Frees the memory belonging to the molecule
 *
 * Modifications:
 *      <list mods with name and date>
 */
void LL_releasemol(mol)
  molecule *mol;
{
  if (mol->links)
    LL_freelinks(mol->links);
  if (mol->attributes)
    LL_freeattributes(mol->attributes);
  if (mol->image)
    LL_freeimage(mol->image);
  mol->links = NULL;
  mol->attributes = NULL;
  mol->image = NULL;
  mol->inWS = mol->edited = 0;
}

/*  */
/*
 * These functions below free various data structures
 */
/**********************************************************************
 * Function: void LL_freeattr(attribute *p)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void LL_freeattr(p)
  attribute *p;
{
  free(p->tag);
  free(p->value);
  free((FREEPTR *)p);
}

/*  */
/**********************************************************************
 * Function: void LL_freeattrgroupval(attribute *p)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void LL_freeattrgroupval(p)
  attribute *p;
{
  if (p) {
    LL_freeattrgroupval(p->next);
    LL_freeattr(p);
  }
}

/*  */
/**********************************************************************
 * Function: void LL_freeattrgroup(attrgroup *p)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void LL_freeattrgroup(p)
  attrgroup *p;
{
  free(p->tag);
  LL_freeattrgroupval(p->value);
  free((FREEPTR *)p);
}

/*  */
/**********************************************************************
 * Function: void LL_freeattributes(attrgroup *p)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void LL_freeattributes(p)
  attrgroup *p;
{
  if (p) {
    LL_freeattributes(p->next);
    LL_freeattrgroup(p);
  }
}

/*  */
/**********************************************************************
 * Function: void LL_freelinkval(linkitem *p)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void LL_freelinkval(p)
  linkitem *p;
{
  if (p) {
    LL_freelinkval(p->next_item);
    free((FREEPTR *)p);
  }
}

/*  */
/**********************************************************************
 * Function: void LL_freelinktag(linktag *p)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void LL_freelinktag(p)
  linktag *p;
{
  free(p->tag);
  LL_freelinkval(p->value);
  free((FREEPTR *)p);
}

/*  */
/**********************************************************************
 * Function: void LL_freelinkgroupval(linktag *p)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void LL_freelinkgroupval(p)
  linktag *p;
{
  if (p) {
    LL_freelinkgroupval(p->next);
    LL_freelinktag(p);
  }
}

/*  */
/**********************************************************************
 * Function: void LL_freelinkgroup(linkgroup *p)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void LL_freelinkgroup(p)
  linkgroup *p;
{
  free(p->tag);
  LL_freelinkgroupval(p->value);
  free((FREEPTR *)p);
}

/*  */
/**********************************************************************
 * Function: void LL_freelinks(linkgroup *p)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void LL_freelinks(p)
  linkgroup *p;
{
  if (p) {
    LL_freelinks(p->next);
    LL_freelinkgroup(p);
  }
}

/*  */
/**********************************************************************
 * Function: void LL_freeimage(image_desc *p)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void LL_freeimage(p)
  image_desc *p;
{
  free(p->value);
  free((FREEPTR *)p);
}
