/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	dbcomm.c
 *
 * SCCSINFO:		@(#)dbcomm.c	1.10 6/6/94
 *
 * ORIGINAL AUTHOR(S):  Peter ]berg, Ralph R|nnquist, 18 Sep 1987
 *
 * MODIFICATIONS:
 *      1993-11-02 Martin Sjlin, moved PASSWD_LENGTH into config.h
 *      1994-01-06 Martin Sjlin, added LL_registercallback.
 *	<list mods with name and date>
 *
 *
 * DESCRIPTION:
 * This file contains the LIBLINCKS functions that handle communications
 * between the database server and the workspace.
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <sys/ioctl.h>
#ifdef  HAVE_USLEEP			/* included in config.h ifndef */
#include <sys/time.h>
#endif /* n HAVE_USLEEP */

#include "dbcodes.h"
#include "sunlincks.h"
#include "xconfig.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_dbcomm.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* molecule table */
/*
 * The molecule (node) table contains entries for both history structure
 * nodes (vs) and versions (inst). Label.vs and Label.inst can be
 * used as molecules in the molecule table.
 * For support for the molecule table, see also
 *      wssupport.c : LL_mallocmol()
 *                    LL_newmol
 *                    moltblsize
 *                    MOLTABLRINC
 *      sunlincks.h : molecule struct
 *      dbcodes.h   : molecule type constants
 */
int LL_freemolindex = 0;
molecule *LL_moltbl;

/* user identity */
char LL_loginuser[100];
int LL_uid;

/* socket definitions */
int    LL_socketno;
int    LL_selectno;
fd_set LL_selectmask;

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_logws.h"
#include "f_rdwrmol.h"
#include "f_structsupport.h"
#include "f_lowcom.h"
#include "f_wssupport.h"
#include "f_sysfuns.h"
#include "f_autostart.h"

/* libshared.a */
extern char *strdup();
extern int configuration();

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern char *sys_errlist[];
extern int sys_nerr;
extern int errno;
extern int pen_off;

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
/* Size of network address */
#define HOSTNAMESIZE 133

/* Size increments for interface table */
#define ITABLEINC 256

/*
 * reserved place in the molecule table used for resolving
 * history structure nodes
 */
#define MT_RES 1

#define MAX_LOGIN_ATTEMPTS 3	/* how many times to try an interactive
				   login */
#define NAME_LENGTH 64		/* how long the username might be */
#define BUFF_LENGTH 132		/* buffer length used in communication */
#define RETRY 99999		/* just a hack to get multiple login tries */

#define PENINTARRAYSIZE 64      /* size of static array in pen message */

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static int DBConnect P_(( char *userid, char *password, int attempts ));
static int NewITix P_(( int wanted ));
static errcode resolvevs P_(( int mol, int moltype ));
static void resolvevs_working P_(( int mol ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
/* Interface table (built from mallocced memory)*/
static int *ITbl = (int *)NULL;
static int ITblsize = 0;
static int freeITindex = 0;
/* coordinate with dbcodes.h */
static struct {
  int (* function)(/* void *extra, int index, int moltype */);
  void *extra;
  } callback_func[MAX_DB_CODE] = {
  {NULL, NULL}, {NULL, NULL}, {NULL, NULL}, {NULL, NULL},
  {NULL, NULL}, {NULL, NULL}, {NULL, NULL}, {NULL, NULL},
  {NULL, NULL}, {NULL, NULL} 
  };

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: errcode LL_DBPreedit(int index, int moltype)
 *
 * Tell db that a new version is being created for history node `index'.
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_DBPreedit(index, moltype)
  int index;
  int moltype;
{
  int ITindex;

  if ((LL_moltbl[index].mark) || pen_off)
    return SUCCESS;
  if ((LL_MoltoIT(index, &ITindex) < 0) ||
      (LL_SendCommand(EDIT_CODE) < 0) ||
      (LL_SendIndex(ITindex) < 0))
    return FAIL;

  if (callback_func[EDIT_CODE].function)
    (void) (*callback_func[EDIT_CODE].function)
      (callback_func[EDIT_CODE].extra, index, moltype); 

  return SUCCESS;
}



/*  */
/**********************************************************************
 * Function: errcode LL_DBStore(int index, int moltype)
 *
 * Stores a molecule with given mol tbl index and the type
 * of the molecule to the db
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_DBStore(index, moltype)
  int index;
  int moltype;	/* OTHER or VS */
{
  linkgroup *trans = (linkgroup *)NULL;
  linktag *latest = (linktag *)NULL;
  linkitem *p = (linkitem *)NULL;
  int ITindex;
  int rtn = SUCCESS;

  if (index >= LL_freemolindex)
    return ERR_LABEL;
  if (!LL_moltbl[index].edited)
    return (SUCCESS);
  if (moltype !=  OTHER) {
    /*
     * If the current node is transient, then we must reconstruct
     * the preceding store state, which in particular means to
     * change the present "SYSTEM_LATEST" to be the parent(s) of
     * the transient version.
     */
    if (!LL_sysgetlinkgroup(index, TH_TRANSIENT, &trans) &&
	trans->value && trans->value->value &&
	!LL_sysgetlink(index, SYSTEM_LATEST, &latest)) {
      p = latest->value;
      latest->value = trans->value->value->next_item;
    }
  }
  if ((LL_MoltoIT(index, &ITindex) < 0) ||
      (LL_SendCommand(STORE_CODE) < 0) ||
      (LL_SendIndex(ITindex) < 0) ||
      (LL_SendMolType(moltype) < 0) ||
      (LL_SendMolecule(&LL_moltbl[index]) < 0)) {
    rtn = FAIL;
  } else {
    LL_moltbl[index].edited = 0;
  }
  if (p)
    latest->value = p;

  if (rtn && callback_func[STORE_CODE].function)
    (void) (*callback_func[STORE_CODE].function)
      (callback_func[STORE_CODE].extra, index, moltype); 

  return rtn;
}

/*  */
/**********************************************************************
 * Function: errcode LL_DBRetrieve(int index, int moltype)
 *
 * Retrieves a molecule with given mol tbl index and moltype from db
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_DBRetrieve(index, moltype)
  int index;
  int moltype;
{
  molecule *mol;
  int newindex = 0;
  int ITindex;
  byte code;
  linkgroup *grp;

  if (index >= LL_freemolindex) {
    if ((newindex = LL_newmol()) == FAIL)
      return FAIL;
  } else {
    newindex = index;
  }

  if ((LL_MoltoIT(newindex, &ITindex) < 0) ||
      (LL_SendCommand(RETRIEVE_CODE) < 0) ||
      (LL_SendIndex(ITindex) < 0) ||
      (LL_SendMolType(moltype) < 0)) {
    return FAIL;
  }
  if (LL_ReceiveStatus(&code) < 0)
    return -code;

  switch (code) {
  case RCMOL:
    LL_releasemol(&LL_moltbl[newindex]);
    if (!(mol = LL_ReceiveMolecule()))
      break;
    LL_moltbl[newindex].links = mol->links;
    LL_moltbl[newindex].attributes = mol->attributes;
    LL_moltbl[newindex].image = mol->image;
    LL_moltbl[newindex].inWS = 1;
    LL_moltbl[newindex].edited = LL_moltbl[newindex].mark = 0;
    mol->links = (linkgroup *)NULL;
    mol->image = (image_desc *)NULL;
    mol->attributes = (attrgroup *)NULL;
    if (moltype != OTHER) {
      if (!LL_sysgetlinkgroup(newindex, TH_TRANSIENT, &grp) && grp->value) {
	LL_freelinkgroupval(grp->value);
	grp->value = (linktag *)NULL;
      }
      if (!LL_sysgetlinkgroup(newindex, BUFFERNODE, &grp) && grp->value) {
	LL_freelinkgroupval(grp->value);
	grp->value = (linktag *)NULL;
      }
    }
    if (LL_ReceiveStatus(&code) < 0)
      return -code;
    break;

  case RCBUFFER:
    LL_releasemol(&LL_moltbl[newindex]);
    LL_moltbl[newindex].links = (linkgroup *)NULL;
    LL_moltbl[newindex].attributes = (attrgroup *)NULL;
    LL_moltbl[newindex].image = (image_desc *)NULL;
    LL_moltbl[newindex].inWS = 1;
    LL_moltbl[newindex].edited = LL_moltbl[newindex].mark = 0;
    break;

  case RCOK:
    return ERR_ALREADY;
    break;

  default:
    return ERR_STATUS;
    break;
  }

  if (callback_func[RETRIEVE_CODE].function)
    (void) (*callback_func[RETRIEVE_CODE].function)
      (callback_func[RETRIEVE_CODE].extra, newindex, moltype); 

  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: errcode LL_DBRelease(int index, int moltype)
 *
 * Releases molecule with given mol tbl index at db
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_DBRelease(index, moltype)
  int index;
  int moltype;
{
  int ITindex;
  byte code;

  if (index >= LL_freemolindex)
    return ERR_LABEL;

  if (callback_func[RELEASE_CODE].function)
    (void) (*callback_func[RELEASE_CODE].function)
      (callback_func[RELEASE_CODE].extra, index, moltype); 

  if (LL_MoltoIT(index, &ITindex) < 0)
    return FAIL;
  if (LL_SendCommand(RELEASE_CODE) < 0)
    return FAIL;
  if (LL_SendIndex(ITindex) < 0)
    return FAIL;
  if (LL_SendMolType(moltype) < 0)
    return FAIL;
  if (LL_ReceiveStatus(&code) < 0)
    return -code;
  LL_releasemol(&LL_moltbl[index]);

  LL_moltbl[index].inWS = LL_moltbl[index].edited = 0;
  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: errcode LL_DBPenMessage(int (*mapfn)(), void *extra)
 *
 * Retrieves a pen-message with given index type and message string
 * from db. For description of mapfn, see usfuns.c
 *
 *
 * Modifications:
 *      1993-08-12 Martin Sjlin  Receives all pen message from
 *                 the dbs at once ( OOB only at least one (1)
 *		   copy is stored at the client). This should make
 *                 the pen stuff less susceptible to overruns.
 */
errcode LL_DBPenMessage( mapfn, extra)
  int (*mapfn)();
  void *extra;
{
  byte  code;
  int   st;
  int   i;
  int   count = 0;
  int   total = 0;
  int   dbsleft = 0;
  int   skip = FALSE;
  int   type = 0;
  int   index = 0;
  char *msg = (char *)NULL;

  do {
    if (LL_SendCommand(MESSAGE_CODE) < 0)
      return FAIL;

    for( i=0; i < 100; i++) {
      st = LL_ReceiveStatus(&code);
#ifdef PENDEBUG
      (void) fprintf( stderr,
		     "LL_DBPenMessage: ReceiveStatus => %d code = %d\n", 
		     st, (int) code);
#endif
      if (st == 0)
	break;
    } /* for */

    if ( i == 100 && st < 0)
      return -code;

    if ( code == RCOK ) {		/* no message left in queue .... */
      if ( total == 0 )			/* call mapfn if first time around */
	(void) (*mapfn)( extra, 0, 0, (char *)NULL, count, total);
      break;
    }
    if ( code != RCMESSAGE )
      return(ERR_STATUS);      
    if ( LL_ReceiveCount(&dbsleft, LL_socketno) < 0 || dbsleft < 0 )
	return(ERR_STATUS);

    /* if first fetch - call mapfn and set up maxvalue (count) */
    if ( total == 0 ) {
      total = dbsleft;
      skip  = (*mapfn)( extra, 0, 0, (char *)NULL, count, total) != 0;
    }

    if ( LL_ReceivePenMessage(&index, &type, &msg) < 0 ) 
      return FAIL;
	
    /* do we need to resolve the index ?  */
    if ( type == PEN_RESOLVE && !resolvevs( index, VSH))
      (void)LL_DBStore( index, VS);

    count++;

    if ( !skip ) 
      skip = (*mapfn)( extra, index, type, msg, count, total) != 0;
    else if ( msg ) 
      free( msg );  /* free allocated memory if not calling mapfn */

  } while (!skip && count <= total);
  
  return(SUCCESS);
}

/*  */
/**********************************************************************
 * Function: errcode LL_DBLogout()
 *
 * Sends a logout command to the database and terminates
 * DOES NOT RETURN
 *
 * Modifications:
 *	1993-10-11 Martin Sjolin. FIRST, try to get in sync with the
 *              database, THEN send logout and read status back. This
 *              is done in case the server is really loaded (since we
 *              cannot use LL_SyncTest after sending the logout, then
 *              the RCLOGO will be caught in ReceiveStatus in the
 *		LL_SyncTest).
 *      <list mods with name and date>.
 */
errcode LL_DBLogout()
{
  byte c;
  int retrycount;
  int st;

  /* Get in sync with dbs */
  (void)LL_SyncTest();

  /* Send logout code */
  if ((LL_SendCommand(LOGO_CODE)) == FAIL) {
    (void)fprintf(stderr, "LL_SendCommand failed in LL_DBLogout.\n");
    return FAIL;
  }

  /* and wait for RCLOGO back or failure */
  for(retrycount = NODBRETRIES; retrycount > 0; retrycount--) {
    if ((st = LL_ReceiveStatus(&c)) == FAIL && c != RCBAD)
      continue;
    
    switch (c) {
    case RCOK:		/* left from LL_SyncTest .... */
      continue;
    case RCLOGO:
      (void)fprintf(stderr, "\nGoodbye!\n");
      return SUCCESS;
    case RCBAD:
      return ERR_RCBAD;
    default:
      (void) fprintf( stderr,
	      "\n%s: %s (%d, %d)\n",
	      "Improper termination",
	      "may need to kill database server process",
	      (int)c, st);
      break;
    } /* switch */
  } /* for */

  return(FAIL);

}

/*  */
/**********************************************************************
 * Function: errcode LL_DBRetry(int mol, int moltype, errcode (*dbfn)())
 *
 * Executes a DB command and repeats a set number of times if it fails
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_DBRetry(mol, moltype, dbfn)
  int mol;
  int moltype;
errcode(*dbfn) ();
{
  int ERRCODE;
  int retrycount = NODBRETRIES;

  while ((ERRCODE = (*dbfn) (mol, moltype)) && (retrycount-- > 0)) {
    if ((ERRCODE = LL_SyncTest()))
      return ERRCODE;
    else if (dbfn == LL_DBRetrieve)
      (void)LL_DBRelease(mol, moltype);
  }
  return ERRCODE;
}

/*  */
/**********************************************************************
 * Function: errcode LL_startup(char *userid, char *password, char *dbid)
 *
 * Initiates the connection to the database server and
 * sets up the interface table initial state.
 *	     user is user name.
 *	     password is user password.
 *	     dbid is the database directory.
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_startup(userid, password, dbid)
  char *userid;
  char *password;
  char *dbid;
{
  int err;
  int attempts = 0;
  int done = 0;

  if (userid && password)
    attempts = MAX_LOGIN_ATTEMPTS - 1;

  while (!done && attempts < MAX_LOGIN_ATTEMPTS) {
    attempts++;

    /* Load run-time configuration */
    if (configuration(dbid) == 0)
      return (FAIL);

    /* Create system molecule entry */
    (void)LL_newmol();
    if (LL_MoltoIT(0, &LL_moltbl[0].ITix) < 0)
      return FAIL;
    ITbl[0] = 0;

    /* Create reserved mol entry # 1 */
    (void)LL_mallocmol();

    /* Connect to DB - pass the user id and password */
    if ((err = DBConnect(userid, password, attempts)) != SUCCESS) {
      if (err != RETRY) {
	return err;
      }
    } else {
      done = 1;
    }
  }
  if (!done)
    return ERR_LOGIN;

  /* Test that the channel is synchronized */
  if (LL_SyncTest() < 0)
    return FAIL;

  /* Ignore Control-C (really SIGINT), but why ? */
  /* (void) Signal(SIGINT, SIG_IGN); removed by marsj */

  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: static int DBConnect(char *userid, char *password)
 *
 * Establishes connection with database server process
 *	     char *userid      is user name.
 *	     char *password    is user password.
 *
 * Modifications:
 *      1994-01-16 Martin Sjlin. Open our own file for reading
 *           and writing if interactive, else we piping from/to
 *           stdin/stdout we get into problems.
 *      <list mods with name and date>
 */
static int DBConnect(userid, password, attempts)
  char *userid;			/* pointer to user name */
  char *password;		/* pointer to user password */
  int attempts;			/* how many times we've tried */
{
  unsigned char c;
  char name[NAME_LENGTH];	/* user name buf when asking for it */
  char passwd[PASSWD_LENGTH];	/* ditto */
  char buff[BUFF_LENGTH];	/* buffer used during comm */
  char host[HOSTNAMESIZE];	/* buffer for host name */
  FILE *intty;			/* file id for input */
#ifdef HAVE_TERMIOS
  struct termios status;
#else
  struct sgttyb status;
#endif
  int to_write = 0;
  int interactive = 0;

  /* set up a flag to guide me through the function */
  if (!userid && !password)
    interactive = 1;

  /* Try to find the monitor host, return name in host */
  if (LL_FindMonitorHost(host, &LL_socketno) == FAIL)
    return FAIL;

  /* Get first handshake - if prompting for password, give some */
  /* more information on stdout, else keep quit */
  (void)read(LL_socketno, (char *)&c, sizeof(c));
  if (c == SPECBYTE) {
    /* if (interactive)
	(void)fprintf(stderr, "Connected to %s.\n\n", host); */
  } else {
    if (interactive)
      (void)fprintf(stderr, "Initiation failure connecting to %s\n\n", host);
    return FAIL;
  }

  /* if either of userid or password is a NULL string, */
  /* prompt for information, either try supplied id/password */
  if (interactive) {
    /* open /dev/tty for reading/writing */
    if ((intty = fopen("/dev/tty","r")) == NULL) {
      perror("Connect(0): Failed to open /dev/tty for input\n");
      return(FAIL);
    }
 
    /* Let user attempt to log in */
    (void)fprintf(stderr, "LINCKS login: ");
    (void)fgets(name, sizeof(name),intty);
    name[strlen(name) - 1] = '\0';	/* Zap trailing newline */

    /* Get status for intty */
#ifdef HAVE_TERMIOS
    if (tcgetattr(fileno(intty), &status) < 0) {
      perror("Connect: tcgetattr call failed");    
#else
    if (ioctl(fileno(intty), TIOCGETP, &status) < 0) {
      perror("Connect: ioctl call (1) failed");
#endif
      (void)fclose(intty);
      return FAIL;
    }

    /* Disable echo on intty */
#ifdef HAVE_TERMIOS
    status.c_lflag  &= ~ECHO;
    if (tcsetattr( fileno(intty), TCSANOW, &status) < 0) {
      perror("Connect: tcsetattr call (1) failed");
#else
    status.sg_flags &= ~(short)ECHO;
    if (ioctl(fileno(intty), TIOCSETP, &status) < 0) {
      perror("Connect: ioctl call (2) failed");
#endif
      (void)fclose(intty);
      return FAIL;
    }
    /* Get password (without echo) */
    (void)fprintf(stderr, "password: ");
    (void)fgets(passwd, sizeof(passwd), intty);
    passwd[strlen(passwd) - 1] = '\0';	/* Zap trailing newline */

    /* Enable echo on intty */
#ifdef HAVE_TERMIOS
    status.c_lflag  |= ECHO;
    if (tcsetattr( fileno(intty), TCSANOW, &status) < 0) {
      perror("Connect: tcsetattr call (2) failed");
#else
    status.sg_flags |= (short)ECHO;
    if (ioctl(fileno(intty), TIOCSETP, &status) < 0) {
      perror("Connect: ioctl call (3) failed");
#endif
      (void)fclose(intty);
      return FAIL;
    }
    /* Copy address into our pointers */
    password = passwd;
    userid = name;

    /* and close file */
   (void)fclose(intty);
  }

  /* Send identification to server for verification */
#ifdef OOBSELECTBUG
  (void)sprintf(buff, "l%c%s%c%s",
#else
  (void)sprintf(buff, "L%c%s%c%s",
#endif
		(unsigned char) strlen(userid), userid, 
		(unsigned char) strlen(password), password);
  to_write = strlen(userid) + strlen(password) + 3;
  if ((write(LL_socketno, buff, (IOSIZE_T)to_write)) < 0) {
    perror("write failed in DBConnect");
  }
  /* Get response - if failed */
  (void)read(LL_socketno, (char *)&c, sizeof(c));

  if (c == 255) {
    (void)fprintf(stderr, "\n");
    if (attempts >= MAX_LOGIN_ATTEMPTS) {
      if (interactive)
	(void)fprintf(stderr, "\n%d login failures. You lose.\n",
		      MAX_LOGIN_ATTEMPTS);
      return ERR_LOGIN;
    }
    return RETRY;
  }
  /* Get user id from dbserver */
  (void)read(LL_socketno, (char *)&c, sizeof(c));
  LL_uid = c;
  (void)strcpy(LL_loginuser, userid);

  /* Print welcome message, if we prompted for password/id */
  if (interactive) {
    (void)fprintf(stderr, "\n\n\t%s %d (%s).  %s %s.\n",
		  "Welcome user", LL_uid, userid, "You're logged in to", host);
  }
  /* Prepare mask for select calls */
  FD_ZERO(&LL_selectmask);
  FD_SET(LL_socketno, &LL_selectmask);
  LL_selectno = LL_socketno+1; /* going throgh 0 to -1 => add one */

  /* Successful login */
  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: errcode LL_ITtoMol(int ITindex, int *molindex)
 *
 * Converts an interface table index to a mol table index
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_ITtoMol(ITindex, molindex)
  int ITindex;
  int *molindex;
{
  if (ITindex == -1)
    *molindex = -1;

  else if (ITindex < freeITindex)
    *molindex = ITbl[ITindex];

  else {
    if (NewITix(ITindex) < 0)
      return FAIL;

    while (ITindex >= freeITindex) {
      if ((*molindex = LL_newmol()) < 0)
	return FAIL;
      LL_moltbl[*molindex].ITix = freeITindex;
      ITbl[freeITindex++] = *molindex;
    }
  }

  /* Return status */
  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: errcode LL_MoltoIT(int molindex, int *ITindex)
 *
 * Converts a mol table index to its interface table index
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_MoltoIT(molindex, ITindex)
  int molindex;
  int *ITindex;
{
  if (molindex == -1)
    *ITindex = -1;

  else if (LL_moltbl[molindex].ITix != -1)
    *ITindex = LL_moltbl[molindex].ITix;

  else {
    if (NewITix(freeITindex) < 0)
      return FAIL;

    *ITindex = LL_moltbl[molindex].ITix = freeITindex++;
    ITbl[*ITindex] = molindex;
  }

  /* Return status */
  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: static int NewITix(int wanted)
 *
 * Mallocs IT table space and returns the requested index
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int NewITix(wanted)
  int wanted;
{
  unsigned newsize;

  newsize = wanted + ITABLEINC;

  if (ITblsize == 0) {
    if (!(ITbl = (int *)calloc((ALLOC_T) newsize, sizeof(int)))) {
      LogMess(LL_uid,
	      "Liblincks: Couldn't allocate IT table space in NewITix, %s",
	      sys_errlist[(errno >= sys_nerr) ? 0 : errno]);
      perror("WARNING!: Cannot Allocate IT Table Space In \"NewITix\"\n");
      return FAIL;
    }
    ITblsize = newsize;
  } else {
    if (!(ITbl = (int *)realloc((FREEPTR *)ITbl, (ALLOC_T) newsize * sizeof(int)))) {
      LogMess(LL_uid,
	      "Liblincks: Couldn't reallocate IT table space in NewITix, %s",
	      sys_errlist[(errno >= sys_nerr) ? 0 : errno]);
      perror("WARNING!: Cannot Reallocate IT Table Space In \"NewITix\"\n");
      return FAIL;
    }
    ITblsize = newsize;
  }

  /* Return ITindex */
  return wanted;
}

/*  */
/**********************************************************************
 * Function: errcode LL_DBResolve(int mol, int moltype)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_DBResolve(mol,moltype)
  int mol;
  int moltype;
{
  if (moltype != VS)
    return ERR_LABEL;
  return resolvevs(mol,VS);
}

/*  */
/**********************************************************************
 * Function: static errcode resolvevs(int mol, int moltype)
 *
 * Takes care of temporal structure resolving.
 * Resolving has to be done  when another process has
 * updated the history structure node
 * since we last fetched or stored it.
 * The moltype variable should be instantiated with VS or VSH
 * upon calling this function.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode resolvevs(mol, moltype)
  int mol, moltype;
{
  attribute *cp;
  int count, ERRCODE;

  /*
   * Copy 'mol' into special reserved place in moltbl and
   * empty out contents of mol in 'mol' 's real place in moltbl
   */

  LL_moltbl[MT_RES].links = LL_moltbl[mol].links;
  LL_moltbl[MT_RES].attributes = LL_moltbl[mol].attributes;
  LL_moltbl[MT_RES].image = LL_moltbl[mol].image;
  LL_moltbl[MT_RES].edited = LL_moltbl[mol].edited;
  LL_moltbl[MT_RES].inWS = LL_moltbl[mol].inWS;
  LL_moltbl[MT_RES].mark = LL_moltbl[mol].mark;

  LL_moltbl[mol].links = (linkgroup *)NULL;
  LL_moltbl[mol].attributes = (attrgroup *)NULL;
  LL_moltbl[mol].image = (image_desc *)NULL;
  LL_moltbl[mol].edited = LL_moltbl[mol].inWS = FALSE;

  /*
   * Retrieve the database copy of the history node.
   */
  if ((ERRCODE = LL_DBRetrieve(mol, moltype)) ||
      (!LL_moltbl[mol].links)) {
    /*
     * The database variant couldn't be retrieved, so we pretend
     * it's raining. (Ralph means : we just restore the 'mol'
     * to the old state.)
     * Reason is either fail or that the workspace copy of the
     * history node currently is the correct one.
     */
    LL_moltbl[mol].links = LL_moltbl[MT_RES].links;
    LL_moltbl[mol].attributes = LL_moltbl[MT_RES].attributes;
    LL_moltbl[mol].image = LL_moltbl[MT_RES].image;
    LL_moltbl[mol].mark = LL_moltbl[MT_RES].mark;
    LL_moltbl[mol].edited = LL_moltbl[MT_RES].edited;
    LL_moltbl[mol].inWS = LL_moltbl[MT_RES].inWS;
    return ERRCODE;
  }
  /* Preserve workspace marking */
  LL_moltbl[mol].mark = LL_moltbl[MT_RES].mark;
  /*
   * Set VSNET:Count to be the biggest of the database copy
   * and the workspace copy of VSNET:Count.
   */
  count = 0;
  if (!LL_sysgetattr(MT_RES, VSNET_COUNT, &cp))
    count = *((int *)cp->value);
  if ((!LL_sysgetattr(mol, VSNET_COUNT, &cp) &&
       (count > *((int *)cp->value))))
    *((int *)cp->value) = count;

  /*
   * Update the history structure
   */
  resolvevs_working(mol);

  /*
   * Reclaim the memory of 'MT_RES' and store the resolved history node.
   */
  if (LL_moltbl[MT_RES].attributes)
    LL_freeattributes(LL_moltbl[MT_RES].attributes);
  if (LL_moltbl[MT_RES].links)
    LL_freelinks(LL_moltbl[MT_RES].links);
  if (LL_moltbl[MT_RES].image)
    LL_freeimage(LL_moltbl[MT_RES].image);

  return SUCCESS;
}


/*  */
/**********************************************************************
 * Function: int next_vs_fragment(int vs)
 *
 * Modifications:
 *      <list mods with name and date>
 */
int next_vs_fragment(vs)
     int vs;
{
  linkitem *p;

  if (LL_sysgetlinkitem(vs, SYSTEM_CHAIN,1,&p))
    return -1;
  return p->link.vs;
}


/*  */
/**********************************************************************
 * Function: static void resolvevs_working(int mol)
 *
 * Merges the proper history from database and workspace copies and
 * updates SYSTEM_LATEST.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void resolvevs_working(mol)
  int mol;
{
  linkgroup *wsgrp = (linkgroup *)NULL;
  linkgroup *wstrans = (linkgroup *)NULL;
  linkgroup *dbgrp = (linkgroup *)NULL;
  linkgroup *chaingrp = (linkgroup *)NULL;
  linkgroup *dbtrans = (linkgroup *)NULL;
  linktag *wsslot = (linktag *)NULL;
  linktag *wsprev = (linktag *)NULL;
  linktag *dbslot = (linktag *)NULL;
  linktag *dbold = (linktag *)NULL;
  linktag *dblatest = (linktag *)NULL;
  linktag *wsbtl = (linktag *)NULL;
  linktag *dbbtl = (linktag *)NULL;
  linkitem *p = (linkitem *)NULL;
  linkitem *q = (linkitem *)NULL;
  char *datetag = (char *)NULL;
  int vs = -1;

  /*
   * The history is reconstructed by adding to the database copy all
   * slots from the workspace copy that are not already there. Since
   * slots are added to the top of the list, the procedure is as
   * follows. Map over the database copy history slots to find the
   * first slot that occurs in the workspace copy. Add to the top of
   * the database copy history all slots prior to this merge point in
   * the workspace copy.
   */

  (void)LL_sysgetlinkgroup(MT_RES, VSNET, &wsgrp);
  (void)LL_sysgetlinkgroup(mol, VSNET, &dbgrp);

  dbold = dbgrp->value;	/* Used below in the update of SYSTEM_LATEST */

  for (vs = mol; vs != -1; vs = next_vs_fragment(vs)) {
    if (LL_sysgetlinkgroup(vs, VSNET, &chaingrp))
      continue;
    mapfields(dbslot,chaingrp) {
      wsprev = (linktag *)NULL;
      mapfields(wsslot,wsgrp) {
	if (dbslot->value->link.inst == wsslot->value->link.inst)
	  break;
	wsprev = wsslot;
      }
      if (wsslot) {
	if (wsprev) {
	  wsprev->next = dbgrp->value;
	  dbgrp->value = wsgrp->value;
	  wsgrp->value = wsslot;
	}
	break;
      }
    }
    if (wsslot)
      break;
  }

  /*
   * If there is no development in the workspace then we are ready.
   */
  if (LL_sysgetlinkgroup(MT_RES, TH_TRANSIENT, &wstrans) ||
      !wstrans->value || !wstrans->value->value) {
    wstrans = (linkgroup *)NULL;
    if (!wsprev)
      return;
  }

  LL_moltbl[mol].edited = TRUE;

  /*
   * Regenerate date tags for the portion added and begin handle
   * SYSTEM_LATEST. This means to add the versions of the local
   * development(s), and then reduce with those versions
   * that appear as predecessors in the portion added.
   * A NOTE on representation:
   *  a history slot has designated positions such that
   *  + first is the version label,
   *  + next is the "store command" label,
   *  + the rest are all immediate predecessors.
   * The transitent slot looks the same except that the "store command"
   * label is omitted.
   */
     
  (void)LL_sysgetlink(mol, SYSTEM_LATEST, &dblatest);
  mapfields(dbslot,dbgrp) {
    if (dbslot == dbold)
      break;
    (void)LL_VSTag(mol, &datetag);
    free(dbslot->tag);
    dbslot->tag = strdup(datetag);
    (void)LL_sysputlinkitem(mol, SYSTEM_LATEST, 1, &(dbslot->value->link));
  }
  mapfields(dbslot,dbgrp) {
    if (dbslot == dbold)
      break;
    for (p = dbslot->value->next_item->next_item; p; p = p->next_item)
      (void)LL_sysdellinkitem(&(dblatest->value),&(p->link));
  }
 
  /*
   * Now the SYSTEM_LATEST is properly updated to the point prior to
   * an eventual transient.
   * Rebuild transient slot if there is one.
   * The transient slot should consist of
   *  + first the transient version label,
   *  + thereafter the present SYSTEM_LATEST
   * and SYSTEM_LATEST should be updated so as to contain only the
   * transient version.
   * NOTE: when the version node is stored while there is a transient
   * version, SYSTEM_LATEST must be temporarily re-assigned to be
   * the predecessors of the transient version rather than the transient
   * version itself (see LL_DBStore).
   */
  if (!LL_sysgetlinkgroup(mol, TH_TRANSIENT, &dbtrans)) {
    LL_freelinkgroupval(dbtrans->value);
    dbtrans->value = (linktag *)NULL;
  }
  if (wstrans) {
    (void)LL_VSTag(mol, &datetag);
    (void)LL_sysputlinkitem(mol, TH_TRANSIENT, datetag, 1,
			    &(wstrans->value->value->link));
    (void)LL_sysgetlinkitem(mol, TH_TRANSIENT, datetag, 1, &p);
    p->next_item = dblatest->value;
    dblatest->value = (linkitem *)NULL;
    (void)LL_sysputlinkitem(mol, SYSTEM_LATEST, 1,
			    &(wstrans->value->value->link));
  }

  /*
   * The association list for binding tables is managed as follows. Add
   * to the database copy every association from the workspace copy
   * that is not already included.
   */
  if (!LL_sysgetlink(MT_RES, BTBL_LINK,&wsbtl)) {
    if (!LL_sysgetlink(mol, BTBL_LINK,&dbbtl)) {
      mapitems(p,wsbtl) {
	p = p->next_item;
	mapitems(q,dbbtl) {
	  q = q->next_item;
	  if (p->link.vs == q->link.vs) {
	    p->link.vs = -1;
	    break;
	  }
	}
      }
    }
    mapitems(p,wsbtl) {
      q = p->next_item;
      if (q->link.vs != -1) {
	(void)LL_sysputlinkitem(mol, BTBL_LINK, 1,&(q->link));
	(void)LL_sysputlinkitem(mol, BTBL_LINK, 1,&(p->link));
      }
      p = q;
    }
  }

  /*
   * The history structure chain is not changed, since it is the
   * database itself that is responsible for managing it. Thus, the
   * chain as it is in the database copy is already correct.
   */
}

/*  */
/**********************************************************************
 * Function: errcode LL_lastinst(linktag *tp)
 *
 *  LL_lastinst: Gets the last instance link of a linktag
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_lastinst(tp)
  linktag *tp;
{
  linkitem *p;

  if (!tp)
    return -1;

  mapitems(p, tp)
    if (!p->next_item)
      return p->link.inst;
  return -1;
}

/*  */
/**********************************************************************
 * Function: int (*LL_registercallback(int what, int (*callback)(), void *extra ))();
 *
 *  LL_registercallback - install lowlevel callback function 
 *
 * Modifications:
 *      <list mods with name and date>
 */
int (*LL_registercallback(what, callback, extra))()
  int what;
  int (* callback)(/* void *extra, int index, int moltype */);
  void *extra;
{
  static int map[] = 
    { EDIT_CODE, STORE_CODE, RETRIEVE_CODE, RELEASE_CODE };
  int (* prev)() = NULL;

  /* if outside limit, return NULL */
  if (0 <= what && what < sizeof(map)) {
    int index = map[what];
    prev = callback_func[index].function;
    callback_func[index].function = callback;
    callback_func[index].extra    = extra;
  }
  return(prev);
}
