/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * MODULE NAME:		rendering.c
 *
 * SCCSINFO:		@(#)rendering.c	1.7 6/6/94
 *
 * ORIGINAL AUTHOR(S):  Michael Jansson, 28-Feb-92
 *
 * MODIFICATIONS:
 *      <list mods with name and date>
 *
 * DESCRIPTION:
 *	This module is part of the GraphWidget.
 *	It contains routines that arranges the layout of the graph,  and 
 *	which updates it on screen. It contains the stubs for node detection
 *	also.
 *
 **************************************************************************
 * INCLUDES:
 **************************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>

#include "graphP.h"

/**************************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 **************************************************************************
 */
#include "f_rendering.h"

/**************************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 **************************************************************************/
/* None */

/**************************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 **************************************************************************/
#include "f_graph.h"
#include "f_graphutils.h"

/**************************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 **************************************************************************/
/* None */

/**************************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 **************************************************************************/
#define DEGREES 64
#define Max(x, y)	((x)>(y) ? (x) : (y))
#define Min(x, y)	((x)<(y) ? (x) : (y))
#define Abs(v)		((v)>=0 ? (v) : -(v))
#define WithIn(v, v1, v2)	((v)>=(v1)) && ((v)<=(v2))

/**************************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 **************************************************************************/
static void DrawAllLinks P_(( GraphWidget graph, GraphNode *gnode ));
static void DrawAllNodes P_(( GraphWidget graph, GraphNode *gnode ));
static void DrawLink P_(( GraphWidget graph, GraphNode *gnode, int x1,
                           int y1, int x2, int y2 ));
static void DrawNode P_(( GraphWidget graph, GraphNode *gnode, int x,
                           int y ));
/**************************************************************************
 * INTERNAL (STATIC) DATA: 
 **************************************************************************/
/* None */

/*  */
/**********************************************************************
 * Function: void DefDrawNode(SIX PARAMETERS)
 * Parameters:
 *	GraphWidget graph
 *	GC gc
 *	int x, y, size
 *	void *value
 *
 * The default method in the widget that is used to draw the nodes.
 * 
 * Modifications:
 *      <list mods with name and date>
 */
void DefDrawNode(graph, gc, x, y, size, value)
  GraphWidget graph;
  GC gc;
  int x, y, size;
  void *value;
{
  char string[256];

  /* Clear the background of the node. */
  XSetForeground(XtDisplay(graph),gc,WhitePixelOfScreen(XtScreen(graph)));

  XFillArc(XtDisplay(graph), XtWindow(graph), gc, x-size/2, y-size/2,
	   (unsigned int)size, (unsigned int)size, 0, 360*DEGREES);

  XSetForeground(XtDisplay(graph), gc,BlackPixelOfScreen(XtScreen(graph)));

  if (((long)value) != 0) {
    /* Draw a circle. */
    XDrawArc(XtDisplay(graph), XtWindow(graph), gc, x-size/2, y-size/2,
	     (unsigned int)size, (unsigned int)size, 0, 360*DEGREES);
  }

  /* Print the value of the node in the middle of the circle. */
  if ((long)value == 0)
    (void)sprintf(string, "Hidden");
  else if ((long)value == (int)(graph->graph.current))
    (void)sprintf(string, "%ld*", (long)value);
  else
    (void)sprintf(string, "%ld", (long)value);
  if (graph->graph.font) {
    x -= (XTextWidth(graph->graph.font, string, strlen(string)))/2;
    y += graph->graph.font->ascent/2;
  }
  XDrawString(XtDisplay(graph), XtWindow(graph), gc,
	      x, y, string, strlen(string));
}


/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: void DefDrawLink(SEVEN PARAMETERS)
 * Parameters:
 *	GraphWidget graph;
 *	GC gc;
 *	int x1, y1, x2, y2, size;
 * 
 * Modifications:
 *      <list mods with name and date>
 */
void DefDrawLink(graph, gc, x1, y1, x2, y2, size)
  GraphWidget graph;
  GC gc;
  int x1, y1, x2, y2, size;
{
  if (y1==y2 && Abs(x2-x1) >=
	     2*Abs(graph->graph.space_x)*graph->graph.scale/FIX_OFFSET) {
    XDrawArc(XtDisplay(graph), XtWindow(graph), gc,
	     Min(x1, x2), y1 - Abs(x2-x1)/4, 
	     (unsigned int)Abs(x2-x1), (unsigned int)Abs(x1-x2)/2,
	     180*DEGREES, 180*DEGREES);
  } else {
    XDrawLine(XtDisplay(graph), XtWindow(graph), gc,
	      x1, y1, x2, y2);
  }
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static void DrawNode(GraphWidget graph,GraphNode *gnode,int x,y)
 *
 * Calculate the position and size of the node and then call the DrawNode
 * callback.
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static void DrawNode(graph, gnode, x, y)
  GraphWidget graph;
  GraphNode *gnode;
  int x, y;
{
  int posx, posy;
  int size;
  void *value;
  GC gc;

  posx = x*graph->graph.scale/FIX_OFFSET;
  posy = y*graph->graph.scale/FIX_OFFSET;
  size = graph->graph.nodesize*graph->graph.scale/FIX_OFFSET;
  value = gnode->value;
  if ((graph->graph.DrawNode == DefDrawNode) &&
      (graph->graph.root == gnode))
    value = (void *)-1L;
  gc = graph->graph.gc;
  (*graph->graph.DrawNode)(graph, gc, posx, posy, size, value);
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static void DrawLink(SIX PARAMTERS)
 * Paramters:
 *	GraphWidget graph;
 *	GraphNode *gnode;
 *	int x1, y1;
 *	int x2, y2;
 *
 * Calculate the position and size of the node and then call the DrawLink
 * callback.
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static void DrawLink(graph, gnode, x1, y1, x2, y2)
  GraphWidget graph;
  GraphNode *gnode;
  int x1, y1;
  int x2, y2;
{
  int posx1, posy1;
  int posx2, posy2;
  int size;

  posx1 = x1*graph->graph.scale/FIX_OFFSET;
  posy1 = y1*graph->graph.scale/FIX_OFFSET;
  posx2 = x2*graph->graph.scale/FIX_OFFSET;
  posy2 = y2*graph->graph.scale/FIX_OFFSET;
  size = graph->graph.nodesize*graph->graph.scale/FIX_OFFSET;
  if (((long)gnode->value) > 0)
    (*graph->graph.DrawLink)(graph,
			     graph->graph.gc,
			     posx1, posy1,
			     posx2, posy2,
			     size);
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: void ComputePositions(GraphNode *gnode)
 *
 * Travers the graph and compute the postition of the nodes.
 * 
 * Modifications:
 *      <list mods with name and date>
 */
void ComputePositions(gnode)
  GraphNode *gnode;
{
  GraphNode **nodes;

  gnode->flags |= GNF_TOUCHED;
  if ((nodes = gnode->children)) {
    while(*nodes) {
      if (!((*nodes)->flags&GNF_TOUCHED)) {
	(*nodes)->x = gnode->x+(*nodes)->dx;
	(*nodes)->y = gnode->y+(*nodes)->dy;
	ComputePositions(*nodes);
      }
      ++nodes;
    }
  }
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static void DrawAllNodes(GraphWidget graph, GraphNode *gnode)
 *
 * Traverse the graph and draw all nodes.
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static void DrawAllNodes(graph, gnode)
  GraphWidget graph;
  GraphNode *gnode;
{
  GraphNode **nodes;

  gnode->flags |= GNF_TOUCHED;
  if ((nodes = gnode->children)) {
    while(*nodes) {
      if (!((*nodes)->flags&GNF_TOUCHED)) {
	DrawAllNodes(graph, *nodes);
      }
      ++nodes;
    }
  }
  DrawNode(graph, gnode, gnode->x, gnode->y);
}

/*  */
/**********************************************************************
 * Function: static void DrawAllLinks(GraphWidget graph, GraphNode *gnode)
 * 
 * Traverse the graph and draw all links.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void DrawAllLinks(graph, gnode)
  GraphWidget graph;
  GraphNode *gnode;
{
  GraphNode **nodes;

  gnode->flags |= GNF_TOUCHED;
  if ((nodes = gnode->children)) {
    while(*nodes) {
      if (!((*nodes)->flags&GNF_TOUCHED)) {
	DrawLink(graph,
		 *nodes,
		 gnode->x, gnode->y,
		 (*nodes)->x,  (*nodes)->y);
	DrawAllLinks(graph, *nodes);
      } else {
	DrawLink(graph,
		 *nodes,
		 gnode->x, gnode->y,
		 (*nodes)->x,  (*nodes)->y);
      }
      ++nodes;
    }
  }
}

/*  */
/**********************************************************************
 * Function: int ComputeDistances(SIX PARAMETERS)
 * Parameters:
 *	enum Gravity gravity;
 *	int space_x;
 *	int space_y;
 *	GraphNode *gnode;
 *	int dx, dy;
 * 
 * Calculate the layout of the graph.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int ComputeDistances(gravity, space_x, space_y, gnode, dx, dy)
  enum Gravity gravity;
  int space_x;
  int space_y;
  GraphNode *gnode;
  int dx, dy;
{
  GraphNode **nodes;
  int ndx, ndy;
  
  /* Don't move a node that is already moved, 
     which is the case in DAG:s and graphs */
  if (gnode->flags&GNF_TOUCHED) {
    return 0;
  }

  gnode->dx = dx;
  gnode->dy = dy;
  gnode->flags |= GNF_TOUCHED;
  if ((nodes = gnode->children)) {
    switch(gravity) {
    case LeftRight:
      ndy = 0;
      while(*nodes) {
	ndy += ComputeDistances(gravity, space_x, space_y,
				*nodes, space_x, ndy);
	++nodes;
      }
      return Max(ndy, space_y);
      break;
    case UpDown:
      ndx = 0;
      while(*nodes) {
	ndx += ComputeDistances(gravity, space_x, space_y,
				*nodes, ndx, space_y);
	++nodes;
      }
      return Max(ndx, space_x);
    }
  }
  if (gravity==LeftRight)
    return space_y;
  return space_x;
}

/*  */
/**********************************************************************
 * Function: void DrawGraph(GraphWidget graph)
 * 
 * Modifications:
 *      <list mods with name and date>
 */
void DrawGraph(graph)
  GraphWidget graph;
{
  /* Draw the lincks. */
  Untouch(graph->graph.root);
  DrawAllLinks(graph, graph->graph.root);
  XFlush(XtDisplay(graph));

  /* Draw the nodes. */
  Untouch(graph->graph.root);
  DrawAllNodes(graph, graph->graph.root);
  XFlush(XtDisplay(graph));
}

/*  */
/**********************************************************************
 * Function: void ExtDumpGraph(SEVEN PARAMETERS)
 * 
 * Parameters:
 *	GraphWidget graph
 *	void (*dumpLink)()
 *	void (*dumpNode)()
 *	void (*prolog)()
 *	void (*epilog)()
 *	void (*comment)()
 *	void *context
 *
 * Dump the graph on some external device,  e.g. create a LaTeX or 
 * postscript printout. This is through the use of four call-back function:
 *		dumpLink: Dump a link (see DrawLink).
 *		dumpNode: Dump a node (see DrawNode).
 *		prolog:	  Dump the prolog. May be ommited (NULL).
 *		epilog:   Dump the epilog. May be ommited (NULL).
 *		comment:  Dump a comment string.  May be ommited (NULL).
 * 
 * Modifications:
 *      <list mods with name and date>
 */
void ExtDumpGraph(graph, dumpLink, dumpNode, prolog, epilog, comment, context)
  GraphWidget graph;
  void (*dumpLink)();
  void (*dumpNode)();
  void (*prolog)();
  void (*epilog)();
  void (*comment)();
  void *context;
{
  GC oldgc;
  void (*oDrawNode)();
  void (*oDrawLink)();

  /* Compute the layout. */
  ClearGraphPart((GraphWidget )NULL, graph);

  /* Change the context. */
  oldgc = graph->graph.gc;
  oDrawNode = graph->graph.DrawNode;
  oDrawLink = graph->graph.DrawLink ;
  graph->graph.gc = (GC)context;
  graph->graph.DrawNode = dumpNode;
  graph->graph.DrawLink = dumpLink;


  /* Do the prolog. */
  if (prolog)
    (*prolog)(context, graph->core.width, graph->core.height);

  /* Draw the lincks. */
  if (comment)
    (*comment)(context, "Drawing the links.");
  Untouch(graph->graph.root);
  DrawAllLinks(graph, graph->graph.root);

  /* Draw the nodes. */
  if (comment)
    (*comment)(context, "Drawing the nodes.");
  Untouch(graph->graph.root);
  DrawAllNodes(graph, graph->graph.root);

  /* Do the epilog. */
  if (epilog)
    (*epilog)(context);

  /* Set back context. */
  graph->graph.gc = oldgc;
  graph->graph.DrawNode = oDrawNode;
  graph->graph.DrawLink = oDrawLink;

}

/*  */
/**********************************************************************
 * Function: GraphNode *DefLocateNode(GraphNode *root, int x, int y, int d)
 * 
 * Modifications:
 *      <list mods with name and date>
 */
GraphNode *DefLocateNode(root, x, y, d)
  GraphNode *root;
  int x;
  int y;
  int d;
{
  GraphNode *node = NULL;
  GraphNode **i;

  root->flags |= GNF_TOUCHED;
  if (WithIn(x,  root->x - d/2,  root->x + d/2) &&
      WithIn(y,  root->y - d/2,  root->y + d/2)) {
    node = root;
  } else if ((i = root->children)) {
    while (*i) {
      if (!(i[0]->flags&GNF_TOUCHED) && (node=DefLocateNode(*i, x, y, d)))
	break;
      i++;
    }
  }

  return node;
}


/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: void DefNodeHit(GraphWidget graph, GraphNode *node)
 * 
 * Modifications:
 *      <list mods with name and date>
 */
void DefNodeHit(graph, node)
  GraphWidget graph;
  GraphNode *node;
{
  /* NOOP */
  return;
}

/**********************************************************************
 * Function: void DetectHit(GraphWidget root, XEvent *event)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void DetectHit(root, event)
  GraphWidget root;
  XEvent *event;
{
  GraphNode *node;

  Untouch(root->graph.root);
  if ((node = (*root->graph.LocateNode)
      (root->graph.root,
       event->xbutton.x*FIX_OFFSET/root->graph.scale,
       event->xbutton.y*FIX_OFFSET/root->graph.scale,
       root->graph.nodesize)))
    (*root->graph.NodeHit)(root, node);
}



