/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	graphutils.c
 *
 * SCCSINFO:		@(#)graphutils.c	1.7 5/3/94
 *
 * ORIGINAL AUTHOR(S):  Michael Jansson, 22-Mar-1991
 *
 * MODIFICATIONS:
 *      <list mods with name and date>
 *
 * DESCRIPTION:
 *	This module is an independent extension of the GraphWidget module.
 *	It contains functions that are used to create and manage a dynamic
 *	GNode structure. The GNode structure is an extension of the
 *	GraphNode structure in graph.h.  It can calculate width and height,
 *	create, locate and mark all nodes of the graphs as untouched.
 */

/******************************************************************************
 * INCLUDES:
 ******************************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <X11/StringDefs.h>

#include "graph.h"
#include "graphutils.h"

/******************************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 ******************************************************************************/
#include "f_graphutils.h"

/******************************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 ******************************************************************************/
/* None */

/******************************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 ******************************************************************************/
#include "f_rendering.h"

/******************************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 ******************************************************************************/
/* None. */

/******************************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 ******************************************************************************/
/* Macros. */
#define MapAllChildren(n, i) for (i=0; (n)->node.children[i]; (i)++)

/******************************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 ******************************************************************************/
static int GraphHeightRec P_(( GraphNode *gnode, int height ));
static int GraphWidthRec P_(( GraphNode *gnode, int width ));
static GNode **ReallocNodeArr P_(( GNode *node ));

/******************************************************************************
 * INTERNAL (STATIC) DATA: 
 ******************************************************************************/
/* None */

/*  */
/**********************************************************************
 * Function: void Untouch(GraphNode *gnode)
 * 
 * Clear the GNF_TOUCHED bit in the nodes of the graph.
 * Applies to all graphs based on GraphNode's.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void Untouch(gnode)
  GraphNode *gnode;
{
  GraphNode **nodes;

  if (gnode->flags&GNF_TOUCHED) {
    gnode->flags &= ~GNF_TOUCHED;
    if ((nodes = gnode->children)) {
      while(*nodes) {
	  Untouch(*nodes);
	++nodes;
      }
    }
  }
}

/*  */
/**********************************************************************
 * Function: static int GraphHeightRec(GraphNode *gnode, int height)
 *
 * Recursive function that returns the height of a subgraph.
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static int GraphHeightRec(gnode, height)
  GraphNode *gnode;
  int height;
{
  GraphNode **nodes;

  if (height<gnode->y)
    height = gnode->y;
  if (!(gnode->flags&GNF_TOUCHED)) {
    gnode->flags |= GNF_TOUCHED;
    if ((nodes = gnode->children)) {
      while(*nodes) {
	height = GraphHeightRec(*nodes, height);
	++nodes;
      }
    }
  }
  return height;
}

/*  */
/**********************************************************************
 * Function: static int GraphWidthRec(GraphNode *gnode, int width)
 * 
 * Recursive function that returns the width of a subgraph.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int GraphWidthRec(gnode, width)
  GraphNode *gnode;
  int width;
{
  GraphNode **nodes;

  if (width<gnode->x)
    width = gnode->x;
  if (!(gnode->flags&GNF_TOUCHED)) {
    gnode->flags |= GNF_TOUCHED;
    if ((nodes = gnode->children)) {
      while(*nodes) {
	width = GraphWidthRec(*nodes, width);
	++nodes;
      }
    }
  }
  return width;
}

/*  */
/**********************************************************************
 * Function: int GraphWidth(GraphNode *node)
 * 
 * Returns the width of the graph. Applies to any graph based on the 
 * GraphNode structure, e.g. GNode etc.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int GraphWidth(node)
  GraphNode *node;
{
  int width = 0;

  Untouch(node);
  width = GraphWidthRec(node, width);
  Untouch(node);
  return width;
}

/*  */
/**********************************************************************
 * Function: int GraphHeight(GraphNode *node)
 * 
 * Returns the height of the graph. Applies to any graph
 * based on the GraphNode structurem, e.g. GNode etc.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int GraphHeight(node)
  GraphNode *node;
{
  int height = 0;

  Untouch(node);
  height = GraphHeightRec(node, height);
  Untouch(node);
  return height;
}

/*  */
/**********************************************************************
 * Function: GNode *LocateParent(GNode *root, void *value)
 * 
 * Locate a GNode with a given value.
 *
 * Modifications:
 *      <list mods with name and date>
 */
GNode *LocateParent(root, value)
  GNode *root;
  void *value;
{
  GNode *node = NULL;
  int i;

  if (root) {
    if (root->node.value==value)
      node = root;
    else if (root->node.children) {
      MapAllChildren(root, i) {
	if ((node=LocateParent((GNode *)root->node.children[i], value)))
	  break;
      }
    }
  }
  return node;
}

/*  */
/**********************************************************************
 * Function: static GNode **ReallocNodeArr(GNode *node)
 *
 * Make sure that the node can carry one more child.
 * Only applies to graphs based on GNode's.
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static GNode **ReallocNodeArr(node)
  GNode *node;
{
  GNode **narr;

  if ((node->used+1)>=node->num) {
    narr = (GNode **)malloc((ALLOC_T) sizeof(GNode *)*(node->num+4));
    if (!narr) {
      (void)fprintf(stderr, "ReallocNodeArr: malloc failed - Out of memory!\n");
    } else {
      (void)memcpy((char *)narr,(char *)node->node.children,
	     sizeof(GNode *)*node->used);
      if (node->node.children)
	free((FREEPTR *)node->node.children);
      node->node.children = (GraphNode **)narr;
      node->num += 4;
    }
  } else {
    narr = (GNode **)node->node.children;
  }
  return narr;
}

/*  */
/**********************************************************************
 * Function: GNode *AddChild(GNode *parent, GNode *child)
 *
 * Add a child to a node.  Only applies to graphs based on GNode`s.
 *
 * Modifications:
 *      <list mods with name and date>
 */
GNode *AddChild(parent, child)
  GNode *parent;
  GNode *child;
{
  if (ReallocNodeArr(parent)) {
    parent->node.children[parent->used++] = (GraphNode *)child;
    parent->node.children[parent->used] = NULL;
  } else {
    child = NULL;
  }
  return child;
}

/*  */
/**********************************************************************
 * Function: void FreeTree(GNode *root, void (*freefn)())
 * 
 * Deallocates the memory of a TREE. Assumes that it is a TREE and that
 * any memory hook up to the value field is already freed.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void FreeTree(root, freefn)
  GNode *root;
  void (*freefn)();
{
  int i;

  if (root) {
    for (i=0; i<root->used; i++)
      FreeTree((GNode *)root->node.children[i], freefn);

    if (freefn) {
      (*freefn)(root);
    }
    if (root->node.children)
      free((FREEPTR *)root->node.children);
    free((FREEPTR *)root);
  }
}

/*  */
/**********************************************************************
 * Function: GNode *CreateNode(GNode *parent, void *value)
 * 
 * Allocate and insert a node into the graph.
 *
 * Modifications:
 *      <list mods with name and date>
 */
GNode *CreateNode(parent, value)
  GNode *parent;
  void *value;
{
  GNode *node;

  /* Create and initiate the new node */
  if (!(node = (GNode *)malloc(sizeof(GNode)))) {
    (void)fprintf(stderr, "Failed - Out of memory!\n");
    return FALSE;
  }
  node->used = 0;
  node->num = 0;
  node->node.flags = 0;
  node->node.x = 0;
  node->node.y = 0;
  node->node.dx = 0;
  node->node.dy = 0;
  node->node.value = value;
  node->node.children = NULL;

  if ((parent != NULL) && !AddChild(parent, node)) {
    free((FREEPTR *)node);
    return NULL;
  }

  return node;
}


/*  */
/**********************************************************************
 * Function: void ComputeLayout(FOUR PARAMETERS)
 *
 * Parameters:
 *	GraphNode *root;
 *	enum Gravity gravity;
 *	int space_x;
 *	int space_y;
 * 
 * Modifications:
 *      <list mods with name and date>
 */
void ComputeLayout(root, gravity, space_x, space_y)
  GraphNode *root;
  enum Gravity gravity;
  int space_x;
  int space_y;
{
  Untouch(root);
  (void)ComputeDistances(gravity, space_x, space_y, root, 0, 0);
  Untouch(root);
  ComputePositions(root);
  Untouch(root);
}
