/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	aimstart.c
 *
 * SCCSINFO:		@(#)aimstart.c	1.11 5/20/94
 *
 * ORIGINAL AUTHOR:  Tim Teitenberg, 1990-03-22
 *
 * MODIFICATIONS:
 *      1994-01-01 Martin Sjlin. register_pen_function returns the 
 *                 previous value (more consistent).
 *      1994-01-13 Martin Sjlin. Removed register_pen_function, we
 *                 are using the RCB_interface instead ...
 *      1994-04-15 Martin Sjlin. Added fourth arg to pen_select_fun
 *      1994-05-14 Martin Sjlin. Added -c and 'use_colors' for pens
 *                 and diffs from app defaults.
 *      <list mods with name and date>
 *
 * DESCRIPTION:
 * 	This file contains the 'main' function for Aim, and declares all global
 * 	status variables.
 *
 *********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************
 * void main(unsigned int argc, char **argv)
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "parser.h"
#include "aimtypes.h"

#include "f_aimstart.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE DATA FOUND IN THIS MODULE:
 *********************************************************************/
label sysroot;			/* A label for the system root object */
label userroot;			/* A label for the user object */
reference_structure *refstructs;/* List of all reference structures. */
label loginlbl;			/* Current login profile root */
int use_development_buttons = 0;/* whether to use all buttons on windows */
int nobuttons = 0;		/* whether to have no buttons at all */
int suppress = 0;		/* whether to suppress certain stuff */
int chain = 0;			/* whether to stack windows */
int use_colors = 0;		/* whether to use colors for pen/diff */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_aim_version.h"
#include "f_aimbuildref.h"
#include "f_aimcommand.h"
#include "f_aimsubr.h"
#include "f_blddisplay.h"
#include "f_comhist.h"
#include "f_libxl_version.h"
#include "f_linckshandler.h"
#include "f_ll_version.h"
#include "f_ls_version.h"
#include "f_pen_notify.h"
#include "f_xstuff.h"

/*******************************************************************n
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
/** aimbuildref.c **/
extern actionentry EXPANSION_ACTIONS[];	/* actions for GPD language */
extern char *EXPANSION_SYNTAX[];/* syntax for GPD language */

extern infonode *currentinfonode;	/* in aimstruct.c */

#ifdef COMHISTOFF
extern int global_comhistoff;
#endif

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define USAGE " [-c -D -N -S -C -q [-u username -p password]] database"
#define COPYRIGHT "\
LINCKS, Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist.\n\
LINCKS comes with ABSOLUTELY NO WARRANTY.  It is free software, and you\n\
are welcome to redistribute it under the conditions of the GNU General\n\
Public License (see the file COPYING in the distribution or contact the\n\
Free Software Foundation)."

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static char *arginit P_(( char **vec ));
static int expandlink P_(( char *paramtag, label *lbl, int count, int
                            length ));
static int expandlinks P_(( char *extra, char *tag, int count, int
                             length ));
/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
static int quiet = 0;
static char *password = NULL;
static char *username = NULL;

#ifdef COMHISTOFF
static char *ID = "@(#)xlincks (command history disabled)";
#else
static char *ID = "@(#)xlincks";
#endif

/*  */
/**********************************************************************
 * Function: void main(unsigned int argc, char **argv)
 *
 * The main function for xlincks
 *
 * Modifications:
 *      <list mods with name and date>
 */
int main(argc, argv)
  int argc;
  char **argv;
{
  extern char LL_loginuser[];
  char *db_dir = (char *)NULL;
  int aimpoll = 1;
  int nopoll = 0;

#ifdef COMHISTOFF
  global_comhistoff = 1;
#endif /* COMHISTOFF */

  db_dir = arginit(argv);

  if (!quiet) {
    (void)printf("%s", COPYRIGHT);
    print_aim_version();
    print_uim_version();
    print_liblincks_version();
    print_libshared_version();
  }

  (void)printf("Logging in to database %s.\n", db_dir);
  if (password != NULL && username != NULL) {
     if(NOPROMPT_LOGIN(username, password, db_dir) != SUCCESS) {
       (void) fprintf( stderr, "NOPROMPT_LOGIN: login failure\n");
       exit(1);
     }
  } else if (LOGIN(db_dir) != SUCCESS) {
    (void)fprintf(stderr, "%s: Couldn't login.\n", db_dir);
    exit(1);
  }
  if (initwindows(&argc, argv) != 0)
    exit(-1);

 /*
  * register the pen functions after login, as we can get pen
  * on following object when retrieving them a second time.
  * Needs to run initwindows before to setup X stuff.
  */
  (void) RCB_REGISTERCALLBACK(RCB_PEN_RESOLVE, PEN_resolve, NULL);
  (void) RCB_REGISTERCALLBACK(RCB_PEN_CHECK, PEN_check, NULL);
  (void) RCB_REGISTERCALLBACK(RCB_PEN_POLL, PEN_poll, NULL);

  /* we do not want to have the function in usfuns.c doing any polling */
  nopoll = 0;
  (void) RCB_REGISTERCALLBACK(RCB_PEN_NOPOLL, NULL, (void *)&nopoll);

  /* the last param is to passed to POLL_oob when it is called */
  aimpoll = 1;
  if (!pen_select_func(PEN_GET_HANDLE(), POLL_oob, (void *)&aimpoll, NULL)) {
    (void)fprintf(stderr, "Failed to add pen filehandle to Xt.\n");
    exit(1);
  }

  /* Initialise the command history */
  if (CH_init("xlincks")) {
    (void)fprintf(stderr, "Could not find the command history.\n");
    exit(1);
  }
  /* Update the command history with the LOGIN command */
  if (CH_update("LOGIN", 0, (char *)NULL, (char *)NULL, (char *)NULL,
		(char *)NULL, 0, (label *)NULL, (label *)NULL, (label *)NULL)) {
    (void)fprintf(stderr, "Could not access the command history.\n");
    exit(1);
  }
  if (GSR_GETSYSTEMROOT(&sysroot) != SUCCESS) {
    (void)fprintf(stderr, "%s: Couldn't load.\n", db_dir);
    exit(1);
  }
  if (GLI_GETLINKITEM(&sysroot, "USERS", LL_loginuser, 1, &userroot)) {
    (void)fprintf(stderr, "%s:  Cannot retrieve user root.\n", db_dir);
    (void)fprintf(stderr, "Either user %s unknown or database corrupted.\n",
		  LL_loginuser);
    exit(1);
  }

  /* Initialise the parser module */
  init_parser();
  metagrammar();

  /* Define the GPD language actions and syntax */
  defineactions(EXPANSION_ACTIONS);
  definegrammar(EXPANSION_SYNTAX);

  /* Reset the list of all reference structures */
  refstructs = (reference_structure *)NULL;

  /* Build and map the command menu */
  initcommands();

  /* Build Login Profile. */
  /* -- expand all objects under 'group' LOGIN with GPDname 'field' --
     -- used to display user's choice of objects at login --
     -- by patol -- */

  (void)copylbl(&loginlbl, &userroot);
  if (GLN_GETLINKNAMES(&loginlbl, "LOGIN", expandlinks, (void *)NULL)) {
    blddisplay(buildref(&userroot, "usergpd",(reference_structure *)NULL));
    (void)copylbl(&loginlbl, &sysroot);
    if (GLN_GETLINKNAMES(&loginlbl, "LOGIN", expandlinks, (void *)NULL)) {
      (void)fprintf(stdout, "No LOGIN profile present.\n");
    }
  }

  /* Initialise UI and let the user do his things. */
  currentinfonode = (infonode *)NULL;
  mainloop();
  return (0);			/* all's well.  this is for lint */
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static int expandlink(char *paramtag, label *lbl, int count, int length)
 *
 * A mapfunction to display a ref.struct made by buildref(), where
 * lbl is the link to the object and paramtag is the name of the
 * GPD to expand as. The GPD name is taken as the links field-
 * name.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int expandlink(paramtag, lbl, count, length)
  char *paramtag;
  label *lbl;
  int count, length;
{
  if (count == 0)
    return (0);
  blddisplay(buildref(lbl, paramtag,(reference_structure *)NULL));
  return (0);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static int expandlinks(char *extra, char *tag, int count, int length)
 *
 * A mapfunction called to map over all fieldnames in a linkgroup
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int expandlinks(extra, tag, count, length)
  char *extra, *tag;
  int count, length;
{
  static char tagcp[1000];

  if (count == 0)
    return (0);
  (void)strcpy(tagcp, tag);
  /*                                    access              param*/
  if (GLV_GETLINKVAL(&loginlbl, "LOGIN", tagcp, expandlink, tagcp)) {
    (void)fprintf(stdout, "Could not get link for %s.\n", tagcp);
  }
  return (0);
}

#define DISPLAY_WARN \
"Set the DISPLAY environment variable before starting xlincks.\n\
You cannot use -display.\n"

/*  */
/**********************************************************************
 * Function: static char *arginit(char **vec)
 *
 * processes the command line options.
 *
 * Modifications:
 *      1993-12-25 Martin Sjlin. Added -u <user> -p <password>
 *                 as for t2lincks and the other tools.
 *      1994-05-14 Martin Sjlin. Added -c for using colors 
 *                 for showing diffs/pens ...
 *      <list mods with name and date>
 */
static char *arginit(vec)
  char **vec;
{
  char *db = (char *)NULL;
  char *display = (char *)NULL;
  char *argument;
  char *first = *vec;
  int user = 0, pword = 0;

  if ((display = (char *)getenv("DISPLAY")) == NULL) {
    (void)fprintf(stderr, DISPLAY_WARN);
    exit(1);
  }

  for (vec++; (argument = *vec) != (char *)NULL; vec++) {
    if (*argument == '-') {
      switch(*++argument) {
      case 'u' :
	if( (username = *++vec) == (char *) NULL) {
	  (void) fprintf( stderr,"Missing username after -u\n");
	  exit(1);
	}
	user = 1;
	break;
      case 'p' :
	if ((password = *++vec) == (char *) NULL) {
	  (void) fprintf( stderr,"Missing password after -p\n"); 
	  exit(1);
	}
	pword = 1;
	break;
      case 'q':
	quiet = 1;
	break;
      case 'D':
        use_development_buttons = 1;
	break;
      case 'N':
	nobuttons = 1;
	break;
      case 'S':
	suppress = 1;
	break;
      case 'C':
	chain = 1;
	break;
      case 'c':
        use_colors = 1;
	break;
      default:
	(void)fprintf(stderr, 
		      "Unknown xlincks switch: -%s.  Ignoring it.\n", argument);
      }
    }
    else {
      if (!db) 
        db = argument;
      else {
	(void)fprintf(stderr, "%s%s %s",
		"Multiple command line args which could\n",
		"be used as database directories.  Ignoring", argument);
      }
    }
  } 
  if (user != pword) {
    (void)fprintf(stderr, "You must use both -u and -p if you use either.\n");
    exit(1);
  }

  if (!db)
    if ((db = (char *)getenv("LINCKSDBDIR")) == (char *)NULL) {
      (void)fprintf(stderr, "Usage: %s %s\n", first, USAGE);
      exit(1);
    }
  return db;
}
