/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * MODULE NAME: 	molfile.c
 *
 * SCCSINFO:		@(#)molfile.c	1.11 6/1/94
 *
 * ORIGINAL AUTHOR(S):  ???, 16 Feb, 1987
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *      This file contains UpdateMolFile, ReadMolFile, GetMolfd, MolAccess
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include <rpc/types.h>	/* broken rpc headers */
#include <rpc/xdr.h>

#include "lincks.h"
#include "dbserver.h"
#include "libshared.h"
#include "xconfig.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_molfile.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_dbserv.h"
#include "f_dbserrors.h"
#include "f_rpc.h"
#include "f_xdr.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
/* Global variable declarations */
extern int filetablesize;	/* dbserv.c */
extern char host[];		/* dbserv.c */
extern UID uid;			/* dbserv.c */
extern TABENTRY *filetable;	/* dbserv.c */
extern u_long lincksport;	/* dbserv.c */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
/* none */

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * INTERNAL (STATIC) DATA: 
 *********************************************************************/
/* none */

/*  */
/**********************************************************************
 * Function: int UpdateMolFile(FILEMOL *molentry, LABEL label, int index)
 * 
 * Writes the molecule entry to the molecule file and
 * calls the monitor to update the index file entry
 * Uses Error and ErrorRPC
 *
 * Modifications:
 *      <list mods with name and date>
 */
int UpdateMolFile(molentry, label, index)
  FILEMOL *molentry;
  LABEL label;
  int index;
{
  static char namebuff[MAXPATHLEN];
  static U_ENTRY update = {(LABEL)0, (UID)0, namebuff, 0L};
  int status;
  int st;
	
  /* Lock the molecule file */
  if ((writelockit(filetable[index].molfile.fd)) == -1) {
    Error(ER_FLOCK, "UpdateMolFile: could not lock molecule file");
    return(FAIL);
  }

  /* Seek to the end of the file for appending */
  update.u_filepos = lseek(filetable[index].molfile.fd, (off_t) 0, L_XTND);
  if (update.u_filepos < 0) {
    Error(ER_LSEEK, "UpdateMolFile: could not seek to end of file");
    unlockit(filetable[index].molfile.fd, (off_t)0, (off_t)0);
    return(FAIL);
  }

  /* Write the molecule entry to file */
  if (write(filetable[index].molfile.fd, (char *)molentry, (IOSIZE_T)FILEMOLSIZE) !=
      FILEMOLSIZE) {
    Error(ER_WRITE, "UpdateMolFile: could not write to file");
    unlockit(filetable[index].molfile.fd, (off_t)update.u_filepos, (off_t)0);
    return(FAIL);
  }

  /* Unlock the file */
  unlockit(filetable[index].molfile.fd, (off_t)update.u_filepos, (off_t)0);

  /* Copy info for the monitor update call */
  (void)strcpy(namebuff, filetable[index].molfile.fname);
  update.u_label = label;
  update.u_uid = uid;
  update.u_filename = namebuff;

  /* Call the monitor */
  if ((st=callrpc(host, (u_long)MONITORPROGNO, lincksport, UPDATEENTRY,
	      xdr_update, (char *)&update, xdr_int, (char *)&status))) {
    RPCTrouble(UPDATEENTRY,st);
    return(FAIL);
  }

  /* Check that all went well */
  if (status) {
    ErrorRPC(UPDATEENTRY, status);
    return(FAIL);
  }

  return(SUCCESS);
}

/*  */
/**********************************************************************
 * Function: FILEMOL *ReadMolFile(A_ENTRY *access, int index)
 * 
 * Retrieves the molecule file entry specified by access
 * Uses Error
 *
 * Modifications:
 *      <list mods with name and date>
 */
FILEMOL *ReadMolFile(access, index)
  A_ENTRY *access;
  int index;
{
  static FILEMOL molbuff;

  /* Short cut if new mol file entry */
  if (access->a_filepos < 0) {
    /* Initialize info */
    molbuff.flink_filepos = 0;
    molbuff.flink_size = 0;
    molbuff.blink_filepos = 0;
    molbuff.blink_size = 0;
    molbuff.attr_filepos = 0;
    molbuff.attr_size = 0;
    molbuff.image_filepos = 0;
    molbuff.image_size = 0;
    return(&molbuff);
  }

  /* Seek to file position for molecule entry */
  if (lseek(filetable[index].molfile.fd, (off_t) access->a_filepos, L_SET) !=
      access->a_filepos) {
    Error(ER_LSEEK, "ReadMolFile: could not seek to file pos");
    return(NULL);
  }

  /* Read molecule entry into buffer */
  if (read(filetable[index].molfile.fd, (char *)&molbuff, (IOSIZE_T)FILEMOLSIZE) < 0) {
    Error(ER_READ, "ReadMolFile: could not read molecule file");
    return(NULL);
  }

  /* Return pointer to molecule entry */
  return(&molbuff);
}

/*  */
/**********************************************************************
 * Function: A_ENTRY *MolAccess(LABEL label, u_long mode)
 * 
 * Performs monitor call to retrieve molecule info
 * mode specifies if access is locked or not
 * Uses Error and ErrorRPC
 *
 * Modifications:
 *      <list mods with name and date>
 */
A_ENTRY *MolAccess(label, mode)
  LABEL label;
  u_long mode;
{
  static L_ENTRY largs = {(LABEL)0, (UID)0};
  static A_ENTRY access = {0, (char *)NULL, 0L, 0};
  int st;

  /* Set up info for monitor call */
  largs.l_uid = uid;
  largs.l_label = label;

  /* Reclaim old access info (rather than reuse memory) */
  if (access.a_filename) {
    xdr_free(xdr_string, (char *)&access.a_filename);
    access.a_filename = NULL;
  }

  /* Call monitor */
  if ((st=callrpc(host, (u_long)MONITORPROGNO, lincksport, mode,
		  xdr_entry, (char *)&largs, xdr_access, (char *)&access))) {
    RPCTrouble(mode,st);
    return(NULL);
  }

  /* Check that all went well */
  if (access.a_status) {
    ErrorRPC(ACCESS, access.a_status);
    return(NULL);
  }

  /* Return information structure */
  return(&access);
}

/*  */
/**********************************************************************
 * Function: int GetMolfd(char *molfilename)
 *
 * Retrieves an index to the file information table
 * containing molecule and data file names & descriptors
 * If a file is presently not in the table it is opened
 * and inserted
 * Uses Error
 * 
 * Modifications:
 *      <list mods with name and date>
 */
int GetMolfd(molfilename)
  char *molfilename;
{
  int i;
  char *namep;

  /* Search for matching table molfile name entry */
  for (i=0; i<filetablesize; ++i)
    if ((filetable[i].useflags & USED) &&
	!strcmp(filetable[i].molfile.fname, molfilename))
      break;

  /* Check if valid entry was found */
  if (i<filetablesize) {
    filetable[i].useflags |= ALIVE;
    return(i);
  }

  /* Check where to put new file entry in table */
  for (i=0; i<filetablesize && (filetable[i].useflags & USED); ++i)
    ;

  /* If there wasn't any space left: kick something else out */
  if (i == filetablesize) {
    i = FindTablePlace();
    free(filetable[i].molfile.fname);
    free(filetable[i].datfile.fname);
  }

  /* Open file and place descriptor in table */
  if ((filetable[i].molfile.fd = open(molfilename, 2)) < 0) {
    Error(ER_OPEN, "GetMolfd: could not open %s", 
	  molfilename ? molfilename : "");
    return(FAIL);
  }

  /* Malloc space for file name and copy it into table */
  if (!(filetable[i].molfile.fname = strdup(molfilename))) {
    Error(ER_MALLOC, "GetMolfd: no more memory");
    return(FAIL);
  }
	
  /* Malloc space for file name and copy it into table */
  if (!(filetable[i].datfile.fname = strdup(molfilename))) {
    Error(ER_MALLOC, "GetMolfd: no more memory");
    return(FAIL);
  }
  
  /* Get corresponding data file name */
  if ((namep = strrchr(filetable[i].datfile.fname, '.')) == NULL ||
      strcmp(namep, ".mol") != 0) {
    Error(ER_MOLFILE_FMT, "GetMolfd: bad molecule file name: %s", 
	  molfilename ? molfilename : "");
    return(FAIL);
  }
  (void)strcpy(namep, ".dat");

  /* Open data file and place descriptor in table */
  if ((filetable[i].datfile.fd = open(filetable[i].datfile.fname, 2)) < 0) {
    Error(ER_OPEN, "GetMolfd: could not open %s", 
	  filetable[i].datfile.fname ? filetable[i].datfile.fname : "");
    return(FAIL);
  }

  /* Mark entry used */
  filetable[i].useflags = USED | ALIVE;

  /* Return index */
  return(i);
}

/*  */
/**********************************************************************
 * Function: int FindTablePlace()
 * 
 * Makes room for a new file entry in the file table
 *
 * Modifications:
 *      <list mods with name and date>
 */
int FindTablePlace()
{
  int i, j;

#ifndef HAVE_RANDOM
  /* initialize the random number generator */
  static int initdone = 0;

  if ( !initdone ) {
    srand48((int) getpid());
    initdone = 1;
  }
#endif	/* n HAVE_RANDOM */

  /* Find file place no longer alive */
  for (i=0; i<filetablesize && (filetable[i].useflags & ALIVE); ++i) ;

  /* If all are alive dump somebody at random */
  if (i == filetablesize)
#ifdef HAVE_RANDOM 
    i = random() % filetablesize;
#else
    i = lrand48() % filetablesize;    
#endif /* n HAVE_RANDOM */

  /* Mark all others as dead */
  for (j=0; j<filetablesize; ++j)
    filetable[j].useflags &= ~ALIVE;

  /* Return found value */
  return(i);
}

/*  */
/**********************************************************************
 * Function: int Unlock(LABEL label)
 * 
 * unlocks a locked molecule entry
 *
 * Modifications:
 *      <list mods with name and date>
 */
int Unlock(label)
  LABEL label;
{
  static L_ENTRY largs = {(LABEL)0, (UID)0};
  static int status = 0;
  int st;

  largs.l_uid = uid;
  largs.l_label = label;

  if ((st=callrpc(host, (u_long)MONITORPROGNO, lincksport, UNLOCK,
		  xdr_entry, (char *)&largs, xdr_int, (char *)&status))) {
    RPCTrouble(UNLOCK,st);
    return FAIL;
  }

  if (status) {
    ErrorRPC(UNLOCK, status);
    return FAIL;
  }
  return SUCCESS;
}
