/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * MODULE NAME: 	dbshandler.c
 *
 * SCCSINFO:		@(#)dbshandler.c	1.12 6/6/94
 *
 * ORIGINAL AUTHOR(S):  ???, 4 Mar, 1987
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *      This file contains the signal catcher Handler
 *
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#ifdef  HAVE_USLEEP			/* included in config.h ifndef */
#include <sys/time.h>
#endif /* n HAVE_USLEEP */
#include <netdb.h>			/* for MAXHOSTNAMELEN */
#ifndef MAXHOSTNAMELEN			/* seems like SCO have here */
#include <sys/socket.h>
#endif /* n MAXHOSTNAMELEN */
#ifdef HAVE_GETDTABLESIZE
#include <sys/resource.h>	/* for getrlimit() */
#endif	/* n HAVE_GETDTABLESIZE */

#include "lincks.h"
#include "dbserver.h"
#include "libshared.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_dbshandler.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_dbserfs.h"
#include "f_dbserrors.h"
#include "f_misc.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern int errno;			/* libc */
extern int sys_nerr;			/* libc */
extern char *sys_errlist[];		/* libc */
extern UID uid;				/* dbserv.c */
extern int ITblsize;			/* dbserv.c */
extern IT_ENTRY *InterfaceTable;	/* dbserfs.c */
extern int sigusr2count;		/* dbserv.c */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif /* MAXHOSTNAMELEN */


/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#ifdef DUMP_ITD
static int ITdump P_(( void ));
#endif /* DUMP_ITD */

/*********************************************************************
 * INTERNAL (STATIC) DATA: 
 *********************************************************************/
/* none */

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: RETSIGTYPE Handler( int sig, int code, struct sigcontext *scp)
 * 
 * Takes care of system signals and performs appropriate
 * actions on them
 *
 * Modifications:
 *      Sat Oct  9 19:28:05 1993 Martin Sjlin. Added check for SIGURG
 *                   and if not defined try SIGPOLL instead (SCO system).
 *      Thu Nov 11 10:25:19 1993 Martin Sjlin - SCO uses SIGUSR1 for
 *                   SIGURG (OOB) and SIGPOLL for async I/O.
 *      <list mods with name and date>
 */
RETSIGTYPE Handler( sig )
  int sig;
{
  int savederrno = errno;
  switch (sig)
    {
#ifdef SIGURG
    case SIGURG:
#else
#ifdef SIGUSR1_FOR_OOB
    case SIGUSR1:
#else
    case SIGPOLL:
#endif /* n SIGUSR1_FOR_OOB */
#endif /* n SIGURG */
      Error(ER_SYSERR, "Urgent condition on socket: %s - %s",
	       	       sys_errlist[errno > sys_nerr ? 0 : errno],
#ifdef DUMP_ITD
	       	       ITdump() == SUCCESS
	                   ? "Interface Table dumped"
		 	   : "could not dump Interface Table");
#else
		 	   "Interface tables no longer dumped!");
#endif /* DUMP_ITD */
      exit(1);
      
#ifndef SIGUSR1_FOR_OOB
    case SIGUSR1:
      /* Error(0, "Entering gobble mode");                           */
      /* Gobble();                  -- Gobble mode disabled. /mikoh  */
#ifndef HAVE_RELIABLE_SIGNALS
     if (Signal(SIGUSR1, Handler) == BADSIG) 
        Error(ER_SYSERR, "Handler: could not register signal SIGUSR1.");
#endif /* n HAVE_RELIABLE_SIGNALS */
      break;
#endif /* n SIGUSR1_FOR_OOB */

    case SIGUSR2:
      /* Increment sigusr2 counter */
#ifndef HAVE_RELIABLE_SIGNALS
      if (Signal(SIGUSR2, Handler) == BADSIG) 
	Error(ER_SYSERR, "Handler: could not register signal SIGUSR2.");
#endif /* n HAVE_RELIABLE_SIGNALS */
      sigusr2count++;
      break;

      /* Remove any file locks before exiting */
#ifdef SIGLOST
      case SIGLOST:
#endif
      case SIGPIPE:
      case SIGTERM:
      case SIGHUP:
      case SIGQUIT:
        LogMessage( uid, "dbs: caught signal %d cleaning up locks", sig);
	{
	  int fd;
	  struct rlimit rlp;

#ifdef RLIMIT_NOFILE
	  if ( getrlimit(RLIMIT_NOFILE, &rlp) < 0) {
	    LogMessage( uid,"Handler: failed to get file limits.");
#endif /* RLIMIT_NOFILE */
#ifdef NOFILE
	    rlp.rlim_cur = NOFILE;
#else /* NOFILE */
#ifdef _NFILE
	    rlp.rlim_cur = _NFILE;
#else /* _NFILE */
	    rlp.rlim_cur = 256;    /* just use ANY value ... */
#endif /* n _NFILE */
#endif /* n NOFILE */
#ifdef RLIMIT_NOFILE
	  }
#endif /* RLIMIT_NOFILE */

	  /* try to unlock all fcntl/flock locks */
	  for( fd=0; fd < rlp.rlim_cur; fd++)
            unlockit(fd, (off_t)0, (off_t)0);
	}
       /* Just fall through -  we try remove stuff in */
       /* in monitor, even if it is dead */

    default:
      /* Clean up edit information in monitor */
      if (RemoveAll() != SUCCESS) {
	LogMessage(uid, "dbs: failed to RemoveAll");	
      }
      Error(ER_SYSERR, "dbs crashed due to signal: %d - %s", sig, 
#ifdef DUMP_ITD
	       	       ITdump() == SUCCESS
	                   ? "Interface Table dumped"
		 	   : "could not dump Interface Table");
#else
		 	   "Interface tables no longer dumped!");
#endif /* DUMP_ITD */
      exit(1);
    } /* Switch */
   errno = savederrno;
}

#ifdef DUMP_ITD
/*  */
/**********************************************************************
 * Function: static int ITdump()
 * 
 * Dumps the Interface Table to a new file
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int ITdump()
{
/* 
 * there's no reason to dump ITD files any more.  no one ever reads
 * them, they clutter up things, and the function uses a non-standard
 * function (tempnam).  whacked!
 */
  char prefix[MAXHOSTNAMELEN];
  char *dumpfn;
  FILE *dumpfp;
  int index;

  (void)sprintf(prefix, "ITD%02d", uid);
  if ((dumpfn  = tempnam(".", prefix)) == NULL)
    return(FAIL);

  if ((dumpfp = fopen(dumpfn, "w")) == NULL)
    return(FAIL);

  (void)fprintf(dumpfp,
 "         0        1        2        3        4        5        6        7        8        9");

  for (index=0; index < ITblsize; ++index)
    {
      if ((index % 10)==0) (void)fprintf(dumpfp, "\n%4d: ", index);
      if (InterfaceTable[index].label == 0)
        (void)fprintf(dumpfp, "  - ");
      else
        (void)fprintf(dumpfp, "%3ld", InterfaceTable[index].label);
      switch (InterfaceTable[index].checknumber)
	{
        case IT_DELETED  : (void)fprintf(dumpfp, "   - |"); break;
        case IT_NOT_SENT : (void)fprintf(dumpfp, "  Rel|"); break;
        default:           (void)fprintf(dumpfp, " %4d|", 
					 InterfaceTable[index].checknumber);
	  break;
	}
    }
  (void)fprintf(dumpfp, "|\n");

  (void)fclose(dumpfp);
  return(SUCCESS);
}
#endif  /* DUMP_ITD */
