/*
**	ldbf.h	- LDBF client library
**      Copyright (c) 1995 Nadezhda Compaby, 
**         Software group, Vlad Seriakov
*/

#ifndef _LDBF_H_
#define _LDBF_H_

typedef struct {
   char name[11];
   unsigned len;
   int dec;
   int type;
   int offset;
   struct DB_s *data;
} FIELD;

typedef struct {     /* info about field  */
  char name[11];     /* name of field */
  char type;         /* type of field */
  int len;           /* length of field */
  int dec;           /* number of decimal numbers after dot */
} FIELDINFO;

typedef struct {
  char name[25];         /* name of tag */
  char expression[70];   /* expression of index */
  char filter[50];       /* filter of index */
  int unique;            /* whether index is unique */
  int descending;        /* order */
} TAGINFO;

typedef struct DB_s {
   int sock;            /* socket of server, returned by dbfConnect */ 
   int handle;          /* database handle, returned by dbfOpen */
   char *record;        /* current record */
   int changed;         /* record changed flag */
   long recno;          /* number of current record */
   long reccount;       /* total number of records in the database */
   long record_width;   /* record widht */
   char tag[11];        /* current tag */
   int n_fields;          /* number of fields */
   FIELD *fields;       /* fields info */
} DB;                   

#define O_TTS  64   /* transaction mode: all updates are transactions */

/*
** Pre-declarations for the API library functions
*/

/**************************************************************************
**	Purpose	: Gets text associated with error code
**	Args	: error code 
**	Returns	: text
*/
char *ldbferr_text(int err_code);

/**************************************************************************
**	Purpose	: Tell the server that connection is closed
**	Args	: Server sock 
**	Returns	: none
*/
int ldbfShutdown(int sock);

/**************************************************************************
**	Purpose	: Form a connection to a DBF server
**	Args	: hostname of server
**	Returns	: socket for further use.  -1 on error
**	Notes	: If host == NULL, localhost is used via UNIX domain socket
*/
int ldbfConnect(char *host);

/**************************************************************************
**	Purpose	: Tell the server which database we want to use
**	Args	: Server sock and DB name,
**                  mode may be O_EXCL to open database in exclusive 
**                              O_TTS  to open in transaction mode
**                               in this mode every updates will be flushed
**	Returns	: NULL on error
*/
DB  *ldbfOpen(int sock,char *db,int mode);

/**************************************************************************
**	Purpose	: get tag info of database
**	Args	: DB 
**	Returns	: NULL on error
*/
TAGINFO  *ldbfTaginfo(DB* data);

/**************************************************************************
**	Purpose	: close the database
**	Args	: DB 
**	Returns	: none
*/
void  ldbfClose(DB* data);

/**************************************************************************
**	Purpose	: close all opened databases
**	Args	: connection number
**	Returns	: none
**      Note    : must be exec at the end because it doesn't frees DBs
*/
void  ldbfCloseall(int sock);

/**************************************************************************
**	Purpose	: starts transaction
**	Args	: connection number
**	Returns	: none
*/
void  ldbfBeginTTS(int sock);

/**************************************************************************
**	Purpose	: aborts current transaction
**	Args	: connection number
**	Returns	: none
*/
void  ldbfAbortTTS(int sock);

/**************************************************************************
**	Purpose	: commit current transaction
**	Args	: connection number
**	Returns	: -1 on error
*/
int  ldbfCommitTTS(int sock);

/**************************************************************************
**	Purpose	: Tell the server to create database
**	Args	: Server sock,DB name,field info and tag info
**	Returns	: NULL on error
*/
DB  *ldbfCreate(int sock,char *db,FIELDINFO *finfo,TAGINFO *tinfo);

/**************************************************************************
**	Purpose	: Seeks the data in the database
**	Args	: DB name and sought string
**	Returns	: -1 on error
*/
int ldbfSeek(DB* data,char *str);

/**************************************************************************
**	Purpose	: Set database pointer to specified record
**	Args	: DB name and record number
**	Returns	: -1 on error
*/
int ldbfGoto(DB* data,long rec);

/**************************************************************************
**	Purpose	: Delete current record
**	Args	: DB name 
**	Returns	: -1 on error
*/
int ldbfDelete(DB* data);

/**************************************************************************
**	Purpose	: Recall specified record
**	Args	: DB name and record number
**	Returns	: -1 on error
*/
int ldbfRecall(DB* data,long rec);

/**************************************************************************
**	Purpose	: Skips specified number of records
**	Args	: DB and record number
**	Returns	: -1 on error
*/
int ldbfSkip(DB* data,long rec);

/**************************************************************************
**	Purpose	: Appends one record into database
**	Args	: DB 
**	Returns	: -1 on error
*/
int ldbfAppend(DB* data);

/**************************************************************************
**	Purpose	: Updates current record in the database
**	Args	: DB 
**	Returns	: -1 on error
*/
int ldbfUpdate(DB* data);

/**************************************************************************
**	Purpose	: Locks specified record
**	Args	: DB and record number
**	Returns	: -1 on error
*/
int ldbfLock(DB* data,long rec);

/**************************************************************************
**	Purpose	: Unlocks specified record
**	Args	: DB and record number
**	Returns	: none
*/
void ldbfUnlock(DB* data,long rec);

/**************************************************************************
**	Purpose	: Locks database
**	Args	: DB 
**	Returns	: -1 on error
*/
int ldbfFlock(DB* data);

/**************************************************************************
**	Purpose	: Unlocks database
**	Args	: DB 
**	Returns	: none
*/
void ldbfFunlock(DB* data);

/**************************************************************************
**	Purpose	: Flush database on disk
**	Returns	: none
*/
void ldbfFlush(DB* data);

/**************************************************************************
**	Purpose	: Reindex database (exclusive mode)
**	Returns	: none
*/
void ldbfReindex(DB* data);
/**************************************************************************
**	Purpose	: Packs database (exclusive mode )
**	Returns	: none
*/
void ldbfPack(DB* data);
/**************************************************************************
**	Purpose	: Zaps database (exclusive mode )
**	Returns	: none
*/
void ldbfZap(DB* data);

/**************************************************************************
**	Purpose	: Get field via name
**	Args	: DB and field name
**	Returns	: NULL on error
*/
FIELD *ldbfField( DB *data, char *field_name );

/**************************************************************************
**	Purpose	: Get field via number
**	Args	: DB and field number
**	Returns	: NULL on error
*/
FIELD *ldbfField_j( DB *data, int j_field );

/**************************************************************************
**	Purpose	: Get field number by its name
**	Args	: DB and field name
**	Returns	: field number,0 on error
*/
int ldbfField_number( DB *data, char *field_name );

/**************************************************************************
**	Returns	: pointer on field in record
*/
char *ldbfPtr( FIELD *field );

/**************************************************************************
**	Purpose	: Assign the field value
**	Args	: FIELD and double value
**	Returns	: none
*/
void ldbfAssign_double( FIELD *field, double d_value );

/**************************************************************************
**	Purpose	: Get double value from field
**	Args	: FIELD
**	Returns	: value of field
*/
double ldbfDouble( FIELD *field );

/**************************************************************************
**	Purpose	: Empty all field of current record
**	Args	: DB 
**	Returns	: none
*/
void ldbfBlank( DB *data );

/**************************************************************************
**	Purpose	: Assign char value to field
**	Args	: FIELD and value
**	Returns	: none
*/
void ldbfAssign_char( FIELD *field, int chr );

/**************************************************************************
**	Purpose	: Get char vlaue from field
**	Args	: FIELD
**	Returns	: char value
*/
int ldbfChar( FIELD *field );

/**************************************************************************
**	Purpose	: Assign string value to field
**	Args	: FIELD and string value
**	Returns	: none
*/
void ldbfAssign( FIELD *field, char *str );

/**************************************************************************
**	Purpose	: Get string value form field
**	Args	: FIELD
**	Returns	: pointer to static area with string
*/
char *ldbfStr( FIELD *field );

/**************************************************************************
**	Purpose	: Assign long value to field
**	Args	: FIELD and long value
**	Returns	: none
*/
void ldbfAssign_long( FIELD *field, long l_value );

/**************************************************************************
**	Purpose	: Get long value form field
**	Args	: FIELD
**	Returns	: field value in long
*/
long ldbfLong( FIELD *field );

/**************************************************************************
**	Purpose	: Assign field value from another field
**	Args	: FIELD and FIELD
**	Returns	: none
*/
void ldbfAssign_field( FIELD *field_to, FIELD *field_from );

/**************************************************************************
**	Purpose	: Get information about field 
**	Args	: FIELD
*/
int ldbfDecimals( FIELD *field );
unsigned ldbfLen( FIELD *field );
char *ldbfName( FIELD *field );
int ldbfType( FIELD *field );

/**************************************************************************
  This is misc function that may easy your programming
*/

/* Delete leading and trailing trash characters from s */
char *trim(char *s);

/* Returns number of words in s delimited with delim */
int wordcnt(char *s,char *delim);

/* Returns position of n'th word */
int wordpos(int n, char *s,char *delim);

/* Return n'th word from s and puts it into buf */
char *getword(int n,char *s,char *delim,char *buf);

/*  Converts a string to a double */
double ldbfatod( char *str, int len_str );

/* Converts a string to a integer */
int ldbfatoi( char *str, int len_str );

/* Converts a string to a long */
long ldbfatol( char *str, int len_str );

/* - formats a double to a string */
void ldbfdtoa( double doub_val, char *out_buffer, int len, int dec);

/*  Converts a long to a string.  Fill with '0's rather than blanks if
    'num' is less than zero.*/
void  ldbfltoa( long l_val, char *ptr, int num);

/*  - From CCYYMMDD to CCYY.MM.DD
   Ex.        c4encode( to, from, "CCYY.MM.DD", "CCYYMMDD" ) ;*/
void  ldbfencode( char *to, char *from, char *t_to, char *t_from);

/* Make all chars in the string in lower case */
void ldbflower( char *str );

/* Make all chars in the string in upper case */
void  ldbfupper( char *str );

/* store long date into string format */
void ldbfdate_assign( char *date_ptr, long ldate );

/* returns character day of week */
char *ldbfcdow( char *date_ptr );

/* returns character month */
char *ldbfcmonth( char *date_ptr );

/* returns day from date string */
int ldbfday( char *date_ptr );

/* returns day of week in number */
int ldbfdow( char *date_ptr );

/* formats date using picture, assumes date_ptr in dbf date format
   CCYYMMDD */
void ldbfdate_format( char *date_ptr, char *result, char *picture );

/* converts date_data using picture in dbf format */ 
void ldbfdate_init( char *date_ptr, char *date_data, char *picture );

/* returns date in long, assumes date_ptr in dbf format */
long ldbfdate( char *date_ptr );

/* return month from date string */
int ldbfmonth( char *date_ptr );

/* return year from date string */
int ldbfyear( char *year_ptr );

/* store curent time */
void ldbftime_now( char *time_data );

/* store current date */
void ldbftoday( char *date_ptr );

#endif
