/*
 * The IPC and multiplexed I/O routines for Berkeley flavors of Unix
 */

#include <stdio.h>
#ifndef MAIN
#include <fcntl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include "ipc.h"

static char pty_name[12];
static int fdin_mask, num_fds;
static struct timeval time_out = {1L, 0L};
static struct timeval *tp;

void
ipc_init(tty_fd, cmd_fd)
int tty_fd, cmd_fd;
{
	int fdout, fdin, fdex;

	fdout = 0;
	fdex = 0;
	fdin_mask = 1;			/* the keyboard */
	num_fds = 1;
	if (tty_fd != -1) {		/* the TTY */
		fdin_mask |= (1 << tty_fd);
		num_fds = tty_fd +1;
	}
	if (cmd_fd != -1) {		/* the shell script */
		fdin_mask |= (1 << cmd_fd);
		if (cmd_fd > tty_fd)
			num_fds = cmd_fd +1;
		tp = &time_out;
	}
	else
		tp = (struct timeval *) NULL;
	return;
}

int
ipc_poll(tty_fd, cmd_fd)
int tty_fd, cmd_fd;
{
	int ret_code, fdout, fdin, fdex;

	ret_code = 0;
	fdin = fdin_mask;
	select(num_fds, &fdin, &fdout, &fdex, tp);

	if (fdin & 1)
		ret_code |= KEY_READY;

	if (tty_fd != -1 && (fdin & (1 << tty_fd)))
		ret_code |= TTY_READY;

	if (cmd_fd != -1 && (fdin & (1 << cmd_fd)))
		ret_code |= CMD_READY;

	return(ret_code);
}

/*
 * Update the file descriptors for poll()/select() that might have changed.
 */

void
ipc_update(tty_fd, cmd_fd)
int tty_fd, cmd_fd;
{
	fdin_mask = 1;
	num_fds = 1;
	if (tty_fd != -1) {
		fdin_mask |= (1 << tty_fd);
		num_fds = tty_fd +1;
	}
	if (cmd_fd != -1) {
		fdin_mask |= (1 << cmd_fd);
		if (cmd_fd > tty_fd)
			num_fds = cmd_fd +1;
		tp = &time_out;
	}
	else
		tp = (struct timeval *) NULL;
	return;
}

/*
 * Open an IPC "channel"
 */

int
ipc_open()
{
	int fd, i, j;
	char *p, *strcpy();
	struct stat sbuf;
	static char letters[] = "pqrs";
	static char hex_num[] = "0123456789abcdef";

	strcpy(pty_name, "/dev/ptyxx");

	for (i=0; i<4; i++) {
		pty_name[8] = letters[i];
		for (j=0; j<16; j++) {
			pty_name[9] = hex_num[j];
					/* if it doesn't exist, quit */
			if (stat(pty_name, &sbuf) < 0)
				return(-1);

			if ((fd = open(pty_name, O_RDWR)) >= 0)
				return(fd);
		}
	}
	return(-1);
}

/*
 * Close an IPC channel
 */

int
ipc_close(fd)
int fd;
{
	return(close(fd));
}

/*
 * Return a unique "key" to pass to the child process 
 */

/* ARGSUSED */
char *
ipc_key(fd)
int fd;
{
	static char tty_name[12];
	char *strcpy();

	strcpy(tty_name, pty_name);
	tty_name[5] = 't';
	return(tty_name);
}
#endif /* MAIN */

/*
 * Attach to an existing IPC channel
 */

int
ipc_attach(dev, mode)
char *dev;
int mode;
{
	return(open(dev, mode));
}

/*
 * Detach from an IPC channel
 */

int
ipc_detach(fd)
int fd;
{
	return(close(fd));
}

/*
 * Read from an IPC channel
 */

int
ipc_read(fd, buf, n)
int fd;
char *buf;
int n;
{
	if (read(fd, buf, n) != n)
		return(1);
	return(0);
}

/*
 * Write to an IPC channel
 */

int
ipc_write(fd, buf, n)
int fd;
char *buf;
int n;
{
	if (write(fd, buf, n) != n)
		return(1);
	return(0);
}
