/*+-----------------------------------------------------------------------
	eculine.c -- ECU line handler
	wht@n4hgf.Mt-Park.GA.US

  Defined functions:
	display_hw_flow_config()
	lCLOCAL(flag)
	lRTSCTS_control(flag)
	lbreak()
	lclear_xmtr_xoff()
	lclose()
	lclose_failed(sig)
	ldcdwatch(flag)
	ldcdwatch_str(flagstr)
	ldraino(inflush_flag)
	lflash_dtr()
	lflush(flush_type)
	lget_xon_xoff(ixon,ixoff)
	lgetc_timeout(msec)
	lgetc_xmtr()
	lgets_timeout(lrwt)
	llookfor(lookfor,msecs,echo_flag)
	lnew_baud_rate(new_baud)
	lopen()
	lopen_failed(sig)
	lputc(lchar)
	lputc_paced(pace_msec,lchar)
	lputs(string)
	lputs_paced(pace_msec,string)
	lquiet(msecs,echo_flag)
	lrdchk_xmtr()
	lreset_ksr()
	lset_baud_rate(ioctl_flag)
	lset_parity(ioctl_flag)
	linst_err_text(lerr)
	lxon_xoff(flag)
	lzero_length_read_detected()
	process_xmtr_rcvd_char(rchar,echo)
	set_xon_xoff_by_arg(arg)
	valid_baud_rate(baud)
	xon_status()

Note: On the Sun or with -DUSE_TERMIOS, termios is used in lieu of
termio.  See ecu.h and search for USE_TERMIOS for vague illumination.

------------------------------------------------------------------------*/
/*+:EDITS:*/
/*:09-10-1992-13:58-wht@n4hgf-ECU release 3.20 */
/*:08-22-1992-15:38-wht@n4hgf-ECU release 3.20 BETA */
/*:08-21-1992-13:39-wht@n4hgf-rewire direct/modem device use */
/*:08-19-1992-14:03-wht@n4hgf-2nd open in lflash_dtr needs O_NDELAY on SVR4 */
/*:08-16-1992-02:52-wht@n4hgf-some vendors use SCO naming but ttyaa/ttyaA */
/*:07-27-1992-05:49-wht@n4hgf-lopen SCO modem line to make CLOCAL effective */
/*:07-24-1992-14:25-wht@n4hgf-more valiant efforts on lclose failure */
/*:07-19-1992-09:19-wht@n4hgf-lopen validation for char special */
/*:05-11-1992-16:35-wht@gyro-speed up lflash_DTR on sun */
/*:05-04-1992-04:43-kortcs!tim-fix EAGAIN on line open with SVR4 */
/*:04-27-1992-19:57-wht@n4hgf-add LINST_ECUUNGETTY error text */
/*:04-24-1992-21:59-wht@n4hgf-more SCO tty name normalizing */
/*:04-24-1992-21:44-wht@n4hgf-add SCO_direct_tty */
/*:04-12-1992-06:31-wht@gyro-was not canceling alarm on lopen error */
/*:03-29-1992-16:27-wht@n4hgf-put three second timer on lopen */
/*:03-17-1992-18:26-wht@n4hgf-optimize parameter 1 to select() */
/*:12-12-1991-05:14-wht@n4hgf-lgetc_timeout can now return a null character */
/*:11-26-1991-19:47-wht@n4hgf-add ldcdwatch_str */
/*:11-11-1991-22:28-wht@n4hgf-ldcdwatch and lCLOCAL code */
/*:11-11-1991-14:38-wht@n4hgf-lzero_length_read_detected code */
/*:09-01-1991-14:18-wht@n4hgf2-on sun, use termios and improve ldraino */
/*:09-01-1991-02:51-wht@n4hgf2-sun CRTSCTS turn on bug fixed */
/*:08-25-1991-14:39-wht@n4hgf-SVR4 port thanks to aega84!lh */
/*:08-11-1991-18:06-wht@n4hgf-SCO_TTY_NAMING considerations */
/*:08-06-1991-14:18-wht@n4hgf-baud rates below 300 get two stop bits */
/*:07-29-1991-01:55-wht@n4hgf-remove unused externs */
/*:07-25-1991-12:56-wht@n4hgf-ECU release 3.10 */
/*:07-17-1991-07:04-wht@n4hgf-avoid SCO UNIX nap bug */
/*:04-09-1991-16:11-wht@n4hgf-use B0 in lflash_DTR */
/*:02-07-1991-01:00-wht@n4hgf-fix code in for lclose retry on remote XOFF */
/*:01-29-1991-14:54-wht@n4hgf-put code in for lclose retry on remote XOFF */
/*:01-25-1991-05:57-wht@n4hgf-cringe - lflush was flushing console not line */
/*:01-09-1991-22:31-wht@n4hgf-ISC port */
/*:01-09-1991-21:26-wht@n4hgf-don't prototype nap() (ISC port) */
/*:08-14-1990-20:40-wht@n4hgf-ecu3.00-flush old edit history */

#define DECLARE_LINEVARS_PUBLIC
#include "ecu.h"
#include "ecukey.h"
#include "termecu.h"
#include <setjmp.h>
#include <pwd.h>

extern int errno;
#if defined(SVR4)
# include <sys/termios.h>
# include <sys/termiox.h>
int hx_flag;
#endif

void lzero_length_read_detected();
void lCLOCAL();

int lgetc_count;
char lopen_err_str[64] = "";
int zero_length_read_detected = 0;
int dcdwatch_set = 0;	/* if true, ldcdwatch() has been called */
static jmp_buf _insurance_jmpbuf;

/*
 * with SCO UNIX, nap doesn't work as advertized; param MUST be > granularity
 * or nap will return immediately; not a problem with XENIX
 */
#define LPUTS_NAP_COUNT	(min((hzmsec * 2),20L))

/*+-------------------------------------------------------------------------
	process_xmtr_rcvd_char(rchar,echo) - feed xmtr-rcvd char to rcvr code

echo: 0 no echo
      1 echo literally
      2 "make printable"
--------------------------------------------------------------------------*/
void
process_xmtr_rcvd_char(rchar,echo)
uint rchar;
register int echo;
{
	if(process_rcvd_char(rchar))
		return;

	if(echo == 1)
	{
		if(rchar == NL)
			fputc(CRET,se);
		fputc(rchar,se);
		if(rchar != CRET)
			plogc(rchar);
	}
	else if(echo == 2)
	{
	char *make_char_graphic();
		pputs(make_char_graphic(rchar,0));
		if(rchar == 0x0A)
			pputs("\n");
	}

}	/* end of process_xmtr_rcvd_char */

/*+-------------------------------------------------------------------------
	lgetc_xmtr() -- xmtr version of get char from line
also called by rcvr code when lgetc_buf empty and vmin == 1

zero_length_read_detected is a public that will set if the
DCD watcher is turned on and DCD is lost
--------------------------------------------------------------------------*/
uchar
lgetc_xmtr()
{
	int itmp;
	uchar char_rtnd;

READ_AGAIN:
	if((itmp = read(shm->Liofd,(char *)&char_rtnd,1)) < 1)
	{
		if(!itmp)
		{
			if(shm->Ldcdwatch)
			{
				lzero_length_read_detected();
				return(0);
			}
			errno = EIO;	/* for termecu processing */
			termecu(TERMECU_LINE_READ_ERROR);
		}
		if(errno == EINTR)			/* if signal interrupted, ... */
		{
			if(sigint)
				return(0);
			goto READ_AGAIN;
		}
		termecu(TERMECU_LINE_READ_ERROR);
	}
	shm->rcvd_chars++;
	shm->rcvd_chars_this_connect++;
	if(shm->Lparity)
		char_rtnd &= 0x7F;
	return(char_rtnd);

}	/* end of lgetc_xmtr */

/*+-------------------------------------------------------------------------
	lrdchk_xmtr() -- rdchk(shm->Liofd) for xmtr
--------------------------------------------------------------------------*/
int
lrdchk_xmtr()
{
	return(rdchk(shm->Liofd));
}	/* end of lrdchk_xmtr */

/*+-------------------------------------------------------------------------
	char *lgets_timeout(LRWT *) - may be called by xmtr only

to1 and to2 are unsigned long values in milliseconds (not
currently supported well under BSD4); to1 is the time to wait
for the first character, to2 the time to wait for subsequent
characters.

if raw_flag 0,     non-printables are stripped from beginning
                   and end of received characters (i.e., modem
                   response reads); NULs discarded, parity stripped
if raw_flag 1,     full raw read buffer returned

0x80 in raw_flag indicates console interrupts should be enabled.
if interrupt thus detected, the procedure returns "!Interrupted"
without reseting variable 'interrupt'

buffer is address to read chars into

bufsize is buffer max size (allowing room for terminating null)
which should be at least 2 if raw_size includes 0x80 bit,
else at least 12 characters if 0x80 omitted.

count is a int which, at return, receives the actual count read

zero_length_read_detected is a public that will set if the
DCD watcher is turned on and DCD is lost

--------------------------------------------------------------------------*/
char *
lgets_timeout(lrwt)
LRWT *lrwt;
{
/**************************/
#if !defined(WORKING_SELECT)
/**************************/

	register actual_count = 0;
	register char *cptr = lrwt->buffer;
	register echo_flag = lrwt->echo_flag;
	int max_count = lrwt->bufsize;
	char *rtn_val;
	int timeout_counter;
	int qc1;
	int qc2;
	int raw_mode = lrwt->raw_flag & 0x0F;
	int check_sigint = (lrwt->raw_flag & 0x80);
	int old_ttymode = get_ttymode();	/* save original tty mode */
	int delim_len;
	long quantum;
	long ltmp;

	delim_len = (lrwt->delim) ? strlen(lrwt->delim) : 0;

	if((shm->Lbaud < 300) && lrwt->to2)
		if(lrwt->to2 < 300L) lrwt->to2 = 300L;
	else if((shm->Lbaud < 1200) && lrwt->to2)
		if(lrwt->to2 < 200L) lrwt->to2 = 100L;

/* shortest interval */
	ltmp = (lrwt->to1 < lrwt->to2) ? lrwt->to1 : lrwt->to2;

/* calculate wait quantum */
	quantum = ltmp / 10L;				/* try for ten ticks */

#if defined(M_I386)
	if(quantum < 40L)
		quantum = 40L;
#else
	if(quantum < 20L)
		quantum = 20L;
#endif
	qc1 = lrwt->to1 / quantum;
	if(!qc1) qc1 = 1L;
	qc2 = lrwt->to2 / quantum;
	if(!qc2) qc2 = 1L;

/* perform the lrtw function using nap() and rdchk()
   input: qc1 is first nap count (for first charcters) 
          qc2 is 2nd nap count (for subsequent characters) 
          quantum is the nap period in milliseconds
          cptr is char* to receive read string
          max_count is max number of characters incl null
          lrwt->raw_flag as described above

  output: lrwt->count is actual count of return result
          lrwt->buffer is return read buffer
*/
	max_count--;				/* leave room for null */

	if(check_sigint)
		ttymode(2);				/* let console interrupt long timeouts */

	timeout_counter = qc1;		/* first timeout */ 
	*cptr = 0;					/* init result string */
	while(timeout_counter--)
	{
		Nap(quantum);

		if(check_sigint && sigint)
			goto INTERRUPTED;

		while(lrdchk_xmtr())
		{
			zero_length_read_detected = 0;
			*cptr = lgetc_xmtr();

			if(zero_length_read_detected)
				goto BOTTOM;

			if(check_sigint && sigint)
				goto INTERRUPTED;

			if(*cptr == 0)
				continue;

			process_xmtr_rcvd_char(*cptr,echo_flag);

			if(!raw_mode && (*cptr == CRET))
					continue;

			*++cptr = 0;
			if(++actual_count == 1)
			{
				if(!lrwt->to2)
					break;
				timeout_counter = qc2;
			}

			if(--max_count == 0)
				goto BOTTOM;

			if(delim_len && (actual_count >= delim_len) &&
					!strncmp(lrwt->delim,cptr - delim_len,delim_len))
				goto BOTTOM;
		}
	}

/********************************/
#else /* do have WORKING_SELECT */
/********************************/
/* --- use select --- */
	register actual_count = 0;
	register char *cptr = lrwt->buffer;
	register max_count = lrwt->bufsize;
	register raw_mode = lrwt->raw_flag & 0x0F;
	register echo_flag = lrwt->echo_flag;
	int check_sigint = (lrwt->raw_flag & 0x80);
	int old_ttymode = get_ttymode();	/* save original tty mode */
	int fdmask;
	int delim_len;
	struct timeval tval;
	char *rtn_val;

	delim_len = (lrwt->delim) ? strlen(lrwt->delim) : 0;

	if((shm->Lbaud < 300) && lrwt->to2)
		if(lrwt->to2 < 300L) lrwt->to2 = 300L;
	else if((shm->Lbaud < 1200) && lrwt->to2)
		if(lrwt->to2 < 200L) lrwt->to2 = 100L;


/* perform the lrtw function

  output: lrwt->count is actual count of return result
          lrwt->buffer is return read buffer
*/
	max_count--;				/* leave room for null */

	if(check_sigint)
		ttymode(2);				/* let console interrupt long timeouts */

	*cptr = 0;					/* init result string */
	while(1)
	{
		if(check_sigint && sigint)
			goto INTERRUPTED;

		errno = 0;
		fdmask = 1 << shm->Liofd; /* Liofd will always be <= 31, right? */
		if(actual_count)
		{
			tval.tv_sec = lrwt->to2 / 1000L;
			tval.tv_usec = (lrwt->to2 % 1000L) * 1000L;
		}
		else
		{
			tval.tv_sec = lrwt->to1 / 1000L;
			tval.tv_usec = (lrwt->to1 % 1000L) * 1000L;
		}
		if(select(shm->Liofd + 1,&fdmask,NULL,NULL,&tval) != 1)
		{
			if(errno == EINTR)
				continue;
			break;
		}

		while(rdchk(shm->Liofd))
		{
			zero_length_read_detected = 0;
			*cptr = lgetc_xmtr();

			if(zero_length_read_detected)
				goto BOTTOM;

			if(check_sigint && sigint)
				goto INTERRUPTED;

			if(*cptr == 0)
				continue;

			process_xmtr_rcvd_char(*cptr,!!echo_flag);

			if(!raw_mode && (*cptr == CRET))
					continue;

			*++cptr = 0;
			actual_count++;

			if(--max_count == 0)
				goto BOTTOM;

			if(delim_len && (actual_count >= delim_len) &&
					!strncmp(lrwt->delim,cptr - delim_len,delim_len))
				goto BOTTOM;
		}
		if(!lrwt->to2)
			break;
	}

#endif	/* WORKING_SELECT */

/********* common post processing for select() / no select() ************/
BOTTOM:
	if(check_sigint)
		ttymode(old_ttymode);
	if(raw_mode)
	{
		lrwt->count = actual_count;
		return(lrwt->buffer);
	}
	cptr = lrwt->buffer;
	while(((*cptr > 0) && (*cptr < SPACE)) || (*cptr >= DEL))
		cptr++;
	rtn_val = cptr;
	actual_count = 0;
	while(((*cptr &= 0x7F) >= SPACE) && (*cptr < DEL))
	{
		cptr++;
		actual_count++;
	}
	*cptr = 0;
	strcpy(lrwt->buffer,rtn_val);
	lrwt->count = actual_count;
	return(lrwt->buffer);

INTERRUPTED:
	ttymode(old_ttymode);
	strcpy(lrwt->buffer,"!Interrupted");
	lrwt->count = strlen(lrwt->buffer);
	return((char *)0);

}	/* end of lgets_timeout */

/*+-------------------------------------------------------------------------
	lgetc_timeout(msec) - may be called by xmtr only

 reads one character from line unless msec passes with no receipt.
 return char if received, else -1 if timeout
--------------------------------------------------------------------------*/
int
lgetc_timeout(msec)
long msec;
{
	uchar rtn_char;
#if !defined(WORKING_SELECT)
	long timeout;

	timeout = msec;
	while(!lrdchk_xmtr())
	{
		if(sigint)
			return(-1);
		if((timeout -= Nap(hzmsec)) <= 0)
			return(-1);
	}

#else

	int fdmask;
	struct timeval tval;

	tval.tv_sec = msec / 1000L;
	tval.tv_usec = (msec % 1000L) * 1000L;
	fdmask = 1 << shm->Liofd; /* Liofd will always be <= 31, right? */
	if(select(shm->Liofd + 1,&fdmask,NULL,NULL,&tval) < 1)
		return(-1);
	if(!lrdchk_xmtr())
		return(-1);
	if(sigint)
		return(-1);
#endif

	rtn_char = lgetc_xmtr();
	return(rtn_char);

}	/* end of lgetc_timeout */

/*+-------------------------------------------------------------------------
	llookfor(lookfor,msecs,echo_flag)
return 1 if successful, else 0 if no match
echo_flag: 0 no echo
           1 echo literally
           2 "make printable"
--------------------------------------------------------------------------*/
int
llookfor(lookfor,msecs,echo_flag)
char *lookfor;
ulong msecs;
int echo_flag;
{
	register lookfor_len = strlen(lookfor);
	register lchar;
	char *lastfew = (char *)malloc(lookfor_len);
	int success_flag = 0;
	int old_ttymode = get_ttymode();

	if(!lastfew)
	{
		pputs("memory exhausted\n");
		return(0);
	}

	ttymode(2);

	memset(lastfew,0,lookfor_len);
	while((lchar = lgetc_timeout(msecs)) >= 0)
	{
		if(!lchar)		/* skip nulls */
			continue;
		process_xmtr_rcvd_char(lchar,echo_flag);
		mem_cpy(lastfew,lastfew + 1,lookfor_len - 1);
		*(lastfew + lookfor_len - 1) = lchar;
		if(!strncmp(lastfew,lookfor,lookfor_len))
		{
			success_flag = 1;
			break;
		}
	}
	free(lastfew);
	ttymode(old_ttymode);
	return(success_flag);
}	/* end of llookfor */

/*+-------------------------------------------------------------------------
	lquiet(msecs,echo_flag)
--------------------------------------------------------------------------*/
void
lquiet(msecs,echo_flag)
ulong msecs;
int echo_flag;
{
	register lchar;
	int old_ttymode = get_ttymode();

	ttymode(2);
	while((lchar = lgetc_timeout(msecs)) >= 0)
	{
		if(sigint)	/* if interrupt, return */
			break;
		if(!lchar)		/* skip nulls */
			continue;
		process_xmtr_rcvd_char(lchar,!!echo_flag);
	}
	ttymode(old_ttymode);

}	/* end of lquiet */

/*+-------------------------------------------------------------------------
	lflush(flush_type) -- flush line driver input &/or output buffers

0 == input buffer
1 == output buffer
2 == both buffers
--------------------------------------------------------------------------*/
void
lflush(flush_type)
int flush_type;
{
	switch(flush_type)
	{
		case 0:
			lgetc_count = 0;
			ioctl(shm->Liofd,TCFLSH,(char *)0); 
			break;
		case 1:
			ioctl(shm->Liofd,TCFLSH,(char *)1); 
			break;
		case 2:
			lgetc_count = 0;
			ioctl(shm->Liofd,TCFLSH,(char *)2); 
			break;
	}
}	/* end of lflush */

/*+-------------------------------------------------------------------------
	lreset_ksr()

  This procedure restores the termio for the
  comm line to the values in Ltermio
--------------------------------------------------------------------------*/
void
lreset_ksr()
{
	ioctl(shm->Liofd,TCSETA,(char *)Ltermio);

}	/* end of lreset_ksr */

/*+-------------------------------------------------------------------------
	ldraino(inflush_flag) - wait for output to drain

If inflush_flag is set, also flush input after output drains
--------------------------------------------------------------------------*/
void
ldraino(inflush_flag)
int inflush_flag;
{
#if defined(sun)
	int retries = 50;
	int outq_count;
	int old_outq_count = 0;

	do {
		ioctl(shm->Liofd,TIOCOUTQ,&outq_count);
		if(!outq_count)
			break;
		if(old_outq_count == outq_count) /* don't hang if flow control lock */
			retries--;
		old_outq_count = outq_count;
		Nap(50);
	} while(outq_count && retries);
	if(inflush_flag)
		ioctl(shm->Liofd,TCFLSH,TCIFLUSH);
#else
	ioctl(shm->Liofd,(inflush_flag) ? TCSETAF : TCSETAW,(char *)Ltermio);
#endif

}	/* end of ldraino */

/*+-----------------------------------------------------------------------
	lputc(lchar) -- write lchar to comm line
------------------------------------------------------------------------*/
void
lputc(lchar)
char lchar;
{
	while(write(shm->Liofd,&lchar,1) < 0)
	{
		if(errno == EINTR)
			continue;
		pperror("lputc write error");
		termecu(TERMECU_XMTR_WRITE_ERROR);
	}
	shm->xmit_chars++;
	shm->xmit_chars_this_connect++;
}	/* end of lputc */

/*+-----------------------------------------------------------------------
	lputc_paced(pace_msec,lchar) -- write lchar to comm line
  with time between each character 
------------------------------------------------------------------------*/
void
lputc_paced(pace_msec,lchar)
register pace_msec;
char lchar;
{

	lputc(lchar);	
	Nap((long)(pace_msec ? pace_msec : LPUTS_NAP_COUNT));

}	/* end of lputc_paced */

/*+-----------------------------------------------------------------------
	lputs(string) -- write string to comm line
------------------------------------------------------------------------*/
void
lputs(string)
register char *string;
{
	while(*string)
		lputc(*string++);
}

/*+-----------------------------------------------------------------------
	lputs_paced(pace_msec,string) -- write string to comm line
  with time between each character 
------------------------------------------------------------------------*/
void
lputs_paced(pace_msec,string)
register pace_msec;
register char *string;
{
	while(*string)
		lputc_paced(pace_msec,*string++);

}	/* end of lputs_paced */

/*+-------------------------------------------------------------------------
	valid_baud_rate(baud) -- returns (positive) baud rate selector
or -1 if invalid baud rate
--------------------------------------------------------------------------*/
valid_baud_rate(baud)
uint baud;
{
	switch(baud)
	{
		case 110: return(B110);
		case 300: return(B300);
		case 600: return(B600);
		case 1200: return(B1200);
		case 2400: return(B2400);
		case 4800: return(B4800);
		case 9600: return(B9600);
		case 19200: return(EXTA);
		case 38400: return(EXTB);
		default: return(-1);
	}

}	/* end of valid_baud_rate */

/*+-----------------------------------------------------------------------
	lset_baud_rate(ioctl_flag)

  If 'ioctl_flag' is set, then perform ioctl call
  is executed after setting baud rate
------------------------------------------------------------------------*/
lset_baud_rate(ioctl_flag)
int ioctl_flag;
{
	int baud_selector = valid_baud_rate(shm->Lbaud);

	if(shm->Liofd < 0)
		return(0);

	if(baud_selector == -1)
		baud_selector = valid_baud_rate(shm->Lbaud = DEFAULT_BAUD_RATE);

	shm->Lmodem_already_init = 0;
	Ltermio->c_cflag &= ~CBAUD;
	Ltermio->c_cflag |= baud_selector;

	if(baud_selector < B300)
		Ltermio->c_cflag |= CSTOPB;
	else
		Ltermio->c_cflag &= ~CSTOPB;

	if(ioctl_flag)
		 ioctl(shm->Liofd,TCSETA,(char *)Ltermio);
	return(0);

}	/* end of lset_baud_rate */

/*+-------------------------------------------------------------------------
	display_hw_flow_config() - display hardware flow control configuration
--------------------------------------------------------------------------*/
#if defined(HW_FLOW_CONTROL) /* see ecu.h */
void
display_hw_flow_config()
{
#undef ____HANDLED
#ifdef RTSFLOW /* SCO */
#define ____HANDLED
	pprintf("RTSFLOW %s CTSFLOW %s",
		(Ltermio->c_cflag & RTSFLOW) ? "on" : "off",
		(Ltermio->c_cflag & CTSFLOW) ? "on" : "off");
#ifdef CRTSFL
	pprintf(" CRTSFL %s",
		(Ltermio->c_cflag & CRTSFL) ? "on" : "off");
#endif /* CRTSFL */
	pprintf("\n");
#endif /* RTSFLOW */

#ifdef RTSXOFF /* SVR4 */
#define ____HANDLED
	pprintf("RTSXOFF %s CTSXON %s\n",
		(hx_flag & RTSXOFF) ? "on" : "off",
		(hx_flag & CTSXON) ? "on" : "off");
#endif /* RTSXOFF */

#if defined(CRTSCTS) /* sun */
#define ____HANDLED
	pprintf(" CRTSCTS %s\n",
		(Ltermio->c_cflag & CRTSCTS) ? "on" : "off");
#endif /* sun */

#ifndef ____HANDLED
porting_attention_needed_here; /* HW_FLOW_CONTROL but no recognized flags */
/*
 * if you are reading this because of a compilation error, you may wish to
 * go ahead and grep for 'RTSFLOW' and 'display_hw_flow_control' to find other
 * hardware control dependencies (like in lRTSCTS_control() below).  This is
 * the only rigrous test in ECU for making sure that if HW_FLOW_CONTROL is on
 * we know what to do about it.
 */
#endif /* ____HANDLED */

}	/* end of display_hw_flow_config */
#endif /* HW_FLOW_CONTROL */

/*+-------------------------------------------------------------------------
	lRTSCTS_control(flag)
--------------------------------------------------------------------------*/
void
lRTSCTS_control(flag)
int flag;
{
#ifdef RTSXOFF /* SVR4 */
	struct termiox flowctrl;

	ioctl(shm->Liofd, TCGETX, &flowctrl);
	switch(flag)
	{
		case 0:
			flowctrl.x_hflag &= ~(RTSXOFF | CTSXON);
			Ltermio->c_iflag |= (IXOFF);
			break;

		case 1:
			flowctrl.x_hflag |= CTSXON;
			flowctrl.x_hflag &= ~RTSXOFF;
			Ltermio->c_iflag &= ~(IXON | IXOFF | IXANY);
			break;
		case 2:
			flowctrl.x_hflag |= RTSXOFF;
			flowctrl.x_hflag &= ~CTSXON;
			Ltermio->c_iflag &= ~(IXON | IXOFF | IXANY);
			break;
		case 3:
			flowctrl.x_hflag |= (RTSXOFF | CTSXON);
			Ltermio->c_iflag &= ~(IXON | IXOFF | IXANY);
			break;
	}
	shm->Lxonxoff = Ltermio->c_iflag & (IXON|IXOFF);
	ioctl(shm->Liofd,TCSETA,(char *)Ltermio);
	ioctl(shm->Liofd, TCSETX, &flowctrl);
	hx_flag = flowctrl.x_hflag;
#else /* !SVR4 */
#if defined(RTSFLOW)	/* only SCO */
	switch(flag & 3)
	{
		case 0:
			Ltermio->c_iflag |= (IXOFF);
			Ltermio->c_cflag &= ~(RTSFLOW | CTSFLOW);
			break;

		case 1:
			Ltermio->c_iflag &= ~(IXON | IXOFF | IXANY);
			Ltermio->c_cflag |= CTSFLOW;
			Ltermio->c_cflag &= ~RTSFLOW;
			break;

		case 2:
			Ltermio->c_iflag &= ~(IXON | IXOFF | IXANY);
			Ltermio->c_cflag |= RTSFLOW;
			Ltermio->c_cflag &= ~CTSFLOW;
			break;

		case 3:
			Ltermio->c_iflag &= ~(IXON | IXOFF | IXANY);
			Ltermio->c_cflag |= (RTSFLOW | CTSFLOW);
			break;
	}
#if defined(CRTSFL)
	if(flag & 4)
	{
		Ltermio->c_iflag &= ~(IXON | IXOFF | IXANY | RTSFLOW | CTSFLOW);
		Ltermio->c_cflag |= CRTSFL;
	}
#endif
	shm->Lxonxoff = Ltermio->c_iflag & (IXON|IXOFF);
	ioctl(shm->Liofd,TCSETA,(char *)Ltermio);

#else
#if defined(CRTSCTS) /* sun */
	switch(flag)
	{
		case 0:
			Ltermio->c_iflag |= (IXOFF);
			Ltermio->c_cflag &= ~(CRTSCTS);
			break;

		default:
			Ltermio->c_iflag &= ~(IXON | IXOFF | IXANY);
			Ltermio->c_cflag |= CRTSCTS;
			break;

	}
	shm->Lxonxoff = Ltermio->c_iflag & (IXON|IXOFF);
	ioctl(shm->Liofd,TCSETA,(char *)Ltermio);

#endif /* sun */
#endif /* RTSFLOW */
#endif /* SVR4 */
}	/* end of lRTSCTS_control */

/*+-------------------------------------------------------------------------
	lnew_baud_rate(new_baud)
--------------------------------------------------------------------------*/
int
lnew_baud_rate(new_baud)
uint new_baud;
{
	if(valid_baud_rate(new_baud) < 0)
		return(-1);
	if(shm->Lbaud != new_baud)
		shm->Lmodem_already_init = 0;
	shm->Lbaud = new_baud;
	lset_baud_rate(1);
	return(0);
}	/* end of lnew_baud_rate */

/*+-----------------------------------------------------------------------
	lset_parity(ioctl_flag)

  If 'ioctl_flag' is set, then perform ioctl call
  is executed after setting parity
------------------------------------------------------------------------*/
void
lset_parity(ioctl_flag)
int ioctl_flag;
{
	if(shm->Liofd < 0)
		return;

	Ltermio->c_cflag &= ~(CS8 | PARENB | PARODD);
	switch(to_lower(shm->Lparity))
	{
		case 'e':
			Ltermio->c_cflag |= CS7 | PARENB;
			Ltermio->c_iflag |= ISTRIP;
			break;
		case 'o':
			Ltermio->c_cflag |= CS7 | PARENB | PARODD;
			Ltermio->c_iflag |= ISTRIP;
			break;
		default:
			ff(se,"invalid parity: '%c' ... defaulting to no parity\r\n",
				to_lower(shm->Lparity));
		case 'n':
			shm->Lparity = 0;
		case 0:
			Ltermio->c_cflag |= CS8;
			Ltermio->c_iflag &= ~(ISTRIP);
			shm->Lparity = 0;
			break;
	}			

	if(ioctl_flag)
		ioctl(shm->Liofd,TCSETA,(char *)Ltermio);

}	/* end of lset_parity */

/*+-------------------------------------------------------------------------
	lclear_xmtr_xoff()
--------------------------------------------------------------------------*/
void
lclear_xmtr_xoff()
{
	ioctl(shm->Liofd,TCXONC,(char *)1); /* restart xmtr output */
}	/* end of lclear_xmtr_xoff */

/*+-------------------------------------------------------------------------
	lbreak()
--------------------------------------------------------------------------*/
void
lbreak()
{
	ioctl(shm->Liofd,TCSBRK,(char *)0);
}	/* end of lbreak */

/*+-------------------------------------------------------------------------
	linst_err_text(lerr)
--------------------------------------------------------------------------*/
char *
linst_err_text(lerr)
int lerr;
{
	static char lerr_s80[80];
	extern uchar last_ugstat;
	char *ugstat_text();

	if(lopen_err_str[0])
		return(lopen_err_str);

	switch(lerr)
	{
		case LINST_INVALID: return("invalid line name");
		case LINST_UNKPID: return("unknown pid is using line");
		case LINST_LCKERR: return("error creating lock file");
		case LINST_NODEV: return("line does not exist");
		case LINST_ALREADY: return("line already open!?");
		case LINST_OPNFAIL:
			sprintf(lerr_s80,"open error (%-.60s)",
				errno_text(errno));
			return(lerr_s80);
		case LINST_ENABLED: return("line enabled for incoming login");
		case LINST_ENABLED_IN_USE: return("line in use by incoming login");
		case LINST_DIALOUT_IN_USE: return("line in use by another dial out");
		case LINST_NOPTY: return("ptys not supported");
		case LINST_WEGOTIT: return("line is locked by this process");
		case LINST_ECUUNGETTY:
			sprintf(lerr_s80,"ecuungetty error (%-.45s)",
				ugstat_text(last_ugstat));
			return(lerr_s80);
		case LINST_ECUUNGETTY2:
			return("ecuungetty execution error");
		case LINST_NOTCHR:
			return("not a character special device");
	}
	if(lerr > 0)
		sprintf(lerr_s80,"pid %d using line",lerr);
	else
		sprintf(lerr_s80,"unknown line error %d",lerr);
	return(lerr_s80);
}	/* end of linst_err_text */

/*+-------------------------------------------------------------------------
	lopen_failed(sig) - see lopen() below
--------------------------------------------------------------------------*/
void
lopen_failed(sig)
int sig;
{
	if(sig != SIGALRM)
		ff(se,"error %d in lopen_failed: tell wht@n4hgf\r\n",sig);
	longjmp(_insurance_jmpbuf,1);

}	/* end of lopen_failed */

/*+----------------------------------------------------------------------
	lopen()
returns negative LINST_ codes if failure else positive pid using line
else 0 if successful open
------------------------------------------------------------------------*/
int
lopen()
{
	int itmp;
	struct stat ttystat;

#ifdef SHARE_DEBUG
	char s80[80];
	sprintf(s80,"lopen Liofd=%d Lline=%s line_lock_status=%d",
	    shm->Liofd,shm->Lline,
	    line_lock_status(shm->Lline));
	ecu_log_event((int)xmtr_pid,s80);
#endif

	/*
	 * system independent checks
	 */
	if(shm->Liofd >= 0)
		return(LINST_ALREADY);
	if(!strcmp(shm->Lline,"/dev/tty"))
		return(LINST_INVALID);
	if(stat(shm->Lline,&ttystat) < 0)
	{
		if(errno == ENOENT)
			return(LINST_NODEV);
		return(LINST_OPNFAIL);
	}
	if((ttystat.st_mode & S_IFMT) != S_IFCHR)
		return(LINST_NOTCHR);
	if(ulindex(shm->Lline,"pty") > -1)
		return(LINST_NOPTY);

	/*
	 * lock the tty 
	 */
	if((itmp = lock_tty(shm->Lline)) && (itmp != LINST_WEGOTIT))
		return(itmp);

	/*
	 * if appropriate, make sure we have ungetty'd the line
	 */
#if defined(USE_ECUUNGETTY)
	ungetty_return_all_but(shm->Lline);
	if(!in_ungetty_list(shm->Lline))
	{
		if(itmp = ungetty_get_line(shm->Lline))
		{
			sprintf(lopen_err_str,"ecuungetty error - %s",linst_err_text(itmp));
			unlock_tty(shm->Lline);
			return(itmp);
		}
	}
#endif

	/*
	 * rarely an open will hang despite our wisdom and prayer
	 */
	if(setjmp(_insurance_jmpbuf))
	{
		alarm(0);
		signal(SIGALRM,SIG_IGN);
		errno = EIO;
		sprintf(lopen_err_str,"open error - %s",errno_text(errno));
		unlock_tty(shm->Lline);
		return(LINST_OPNFAIL);
	}

	/*
	 * open the tty using non-blocking I/O to bypass DCD wait
	 * handle EAGAIN for SVR4 per kortcs!tim
	 */

	for (itmp=0; ; ++itmp)
	{
		signal(SIGALRM,lopen_failed);
#ifdef sun
		alarm(10);
#else
		alarm(5);
#endif
		shm->Liofd = open(shm->Lline,O_RDWR | O_NDELAY,0666);
		alarm(0);
		signal(SIGALRM,SIG_IGN);
		if(shm->Liofd >= 0)
			break;
		if((itmp < 5) && (errno == EAGAIN))
		{
			(void)signal(SIGALRM,SIG_DFL);
			alarm(0);
			sleep(2);
			continue;
		}
		if (errno == EACCES)
		{
			struct passwd *pw = getpwuid(ttystat.st_uid);
			endpwent();
			if(pw)
			{
				sprintf(lopen_err_str,
					"cannot open line owned by %s (mode=%3o)",
					pw->pw_name,ttystat.st_mode & 0777);
			}
			else
			{
				sprintf(lopen_err_str,
					"open error - try chmod +rw %s",shm->Lline);
			}
		}
		else 
			sprintf(lopen_err_str,"open error - %s",errno_text(errno));
		unlock_tty(shm->Lline);
		return(LINST_OPNFAIL);
	}

	/*
	 * turn off non-blocking I/O and set initial termio, including CLOCAL
	 */
	fcntl(shm->Liofd,F_GETFL,&itmp);
	itmp &= ~O_NDELAY;
	fcntl(shm->Liofd,F_SETFL,itmp);

	ioctl(shm->Liofd,TCGETA,(char *) Ltermio);
	Ltermio->c_iflag = (IGNPAR | IGNBRK | shm->Lxonxoff);
	Ltermio->c_oflag = 0;
	Ltermio->c_cflag |= (CLOCAL | CREAD | HUPCL);
	Ltermio->c_lflag = 0;
	Ltermio->c_cc[VMIN]   = 1;
	Ltermio->c_cc[VTIME]  = 1;
	lset_baud_rate(0);		/* do not perform ioctl */
	lset_parity(1);			/* do perform ioctl */

#if defined(SVR4)
	hx_flag = 0;			/* hardware flow control "memory" */
#endif

	lopen_err_str[0] = 0;	/* no error this time! */
	return(0);

}	/* end of lopen */

/*+-------------------------------------------------------------------------
	lclose_failed(sig) - see lclose() below
--------------------------------------------------------------------------*/
void
lclose_failed(sig)
int sig;
{
	if(sig != SIGALRM)
		ff(se,"error %d in lclose_failed: tell wht@n4hgf\r\n",sig);
	longjmp(_insurance_jmpbuf,1);

}	/* end of lclose_failed */

/*+-----------------------------------------------------------------------
	lclose() - close the line

The FAS driver and others hang on a close until all output for a line
has drained.  Sometimes during a hangup, a noisy XOFF can be received.
Other changeces for failure include a DCE which drops CTS and leaves
it off, locking the line up if there is output waiting to go out.
To make sure the line is actually closed in these situations, a SIGLARM
handler is used.
------------------------------------------------------------------------*/
void
lclose()
{
	struct termio ttio;
	int attempt2 = 0;

#ifdef SHARE_DEBUG
	char s80[80];
	sprintf(s80,"lclose Liofd=%d Lline=%s line_lock_status=%d",
		shm->Liofd,shm->Lline,
		line_lock_status(shm->Lline));
	ecu_log_event((int)xmtr_pid,s80);
#endif

	if(shm->Liofd < 0)
		return;
ATTEMPT: /* endless loop because we cannot get out anyway unless success */
	signal(SIGALRM,lclose_failed);
#ifdef sun
	alarm(10);
#else
	alarm(5);
#endif
	if(setjmp(_insurance_jmpbuf))
	{	/* close failed */
		signal(SIGALRM,SIG_IGN);
		alarm(0);
		ff(se,"\r\nclose failed (remote XOFF?) ... retrying close\r\n");
		lclear_xmtr_xoff();
		ttio = *Ltermio;
		ttio.c_iflag &= ~(IXON | IXOFF);
		ttio.c_cflag &= (CSIZE | CSTOPB | CREAD | PARENB | PARODD);
		ioctl(shm->Liofd,TCSETA,(char *)&ttio);
		lflush(2);
		attempt2 = 1;
		goto ATTEMPT;
	}
	if(!attempt2)
	{
		lclear_xmtr_xoff();
		ldraino(1);
	}
	lCLOCAL(1);
	close(shm->Liofd);
	signal(SIGALRM,SIG_IGN);
	alarm(0);
	unlock_tty(shm->Lline);
	shm->Lmodem_already_init = 0;
	shm->Lconnected = 0;
	shm->Liofd = -1;

}	/* end of lclose */

/*+-------------------------------------------------------------------------
	lflash_dtr() - flash DTR

DTR is lowered for 300 msec and raised again.  After raising,
we pause a while for a possibly slow DCE to rereap it's fecal material

expects:  Ltermio - current termio status of line
          shm->Liofd - current line fd
          shm->Lline - /dev/ttyxx name of line

On SunOS and SVR4, an open/close of the line is required to get DTR back
up. SVR3 does not seem to need this (ISC asy, SCO sio, Uwe Doering's FAS)
but we do it anyway
--------------------------------------------------------------------------*/
void
lflash_dtr()
{
#undef NEED_REOPEN
#if defined(sun) || defined(SVR4)
#define NEED_REOPEN
	int tempfd;
#endif
	struct termio b0t;

	/*
	 * copy termio but CBAUD to B0
	 */
	b0t = *Ltermio;
	b0t.c_cflag &= ~CBAUD;	/* B0 */

	/*
	 * drop DTR for a while
	 */
	ioctl(shm->Liofd,TCSETA,(char *)&b0t);		/* drop DTR */

	/*
	 * DTR will not come back on some systems without open/close line
	 */
#ifdef NEED_REOPEN
	if ((tempfd = open(shm->Lline, O_NDELAY | O_RDWR, 0666)) != -1)
		close(tempfd);
	else
	{
		int save_errno = errno;
		char s128[128];
		sprintf(s128,"FLASH DTR line reopen failed (%.80s)",
			errno_text(errno));
		ecu_log_event(shm->xmtr_pid,s128);
		pputs(s128);
		pputs("\n");
		errno = save_errno;
		termecu(TERMECU_LINE_OPEN_ERROR);
	}
#else
	/*
	 * ensure DTR low for 300 msec
	 * (the tempfd open/close takes plenty long enough)
	 */
	Nap(300L);
#endif

	/*
	 * reestablish baud rate 
	 * (raise DTR if the open/close line did not do it)
	 */
	ioctl(shm->Liofd,TCSETA,(char *)Ltermio);	/* raise DTR */
	Nap(300L);	/* nap to give a lazy DCE some time */

#undef NEED_REOPEN

}	/* end of lflash_dtr */

/*+-------------------------------------------------------------------------
	lxon_xoff(flag)
IXON specifies whether or not we respond to xon/xoff characters
IXOFF specifies whether or not we generate XON/XOFF characters
--------------------------------------------------------------------------*/
void
lxon_xoff(flag)
int flag;
{
	if(flag & IXON)
		Ltermio->c_iflag |= IXON;
	else
		Ltermio->c_iflag &= ~IXON;

	if(flag & IXOFF)
		Ltermio->c_iflag |= IXOFF;
	else
		Ltermio->c_iflag &= ~IXOFF;
	shm->Lxonxoff = Ltermio->c_iflag & (IXON|IXOFF);
	ioctl(shm->Liofd,TCSETA,(char *)Ltermio);
#if 0
	Nap(400L);
#endif

}	/* end of lflash_dtr */

/*+-------------------------------------------------------------------------
	lget_xon_xoff(ixon,ixoff)
--------------------------------------------------------------------------*/
void
lget_xon_xoff(ixon,ixoff)
int *ixon;
int *ixoff;
{
	*ixon  = Ltermio->c_iflag & IXON;
	*ixoff = Ltermio->c_iflag & IXOFF;
}	/* end of lget_xon_xoff */

/*+-------------------------------------------------------------------------
	set_xon_xoff_by_arg(arg)
--------------------------------------------------------------------------*/
int
set_xon_xoff_by_arg(arg)
char *arg;
{
	if(ulcmpb(arg,"on") < 0)
		shm->Lxonxoff = IXON | IXOFF;
	else if(ulcmpb(arg,"off") < 0)
		shm->Lxonxoff = 0;
	else if(ulcmpb(arg,"out") < 0)
		shm->Lxonxoff = IXON;
	else if(ulcmpb(arg,"in") < 0)
		shm->Lxonxoff = IXOFF;
	else
		return(-1);

	Ltermio->c_iflag &= ~(IXON|IXOFF);
	Ltermio->c_iflag |= shm->Lxonxoff;
	ioctl(shm->Liofd,TCSETA,(char *)Ltermio);
	return(0);

}	/* end of set_xon_xoff_by_arg */

/*+-------------------------------------------------------------------------
	xon_status()
--------------------------------------------------------------------------*/
char *
xon_status()
{
	switch(shm->Lxonxoff)
	{
		case 0            : return("off");
		case IXON         : return("in off, out on");
		case        IXOFF : return("in on, out off");
		case IXON | IXOFF : return("on");
	}
	return("logic error");
}	/* end of xon_status */

/*+-------------------------------------------------------------------------
	lCLOCAL(flag) - set line CLOCAL state

flag == 0: turn off CLOCAL to catch DCD loss
     == 1: turn on CLOCAL to ignore modem signals

does not let CLOCAL be turned off if not Lconnected
also resets global zero_length_read_detected
--------------------------------------------------------------------------*/
void
lCLOCAL(flag)
int flag;
{

	if(flag)
		Ltermio->c_cflag |= CLOCAL;
	else if(shm->Lconnected)
		Ltermio->c_cflag &= ~CLOCAL;
	else
		Ltermio->c_cflag |= CLOCAL;

	zero_length_read_detected = 0;
	lreset_ksr();	/* do the ioctl */

#ifdef DEBUG_CLOCAL
	{
		char s128[128];
		sprintf(s128,"lCLOCAL(%d) connected=%c CLOCAL set %o",
			flag,shm->Lconnected?'y':'n',Ltermio->c_cflag & CLOCAL?1:0);
		ecu_log_event((int)xmtr_pid,s128);
		pprintf("%s\n",s128);
	}
#endif

}	/* end of lCLOCAL */

/*+-------------------------------------------------------------------------
	ldcdwatch(flag) - set DCD watcher state
--------------------------------------------------------------------------*/
void
ldcdwatch(flag)
int flag;
{
	shm->Ldcdwatch = flag;
	dcdwatch_set = 1;
	lCLOCAL(!flag);
}	/* end of ldcdwatch */

/*+-------------------------------------------------------------------------
	ldcdwatch_str(flagstr) - string version of ldcdwatch

return 0 if successful or -1 if bad flagstr
--------------------------------------------------------------------------*/
int
ldcdwatch_str(flagstr)
char *flagstr;
{
	static STR_CLASSIFY sc[] = {
		{ "1",			1,DCDW_ON },
		{ "yes",		1,DCDW_ON },
		{ "on",			2,DCDW_ON },
		{ "0",			1,DCDW_ON },
		{ "no",			1,DCDW_OFF },
		{ "off",		3,DCDW_OFF },
		{ "terminate",	1,DCDW_TERMINATE },
		{ (char *)0,	0,-1 }
	};
	int token;

	if((token = str_classify(sc,flagstr)) < 0)
		return(-1);

	ldcdwatch(token);
	return(0);
		
}	/* end of ldcdwatch_str */

/*+-------------------------------------------------------------------------
	lzero_length_read_detected() - read from line returned zero length

This must mean CLOCAL was off and DCD is/went low.  We do different things
depending in the xmtr and rcvr process

If we return, the condition has ben handled and reads may be retried
safely or other appropriate operations performed; otherwise ECU is
terminated.
--------------------------------------------------------------------------*/
void
lzero_length_read_detected()
{

	zero_length_read_detected = 1;
	if(getpid() == xmtr_pid)	/* if we are in the transmitter */
	{
#ifdef DEBUG_CLOCAL
		ecu_log_event((int)xmtr_pid,"lzero xmtr");
		pprintf("lzero xmtr\n");
#endif
		if(shm->Lconnected)
		{
			extern ulong colors_current;
			ulong colors_at_entry = colors_current;

			fputs("\r\n",se);
			setcolor(colors_notify);
			fputs("[connection terminated]",se);
			setcolor(colors_at_entry);
			fputs("\r\n",se);
			DCE_now_on_hook();	/* does a lCLOCAL(1); */
		}
		else
			lCLOCAL(1);
		Nap(1000L);
		lflush(2);

		if(shm->Ldcdwatch == DCDW_TERMINATE)
			termecu(0);
		shmx_unpause_rcvr();
	}
	else	/* we are in the receiver */
	{
#ifdef DEBUG_CLOCAL
		ecu_log_event((int)xmtr_pid,"lzero rcvr");
		pprintf("lzero rcvr\n");
#endif
		/*
		 * make line "safe" to read from immediately;
		 * however, if CLOCAL was set and we get a zero length read,
		 * we are in some kind of unknown trouble
		 */
		if(Ltermio->c_cflag & CLOCAL)	/* zero len read with CLOCAL? */
		{								/* then die ECU */
			errno = EIO;
			termecu(TERMECU_LINE_READ_ERROR);
		}
		lCLOCAL(1);
		shmr_notify_xmtr_of_DCD_loss();
		pause();		/* wait for unpause */
	}

}	/* end of lzero_length_read_detected */

/* vi: set tabstop=4 shiftwidth=4: */
/* end of eculine.c */
