/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994,1995 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@rz.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: /sda4/users/nau/src/pcb/RCS/printpanner.c,v 2.3 1994/10/29 17:27:15 nau Exp $";

/* print dialog panner routines
 * only available for X11R5 and later
 * all routines are defined 'empty' for X11R4
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <X11/Xlib.h>

#include "global.h"

#include "data.h"
#include "menu.h"
#include "misc.h"
#include "printpanner.h"

/* ---------------------------------------------------------------------------
 * some local types
 */
typedef struct					/* media description */
{
	String		Name;			/* name and size (in mil) */
	Dimension	Width,
				Height;
} MediaType, *MediaTypePtr;

/* ---------------------------------------------------------------------------
 * some local identifiers
 */
static	int				Index = -1;		/* media index */
static	MediaType		Media[] = {		/* first one is user defined */
	{ "user defined", 0, 0 },
	{ "a3", 11690, 16530 },
	{ "a4", 8260, 11690 },
	{ "a5", 5830, 8260 },
	{ "letter", 8500, 11000 },
	{ "tabloit", 11000, 17000 },
	{ "ledger", 17000, 11000 },
	{ "legal", 8500, 14000 },
	{ "executive", 7500, 10000 },
	{ NULL, 0, 0 }};
static	PopupEntryType	MediaMenuEntries[ENTRIES(Media)+1];
static	PopupMenuType	MediaMenu =
	{ "media", NULL, MediaMenuEntries, NULL, NULL, NULL };
static	MenuButtonType	MediaMenuButton =
	{ "media", "Media", &MediaMenu, NULL };

/* ---------------------------------------------------------------------------
 * a local prototype for both cases
 */
static	void	InitDefaultMedia(void);

/***************************************************************************** 
 * routines for X11R5 an newer
 *****************************************************************************/
#if XlibSpecificationRelease >= 5

#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Panner.h>

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	void	CB_Media(Widget, XtPointer, XtPointer);
static	Widget	PrintPannerCreateMediaMenu(Widget, Widget, Widget);

/* ---------------------------------------------------------------------------
 * some local identifiers
 */
static	Widget			Panner,			/* the panner widget */
						SizeField;		/* label to display the media size */
static	Dimension		Width, Height;	/* layout size */

/* ---------------------------------------------------------------------------
 * updates the panner widget
 */
void PrintPannerUpdate(float Scale, Boolean RotateFlag)
{
	Dimension	width, height;

		/* scale layout size */
	width = (Dimension) ((float) Width *Scale);
	height = (Dimension) ((float) Height *Scale);

		/* update slider */
	XtVaSetValues(Panner,
		XtNsliderWidth, RotateFlag ? height : width,
		XtNsliderHeight, RotateFlag ? width : height,
		NULL);
}

/* ---------------------------------------------------------------------------
 * callback for media size selection (menu)
 */
static void CB_Media(Widget W, XtPointer ClientData, XtPointer CallData)
{
	MediaTypePtr	media = &Media[(int) ClientData];
	char			size[128];
	Dimension		mediawidth,
					mediaheight;

		/* set visible size of media, use global identifiers because
		 * the size will also be used in printdialog.c
		 */
	Settings.MediaWidth = media->Width;
	Settings.MediaHeight = media->Height;
	mediawidth = Settings.MediaWidth -2*Settings.MediaMarginLeft;
	mediaheight = Settings.MediaHeight -2*Settings.MediaMarginBottom;

	sprintf(size, "media size:     %4.2f\" x %4.2f\"\n"
				  "without margin: %4.2f\" x %4.2f\"\n",
		(float) Settings.MediaWidth /1000.0,
		(float) Settings.MediaHeight /1000.0,
		(float) mediawidth /1000.0,
		(float) mediaheight /1000.0);

		/* update widgets */
	XtVaSetValues(SizeField, XtNlabel, size, NULL);
	XtVaSetValues(Panner,
		XtNcanvasHeight, mediaheight,
		XtNcanvasWidth, mediawidth,
		NULL);
}

/* ---------------------------------------------------------------------------
 * creates 'media size' menu
 */
static Widget PrintPannerCreateMediaMenu(Widget Parent, Widget Top, Widget Left)
{
	int		i;

		/* copy media description to menuentry structure */
	for (i = 0; i < ENTRIES(Media); i++)
	{
		MediaMenuEntries[i].Name = Media[i].Name;
		MediaMenuEntries[i].Label = Media[i].Name;
		MediaMenuEntries[i].Callback = CB_Media;
		MediaMenuEntries[i].ClientData = (char *) i;
		MediaMenuEntries[i].W = NULL;
	}
	return(InitMenuButton(Parent, &MediaMenuButton, Top, Left));
}

/* ---------------------------------------------------------------------------
 * creates a print command dialog
 */
Widget PrintPannerCreate(Widget Parent, Widget Top, Widget Left)
{
	Widget		label,
				menu;
	BoxTypePtr	box;

		/* init media size if called the first time */
	if (Index == -1)
		InitDefaultMedia();

	label = XtVaCreateManagedWidget("comment", labelWidgetClass,
		Parent,
		LAYOUT_TOP,
		XtNfromVert, Top,
		XtNfromHoriz, Left,
		XtNlabel, "select media and offsets:",
		NULL);
	menu = PrintPannerCreateMediaMenu(Parent, label, Left);
	SizeField = XtVaCreateManagedWidget("sizeField", labelWidgetClass,
		Parent,
		LAYOUT_TOP,
		XtNfromVert, menu,
		XtNfromHoriz, Left,
		XtNlabel, "",
		NULL);
	Panner = XtVaCreateManagedWidget("panner", pannerWidgetClass,
		Parent,
		LAYOUT_TOP,
		XtNdefaultScale, 1,
		XtNfromVert, Top,
		XtNfromHoriz, SizeField,
		XtNrubberBand, True,
		XtNresizable, True,
		NULL);

		/* get layout size and initialize size label;
		 * layout has already be checked to be non-empty
		 */
	box = GetDataBoundingBox(PCB->Data);
	Width = box->X2 -box->X1;
	Height = box->Y2 -box->Y1;
	CB_Media(NULL, (XtPointer) Index, NULL);
	return(Panner);
}

/* ----------------------------------------------------------------------
 * return the selected offset and height of slider
 */
void PrintPannerGetOffset(Position *OffsetX, Position *OffsetY)
{
	XtVaGetValues(Panner,
		XtNsliderX, OffsetX,
		XtNsliderY, OffsetY,
		NULL);
}

/* ----------------------------------------------------------------------
 * set the panner offset
 */
void PrintPannerSetSliderPosition(Position OffsetX, Position OffsetY)
{
	XtVaSetValues(Panner,
		XtNsliderX, OffsetX,
		XtNsliderY, OffsetY,
		NULL);
}

#else

/***************************************************************************** 
 * routines for X11R4 an older; just some empty routines
 *****************************************************************************/
void PrintPannerUpdate(float Scale, Boolean RotateFlag)
{
}

Widget PrintPannerCreate(Widget Parent, Widget Top, Widget Left)
{
	if (Index == -1)
		InitDefaultMedia();
	Settings.MediaWidth = Media[Index].Width;
	Settings.MediaHeight = Media[Index].Height;
	return(Top);
}

void PrintPannerSetSliderPosition(Position OffsetX, Position OffsetY)
{
}

void PrintPannerGetOffset(Position *OffsetX, Position *OffsetY)
{
		/* always pass zero as offset */
	*OffsetX = 0;
	*OffsetY = 0;
}

#endif

/***************************************************************************** 
 * routines common to all X releases
 *****************************************************************************/
/* ---------------------------------------------------------------------------
 * initializes the user defined mediasize and sets it as default
 * only called once
 */
static void InitDefaultMedia(void)
{
	int				i,
					x, y;
	unsigned int	width, height;
	char			*p1, *p2;

		/* check if the passed string is equal to a known media type
		 * first entry is user defined, last one holds a zero pointer only
		 */
	for (i = 1; i < ENTRIES(Media)-1; i++)
	{
		p1 = Media[i].Name;
		p2 = Settings.Media;

			/* ignore case */
		for (; *p1 && *p2 && (tolower(*p1) == tolower(*p2)); p1++, p2++);

			/* found a match */
		if (*p1 == *p2)
			break;
	}

	if (i != ENTRIES(Media)-1)
	{
			/* used the found entry */
		Index = i;
		width = Media[i].Width;
		height = Media[i].Height;
	}
	else
	{
			/* no match, try to evaluate it as geometry string;
			 * use first entry (user defined)
			 */
		Index = 0;
		i = XParseGeometry(Settings.Media, &x, &y, &width, &height);

			/* check the syntax */
		if (!(i & WidthValue) || !(i & HeightValue) ||
			(i & XValue) || (i & YValue))
		{
				/* failed; use default setting (a4) */
			width = 8260;
			height = 11690;
		}
	}

		/* set size as user default; 'Index' is already set to != -1
		 * to make sure that routine wont be called again
		 */
	Media[0].Width = width;
	Media[0].Height = height;
}
