/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994,1995 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@rz.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: /sda4/users/nau/src/pcb/RCS/printdialog.c,v 2.4 1994/10/29 17:26:45 nau Exp nau $";

/* print dialog routines
 */

#include <stdlib.h>

#include "global.h"

#include "data.h"
#include "dialog.h"
#include "misc.h"
#include "mymem.h"
#include "print.h"
#include "printdialog.h"
#include "printpanner.h"

#include <X11/Shell.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/Toggle.h>

/* ---------------------------------------------------------------------------
 * define resource for X11R4 (names have changed from R4 to R5)
 * and scrollbar sizes
 */
#ifndef	XtNtopOfThumb
#define	XtNtopOfThumb	XtNtop
#endif

#define	MIN_SCALE			0.5
#define	MAX_SCALE			3.0
#define	THUMB_LENGTH		10
#define	SCROLLBAR_LENGTH	150


/* ---------------------------------------------------------------------------
 * some local identifiers
 */
static	Widget		Popup,				/* the popup shells */
					ScrollbarW,			/* scaling scrollbar */
					ScaleW;				/* scale label */
static	float		Scale = 1.0;		/* initial scaling factor */
static	int			ReturnCode;			/* returncode of buttons */
static	Boolean		RotateFlag = True;	/* initial rotation flag */

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	void	UpdateScale(void);
static	void	VisibleHeightAndWidth(Dimension *, Dimension *);
static	void	UpdateScrollbar(float);
static	void	CB_CancelOrOK(Widget, XtPointer, XtPointer);
static	void	CB_Rotate(Widget, XtPointer, XtPointer);
static	void	CB_ScrollProc(Widget, XtPointer, XtPointer);
static	void	CB_JumpProc(Widget, XtPointer, XtPointer);

/* ---------------------------------------------------------------------------
 * updates scale label
 */
static void UpdateScale(void)
{
	char	s[10];

	sprintf(s, "%4.2f", Scale);
	XtVaSetValues(ScaleW, XtNlabel, s, NULL);
}

/* ----------------------------------------------------------------------
 * returns the visible size of the layout
 */
static void VisibleHeightAndWidth(Dimension *W, Dimension *H)
{
	BoxTypePtr	box;

		/* get layout size and scale it; layout has already be checked
		 * to be non-empty
		 */
	box = GetDataBoundingBox(PCB->Data);
	*W = (Dimension) ((float) (RotateFlag ?
		(box->Y2 -box->Y1) : (box->X2 -box->X1)) *Scale);
	*H = (Dimension) ((float) (RotateFlag ?
		(box->X2 -box->X1) : (box->Y2 -box->Y1)) *Scale);
}

/* ---------------------------------------------------------------------------
 * updates the position of the scrollbar thumb
 * the thumb position is passed (range 0..1)
 */
static void UpdateScrollbar(float NewScale)
{
	float	top;

		/* set new scale, ignore all but first digit behind decimal dot */
	NewScale = MIN(NewScale, MAX_SCALE);
	NewScale = MAX(NewScale, MIN_SCALE);
	Scale = ((int) (10.0*NewScale)) /10.0;
	top = (Scale -MIN_SCALE) / (MAX_SCALE -MIN_SCALE);
	top = MIN(top, (1.0 -((float) THUMB_LENGTH /(float) SCROLLBAR_LENGTH)));
	XawScrollbarSetThumb(ScrollbarW, top, -1.0);
	UpdateScale();
	PrintPannerUpdate(Scale, RotateFlag);
}

/* ---------------------------------------------------------------------------
 * callback function for OK and cancel button
 */
static void CB_CancelOrOK(Widget W, XtPointer ClientData, XtPointer CallData)
{
	ReturnCode = (int) ClientData;
}

/* ---------------------------------------------------------------------------
 * callback function if rotate toggle button
 * just updates the slider
 */
static void CB_Rotate(Widget W, XtPointer ClientData, XtPointer CallData)
{
	XtVaGetValues(W, XtNstate, &RotateFlag, NULL);
	PrintPannerUpdate(Scale, RotateFlag);
}

/* ---------------------------------------------------------------------------
 * callback function for scrolling
 * see Athena Widget manual for details
 */
static void CB_ScrollProc(Widget W, XtPointer ClientData, XtPointer CallData)
{
	float	top;
	int		delta = (int) CallData;

		/* get thumb postion */
	XtVaGetValues(W, XtNtopOfThumb, &top, NULL);
	top += ((float) delta/ (float) SCROLLBAR_LENGTH);
	top = MIN(1.0, MAX(top, 0.0));
	UpdateScrollbar((float) (MAX_SCALE -MIN_SCALE) *top +(float) MIN_SCALE);
}

/* ---------------------------------------------------------------------------
 * callback function for scrolling
 * see Athena Widget manual for details
 */
static void CB_JumpProc(Widget W, XtPointer ClientData, XtPointer CallData)
{
	float	top = *(float *) CallData;

	UpdateScrollbar((float) (MAX_SCALE -MIN_SCALE) *top +(float) MIN_SCALE);
}

/* ---------------------------------------------------------------------------
 * print command dialog
 * prints the package layer only if flag is set
 */
void PrintDialog(Boolean PackageOnly)
{
			Widget				masterform,
								label,
								rotate,
								mirror,
								color,
								last,
								input;
			Dimension			width,
								height;
	static	Position			offsetx = 0,
								offsety = 0;
	static	String				command = NULL;
	static	Boolean				mirrorflag = False,
								colorflag = False;
	static	DialogButtonType	buttons[] = {
		{ "defaultButton", "   OK   ", CB_CancelOrOK,
			(XtPointer) OK_BUTTON, NULL },
		{ "cancelButton", "No/Cancel", CB_CancelOrOK,
			(XtPointer) CANCEL_BUTTON, NULL }};

		/* initialize command if called the first time */
	if (!command)
		command = MyStrdup(Settings.PrintCommand ? Settings.PrintCommand : "lp",
			"PrintDialog()");

		/* create a popup shell */
	Popup = XtVaCreatePopupShell("popup", transientShellWidgetClass,
		Output.Toplevel,
		XtNallowShellResize, True,
		XtNmappedWhenManaged, False,
		XtNinput, True,
		NULL);

		/* the form that holds everything */
	masterform = XtVaCreateManagedWidget("printMasterForm", formWidgetClass,
		Popup,
		XtNresizable, True,
		NULL);

		/* now all buttons, labels ... */
	label = XtVaCreateManagedWidget("comment", labelWidgetClass,
		masterform,
		LAYOUT_TOP,
		XtNlabel, "print settings:",
		NULL);
	label = XtVaCreateManagedWidget("comment", labelWidgetClass,
		masterform,
		LAYOUT_TOP,
		XtNfromVert, label,
		XtNlabel, "select output scale here:",
		NULL);
	ScrollbarW = XtVaCreateManagedWidget("scrollbar", scrollbarWidgetClass,
		masterform,
		LAYOUT_TOP,
		XtNfromVert, label,
		XtNlength, SCROLLBAR_LENGTH,
		XtNminimumThumb, THUMB_LENGTH,
		XtNorientation, XtorientHorizontal,
		NULL);
	ScaleW = XtVaCreateManagedWidget("scale", labelWidgetClass,
		masterform,
		LAYOUT_TOP,
		XtNfromVert, label,
		XtNfromHoriz, ScrollbarW,
		NULL);
	label = XtVaCreateManagedWidget("comment", labelWidgetClass,
		masterform,
		LAYOUT_TOP,
		XtNfromVert, ScrollbarW,
		XtNlabel, "select field to rotate output 90 degree:",
		NULL);
	rotate = XtVaCreateManagedWidget("rotate", toggleWidgetClass,
		masterform,
		LAYOUT_TOP,
		XtNfromVert, label,
		XtNstate, RotateFlag,
		NULL);
	label = XtVaCreateManagedWidget("comment", labelWidgetClass,
		masterform,
		LAYOUT_TOP,
		XtNfromVert, rotate,
		XtNlabel, "select field to mirror output data:",
		NULL);
	mirror = XtVaCreateManagedWidget("mirror", toggleWidgetClass,
		masterform,
		LAYOUT_TOP,
		XtNfromVert, label,
		XtNstate, mirrorflag,
		NULL);
	label = XtVaCreateManagedWidget("comment", labelWidgetClass,
		masterform,
		LAYOUT_TOP,
		XtNfromVert, mirror,
		XtNlabel, "select field to generate color output:",
		NULL);
	color = XtVaCreateManagedWidget("color", toggleWidgetClass,
		masterform,
		LAYOUT_TOP,
		XtNfromVert, label,
		XtNstate, colorflag,
		NULL);
	last = PrintPannerCreate(masterform, color, NULL);
	label = XtVaCreateManagedWidget("comment", labelWidgetClass,
		masterform,
		LAYOUT_TOP,
		XtNfromVert, last,
		XtNlabel, "enter filename or command (starts with '|'):",
		NULL);
	input = XtVaCreateManagedWidget("input", asciiTextWidgetClass,
		masterform,
		XtNresizable, True,
		XtNresize, XawtextResizeWidth,
		XtNstring, command,
		XtNwrap, XawtextWrapNever,
		XtNeditType, XawtextEdit,
		XtNfromVert, label,
		LAYOUT_TOP,
		NULL);

		/* now we add the buttons, the first one is always default */
	AddButtons(masterform, input, buttons, ENTRIES(buttons));

		/* add callback functions, update panner and thumb position */
	XtAddCallback(rotate, XtNcallback, CB_Rotate, NULL);
	XtAddCallback(ScrollbarW, XtNjumpProc, CB_JumpProc, NULL);
	XtAddCallback(ScrollbarW, XtNscrollProc, CB_ScrollProc, NULL);
	UpdateScale();
	PrintPannerUpdate(Scale, RotateFlag);
	UpdateScrollbar(Scale);

		/* override default translations and
		 * install accelerators for buttons
		 */
	XtOverrideTranslations(input, XtParseTranslationTable(InputTranslations));
	XtInstallAccelerators(input, buttons[0].W);
	XtInstallAccelerators(input, buttons[1].W);

		/* set focus to input line */
	XtSetKeyboardFocus(masterform, input);

		/* the panner widget has to be realized before the
		 * offset can be set
		 */
	XtRealizeWidget(Popup);
	PrintPannerSetSliderPosition(offsetx, offsety);

		/* now display dialog window and handle input */
	StartDialog(Popup);
	DialogEventLoop(&ReturnCode);

		/* get button settings and offsets */
	XtVaGetValues(rotate, XtNstate, &RotateFlag, NULL);
	XtVaGetValues(mirror, XtNstate, &mirrorflag, NULL);
	XtVaGetValues(color, XtNstate, &colorflag, NULL);
	VisibleHeightAndWidth(&width, &height);
	PrintPannerGetOffset(&offsetx, &offsety);

		/* release old command and allocate new string */
	SaveFree(command);
	XtVaGetValues(input, XtNstring, &command, NULL);
	command = StripWhiteSpaceAndDup(command);

		/* now everything is saved; destroy the dialog */
	EndDialog(Popup);

		/* transform offset of upper left to lower left corner because
		 * PostScript and X11 use different directions for the y-axis */
	if (ReturnCode == OK_BUTTON && command)
		PrintPS(command, Scale,
			mirrorflag, RotateFlag, colorflag, PackageOnly,
			offsetx +Settings.MediaMarginLeft,
			Settings.MediaHeight -Settings.MediaMarginBottom -offsety -height);
}
