/*
 * warper.c: the warping routine.
 */

#include <stdio.h>
#include <stdlib.h>

#include "warper.h"

static GETFN _getPixel;
static SETFN _setPixel;

/*
 * Since we have to walk on six lines independently, we must 
 * store all necessary data:
 * (xi,yi) = source point;
 * (xf,yf) = destination point;
 * dx = abs(xf-xi);
 * dy = abs(yf-yi);
 * d = decisional variable;
 * incrE = increment to d during an east movement;
 * incrN = increment to d during a north movement;
 * (x,y) = current point;
 * dir = direction;
 * n = number of points of the segment.
 */ 

static int dx[6],dy[6],d[6],xf[6],xi[6];
static int yf[6],yi[6],incrE[6],incrN[6];
static int x[6],y[6],n[6],dir[6];

/* used only for the first four lines */

static int xmap[4][MAXWAREASIZE],ymap[4][MAXWAREASIZE];

/* init data for line i */

static void initLine(int i)
	{
	dx[i]=xf[i]-xi[i];
	dy[i]=yf[i]-yi[i];
	incrE[i]=abs(2*dy[i]);
	incrN[i]=abs(2*dx[i]);
	x[i]=xi[i];
	y[i]=yi[i];
	n[i]=abs(dx[i])+abs(dy[i])+1;
	dir[i]=(dx[i]>dy[i])?1:0;
	if(dir[i])
		d[i]=2*abs(dy[i])-abs(dx[i]);
	else
		d[i]=2*abs(dx[i])-abs(dy[i]);
	}

/* computes the next point for line i */

static void nextStep(int i)
	{
	if(dir[i])
		{
		if(d[i]<=0)
			{
			d[i]+=incrE[i];
			x[i]++;
			}
		else
			{
			d[i]-=incrN[i];
			if(dy[i]>0) y[i]++;
			else y[i]--;
			}
		}
	else
		{
		if(d[i]<=0)
			{
			d[i]+=incrN[i];
			y[i]++;
			}
		else
			{
			d[i]-=incrE[i];
			if(dx[i]>0) x[i]++;
			else x[i]--;
			}
		}

	}

/* put a pixel in the destination area */

static void setPixel(int i,Pixel value)
	{
	_setPixel(x[i],y[i],value);
	}

/* get a pixel from the source area */

static Pixel getPixel(int i)
	{
	return(_getPixel(x[i],y[i]));
	}

/* check whether line i is ended */

static int again(int i)
	{
	if(dir[i])
		{
		if(dy[i]>0) return(x[i]<=xf[i] && y[i]<=yf[i]);
		else return(x[i]<=xf[i] && y[i]>=yf[i]);
		}
	else
		{
		if(dx[i]>0) return(y[i]<=yf[i] && x[i]<=xf[i]);
		else return(y[i]<=yf[i] && x[i]>=xf[i]);
		}

	}

/* maps a line on the source area into the corresponding
   line on the destination area */

static void warpLine()
	{
	long q,loop;
	long accr,incr;
	int i,temp;

	initLine(5);
	initLine(4);
	accr=0;
	if(n[4]-n[5]>0)
		{

		/* the destination is greater than the source */

		q=((long)n[4]<<8)/n[5];
		loop=(q>>8)-1;
		incr=q-256;
		while(again(4))
			{
			temp=getPixel(5);
			setPixel(4,temp);
			nextStep(4);
			for(i=0;i<loop;i++)
				{
				setPixel(4,temp);
				nextStep(4);
				}
			if((accr+=(incr-(i<<8)))>256)
				{
				setPixel(4,temp);
				nextStep(4);
				accr-=256;
				}
			nextStep(5);
			}
		}
	else
		{

		/* the source is grater than the destination */

		q=((long)n[5]<<8)/n[4];
		loop=(q>>8)-1;
		incr=q-256;
		while(again(5))
			{
			temp=getPixel(5);
			setPixel(4,temp);
			nextStep(5);
			for(i=0;i<loop;i++)
				{
				nextStep(5);
				}
			if((accr+=incr-(i<<8))>256)
				{
				nextStep(5);
				accr-=256;
				}
			nextStep(4);
			}
		}
	}

/* computes np points of line i (np>=n[i]) */
 
static void mapLine(int i,int np)
	{
	long q,accr;
	long loop,incr;
	int j,h;
	int *xptr,*yptr;

	q=((long)np<<8)/n[i];
	loop=(q>>8)-1;
	incr=q-256;
	accr=0;
	xptr=&(xmap[i][0]);
	yptr=&(ymap[i][0]);
	for(h=0;h<np;h++)
		{
		*(xptr+h)=x[i];
		*(yptr+h)=y[i];
		for(j=0;j<loop;j++)
			{
			h++;
			*(xptr+h)=x[i];
			*(yptr+h)=y[i];
			}
		if((accr+=incr-(j<<8))>256)
			{
			h++;
			*(xptr+h)=x[i];
			*(yptr+h)=y[i];
			accr-=256;
			}
		nextStep(i);
		}
	}

/* set the pointers to the functions used for setting and getting 
   pixels to and from areas */

void initWarper(GETFN getP,SETFN setP)
	{
	_getPixel=getP;
	_setPixel=setP;
	}

/* just warp an area */

void warpArea(WArea source,WArea dest)
	{
	int max,i;

	/* old area vertices */

	xi[0]=source.x[0];
	yi[0]=source.y[0];
	xi[1]=source.x[1];
	yi[1]=source.y[1];
	xf[1]=source.x[2];
	yf[1]=source.y[2];
	xf[0]=source.x[3];
	yf[0]=source.y[3];

	/* new area vertices */

	xi[2]=dest.x[0];
	yi[2]=dest.y[0];
	xi[3]=dest.x[1];
	yi[3]=dest.y[1];
	xf[3]=dest.x[2];
	yf[3]=dest.y[2];
	xf[2]=dest.x[3];
	yf[2]=dest.y[3];

	/* let's init the four border lines and compute the longest line */

	for(i=0;i<4;i++)
		initLine(i);
	for(max=0,i=1;i<4;i++)
		if(n[i]>n[max]) max=i;

	/* let's map the four lines so as they all have n[max] points */

	for(i=0;i<4;i++)
		mapLine(i,n[max]);

	/* let's warp resulting line */

	for(i=0;i<n[max];i++)
		{

		/* source */

		xi[5]=xmap[0][i];
		yi[5]=ymap[0][i];
		xf[5]=xmap[1][i];
		yf[5]=ymap[1][i];

		/* destination */

		xi[4]=xmap[2][i];
		yi[4]=ymap[2][i];
		xf[4]=xmap[3][i];
		yf[4]=ymap[3][i];

		warpLine();
		}
	}

