;;;; lisp.jl -- Some Lispy functions
;;;  Copyright (C) 1993, 1994 John Harper <jsh@ukc.ac.uk>

;;; This file is part of Jade.

;;; Jade is free software; you can redistribute it and/or modify it
;;; under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.

;;; Jade is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.

;;; You should have received a copy of the GNU General Public License
;;; along with Jade; see the file COPYING.  If not, write to
;;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

(defvar features ()
  "A list of symbols definining which ``features'' Jade currently has loaded.
This is used by the `featurep', `provide' and `require' functions.")

(defun require (feature &optional file)
  "If FEATURE (a symbol) has not already been loaded, load it. The file
loaded is either FILE (if given), or the print name of FEATURE."
  (unless (member feature features)
    (load (unless file (symbol-name feature)))))

(defun provide (feature)
  "Show that the feature FEATURE (a symbol) has been loaded."
  (unless (member feature features)
      (setq features (cons feature features))))

(defun featurep (feature)
  "Return non-nil if feature FEATURE has already been loaded."
  (member feature features))

(defun autoload (symbol &rest autoload-defn)
  "Tell the evaluator that the function value of SYMBOL will be initialised
from a named file. The AUTOLOAD-DEFN is the contents of the SYMBOL's
autoload definition. Currently two items are used, the first is the name
of the file to load the value of SYMBOL from. The second says whether or
not the function SYMBOL may be called interactively (as a command)."
  (fset symbol (cons 'autoload autoload-defn)))

(defun add-hook (hook-symbol new-func &optional at-end)
  "Arrange it so that FUNCTION-NAME is added to the hook-list stored in
symbol, HOOK-SYMBOL. It will added at the head of the list unless AT-END
is non-nil in which case it is added at the end."
  (unless (boundp hook-symbol)
    (set hook-symbol nil))
  (if at-end
      (set hook-symbol (nconc (symbol-value hook-symbol) (cons new-func nil)))
    (set hook-symbol (cons new-func (symbol-value hook-symbol)))))

(defun remove-hook (hook-symbol old-func)
  "Remove FUNCTION-NAME from the hook HOOK-SYMBOL."
  (set hook-symbol (delete old-func (symbol-value hook-symbol))))

(defun prin1-to-string (arg)
  "Return a string representing OBJECT."
  (format nil "%S" arg))

(defun read-from-string (string &optional start)
  "Reads an object from STRING, starting at character number START (default
is 0)."
  (read (make-string-input-stream string start)))

;; Some function pseudonyms
(defmacro setcar (&rest args)
  (cons 'rplaca args))
(defmacro setcdr (&rest args)
  (cons 'rplacd args))
(defmacro string= (&rest args)
  (cons 'equal args))
(fset 'string-equal-p (symbol-function 'string=))
(defmacro string< (&rest args)
  (cons '< args))
(fset 'string-less-p (symbol-function 'string<))

(defun error (&rest args)
  (signal 'error (list (apply 'format nil args))))

(defun eval-and-print (form)
  "Eval FORM then print its value in the status line."
  (interactive "xEval: ")
  (prin1 (eval form) t))
