
/*
 * Copyright 1992 the Board of Trustees of the Leland Stanford Junior
 * University. Official permission to use this software is included in
 * the documentation. It authorizes you to use this file for any
 * non-commercial purpose, provided that this copyright notice is not
 * removed and that any modifications made to this file are commented
 * and dated in the style of the example below.
 */

/*
 *
 *  source file:   ./xtpanel/tree.c
 *
 * Steve Cole, Dave Nichols (SEP), August 28 1992
 *      Inserted this sample edit history entry.
 *      Please log any further modifications made to this file:
 * Dave Nichols (SEP), Feb 23 1993
 *      Expand variables and commands in all strings as they are
 *      read. (Use expand_string in get_value ).
 */

#include <stdio.h>

#include "tree.h"
#include "expand_string.h"

/* 
  
  This is an entry stucture as defined in tree.h.
  
  Each item in a tree is an entry.
  
  struct _ent {
  char* tag;
  char* value;
  struct _ent* child;
  struct _ent* next;
  } ;
  
  */

/* a tree contains either nodes or leaves.
 * each entry must have a tag ( its name )
 *
 * nodes have a child but no value.
 *
 * leaves have a value but no children.
 *
 * either may have a sibling (next)
 */

/* create a new empty entry, the tag is not null terminated */
entry* new_entry( tag, len )
     char*tag;
     int len;
{
    entry* a;
    a = (entry*) malloc( sizeof( entry ) );
    
    a->tag = malloc( len+1 );
    strncpy( a->tag, tag, len );
    a->tag[len] = '\0';
    
    a->value = (char*)0;
    a->child= (entry*)0;
    a->next= (entry*)0;
    return a;
}

/* add a sibling to an entry, if this entry already has one it
 * will go and find the last sibling without a "next" field.  */
void add_sibling( orig, sib )
     entry* orig, *sib;
{
    entry *curr;
    curr=orig;
    
    while( curr->next !=  (entry*)0 ){ curr = curr->next; }
    
    curr->next = sib;
}

/* add a child to an entry. It will be the sibling of any existing child */
void add_child( parent, child )
     entry *parent,*child;
{
    if( parent->child ==  (entry*)0 ){
	parent->child = child;
    }else{
	add_sibling( parent->child, child );
    }
}


/* retrieve an entry given the tag, if prev is non zero the entry will 
   be the next matching entry after prev */
entry* get_next( root, tag, prev )
     entry* root;
     char* tag;
     entry* prev;
{
    entry *ent;
    
    /* start with the first child */
    ent = root->child;
    
    /* first find the previous entry */
    if( prev != (entry*)0 ){
	while( ent != (entry*)0  && ent != prev ) ent = ent->next;
	/* found prev, start at the next one */
	if( ent != (entry*)0 ) ent = ent->next;
    }
    
    while( ent != (entry*)0  ){
	if( !strcmp( ent->tag, tag ) )  break;
	ent = ent->next;
    }
    
    return ent;
}

/* retreive the next "action" or "test" tag, a special helper routine for 
 * actions.c
 */

entry* get_next_act_or_test( root, prev )
     entry* root;
     entry* prev;
{
    entry *ent;

    /* start with the first child */
    ent = root->child;

    /* first find the previous entry */
    if( prev != (entry*)0 ){
        while( ent != (entry*)0  && ent != prev ) ent = ent->next;
        /* found prev, start at the next one */
        if( ent != (entry*)0 ) ent = ent->next;
    }

    /* find a tag called "action" or "test" */
    while( ent != (entry*)0  ){
     if( (!strcmp( ent->tag, "action" ))||(!strcmp(ent->tag,"test") )) break;
     ent = ent->next;
    }

    return ent;
}



/* retrieve a copy of the value string given a tag.
   if the tag is not found the default string will be copied */
char* get_value( root, tag, defstring )
     entry* root;
     char* tag;
     char * defstring;
{
    char *retval;
    entry * ent;
    
    retval = (char*)0;
    
    ent = get_next( root, tag, 0 );
   
 
    if( ent != (entry*)0 &&  ent->value != (char*)0 ) {
	retval = expand_string( (char*)0, ent->value );
    }else{
	retval = expand_string( (char*)0, defstring );
    }
    
    return retval;
}

/* See if a given tag exists */
int is_specified( root, tag )
     entry* root;
     char* tag;
{
    int retval;
    entry * ent;
    
    retval = 0;
    
    ent = get_next( root, tag, 0 );
    
    if( ent != (entry*)0 &&  ent->value != (char*)0 ) {
	retval = 1;
    }else{
	retval = 0;
    }
    
    return retval;
}

/* print the tree rooted at this entry, useful for debugging */
void print_entry( file, ent, num_tab )
     FILE* file;
     entry *ent;
     int num_tab;
{
    char tabs[20];
    int i;
    entry *ch;
    
    for ( i=0; i<num_tab; i++ ) tabs[i] = '\t';
    tabs[num_tab]='\0';
    
    fprintf(file,"%s tag=%s\n",tabs,ent->tag);
    if( ent->value != 0 )  fprintf(file,"%s value=%s\n",tabs,ent->value);
    
    ch = ent->child;
    while( ch != 0 ) {
	print_entry( file, ch, num_tab+1 );
        ch = ch->next;
    }
    
    fprintf(file,"\n");
    
}
