

/*
 * Copyright 1992 the Board of Trustees of the Leland Stanford Junior
 * University. Official permission to use this software is included in
 * the documentation. It authorizes you to use this file for any
 * non-commercial purpose, provided that this copyright notice is not
 * removed and that any modifications made to this file are commented
 * and dated in the style of the example below.
 */

/*
 *
 *  source file:   ./xtpanel/actions.c
 *
 * Steve Cole, Dave Nichols (SEP), August 28 1992
 *      Inserted this sample edit history entry.
 *      Please log any further modifications made to this file:
 * Steve Cole, Dave Nichols (SEP), November 20 1992 
 *      new actions: ASSIGN, SET.
 *      objects can have multiple actions.
 * 	    backquoted strings in actions get executed at action time.
 *  Dave Nichols (SEP), January 19 1993 
 *      Use expand_string to evaluate actions at runtime, this avoids
 *      duplicate code and there is much less parsing to do.
 *  Dave Nichols (SEP), April 25 1993
 *      Introduced conditional actions "test" objects.
 *      QUIT now can have an action format which is passed through atoi to 
 *      generate a return code.
 *  Dave Nichols (SEP), May 7 1993
 *      Fixed all_string_actions to handle conditionals.
 */

#include <stdio.h>
#include <string.h>

#include "tree.h"

#include "actions.h"
#include "object.h"
#include "eval_command.h"
#include "string_buf.h"
#include "expand_string.h"


extern int quitflag;

/*
 * Function name: parse_actions
 * Description: parses all the actions for an object
 * Arguments: name (object->name) root (level in tree)
 * Returns: pointer to new structure of type action
 */
action* parse_actions( name, root )
     char *name;
     entry *root;
{
    action *topact;
    char *act_string;
    action *act;
    entry *ent;
    
    /* Initially no actions */
    topact = (action*) 0;
    
    /* loop over action strings, parse each */
    for (ent = get_next_act_or_test(root,(entry*)0);
	 ent != ((entry*) 0);
	 ent = get_next_act_or_test(root,ent)) {
	
	/* create a new action struct */
	if( topact == (action*) 0){
	    act = topact =  new_action( 0 );
	}else{
	    act = (action*) new_action(topact);
	}
	
	if( !strcmp( ent->tag, "test" )){
	    /* parse a test object from the current tree node */
	    parse_one_test( act, name, ent);
	    
	}else{
	    /* parse an action from the current tree node */
	    /* NOTE: don't use get_value to get the string because that
	     * will expand out the action string and we don't want this 
	     * to happen until action time */
	    if( ent != (entry*)0 &&  ent->value != (char*)0 ) {
		act_string = strdupl( ent->value );
	    }else{
		act_string = strdupl( "NONE" );
	    }
	    parse_one_action( act, name, act_string );
	}
    }
    
    /* make an action of "NONE" if none were specified */
    if( topact == 0 ){
	act = topact =  new_action( 0 );
	parse_one_action( act, name, "NONE" );
    }
    
    /* return the head of the tree */
    return topact;
}
/*
 * Function name: parse_one_test
 * Description: parses a single test for an object
 * Arguments: act (action structure) name (object->name) 
 * root (tree node for test)
 * Returns: none
 */
void parse_one_test( act, name, root )
     action *act;
     char* name;
     entry* root;
{
    entry *true, *false, *cond;
    char *condition;
    
    /* get the condition */
    /* don't use get_value because we don't want the string expanded */
    cond = get_next( root, "condition", (entry*)0 );
    if( cond != (entry*)0 && cond->value != (char*)0 ) {
	act->condition=strdupl( cond->value );
    }else{
	fprintf( stderr,"A test object must have a condition.\n");
	fprintf(stderr,"The test object called \"%s\" does not.\n\n",name);
	exit(-1);
    }
    
    
    /* find the "true" entry */
    true = get_next( root, "true", (entry*)0 );
    if( true == 0 ){ 
	act->true_acts =0;
    }else{ 
	act->true_acts = parse_actions( name, true );
    }	  
    /* find the "false" entry */
    false = get_next( root, "false", (entry*)0 );
    if( false == 0 ){ 
	act->false_acts=0;
    }else{
	act->false_acts = parse_actions( name, false);
    }
}
/*
 * Function name: parse_one_action
 * Description: parses a single action string for an object
 * Arguments: act (action structure) name (object->name) act_string (to parse)
 * Returns: none
 */
void parse_one_action( act, name, act_string )
     action *act;
     char* name;
     char* act_string;
{
    char *startpos;
    int len;
    int brackets;
    
    /* strip leading blanks */
    len = strspn( act_string, " \t\n" );
    act_string += len;
    
    
    /* no action specified or "NONE" explicitly specified */
    if( !strncmp( act_string, "NONE" , 4 ) ){
  	act->type=NONE; return;
    }
    
    /* "QUIT" explicitly specified, the format string should be something
     * that atoi can convert to a number this will be our return code */
    if( !strncmp( act_string, "QUIT" , 4 ) ){
	quitflag=1; 
	act->type=QUIT; 
	act_string += 4;
	
    }else if( !strncmp( act_string, "PRINT" , 5 ) ){
	act->type=PRINT;
	act_string += 5;
	
    }else if(!strncmp( act_string, "STRING" , 6 ) ){
	act->type=STRING;
	act_string += 6;
	
    }else if(!strncmp( act_string, "ASSIGN" , 6 ) ){
	act->type=ASSIGN;
	act_string += 6;
	/* the next word is the object name to assign it to */
	for(startpos=act_string; *startpos==' '; startpos++); 
        len = strcspn( startpos, " " ); act->assignto = malloc( len + 1); 
	strncpy( act->assignto, startpos, len ); act->assignto[len]='\0'; 
	act_string = startpos+len+1;
	
    }else if(!strncmp( act_string, "SET" , 3 ) ){
	act->type=SET;
	act_string += 3;
	/* the next word is the object name to assign it to */ 
	for(startpos=act_string; *startpos==' '; startpos++); 
        len = strcspn( startpos, " " ); act->assignto = malloc( len + 1); 
	strncpy( act->assignto, startpos, len ); act->assignto[len]='\0'; 
	act_string = startpos+len+1;
	/* the next word is the tag to assign it to */
	for(startpos=act_string; *startpos==' '; startpos++); 
        len = strcspn( startpos, " " ); act->totag = malloc( len + 1); 
	strncpy( act->totag, startpos, len ); act->totag[len]='\0'; 
	act_string = startpos+len+1;
	
    }else if(!strncmp( act_string, "SYSTEM" , 6 ) ){
	act->type=SYSTEM;
	act_string += 6;
    }else {
	act->type=SYSTEM;
    }
    
    act->format = strdupl( act_string );
    
}

/*
 * Function name: new_action
 * Description: creates a structure for a new action
 * Arguments: action* topact ( first item in action list for an object)
 * Returns: pointer to new structure of type action
 */
action* new_action(topact)
     action* topact;
{
    action *a, *b;
    
    /* make a new action */ 
    a = (action*) malloc( sizeof( action ) );
    a->assignto = (char *) 0;
    a->next= (action*) 0;
    a->false_acts=(action*)0;
    a->true_acts=(action*)0;
    a->condition=(char*)0;
    
    if( topact == (action*)0 ) return a;
    
    /* find the tail of the list */
    for( b=topact; b->next != (action*)0; b = b->next );
    
    b->next = a;
    
    return a;
}

/*
 * Function name: performs_actions
 * Description: performs all the actions for an object
 * Arguments: name (object->name) topact (head of action list) 
 *            all (>0 do all actions; =0 string actions only)
 * Returns: int
 */
extern void flush_display();

int perform_actions( name, topact, all )
     char *name;
     action *topact;
     int all;
{
    action *act;
    int retcode=0;
    
    for (act=topact; act != (action*) 0; act=act->next) {
	
	if( all && act->condition != 0 ){
	    /* this is a test, evaluate the condition */
	    if( eval_condition( name, act->condition ) == 0 ){
		if( act->true_acts != 0 ) 
		  retcode += perform_actions( name, act->true_acts, all );
	    }else{
		if( act->false_acts != 0 ) 
		  retcode += perform_actions( name, act->false_acts, all );
	    }
	    
	}else{
	    /* this is an unconditional action, so do it */
	    if (all || act->type == STRING || act->type == NONE)
	      retcode += perform_one_action( name, act );
	    flush_display();
	}
	
    }
    
    return retcode;
}

int eval_condition( name, cond )
     char*name;
     char* cond;
{
    char* condline;
    char* condline2;
    int ret;
    /* expand out the condition string */
    condline = expand_string( name, cond );
    /* evaluate the command and return the status */
    /* Throw away any output */
/*    condline2 = malloc( strlen(condline) + 20 );*/
/*    strcpy( condline2,"(");*/
/*    strcat( condline2,condline);*/
/*    strcat( condline2,") >/dev/null 2>&1" );*/
/*    ret = system( condline2 );*/
/*    free( condline );*/
/*    free( condline2 );*/
    ret = system( condline );
    free( condline );
    return ret;
}

/*
 * Function name: perform_one_action
 * Description: performs a single action for an object
 * Arguments: name (object->name) act (action) 
 * Returns: int
 */
int perform_one_action( name, act )
     char *name;
     action *act;
{
    char *action_line;
    
    switch( act->type ){
	case( NONE ):
	  return 0;
	break;
	
	case( QUIT ):
	  action_line = expand_string( name, act->format );
	quit_xtpanel( atoi(action_line) );
	break;
	
	case( PRINT ):
	  action_line = expand_string( name, act->format );
	printf( "%s",action_line );
	fflush(stdout);
	free( action_line );
	return 1;
	break;
	
	case( STRING ):
	  action_line = expand_string( name, act->format );
	set_string( name, action_line );
	free( action_line );
	return 0;
	break;
	
	case( ASSIGN ):
	  action_line = expand_string( name, act->format );
	update_object( act->assignto, action_line );
	free( action_line );
	return 0;
	break;
	
	case( SET ):
	  action_line = expand_string( name, act->format );
	update_tag( act->assignto, act->totag, action_line );
	free( action_line );
	return 0;
	break;
	
	case( SYSTEM ):
	  action_line = expand_string( name, act->format );
	system( action_line );
	free( action_line );
	return 1;
	break;
    }
    
    return 0;
}

/*
 * Function name: all_string_actions
 * Description: determines whether all actions for an object are string type
 * Arguments: topact (head of action list )
 * Returns: int (=0 if not all actions are strings, =1 otherwise)
 */
int all_string_actions( topact )
     action *topact;
{
    action *act;
    int all_strings = 1;
    for (act=topact; act != (action*) 0; act=act->next) {
	if( act->condition != 0 ){
            if( act->true_acts != 0 ) 
		all_strings *= all_string_actions( act->true_acts );
            if( act->false_acts != 0 ) 
		all_strings *= all_string_actions( act->false_acts );
	}else{
	    if (act->type == PRINT || act->type == SYSTEM || 
	        act->type == ASSIGN || act->type == SET) all_strings = 0;
        }
    }
    return all_strings;
}



void print_action( name, act )
     char *name;
     action* act;
{
    int i;
    
    switch( act->type ){
	case( NONE ):	
	  printf(" action type NONE \n");
	return;
	break;
	case( QUIT ):
	  printf(" action type QUIT \n");
	return;
	break;
	
	case( PRINT ):
	  printf(" action type PRINT \n");
	break;
	
	case( SYSTEM ):
	  printf(" action type SYSTEM \n");
	break;
	
	case( STRING ):
	  printf(" action type STRING \n");
	break;
	
	case( ASSIGN ):
	  printf(" action type ASSIGN \n");
	break;
	
	case( SET ):
	  printf(" action type SET \n");
	break;
    }
    
    printf(" action format = ->%s<- \n",act->format);
    
    printf(" expanded format = ->%s<- \n",expand_string( name, act->format) );
    
    
}

