-------------------------------------------------------------------------
--
--      CoLibri V2.0 - SQL-Interface (Informix 4.0)
--
-------------------------------------------------------------------------
--
--      Copyright (C) Siemens Nixdorf Informationssysteme AG 1992
--      All rights reserved
--
-------------------------------------------------------------------------
--
-- Author        : A. Graffigna
-- Creation      : 01.10.1992
--
-------------------------------------------------------------------------
--
-- Last Modified By: Torsten Dumm
-- Last Modified On: Wed Feb 16 14:24:19 1994
-- Update Count    : 10
--
-------------------------------------------------------------------------
--
-- Module        : SqlDate.cs
-- Purpose       : Module for supporting the SQL data type SQLDATE and
--                 SQLDATETIME. 
-- Updates       : 
--                 Torsten Dumm 08.02.94
--                 module modified for the CooL Version 2.0
--
-- Exports       : CONST     YEARS       ;
--                 CONST     MONTHS      ;
--                 CONST     DAYS        ;
--                 CONST     HOURS       ;
--                 CONST     MINUTES     ;
--                 CONST     SECONDS     ;
--                 CONST     FRACTIONS   ;
--                 CONST     FRACTIONS_1 ;
--                 CONST     FRACTIONS_2 ;
--                 CONST     FRACTIONS_3 ;
--                 CONST     FRACTIONS_4 ;
--                 CONST     FRACTIONS_5 ;
--
--                 TYPE      SQLDATETIME ;
--                 TYPE      SQLDATE
--
--                 FUNCTION  DateToSqlDate            (..) ;
--                 FUNCTION  getCurrentSqlDate        (..) ;
--                 FUNCTION  getSqlDate               (..) ;
--                 FUNCTION  getDayOfWeekOfSqlDate    (..) ;
--                 FUNCTION  isLeapYear               (..) ;
--                 FUNCTION  isSqlDateEqualNull       (..) ;
--                 FUNCTION  putSqlDate               (..) ;
--                 FUNCTION  setSqlDateToNull         (..) ;
--                 PROCEDURE SqlDateToDate            (..) ;
--                 PROCEDURE SqlDateToString          (..) ;
--                 PROCEDURE SqlDateToStringFormatted (..) ;
--                 FUNCTION  StringFormattedToSqlDate (..) ;
--                 FUNCTION  StringToSqlDate          (..) ;
--
--                 PROCEDURE CharToSqlDatetime        (..) ;
--                 PROCEDURE copySqlDatetime          (..) ;
--                 PROCEDURE getCurrentSqlDatetime    (..) ;
--                 FUNCTION  isSqlDatetimeEqualNull   (..) ;
--                 PROCEDURE setSqlDatetimeToNull     (..) ;
--                 PROCEDURE SqlDateToSqlDatetime     (..) ;
--                 FUNCTION  SqlDatetime              (..) ;
--                 PROCEDURE SqlDatetimeToChar        (..) ;
--                 FUNCTION  SqlDatetimeToSqlDate     (..) ;
--                 PROCEDURE SqlDatetimeToTimeStamp   (..) ;
--                 PROCEDURE TimeStampToSqlDatetime   (..) ;
-------------------------------------------------------------------------

SPECIFICATION SqlDate

IMPORT Date      FROM Date;
IMPORT TimeStamp FROM TimeSt;
IMPORT dec_t     FROM SqlDecimal;

-------------------------------------------------------------------------
-- The data type SQLDATE.
-------------------------------------------------------------------------

TYPE SQLDATE = RECORD
                  value     : INT ;
	          indicator : INT ;
               END RECORD;

PROCEDURE DateToSqlDate (IN date : Date
			 ) : SQLDATE;
  ----------------------------------------------------------------------
  -- This function converts the Date to a SQLDATE.
  -- Exceptions : None .
  ----------------------------------------------------------------------

PROCEDURE getCurrentSqlDate : SQLDATE ;
  ----------------------------------------------------------------------
  -- This function delivers the actual date.
  -- Exceptions : None .
  ----------------------------------------------------------------------

PROCEDURE getSqlDate (IN  sql_date : SQLDATE,
		      OUT year     : INT,
		      OUT month    : INT,
		      OUT day      : INT);
  ----------------------------------------------------------------------
  -- This procedure takes the SQLDATE apart to the single components year,
  -- month and day.
  -- Exceptions : ValueIsNull .
  ----------------------------------------------------------------------

PROCEDURE getDayOfWeekOfSqlDate (IN sql_date : SQLDATE
				 ) : INT ;
  ----------------------------------------------------------------------
  -- This function computes the day of the week.
  -- Exceptions : ValueIsNull .
  ----------------------------------------------------------------------

PROCEDURE isLeapYear (IN year : INT 
		      ) : BOOL;
  ----------------------------------------------------------------------
  -- This function checks whether the input parameter contains a leap year.
  -- Exceptions : None .
  ----------------------------------------------------------------------
  
PROCEDURE isSqlDateEqualNull (IN sql_date : SQLDATE
			      ) : BOOL;
  ----------------------------------------------------------------------
  -- This function checks if the input variable sql_date contains
  -- the null value.
  -- Exceptions : None.
  ----------------------------------------------------------------------
   
PROCEDURE putSqlDate (IN year  : INT,
		      IN month : INT,
		      IN day   : INT
		      ) : SQLDATE;
  ----------------------------------------------------------------------
  -- This function fits the single components year, month and day
  -- together to a SQLDATE.
  -- Exceptions : InvalidYear
  --              InvalidMonth
  --              InvalidDay .
  ----------------------------------------------------------------------

PROCEDURE setSqlDateToNull : SQLDATE;
  ----------------------------------------------------------------------
  -- This function sets a variable of type SQLDATE to the null value.
  -- Exceptions : None.
  ----------------------------------------------------------------------

PROCEDURE SqlDateToDate (IN    sql_date : SQLDATE,
			 INOUT date     : Date);
  ----------------------------------------------------------------------
  -- This procedure converts the SQLDATE to a Date.
  -- Exceptions : ValueIsNull .
  ----------------------------------------------------------------------

PROCEDURE SqlDateToString (IN sql_date : SQLDATE
			   ) : STRING;
  ----------------------------------------------------------------------
  -- This procedure converts the SQLDATE to a string.
  -- The format of the delivered string representation is determined by
  -- the environment variable DBDATE.
  -- The variable string has to contain enough memory space for at least
  -- 12 characters.
  -- The produced string will be terminated by a null-character.
  -- Exceptions : ValueIsNull .
  ----------------------------------------------------------------------

PROCEDURE SqlDateToStringFormatted (IN sql_date : SQLDATE,
				    IN format   : STRING
				    ) : STRING;
  ----------------------------------------------------------------------
  -- This procedure converts the SQLDATE to a String.
  -- The format of the delivered string representation is determined by
  -- the input variable format.
  -- The variable string has to contain enough memory space.
  -- The produced string will be terminated by a null-character.
  -- For this function the INFORMIX message files are necessary.
  -- Exceptions : ValueIsNull
  --              DateConversionFailed
  --              MemoryOverflow .
  ----------------------------------------------------------------------

PROCEDURE StringFormattedToSqlDate (IN string : STRING,
				    IN format : STRING
				    ) : SQLDATE;
  ----------------------------------------------------------------------
  -- This function converts the string to a SQLDATE.
  -- The string has to contain a valid representation of a date value.
  -- This representation is determined by the input parameter format.
  -- The string has to be finished with the null-character.
  -- For this function the INFORMIX message files are necessary.
  -- Exceptions : LengthOfStringIsZero
  --              LengthOfFormatIsZero
  --              InvalidYear
  --              InvalidMonth
  --              InvalidDay
  --              InvalidLength
  --              DateComponentIsMissing .
  ----------------------------------------------------------------------

PROCEDURE StringToSqlDate (IN string : STRING
			   ) : SQLDATE ;
  ----------------------------------------------------------------------
  -- This function converts the string to a SQLDATE.
  -- The string has to contain a valid representation of a date value.
  -- This representation is determined by the environment variable DBDATE.
  -- The string has to be finished with the null-character.
  -- If the string has an actual length of zero the function result will
  -- be the actual date.
  -- Exceptions : DateConversionFailed .
  ----------------------------------------------------------------------

-------------------------------------------------------------------------
-- The data type SQLDATETIME.
-------------------------------------------------------------------------

-- time units for datetime qualifier

CONST YEARS       : INT =  0 ;  -- TU_YEAR
CONST MONTHS      : INT =  2 ;  -- TU_MONTH
CONST DAYS        : INT =  4 ;  -- TU_DAY
CONST HOURS       : INT =  6 ;  -- TU_HOUR
CONST MINUTES     : INT =  8 ;  -- TU_MINUTE
CONST SECONDS     : INT = 10 ;  -- TU_SECOND
CONST FRACTIONS   : INT = 12 ;  -- TU_FRAC
CONST FRACTIONS_1 : INT = 11 ;  -- TU_F1
CONST FRACTIONS_2 : INT = 12 ;  -- TU_F2
CONST FRACTIONS_3 : INT = 13 ;  -- TU_F3
CONST FRACTIONS_4 : INT = 14 ;  -- TU_F4
CONST FRACTIONS_5 : INT = 15 ;  -- TU_F5

TYPE dtime_t = RECORD
	          dt_qual : SHORT INT ;
	          dt_dec  : dec_t ;
               END RECORD;

TYPE SQLDATETIME = RECORD
                      value     : dtime_t ;
	              indicator : INT ;
		      from      : INT ;
		      to        : INT ;
	           END RECORD;

PROCEDURE copySqlDatetime (IN    source      : SQLDATETIME,
			   INOUT destination : SQLDATETIME);
  ----------------------------------------------------------------------
  -- This procedure copies source to destination.
  -- The components of the destination variable determine which components
  -- of the source variable will be copied.
  -- Exceptions : ValueIsNull
  --              InvalidComponent .
  ----------------------------------------------------------------------

PROCEDURE getCurrentSqlDatetime (INOUT sql_datetime : SQLDATETIME);
  ----------------------------------------------------------------------
  -- This procedure delivers the actual date and time.
  -- Exceptions : None .
  ----------------------------------------------------------------------

PROCEDURE isSqlDatetimeEqualNull (IN sql_datetime : SQLDATETIME 
				  ) : BOOL;
  ----------------------------------------------------------------------
  -- This function checks if the input variable sql_datetime contains
  -- the null value.
  -- Exceptions : None.
  ----------------------------------------------------------------------

PROCEDURE setSqlDatetimeToNull (INOUT sql_datetime : SQLDATETIME);
  ----------------------------------------------------------------------
  -- This procedure sets a variable of type SQLDATETIME to the null value.
  -- Exceptions : None.
  ----------------------------------------------------------------------

PROCEDURE SqlDatetime (IN from   : INT ,
		       IN to     : INT ,
		       IN string : STRING
		       ) : SQLDATETIME;
  ----------------------------------------------------------------------
  -- This function initializes a variable of type SQLDATETIME with a
  -- C-String which has to contain a valid representation of a date and
  -- time value.
  -- If the C-String has an actual length of zero the function result
  -- will be the actual date and time.
  -- The parameters "from" and "to" determine the components which have
  -- to be part of the variable.
  -- "from" has to be larger than "to".
  -- Exceptions : IllegalComponent
  --              WrongOrderOfComponents
  --              NoConversionPossible
  --              TooMuchValuesAtBeginning
  --              TooMuchValuesAtEnd
  --              StringContainsCharacters
  --              RangeOverflow
  --              ValueOverflow
  --              ResultOverflow
  --              IncompatibleConversion
  --              InvalidComponent .
  ----------------------------------------------------------------------

PROCEDURE SqlDatetimeToSqlDate (IN sql_datetime : SQLDATETIME 
				) : SQLDATE;
  ----------------------------------------------------------------------
  -- This function converts the SQLDATETIME to a SQLDATE.
  -- Exceptions : ValueIsNull .
  ----------------------------------------------------------------------

PROCEDURE SqlDatetimeToString (IN sql_datetime : SQLDATETIME
			       ) : STRING;
  ----------------------------------------------------------------------
  -- This procedure converts the SQLDATETIME to a C-String.
  -- The variable string has to contain enough memory space for at least
  -- 26 characters.
  -- The produced C-String will be terminated by a null-character.
  -- Exceptions : ValueIsNull .
  ----------------------------------------------------------------------

PROCEDURE SqlDatetimeToTimeStamp (IN    sql_datetime : SQLDATETIME,
				  INOUT time_stamp   : TimeStamp);
  ----------------------------------------------------------------------
  -- This procedure converts the SQLDATETIME to a TimeStamp.
  -- Exceptions : ValueIsNull .
  ----------------------------------------------------------------------

PROCEDURE SqlDateToSqlDatetime (IN    sql_date     : SQLDATE,
				INOUT sql_datetime : SQLDATETIME);
  ----------------------------------------------------------------------
  -- This procedure converts the SQLDATE to a SQLDATETIME.
  -- Exceptions : ValueIsNull .
  ----------------------------------------------------------------------

PROCEDURE StringToSqlDatetime (IN    string       : STRING,
			       INOUT sql_datetime : SQLDATETIME);
  ----------------------------------------------------------------------
  -- This procedure converts the C-String to a SQLDATETIME.
  -- The C-String has to contain a valid representation of a date and
  -- time value.
  -- The C-String has to be finished with the null-character.
  -- Exceptions : LengthOfStringIsZero
  --              NoConversionPossible
  --              TooMuchValuesAtBeginning
  --              TooMuchValuesAtEnd
  --              StringContainsCharacters
  --              RangeOverflow
  --              ValueOverflow
  --              ResultOverflow
  --              IncompatibleConversion
  --              InvalidComponent .
  ----------------------------------------------------------------------

PROCEDURE TimeStampToSqlDatetime (IN    time_stamp   : TimeStamp,
				  INOUT sql_datetime : SQLDATETIME);
  ----------------------------------------------------------------------
  -- This procedure converts the TimeStamp to a SQLDATETIME.
  -- Exceptions : None .
  ----------------------------------------------------------------------

END SPECIFICATION;

