/***************************************************************************/
/*                      SQL_LANG.C - language interface                    */
/*-------------------------------------------------------------------------*/
/*      Copyright (C) Siemens Nixdorf Informationssysteme AG 1992          */
/*      All rights reserved                                                */
/*	Last modified by Michael Watzek					   */
/***************************************************************************/

# include "sql_defs.h"

static char sccsid[] = "@(#) sql_lang.c 1.0 1992-11-29";

/***************************************************************************/
/*	functions for realizing generated conversions
/***************************************************************************/


static char *get_CooL_mode();


/*---- GET_COOL_LANGUAGE_TYPE ---------------------------------------------*/

char *get_CooL_Language_type( type )
SQL_TYPE *type;
{
    static char working_buffer[30];

    /*
       This function returns the language dependent type string for 
       the SQL type. The SOI assumes that the supported language knows
       composite data structures as records (CooL) or structs (C/C++).
       The generated type strings are type names which are defined with
       a TYPE-definition (CooL) or with a typedef-construct (C/C++). 
    */

    switch( type->typecode )
        {
        case CHAR:
        case VARCHAR:
            sprintf( working_buffer, "%s", "STRING" );
            break;
        case INTEGER:
        case SERIAL:
        case SMALLINT:
            sprintf( working_buffer, "%s", "INT" );
            break;
        case SMALLFLOAT:
        case FLOAT:
            sprintf( working_buffer, "%s", "DOUBLE" );
            break;
        case DECIMAL:
        case MONEY:
            sprintf( working_buffer, "%s", "BCD" );
            break;
        case DATE:
            sprintf( working_buffer, "%s", "Date" );
            break;
        case DATETIME:
            sprintf( working_buffer, "%s", "Timestamp" );
            break;
        case INTERVAL:
            sprintf( working_buffer, "%s", "Duration" );
        }    
    return working_buffer;
}


/*---- PRINT_SQL_COOL_ARGUMENTS -------------------------------------------*/

void print_SQLCooL_Arguments( parameter_list )
PAR_LIST *parameter_list;
{ PAR_LIST *Current_Parameter;

    for( Current_Parameter = parameter_list;
	 Current_Parameter != PAR_LIST_NULL;
	 Current_Parameter = Current_Parameter->next )
    {	fprintf( Implementationfile, "%sSQLCooL", get_symbol( Current_Parameter->ident.Hash ) );
	if( Current_Parameter->next != PAR_LIST_NULL )
	    fprintf( Implementationfile, ", " );
}   }


/*---- PRINT_COOL_PARAMETER_LIST ------------------------------------------*/

void print_CooL_parameter_list( file, parameter_list )
FILE *file;
PAR_LIST *parameter_list;
{
    PAR_LIST *Current_Parameter;

    for ( Current_Parameter = parameter_list;
        Current_Parameter != PAR_LIST_NULL;
          Current_Parameter = Current_Parameter->next )
        {
        fprintf( file, "%s %s : %s",
                 get_CooL_mode( Current_Parameter->mode ),
                 get_symbol( Current_Parameter->ident.Hash ),
                 get_CooL_Language_type( &(Current_Parameter->type) ) );
        if ( Current_Parameter->next != PAR_LIST_NULL )
            fprintf( file, ", " );
        }
}


/*---- PRINT_SQL_COOL_PARAMETER_LIST ------------------------------------------*/

void print_SQLCooL_parameter_list( parameter_list )
PAR_LIST *parameter_list;
{
    PAR_LIST *Current_Parameter;

    for ( Current_Parameter = parameter_list;
        Current_Parameter != PAR_LIST_NULL;
          Current_Parameter = Current_Parameter->next )
        {
        fprintf( Implementationfile, "%s %sSQLCooL : %s",
                 get_CooL_mode( Current_Parameter->mode ),
                 get_symbol( Current_Parameter->ident.Hash ),
                 get_Language_type( &(Current_Parameter->type) ) );
        if ( Current_Parameter->next != PAR_LIST_NULL )
            fprintf( Implementationfile, ", " );
        }
}


/*---- PRINT_SQL_COOL_DECLARATIONS -----------------------------------------*/

void print_SQLCooL_Declarations( parameter_list )
PAR_LIST *parameter_list;
{
    PAR_LIST *Current_Parameter;

    for ( Current_Parameter = parameter_list;
          Current_Parameter != PAR_LIST_NULL;
          Current_Parameter = Current_Parameter->next )
           fprintf( Implementationfile, "VAR %sSQLCooL : %s;\n",
           				get_symbol( Current_Parameter->ident.Hash ),
           				get_Language_type( &(Current_Parameter->type) ) );
}


/*---- GET_PARAMETER -------------------------------------------------*/

char *get_parameter( type )
SQL_TYPE *type;
{
    static char working_buffer[200];

    switch( type->typecode )
        {
        case CHAR:
            sprintf( working_buffer, "%d", type->Char.length );
            break;
        case VARCHAR:
            sprintf( working_buffer, "%d, %d", type->VarChar.maxlength, type->VarChar.minlength );
            break;
        case DATETIME:
            sprintf( working_buffer, "%d, %d", 
                                        TU_START( type->Interval.komponent ),
                                         TU_END( type->Interval.komponent ) );
            break;
        case INTERVAL:
            sprintf( working_buffer, "%d, %d, %d,", 
                                   GET_INTVL_FIRST( type->Interval.komponent ),
                                        TU_START( type->Interval.komponent ),
                                         TU_END( type->Interval.komponent ) );
        }
    return working_buffer;
}


/*---- CONVERT_FROM_COOL_TO_SQLCOOL ----------------------------------------*/

void print_CooL_To_SQLCooL_Conversions( prefix1, prefix2, parameter_list )
char	 *prefix1;
char	 *prefix2;
PAR_LIST *parameter_list;
{   PAR_LIST *Current_Parameter;

    for ( Current_Parameter = parameter_list;
          Current_Parameter != PAR_LIST_NULL;
          Current_Parameter = Current_Parameter->next )
    {   if( strcmp( get_CooL_mode( Current_Parameter->mode ), "IN" ) == 0 )
	{   switch( Current_Parameter->type.typecode )
	    {   case CHAR:
		    fprintf( Implementationfile, "  %s%sSQLCooL := SqlChar( %s%s, %s );\n",
						 prefix2, get_symbol(Current_Parameter->ident.Hash),
						 prefix1, get_symbol(Current_Parameter->ident.Hash),
           					 get_parameter(&(Current_Parameter->type)) );
		    break;
		case VARCHAR:
		    fprintf( Implementationfile, "  %s%sSQLCooL := SqlVarChar( %s%s, %s );\n",
						 prefix2, get_symbol(Current_Parameter->ident.Hash),
						 prefix1, get_symbol(Current_Parameter->ident.Hash),
           					 get_parameter(&(Current_Parameter->type)) );
		    break;
		case INTEGER:
		    fprintf( Implementationfile, "  %s%sSQLCooL := IntToSqlInteger( %s%s );\n",
						 prefix2, get_symbol(Current_Parameter->ident.Hash),
           					 prefix1, get_symbol(Current_Parameter->ident.Hash) );
		    break;
		case SERIAL:
		    fprintf( Implementationfile, "  %s%sSQLCooL := IntToSqlSerial( %s%s );\n",
						 prefix2, get_symbol(Current_Parameter->ident.Hash),
           					 prefix1, get_symbol(Current_Parameter->ident.Hash) );
		    break;
		case SMALLINT:
		    fprintf( Implementationfile, "  %s%sSQLCooL := IntToSqlSmallint( %s%s );\n",
						 prefix2, get_symbol(Current_Parameter->ident.Hash),
           					 prefix1, get_symbol(Current_Parameter->ident.Hash) );
		    break;
		case SMALLFLOAT:
		    fprintf( Implementationfile, "  %s%sSQLCooL := DoubleToSqlSmallfloat( %s%s );\n",
						 prefix2, get_symbol(Current_Parameter->ident.Hash),
           					 prefix1, get_symbol(Current_Parameter->ident.Hash) );
		    break;
		case FLOAT:
		    fprintf( Implementationfile, "  %s%sSQLCooL := DoubleToSqlFloat( %s%s );\n",
						 prefix2, get_symbol(Current_Parameter->ident.Hash),
           					 prefix1, get_symbol(Current_Parameter->ident.Hash) );
		    break;
		case DECIMAL:
		case MONEY:
		    fprintf( Implementationfile, "  %s%sSQLCooL := BcdToSqlDecimal( %s%s );\n",
						 prefix2, get_symbol(Current_Parameter->ident.Hash),
           					 prefix1, get_symbol(Current_Parameter->ident.Hash) );
		    break;
		case DATE:
		    fprintf( Implementationfile, "  %s%sSQLCooL := DateToSqlDate( %s%s );\n",
						 prefix2, get_symbol(Current_Parameter->ident.Hash),
           					 prefix1, get_symbol(Current_Parameter->ident.Hash) );
		    break;
		case DATETIME:
		    fprintf( Implementationfile, "  TimeStampToSqlDateTime( %s%s, %s%sSQLCooL );\n",
						 prefix1, get_symbol(Current_Parameter->ident.Hash),
           					 prefix2, get_symbol(Current_Parameter->ident.Hash) );
		    break;
		case INTERVAL:
		    fprintf( Implementationfile, "  DurationToSqlInterval( %s%s, %s%sSQLCooL );\n",
						 prefix1, get_symbol(Current_Parameter->ident.Hash),
           					 prefix2, get_symbol(Current_Parameter->ident.Hash) );
}   }	}   }


/*---- CONVERT_FROM_SQLCOOL_TO_COOL ----------------------------------------*/

void print_SQLCooL_To_CooL_Conversions( prefix1, prefix2, parameter_list )
char	 *prefix1;
char	 *prefix2;
PAR_LIST *parameter_list;
{   PAR_LIST *Current_Parameter;

    for ( Current_Parameter = parameter_list;
          Current_Parameter != PAR_LIST_NULL;
          Current_Parameter = Current_Parameter->next )
    {   if( strcmp( get_CooL_mode( Current_Parameter->mode ), "IN" ) != 0 )
	{   switch( Current_Parameter->type.typecode )
	    {   case CHAR:
		    fprintf( Implementationfile, "  %s%s := SqlCharToString( %s%sSQLCooL );\n",
						 prefix2, get_symbol(Current_Parameter->ident.Hash),
           					 prefix1, get_symbol(Current_Parameter->ident.Hash) );
		    break;
		case VARCHAR:
		    fprintf( Implementationfile, "  %s%s := SqlVarCharToString( %s%sSQLCooL );\n",
						 prefix2, get_symbol(Current_Parameter->ident.Hash),
           					 prefix1, get_symbol(Current_Parameter->ident.Hash) );
		    break;
		case INTEGER:
		    fprintf( Implementationfile, "  %s%s := SqlIntegerToInt( %s%sSQLCooL );\n",
						 prefix2, get_symbol(Current_Parameter->ident.Hash),
           					 prefix1, get_symbol(Current_Parameter->ident.Hash) );
		    break;
		case SERIAL:
		    fprintf( Implementationfile, "  %s%s := SqlSerialToInt( %s%sSQLCooL );\n",
						 get_symbol(Current_Parameter->ident.Hash),
           					 get_symbol(Current_Parameter->ident.Hash) );
		    break;
		case SMALLINT:
		    fprintf( Implementationfile, "  %s%s := SqlSmallintToInt( %s%sSQLCooL );\n",
						 prefix2, get_symbol(Current_Parameter->ident.Hash),
           					 prefix1, get_symbol(Current_Parameter->ident.Hash) );
		    break;
		case SMALLFLOAT:
		    fprintf( Implementationfile, "  %s%s := SqlSmallfloatToDouble( %s%sSQLCooL );\n",
						 prefix2, get_symbol(Current_Parameter->ident.Hash),
           					 prefix1, get_symbol(Current_Parameter->ident.Hash) );
		    break;
		case FLOAT:
		    fprintf( Implementationfile, "  %s%s := SqlFloatToDouble( %s%sSQLCooL );\n",
						 prefix2, get_symbol(Current_Parameter->ident.Hash),
           					 prefix1, get_symbol(Current_Parameter->ident.Hash) );
		    break;
		case DECIMAL:
		case MONEY:
		    fprintf( Implementationfile, "  %s%s := SqlDecimalToBcd( %s%sSQLCooL );\n",
						 prefix2, get_symbol(Current_Parameter->ident.Hash),
           					 prefix1, get_symbol(Current_Parameter->ident.Hash) );
		    break;
		case DATE:
		    fprintf( Implementationfile, "  SqlDateToDate( %s%sSQLCooL, %s%s );\n",
						 prefix1, get_symbol(Current_Parameter->ident.Hash),
           					 prefix2, get_symbol(Current_Parameter->ident.Hash) );
		    break;
		case DATETIME:
		    fprintf( Implementationfile, "  SqlDateTimeToTimeStamp( %s%sSQLCooL, %s%s );\n",
						 prefix1, get_symbol(Current_Parameter->ident.Hash),
           					 prefix2, get_symbol(Current_Parameter->ident.Hash) );
		    break;
		case INTERVAL:
		    fprintf( Implementationfile, "  SqlIntervalToDuration( %s%sSQLCooL, %s%s );\n",
						 prefix1, get_symbol(Current_Parameter->ident.Hash),
           					 prefix2, get_symbol(Current_Parameter->ident.Hash) );
}   }	}   }


/*---- PRINT_ASSIGNMENT_PROCS ------------------------------------------------*/

static void print_assignment_procs( ident, fields )
YY_IDENT *ident;
PAR_LIST *fields;
{ PAR_LIST *Current_Field;

    fprintf( Implementationfile, "\nPROCEDURE %s_COOL_TO_SQLCooL( IN row : %s_R ) : %s_RSQLCooL\n",
		get_symbol(ident->Hash), get_symbol(ident->Hash), get_symbol(ident->Hash) );
    fprintf( Implementationfile, "VAR rowSQLCooL : %s_RSQLCooL;\n", get_symbol(ident->Hash),
								    get_symbol(ident->Hash) );
    fprintf( Implementationfile, "DO\n" );
    print_CooL_To_SQLCooL_Conversions( "row.", "rowSQLCooL.", fields );
    fprintf( Implementationfile, "  RETURN rowSQLCooL;\n" );
    fprintf( Implementationfile, "END DO;\n" );


    fprintf( Implementationfile, "\nPROCEDURE %sSQLCooL_TO_COOL( IN rowSQLCooL : %s_RSQLCooL ) : %s_R\n",
		get_symbol(ident->Hash), get_symbol(ident->Hash), get_symbol(ident->Hash) );
    fprintf( Implementationfile, "VAR row : %s_R;\n", get_symbol(ident->Hash),
						      get_symbol(ident->Hash) );
    fprintf( Implementationfile, "DO\n" );
    print_SQLCooL_To_CooL_Conversions( "rowSQLCooL.", "row.", fields );
    fprintf( Implementationfile, "  RETURN row;\n" );
    fprintf( Implementationfile, "END DO;\n" );
}


/*---- GENERATE_COOL_RECORDS ------------------------------------------------*/

static void generate_CooL_records( ident, fields )
YY_IDENT *ident;
PAR_LIST *fields;
{ PAR_LIST *Current_Field;

    fprintf( Implementationfile, "\nTYPE %s_RSQLCooL =\nRECORD\n", get_symbol( ident->Hash ) );
    for ( Current_Field = fields;
	  Current_Field != PAR_LIST_NULL; 
	  Current_Field = Current_Field->next )
        fprintf( Implementationfile, "  %sSQLCooL : %s;\n", get_symbol( Current_Field->ident.Hash ),
						    get_Language_type( &(Current_Field->type) ) );

    fprintf( Implementationfile, "END RECORD;\n" );

    fprintf( Specificationfile, "\nTYPE %s_R =\nRECORD\n", get_symbol( ident->Hash ) );
    for ( Current_Field = fields;
	  Current_Field != PAR_LIST_NULL; 
	  Current_Field = Current_Field->next )
        fprintf( Specificationfile, "  %s : %s;\n", get_symbol( Current_Field->ident.Hash ),
						    get_CooL_Language_type( &(Current_Field->type) ) );

    fprintf( Specificationfile, "END RECORD;\n" );
    print_assignment_procs( ident, fields );
}

/***************************************************************************/
/*                      service functions                                  */
/***************************************************************************/

/*---- GET_LANGUAGE_TYPE --------------------------------------------------*/

char *get_Language_type( type )
SQL_TYPE *type;
{
    static char working_buffer[30];

    /*
       This function returns the language dependent type string for 
       the SQL type. The SOI assumes that the supported language knows
       composite data structures as records (CooL) or structs (C/C++).
       The generated type strings are type names which are defined with
       a TYPE-definition (CooL) or with a typedef-construct (C/C++). 
    */

    switch( type->typecode )
        {
        case CHAR:
        case VARCHAR:
            sprintf( working_buffer, "%s", "SQLCHAR" );
            break;
        case INTEGER:
            sprintf( working_buffer, "%s", "SQLINTEGER" );
            break;
        case SERIAL:
            sprintf( working_buffer, "%s", "SQLSERIAL" );
            break;
        case SMALLINT:
            sprintf( working_buffer, "%s", "SQLSMALLINT" );
            break;
        case SMALLFLOAT:
            sprintf( working_buffer, "%s", "SQLSMALLFLOAT" );
            break;
        case FLOAT:
            sprintf( working_buffer, "%s", "SQLFLOAT" );
            break;
        case DECIMAL:
            sprintf( working_buffer, "%s", "SQLDECIMAL" );
            break;
        case MONEY:
            sprintf( working_buffer, "%s", "SQLMONEY" );
            break;
        case DATE:
            sprintf( working_buffer, "%s", "SQLDATE" );
            break;
        case DATETIME:
            sprintf( working_buffer, "%s", "SQLDATETIME" );
            break;
        case INTERVAL:
            sprintf( working_buffer, "%s", "SQLINTERVAL" );
        }    
    return working_buffer;
}

/*---- GET_CLANGUAGE_TYPE --------------------------------------------------*/

char *get_CLanguage_type( type )
SQL_TYPE *type;
{
    static char working_buffer[30];

    switch( type->typecode )
        {
        case CHAR:
        case VARCHAR:
            sprintf( working_buffer, "%s", "SQL_CHAR" );
            break;
        case INTEGER:
            sprintf( working_buffer, "%s", "SQL_INTEGER" );
            break;
        case SERIAL:
            sprintf( working_buffer, "%s", "SQL_SERIAL" );
            break;
        case SMALLINT:
            sprintf( working_buffer, "%s", "SQL_SMALLINT" );
            break;
        case SMALLFLOAT:
            sprintf( working_buffer, "%s", "SQL_SMALLFLOAT" );
            break;
        case FLOAT:
            sprintf( working_buffer, "%s", "SQL_FLOAT" );
            break;
        case DECIMAL:
            sprintf( working_buffer, "%s", "SQL_DECIMAL" );
            break;
        case MONEY:
            sprintf( working_buffer, "%s", "SQL_MONEY" );
            break;
        case DATE:
            sprintf( working_buffer, "%s", "SQL_DATE" );
            break;
        case DATETIME:
            sprintf( working_buffer, "%s", "SQL_DATETIME" );
            break;
        case INTERVAL:
            sprintf( working_buffer, "%s", "SQL_INTERVAL" );
        }    
    return working_buffer;
}

/*---- GET_INIT_FUNCTION_NAME ----------------------------------------------*/

char *get_init_function_name( type )
SQL_TYPE *type;
{
    static char working_buffer[30];

    switch( type->typecode )
        {
        case CHAR:
            sprintf( working_buffer, "%s", "SqlChar" );
            break;
        case VARCHAR:
            sprintf( working_buffer, "%s", "SqlVarChar" );
            break;
        case INTEGER:
            sprintf( working_buffer, "%s", "IntToSqlInteger" );
            break;
        case SERIAL:
            sprintf( working_buffer, "%s", "IntToSqlSerial" );
            break;
        case SMALLINT:
            sprintf( working_buffer, "%s", "IntToSqlSmallint" );
            break;
        case SMALLFLOAT:
            sprintf( working_buffer, "%s", "DoubleToSqlSmallfloat" );
            break;
        case FLOAT:
            sprintf( working_buffer, "%s", "DoubleToSqlFloat" );
            break;
        case DECIMAL:
        case MONEY:
            sprintf( working_buffer, "%s", "IntToSqlDecimal" );
            break;
        case DATE:
            sprintf( working_buffer, "%s", "getCurrentSqlDate" );
            break;
        case DATETIME:
            sprintf( working_buffer, "%s", "SqlDatetime" );
            break;
        case INTERVAL:
            sprintf( working_buffer, "%s", "SqlInterval" );
        }    
    return working_buffer;
}

/***************************************************************************/
/*            procedure for generating import list                         */
/***************************************************************************/

/*---- GENERATE_IMPORT_LIST -----------------------------------------------*/

void print_CooL_import_list()
{
  fprintf( Implementationfile, "\nIMPORT GetLength FROM String;\n" );

  fprintf( Implementationfile, "\nIMPORT SQLCHAR FROM SqlChar;\n" );
  fprintf( Implementationfile, "IMPORT getSqlCharLength,\n" );
  fprintf( Implementationfile,   "       isSqlCharEqualNull,\n" );
  fprintf( Implementationfile,   "       setSqlCharToNull,\n" );
  fprintf( Implementationfile,   "       SqlChar,\n" );
  fprintf( Implementationfile,   "       SqlCharToString,\n" );
  fprintf( Implementationfile,   "       SqlVarChar,\n" );
  fprintf( Implementationfile,   "       StringToSqlChar FROM SqlChar;\n" );

  fprintf( Implementationfile, "\nIMPORT SQLSMALLINT FROM SqlInteger;\n" );
  fprintf( Implementationfile, "IMPORT IntToSqlSmallint,\n" );
  fprintf( Implementationfile,   "       isSqlSmallintEqualNull,\n" );
  fprintf( Implementationfile,   "       setSqlSmallintToNull,\n" );
  fprintf( Implementationfile,   "       SqlSmallintToInt FROM SqlInteger;\n" );

  fprintf( Implementationfile, "\nIMPORT SQLINTEGER FROM SqlInteger;\n" );
  fprintf( Implementationfile, "IMPORT IntToSqlInteger,\n" );
  fprintf( Implementationfile,   "       isSqlIntegerEqualNull,\n" );
  fprintf( Implementationfile,   "       setSqlIntegerToNull,\n" );
  fprintf( Implementationfile,   "       SqlIntegerToInt FROM SqlInteger;\n" );

  fprintf( Implementationfile, "\nIMPORT SQLSERIAL FROM SqlInteger;\n" );
  fprintf( Implementationfile, "IMPORT IntToSqlSerial,\n" );
  fprintf( Implementationfile,   "       SqlSerialToInt FROM SqlInteger;\n" );

  fprintf( Implementationfile, "\nIMPORT SQLSMALLFLOAT FROM SqlFloat;\n" );
  fprintf( Implementationfile, "IMPORT DoubleToSqlSmallfloat,\n" );
  fprintf( Implementationfile,   "       isSqlSmallfloatEqualNull,\n" );
  fprintf( Implementationfile,   "       setSqlSmallfloatToNull,\n" );
  fprintf( Implementationfile,   "       SqlSmallfloatToDouble FROM SqlFloat;\n" );

  fprintf( Implementationfile, "\nIMPORT SQLFLOAT FROM SqlFloat;\n" );
  fprintf( Implementationfile, "IMPORT DoubleToSqlFloat,\n" );
  fprintf( Implementationfile,   "       isSqlFloatEqualNull,\n" );
  fprintf( Implementationfile,   "       setSqlFloatToNull,\n" );
  fprintf( Implementationfile,   "       SqlFloatToDouble FROM SqlFloat;\n" );

  fprintf( Implementationfile, "\nIMPORT SQLDECIMAL FROM SqlDecimal;\n" );
  fprintf( Implementationfile, "IMPORT addSqlDecimal,\n" );
  fprintf( Implementationfile,   "       BcdToSqlDecimal,\n" );
  fprintf( Implementationfile,   "       compareSqlDecimal,\n" );
  fprintf( Implementationfile,   "       divideSqlDecimal,\n" );
  fprintf( Implementationfile,   "       DoubleToSqlDecimal,\n" );
  fprintf( Implementationfile,   "       IntToSqlDecimal,\n" );
  fprintf( Implementationfile,   "       isSqlDecimalEqualNull,\n" );
  fprintf( Implementationfile,   "       roundSqlDecimal,\n" );
  fprintf( Implementationfile,   "       setSqlDecimalToNull,\n" );
  fprintf( Implementationfile,   "       SqlDecimal,\n" );
  fprintf( Implementationfile,   "       SqlDecimalToBcd,\n" );
  fprintf( Implementationfile,   "       SqlDecimalToDouble,\n" );
  fprintf( Implementationfile,   "       SqlDecimalToInt,\n" );
  fprintf( Implementationfile,   "       SqlDecimalToString,\n" );
  fprintf( Implementationfile,   "       StringToSqlDecimal,\n" );
  fprintf( Implementationfile,   "       substractSqlDecimal,\n" );
  fprintf( Implementationfile,   "       truncateSqlDecimal FROM SqlDecimal;\n" );

  fprintf( Implementationfile,  "\nIMPORT SQLMONEY FROM SqlDecimal;\n" );

  fprintf( Implementationfile, "\nIMPORT SQLDATE FROM SqlDate;\n" );
  fprintf( Implementationfile,   "IMPORT DateToSqlDate,\n" );
  fprintf( Implementationfile,   "       getSqlDate,\n" );
  fprintf( Implementationfile,   "       getDayOfWeekOfSqlDate,\n" );
  fprintf( Implementationfile,   "       isLeapYear,\n" );
  fprintf( Implementationfile,   "       isSqlDateEqualNull,\n" );
  fprintf( Implementationfile,   "       putSqlDate,\n" );
  fprintf( Implementationfile,   "       setSqlDateToNull,\n" );
  fprintf( Implementationfile,   "       SqlDateToString,\n" );
  fprintf( Implementationfile,   "       SqlDateToStringFormatted,\n" );
  fprintf( Implementationfile,   "       StringFormattedToSqlDate,\n" );
  fprintf( Implementationfile,   "       StringToSqlDate FROM SqlDate;\n" );

  fprintf( Implementationfile, "\nIMPORT SQLDATETIME FROM SqlDate;\n" );
  fprintf( Implementationfile,   "IMPORT copySqlDatetime,\n" );
  fprintf( Implementationfile,   "       getCurrentSqlDatetime,\n" );
  fprintf( Implementationfile,   "       isSqlDatetimeEqualNull,\n" );
  fprintf( Implementationfile,   "       SqlDatetime,\n" );
  fprintf( Implementationfile,   "       SqlDatetimeToSqlDate,\n" );
  fprintf( Implementationfile,   "       SqlDatetimeToString,\n" );
  fprintf( Implementationfile,   "       SqlDatetimeToTimeStamp,\n" );
  fprintf( Implementationfile,   "       SqlDateToSqlDatetime,\n" );
  fprintf( Implementationfile,   "       StringToSqlDatetime,\n" );
  fprintf( Implementationfile,   "       TimeStampToSqlDatetime FROM SqlDate;\n" );

  fprintf( Implementationfile, "\nIMPORT SQLINTERVAL FROM SqlInterval;\n" );
  fprintf( Implementationfile,   "IMPORT DurationToSqlInterval,\n" );
  fprintf( Implementationfile,   "       isSqlIntervalEqualNull,\n" );
  fprintf( Implementationfile,   "       setSqlIntervalToNull,\n" );
  fprintf( Implementationfile,   "       SqlInterval,\n" );
  fprintf( Implementationfile,   "       SqlIntervalToDuration,\n" );
  fprintf( Implementationfile,   "       SqlIntervalToString,\n" );
  fprintf( Implementationfile,   "       StringToSqlInterval FROM SqlInterval;\n" );

}

/***************************************************************************/
/*            procedure for generating system specification                */
/***************************************************************************/

/*---- GENERATE_SYSTEM_SPECIFICATION --------------------------------------*/

void generate_system_specification( System_Name )
char *System_Name;
{

    /* In this procedure the prolog of the language interface is generated.
       In C/C++ a prolog has to written into the language and header file. */

    switch ( current_Language )
        {
        case L_COOL:
            if ( strcmp( System_Name, "informix40" ) == 0 )
                {
                fprintf( Specificationfile, 
                            "-- SOI-CooL-Specification-file for INFORMIX 4.0\n\n" );
                fprintf( Implementationfile, 
                            "-- SOI-CooL-Implemenation-file for INFORMIX 4.0\n\n" );
		if ( verbose_Flag )
		  fprintf( Protocolfile,
		   "generated database interface: INFORMIX 4.0\n\n" );
                }
            else if ( strcmp( System_Name, "informix41" ) == 0 )
	      {
                fprintf( Specificationfile, 
                            "-- SOI-CooL-Specification-file for INFORMIX 4.1\n\n" );
                fprintf( Implementationfile, 
                            "-- SOI-CooL-Implemenation-file for INFORMIX 4.1\n\n" );
		if ( verbose_Flag )
		  fprintf( Protocolfile,
		   "generated database interface: INFORMIX 4.1\n\n" );
	      }
            else if ( strcmp( System_Name, "informix50" ) == 0 )
	      {
                fprintf( Specificationfile, 
                            "-- SOI-CooL-Specification-file for INFORMIX 5.0\n\n" );
                fprintf( Implementationfile, 
                            "-- SOI-CooL-Implemenation-file for INFORMIX 5.0\n\n" );
		if ( verbose_Flag )
		  fprintf( Protocolfile,
		   "generated database interface: INFORMIX 5.0\n\n" );
	      }
	    generate_informix_prolog();
	    fprintf( Specificationfile, "\nSPECIFICATION %s\n", Modulename );
	    fprintf( Implementationfile, "\nIMPLEMENTATION %s\n", Modulename );
	    print_CooL_import_list();
	    fprintf( Specificationfile, "\nEXCEPTION DatabaseError;" );
	    fprintf( Specificationfile, "\nEXCEPTION DBObjectAlreadyExists;\n\n" );
	    fprintf( Implementationfile, "\nVAR Sql_Error : INT;\n" );
	    break;
        case L_CPP:
            {
            if ( strcmp( System_Name, "informix40" ) == 0 )
                {
                fprintf( Languagefile, 
                            "/* SOI-C++-Interface for INFORMIX 4.0 */\n\n");
                fprintf( Headerfile, 
                    "/* Headerfile SOI-C++-Interface for INFORMIX 4.0 */\n\n");
		if ( verbose_Flag )
		  fprintf( Protocolfile,
		   "generated database interface: INFORMIX 4.0\n\n" );
                }

            if ( strcmp( System_Name, "informix41" ) == 0 )
                {
                fprintf( Languagefile, 
                            "/* SOI-C++-Interface for INFORMIX 4.1 */\n\n");
                fprintf( Headerfile, 
                    "/* Headerfile SOI-C++-Interface for INFORMIX 4.1 */\n\n");
		if ( verbose_Flag )
		  fprintf( Protocolfile,
		   "generated database interface: INFORMIX 4.1\n\n" );
                }
            if ( strcmp( System_Name, "informix50" ) == 0 )
                {
                fprintf( Languagefile, 
                            "/* SOI-C++-Interface for INFORMIX 5.0 */\n\n");
                fprintf( Headerfile, 
                    "/* Headerfile SOI-C++-Interface for INFORMIX 5.0 */\n\n");
		if ( verbose_Flag )
		  fprintf( Protocolfile,
		   "generated database interface: INFORMIX 5.0\n\n" );
                }

            generate_informix_prolog();

            /* The headerfile including the database specific definitions
               has to be only included in the headerfile generated out of
               the description file */

            fprintf( Headerfile, "\n# include \"SqlInf.h\"\n" );

            if ( Outfile_Name == (char *)0 )
                fprintf( Languagefile, "\n# include \"sql_cpp.h\"\n" );
            else
                fprintf( Languagefile, "\n# include \"%s.h\"\n",
                                                            Outfile_Name );
            }
        }
}

/***************************************************************************/
/*            procedure for generating constant specifcations              */
/***************************************************************************/

/*---- GENERATE_CONST_SPECIFICATION ---------------------------------------*/

void generate_const_specification( Symbol, Value )
char *Symbol;
int Value;
{

    /* The constant specification is generated for CooL in the CooL-file
       and for C/C++ in the corresponding headerfile */

    switch ( current_Language )
        {
        case L_COOL:
            fprintf( Specificationfile, "CONST %s : INT = %d;\n", Symbol, Value );
            break;
        case L_CPP:
            fprintf( Headerfile, "# define %s %d\n", Symbol, Value );
        }
    if ( verbose_Flag )
      fprintf( Protocolfile, "Constant: \"%s\", value: %d\n", Symbol, Value );
}

/***************************************************************************/
/*           procedures for generating view record types                   */
/***************************************************************************/

/*---- GENERATE_CPP_RECORD -------------------------------------------------*/

static void generate_Cpp_record( ident, fields )
YY_IDENT *ident;
PAR_LIST *fields;
{
    PAR_LIST *Current_Field;

    /* In C/C++ the view record definition is generated in the corresponding
       header file */

    fprintf( Headerfile, "\ntypedef struct\n{\n" ); 
    for ( Current_Field = fields; Current_Field != PAR_LIST_NULL; 
                                        Current_Field = Current_Field->next )
        {
        fprintf( Headerfile, "  %s %s;\n",
                                get_Language_type( &(Current_Field->type) ),
                                    get_symbol( Current_Field->ident.Hash ) );
        }
    fprintf( Headerfile, "} %s_R;\n", get_symbol( ident->Hash ) );
}

/*---- GENERATE_LANGUAGE_RECORD --------------------------------------------*/

void generate_Language_record( ident, fields )
YY_IDENT *ident;
PAR_LIST *fields;
{
    switch ( current_Language )
        {
        case L_COOL:
            generate_CooL_records( ident, fields );
            break;
        case L_CPP:
            generate_Cpp_record( ident, fields );
            break;
        }
    if ( verbose_Flag )
      fprintf( Protocolfile, 
	       "\trecord type: \"%s_R\"\n", get_symbol( ident->Hash ) );
}

/***************************************************************************/
/*            procedures for generating the init procedures                */
/***************************************************************************/

/*---- GET_INIT_PARAMETER -------------------------------------------------*/

char *get_init_parameter( type )
SQL_TYPE *type;
{
    static char working_buffer[200];

    switch( type->typecode )
        {
        case CHAR:
            sprintf( working_buffer, "\"\", %d", type->Char.length );
            break;
        case VARCHAR:
            sprintf( working_buffer, "\"\", %d, %d", type->VarChar.maxlength,
                                                     type->VarChar.minlength );
            break;
        case INTEGER:
        case SERIAL:
        case SMALLINT:
        case DECIMAL:
        case MONEY:
	    sprintf( working_buffer, "0" );
	    break;
        case SMALLFLOAT:
        case FLOAT:
	    sprintf( working_buffer, "0.0" );
	    break;
        case DATE:
            sprintf( working_buffer, "");
            break;
        case DATETIME:
            sprintf( working_buffer, "%d, %d, \"\"", 
                                        TU_START( type->Interval.komponent ),
                                         TU_END( type->Interval.komponent ) );
            break;
        case INTERVAL:
            sprintf( working_buffer, "%d, %d, %d, \"\"", 
                                   GET_INTVL_FIRST( type->Interval.komponent ),
                                        TU_START( type->Interval.komponent ),
                                         TU_END( type->Interval.komponent ) );
        }
    return working_buffer;
}

/*---- GENERATE_COOL_INIT_PROCEDURE ---------------------------------------*/

static void generate_CooL_init_procedure( ident, fields )
YY_IDENT *ident;
PAR_LIST *fields;
{
    PAR_LIST *Current_Field;

    fprintf( Implementationfile, "\nPROCEDURE init_%s_RSQLCooL() : %s_RSQLCooL\n", 
                      get_symbol( ident->Hash ), get_symbol( ident->Hash ) );
    fprintf( Implementationfile, "VAR record : %s_RSQLCooL;\nDO\n", get_symbol( ident->Hash ) );
    for ( Current_Field = fields; Current_Field != PAR_LIST_NULL; 
            Current_Field = Current_Field->next )
        {
        fprintf( Implementationfile, "  record.%sSQLCooL := %s(%s);\n",
                            get_symbol( Current_Field->ident.Hash ),
                            get_init_function_name( &(Current_Field->type) ),
                            get_init_parameter( &(Current_Field->type) ) );
        }
    fprintf( Implementationfile, "  RETURN record;\n" );
    fprintf( Implementationfile, "END DO;\n" );
}

/*---- GENERATE_CPP_INIT_PROCEDURE ----------------------------------------*/

static void generate_Cpp_init_procedure( ident, fields )
YY_IDENT *ident;
PAR_LIST *fields;
{
    PAR_LIST *Current_Field;

    fprintf( Languagefile, "\n%s_R init_%s_R()\n{\n",
                      get_symbol( ident->Hash ), get_symbol( ident->Hash ) );
    fprintf( Languagefile, "%s_R record;\n", get_symbol( ident->Hash ) ); 
    for ( Current_Field = fields; Current_Field != PAR_LIST_NULL; 
            Current_Field = Current_Field->next )
        {
        fprintf( Languagefile, "  record.%s = init_%s(%s);\n",
                            get_symbol( Current_Field->ident.Hash ),
                            get_Language_type( &(Current_Field->type) ),
                            get_init_parameter( &(Current_Field->type) ) );
        }
    fprintf( Languagefile, "  return record;\n}\n" );

   /* In C/C++ an external definition of the init function has to be stored
      into the corresponding header file */

    fprintf( Headerfile, "\nextern %s_R init_%s_R();\n",
                       get_symbol( ident->Hash ), get_symbol( ident->Hash ) );
}

/*---- GENERATE_INIT_PROCEDURE --------------------------------------------*/

void generate_init_procedure( ident, fields )
YY_IDENT *ident;
PAR_LIST *fields;
{
    switch ( current_Language )
        {
        case L_COOL:
            generate_CooL_init_procedure( ident, fields );
            break;
        case L_CPP:
            generate_Cpp_init_procedure( ident, fields );
            break;
        }
    if ( verbose_Flag )
      fprintf( Protocolfile, 
	      "\tinit procedure for record type: \"init_%s_R()\"\n", 
	                                    get_symbol( ident->Hash ) );
}

/***************************************************************************/
/*         procedures for generating the statement specification           */
/***************************************************************************/

/*---- PRINT_ARGUMENTS ----------------------------------------------------*/

static void print_arguments( file, parameter_list )
FILE *file;
PAR_LIST *parameter_list;
{
    PAR_LIST *Current_Parameter;

    for ( Current_Parameter = parameter_list;
	     Current_Parameter != PAR_LIST_NULL;
		Current_Parameter = Current_Parameter->next )
	{
	fprintf( file, "%s",
			get_symbol( Current_Parameter->ident.Hash ) );
	if ( Current_Parameter->next != PAR_LIST_NULL )
	fprintf( file, ", " );
	}
}

/*---- GET_COOL_MODE ------------------------------------------------------*/

static char *get_CooL_mode( mode )
PMODE mode;
{
    switch ( mode )
        {
        case M_IN:
            return "IN";
        case M_OUT:
            return "OUT";
        case M_INOUT:
            return "INOUT";
        }
}

/*---- GET_CPP_MODE -------------------------------------------------------*/

static char *get_Cpp_mode( mode )
PMODE mode;
{
    switch ( mode )
        {
        case M_IN:
            return "";
        case M_OUT:
            return "&";
        case M_INOUT:
            return "&";
        }
}


/*---- PRINT_CPP_PARAMETER_LIST -------------------------------------------*/

void print_Cpp_parameter_list( Outfile, parameter_list )
FILE *Outfile;
PAR_LIST *parameter_list;
{
    PAR_LIST *Current_Parameter;

    for ( Current_Parameter = parameter_list;
        Current_Parameter != PAR_LIST_NULL;
          Current_Parameter = Current_Parameter->next )
        {
        fprintf( Outfile, "%s %s%s",
                            get_Language_type( &(Current_Parameter->type) ),
                              get_Cpp_mode( Current_Parameter->mode ),
                                get_symbol( Current_Parameter->ident.Hash ) );
        if ( Current_Parameter->next != PAR_LIST_NULL )
            fprintf( Outfile, ", " );
        }
}

/*---- GENERATE_COOL_STATEMENT --------------------------------------------*/

void generate_CooL_statement( ident, parameter_list )
YY_IDENT *ident;
PAR_LIST *parameter_list;
{
    char *Symbol;
    PAR_LIST *Current_Parameter;

    Symbol = get_symbol( ident->Hash );
    if ( parameter_list == PAR_LIST_NULL )
        {
        fprintf( Specificationfile, "\nPROCEDURE %s();\n", Symbol );
        fprintf( Implementationfile, "\nFOREIGN PROCEDURE SQL_%s() : INT;\n", Symbol);
        fprintf( Implementationfile, "\nPROCEDURE %s()\n", Symbol );
	fprintf( Implementationfile, "DO\n" );
        fprintf( Implementationfile, "  Sql_Error := SQL_%s();\n", Symbol );
        }
    else
        {
        fprintf( Specificationfile, "\nPROCEDURE %s( ", Symbol );
        print_CooL_parameter_list( Specificationfile, parameter_list );
        fprintf( Specificationfile, " );\n" );

        fprintf( Implementationfile, "\nFOREIGN PROCEDURE SQL_%s( ", Symbol );
        print_SQLCooL_parameter_list( parameter_list );
        fprintf( Implementationfile, " ) : INT;\n" );

        fprintf( Implementationfile, "\nPROCEDURE %s( ", Symbol );
        print_CooL_parameter_list( Implementationfile, parameter_list );
        fprintf( Implementationfile, " )\n" );

	print_SQLCooL_Declarations( parameter_list );
	fprintf( Implementationfile, "DO\n" );
	print_CooL_To_SQLCooL_Conversions( "", "", parameter_list );
        fprintf( Implementationfile,"  Sql_Error := SQL_%s( ", Symbol );
        print_SQLCooL_Arguments( parameter_list );
        fprintf( Implementationfile, " );\n" );
	print_SQLCooL_To_CooL_Conversions( "", "", parameter_list );
    }
    fprintf( Implementationfile, "  IF Sql_Error <> 0 THEN RAISE DatabaseError;\n" );
    fprintf( Implementationfile, "  END IF;\n" );
    fprintf( Implementationfile, "END DO;\n" );
}

/*---- GENERATE_CPP_STATEMENT ---------------------------------------------*/

void generate_Cpp_statement( ident, parameter_list )
YY_IDENT *ident;
PAR_LIST *parameter_list;
{
    char *Symbol;
    PAR_LIST *Current_Parameter;

    /* In C/C++ both function specifications has to be included into
       the header file */

    Symbol = get_symbol( ident->Hash );
    if ( parameter_list == PAR_LIST_NULL )
        {
        fprintf( Headerfile, "\nextern int SQL_%s();\n", Symbol);
        fprintf( Headerfile, "\nextern void %s();\n", Symbol );

        fprintf( Languagefile, "\nvoid %s()\n{\n", Symbol );
        }
    else
        {
        fprintf( Headerfile, "\nextern int SQL_%s( ", Symbol );
        print_Cpp_parameter_list( Headerfile, parameter_list );
        fprintf( Headerfile, " );\n" );
        fprintf( Headerfile, "\nextern void %s( ", Symbol );
        print_Cpp_parameter_list( Headerfile, parameter_list );
        fprintf( Headerfile, " );\n" );

        fprintf( Languagefile, "\nvoid %s( ", Symbol );
        print_Cpp_parameter_list( Languagefile, parameter_list );
        fprintf( Languagefile, " )\n{\n" );
        }
    if ( parameter_list == PAR_LIST_NULL )
        fprintf( Languagefile, "  Sql_Error = SQL_%s();\n", Symbol );
    else
        {
        fprintf( Languagefile,"  Sql_Error = SQL_%s( ", Symbol );
        print_arguments( parameter_list );
        fprintf( Languagefile, " );\n" );
        }
    fprintf( Languagefile, "  if (Sql_Error) RAISE( DatabaseError );\n" );
    fprintf( Languagefile, "};\n" );
}

/*---- GENERATE_LANGUAGE_STATEMENT ----------------------------------------*/

void generate_Language_statement( ident, parameter_list )
YY_IDENT *ident;
PAR_LIST *parameter_list;
{
    if ( verbose_Flag )
      fprintf( Protocolfile, "\nstatement \"%s()\"\n",
	                                    get_symbol( ident->Hash ) );
    switch ( current_Language )
        {
        case L_COOL:
            generate_CooL_statement( ident, parameter_list );
            break;
        case L_CPP:
            generate_Cpp_statement( ident, parameter_list );
            break;
        }
}

/***************************************************************************/
/*             procedures for generating the object type                   */
/***************************************************************************/

/*---- PRINT_COOL_OBJECT_USED_EXTERNALS -----------------------------------*/

void print_CooL_object_used_externals( Object_Name, parameters, cursor_type )
char *Object_Name;
PAR_LIST *parameters;
CURSOR_TYPE cursor_type;
{
    fprintf( Implementationfile, "\nFOREIGN VAR %s_count : INT;\n", Object_Name );

    /* this specification differs from that for C++ */
    fprintf( Implementationfile, "FOREIGN PROCEDURE SQL_%s_Declare();\n",
                                                                Object_Name );
    fprintf( Implementationfile, "FOREIGN PROCEDURE SQL_%s_Free() : INT;\n",
                                                                Object_Name );

    if ( parameters == PAR_LIST_NULL )
        fprintf( Implementationfile, "FOREIGN PROCEDURE SQL_%s_Open() : INT;\n",
                                Object_Name );
    else
        {
        fprintf( Implementationfile, "FOREIGN PROCEDURE SQL_%s_Open( ", Object_Name );
        print_SQLCooL_parameter_list( parameters );
        fprintf( Implementationfile, " ) : INT;\n" );
        }
    fprintf( Implementationfile, "FOREIGN PROCEDURE SQL_%s_Close() : INT;\n",
                                                                Object_Name );
    switch ( cursor_type )
        {
        case C_SELECT:
          fprintf( Implementationfile,
        "FOREIGN PROCEDURE SQL_%s_Fetch_previous( OUT rowSQLCooL : %s_RSQLCooL ) : INT;\n",
                        Object_Name , Object_Name );
          fprintf( Implementationfile,
        "FOREIGN PROCEDURE SQL_%s_Fetch_last( OUT rowSQLCooL : %s_RSQLCooL ) : INT;\n",
                                                Object_Name, Object_Name );
          fprintf( Implementationfile,
        "FOREIGN PROCEDURE SQL_%s_Fetch_first( OUT rowSQLCooL : %s_RSQLCooL ) : INT;\n",
                                                Object_Name, Object_Name );
          fprintf( Implementationfile, 
        "FOREIGN PROCEDURE SQL_%s_Fetch_next( OUT rowSQLCooL : %s_RSQLCooL ) : INT;\n",
                                                Object_Name, Object_Name );
            break;
        case C_UPDATE:
          fprintf( Implementationfile, 
        "FOREIGN PROCEDURE SQL_%s_Fetch_next( OUT rowSQLCooL : %s_RSQLCooL ) : INT;\n",
                                                Object_Name, Object_Name );
          fprintf( Implementationfile, 
        "FOREIGN PROCEDURE SQL_%s_Update( rowSQLCooL : %s_RSQLCooL ) : INT;\n",
                                                Object_Name, Object_Name );
          fprintf( Implementationfile, 
        "FOREIGN PROCEDURE SQL_%s_Delete() : INT;\n", Object_Name );
            break;
        case C_INSERT:
          fprintf( Implementationfile,
            "FOREIGN PROCEDURE SQL_%s_Put( rowSQLCooL : %s_RSQLCooL ) : INT;\n",
                                                Object_Name, Object_Name );
          fprintf( Implementationfile,
	    "FOREIGN PROCEDURE SQL_%s_Flush() : INT;\n",
                                                Object_Name );
        }
        
}
 
/*---- PRINT_CPP_OBJECT_USED_EXTERNALS ------------------------------------*/

void print_Cpp_object_used_externals( Object_Name, parameters, cursor_type )
char *Object_Name;
PAR_LIST *parameters;
CURSOR_TYPE cursor_type;
{
    /* The class variable "object_name"_count is generated as initialized
       variable declaration in the language file. This differs from CooL
       but it uses the mechanism of class variables in C++ */

    fprintf( Languagefile, "\nint %s::%s_count = 0;\n", Object_Name,
	                                                       Object_Name );

    fprintf( Headerfile, "extern int SQL_%s_Declare();\n", Object_Name );
    fprintf( Headerfile, "extern int SQL_%s_Free();\n", Object_Name );

    if ( parameters == PAR_LIST_NULL )
        fprintf( Headerfile, "extern int SQL_%s_Open();\n", Object_Name );
    else
        {
        fprintf( Headerfile, "extern int SQL_%s_Open( ", Object_Name );
        print_Cpp_parameter_list( Headerfile, parameters );
        fprintf( Headerfile, " );\n" );
        }
    fprintf( Headerfile, "extern int SQL_%s_Close();\n", Object_Name );

    switch ( cursor_type )
        {
        case C_SELECT:
            fprintf( Headerfile,
                      "extern int SQL_%s_Fetch_previous( %s_R &row );\n",
                                                Object_Name , Object_Name );
            fprintf( Headerfile,
                      "extern int SQL_%s_Fetch_last( %s_R &row );\n",
                                                Object_Name , Object_Name );
            fprintf( Headerfile,
                      "extern int SQL_%s_Fetch_first( %s_R &row );\n",
                                                Object_Name , Object_Name );
            fprintf( Headerfile,
                      "extern int SQL_%s_Fetch_next( %s_R &row );\n",
                                                Object_Name , Object_Name );
            break;
        case C_UPDATE:
            fprintf( Headerfile,
                      "extern int SQL_%s_Fetch_next( %s_R &row );\n",
                                                Object_Name , Object_Name );
            fprintf( Headerfile,
                      "extern int SQL_%s_Update( %s_R row );\n",
                                                Object_Name , Object_Name );
            fprintf( Headerfile,
                      "extern int SQL_%s_Delete();\n", Object_Name );
            break;
        case C_INSERT:
             fprintf( Headerfile,
                      "extern int SQL_%s_Put( %s_R row );\n",
                                                Object_Name, Object_Name );
             fprintf( Headerfile, "extern int SQL_%s_Flush();\n", Object_Name );
        }
        
}

/*---- PRINT_COOL_CURSOR_METHOD -------------------------------------------*/

static void print_CooL_cursor_method( Object_Name, Method_Name, Parameter )
char *Object_Name;
char *Method_Name;
char *Parameter;
{
    if ( *Parameter == '\0' )
    {	fprintf( Implementationfile, "  METHOD %s()\n", Method_Name );
	fprintf( Implementationfile, "  DO\n" );
	fprintf( Implementationfile, "    Sql_Error := SQL_%s_%s();\n", Object_Name, Method_Name );
    }
    else
    {   fprintf( Implementationfile, "  METHOD %s( %s : %s_R )\n", Method_Name, Parameter, Object_Name );
	fprintf( Implementationfile, "  VAR %sSQLCooL : %s_RSQLCooL;\n", Parameter, Object_Name );
	fprintf( Implementationfile, "  DO\n" );
	fprintf( Implementationfile, "    %sSQLCooL := %s_COOL_TO_SQLCooL( %s );\n",
					Parameter, Object_Name, Parameter );
	fprintf( Implementationfile, "    Sql_Error := SQL_%s_%s( %sSQLCooL );\n",
				     Object_Name, Method_Name, Parameter );
    }

    fprintf( Implementationfile, "    IF Sql_Error <> 0 THEN RAISE DatabaseError;\n" );
    fprintf( Implementationfile, "    END IF;\n  END DO;\n" );
}

/*---- PRINT_CPP_CURSOR_METHOD --------------------------------------------*/

static void print_Cpp_cursor_method( Object_Name, Method_Name, Parameter )
char *Object_Name;
char *Method_Name;
char *Parameter;
{
  if ( *Parameter == '\0' )
    fprintf( Languagefile, "\nvoid %s::%s()\n{\n", Object_Name, Method_Name );
  else
    fprintf( Languagefile, "void %s::%s( %s_R %s )\n{\n", Object_Name,
					Method_Name, Object_Name, Parameter );
  fprintf( Languagefile, "  Sql_Error = SQL_%s_%s(%s);\n",
					Object_Name, Method_Name, Parameter );
  fprintf( Languagefile, "  if ( Sql_Error != 0 ) RAISE(DatabaseError);\n" );
  fprintf( Languagefile, "}\n" );
}

/*---- PRINT_COOL_CURSOR_FETCH_METHOD ---------------------------------------*/

static void print_CooL_cursor_fetch_method( Object_Name, Method_Name )
char *Object_Name;
char *Method_Name;
{
  fprintf( Implementationfile, "  METHOD %s() : %s_R\n", Method_Name, Object_Name );
  fprintf( Implementationfile, "  VAR row : %s_R;\n", Object_Name );
  fprintf( Implementationfile, "  VAR rowSQLCooL : %s_RSQLCooL;\n", Object_Name );
  fprintf( Implementationfile, "  DO\n" );
  fprintf( Implementationfile, "    rowSQLCooL := init_%s_RSQLCooL();\n", Object_Name );
  fprintf( Implementationfile, "    Sql_Error := SQL_%s_%s( rowSQLCooL );\n", Object_Name, Method_Name );
  fprintf( Implementationfile, "    row := %sSQLCooL_TO_COOL( rowSQLCooL );\n", Object_Name );
  fprintf( Implementationfile, "    IF Sql_Error <> 0 THEN RAISE DatabaseError;\n    END IF;\n" );
  fprintf( Implementationfile, "    RETURN row;\n" );
  fprintf( Implementationfile, "  END DO;\n" );
}

/*---- PRINT_CPP_CURSOR_FETCH_METHOD ----------------------------------------*/

static void print_Cpp_cursor_fetch_method( Object_Name, Method_Name )
char *Object_Name;
char *Method_Name;
{
  fprintf( Languagefile, "\n%s_R %s::%s()\n{\n", Object_Name, Object_Name,
								 Method_Name );
  fprintf( Languagefile, "  %s_R row;\n", Object_Name );
  fprintf( Languagefile, "  row = init_%s_R();\n", Object_Name );
  fprintf( Languagefile, "  Sql_Error = SQL_%s_%s(row);\n",
						    Object_Name, Method_Name );
  fprintf( Languagefile, "  if ( Sql_Error != 0 ) RAISE(DatabaseError);\n" );
  fprintf( Languagefile, "  return row;\n}\n" );
}

/*---- GENERATE_COOL_OBJECT_TYPE ------------------------------------------*/

void generate_CooL_object_type( ident, parameters, cursor_type )
YY_IDENT *ident;
PAR_LIST *parameters;
CURSOR_TYPE cursor_type;
{
    char *Object_Name;

    Object_Name = get_symbol( ident->Hash );

    print_CooL_object_used_externals( Object_Name, parameters, cursor_type );

    fprintf( Specificationfile, "\nTYPE %s = OBJECT\n", Object_Name );
    fprintf( Specificationfile, "PUBLIC\n" );
    fprintf( Implementationfile, "\nTYPE %s = OBJECT\n", Object_Name );

    fprintf( Specificationfile, "  METHOD Finally();\n" );
    if ( parameters == PAR_LIST_NULL ) 
        fprintf( Specificationfile, "  METHOD Open();\n" );
    else
        {
        fprintf( Specificationfile, "  METHOD Open( " );
        print_CooL_parameter_list( Specificationfile, parameters );
        fprintf( Specificationfile, " );\n" );
        }

    fprintf( Specificationfile, "  METHOD Close();\n" );
    switch ( cursor_type )
	{
        case C_SELECT:
            fprintf( Specificationfile, "  METHOD Fetch_previous() : %s_R;\n",
                                                                Object_Name );
            fprintf( Specificationfile, "  METHOD Fetch_last() : %s_R;\n",
                                                                Object_Name );
            fprintf( Specificationfile, "  METHOD Fetch_first() : %s_R;\n",
                                                                Object_Name );
            fprintf( Specificationfile, "  METHOD Fetch_next() : %s_R;\n",
                                                                Object_Name );
            break;
	case C_UPDATE:
	    fprintf( Specificationfile, "  METHOD Fetch_next() : %s_R;\n",
								Object_Name );
	    fprintf( Specificationfile, "  METHOD Update( row : %s_R );\n",
								Object_Name );
	    fprintf( Specificationfile, "  METHOD Delete();\n" );
	    break;
	case C_INSERT:
	    fprintf( Specificationfile, "  METHOD Put( row : %s_R );\n",
								Object_Name );
	    fprintf( Specificationfile, "  METHOD Flush();\n" );
	}

    fprintf( Specificationfile, "PROTECTED\n" );
    fprintf( Specificationfile, "STATE\n" );
    fprintf( Implementationfile, "BODY\n" );

    fprintf( Implementationfile, "  INITIALLY\n" );
    fprintf( Implementationfile, "  DO\n" );
    fprintf( Implementationfile, "    IF %s_count < 1 THEN\n", Object_Name );
    fprintf( Implementationfile, "      SQL_%s_Declare();\n", Object_Name );
    fprintf( Implementationfile, "      %s_count := 1;\n", Object_Name );
    fprintf( Implementationfile, "     ELSE\n" );
    fprintf( Implementationfile, "       DELETE CURRENT;\n" );
    fprintf( Implementationfile, "       RAISE DBObjectAlreadyExists;\n" );
    fprintf( Implementationfile, "    END IF;\n  END DO;\n" );

    fprintf( Implementationfile, "  METHOD Finally()\n" );
    fprintf( Implementationfile, "  DO\n" );
    fprintf( Implementationfile, "    Sql_Error := SQL_%s_Free();\n", Object_Name );
    fprintf( Implementationfile, "    IF Sql_Error <> 0 THEN\n" );
    fprintf( Implementationfile, "      RAISE DatabaseError;\n" );
    fprintf( Implementationfile, "    END IF;\n" );
    fprintf( Implementationfile, "    %s_count := 0;\n", Object_Name );
    fprintf( Implementationfile, "    DELETE CURRENT;\n  END DO;\n" );

    if ( parameters == PAR_LIST_NULL )
	print_CooL_cursor_method( Object_Name, "Open", "" );
    else
	{
	fprintf( Implementationfile, "  METHOD Open( " );
	print_CooL_parameter_list( Implementationfile, parameters );
	fprintf( Implementationfile, " )\n" );
	print_SQLCooL_Declarations( parameters );
	fprintf( Implementationfile, "DO\n" );
	print_CooL_To_SQLCooL_Conversions( "", "", parameters );
	fprintf( Implementationfile, "      Sql_Error := SQL_%s_Open(", Object_Name );
	print_SQLCooL_Arguments( parameters );
        fprintf( Implementationfile, ");\n" );
	fprintf( Implementationfile, 
                       "      IF Sql_Error <> 0 THEN RAISE DatabaseError;\n" );
        fprintf( Implementationfile, "    END IF;\n  END DO;\n" );
	}

    print_CooL_cursor_method( Object_Name, "Close", "" );

    switch ( cursor_type )
        {
        case C_SELECT:
            print_CooL_cursor_fetch_method( Object_Name, "Fetch_previous" );
            print_CooL_cursor_fetch_method( Object_Name, "Fetch_last" );
            print_CooL_cursor_fetch_method( Object_Name, "Fetch_first" );
            print_CooL_cursor_fetch_method( Object_Name, "Fetch_next" );
	    break;
        case C_UPDATE:
            print_CooL_cursor_fetch_method( Object_Name, "Fetch_next" );
            print_CooL_cursor_method( Object_Name, "Update", " row" );
            print_CooL_cursor_method( Object_Name, "Delete", "" );
            break;
        case C_INSERT:
            print_CooL_cursor_method( Object_Name, "Put", " row" );
            print_CooL_cursor_method( Object_Name, "Flush", "" );
	}
    fprintf( Specificationfile, "END OBJECT;\n" );
    fprintf( Implementationfile, "END OBJECT;\n" );
}


/*---- GENERATE_CPP_OBJECT_TYPE -------------------------------------------*/

void generate_Cpp_object_type( ident, parameters, cursor_type )
YY_IDENT *ident;
PAR_LIST *parameters;
CURSOR_TYPE cursor_type;
{
    char *Object_Name;

    Object_Name = get_symbol( ident->Hash );

    print_Cpp_object_used_externals( Object_Name, parameters, cursor_type );

    fprintf( Headerfile, "\nclass %s\n{\n", Object_Name );
    fprintf( Headerfile, "protected:\n  static int %s_count;\n", Object_Name );
    fprintf( Headerfile, "public:\n" );
    fprintf( Headerfile, "  %s();\n", Object_Name );  /* Constructor */
    fprintf( Headerfile, "  ~%s();\n", Object_Name ); /* Destructor */
    fprintf( Headerfile, "  virtual void Finally();\n" );
    if ( parameters == PAR_LIST_NULL ) 
        fprintf( Headerfile, "  virtual void Open();\n" );
    else
        {
        fprintf( Headerfile, "  virtual void Open( " );
	print_Cpp_parameter_list( Headerfile, parameters );
	fprintf( Headerfile, " );\n" );
	}

    fprintf( Headerfile, "  virtual void Close();\n" );
    switch ( cursor_type )
	{
	case C_SELECT:
	    fprintf( Headerfile, "  virtual %s_R Fetch_previous();\n",
							     Object_Name );
	    fprintf( Headerfile, "  virtual %s_R Fetch_last();\n",
							     Object_Name );
	    fprintf( Headerfile, "  virtual %s_R Fetch_first();\n",
							     Object_Name );
	    fprintf( Headerfile, "  virtual %s_R Fetch_next();\n",
							      Object_Name );
	    break;
	case C_UPDATE:
	    fprintf( Headerfile, "  virtual %s_R Fetch_next();\n",
								Object_Name );
	    fprintf( Headerfile, "  virtual void Update( %s_R row );\n",
								Object_Name );
	    fprintf( Headerfile, "  virtual void Delete();\n" );
	    break;
	case C_INSERT:
	    fprintf( Headerfile, "  virtual void Put( %s_R row );\n",
							       Object_Name );
	    fprintf( Headerfile, "  virtual void Flush();\n" );
	}
    fprintf( Headerfile, "};\n" );

    fprintf( Languagefile, "\n%s::%s()\n{\n", Object_Name, Object_Name );
    fprintf( Languagefile, "  if (%s_count < 1) {\n", Object_Name );
    fprintf( Languagefile, "    SQL_%s_Declare();\n", Object_Name );
    fprintf( Languagefile, "    %s_count = 1; }\n", Object_Name );
    fprintf( Languagefile, "   else {\n" );
    fprintf( Languagefile, "     this->%s::~%s();\n", Object_Name, 
	                                                      Object_Name );
    fprintf( Languagefile, "     RAISE(DBObjectAlreadyExists); }\n" );
    fprintf( Languagefile, "}\n" );

    fprintf( Languagefile, "\n%s::~%s() {}\n", Object_Name, Object_Name );

    fprintf( Languagefile, "\nvoid %s::Finally()\n{\n", Object_Name );
    fprintf( Languagefile, "  Sql_Error = SQL_%s_Free();\n", Object_Name );
    fprintf( Languagefile, "  if (Sql_Error != 0)\n" );
    fprintf( Languagefile, "    RAISE(DatabaseError);\n" );
    fprintf( Languagefile, "  %s_count = 0;\n", Object_Name );
    fprintf( Languagefile, "  this->%s::~%s();\n}\n", Object_Name, 
	                                                     Object_Name );

    if ( parameters == PAR_LIST_NULL )
	print_Cpp_cursor_method( Object_Name, "Open", "" );
    else
	{
	fprintf( Languagefile, "\nvoid %s::Open( ", Object_Name );
	print_Cpp_parameter_list( Languagefile, parameters );
	fprintf( Languagefile, " )\n{\n" );
	fprintf( Languagefile, "  Sql_Error = SQL_%s_Open(", Object_Name );
	print_arguments( parameters );
	fprintf( Languagefile, ");\n" );
	fprintf( Languagefile,
		       "  if (Sql_Error != 0) RAISE(DatabaseError);\n" );
	fprintf( Languagefile, "}\n" );
	}

    print_Cpp_cursor_method( Object_Name, "Close", "" );

    switch ( cursor_type )
	{
	case C_SELECT:
	    print_Cpp_cursor_fetch_method( Object_Name, "Fetch_previous" );
	    print_Cpp_cursor_fetch_method( Object_Name, "Fetch_last" );
	    print_Cpp_cursor_fetch_method( Object_Name, "Fetch_first" );
	    print_Cpp_cursor_fetch_method( Object_Name, "Fetch_next" );
	    break;
	case C_UPDATE:
	    print_Cpp_cursor_fetch_method( Object_Name, "Fetch_next" );
	    print_Cpp_cursor_method( Object_Name, "Update", " row " );
	    print_Cpp_cursor_method( Object_Name, "Delete", "" );
	    break;
	case C_INSERT:
	    print_Cpp_cursor_method( Object_Name, "Put", " row " );
	    print_Cpp_cursor_method( Object_Name, "Flush", "" );
	}
}

/*---- GENERATE_OBJECT_TYPE -----------------------------------------------*/

void generate_object_type( ident, parameters, cursor )
YY_IDENT *ident;
PAR_LIST *parameters;
SQL_CURSOR *cursor;
{
    switch ( current_Language )
      {
      case L_COOL:
	generate_CooL_object_type( ident, parameters, cursor->Select.type );
	break;
      case L_CPP:
	generate_Cpp_object_type( ident, parameters, cursor->Select.type );
	break;
      }
    if ( verbose_Flag )
      {
	fprintf( Protocolfile, "\tobject type: \"%s\"\n",
	                                        get_symbol( ident->Hash ) );
	fprintf( Protocolfile, "\t  methods: \"Finally()\"\n" );
	fprintf( Protocolfile, "\t           \"Open()\"\n" );
	fprintf( Protocolfile, "\t           \"Close()\"\n" );
	switch ( cursor->Select.type )
	  {
	  case C_SELECT:
	    fprintf( Protocolfile, "\t           \"Fetch_next()\"\n" );
	    fprintf( Protocolfile, "\t           \"Fetch_previous()\"\n" );
	    fprintf( Protocolfile, "\t           \"Fetch_last()\"\n" );
	    fprintf( Protocolfile, "\t           \"Fetch_first()\"\n" );
	    break;
	  case C_UPDATE:
	    fprintf( Protocolfile, "\t           \"Fetch_next()\"\n" );
	    fprintf( Protocolfile, "\t           \"Delete()\"\n" );
	    fprintf( Protocolfile, "\t           \"Update()\"\n" );
	    break;
	  case C_INSERT:
	    fprintf( Protocolfile, "\t           \"Put()\"\n" );
	    fprintf( Protocolfile, "\t           \"Flush()\"\n" );
	  }
      }
}
