/***************************************************************************/
/*                    SQL_GRAM.Y - SQLGEN-interface grammar                */
/*-------------------------------------------------------------------------*/
/*      Copyright (C) Siemens Nixdorf Informationssysteme AG 1992          */
/*      All rights reserved                                                */
/***************************************************************************/

/* This grammar is for the generator of the interface for INFORMIX 4.x */

/*---- reserved words -----------------------------------------------------*/

%token SQL_END
%token VIEW
%token AS
%token IS
%token LP
%token RP
%token CM
%token SM
%token CONST
%token ASSIGN
%token SYSTEM
%token STATEMENT
%token IN
%token OUT
%token INOUT
%token CURSOR
%token OPEN
%token WITH
%token FOR
%token SELECT
%token INSERT
%token UPDATE
%token MUL
%token DIV
%token PLUS
%token MINUS
%token INTO
%token CHAR
%token VARCHAR
%token INTEGER
%token SERIAL
%token SMALLINT
%token SMALLFLOAT
%token FLOAT
%token DECIMAL
%token MONEY
%token DATE
%token TO
%token DATETIME
%token INTERVAL
%token YEAR
%token MONTH
%token DAY
%token HOUR
%token MINUTE
%token SECOND
%token FRACTION

/*---- complex tokens -----------------------------------------------------*/

%token IDENTIFIER
%token NUMBER

%token sql_select
%token sql_statement

%type <ident>          IDENTIFIER

%type <number>         constant_expression
                       mul_const_expression
                       unary_const_expression
                       primary_const_expression
                       NUMBER
                       datetime_components
                       interval_components
                       time
                       fraction
                       datetime_simple_time
                       simple_time
                       simple_time_length
                       update_clause
                       mode

%type <list>           identifier_list
                       columns

%type <par_list>       cursor_parameter_list
                       parameters
                       statement_parameter_list
                       statement_parameters

%type <type>           SQL_type

%type <cursor>         cursor_clause

%type <sql>            sql_select sql_statement

%type <statement_list> sql_single_statement sql_statements

/*---- grammar rules ------------------------------------------------------*/

%start description_unit

%{
# define yyerror sql_error
# ifndef M_UNIX
# define YYSTYPE YYSTYPE
# endif
# include "sql_defs.h"

static char sccsid[] = "@(#) sql_gram.y 1.0 1992-11-29";

PAR_LIST *Actual_View;

#ifdef YYDEBUG
extern int yydebug;

yydebug = 1;
#endif

/* local functions (extension by michael) */
LIST *make_list( YY_IDENT * );
PAR_LIST *make_par_list( PMODE, YY_IDENT *, SQL_TYPE *);
STATEMENT_LIST *make_statement_list( SQL_TEXT *);

%}

%%

description_unit:

    /*
      The token SQL_END is inserted to make it possible to recover from
      syntax errors in the descriptions. Without this token it would be
      not possible to recover from errors in the descriptions because
      the yacc parser driver performs in the position of SQL_END normally
      a default reduction which leads to the situation that the parser 
      goes into state accept and then states are poped from the parser
      stack until a state is reached in that an error token can be find.
    */
      system_specification descriptions SQL_END
    ;

system_specification:
      SYSTEM IDENTIFIER semikolon
      ={
        generate_system_specification( get_symbol( $2.Hash ) );
       }
    | error
      ={
        /* 
           This error rule is also used if the keyword is not followed
           by an identifier. Because in both cases the same correction has
           to be done and illegal tokens can be discarded without any
           resulting new errors it is possible to use the same syntax rule.
         */ 
        sql_information( 
                   "illegal system specification, INFORMIX4.0 is assumed" );
        generate_system_specification( "informix40" );
       }
    ;

descriptions:
      description
    | descriptions description
    ;

description:
      view_specification
    | const_specification
    | statement_specification
    | error
      ={ /* This rule can be reduced because the token SQL_END is
            appended in the rule description_unit */
        sql_information("view, const or statement specification is aspected");
       } 
    ;

/*---- const specification ------------------------------------------------*/

const_specification:
      CONST { yyerrok; } IDENTIFIER ASSIGN constant_expression semikolon
      ={
        TSTAB *Stab;

        Stab = get_stab( $3.Hash );
        if ( Stab == STAB_NULL )
            {
            Stab = insert_Entry( $3.Hash );
            Stab->Kind = S_CONSTANT;
            Stab->Value = $5;
            generate_const_specification( get_symbol( $3.Hash ), $5 );
            }
        else
            sql_error( "redefinition of identifier" );
       }
    | CONST error
      ={
    sql_information( "illegal const specification" );
       }
    ;

constant_expression:
      mul_const_expression
    | constant_expression PLUS mul_const_expression
      ={
        $$ = $1 + $3;
       }
    | constant_expression MINUS mul_const_expression
      ={
        $$ = $1 - $3;
       }
    ;

mul_const_expression:
      unary_const_expression
    | mul_const_expression MUL unary_const_expression
      ={
        $$ = $1 * $3;
       }
    | mul_const_expression DIV unary_const_expression
      ={
        $$ = $1 / $3;
       }
    ;

unary_const_expression:
      primary_const_expression
    | MINUS primary_const_expression
      ={
        $$ = -($2);
       }
    | PLUS primary_const_expression
      ={
        $$ = $2;
       }
    ;

primary_const_expression:
      NUMBER
    | IDENTIFIER
      ={
        TSTAB *Stab;

        Stab = get_stab( $1.Hash );
        if ( Stab->Kind != S_CONSTANT )
            sql_error( "illegal use of identifier in constant expression" );
        else
            $$ = Stab->Value;
       }
    | LP constant_expression RP
      ={
        $$ = $2;
       }
    ;

/*---- cursor specification -----------------------------------------------*/

view_specification:
      VIEW { yyerrok; } IDENTIFIER LP parameters RP
      ={

    if ( verbose_Flag )
      fprintf( Protocolfile, "\nView \"%s\":\n\n", get_symbol( $3.Hash ) );

        /* language interface */

        generate_Language_record( &($3), $5 );
        generate_init_procedure( &($3), $5 );

        /* esql interface */

        Actual_View = $5;
        generate_ESQL_record( &($3), $5 );
        generate_Hostvariables( get_symbol( $3.Hash ), $5 );
        }  AS { $<ident>$= $3; } cursor_specification SM
    | VIEW error
      ={
    sql_information( "illegal view specification" );
       }
    ;

cursor_specification:
      CURSOR cursor_parameter_list FOR cursor_clause
      ={
        generate_object_type( &($<ident>0), $2 , &($4) );
        if ( $2 != PAR_LIST_NULL && $4.Select.type != C_INSERT )
            mark_hostvariables( $2, &($4.Select.text) );
        generate_cursor_functions( get_symbol( $<ident>0.Hash ), $2, &($4) );
       }
    ;

cursor_clause:
      SELECT { SQL_TEXT_Flag = TRUE;
               SELECT_Flag = TRUE; } sql_select update_clause
      ={
        $$.Select.type = $4;
        $$.Select.text = $3;
       }
    | INSERT INTO IDENTIFIER columns
      ={
        $$.Insert.type = C_INSERT;
        $$.Insert.columns = $4;
        $$.Insert.table = $3;
       }
    ;

update_clause:
      ={ 
        $$ = C_SELECT;
       }
    | FOR UPDATE
      ={ 
        $$ = C_UPDATE;
       }
    ;

columns:
      ={
        $$ = LIST_NULL;
       }
    | LP identifier_list RP
      ={
        $$ =$2;
       }
    ;

statement_specification:
      STATEMENT { yyerrok; } IDENTIFIER statement_parameter_list IS
        { SQL_TEXT_Flag = TRUE; } sql_statements
      ={
        generate_Language_statement( &($3), $4 );
        if ( $4 != PAR_LIST_NULL )
            {
            STATEMENT_LIST *Pointer_to_statement; 
            generate_Hostvariables( get_symbol( $3.Hash ), $4 );
            for ( Pointer_to_statement = $7;
                    Pointer_to_statement != STATEMENT_LIST_NULL;
                        Pointer_to_statement = Pointer_to_statement->next )
                mark_hostvariables( $4, &(Pointer_to_statement->statement) );
            }
        generate_ESQL_statement( get_symbol( $3.Hash ), $4, $7 );
       }
    | STATEMENT error
      ={
    sql_information( "illegal statement specification" );
       }
    ;

sql_single_statement:
      sql_statement SM
      ={
        more_SQL_possible = TRUE;
        $$ = make_statement_list( &($1) );
       }
    ;

sql_statements:
      sql_single_statement
    | sql_single_statement sql_statements
      ={
        $$->next = $2;
       }
    ;

cursor_parameter_list:
      ={
        $$ = PAR_LIST_NULL;
       }
    | OPEN WITH LP parameters RP
      ={
        $$ =$4;
       }
    ;

parameters:
      IDENTIFIER SQL_type
      ={
        $$ = make_par_list( M_IN, &($1), &($2) );
       }
    | IDENTIFIER SQL_type CM parameters
      ={
        $$ = make_par_list( M_IN, &($1), &($2) );
        $$->next = $4;
       }
    ;

statement_parameter_list:
      ={
        $$ = PAR_LIST_NULL;
       }
    | LP RP
      ={
        $$ = PAR_LIST_NULL;
       }
    | LP statement_parameters RP
      ={
        $$ =$2;
       }
    ;

statement_parameters:
      mode IDENTIFIER SQL_type
      ={
        $$ = make_par_list( $1, &($2), &($3) );
       }
    | mode IDENTIFIER SQL_type CM statement_parameters
      ={
        $$ = make_par_list( $1, &($2), &($3) );
        $$->next = $5;
       }
    ;

mode:
      ={
        $$ = M_IN;
       }
    | IN
      ={
        $$ = M_IN;
       }
    | OUT
      ={
        $$ = M_OUT;
       }
    | INOUT
      ={
        $$ = M_INOUT;
       }
    ;

identifier_list:
      IDENTIFIER
      ={
        $$ = make_list( &($1) );
       }
    | IDENTIFIER CM identifier_list
      ={
        $$ = make_list( &($1) );
        $$->next = $3;
       }
    ;

SQL_type:
      CHAR
      ={
        $$.Char.typecode = CHAR;
        $$.Char.length = 1;
       }
    | CHAR LP constant_expression RP
      ={
        $$.Char.typecode = CHAR;
        $$.Char.length = $3;
       }
    | VARCHAR LP constant_expression RP
      ={
        if ( $3 > 255 )
            sql_error( "maximum length of varchar exceeded" );
        if ( $3 < 1 )
            sql_error( "maximum length of varchar must be at least 1" );
        $$.VarChar.typecode = VARCHAR;
        $$.VarChar.maxlength = $3;
        $$.VarChar.minlength = 0;
       }
    | VARCHAR LP constant_expression CM constant_expression RP
      ={
        if ( $3 > 255 )
            sql_error( "maximum length of varchar exceeded" );
        if ( $3 < 1 )
            sql_error( "maximum length of varchar must be at least 1" );
        if ( $5 < 0 )
            sql_error( "minimum length of varchar must be at least 0" );
        if ( $5 > $3 )
            sql_error( 
             "maximum length must be greater than or equal minimum length" );
        $$.VarChar.typecode = VARCHAR;
        $$.VarChar.maxlength = $3;
        $$.VarChar.minlength = $5;
       }
    | INTEGER
      ={
        $$.typecode = INTEGER;
       }
    | SERIAL
      ={
        $$.typecode = SERIAL;
       }
    | SMALLINT
      ={
        $$.typecode = SMALLINT;
       }
    | SMALLFLOAT
      ={
        $$.typecode = SMALLFLOAT;
       }
    | FLOAT
      ={
        $$.typecode = FLOAT;
       }
    | DECIMAL
      ={
        $$.Decimal.typecode = DECIMAL;
        $$.Decimal.length = 16;
        $$.Decimal.precision = 2;
       }
    | DECIMAL LP constant_expression RP
      ={
        $$.Decimal.typecode = DECIMAL;
        $$.Decimal.length = $3;
        $$.Decimal.precision = 2;
       }
    | DECIMAL LP constant_expression CM constant_expression RP
      ={
        $$.Decimal.typecode = DECIMAL;
        $$.Decimal.length = $3;
        $$.Decimal.precision = $5;
       }
    | MONEY
      ={
        $$.Decimal.typecode = MONEY;
        $$.Decimal.length = 16;
        $$.Decimal.precision = 2;
       }
    | MONEY LP constant_expression RP
      ={
        $$.Decimal.typecode = MONEY;
        $$.Decimal.length = $3;
        $$.Decimal.precision = 2;
       }
    | MONEY LP constant_expression CM constant_expression RP
      ={
        $$.Decimal.typecode = MONEY;
        $$.Decimal.length = $3;
        $$.Decimal.precision = $5;
       }
    | DATE
      ={
        $$.typecode = DATE;
       }
    | DATETIME datetime_components
      ={
        $$.Interval.typecode = DATETIME;
        $$.Interval.komponent = $2;
       }
    | INTERVAL interval_components
      ={
        $$.Interval.typecode = INTERVAL;
        $$.Interval.komponent = $2;
       }
    ;

datetime_components:
      datetime_simple_time TO time
      ={
        if ( $1 > $3 )
            sql_error( "start must be greater or equal than end" );
        $$ = ($1 << 4) | $3;
       }
    ;

interval_components:
      simple_time simple_time_length TO time
      ={
        if ( ($1 == TU_YEAR || $1 == TU_MONTH) && 
                                ($4 != TU_YEAR && $4 != TU_MONTH) )
            sql_error( "illegal combination of components" );
        if ( ($1 != TU_YEAR && $1 != TU_MONTH) &&
                                ($4 == TU_YEAR || $4 == TU_MONTH) )
            sql_error( "illegal combination of components" );
        if ( $1 > $4 )
            sql_error( "start must be greater or equal than end" );
        $$ = ($2 << 8) | ($1 << 4) | $4;
       }
    | fraction TO time
      {
       $$ = ($1 << 4) | $3;
      }
    ;

simple_time_length:
      ={
        switch( $<number>0 )
            {
            case TU_YEAR:
                $$ = INTERVAL_YEAR();
                break;
            case TU_MONTH:
                $$ = INTERVAL_MONTH();
                break;
            case TU_DAY:
                $$ = INTERVAL_DAY();
                break;
            case TU_HOUR:
                $$ = INTERVAL_HOUR();
                break;
            case TU_MINUTE:
                $$ = INTERVAL_MINUTE();
                break;
            case TU_SECOND:
                $$ = INTERVAL_SECOND();
            }
       }
    | LP NUMBER RP
      ={
        if ( $2 > MAX_INTERVAL_FIRST() )
            {
            sql_warning( "length of first interval component out of range" );
            switch( $<number>0 )
          {
                case TU_YEAR:
                    $$ = INTERVAL_YEAR();
                    break;
                case TU_MONTH:
                    $$ = INTERVAL_MONTH();
                    break;
                case TU_DAY:
                    $$ = INTERVAL_DAY();
                    break;
                case TU_HOUR:
                    $$ = INTERVAL_HOUR();
                    break;
                case TU_MINUTE:
                    $$ = INTERVAL_MINUTE();
                    break;
                case TU_SECOND:
                    $$ = INTERVAL_SECOND();
                }
            }
        else
            $$ = $2;
       }
    ;

datetime_simple_time:
      simple_time
    | fraction
    ;

simple_time:
      YEAR
      ={
        $$ = TU_YEAR;
       }
    | MONTH
      ={
        $$ = TU_MONTH;
       }
    | DAY
      ={
        $$ = TU_DAY;
       }
    | HOUR
      ={
        $$ = TU_HOUR;
       }
    | MINUTE
      ={
        $$ = TU_MINUTE;
       }
    | SECOND
      ={
        $$ = TU_SECOND;
       }
    ;

fraction:
      FRACTION
      ={
        $$ = TU_FRAC;
       }
    ;

time:
      simple_time
    | fraction
    | FRACTION LP constant_expression RP
      ={
        switch ( $3 )
            {
            case 1:
                $$ = TU_F1;
                break;
            case 2:
                $$ = TU_F2;
                break;
            case 3:
                $$ = TU_F3;
                break;
            case 4:
                $$ = TU_F4;
                break;
            case 5:
                $$ = TU_F5;
                break;
            default:
                sql_warning( "fraction precision out of range" );
                $$ = TU_FRAC;
            }
       }
    ;

semikolon:
      SM
    | error
      ={
        sql_information( "missing \';\' inserted" );
        yyerrok;
       }

%%

/*---- MAKE_LIST ----------------------------------------------------------*/

LIST *make_list( ident )
YY_IDENT *ident;
{
    LIST *Pointer_to_new_List;

    Pointer_to_new_List = Get_List();
    Pointer_to_new_List->ident = *ident;
    return( Pointer_to_new_List );
}

/*---- MAKE_PAR_LIST ------------------------------------------------------*/

PAR_LIST *make_par_list( mode, ident, type )
PMODE mode;
YY_IDENT *ident;
SQL_TYPE *type;
{
    PAR_LIST *Pointer_to_new_Par_List;

    Pointer_to_new_Par_List = Get_Par_List();
    Pointer_to_new_Par_List->ident = *ident;
    Pointer_to_new_Par_List->type = *type;
    Pointer_to_new_Par_List->mode = mode;
    return( Pointer_to_new_Par_List );
}

/*---- MAKE_STATEMENT_LIST -------------------------------------------------*/

STATEMENT_LIST *make_statement_list( statement )
SQL_TEXT *statement;
{
    STATEMENT_LIST *Pointer_to_new_Statement_List;

    Pointer_to_new_Statement_List = Get_Statement_List();
    Pointer_to_new_Statement_List->statement = *statement;
    return( Pointer_to_new_Statement_List );
}
