;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Emacs-Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;
;; landESP.el -- language dependent editing support package
;; 
;;   Copyright (C) sietec Systemtechnik GmbH & Co OHG 1993
;;   All rights reserved
;; 
;; AtFSID          : $__Header$
;; Author          : Martin Weber
;; 
;; Created On      : Fri Apr 23 11:14:09 1993
;; Last Modified By: Karl-Heinz Koester
;; Last Modified On: Thu Sep 22 17:34:50 1994
;; Update Count    : 373
;; 
;; HISTORY
;; 4-Jul-1994		Martin Weber TA1 Tel.22260	
;;    Last Modified: Fri Jul  1 17:32:51 1994 #255 (Martin Weber TA1 Tel.22260)
;;    focus key syntax enhanced: landESP expects a symbol representing
;;    the file structure of the file-specific language mode. The symbol defines
;;    a list of the key syntax for the unit followed by a list of symbols
;;    defining subunits of that unit.
;;    focus mechanism enhanced: If the package 'file-part' is available which
;;    is delivered with lucid emacs, you can be focused on multiple units
;;    in parallel, and every focus may have its own screen.
;; 5-Aug-1993		Martin Weber	
;;    Last Modified: Thu Aug  5 13:27:14 1993 #175 (Martin Weber)
;;    landESP-next-error and landESP-previous-error added which replace
;;    next-error and previous-error
;; 30-Jul-1993		Martin Weber	
;;    Last Modified: Fri Jul 30 16:28:06 1993 #163 (Martin Weber)
;;    top level is now dired
;; 22-Jul-1993		Martin Weber	
;;    Last Modified: Thu Jul 22 13:42:25 1993 #149 (Martin Weber)
;;    Added new functionality for focusing: focus on identifier, a 
;;    project file which defines a list of files to be considered, etc.
;; 9-Jul-1993		Lutz Hilken	
;;    Last Modified: Thu Jul  8 17:55:13 1993 #43 (Lutz Hilken)
;;    Now some functions are redefined for use in lucid emacs
;;    (popup-menus for alternatives and focusing)
;; 27-Apr-1993		Martin Weber	
;;    Last Modified: Mon Apr 26 15:26:14 1993 #14 (Martin Weber)
;;    extended dynamic abbreviations added (again!) This file requires 
;;    now dabbrev-ex 
;; 23-Apr-1993		Michael Bouschen	
;;    Last Modified: Fri Apr 23 14:31:16 1993 #5 (Michael Bouschen)
;;    New indentation syntax and functionality.
;; PURPOSE
;; 	
;;  this package provides functions to support language dependent
;;  editing modes. It consists of three parts:
;;  - a focussing part where you can focus on certain units within
;;    a file (or within another unit)
;;  - an indentation part which handles the layout
;;  - a construct printing part used to print construct templates
;;
;;  A detailed description can be found in the file 'landESP.info'
;;
;;  TODO:
;;  Lots of things.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;  OK, here we define some mode specific key bindings
;;

(defun landESP-replace-binding (map f1 f2)
  (mapcar '(lambda (key)
	     (define-key map key f2))
	  (where-is-internal f1)))

(defvar landESP-mode-map nil)
(setq landESP-mode-map
      (let ((map (make-sparse-keymap)))
	(define-key map "\t" 'landESP-indent-line)
	(define-key map "\C-M" 'landESP-electric-indent-line)
	(define-key map "\C-c\t" 'landESP-beautify-region)
	(define-key map "\C-c\C-f" 'landESP-narrow-focus)
	(define-key map "\C-c\C-b" 'landESP-widen-focus)
	(define-key map "\C-c\C-n" 'landESP-next-focus)
	(define-key map "\C-c\C-p" 'landESP-previous-focus)
	(define-key map "\C-c\C-d" 'landESP-direct-focus)
	(define-key map "\C-c\C-i" 'landESP-focus-on-identifier-definition)
	(define-key map "\C-c\C-h" 'landESP-focus-on-history)
	(define-key map "\C-c\C-l" 'landESP-focus-on-last)
	(landESP-replace-binding map 'next-error 'landESP-next-error)
	(landESP-replace-binding map 'previous-error 'landESP-previous-error)
	map))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Before we define the rest we need some functions to make landESP
;; compatible to the various emacses around.
;; Currently supported versions are:
;; - Gnu emacs 19.x
;; - Lucid emacs 19.x
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(cond ((boundp 'epoch::version) ;;; we do not support epoch because
                                ;;; we do not have one around
       (defun landESP-set-screen-size (width height)
	 "Set the size of the current screen to WIDTH and HEIGHT."
	 (if landESP-chaotic-display
	     (change-screen-size width height)))
       (defun landESP-popup-menu (e) t)
       (defun landESP-define-menu (map mode specific-menu) t))
      ((string-match "Lucid" (emacs-version))
       (defun landESP-set-screen-size (width height)
	 "Set the size of the current screen to WIDTH and HEIGHT."
	 (if landESP-chaotic-display
	     (set-screen-size (selected-screen) width height)))
       (defun landESP-define-menu (map mode specific-menu) 
	 (if specific-menu
	     (progn
	       (setq landESP-menu (append landESP-default-menu 
					  specific-menu))
	       (if (and current-menubar 
			(not (assoc mode
				    current-menubar)))
		   (progn
		     (set-buffer-menubar
		      (copy-sequence current-menubar))
		     (add-menu nil mode
			       landESP-menu))))))
       (defun landESP-popup-menu (e)
	 (interactive "e")
	 ;;  (mouse-set-point e)
	 ;;  (beginning-of-line)
	 (popup-menu (cons mode-name landESP-menu)))
       (define-key landESP-mode-map 'button3 'landESP-popup-menu))
      ((string-match "GNU" (emacs-version))
       (defun landESP-set-screen-size (width height)
	 "Set the size of the current screen to WIDTH and HEIGHT."
	 (if landESP-chaotic-display
	     (set-screen-size (selected-screen) width height)))
       (defun landESP-popup-menu (e) t)
       (if (boundp 'menu-bar-help-menu)
	   (progn
	     (defun landESP-define-menu (map mode specific-menu)
	       (if specific-menu
		   (let ((menu (nreverse
				(append (car landESP-default-menu)
					specific-menu)))
			 item)
		     (setq landESP-menu menu)
		     (define-key map [menu-bar] (make-sparse-keymap))
		     (define-key map [menu-bar landESP]
		       (cons mode (make-sparse-keymap mode)))
		     (while menu
		       (setq item (car menu)
			     menu (cdr menu))
		       (if (vectorp item)
			   (define-key map
			     (vector 'menu-bar 'landESP (elt item 1))
			     (cons (elt item 0)
				   (elt item 1))))))
		 )))
	 (defun landESP-define-menu (map mode specific-menu) t)))
      (t
       (defun landESP-set-screen-size (width height)
	 "Do not do anything."
	 t)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; some minor stuff
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; mini-cl requires epoch version 3.2 or later
;; but we use only push and pop from mini-cl so they are included here
;; together with the gensym function and two variables that are needed by pop
;;(require 'mini-cl)
(defvar *gensym-index* 0
  "Integer used by gensym to produce new names.")
(defvar *gensym-prefix* "G$$_"
  "Names generated by gensym begin with this string by default.")
(defun gensym (&optional prefix)
  "Generate a fresh uninterned symbol.
There is an  optional argument, PREFIX.  PREFIX is the
string that begins the new name. Most people take just the default,
except when debugging needs suggest otherwise."
  (if (null prefix)
      (setq prefix *gensym-prefix*))
  (let ((newsymbol nil)
        (newname   ""))
    (while (not newsymbol)
      (setq newname (concat prefix *gensym-index*))
      (setq *gensym-index* (+ *gensym-index* 1))
      (if (not (intern-soft newname))
          (setq newsymbol (make-symbol newname))))
    newsymbol))

(defmacro push (item ref)
  "(push ITEM REF) -> cons ITEM at the head of the g.v. REF (a list)"
  (list 'setq ref (list 'cons item ref)))

(defmacro pop (ref)
  "(pop REF) -> (prog1 (car REF) (setq REF (cdr REF)))"
  (let ((listname (gensym)))
    (list 'let (list (list listname ref))
          (list 'prog1
                (list 'car listname)
                (list 'setq ref (list 'cdr listname))))))

(if (fboundp 'turn-on-font-lock)
    ()
  (defun turn-on-font-lock ()
    (font-lock-mode 1)))
(defvar old-turn-on-font-lock nil)
(fset 'old-turn-on-font-lock (symbol-function 'turn-on-font-lock))

(defvar old-beginning-of-defun nil)
(fset 'old-beginning-of-defun (symbol-function 'beginning-of-defun))

(defvar old-switch-to-buffer nil)
(fset 'old-switch-to-buffer (symbol-function 'switch-to-buffer))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(require 'dabbrev-ex)
(require 'dired)

(defvar landESP-chaotic-display nil
  "*if t try to change the size of a screen according to the size of a focus.
This is not recommended by common user interface conventions.")

(defvar landESP-max-screens 5
  "*This defines the maximum number of screens created by the focus mechanism.
If it is nil there is no maximum.")

(defvar landESP-with-file-part t
  "*if t use make-file-part instead of narrow-focus.")
(condition-case ()
    (require 'file-part)
  (error (setq landESP-with-file-part nil)))

(defvar landESP nil)
(make-variable-buffer-local 'landESP)

(defun landESP-initialize (file-type indent-table recover-expr 
				     &optional map mode menu indent-factor)
  "Function need to be called by every user's mode-specific hook.
FILE-TYPE is the name of the focus expression representing a file.
INDENT-TABLE is the name of your mode-specific indentation table (see
landESP-keywords-indent-table). RECOVER-EXPR is used for
landESP-error-recovery-expression. Optional map, mode and menu are used
to define a menu entry in the menubar; INDENT-FACTOR is the number
of blanks to be inserted for each indentation level (default is 2)."
  (if landESP
      ()
    (let ((entry (list (file-name-nondirectory buffer-file-name)
		       (point-min)
		       (point-max)
		       0
		       (point)
		       (current-buffer)
		       buffer-file-name
		       file-type
		       'closed)))
      (setq landESP-file-type file-type)
      (setq landESP-keywords-indent-table indent-table)
      (setq landESP-error-recovery-expression recover-expr)
      (setq landESP-last-buffer-size (buffer-size))
      (setq landESP-focus-list-stack 
	    (cons (cons entry (list entry)) landESP-focus-list-stack))
      (if indent-factor 
	  (setq landESP-indent indent-factor)))
    (setq landESP t)
    (setq selective-display t)
    (landESP-define-menu map mode menu)
    (setq mode-line-buffer-identification
	  (concat " " 
		  (landESP-focus-name-sans-directory
		   (landESP-get-current-focus))))))
;;  (landESP-print-modeline))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;  functions and vars for focussing
;;

(defvar landESP-last-buffer-size 0
  "Size of the buffer before the last focus change.")
(make-variable-buffer-local 'landESP-last-buffer-size)

(defvar landESP-focus-list-stack nil
  "Holds the focus lists of the actual level (as stack).")
(make-variable-buffer-local 'landESP-focus-list-stack)

(defconst landESP-default-file-type '((("\\`." 0))))

(defvar landESP-file-type 'landESP-default-file-type
  "focus expression representing a file.")
(make-variable-buffer-local 'landESP-file-type)

(defun landESP-get-current-focus()
  (landESP-correct-current-focus nil))

(defun landESP-get-focus-type ()
  "returns the type of the current focus."
  (landESP-type (car (car landESP-focus-list-stack))))

(defun landESP-correct-current-focus (update-history)
  "returns the corrected current focus. If update-history is t the current
focus is added to the focus history."
  (let ((entry (car (car landESP-focus-list-stack))))
    (setq entry (list (landESP-key entry)
		      (landESP-start entry)
		      (landESP-end entry)
		      (landESP-count entry)
		      (point)
		      (current-buffer)
		      (landESP-focus-name entry)
		      (landESP-type entry)
		      (landESP-end-flag entry)))
    (setq landESP-focus-list-stack
	  (cons (cons entry  
		      (cdr (car landESP-focus-list-stack)))
		(cdr landESP-focus-list-stack)))
    (if update-history
	(landESP-history-register 
	 (cons (landESP-focus-name-sans-directory
		(landESP-get-current-focus))
	       (car (car landESP-focus-list-stack)))))
    entry))
	
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; History handling
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar landESP-history nil
  "History for focus actions performed. The last performed focus is
the last element in this list.")

(defvar landESP-history-size 10
  "Maximum number of history entries.")

(defun landESP-history-register (entry)
  "Add ENTRY to the focus history."
  (if (nth landESP-history-size landESP-history)
      (setq landESP-history 
	    (reverse (cdr (reverse landESP-history)))))
  (setq landESP-history (append (list entry) landESP-history)))

(defun landESP-focus-on-history ()
  "Focus on a history entry choosen by the user."
  (interactive)
  (let* ((num 0)
	 (unsorted-list (mapcar '(lambda (elem)
				   (setq num (1+ num))
				   (cons (format "%02d: %s" num (car elem))
					 (cdr elem)))
				(reverse landESP-history))))
    (landESP-correct-current-focus t)
    (if (landESP-focus-on-entry 
	 (cdr (assoc (car (landESP-completing-read 0 unsorted-list))
		     unsorted-list))
	 t)
	(landESP-print-modeline))))

(defun landESP-focus-on-last ()
  "Focus on the last history entry."
  (interactive)
  (let ((length (length landESP-history)))
    (if (< length 1)
	()
      (landESP-correct-current-focus t)
      (landESP-focus-on-entry (cdr (car (cdr landESP-history)))
			      t)
      (landESP-print-modeline))))
  
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; 
;; handling of regular expressions as focus keys
;;
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar alternate-comment-start ""
  "an alternative to comment-start")
(setq-default alternate-comment-start "")
(make-variable-buffer-local 'alternate-comment-start)

(defun OPT (expression &rest rest)
  "converts the EXPRESSION(s)  into an optional expression"
  (concat "\\(" expression 
	  (let ((stringlist rest) (expr ""))
	      (while stringlist
		(setq expr (concat expr (car stringlist)))
		(setq stringlist (cdr stringlist)))
	      expr)
	  "\\)?"))


(defun ALT (expression &rest rest)
  "converts the EXPRESSION(s)  into an expression where the EXPRESSION(s)
are alternatives"
    (concat "\\(" expression 
	    (let ((stringlist rest) (expr ""))
	      (while stringlist
		(setq expr (concat expr "\\|" (car stringlist)))
		(setq stringlist (cdr stringlist)))
	      expr)
	    "\\)"))

(defun PAREN (expression &rest rest)
  "puts EXPRESSION(s) into parentheses"
  (concat "\\(" expression 
	  (let ((stringlist rest) (expr ""))
	      (while stringlist
		(setq expr (concat expr (car stringlist)))
		(setq stringlist (cdr stringlist)))
	      expr)
	  "\\)"))

(defconst WS "[ \t\n\^M]+") ;;; white space, the \^M is needed if we use
                            ;;; selective display

(defconst OPT_WS "[ \t\n\^M]*") ;;; optional white space

(defconst ID "\\b\\w+\\b")   ;;; a word

(defconst landESP-curly-brackets
  '("{" nil "}"))

(defconst landESP-brackets
  '("(" nil ")"))

(defconst landESP-braces
  '("[" nil "]"))

(defun landESP-search-next-expr (expr)
  "Search forward until EXPR is found and point is not within comment
or string. Return point if found, nil otherwise."
  (let (stop found)
    (while (not stop)
      (setq found 
	    (re-search-forward expr nil 1))
      (if found
	  (if (or (landESP-in-string-or-comment (match-beginning 0))
		  (landESP-in-string-or-comment (point)))
	      (setq found nil)
	    (setq stop t))
	(setq stop t)))
    (if found
	(point)
      nil)))

(defun landESP-search-previous-expr (expr)
  "Search backward until EXPR is found and point is not within comment
or string. Return point if found, nil otherwise."
  (let (stop found)
    (while (not stop)
      (setq found 
	    (re-search-backward expr nil 1))
      (if found
	  (if (or (landESP-in-string-or-comment (match-end 0))
		  (landESP-in-string-or-comment (point)))
	      (setq found nil)
	    (setq stop t))
	(setq stop t)))
    (if found
	(point)
      nil)))

(defun landESP-in-string-or-comment (pos)
  "returns t if the position is within a string or a comment."
  (save-excursion
    (let (context last-match)
      (setq last-match (match-data))
      (if (fboundp 'buffer-syntactic-context)
	  (progn
	    (goto-char pos)
	    (setq context (buffer-syntactic-context))
	    (store-match-data last-match)
	    (or (eq context 'string)
		(eq context 'comment)
		(eq context 'block-comment)))
	(prog1
	    (or (landESP-in-string pos) (landESP-in-comment pos))
	  (store-match-data last-match))))))

(defun landESP-in-comment (pos)
  "Return t if pos is within comment, nil otherwise."
  (save-excursion
    (goto-char pos)
    (if (re-search-backward (ALT comment-start alternate-comment-start) nil 1)
	(progn
	  (forward-comment 1)
	  (< pos (point))))))
  
(defun landESP-in-string (pos)
  "Return t if pos is within string,, nil otherwise."
  (save-excursion
    (goto-char pos)
    (if (re-search-backward "\\s'" nil 1)
	(progn
	  (re-search-forward "\\s'" nil 1)
	  (< pos (point))))))
  
(defun landESP-make-focus-list (entry)
  "compute a list of all subunits of this entry."
  (save-excursion
    (let (focus-lists subunits)
      (setq subunits (cdr (eval (landESP-type entry))))
      (setq focus-lists (mapcar 'landESP-make-expr-list subunits))
      (landESP-merge-focus-lists focus-lists))))

(defun landESP-make-expr-list (entry-type)
  "This function returns a list of focus entries of ENTRY-TYPE.
ENTRY-TYPE is a list consisting of the focus matching expression
and the subunit types of that entry."
  (goto-char (point-min))
  (let* ((expr (car (eval entry-type)))
	 (expr-data (landESP-search-expr expr entry-type))
	 entry-list)
    (while expr-data
      (let ((unit-start (nth 0 expr-data))
	    (unit-end (1- (nth 1 expr-data)))
	    (key-start (nth 2 expr-data))
	    (key-end (nth 3 expr-data))
	    (mark (nth 4 expr-data))
	    (end-flag (nth 5 expr-data)))
	(setq expr-data (landESP-search-expr expr entry-type))
	(setq entry-list 
	      (append entry-list
		      (list (list (buffer-substring key-start key-end) 
				  unit-start
				  unit-end
				  0
				  (marker-position mark)
				  (marker-buffer mark)
				  (concat 
				   (landESP-focus-name
				    (landESP-get-current-focus))
				   "|" (buffer-substring key-start key-end))
				  entry-type
				  end-flag))))))
    entry-list))

(defun landESP-merge-focus-lists (focus-lists)
  "This function merges the FOCUS-LISTS, sorting them after the start
position of the focus entries and changing the end position of the focus
entries so that there is no space between entries. The result is a sorted
list of focus entries."
  (let ((entry-list (apply 'append focus-lists))
	(count 0)
	(last-entry-end (1- (point-min)))
	focus-list entry)
    (setq entry-list (sort entry-list 'landESP-entry-before))
    (while entry-list
      (setq entry (car entry-list))
      (setq entry-list (cdr entry-list))
      (while (and entry-list
		  (> (landESP-end entry)
		     (landESP-start (car entry-list))))
	(setq entry-list (cdr entry-list)))
      (setq entry (list (landESP-key entry)
			(1+ last-entry-end)
			(setq last-entry-end
			      (cond ((equal (landESP-end-flag entry)
					    'closed)
				     (landESP-end entry))
				    (entry-list
				     (1- (landESP-start (car entry-list))))
				    (t
				     (point-max))))
			count
			(landESP-position entry)
			(landESP-buffer entry)
			(landESP-focus-name entry)
			(landESP-type entry)
			(landESP-end-flag entry)))
      (setq count (1+ count))
      (setq focus-list (append focus-list (list entry))))
    focus-list))

(defun landESP-entry-before (e1 e2)
  "Returns t if the position of e1 is before e2, nil otherwise."
  (< (landESP-start e1) (landESP-start e2)))

(defun landESP-search-expr (expr-list entry-type)
  "Searches for next match with EXPR-LIST. ENTRY-TYPE is the type of
entries we are looking for. If a match is found point is
at the end of the match and the function returns a list of six items:
start position of the match, end position of the match, start position
of the ID within the match, end position of the ID, the pointer-marker,
and a symbol showing whether the part has a well defined end or whether 
its end is defined by the start position of the next part.
If no match is found nil is returned and point is at point-max.
The EXPR-LIST contains two items: A list of a regular expression and a
number indicating the part of that expression which
will form the ID, and a list where each item may be one of the following:
  <ID         represents the start position of the ID
  ID>         represents the end position of the ID
  `(END ..)'  a list of two items where the first item is the symbol END 
              and the second is an expression matching the end of a part.
  a list      is a list of two items and represents an enclosing bracket;
              the two items are regular expressions matching the open and
              close bracket, respectively. This item is matched by an open
              bracket followed by whatever followed by a close bracket
              that matches the open bracket.
  a string    represents a regular expression to be matched. This not allowed
              as the first item in the list!

If an item in the list is a list representing a bracket, the first 
regular expression before the bracket expression must end with the
first characters of the corresponding open bracket.
A match is found if the regular expression is matched by the first part
of the match, if the following list items are each matched by
consecutive parts of the match, if the optional END-expression can be found
somewhere after the last matching part, if all following items are each
matched by consecutive parts of the match, and if the function defined by the 
variable landESP-confirm-entry and called with start, end, id-start, id-end 
and entry-type as arguments returns t."
  (let ((end-matched 'open)
	found id-start id-end start end)
    (if expr-list
	(let ((start-expr (car (car expr-list)))
	      (id-match (car (cdr (car expr-list))))
	      (rest-expr (cdr expr-list)))
	  (while (and 
		  (not found)
		  (progn
;		    (forward-comment 1)
		    (landESP-search-next-expr start-expr)))
	    (setq start (match-beginning 0))
	    (setq id-start (match-beginning id-match))
	    (setq end (match-end 0))
	    (setq id-end (match-end id-match))
	    (if (not rest-expr)
		(setq found end)
	      (setq found (landESP-match-expr rest-expr)))
	    (if found
		(progn
		  (setq end found)
		  (setq found 
			(funcall landESP-confirm-entry 
				 start end id-start id-end entry-type)))))
	  (if found
	      (prog1
		  (list start end id-start id-end (point-marker) end-matched)
		(goto-char end)))))))

(defvar landESP-confirm-entry '(lambda (start end id-start id-end type) t))
(make-variable-buffer-local 'landESP-confirm-entry)

(defun landESP-match-expr (expr-list)
  "If we are looking at a string matched by EXPR-LIST, the function returns 
the position after the matched string."
  (save-excursion
    (let ((start-expr (car expr-list))
	  (rest-expr (cdr expr-list))
	  (end (point)))
      (cond ((eq start-expr '<ID)
	     (landESP-skip-comments)
	     (setq id-start (point)))
	    ((eq start-expr 'ID>)
	     (setq id-end (point)))
	    ((landESP-is-bracket start-expr)
	     (setq end (landESP-match-bracket (landESP-open-bracket start-expr)
					      (landESP-bracket-expr start-expr)
					      (landESP-close-bracket start-expr))))
	    ((landESP-is-end-expr start-expr)
	     (setq end-matched 'closed) ;;; variable defined in dynamic context
	     (setq end (landESP-search-next-expr (car (cdr start-expr)))))
	    (start-expr
	     (landESP-skip-comments)
	     (if (looking-at start-expr)
		 (progn
		   (goto-char (match-end 0))
		   (setq end (point)))
	       (setq end nil))))
      (if end
	  (if rest-expr
	      (progn
		(goto-char end)
		(setq end (landESP-match-expr rest-expr)))
	    end)))))

(defun landESP-skip-comments ()
  (let ((max (point-max)))
    (while (and (< (point) max) 
		(forward-comment 1))))) ;;; skip comments

(defun landESP-match-bracket (open expression close)
  "The function returns the position after the match if we are looking
at EXPRESSION followed by anything, enclosed in OPEN and CLOSE expressions
and point is immediately after the first character of the OPEN expression. 
If EXPRESSION is nil anything between the brackets is allowed."
  (save-excursion
    (if (re-search-backward open nil t)
	;; find matching close
	(let ((open-close (concat "\\(" open "\\|" close "\\)"))
	      (open-count 1)
	      (found t))
	  (goto-char (match-end 0))
	  (if expression
	      (progn
		(setq found (landESP-match-expr expression))
		(if found
		    (goto-char found))))
	  (while (and (> open-count 0) found)
	    (setq found (landESP-search-next-expr open-close))
	    (if found
		(save-excursion
		  (goto-char (match-beginning 0))
		  (if (looking-at open)
		      (setq open-count (1+ open-count))
		    (setq open-count (1- open-count))))))
	  (if found
	      (point)
	    nil)))))

(defun landESP-is-bracket (expr)
  (and (listp expr) (= (length expr) 3)))

(defun landESP-is-end-expr (expr)
  (and (listp expr) (equal (car expr) 'END)))

(defun landESP-open-bracket (expr)
  (car expr))

(defun landESP-bracket-expr (expr)
  (car (cdr expr)))

(defun landESP-close-bracket (expr)
  (car (cdr (cdr expr))))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; 
;; focussing
;;
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defconst landESP-default-screen-width 80
  "The default width of a screen for a new focus.")
(defconst landESP-default-screen-height 45
  "The default height of a screen for a new focus.")

;; A focus is defined by an `entry'. An entry is a list of 8 items, defining
;; 10 properties. Each of the following functions returns one property
(defun landESP-key (entry)       ;; returns the KEY of an entry. The key
  (nth 0 entry))                 ;; is a string identifying the entry
(defun landESP-start (entry)     ;; returns the start position of an entry
  (nth 1 entry))
(defun landESP-end (entry)       ;; returns the end position of an entry
  (nth 2 entry))
(defun landESP-count (entry)     ;; returns the index of the entry within
  (nth 3 entry))                 ;; the list of entries on the same level
(defun landESP-position (entry)  ;; returns the position of point in the entry
  (nth 4 entry))
(defun landESP-buffer (entry)    ;; returns the entry's buffer
  (nth 5 entry))
(defun landESP-focus-name (entry);; returns the entry's focus-name. It is
  (nth 6 entry))                 ;; the concatenation of the file and the
                                 ;; keys of all entries in the focus hierarchy,
                                 ;; separated by pipes `|'
(defun landESP-focus-name-sans-directory (entry)      ;; focus-name without the
  (if (string-match ".*\/" (landESP-focus-name entry));; file's path
      (substring (landESP-focus-name entry) (match-end 0))
    (landESP-focus-name entry)))
(defun landESP-file (entry)      ;; returns the file containing entry
  (if (string-match "[^|]*|" (landESP-focus-name entry))
      (substring (landESP-focus-name entry) 0 (1- (match-end 0)))
    (landESP-focus-name entry)))
(defun landESP-type (entry)      ;; returns entry's type as a symbol
  (nth 7 entry))
(defun landESP-end-flag (entry)  ;; indicates whether the key expression
  (nth 8 entry))                 ;; used during parsing defines the whole
                                 ;; entry or only its start

(defun landESP-correct-focus-entry (pos amount entry)
  (list (landESP-key entry)
	(+ (if (> (landESP-start entry) pos)
	       amount
	     0)
	   (landESP-start entry))
	(+ (if (> (landESP-end entry) pos)
	       amount
	     0)
	   (landESP-end entry))
	(landESP-count entry)
	(landESP-position entry)
	(landESP-buffer entry)
	(landESP-focus-name entry)
	(landESP-type entry)
	(landESP-end-flag entry)))
		     
(defun landESP-correct-focus-list (pos amount focus-list)
  "amount characters have been inserted (deleted) after position pos.
Therefor all entries of the focus-list behind the first entry with
its start position (car (cdr entry)) > pos have to be corrected, i.e.
the start position <- start position + amount."
  (if focus-list
      (cons (landESP-correct-focus-entry pos amount (car focus-list))
	    (landESP-correct-focus-list pos amount (cdr focus-list)))))

(defun landESP-correct-focus-list-stack (pos amount stack)
  (if stack
      (cons (landESP-correct-focus-list pos amount (car stack))
	    (landESP-correct-focus-list-stack pos amount (cdr stack)))))

(defun landESP-correct-focus ()
  "Actualize the focus lists."
  (if (not (= landESP-last-buffer-size (buffer-size)))
      (let ((amount (- (buffer-size) landESP-last-buffer-size)))
	(setq landESP-last-buffer-size (buffer-size))
	(setq landESP-focus-list-stack
	      (landESP-correct-focus-list-stack 
	       (landESP-start (car (car landESP-focus-list-stack)))
	       amount 
	       landESP-focus-list-stack)))))

(defun landESP-where-in-focus-list (pos focus-list)
  "looks through the list and returns element where pos lies between
start and end of that element."
  (let ((list focus-list))
    (while (and list 
		(not (and (<= (landESP-start (car list)) pos)
			  (>= (landESP-end (car list)) pos))))
      (setq list (cdr list)))
    (if list 
	(car list)
      nil)))

(defun landESP-show-focus ()
  "Print the current focus hierarchie."
  (interactive)
  (message (landESP-focus-name-sans-directory (landESP-get-current-focus))))

(defun landESP-print-modeline ()
  "This function is called whenever a focus is shown. It resizes the screen
and updates the modeline, but only if landESP-print-modeline-flag is not
defined in the dynamic context. Usually functions define this flag if they
want to change the focus only as intermediate steps without affecting
any screen."
  (if (boundp 'landESP-print-modeline-flag)
      ()
    (landESP-set-screen-size
     landESP-default-screen-width
     (min landESP-default-screen-height
	  (+ 4 (count-lines 
		(point-min) (point-max)))))
    (setq mode-line-buffer-identification
	  (concat " " 
		  (landESP-focus-name-sans-directory
		   (landESP-get-current-focus))))
    (force-mode-line-update)))

(defun substring-is-in-list (name alist)
  "if name is substring of a list element, return the element; else nil"
  (if alist
      (if (string-match name (car (car alist)) nil)
	  (car (car alist))
	(substring-is-in-list name (cdr alist)))
    nil))

(defun landESP-completing-read (entry &optional list)
  "The same as completing-read, but with no function bound to SPACE."
  (save-excursion
    (let ((alist (or list (landESP-make-focus-list entry))))
      (if alist
	  (list (completing-read "Unit name: " alist nil t)
		alist)
	(error "No units found!")))))

(defun landESP-make-file-part-focus (entry pos new-screen)
  "make a file-part containing ENTRY, with point at POS. If NEW-SCREEN
is t create a new screen."
  (let ((start (landESP-start entry))
	(end (landESP-end entry))
	(mode major-mode)
	(old-mode-name mode-name)
	(old-file-type landESP-file-type)
	(master-buffer (landESP-buffer entry))
	(part-buffer (get-buffer
		      (landESP-focus-name-sans-directory entry))))
    (if part-buffer
	(switch-to-buffer part-buffer)
      (make-file-part start end 
		      (landESP-focus-name-sans-directory entry)
		      master-buffer)
      (setq part-buffer (current-buffer))
      (setq buffer-file-name
	    (landESP-focus-name-sans-directory entry))
      (funcall mode)
      (setq mode-name old-mode-name)
      (setq file-type old-file-type))
    (goto-char (1+ (- pos start)))
    (if (and new-screen 
	     (or (not landESP-max-screens)
		 (< (length (screen-list)) landESP-max-screens)))
	(progn
	  (switch-to-buffer master-buffer)
	  (switch-to-buffer-new-screen part-buffer)
	  (landESP-set-screen-size
	   landESP-default-screen-width
	   (min landESP-default-screen-height
		(+ 4 (count-lines 
		      (point-min) (point-max)))))
	  ))))
  
(defun landESP-focus (new-screen stay focus-stack)
  "When called all information for the focus must already have been set.
FOCUS-STACK contains the focus stack where the current entry is caar of
that stack. If STAY is t leave point where it is if possible.
If NEW-SCREEN is t create a new screen for the focus.
If landESP-with-file-part is nil ignore NEW-SCREEN and just narrow to region.
Otherwise call make-file-part for the new focus.
At the end set landESP-focus-list-stack to focus-stack."
  (let ((entry (car (car focus-stack))))
    (if entry
	(let ((start (landESP-start entry))
	      (end   (landESP-end entry))
	      (pos))
	  (setq pos (if (or (< end (point)) (< (point) start) (not stay))
			start
		      (point)))
	  (if landESP-with-file-part
	      (landESP-make-file-part-focus entry pos new-screen)
	    (narrow-to-region start end)
	    (goto-char pos))
	  (setq landESP-focus-list-stack focus-stack)
	  (landESP-correct-current-focus nil)
	  t))))


(defun landESP-focus-extern ()
  "Focusses on the pattern contained in the file $HOME/.Emacs-focus."
  ;(interactive) ;; implemented later...
  ;(setq landESP-focus-list-stack '((("FILE" 1 1))))
  ;(widen)
  (let ((buffername (buffer-name)) beg alist name)
    (generate-new-buffer "*function*")
    (set-buffer "*function*")
    (shell-command "cat $HOME/.Emacs-focus" t)
    (while (not (= (point) (point-max)))
      (setq beg (point))
      (end-of-line 1)
      (setq name (buffer-substring beg (point)))
      (forward-line 1)
      (save-excursion
	(set-buffer buffername)
	(setq alist (landESP-make-focus-list (landESP-get-current-focus)))
	(if (or (not alist) (string= name ""))
	    nil
	  (landESP-narrow-focus (substring-is-in-list name alist) alist)))
      (set-buffer "*function*"))
    (kill-buffer "*function*")
    (set-buffer buffername))
  (goto-char (point-min)))

(defun landESP-narrow-focus (name &optional focus-list)
  "Narrows the current focus to a subunit. A list of subunits is
computed and the user is asked to select one. If he just presses
return the subunit where the cursor points to is selected. If the
cursor points to no subunit the first subunit is selected."
  (interactive (landESP-completing-read (landESP-get-current-focus)))
  (if name
      (progn
	(landESP-correct-current-focus t)
	(landESP-narrow-focus-intern name focus-list t)
	(landESP-print-modeline))
    (error "You are already at bottom level!")))

(defun landESP-narrow-focus-intern (name focus-list new-screen)
  "Narrows the current focus to the entry in focus-list with key=name. 
If name is the empty string then focus on the entry where point is 
currently located. But do not tell the user about the new focus."
  (let (stay stack)
    (landESP-correct-focus)
    (setq stack 
	  (cons
	   (cons (if (string< "" name)
		     (assoc name focus-list)
		   (setq stay t)    ;nothing selected: focus
					;on the unit we are in and don't
					;move the cursor
		   (cond ((landESP-where-in-focus-list (point) focus-list))
			 ((car focus-list)))) ; select first unit
		 focus-list)
	   landESP-focus-list-stack))
    (landESP-focus new-screen stay stack)))

(defun landESP-narrow-to-point ()
  "Focus to the unit under point."
  (interactive)
  (landESP-correct-current-focus t)
  (if (landESP-narrow-to-point-intern t)
      (landESP-print-modeline)
    (error "You are already at bottom level!")))

(defun landESP-narrow-to-point-intern (new-screen)
  "Focus to the unit under point, but do not tell the user."
  (let ((focus-list (save-excursion
		      (landESP-make-focus-list (landESP-get-current-focus))))
	stack)
    (if focus-list
	(progn
	  (landESP-correct-focus)
	  (setq stack 
		(cons
		 (cons (cond ((landESP-where-in-focus-list (point) focus-list))
			     ((car focus-list))) ; select first unit
		       focus-list)
		 landESP-focus-list-stack))
	  (landESP-focus new-screen t stack))
      nil)))

(defun landESP-direct-focus ()
  "Move focus to another unit on this level."
  (interactive)
  (landESP-correct-current-focus t)
  (if (eq (landESP-type (landESP-get-current-focus)) landESP-file-type)
      (landESP-find-file)
    (landESP-widen-focus-intern nil)
    (let* ((read-focus (landESP-completing-read (landESP-get-current-focus)))
	   (name (car read-focus))
	   (focus-list (car (cdr read-focus))))
      (if name
	  (progn
	    (landESP-narrow-focus-intern name focus-list nil)
	    (landESP-print-modeline))
	(error "You are already at bottom level!")))))


(defun landESP-widen-focus ()
  "Widen the current focus to the surrounding unit."
  (interactive)
  (landESP-correct-current-focus t)
  (if (eq (landESP-type (landESP-get-current-focus)) landESP-file-type)
      (landESP-dired-project)
    (landESP-widen-focus-intern t)
    (landESP-print-modeline)))

(defun landESP-widen-focus-intern (remove-screen)
  "Widen the current focus to the surrounding unit, but do not tell the user."
  (interactive)
  (if landESP-with-file-part
      (landESP-exit-focus remove-screen)
    (widen)
    (pop landESP-focus-list-stack))
  (landESP-correct-focus)
  (if landESP-with-file-part
      nil
    (if (eq (landESP-type (landESP-get-current-focus)) landESP-file-type)
	()
      (landESP-focus nil t
		     landESP-focus-list-stack))))

(defun landESP-exit-focus (remove-screen)
  "saves and then deletes the buffer and the screen containing the current
focus"
  (let* ((pos (1- (+ (landESP-start (landESP-get-current-focus)) (point))))
	 (master-buffer (extent-buffer file-part-master-extent))
	 (master-window (get-buffer-window master-buffer t))
	 (master-screen (if master-window 
			    (window-screen master-window))))
    (save-buffer)
    (kill-buffer nil)
    (if remove-screen
	(progn
	  (if (> (length (screen-list)) 1)
	      (delete-screen))
	  (if master-screen
	      (progn
		(select-screen master-screen)
		(raise-screen master-screen)))))
    (switch-to-buffer master-buffer)
    (goto-char pos)))

(defun landESP-next-focus ()
  "Move focus to the next unit on this level."
  (interactive)
  (landESP-correct-current-focus t)
  (if (landESP-move-focus 1)
      (landESP-print-modeline)
    (error "no next unit found!")))

(defun landESP-previous-focus ()
  "Move focus to the previous unit on this level."
  (interactive)
  (landESP-correct-current-focus t)
  (if (landESP-move-focus -1)
      (landESP-print-modeline)
    (error "no previous unit found!")))

(defun landESP-move-focus (count)
  "Move focus to the COUNT'th next unit on this level."
  ;; If we are focused on a file we use the following mechanism:
  (if (eq (landESP-type (landESP-get-current-focus)) landESP-file-type)
      (let* ((known-files (landESP-file-list))
	     (pos (landESP-position-of-key known-files buffer-file-name))
	     neighbor)
	(if (zerop pos)
	    (error (format "File %s not in your project (see %s)."
			   buffer-file-name landESP-project-file))
	  (setq neighbor (nth (+ (1- pos) count) known-files))
	  (if (and neighbor (>= (+ (1- pos) count) 0))
	      (progn
		(find-file (car neighbor))
		(while (not (eq (landESP-type (landESP-get-current-focus)) 
				landESP-file-type))
		  (landESP-widen-focus-intern nil))
		t)
	    nil)))
    (landESP-correct-focus)
    ;; We are not focused on a file. So continue here:
    (let* ((index (+ count 
		     (landESP-count
		      (car (car landESP-focus-list-stack)))))
	   (new-focus (nth index
			   (cdr (car landESP-focus-list-stack))))
	   stack)
      (if (and (>= index 0) new-focus)
	  (progn
	    (setq stack 
		  (cons (cons new-focus
			      (cdr (car landESP-focus-list-stack)))
			(cdr landESP-focus-list-stack)))
	    (if landESP-with-file-part
		(landESP-exit-focus nil)
	      (widen))
	    (landESP-focus nil nil stack))
	nil))))

(defun landESP-nth-focus (pos)
  "Move focus to the POS'th unit - starting with 1 - on this level.
If POS is zero or negative focus on the \(last minus POS\)th unit."
  ;; if we are focused on a file we use the following mechanism:
  (if (eq (landESP-type (landESP-get-current-focus)) landESP-file-type)
      (let* ((known-files (landESP-file-list))
	     (index (if (> pos 0) pos (- (length known-files) pos)))
	     new)
	(setq new (nth (1- index) known-files))
	(if (and new (>= (1- index) 0))
	    (progn
	      (find-file (car new))
	      (while (not (eq (landESP-type (landESP-get-current-focus)) 
			      landESP-file-type))
		(landESP-widen-focus-intern nil))
	      t)
	  nil))
    ;; we are not focused on a file. So continue here:
    (landESP-correct-focus)
    (let* ((index (if (> pos 0) pos 
		    (- (length (cdr (car landESP-focus-list-stack)))
		       pos)))
	   (new-focus (nth (1- index) (cdr (car landESP-focus-list-stack))))
	   stack)
      (if new-focus
	  (progn
	    (setq stack 
		  (cons (cons new-focus
			      (cdr (car landESP-focus-list-stack)))
			(cdr landESP-focus-list-stack)))
	    (if landESP-with-file-part
		(landESP-exit-focus nil)
	      (widen))
	    (landESP-focus nil nil stack))
	nil))))

(defun landESP-position-of-key (alist key)
  (let ((key-pos 0)
	(found nil))
    (while (and alist (not found))
      (if (string= key (car (car alist)))
	  (setq found t)
	(setq alist (cdr alist)))
      (setq key-pos (1+ key-pos)))
    (if found key-pos 0)))

(defun landESP-select-entry-buffer (entry)
  "Find the file of ENTRY and - if we are working with file-part - select 
the innermost file-part containing the ENTRY."
  (if (buffer-name (landESP-buffer entry))
      (switch-to-buffer (landESP-buffer entry))
    (find-file (landESP-file entry))
    (if landESP-with-file-part
	(let* ((master-buffer-name (landESP-focus-name-sans-directory entry))
	       (master-buffer (get-buffer master-buffer-name)))
	  ;; To find the innermost focus containing ENTRY we use the
	  ;; focus-name of ENTRY. It is a concatenation of the names of all
	  ;; focuses in the focus hierarchy with the name of ENTRY being the
	  ;; last one. The names are separated by a pipe `|`, hence the weird
	  ;; string-match below.
	  ;; Each focus's buffer has the same name as the focus it contains.
	  ;; So we just look for buffers with the rigth name, starting with
	  ;; the ocus-name of ENTRY.
	  (while (and (not master-buffer)
		      (string-match ".*|" master-buffer-name))
	    (setq master-buffer-name
		  (substring master-buffer-name 0 (1- (match-end 0)))
		  master-buffer (get-buffer master-buffer-name)))
	  (if master-buffer
	      (switch-to-buffer master-buffer))))))

(defun landESP-focus-on-entry (entry new-screen)
  "Move focus to focus of entry."
  (let ((old-buffer (current-buffer)))
    (if (not entry)
	t
      (landESP-correct-focus)
      (landESP-select-entry-buffer entry)
      (if (equal (car (car landESP-focus-list-stack)) entry)
	  t
	;; Hmmm, we wanted to be focused on the innermost focus containing
	;; ENTRY, ideally on ENTRY itself. But obviously we are focused
	;; too far down in the focus hierarchy - this can happen if we do not
	;; work with-file-part. So we have to climb up through the focus
	;; hierarchy til we are focused on ENTRY or on the innermost focus
	;; containing ENTRY:
	(while (or (> (length (landESP-focus-name (landESP-get-current-focus)))
		      (length (landESP-focus-name entry)))
		   (not (string= (landESP-focus-name
				  (landESP-get-current-focus))
				 (substring (landESP-focus-name entry)
					    0
					    (length
					     (landESP-focus-name
					      (landESP-get-current-focus)))))))
	  (landESP-widen-focus-intern nil))
	;; now, if we are focused on ENTRY we are done...
	(if (string= (landESP-focus-name
		      (landESP-get-current-focus))
		     (landESP-focus-name entry))
	    t ;; OK we are focused
	  ;; still focused on a surrounding focus, so let us step down
	  ;; the hierarchy until we are focused on ENTRY:
	  (let ((key-rest (substring 
			   (landESP-focus-name entry)
			   (1+ (length (landESP-focus-name
					(landESP-get-current-focus))))))
		key index)
	    (while (not (string= (landESP-focus-name
				  (landESP-get-current-focus))
				 (landESP-focus-name entry)))
	      (setq index (string-match "|" key-rest))
	      (if index
		  (setq key (substring key-rest 0 index)
			key-rest (substring key-rest (1+ index)))
		(setq key key-rest))
	      (if (landESP-narrow-focus-intern
		   key
		   (landESP-make-focus-list (landESP-get-current-focus))
		   nil)
		  t
		(error "internal error in focus-on-entry.")))))
	(if (landESP-buffer entry)
	    (goto-char (landESP-position entry))))
      ;; Yup, we did it! At last we are focused on ENTRY. Just let us see
      ;; if we have to show it on a new screen:
      (if (and new-screen
	       landESP-with-file-part
	       (or (not landESP-max-screens)
		   (< (length (screen-list)) landESP-max-screens)))
	  (let ((buffer (current-buffer)))
	    (switch-to-buffer old-buffer)
	    (switch-to-buffer-new-screen buffer)
	    (landESP-set-screen-size
	     landESP-default-screen-width
	     (min landESP-default-screen-height
		  (+ 4 (count-lines 
			(point-min) (point-max)))))
	    ))
      t)))

(defun landESP-focus-on-expr 
  (buffer expr &optional entry-type take-first new-screen)
  "Move focus in buffer BUFFER to unit \(of type entry-type\) where EXPR
is located. If TAKE-FIRST is t just use the first entry found, otherwise
ask the user. If NEW-SCREEN is t create a new screen for the focus."
  (landESP-correct-focus)
  (let ((start-entry (landESP-get-current-focus)))
    (switch-to-buffer buffer)
    ;; the focus could be a part of this buffer, so climb up the focus
    ;; hierarchy until we are focused on the file:
    (while (not (equal (landESP-type (landESP-get-current-focus))
		       landESP-file-type))
      (landESP-widen-focus-intern nil))
    (goto-char (point-min))
    ;; Now try to find EXPR and focus on it
    (if (and (re-search-forward expr nil t)
	     (landESP-focus-down-on-expr (point) expr entry-type
					 take-first new-screen))
	start-entry
      (landESP-focus-on-entry start-entry nil)
      nil)))

(defun landESP-focus-down-on-expr 
  (pos expr &optional entry-type take-first new-screen)
  "Focus on unit in the current focus after position POS
with EXPR as subexpression: 
if TAKE-FIRST  is t take the first unit found, else take any and ask the
user whether she wants to focus on it or not. If entry-type is not nil
focus only on units of that type."
  (let* ((old-entry (landESP-get-current-focus))
	 (alist (landESP-make-focus-list old-entry))
	 (whole-list alist)
	 found)
    ;; skip focus units before point
    (while (and alist (< (landESP-end (car alist)) pos))
      (setq alist (cdr alist)))
    (while (and alist (not (string= expr "")) (not found))
      (let ((focus (car alist)))
	;; if EXPR matches the key of the focus and if the focus is of
	;; the right type then we have found a canditate.
	(if (and (string-match expr (landESP-key focus))
		 (if entry-type
		     (eq entry-type (landESP-type focus))
		   t))
	    (progn
	      (landESP-narrow-focus-intern (landESP-key focus)
					   whole-list new-screen)
	      (setq found 
		    (or take-first
			(y-or-n-p 
			 (concat "Focus: "
				 (landESP-focus-name-sans-directory
				  (landESP-get-current-focus))))
			(landESP-focus-down-on-expr pos expr 
						    entry-type
						    take-first
						    new-screen)))
	      (if found () (landESP-widen-focus-intern new-screen)))
	  ;; we have not yet found the right focus. But maybe it is
	  ;; a part of focus:
	  (landESP-narrow-focus-intern (landESP-key focus) 
				       whole-list new-screen)
	  (setq found (landESP-focus-down-on-expr pos expr entry-type
						  take-first new-screen))
	  (if found () (landESP-widen-focus-intern new-screen)))
	;; If we have not yet found the right focus then try the next one
	(if found
	    ()
	  (setq alist (cdr alist)))))
    found))
    
(defun landESP-global-focus-on-expr
  (expr &optional files entry-type take-first new-screen)
  "Focus on entries where EXPR is a subexpression of
the entry's key. If FILES is not nil it must be a list of files to look
through; otherwise the landESP-file-list is used. If ENTRY-TYPE is not
nil the found focus must also be of that type. If TAKE-FIRST is nil and
there are more than one entries ask the user."
  (let ((start-entry (landESP-get-current-focus))
	(file-list (or files (landESP-file-list)))
	new-buffer
	found
	landESP-print-modeline-flag) ;;; DO NOT REMOVE THIS!!
    ;; browse through all files and try to focus on EXPR in each file.
    (while (and file-list (not found))
      (if (file-readable-p (car (car file-list)))
	  (progn
	    (setq new-buffer (find-file-noselect (car (car file-list))))
	    (if (landESP-focus-on-expr new-buffer expr
				       entry-type take-first new-screen)
		(setq found t)
	      (setq file-list (cdr file-list))))
	(setq file-list (cdr file-list))))
    (if found 
	start-entry
      (landESP-focus-on-entry start-entry nil)
      (message "unit not found.")
      nil)))

(defun landESP-focus-on-identifier-definition ()
  "Focus on entries where the identifier under point is a subexpression of
the entry's key. If there are more than one entries ask the user."
  (interactive)
  (landESP-correct-current-focus t)
  (let ((id (buffer-substring
	     (progn
	       (re-search-forward "\\w+")
	       (point))
	     (progn
	       (backward-word 1)
	       (point)))))
    (landESP-global-focus-on-expr 
     (concat "\\b" id "\\b")
     (landESP-file-list (landESP-grep-command id))
     nil nil t))
  (landESP-print-modeline))

(defvar landESP-project-file "~/.files")

(defun landESP-set-project-file (file)
  (interactive "fset project file to: ")
  (setq landESP-project-file file))

(defun landESP-file-list-string ()
  (save-excursion
    (let ((args "")
	  beg name )
      (if (file-readable-p landESP-project-file)
	  (progn
	    (find-file landESP-project-file)
	    (goto-char (point-min))
	    (while (not (= (point) (point-max)))
	      (if (looking-at "\\ *#")
		  ()
		(setq beg (point))
		(end-of-line 1)
		(setq name (buffer-substring beg (point)))
		(setq args (concat args " " name)))
	      (forward-line 1)))
	(setq args " * "))
      args)))

(defun landESP-file-list ( &optional list-command)
  (save-excursion
    (let ((current-file (buffer-file-name))
	  (command "sh")
	  (args (or list-command (concat "ls -F" (landESP-file-list-string))))
	  beg alist name)
      (generate-new-buffer "*files*")
      (switch-to-buffer "*files*")
      (insert args)
      (call-process-region (point-min) (point-max) command t t)
      (goto-char (point-min))
      (while (not (= (point) (point-max)))
	(setq beg (point))
	(end-of-line 1)
	(setq name (buffer-substring beg (point)))
	(setq alist (append alist (list (list name name))))
	(forward-line 1))
      (kill-buffer "*files*")
      (or alist (cons (list current-file current-file) alist)))))

(defun landESP-next-error (&optional argp)
  (interactive "P")
  (fset 'switch-to-buffer 
	(function (lambda (buffer) 
		    (old-switch-to-buffer buffer)
		    (if (and (boundp 'landESP) landESP)
			(progn
			  (while (not (eq (landESP-type 
					   (landESP-get-current-focus))
					  landESP-file-type))
			    (landESP-widen-focus-intern t))
			  (landESP-print-modeline))))))
  (condition-case ()
      (next-error argp)
    (error nil))
  (fset 'switch-to-buffer (symbol-function 'old-switch-to-buffer)))

(defun landESP-previous-error (&optional argp)
  "\\[next-error] backwards."
    (interactive "P")
    (landESP-next-error (cond ((null argp) -1)
			      ((numberp argp) (- argp))
			      (t argp))))

(defun landESP-grep-command ( string)
  (concat "grep -l " string (landESP-file-list-string)))

(defun landESP-grep (string)
  (interactive "sRun grep on project files. Expression: ")
  (fset 'set-font-lock-mode (symbol-function 'ignore))
  (grep (concat string (landESP-file-list-string)))
  (fset 'set-font-lock-mode (symbol-function 'old-set-font-lock-mode))
  (set-font-lock-mode))

(defun landESP-find-file ()
  "Gives you a list of files where you can choose one to edit. The file list
is computed by reading the file landESP-project-file."
  (interactive)
  (let ((file (car (landESP-completing-read nil (landESP-file-list)))))
    (if file
	(progn
	  (landESP-correct-current-focus t)
	  (let (landESP-print-modeline-flag)
	    (find-file file))
	  (while (not (eq (landESP-type (landESP-get-current-focus))
			  landESP-file-type))
	    (landESP-widen-focus-intern t))
	  (landESP-print-modeline)))))

(defvar landESP-project-buffer nil)

(defun landESP-dired-project()
  "dired for all files belonging to your project."
  (interactive)
  (if (condition-case ()
	  (and (bufferp landESP-project-buffer )
	       (or (switch-to-buffer landESP-project-buffer) t))
	(error nil))
      ()
    (let ((files (landESP-file-list))
	  dired-list old-dir
	  subdir-alist )
      (setq dired-list (mapcar '(lambda (elem) (car elem))
			       (landESP-file-list 
				(concat "ls -l" (landESP-file-list-string)))))
      (setq landESP-project-buffer (generate-new-buffer "project"))
      (switch-to-buffer "project")
      (while dired-list
	(insert "  " (car dired-list) "\n")
	(setq dired-list (cdr dired-list)))
      (goto-char (point-min))
      (while (re-search-forward "\\([^ \t]*/\\)+" nil t)
	(replace-match "" nil nil))
      (goto-char (point-min))
      (while files
	(if (string= old-dir (file-name-directory (car (car files))))
	    ()
	  (setq old-dir (file-name-directory (car (car files))))
	  (insert "  " old-dir ":\n")
	  (setq subdir-alist 
		(append (list (cons (or (file-name-directory (car (car files)))
					".") 
				    (point-marker)))
			subdir-alist)))
	(setq files (cdr files))
	(forward-line))
      (goto-char (point-min))
      (dired-mode)
      (local-set-key "g" 'landESP-new-project-buffer)
      (set (make-local-variable 'dired-subdir-alist)
	   subdir-alist))))

(defun landESP-new-project-buffer ()
  (interactive)
  (kill-buffer nil)
  (landESP-dired-project))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;  functions and vars to indent source code
;;

(defvar landESP-error-recovery-expression nil
  "*expression used by the compiler for error recovery; usually end of
a statement or sth. similar. Here it is used to restrict the area searched 
back for for the outermost open parenthesis. This is only used if
landESP-indent-parens-special is set to t. You should choose an expression 
which does occir relatively often but which does not occur within parentheses")
(make-variable-buffer-local 'landESP-error-recovery-expression)

(defvar landESP-indent-parens-special t
  "*set this variable to nil if you don't want special indentation within 
parens: all lines within parens start in the column of the first character 
following the open paren.")
(make-variable-buffer-local 'landESP-indent-parens-special)

(defun landESP-calculate-indent ()
  "calculates the indentation of the current line. The algorithm is as
follows:
IF the current line is an IGNORE line just return the current indentation.
IF the current line( let's call it LINE1) is inside parentheses and the 
   variable landESP-indent-parens-special is t
THEN return the position of the character following the innermost
     opening parenthesis
ELSIF the current line is an ABSOLUTE line 
THEN return its indentation specified in landESP-keywords-indent-table
ELSE search backwards for a line (LINE0) which is not inside parentheses
     and which is no IGNORE line; add to the indentation of this line two
     amounts which are  determined by their leading expression.
\(See variable landESP-keywords-indent-table for more information\)"
  (let (first-pos-next-line line-start line-end)
    (save-excursion
      (beginning-of-line 1)
      (if (= (point) (point-min))
	  0
	(end-of-line 1)
	(setq line-end (point))
	(beginning-of-line 1)
	(setq line-start (point))
	(goto-first-word-of-line)
	(setq first-pos-next-line (point)) 
	(cond ((landESP-ignore-position (point)))
	      ((and landESP-indent-parens-special
		    (looking-at "(") 
		    (not (landESP-keyword-indent 
			  (eval landESP-keywords-indent-table))))
	       (current-indentation))
	      ((landESP-after-open-if-in-parens line-start))
	      ((landESP-absolute-position (point)))
	      ((progn
		 (landESP-skip-backward-ignored-lines)
		 (+ (if (looking-at "\\ *\n")
			0
		      (current-indentation))
		    (landESP-indent-factor (point)
					   line-end)
		    (landESP-indent-factor first-pos-next-line 
					   nil)))))))))

(defun landESP-ignore-position (pos)
  "Return the current indentation of a line beginning with text at
position pos if the positioning is IGNORE (unchanged position).
Otherwise return nil."
  (save-excursion
    (let (element)
      (setq element (landESP-get-keyword-at-position pos))
      (if (and element (eq (car (cdr element)) 'IGNORE))
	  (current-indentation)
	nil))))

(defun landESP-get-keyword-at-position (pos)
  "Return the entry of the keyword-indent-table corresponding to the string
at position pos. If there is no such entry return nil."
  (save-excursion
    (goto-char pos)
    (landESP-keyword-indent 
     (eval landESP-keywords-indent-table))))

(defun landESP-absolute-position (pos)
  "Return the indentation of a line beginning with text at position pos
   if the positioning is ABSOLUTE (absolute position). Otherwise return nil."
  (save-excursion
    (let (element)
      (setq element (landESP-get-keyword-at-position pos))
      (if (and element (eq (car (cdr element)) 'ABSOLUTE))
	  (* landESP-indent
	     (car (cdr (cdr element))))
	nil))))

(defun landESP-skip-backward-ignored-lines ()
  (let (element (continue t))
    (while continue
      (beginning-of-previous-line)
      (if (= (point) (point-min))
	  (progn
	    (goto-first-word-of-line)
	    (setq continue nil))
	(goto-first-word-of-line)
	(while (landESP-after-open-if-in-parens (point))
	  (backward-word 1)
	  (goto-first-word-of-line))
	(setq element (landESP-keyword-indent 
		       (eval landESP-keywords-indent-table)))
	(if (not element)
	    t
 	  (if (eq (car (cdr element)) 'IGNORE)
	      t
	    (setq continue nil)))))))

(defun landESP-after-open-if-in-parens (pos)
  "If the variable landESP-indent-parens-special is set the following is done:
IF pos is inside parentheses
THEN goto the position of the opening parentheses and
     Return the column following an opening parenthesis and any white space
ELSE return nil."
  (if landESP-indent-parens-special
      (condition-case ()
	  (let ((end (point))(final-pos (point))
		indent)
	    (save-excursion
	      (save-restriction
		(goto-char pos)
		(if (= (char-after (point)) ?\()
		    (if (landESP-get-keyword-at-position (point))
			(signal 'error nil)
		      (setq final-pos (point))
		      (forward-char 1)
		      (skip-chars-forward " \t\n")
		      (setq indent (current-column)))
		  (if (re-search-backward landESP-error-recovery-expression 
					  (point-min) t 1)
		      (narrow-to-region  end (progn
					       (beginning-of-line)
					       (point)))
		    (narrow-to-region end (point-min)))
		  (goto-char (point-max))
		  (goto-char (or (scan-lists (point) -1 1) (point-min)))
		  (if (= (char-after (point)) ?\()
		      (if (landESP-get-keyword-at-position (point))
			  (signal 'error nil)
			(setq final-pos (point))
			(forward-char 1)
			(skip-chars-forward " \t\n")
			(setq indent (current-column)))
		    (signal 'error nil)))))
	    (goto-char final-pos)
	    indent)
	(error nil))))

(defun landESP-indent-factor (pos &optional limit)
  "Return the indentation of a line beginning with text at position pos
   in respect to the previous line (limit = nil) or a line in respect to the
   previous line with text beginning at pos (limit <> nil), if positioning 
   is RELATIVE."
  (save-excursion
    (save-restriction
      (if pos
	  (let (element)
	    (goto-char pos)
            (if limit
                (narrow-to-region (point-min) limit))
	    (setq element (landESP-keyword-indent 
		  	   (eval landESP-keywords-indent-table)))
	    (if element
	        (* landESP-indent
		   (if limit
		       (car (cdr (cdr (cdr element))))
		     (if (eq (car (cdr element)) 'RELATIVE)
		         (car (cdr (cdr element)))
		       (message "Syntax error in keywords-indent-table")
		       0)))
	      0))))))

(defun landESP-keyword-indent (keywords-table)
  "Returns the element of the keywords-table where the string = the string
   we are looking-at and returns nil if the string could not be found."
  (let ((keywords-list keywords-table)
	result)
    (while (and (not result) keywords-list)
      (if (looking-at (car (car keywords-list)))
	  (setq result (car keywords-list))
	(setq keywords-list (cdr keywords-list))))
    result))

(defun goto-first-word-of-line ()
  "goto the first non-white character of the line;
   point is moved to that position (no save-excursion!)"
  (beginning-of-line)
  (skip-chars-forward " \t"))

(defvar landESP-keywords-indent-table nil 
"holds variable with the mode specific keyword indentations.  The
value is a list of quadruples. Each quadruple consists of a regular
expression EXP, an attribute ATT (ABSOLUTE, RELATIVE or IGNORE) and two
numbers IN1 and IN2, and has the following meaning: Let's assume we
have two lines LINE0 and LINE1, and we want to indent LINE1 relative
to LINE0. There are three cases:
IF ATT of LINE1 is ABSOLUTE
THEN indentation is IN1.
IF ATT of LINE1 is IGNORE
THEN indentation is the current indentation.
IF ATT of LINE1 is RELATIVE
THEN the absolute indentation of LINE1
     depends on the indentation of LINE0 and the starting characters of
     both lines. Let START0 and START1 be the indentations of the two
     lines, and let LINEi.INj be the values INi of the first quadruple, where
     EXP matches the beginning of LINEi 
     START1 = START0 + c *(LINE0.IN2 + LINE1.IN1) 
     c is the value of the variable landESP-indent which defaults to 2.")  
(make-variable-buffer-local 'landESP-keywords-indent-table)

(defvar landESP-indent 2
  "*holds value of indentation factor.")
(make-variable-buffer-local 'landESP-indent)

(defvar landESP-indent-calculation '(landESP-calculate-indent)
  "*holds function to calculate the mode specific indentation.
   The function calculates the indentation of the current line.
   The function landESP-calculate-indent is called as default.")
(make-variable-buffer-local 'landESP-indent-calculation)

(defvar landESP-electric-newline t
  "*indent this line and next line if t.")
(make-variable-buffer-local 'landESP-electric-newline)

(defun landESP-electric-indent-line ()
  "see landESP-electric-newline."
  (interactive)
  (if landESP-electric-newline
      (progn
	(landESP-indent-line)
	(newline)
	(if (looking-at "\\ *$")
	    (progn
	      (insert "a")
	      (landESP-indent-line)
	      (delete-backward-char 1))
	  (landESP-indent-line)))
    (newline)))

(defun landESP-indent-line ()
  "Indent current line as mode specific source code;
return the shift amount, i.e. the number of columns
the line was shifted (this is NOT always the indentation
because a line could already have leading blanks!)."
  (interactive)
  (fset 'beginning-of-defun 
	(function (lambda (&optional arg) (previous-line (or arg 1)))))
  (let ((indent (eval landESP-indent-calculation)) shift-amt beg
	(pos (- (point-max) (point))))
    (beginning-of-line)
    (setq beg (point))
    (skip-chars-forward " \t")
    (setq shift-amt (- indent (current-column)))
    (if (zerop shift-amt)
	nil
      (delete-region beg (point))
      (indent-to indent))
    ;; If initial point was within line's indentation,
    ;; position after the indentation.  Else stay at same point in text.
    (if (> (- (point-max) pos) (point))
	(goto-char (- (point-max) pos)))
    (fset 'beginning-of-defun (symbol-function 'old-beginning-of-defun))
    shift-amt))

(defun landESP-beautify-region (start end)
  "Beautifies the selected region by indenting every line by TAB."
  (interactive "r")
  (let ((pos (- (point-max) end)))
    (goto-char start)
    (while (> (- (point-max) (point)) pos)
      (landESP-indent-line)
      (forward-line 1))
    (goto-char (- (point-max) pos))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;  functions and vars to print language constructs
;;
;; A construct is a list of parts. Each of the parts
;; is either a string, a lisp expression or one of the atoms @, CR and <>. 
;; To print a line
;; one of the following actions is executed for every part:
;; - A string is just printed.
;; - A lisp expression is evaluated and the result is printed.
;; - The atom @ represents the position where the cursor will be placed
;;   within the construct after the whole construct has been printed.
;; - The atom CR represents a newline
;; - The atom <> is a placeholder for an optional argument
;; Each line of the construct is automatically indented.

(defvar landESP-point-stack '(1)
  "the stack of positions to go to after insertion of a construct.")

(defvar landESP-point-set-stack nil
  "stack of flags indicating if the corresponding element in
landESP-point-stack is set in a construct.")

(defun landESP-newline ()
  "Goto next line."
  (interactive)
  (newline 1)
  (indent-relative-maybe))

(defun landESP-tab ()
  "Indent to next tab stop."
  (interactive)
  (if (eq (current-indentation) 
	  (progn (indent-relative-maybe)
		 (current-indentation)))
      (indent-to (* (1+ (/ (current-indentation) landESP-indent)) landESP-indent))))

(defun landESP-print (construct)
  "insert the construct before the current line and set point to a
user defined position or to the beginning of the construct."
  (push (point) landESP-point-stack)
  (push nil landESP-point-set-stack)
  (setq landESP-quit-flag nil)
  (setq landESP-printed-parts 0)
;;  (end-of-line blabla)
  (landESP-print-part construct)
;;  (newline 1)
  (end-of-line)
  (let ((end (point))
	new-end )
    (landESP-indent-line)
    (setq new-end (point))
    (landESP-correct-point-+-start-stack end new-end)
    (setq landESP-last-endpos (+ landESP-last-endpos (- new-end end))))
  (delete-blank-lines)
  (pop landESP-point-set-stack)
  (goto-char (pop landESP-point-stack)))

(defvar landESP-quit-flag nil)
(defvar landESP-printed-parts 0)
(defun landESP-quit-recursive-edit ()
  (interactive)
  (setq landESP-quit-flag t)
  (exit-recursive-edit))

(defun landESP-print-part (part)
;;The function inserts a string represented by part at the current point
;;into the file. PART can represent a string in the following ways:
;;a string starting with ?
;;    The user is prompted with the rest of the string following the ? and 
;;    asked to insert a string. This string is inserted.
;;a string (not starting with ?): 
;;    the string is inserted
;;a lisp symbol:
;;    the symbol is evaluated and the result inserted
;;a lisp expression:
;;    the expression is evaluated and the result inserted
;;@
;;    This is a special symbol representing the position where the cursor will
;;    be placed at the end of landESP-print
;;<>
;;    This represents an optional result (see OPT)
;;CR
;;    This represents a newline
;;(SEQ PART1 PART2 ... PARTn)
;;    This represents a sequence of PARTs
;;(ALT (name1 PART1)
;;     (name2 PART2)
;;     ...
;;     (namen PARTn))
;;    This represents alternatives. The user is shown a list of the names.
;;    She can choose PARTi by giving namei.
;;(OPT optPART PART1 ... PARTn)
;;    This represents an optional argument. optPART is evaluated first.
;;    If its result is a non-empty string then PART1 to PARTn are evaluated 
;;    as in a sequence. If one of them is the atom <> it will insert the 
;;    result string of optPART

  (if landESP-quit-flag
      ()
    (condition-case ()
	(progn
	  (push (point) landESP-start-stack)
	  (cond ((stringp part)
		 (setq landESP-printed-parts (1+ landESP-printed-parts))
		 (cond ((char-equal (string-to-char "?") (string-to-char part))
			(let ((ret-function (key-binding "\C-m"))
			      (quit-function (key-binding "\C-g")))
			  (message (concat (substring part 1) " (RET to exit)"))
			  (local-set-key "\C-m" 'exit-recursive-edit)
			  (local-set-key "\C-g" 'landESP-quit-recursive-edit)
			  (recursive-edit)
			  (local-set-key "\C-m" ret-function)
			  (local-set-key "\C-g" quit-function)
			  (if landESP-quit-flag
			      (signal 'quit nil))
			;;  (let ((end (point))
			;;	new-end )
			;;    (landESP-indent-line)
			;;    (setq new-end (point))
			;;    (landESP-correct-point-+-start-stack end new-end))
			  ))
		       ((insert part))))
		((eq part '@)
		 (landESP-newpoint))
		((eq part 'CR)
		 (let ((end (point))
		       new-end )
		   (newline 1)
		   (backward-char 1)
		   (landESP-indent-line)
		   (setq new-end (point))
		   (landESP-correct-point-+-start-stack end new-end)
		   (forward-char 1)
		   (insert "-")
		   (landESP-indent-line)
		   (delete-backward-char 1)))
		((symbolp part)
		 (landESP-print-part (eval part)))
		((listp part)
		 (cond ((eq (car part) 'SEQ)
			(landESP-print-sequence (cdr part)))
		       ((eq (car part) 'ALT)
			(landESP-print-alternatives (cdr part)))
		       ((eq (car part) 'OPT)
			(landESP-print-optional (cdr part)))
		       (t
			(landESP-print-part (eval part))))))
	  (setq landESP-last-startpos (pop landESP-start-stack))
	  (setq landESP-last-endpos (point)))
      (quit (undo-start)
	    (undo-more landESP-printed-parts)))))

(defun landESP-correct-point-+-start-stack (end new-end)
  (let ((bol (save-excursion
	       (beginning-of-line)
	       (point)))
	(n (- new-end end)))
    (setq landESP-start-stack (landESP-correct-stack landESP-start-stack
						     bol n))
    (setq landESP-point-stack (landESP-correct-stack 
			       landESP-point-stack
			       bol n))))

(defun landESP-correct-stack (stack bol n)
  (if stack
      (if (> (car stack) bol)
	  (cons (+ (car stack) n)
		(landESP-correct-stack (cdr stack) bol n))
	stack)))

(defun landESP-print-sequence (sequence)
  (while sequence
    (landESP-print-part (car sequence))
    (setq sequence (cdr sequence))))

(defun landESP-print-alternatives (alternatives)
  (if alternatives
      (let ((chosen (completing-read "alternatives:" alternatives nil 1)))
	(if (string< "" chosen )
	    (landESP-print-part
	     (car (cdr (assoc chosen alternatives))))))))

(defun landESP-print-optional (optional)
  (let (pos 
	(cont (cdr optional)))
    (while (and cont 
		(not (eq (car cont) '<>)))
      (landESP-print-part (car cont))
      (setq cont (cdr cont)))
    (setq pos (point))
    (landESP-print-part (car optional))
    (if (= pos (point))
	(delete-region (landESP-current-start) pos)
      (landESP-print-sequence (cdr cont)))))

(defvar	landESP-start-stack nil)
(make-variable-buffer-local 'landESP-start-stack)

(defvar landESP-last-startpos 0)
(make-variable-buffer-local 'landESP-last-startpos)

(defun landESP-last-start ()
  "returns the start position of the last inserted construct."
  landESP-last-startpos)

(defun landESP-current-start ()
  "returns the start position of the construct currently being inserted."
  (car landESP-start-stack))

(defvar landESP-last-endpos 0)
(make-variable-buffer-local 'landESP-last-endpos)

(defun landESP-last-end ()
  "returns the end position of the last inserted construct."
  landESP-last-endpos)

(defun landESP-newpoint ()
  "store the current position."
  (pop landESP-point-set-stack)
  (pop landESP-point-stack)
  (push t landESP-point-set-stack)
  (push (point) landESP-point-stack))

(if (string-match "Lucid" emacs-version)
    (progn
      (require 'menu-utils)
      
      (defun landESP-make-popup-tuples (old-list)
	(let (alist)
	  (while old-list
	    (let ((head (car old-list)))
	      (setq alist (append alist
				  (list (cons (car head) (car (cdr head))))))
	      (setq old-list (cdr old-list))))
	  (sort alist (function
		       (lambda (p1 p2)
			 (string< (car p1) (car p2)))))))
      
      (defun landESP-make-popup-list (old-list)
	(let (alist)
	  (while old-list
	    (let ((head (car old-list)))
	      (setq alist (append alist (list (list (car head)))))
	      (setq old-list (cdr old-list))))
	  (sort alist (function
		       (lambda (p1 p2)
			 (string< (car p1) (car p2)))))))
      
      (defun landESP-print-alternatives (alternatives)
	(if alternatives
	    (if current-mouse-event
		(landESP-print-part 
		 (popup-choice-menu (landESP-make-popup-tuples alternatives)
				    "Choose alternative: "))
	      (let ((chosen (completing-read 
			     "alternatives: " alternatives nil 1)))
		(if (string< "" chosen )
		    (landESP-print-part
		     (car (cdr (assoc chosen alternatives)))))))))
      
      (defun landESP-completing-read (entry &optional list)
	"The same as completing-read, but with no function bound to SPACE."
	(save-excursion
	  (let ((alist (or list (landESP-make-focus-list entry))))
	    (if alist
		(if current-mouse-event
		    (list (popup-choice-menu (landESP-make-popup-list alist)
					     "Unit name: ")
			  alist)
		  (list (completing-read "Unit name: " alist nil t)
			alist))
	      (error "No units found!")))))
      ))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; menu bar for lucid emacs
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar landESP-default-menu
  '(("Focussing"
     ["Down one level"                landESP-narrow-focus t]
     ["Down at point"                 landESP-narrow-to-point t]
     ["Up one level"                  landESP-widen-focus t]
     ["Same level, direct selection"  landESP-direct-focus t]
     ["Same level, next item"         landESP-next-focus t]
     ["Same level, previous item"     landESP-previous-focus t]
     ["Focus on identifier definition" 
      landESP-focus-on-identifier-definition t]
     ["Focus history"                 landESP-focus-on-history t]
     ["Focus on last unit"            landESP-focus-on-last t]
     ["Focus on file"                 landESP-find-file t]
     )
    ("Run"
     ["Compile..."                 compile                           t]
     ["Kill Compilation"           kill-compilation                  t]
     "-----"
     ["Next Error"                 landESP-next-error                t]
     ["Previous Error"             landESP-previous-error            t]
     ;;	  ["Goto Error"                 compile-goto-error                t]
     "-----"
     ["GDB Debuger"                gdb                               t]
     "-----"
     )))
(defvar landESP-menu nil)
(make-variable-buffer-local 'landESP-menu)

(defvar landESP-specific-menu nil)
(make-variable-buffer-local 'landESP-specific-menu)

(defvar landESP-mode-string "")
(make-variable-buffer-local 'landESP-mode-string)

(defun landESP-make-font-lock-keywords (table)
  (mapcar '(lambda (x) (concat "\\b" x "\\b")) table))

(defun landESP-substitute (start end flag)
  (subst-char-in-region start end
			(if (= flag ?\n) ?\^M ?\n)
			flag t))

(defun landESP-toggle-focus-visibility (focus flag)
  (if focus
      (save-excursion
	(goto-char (landESP-start focus))
	(search-forward (landESP-key focus) (landESP-end focus) t)
	(landESP-substitute (point) (landESP-end focus) flag))))

(defun landESP-show-all ()
  "Shows the whole body of the current buffer."
  (interactive)
  (landESP-substitute (point-min) (point-max) ?\n))

(defun landESP-hide-all ()
  "Hides the whole body of the current buffer."
  (interactive)
  (landESP-substitute (point-min) (point-max) ?\^M))

(defun landESP-toggle-visibility-of-focus-body (flag &optional entry)
  (let* (focus-list focus)
    (if entry
	(setq focus entry)
      (setq focus-list (save-excursion
			 (landESP-make-focus-list (landESP-get-current-focus))))
      (if focus-list
	  (progn
	    (landESP-correct-focus)
	    (setq focus (cond ((landESP-where-in-focus-list (point) focus-list))
			      ((car focus-list))))))
      (landESP-toggle-focus-visibility (landESP-get-current-focus) flag))))

(defun landESP-hide-focus-body (&optional entry)
  "Hides all except the first line of ENTRY."
  (interactive)
  (landESP-toggle-visibility-of-focus-body ?\^M entry))

(defun landESP-show-focus-body (&optional entry)
  "Shows all lines of ENTRY."
  (interactive)
  (landESP-toggle-visibility-of-focus-body ?\n entry))

(defun landESP-toggle-visibility-of-bodies (flag &optional entry-list)
  (save-excursion
    (let* (focus-list focus)
      (if entry-list
	  (setq focus-list entry-list)
	(setq focus-list (save-excursion
			   (landESP-make-focus-list
			    (landESP-get-current-focus)))))
      (if focus-list
	  (landESP-correct-focus))
      (while focus-list
	(setq focus (car focus-list))
	(setq focus-list (cdr focus-list))
	(goto-char (landESP-start focus))
	(search-forward (landESP-key focus) (landESP-end focus) t)
	(landESP-substitute (point) (landESP-end focus) flag)))))

(defun landESP-hide-all-bodies (&optional entry-list)
  "Hides all focus bodies in the current entry."
  (interactive)
  (landESP-toggle-visibility-of-bodies ?\^M entry-list))

(defun landESP-show-all-bodies (&optional entry-list)
  "Shows all focus bodies in the current entry."
  (interactive)
  (landESP-toggle-visibility-of-bodies ?\n entry-list))

(defun landESP-hide-bodies-of-parts (&optional entry)
  "Hides all except the first line of all parts of ENTRY (default for entry
is the entry where the cursor is."
  (interactive)
  (landESP-narrow-to-point-intern nil)
  (landESP-hide-all-bodies)
  (landESP-widen-focus-intern nil))

(defun landESP-highlight-entries (face &optional entry-list)
  (save-excursion
    (let* (focus-list focus)
      (if entry-list
	  (setq focus-list entry-list)
	(setq focus-list (save-excursion
			   (landESP-make-focus-list
			    (landESP-get-current-focus)))))
      (if focus-list
	  (landESP-correct-focus))
      (while focus-list
	(setq focus (car focus-list))
	(setq focus-list (cdr focus-list))
	(goto-char (landESP-start focus))
	(search-forward (landESP-key focus) (landESP-end focus) t)
	(font-lock-set-face (match-beginning 0) (match-end 0) face)))))

(defun landESP-show-entries ()
  (interactive)
  (landESP-highlight-entries 'font-lock-function-name-face))

(provide 'landESP)

