-- ***************************************************************
-- ---------------------------------------------------------------
--
-- StringList.cs --
-- 
--   Copyright (C) sietec Systemtechnik GmbH & Co OHG 1994
--   All rights reserved
-- 
-- Author          : Herbert Gold
-- 
-- PURPOSE 
--       This object type realizes the class List for STRING.
--
-- REFER
--      |>Documents with client documentation<|
--
-- ---------------------------------------------------------------
--
-- ***************************************************************

SPECIFICATION StringList


TYPE StringList = OBJECT

-- -----------------------------------------------------------------
-- General
-- -----------------------------------------------------------------
-- An object of the type StringList contains a cursor on a list of
-- entries. The entries are referred by the cursor, which referes
-- to one specific entry. This entry contains the concrete
-- object and the link mechanism, say the next and previous objects.
-- Contents of the StringList container is also the number of entries in
-- the list and the beginning and the end of that list. 
-- -----------------------------------------------------------------

PUBLIC

-- -----------------------------------------------------------------
-- Construction of a list (done internally by INITIALLY)
-- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Copying
-- -----------------------------------------------------------------
  
  METHOD Copy () : StringList;
  -- -----------------------------------------------------------------
  -- Copy all entries of the container to a new one and return it.
  -- Leave the cursor unchanged. 
  -- -----------------------------------------------------------------

  METHOD GetNumberOfEntries () : INT;
  -- -----------------------------------------------------------------
  -- Deliver the cardinality of the container. Leave the cursor
  -- unchanged. 
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Selective access
-- -----------------------------------------------------------------

  METHOD AppendEntry (IN entry : STRING);
  -- -----------------------------------------------------------------
  -- Insert the ENTRY into this container at the end.
  -- The position of this ENTRY is the actual cursor position.
  -- -----------------------------------------------------------------

  METHOD InsertEntry (IN entry : STRING);
  -- -----------------------------------------------------------------
  -- Insert the ENTRY into this container after the entry at the
  -- actual cursor position. Leave the cursor unchanged.
  -- -----------------------------------------------------------------

  METHOD RemoveEntry ();
  -- -----------------------------------------------------------------
  -- Delete the entry of this container at the actual cursor
  -- position. Set the cursor to the next entry.
  -- -----------------------------------------------------------------
 
  METHOD RemoveEntries ();
  -- -----------------------------------------------------------------
  -- Remove all entries of this container.
  -- -----------------------------------------------------------------

  METHOD GetEntryAtCursor () : STRING;
  -- -----------------------------------------------------------------
  -- Deliver the contents of this container at the actual cursor
  -- position. Leave the cursor unchanged.
  -- -----------------------------------------------------------------

  METHOD SetEntryAtCursor (IN entry : STRING);
  -- -----------------------------------------------------------------
  -- Write the ENTRY in this container at the actual cursor
  -- position. Leave the cursor unchanged.
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Navigation through the container
-- -----------------------------------------------------------------

  METHOD SetCursorToBeginMargin ();
  METHOD SetCursorToEndMargin ();
  METHOD SetCursorToFirst ();
  METHOD SetCursorToLast ();
  METHOD SetCursorToNext ();
  METHOD SetCursorToPrevious ();
  -- -----------------------------------------------------------------
  -- Move the cursor to the beginning or at the end of this container.
  -- Move the cursor to the first or last position of this container
  -- containing an entry. 
  -- SetCursorToNext and SetCursorToPrevious move the cursor relativ
  -- to the actual position back or forth.
  -- -----------------------------------------------------------------

  METHOD GetPositionOfCursor () : INT;
  -- -----------------------------------------------------------------
  -- Deliver the number of the element at the actual cursor position.
  -- The first element has the number 1.
  -- -----------------------------------------------------------------

  METHOD SetCursorToPosition (IN position : INT);
  -- -----------------------------------------------------------------
  -- Move the cursor to the POSITIONth position
  -- -----------------------------------------------------------------

  METHOD StepToNextMatch
    (IN predicate : PROCEDURE (IN STRING) : BOOL);
  METHOD StepToPreviousMatch
    (IN predicate : PROCEDURE (IN STRING) : BOOL);
  -- -----------------------------------------------------------------
  -- Move the cursor back or forth to the entry which satisfies the
  -- PREDICATE. Start the search at the next or previous entry to the
  -- actual position.
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Predicates
-- -----------------------------------------------------------------

  METHOD IsEmpty () : BOOL;
  -- ---------------------------------------------------------------
  -- Test this container for emptyness. Leave the cursor unchanged.
  -- ---------------------------------------------------------------

  METHOD IsEqual 
    (IN other_list  : StringList) : BOOL;
  -- -----------------------------------------------------------------
  -- Compare this conatiner to OtherList. Leave the cursor unchanged.
  -- -----------------------------------------------------------------

  METHOD IsBeginMargin () : BOOL;
  METHOD IsEndMargin   () : BOOL;
  -- -----------------------------------------------------------------
  -- Is the cursor position at the begin or end margin of this 
  -- list.
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Complex operations on the container
-- -----------------------------------------------------------------

  METHOD Merge (IN list : StringList);
  -- -----------------------------------------------------------------
  -- Merge the StringList at the actual cursor position of this container.
  -- Leave the cursor unchanged. 
  -- -----------------------------------------------------------------

  METHOD Apply (IN routine : PROCEDURE (INOUT STRING));
  -- -----------------------------------------------------------------
  -- Apply ROUTINE to every entry of this container. Start at the
  -- actual cursor position. Leave the cursor unchanged.
  -- -----------------------------------------------------------------

  METHOD RemoveMatchingEntries 
    (IN predicate : PROCEDURE (IN STRING) : BOOL);
  -- -----------------------------------------------------------------
  -- Delete all entries in this container, which satisfies the
  -- PREDICATE. Start at the actual cursor position. Set the cursor
  -- after the operation to the first entry.
  -- -----------------------------------------------------------------

  METHOD RemoveDuplicateEntries;
  -- -----------------------------------------------------------------
  -- Delete all entries in this container, which are equal. 
  -- Set the cursor after the operation to the first entry.
  -- -----------------------------------------------------------------

  METHOD GetNumberOfMatches
    (IN predicate : PROCEDURE (IN STRING) : BOOL) : INT;
  -- -----------------------------------------------------------------
  -- Deliver the number of entries in this container, which satisfy
  -- the PREDICATE. Leave the cursor unchanged. 
  -- -----------------------------------------------------------------

  METHOD Split () : StringList;
  -- -----------------------------------------------------------------
  -- Split this container and return the List starting at the actual
  -- cursor position. The last entry of this container will be the
  -- entry before the actual position. Set the cursor after this
  -- operation to the first entry. 
  -- -----------------------------------------------------------------

  METHOD Substitute 
    (IN predicate : PROCEDURE (IN STRING) : BOOL,
     IN entry     : STRING);
  -- -----------------------------------------------------------------
  -- Replace the contents of all entries of this container which
  -- satisfy the PREDICATE with ENTRY. Start at the actual cursor
  -- position. Leave the cursor unchanged.
  -- -----------------------------------------------------------------

  METHOD Head () : STRING;
  -- -----------------------------------------------------------------
  -- Return the first entry of this container. Leave the cursor
  -- unchanged. 
  -- -----------------------------------------------------------------

  METHOD Tail () : StringList ;
  -- -----------------------------------------------------------------
  -- Return this container without the first entry. Leave the cursor
  -- unchanged. The cursor of the new list is set to the first
  -- entry. 
  -- -----------------------------------------------------------------

PROTECTED
  
STATE

  cursor          : StringCursor;  -- actual position in this container
  numberOfEntries : INT;	   -- number of entries in this container
  beginMargin     : StringNode ;   -- Left end of this container
  endMargin       : StringNode ;   -- Right end of this container

END OBJECT;

-- ***************************************************************

EXCEPTION ContainerEmpty;
EXCEPTION PositionOutOfBounds;
EXCEPTION NoMatch;

-- ***************************************************************

--
TYPE StringCursor = OBJECT

PUBLIC

   METHOD SetNode( IN node : StringNode );
   METHOD SetPosition( IN position : INT );
   METHOD Copy : StringCursor;
   METHOD GetNode : StringNode;
   METHOD GetPosition : INT;

PROTECTED
  
STATE
   Position : INT;
   ActNode  : StringNode;

END OBJECT;


-- ***************************************************************
--
-- StringNode -- 
-- 
-- ---------------------------------------------------------------
TYPE StringNode = ABSTRACT OBJECT

-- ---------------------------------------------------------------
-- General
-- ---------------------------------------------------------------
-- Objects of this class represent list entries, the begin
-- margin and the end margin. This class is a deffered class, which
-- only provides the interface. The subclasses are defining the
-- margins and the entry nodes.
-- ---------------------------------------------------------------

PUBLIC

-- -----------------------------------------------------------------
-- Access to the STATE
-- -----------------------------------------------------------------

  ABSTRACT METHOD GetEntry () : STRING;
  ABSTRACT METHOD SetEntry (IN entry : STRING);
  -- -----------------------------------------------------------------
  -- Deliver and set the entry of this list entry.
  -- -----------------------------------------------------------------

  ABSTRACT METHOD GetFront () : StringNode ;
  ABSTRACT METHOD GetTail  () : StringNode ;
  -- -----------------------------------------------------------------
  -- Deliver the list before and after this list entry.
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Adding and removing entires
-- -----------------------------------------------------------------

  ABSTRACT METHOD Insert (IN entry : StringNode );
  -- -----------------------------------------------------------------
  -- Insert ENTRY after this entry.
  -- -----------------------------------------------------------------

  ABSTRACT METHOD Delete ();
  -- -----------------------------------------------------------------
  -- Remove this entry.
  -- -----------------------------------------------------------------

  ABSTRACT METHOD IsBeginMargin () : BOOL;
  ABSTRACT METHOD IsEndMargin   () : BOOL;
  -- -----------------------------------------------------------------
  -- Test this entry for the margins.
  -- -----------------------------------------------------------------
  
  ABSTRACT METHOD SetFront (IN front : StringNode ); 
  ABSTRACT METHOD SetTail  (IN tail  : StringNode ); 

PROTECTED
  
STATE

END OBJECT;

-- ---------------------------------------------------------------


-- ---------------------------------------------------------------
TYPE Entry = StringNode
OBJECT

-- ---------------------------------------------------------------
-- General 
-- ---------------------------------------------------------------
-- Objects of this class define normal list entries. This class
-- inherits all methods of list but handles this object like an
-- entry. 
-- ---------------------------------------------------------------

PUBLIC

  METHOD GetEntry () : STRING;
  METHOD SetEntry (IN entry : STRING);
  -- -----------------------------------------------------------------
  -- Deliver and set the entry of this list entry.
  -- -----------------------------------------------------------------

  METHOD GetFront () : StringNode ;
  METHOD GetTail  () : StringNode ;
  -- -----------------------------------------------------------------
  -- Deliver the list before and after this list entry.
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Adding and removing entires
-- -----------------------------------------------------------------

  METHOD Insert (IN entry : StringNode);
  -- -----------------------------------------------------------------
  -- Insert ENTRY after this entry.
  -- -----------------------------------------------------------------

  METHOD Delete ();
  -- -----------------------------------------------------------------
  -- Remove this entry.
  -- -----------------------------------------------------------------

  METHOD IsBeginMargin () : BOOL;
  METHOD IsEndMargin   () : BOOL;
  -- -----------------------------------------------------------------
  -- Test this entry for the margins.
  -- -----------------------------------------------------------------

  METHOD SetFront (IN front : StringNode); 
  METHOD SetTail  (IN tail  : StringNode); 
  
PROTECTED
  
STATE

  Entry : STRING;		 -- The entry of a given type
  Front : StringNode ;		 -- The previous list
  Tail  : StringNode ;		 -- The next list

END OBJECT;

-- ---------------------------------------------------------------

EXCEPTION NoEntry;
EXCEPTION NoPreviousEntry;
EXCEPTION NoNextEntry;
EXCEPTION ImpossibleInsert;
EXCEPTION ImpossibleDelete;

-- ***************************************************************
END SPECIFICATION (* StringList *);
-- 
-------------------------------------------------------------------------------
