-- ***************************************************************
--
-- IdentMap.cs -- 
-- 
--   Copyright (C) sietec Systemtechnik GmbH & Co OHG 1994
--   All rights reserved
-- 
-- Author          : Herbert Gold
-- 
-- PURPOSE 
--      This module realizes the generic container class for a
--      IdentityMap, (also called "Identity-Dictionary").
--      It is basically a Map, but each entry (value) can be entered only
--      once into the container. Thus one can realize 1:1 mappings.
--
-- REFER
--
-- ---------------------------------------------------------------
--
-- ***************************************************************

SPECIFICATION IdentMap

IMPORT Map FROM Map;

TYPE IdentityMap [KeyType, EntryType] = OBJECT 

-- -----------------------------------------------------------------
-- General
-- -----------------------------------------------------------------
-- Objects of this class represents a view to a mapping between an
-- object of a key type to an object of an entry type. The basic
-- requirement to the entry type is, that it may appear only once in the map.
-- -----------------------------------------------------------------

PUBLIC
  
-- -----------------------------------------------------------------
-- Creating a map
-- -----------------------------------------------------------------
-- The cursor is set to an initialized state, the map is set empty.
-- -----------------------------------------------------------------
-- INITIALLY (see obove)

-- -----------------------------------------------------------------
-- Changing the cursor position
-- -----------------------------------------------------------------
-- This operations are used to set the cursors position to a new,
-- absolute or relative position.
-- -----------------------------------------------------------------
  
  METHOD SetCursorToFirst ();
  METHOD SetCursorToLast ();
  METHOD SetCursorToNext ();
  METHOD SetCursorToPrevious ();
  -- -----------------------------------------------------------------
  -- Move the cursor to the first or last, to the next or previous
  -- entry of the map.
  -- -----------------------------------------------------------------

  METHOD SetCursorToElement (IN key : KeyType);
  -- -----------------------------------------------------------------
  -- This method moves the cursor to the element with the key value of
  -- Key. 
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Element access related to the cursor
-- -----------------------------------------------------------------
-- The entry value of the element at the actual cursor position may
-- be accessed by reading and writing. The key value may
-- only accessed by reading to keep the order of the map keys.
-- -----------------------------------------------------------------

  METHOD GetEntryAtCursor () : EntryType;
  METHOD SetEntryAtCursor (IN entry : EntryType);
  -- -----------------------------------------------------------------
  -- The entry value of the element at the actual cursor position is
  -- read or written. An exception is raised on attempt to write an
  -- existing entry.
  -- -----------------------------------------------------------------

  METHOD GetKeyAtCursor () : KeyType;
  -- -----------------------------------------------------------------
  -- The key entry of the element at the actual cursor position is 
  -- read. 
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Inserting new elements and replacing entries
-- -----------------------------------------------------------------
-- New elements may be added to the map or the entry may be changed.
-- Both operations work with a given key value. 
-- -----------------------------------------------------------------

  METHOD InsertElement (IN key : KeyType, IN entry : EntryType);
  -- -----------------------------------------------------------------
  -- This method inserts a new element described by Key and
  -- Entry into this map. If there is already an element with the
  -- same key value than Key, an exception is raised. 
  -- If there is already an element with the same entry value, then
  -- also an exception is raised.
  -- -----------------------------------------------------------------
  
  METHOD ReplaceEntry (IN key : KeyType, IN entry : EntryType);
  -- -----------------------------------------------------------------
  -- This Method replaces the entry value of the element described by
  -- Key with Entry in this map. If there is no element with
  -- the Key, an exception is raised.
  -- If the entry exists for another key, an exception is raised too.
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Delete an element
-- -----------------------------------------------------------------
-- The element with a given key value is removed from the map.
-- -----------------------------------------------------------------

  METHOD RemoveElement (IN key : KeyType);
  -- -----------------------------------------------------------------
  -- This method removes the element with the Key value from this
  -- map. If the cursor was set to the element with Key, it is
  -- undefined after this operation. 
  -- -----------------------------------------------------------------

  METHOD RemoveEntry (IN entry : EntryType);
  -- -----------------------------------------------------------------
  -- This method removes the element with the entry value from this
  -- map. If the cursor was set to the element with the entry, it is
  -- undefined after this operation.
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Get an element related to a key
-- -----------------------------------------------------------------

  METHOD GetEntry (IN key : KeyType) : EntryType;
  -- -----------------------------------------------------------------
  -- This method delivers the entry value related to the Key value
  -- from this map. 
  -- -----------------------------------------------------------------

  METHOD GetKey (IN entry : EntryType ) : KeyType;
  -- -----------------------------------------------------------------
  -- This method delivers the key value related to the entry value
  -- from this map.
  -- -----------------------------------------------------------------

-- -----------------------------------------------------------------
-- Counting elements
-- -----------------------------------------------------------------
-- The number of elements is delivered, which satisfies a given
-- criterion. 
-- -----------------------------------------------------------------

  METHOD GetNumberOfElements () : INT;
  -- -----------------------------------------------------------------
  -- This method delivers the number of elements in this map.
  -- -----------------------------------------------------------------
(*
  METHOD GetNumberOfMatches
    (IN predicate : PROCEDURE (IN EntryType) : BOOL) : INT;
  -- -----------------------------------------------------------------
  -- This method searches in this map for all elements, which satisfy
  -- the given Predicate. The number of matching elements is returned.
  -- -----------------------------------------------------------------


-- -----------------------------------------------------------------
-- Operation mapping
-- -----------------------------------------------------------------
-- A given operation is applied to all element enties of a map.
-- -----------------------------------------------------------------
                                        v !!!!
  METHOD Apply (IN routine : PROCEDURE (IN KeyType, IN EntryType));
  -- -----------------------------------------------------------------
  -- This method applies the Routine to all key: entry pairs of all
  -- elements in this map.
  -- -----------------------------------------------------------------
*)
-- -----------------------------------------------------------------
-- Predicates
-- -----------------------------------------------------------------
-- Some tests for the entries of a map
-- -----------------------------------------------------------------

  METHOD IsFirst () : BOOL;
  -- ---------------------------------------------------------------
  -- This method tests, whether the cursor is related to the first
  -- element of this map.
  -- ---------------------------------------------------------------

  METHOD IsLast () : BOOL;
  -- ---------------------------------------------------------------
  -- This method tests, whether the cursor is related to the last
  -- element of this map.
  -- ---------------------------------------------------------------

  METHOD IsEmpty () : BOOL;
  -- ---------------------------------------------------------------
  -- This method tests this map for emptyness.
  -- ---------------------------------------------------------------

(* -- NOT IMPLEMENTED
  METHOD IsEqual 
    (IN second_map : IdentityMap [KeyType, EntryType]) 
      : BOOL;
  -- -----------------------------------------------------------------
  -- This method compares this map to SecondMap. Two lists are not
  -- equal, if one map contains an element, the other doesn't contain.
  -- To test the equality of two elements, the key values and the
  -- entry values are compared. 
  -- -----------------------------------------------------------------
 *)

  METHOD Contains (IN key : KeyType) : BOOL;
  -- -----------------------------------------------------------------
  -- This method tests the existence of the element with the key value
  -- Key in this map.
  -- -----------------------------------------------------------------

  METHOD ContainsEntry (IN entry : EntryType) : BOOL;
  -- -----------------------------------------------------------------
  -- This method tests the existence of the element with the entry value
  -- Entry in this map.
  -- -----------------------------------------------------------------

PROTECTED

STATE

  keyMap      : Map [KeyType, EntryType];
  entryMap    : Map [EntryType, KeyType]; 

END OBJECT;

EXCEPTION CursorUndefined;
EXCEPTION KeyAlreadyExists;
EXCEPTION InvalidKey;           -- ?? ReplaceEntry
EXCEPTION EntryAlreadyExists;
 
-- ***************************************************************
END SPECIFICATION (* IdentMap *);
