#include <stdlib.h>

#include "idents.h"
#include "errmsg.h"

#define PRIVATE static
#define PUBLIC

/*--------------------------------------------------------------------*/

PRIVATE void InitIdents (void);
PRIVATE void allocate_idstringtab (void);
PRIVATE void allocate_idtab (void);

/*--------------------------------------------------------------------*/

#define HashTabSize       2048
#define STRINGTAB_PIECE  10000
#define STRINGTAB_EXTRA    500

/* typedef struct IdentRec *IDENT; -> idents.h */

struct IdentRec
{
   char  *firstposptr;
   long  length;
   Ident next;
   long  meaning;
   long  modulemeaning;
};

PRIVATE char *idstringtab_ptr;
PRIVATE char *idstringtab_endptr;

struct IdentRec *idtab_ptr;
struct IdentRec *idtab_endptr;

PRIVATE Ident HashTab [HashTabSize];

PRIVATE int initialized = 0;

/*--------------------------------------------------------------------*/

PUBLIC void slice_to_id (char *idstart, char *idstop, Ident *ref_id)
   /* char *idstart; position of first character */
   /* char *idstop;  position  a f t e r  last character */
   /* Ident *ref_id; */
{
   long  hash, length;
   Ident chain;
   Ident  NewId;

   if (! initialized) InitIdents();

   length = idstop-idstart;
   hash = ( length*256 + ((*idstart)&0xf)*16 + (*(idstop-1)&0xf) ) 
   & (HashTabSize-1);
   chain = HashTab[hash];

   for(;;) {
      if (chain == 0) {
      
	 /* not in table */
	 
	 register char *i, *freeptr, *stop;

	 NewId = idtab_ptr;
	    
	 if (idtab_ptr == idtab_endptr)
	    allocate_idtab();
         else
	    idtab_ptr++;
	    
	 /* copy id repr to idstringtab space */
	 i = idstart;
	 if (idstringtab_ptr > idstringtab_endptr)
	   allocate_idstringtab();
	 freeptr = idstringtab_ptr;
	 stop = idstop;
	 while (i < stop) {
	    *freeptr++ = *i++;
	 }
	 *freeptr = '\0';
	 freeptr++;
	 
	 /* initialize NewId */
	 NewId->firstposptr = idstringtab_ptr;
	 NewId->length = length;
	 NewId->next = HashTab[hash];
	 NewId->meaning = 0;
	 NewId->modulemeaning = 0;
	    
	 /* set hash table */
	 HashTab[hash] = NewId;

	 /* initialze idstringtab pointer for next id */
	 idstringtab_ptr = freeptr;
 
	 break;
      }

      /* current token == ident at chain ? */
      
      if (chain->length == length) {
         register char *i, *j;
	 i = idstart; j = chain->firstposptr;
	 while (i != idstop && *i == *j) {
	    i++; j++;
         }

	 if (i == idstop && *j == '\0') {
	    
	    /* found */
	    
	    NewId = chain;
	    break;
	 }
      }

      chain = chain->next;
   }

   *ref_id = NewId;
}

/*--------------------------------------------------------------------*/

PUBLIC void string_to_id (char *idstart, Ident *ref_id)
   /* char *idstart; */
   /* Ident *ref_id; */
{
   char *idstop;

   idstop = idstart;
   while (*idstop != '\0') idstop++;
   slice_to_id (idstart, idstop, ref_id);
}

/*--------------------------------------------------------------------*/

PUBLIC void id_to_string (Ident id,  char **repr)
   /* Ident id; */
   /* char **repr; */
{
   *repr = id->firstposptr;
}

/*--------------------------------------------------------------------*/

PRIVATE void InitIdents (void)
{
   long i;

   for (i = 0; i<=HashTabSize-1; i++) HashTab[i] = 0;

   allocate_idtab ();
   allocate_idstringtab ();

   initialized = 1;
}

/*--------------------------------------------------------------------*/

PRIVATE void allocate_idstringtab (void)
{
   idstringtab_ptr =
      (char *) malloc (STRINGTAB_PIECE + STRINGTAB_EXTRA);
   if (idstringtab_ptr == 0) {
      printf("memory allocation failed\n");
      exit(1);
   }
   idstringtab_endptr = idstringtab_ptr + STRINGTAB_PIECE - 1;
}

/*--------------------------------------------------------------------*/

#define IDTABPIECESIZE 500
typedef struct IdentRec IDTAB [IDTABPIECESIZE];

PRIVATE void allocate_idtab (void)
{
   idtab_ptr =
      (struct IdentRec *)
      malloc (sizeof (IDTAB /*struct IdentRec [IDTABPIECESIZE]*/ ) );
   if (idtab_ptr == 0) {
      printf("memory allocation failed\n");
      exit(1);
   }
   idtab_endptr = & idtab_ptr[IDTABPIECESIZE - 1];
}

/*--------------------------------------------------------------------*/

PUBLIC void DefMeaning (Ident id, long m) /* Ident id; long m; */
{
   id->meaning = m;
}

/*--------------------------------------------------------------------*/

PUBLIC void UndefMeaning (Ident id) /* Ident id; */
{
   id->meaning = 0;
}

/*--------------------------------------------------------------------*/

PUBLIC int HasMeaning (Ident id, long *m) /* Ident id; long *m; */
{
   if (id->meaning == 0)
      return 0;
   *m = id->meaning;
   return 1;
}

/*--------------------------------------------------------------------*/

PUBLIC void DefModuleMeaning (Ident id, long m) /* Ident id; long m; */
{
   id->modulemeaning = m;
}

/*--------------------------------------------------------------------*/

PUBLIC void UndefModuleMeaning (Ident id) /* Ident id; */
{
   id->modulemeaning = 0;
}

/*--------------------------------------------------------------------*/

PUBLIC int HasModuleMeaning (Ident id, long *m) /* Ident id; long *m; */
{
   if (id->modulemeaning == 0)
      return 0;
   *m = id->modulemeaning;
   return 1;
}

/*--------------------------------------------------------------------*/
