
'module' grammar

'use' ast extspecs misc context coder print

-------------------------------------------------------------------------------
-- Start
-------------------------------------------------------------------------------

'root'
     Init
     Compiler
     
'nonterm' Compiler

     'rule' Compiler :
	  SyntaxAnalysisSpec (-> Spec, Imports)
	  SyntaxPostlude (Spec, nil)
	  ContextAnalysisSpec (Spec, Imports -> Decls)
	  ContextPostlude (Spec, nil)
	  GenerateSpecification (Decls)

     'rule' Compiler :
	  SyntaxAnalysisImpl (-> Impl, Spec, Imports)
	  SyntaxPostlude (Spec, Impl)
	  ContextAnalysisImpl (Spec, Impl, Imports -> SDecls, IDecls)
	  ContextPostlude (SDecls, IDecls)
	  GenerateImplementation (SDecls, IDecls)
     
     'rule' Compiler :
	  SyntaxAnalysisModule (-> Impl, Spec, Imports)
	  SyntaxPostlude (Spec, Impl)
	  ContextAnalysisImpl (Spec, Impl, Imports -> SDecls, IDecls)
	  ContextPostlude (SDecls, IDecls)
	  GenerateSpecification (SDecls)
	  GenerateImplementation (SDecls, IDecls)

-------------------------------------------------------------------------------
-- Initialization
-------------------------------------------------------------------------------

'nonterm' Init

     'rule' Init :
	  InitPrint
	  InitMisc
	  InitContext
	  InitCoder

-------------------------------------------------------------------------------
-- Modules
-------------------------------------------------------------------------------

'nonterm' SyntaxAnalysisSpec (-> spec: MODULE, imports: MODULE)
     
     'rule' SyntaxAnalysisSpec (-> SpecModule, ModSpecList) :
	  SpecModule (-> SpecModule)
	  EOFILE
	  ImportedModuleList (-> ModSpecList)
	  
'nonterm' SyntaxAnalysisImpl (-> impl: MODULE, spec: MODULE, 
			         imports: MODULE)

     'rule' SyntaxAnalysisImpl (-> ImplModule, SpecModule, ImportModules) :
	  ImplModule (-> ImplModule)
	  EOFILE
	  NEXTUNIT
	  SpecModule (-> SpecModule)
	  EOFILE
	  ImportedModuleList (-> ImportModules)

'nonterm' SyntaxAnalysisModule (-> impl : MODULE, spec : MODULE, 
			           imports : MODULE)
     
     'rule' SyntaxAnalysisModule (-> ImplModule, SpecModule, ImportModules)
	  SpecModule (-> SpecModule)
	  ImplModule (-> ImplModule)
          EOFILE
	  ImportedModuleList (-> ImportModules)

'nonterm' ImplModule (-> MODULE)

     'rule' ImplModule (-> implementation (Id, Imports, Decls)):
	  "IMPLEMENTATION"
	  ImplHead (-> Pos, I) 
	  ImportList (-> Imports)
	  ImplDeclList (-> Decls)
	  "END" "IMPLEMENTATION" ";"
	  NewId (Pos, I, actual -> Id)
	  
'nonterm' ImplHead (-> POS, Ident)

     'rule' ImplHead (-> Pos, I):
	  IDENTIFIER (-> I) @ (-> Pos)
	  CheckImplModuleName (Pos, I)
	  
'nonterm' SpecModule (-> MODULE)

     'rule' SpecModule (-> specification (Id, Imports, Decls)) :
	  "SPECIFICATION"
	  SpecHead (-> Pos, I)
	  ImportList (-> Imports)
	  SpecDeclList (-> Decls)
	  "END" "SPECIFICATION" ";"
	  NewId (Pos, I, actual -> Id)

'nonterm' SpecHead (-> POS, Ident)

     'rule' SpecHead (-> Pos, I):
	  IDENTIFIER (-> I) @ (-> Pos)
	  CheckSpecModuleName (Pos, I)
	  
'nonterm' ImportedModuleList (-> MODULE)

     'rule' ImportedModuleList (-> seq (Mod, ModList)):
	  NEXTUNIT
	  SpecModule (-> Mod)
	  SkipImplModule
	  ImportedModuleList (-> ModList)
	  
     'rule' ImportedModuleList (-> nil) :
	  
'nonterm' SkipImplModule

     'rule' SkipImplModule :
	  SkipRestOfFile
	  
-------------------------------------------------------------------------------
-- Imports
-------------------------------------------------------------------------------

'nonterm' ImportList (-> IMPORT)

     'rule' ImportList (-> seq (Import, ImportList)) :
	  Import (-> Import)
	  ";"
	  ImportList (-> ImportList)
	  
     'rule' ImportList (-> nil) :
	  
'nonterm' Import (-> IMPORT)
     
     'rule' Import (-> ImportModules) :
	  "IMPORT" @ (-> Pos)
	  IdentifierList (-> IdList)
	  ExpandImportModules (IdList -> ImportModules)
	  
     'rule' Import (-> ImportItems) :
	  "IMPORT" @ (-> ItemPos)
	  IdentifierList (-> IdList)
	  "FROM" @ (-> ModPos)
	  IDENTIFIER (-> ModIdent)
	  InsertImportFile (ModIdent)
	  NewId (ModPos, ModIdent, actual -> ModId)
	  ExpandImportItems (IdList, ModId -> ImportItems)

     'rule' Import (-> importforeign (File)) :
	  "IMPORT" "FOREIGN"   
	  STRINGCONST (-> File) 

'action' ExpandImportModules (IDLIST -> IMPORT)
	  
     'rule' ExpandImportModules (idlist(Id, nil) -> importmodule (Id)) :
	  Id'Ident -> ModIdent
	  InsertImportFile (ModIdent)
	  
     'rule' ExpandImportModules (idlist(Id, IdList) -> 
				 seq (importmodule (Id), ImportModules)) :
	  Id'Ident -> ModIdent
	  InsertImportFile (ModIdent)
	  ExpandImportModules (IdList -> ImportModules)
	  
'action' ExpandImportItems (IDLIST, ID -> IMPORT)
	  
     'rule' ExpandImportItems (idlist (Id, nil), ModId ->
			       importitem (ModId, Id)) :
	  
     'rule' ExpandImportItems (idlist (Id, IdList), ModId ->
			       seq (importitem (ModId, Id), ImportItems)) :
	  ExpandImportItems (IdList, ModId -> ImportItems)

-------------------------------------------------------------------------------
-- Declarations
-------------------------------------------------------------------------------

'nonterm' SpecDeclList (-> DECL)
     
     'rule' SpecDeclList (-> seq (Pos, D, L)) :
	  SpecDecl (-> D) @ (-> Pos)
	  ";"
	  SpecDeclList (-> L)
	  
     'rule' SpecDeclList (-> nil (Pos)) :
	  @ (-> Pos)
	  
'nonterm' SpecDecl (-> DECL)
     
     'rule' SpecDecl (-> ClassDecl) :
	  ClassSpecDecl (-> ClassDecl)
	  
     'rule' SpecDecl (-> ProcDecl) :
	  ProcSpecDecl (-> ProcDecl)
	  
     'rule' SpecDecl (-> VarDeclList) :
	  "EXPORT" @ (-> Pos)
	  "FOREIGN"
	  "VAR"
	  IdentifierList (-> IdList)
	  ":"
	  Type (-> Type)
	  VarInit (-> Init)
	  ExpandGlobalVarDecls (Pos, IdList, foreign, Type, Init -> 
				VarDeclList) 
	   
     'rule' SpecDecl (-> Decl) :
	  Decl (-> Decl)
	  
'nonterm' ImplDeclList (-> DECL)
     
     'rule' ImplDeclList (-> seq (Pos, D, L)) :
	  ImplDecl (-> D) @ (-> Pos)
	  ";"
	  ImplDeclList (-> L)
	  
     'rule' ImplDeclList (-> nil (Pos)) :
	  @ (-> Pos)
	  
'nonterm' ImplDecl (-> DECL)
     
     'rule' ImplDecl (-> ClassDecl) :
	  ClassImplDecl (-> ClassDecl)
	  
     'rule' ImplDecl (-> ProcDecl) :
	  ProcImplDecl (-> ProcDecl)
	  
     'rule' ImplDecl (-> Decl) :
	  Decl (-> Decl)
	  
'nonterm' Decl (-> DECL)
     
     'rule' Decl (-> TypeDecl) :
	  TypeDecl (-> TypeDecl)
	  
     'rule' Decl (-> ConstDecl) :
	  ConstDecl (-> ConstDecl)
	  
     'rule' Decl (-> VarDecl) :
	  VarDecl (-> VarDecl)
	  
     'rule' Decl (-> ExceptDecl) :
	  ExceptDecl (-> ExceptDecl)
	  
     'rule' Decl (-> ForeignVarDecl)
	  ForeignVarDecl (-> ForeignVarDecl)
     
     'rule' Decl (-> ForeignProcDecl)
	  ForeignProcDecl (-> ForeignProcDecl)
     
     'rule' Decl (-> ForeignTypeDecl) :
	  ForeignTypeDecl (-> ForeignTypeDecl)

-------------------------------------------------------------------------------
-- Type Declaration
-------------------------------------------------------------------------------

'nonterm' TypeDecl (-> DECL)
     
     'rule' TypeDecl (-> type (Pos, Id, Type)) :
	  "TYPE" @ (-> Pos)
	  IDENTIFIER (-> I) @(-> IPos) 
	  FGenericParamSpec (-> Params)
	  "="
 	  Type (-> Type)
	  NewId (IPos, I, actual -> Id)
	  (|
	       where (Params -> nil)
	  ||
	       Error ("invalid generic parameters in non object types", Pos)
	  |)
	   
-------------------------------------------------------------------------------
-- Foreign types
-------------------------------------------------------------------------------

'nonterm' ForeignTypeDecl (-> DECL)
     
     'rule' ForeignTypeDecl (-> foreigntype (Pos, Id, Type)) :
	  "FOREIGN" "TYPE" @ (-> Pos)
	  IDENTIFIER (-> I) @(-> IPos) 
	  "="
 	  Type (-> Type)
	  NewId (IPos, I, actual -> Id)
	   
-------------------------------------------------------------------------------
-- Formal generic parameters
-------------------------------------------------------------------------------
	  
'nonterm' FGenericParamSpec (-> FPARAMLIST)
	  
     'rule' FGenericParamSpec (-> FParams) :
	  "[" FGenericParamList (-> FParams) "]"
	  
     'rule' FGenericParamSpec (-> nil) :
	  
'nonterm' FGenericParamList (-> FPARAMLIST)
     
     'rule' FGenericParamList (-> fparamlist (FParam, Tail))
	  FGenericParam (-> FParam)
	  ","
	  FGenericParamList (-> Tail)
	  
     'rule' FGenericParamList (-> fparamlist (FParam, nil)) :
	  FGenericParam (-> FParam)

'nonterm' FGenericParam (-> FPARAM)
     
     'rule' FGenericParam (-> unconstrained (Pos, ObjKey)) :
	  IDENTIFIER (-> I) @ (-> Pos)
	  NewId (Pos, I, actual -> ObjKey)
	  
     'rule' FGenericParam (-> constrained (Pos, ObjKey, 
					   ApplKey)) :
	  IDENTIFIER (-> I) @ (-> Pos)
	  "->"
	  IdAppl (-> ApplKey)
	  NewId (Pos, I, actual -> ObjKey)

-------------------------------------------------------------------------------
-- Class specific stuff
-------------------------------------------------------------------------------

'nonterm' ClassSpecDecl (-> DECL)	  
	  
     'rule' ClassSpecDecl (-> classspec (Pos, Id, Params, ClassInterface)) :
	  "TYPE" @ (-> Pos)
	  IDENTIFIER (-> I) @(-> IPos) 
	  FGenericParamSpec (-> Params)
	  "="
	  ClassInterface (-> ClassInterface)
	  "END" "OBJECT"
	  NewId (IPos, I, actual -> Id)
	  
'nonterm' ClassImplDecl (-> DECL)	  
	  
     'rule' ClassImplDecl (-> class (Pos, Id, Params, ClassInterface, 
				     ClassImpl)):
	  "TYPE" @ (-> Pos)
	  IDENTIFIER (-> I) @(-> IPos) 
	  FGenericParamSpec (-> Params)
	  "="
	  ClassInterface (-> ClassInterface)
	  "BODY"
	  MethodImplList (-> ClassImpl)
	  "END" "OBJECT"
	  NewId (IPos, I, actual -> Id)
	  
     'rule' ClassImplDecl (-> classimpl (Pos, Id, ClassImpl)) :
	  "TYPE" @ (-> Pos)
	  IDENTIFIER (-> I) @(-> IPos) 
	  FGenericParamSpec (-> Params)
	  "="
	  SuperOpt (-> Super)
	  AbstractOpt (-> Abstr)
	  "OBJECT"
	  FParamSpec (-> ObjParams)
	  "BODY"
	  MethodImplList (-> ClassImpl)
	  "END" "OBJECT"
	  NewId (IPos, I, actual -> Id)
	  (|
	       where (Params -> nil)
	  ||
	       Error ("invalid formal generic parameters in class implementation",
		      Pos)
	  |)
	  (|
	       where (Super -> none)
	  ||
	       Error ("invalid super type in class implementation", Pos)
	  |)
	  (|
	       where (Abstr -> concrete)
	  ||
	       Error ("invalid ABSTRACT attribute in class implementation", Pos)
	  |)
	  (|
	       where (ObjParams -> nil)
	  ||
	       Error ("invalid object parameters in class implementation", Pos)
	  |)
	  
'nonterm' ClassInterface (-> CLASSINTERFACE)

     'rule' ClassInterface (-> localdefs (Super, Abstr, ObjParams,
					  seq (Pos, Public, Protect), 
					  InstVars)) :
	  SuperOpt (-> Super)
	  AbstractOpt (-> Abstr)
	  "OBJECT" 
	  FParamSpec (-> ObjParams)
	  Public 
	  MethodSpecList (-> Public) @ (-> Pos)
	  Protected
	  MethodSpecList (-> Protect)
	  "STATE"
	  InstVarDeclList (-> InstVars)

'nonterm' Public 

     'rule' Public 
	  "PUBLIC"
	  AccessMode <- public

'nonterm' Protected 

     'rule' Protected
	  "PROTECTED"
	  AccessMode <- protected

'nonterm' SuperOpt (-> SUPERTYPE) 
     
     'rule' SuperOpt (-> super (ApplKey)) :
	  IdAppl (-> ApplKey)
     
     'rule' SuperOpt (-> error) :
                                  -- instantiation of an generic type 
                                  -- not yet implemented
	  IdAppl (-> ApplKey)
	  "[" @( -> Pos)
	  AGenParamList (-> _)
	  "]"
	  Error ("generic instantiation as supertype : not yet implemented",
		 Pos) 
	  
     'rule' SuperOpt (-> none) :

'nonterm' AbstractOpt (-> ABSTRACT)
     
     'rule' AbstractOpt (-> abstract) :
	  "ABSTRACT"
	  
     'rule' AbstractOpt (-> concrete) :
	  
-------------------------------------------------------------------------------
-- Method declarations
-------------------------------------------------------------------------------

'var' AccessMode : ACCESSMODE

'nonterm' MethodSpecList (-> DECL)
     
     'rule' MethodSpecList (-> seq (Pos, MethodSpec, Tail)) :
	  MethodSpec (-> MethodSpec) @(-> Pos)
	  ";"
	  MethodSpecList (-> Tail)
     
     'rule' MethodSpecList (-> nil (Pos)) :
	  @ (-> Pos)
	  
'nonterm' MethodSpec (-> DECL)
     
     'rule' MethodSpec (-> methodspec (Pos, Id, Mode, Abstr, 
				       firstdef, FParams, Result)) :
	  AbstractOpt (-> Abstr)
	  "METHOD" @ (-> Pos)
	  ExtendedSignature (-> Id, FParams, Result)
	  AccessMode -> Mode
	  
'nonterm' MethodImplList (-> DECL)
     
     'rule' MethodImplList (-> seq (Pos, MethodImpl, Tail)) :
	  MethodImpl (-> MethodImpl) @(-> Pos)
	  ";"
	  MethodImplList (-> Tail)
     
     'rule' MethodImplList (-> nil (Pos)) :
	  @ (-> Pos)
  
'nonterm' MethodImpl (-> DECL)
     
     'rule' MethodImpl (-> methodimpl (Pos, Id, Redef, FParams, 
				       Result, Locals, Body)) :
	  Redefined (-> Redef)
	  "METHOD" @ (-> Pos)
	  ExtendedSignature (-> Id, FParams, Result)
	  LocalVarDeclList (-> Locals)
	  RoutineBody (-> Body)
	  
     'rule' MethodImpl (-> initially (Pos, FParams, Locals, Body)) :
	  "INITIALLY" @ (-> Pos)
	  FParamSpec (-> FParams) 
	  LocalVarDeclList (-> Locals)
	  RoutineBody (-> Body)
	  
     'rule' MethodImpl (-> finally (Pos, Locals, Body)) :
	  "FINALLY" @ (-> Pos)
	  LocalVarDeclList (-> Locals)
	  RoutineBody (-> Body)
	  
'nonterm' Redefined (-> METHODFLAG)
     
     'rule' Redefined (-> redef) :
	  "REDEFINED"
	  
     'rule' Redefined (-> firstdef) :

-------------------------------------------------------------------------------
-- Procedure declaration
-------------------------------------------------------------------------------

'nonterm' ProcSpecDecl (-> DECL)
	  
     'rule' ProcSpecDecl (-> procspec (Pos, Id, ExportFlag, CallConv, 
				       FParams, Result)) :
	  ProcSpecAttr (-> Pos, ExportFlag, CallConv)
	  ExtendedSignature (-> Id, FParams, Result)

'nonterm' ProcSpecAttr (-> POS, EXPORTFLAG, CALLCONV)
     
     'rule' ProcSpecAttr (-> Pos, root, CallConv) :
	  "ROOT" @ (-> Pos)
	  CallingConvention (-> CallConv)
	  "PROCEDURE"
     
     'rule' ProcSpecAttr (-> Pos, foreign, CallConv) :
	  "EXPORT" @ (-> Pos) "FOREIGN" 
	  CallingConvention (-> CallConv)
	  "PROCEDURE"
	  
     'rule' ProcSpecAttr (-> Pos, cool, c) :
	  "PROCEDURE" @ (-> Pos)
	  
'nonterm' ProcImplDecl (-> DECL)
	  
     'rule' ProcImplDecl (-> procimpl (Pos, Id, FParams, Result, Locals, 
				       Body)) :
	  "PROCEDURE" @ (-> Pos) 
	  ExtendedSignature (-> Id, FParams, Result)
	  LocalVarDeclList (-> Locals)
	  RoutineBody (-> Body)
 
'nonterm' ExtendedSignature (-> RoutineId : ID, FParams : FPARAMLIST,
			     Result: TYPEINDEX)
     
     'rule' ExtendedSignature (-> Id, FParams, Result) :
	  IDENTIFIER (-> I) @ (-> IPos)
	  FParamSpec (-> FParams) 
	  ResultType (-> Result)
	  NewId (IPos, I, actual -> Id)
	  
'nonterm' RoutineBody (-> BLOCK)

     'rule' RoutineBody (-> block (BeginPos, EndPos, Body)) :
	  "DO" @ (-> BeginPos)
	  StmtList (-> Body)
	  "END" @ (-> EndPos) "DO"
	  
-------------------------------------------------------------------------------
-- Formal parameters
-------------------------------------------------------------------------------

'nonterm' FParamSpec (-> FPARAMLIST)
     
     'rule' FParamSpec (-> FParams) :
	  "(" FParamList (-> FParams) ")"
	  
     'rule' FParamSpec (-> nil) :
	  
'nonterm' FParamList (-> FPARAMLIST)
     
     'rule' FParamList (-> L) :
	  FParamList_h (-> L)
	  
     'rule' FParamList (-> nil) :
	  
'nonterm' FParamList_h (-> FPARAMLIST)
     
     'rule' FParamList_h (-> FParams) :
	  Mode (-> Mode) @ (-> Pos)
	  IdentifierList (-> IdList)
	  ":"
	  Type (-> Type)
	  ","
	  FParamList_h (-> Tail)
	  ExpandFParamDecls (Pos, IdList, Mode, Type, Tail -> FParams)
	  
     'rule' FParamList_h (-> FParams) :
	  Mode (-> Mode) @ (-> Pos)
	  IdentifierList (-> IdList)
	  ":"
	  Type (-> Type)
	  ExpandFParamDecls (Pos, IdList, Mode, Type, nil -> FParams)

     'rule' FParamList_h (-> ellipsis (Pos))
	  ".." @ (-> Pos)
	  
'nonterm' Mode (-> PMODE)
     
     'rule' Mode (-> in) :
	  "IN"
     
     'rule' Mode (-> out) :
	  "OUT"	  
     
     'rule' Mode (-> inout) :
	  "INOUT"
	  
     'rule' Mode (-> in) :
	  
'nonterm' ResultType (-> TYPEINDEX)
     
     'rule' ResultType (-> Type) :
	  ":" Type (-> Type)
	  
     'rule' ResultType (-> Void) :
	  @ (-> Pos)
	  NewTypeIndex (Pos, simple (void) -> Void)
	  
'action' ExpandFParamDecls (POS, IDLIST, PMODE, TYPEINDEX, FPARAMLIST -> 
			    FPARAMLIST)
	  
     'rule' ExpandFParamDecls (_, nil, _, _, RestList -> RestList)
			       
     'rule' ExpandFParamDecls (Pos, idlist (Id, IdList), Mode, Type, 
			       RestList ->
			       fparamlist (fparam (Pos, Id, Mode, Type),
					   Tail)) :
	  ExpandFParamDecls (Pos, IdList, Mode, Type, RestList -> Tail)
	  
-------------------------------------------------------------------------------
-- Foreign procedures
-------------------------------------------------------------------------------
     
'nonterm' ForeignProcDecl (-> DECL) 
     
     'rule' ForeignProcDecl (-> foreignproc (Pos, Id, CallConv, Params, 
					     ReturnType)) :
	  "FOREIGN" 
	  CallingConvention (-> CallConv)
	  "PROCEDURE" @ (-> Pos)
	  IDENTIFIER (-> I) @ (-> IPos)
	  FParamSpec (-> Params)
	  ResultType (-> ReturnType)
	  NewId (IPos, I, actual -> Id)

'nonterm' CallingConvention (-> CALLCONV)
     
     'rule' CallingConvention (-> pascal) :
	  "PASCAL"
     
     'rule' CallingConvention (-> c) :
	  
-------------------------------------------------------------------------------
-- Variables
-------------------------------------------------------------------------------

'nonterm' VarDecl (-> DECL)

     'rule' VarDecl (-> VarDeclList) :
	  "VAR" @ (-> Pos)
	  IdentifierList (-> IdList)
	  ":"
	  Type (-> Type)
	  VarInit (-> Init)
	  ExpandGlobalVarDecls (Pos, IdList, cool, Type, Init -> VarDeclList) 
			    
'nonterm' VarInit (-> EXPR)	  
	  
     'rule' VarInit (-> Expr) : 
	  ":=" Expr (-> Expr)
	  
     'rule' VarInit (-> nil (Pos))
	  @ (-> Pos)
	  
'action' ExpandGlobalVarDecls (POS, IDLIST, EXPORTFLAG, TYPEINDEX, EXPR -> 
			       DECL)
	  
     'rule' ExpandGlobalVarDecls (Pos, idlist (Id, nil), Export, Type, Init 
				  -> globalvar (Pos, Id, Export, Type, Init)) :
	  
     'rule' ExpandGlobalVarDecls (Pos, idlist (Id, IdList), Export, Type, Init 
				  -> seq (Pos, globalvar (Pos, Id, Export, 
							  Type, Init),
					  VarDecls)) :
	  ExpandGlobalVarDecls (Pos, IdList, Export, Type, Init -> VarDecls)
	  
-------------------------------------------------------------------------------
-- Local Variables
-------------------------------------------------------------------------------

'nonterm' LocalVarDeclList (-> DECL)
     
     'rule' LocalVarDeclList (-> seq (Pos, VarDecl, Tail)) :
	  LocalVarDecl (-> VarDecl) @ (-> Pos) 
	  ";"
	  LocalVarDeclList (-> Tail)
     
     'rule' LocalVarDeclList (-> nil (Pos)) :
	  @ (-> Pos)
	  
'nonterm' LocalVarDecl (-> DECL)

     'rule' LocalVarDecl (-> VarDeclList) :
	  "VAR" @ (-> Pos)
	  IdentifierList (-> IdList) 
	  ":"
	  Type (-> Type)
	  VarInit (-> Init)
	  ExpandLocalVarDecls (Pos, IdList, Type, Init -> VarDeclList) 
	  
'action' ExpandLocalVarDecls (POS, IDLIST, TYPEINDEX, EXPR -> DECL)
	  
     'rule' ExpandLocalVarDecls (Pos, idlist (Id, nil), Type, Init -> 
				 localvar (Pos, Id, Type, Init)) :
	  
     'rule' ExpandLocalVarDecls (Pos, idlist (Id, IdList), Type, Init -> 
				 seq (Pos, localvar (Pos, Id, Type, Init),
				      VarDecls)) :
	  ExpandLocalVarDecls (Pos, IdList, Type, Init -> VarDecls)
	 
-------------------------------------------------------------------------------
-- Instance Variables
-------------------------------------------------------------------------------
	  
'nonterm' InstVarDeclList (-> DECL)
     
     'rule' InstVarDeclList (-> seq (Pos, VarDecl, Tail)) :
	  InstVarDecl (-> VarDecl) @ (-> Pos) 
	  ";"
	  InstVarDeclList (-> Tail)
     
     'rule' InstVarDeclList (-> nil (Pos)) :
	  @ (-> Pos)
	  
'nonterm' InstVarDecl (-> DECL)

     'rule' InstVarDecl (-> InstVarDecls) :
	  IdentifierList (-> IdList) @ (-> Pos)
	  ":"
	  Type (-> Type)
	  ExpandInstVarDecls (Pos, IdList, Type -> InstVarDecls)
	  
'action' ExpandInstVarDecls (POS, IDLIST, TYPEINDEX -> DECL)
	  
     'rule' ExpandInstVarDecls (Pos, idlist (Id, nil), Type ->
				instvar (Pos, Id, Type)) :
	  
     'rule' ExpandInstVarDecls (Pos, idlist (Id, IdList), Type ->
				seq (Pos, instvar (Pos, Id, Type),
				     VarDecls)) :
	  ExpandInstVarDecls (Pos, IdList, Type -> VarDecls)

-------------------------------------------------------------------------------
-- Foreign Variables
-------------------------------------------------------------------------------

'nonterm' ForeignVarDecl (-> DECL)

     'rule' ForeignVarDecl (-> VarDeclList) :
	  "FOREIGN" "VAR" @ (-> Pos)
	  IdentifierList (-> IdList)
	  ":"
	  Type (-> Type)
	  ExpandForeignVarDecls (Pos, IdList, Type -> VarDeclList)
	  
'action' ExpandForeignVarDecls (POS, IDLIST, TYPEINDEX -> DECL)
     
     'rule' ExpandForeignVarDecls (Pos, idlist (Id, nil), Type -> 
				   foreignvar (Pos, Id, Type)) :
	  
     'rule' ExpandForeignVarDecls (Pos, idlist (Id, IdList), Type -> 
				   seq (Pos, foreignvar (Pos, Id, Type), 
					VarDecls)) :
	  ExpandForeignVarDecls (Pos, IdList, Type -> VarDecls)
	  
-------------------------------------------------------------------------------
-- Constant Declarations
-------------------------------------------------------------------------------

'nonterm' ConstDecl (-> DECL)

     'rule' ConstDecl (-> ConstDecls) :
	  "CONST" @ (-> Pos)
	  IdentifierList (-> IdList)
	  ":" 
	  Type (-> Type)
	  "="
	  Expr (-> Init)
	  ExpandConstDecls (Pos, IdList, Type, Init -> ConstDecls)
	  
'action' ExpandConstDecls (POS, IDLIST, TYPEINDEX, EXPR -> DECL)
	  
     'rule' ExpandConstDecls (Pos, idlist (Id, nil), Type, Init ->
			      const (Pos, Id, Type, Init)) :
	  
     'rule' ExpandConstDecls (Pos, idlist (Id, IdList), Type, Init ->
			      seq (Pos, const (Pos, Id, Type, Init),
				   VarDecls)) :
	  ExpandConstDecls (Pos, IdList, Type, Init -> VarDecls)
	  
-------------------------------------------------------------------------------
-- Exception Declarations
-------------------------------------------------------------------------------

'nonterm' ExceptDecl (-> DECL)

     'rule' ExceptDecl (-> ExceptDecls) :
	  "EXCEPTION" @ (-> Pos)
	  IdentifierList (-> IdList)
	  FParamSpec (-> FParams)
	  ExpandExceptDecls (Pos, IdList, FParams -> ExceptDecls)
			
'action' ExpandExceptDecls (POS, IDLIST, FPARAMLIST -> DECL)
	  
     'rule' ExpandExceptDecls (Pos, idlist (Id, nil), FParams ->
			       exception (Pos, Id, FParams)) :
	  
     'rule' ExpandExceptDecls (Pos, idlist (Id, IdList), FParams ->
			       seq (Pos, exception (Pos, Id, FParams),
				    ExceptDecls)) :
	  ExpandExceptDecls (Pos, IdList, FParams -> ExceptDecls)
	  
-------------------------------------------------------------------------------
-- Types
-------------------------------------------------------------------------------

'nonterm' Type (-> TYPEINDEX)
     
     'rule' Type  (-> Type) :
	  SimpleType (-> SimpleType) @ (-> Pos)
	  NewTypeIndex (Pos, simple (SimpleType) -> Type)
	    
     'rule' Type (-> Type) :
	  ComposedType (-> CompType) @ (-> Pos)
	  NewTypeIndex (Pos, composite (CompType) -> Type)
	  
     'rule' Type (-> Type)
	  GenericType (-> GenType) @ (-> Pos)
	  NewTypeIndex (Pos, generic (GenType) -> Type)
	  
-------------------------------------------------------------------------------
-- Simple Types
-------------------------------------------------------------------------------

'nonterm' SimpleType (-> SIMPLETYPE)
     
     'rule' SimpleType (-> bool) :
	  "BOOL"
	    
     'rule' SimpleType (-> int) :
	  "INT"
	  
     'rule' SimpleType (-> unsignedint) :
	  "UNSIGNED" "INT"
	  
     'rule' SimpleType (-> shortint) :
	  "SHORT" "INT"
	  
     'rule' SimpleType (-> unsignedshortint) :
	  "UNSIGNED" "SHORT" "INT"
	  
     'rule' SimpleType (-> longint) :
	  "LONG" "INT"
	  
     'rule' SimpleType (-> unsignedlongint) :
	  "UNSIGNED" "LONG" "INT"
	  
     'rule' SimpleType (-> float) :
	  "FLOAT"
	  
     'rule' SimpleType (-> double) :
	  "DOUBLE"
	  
     'rule' SimpleType (-> char) :
	  "CHAR"
	  
     'rule' SimpleType (-> string) :
	  "STRING"
	  
     'rule' SimpleType (-> address) :
	  "ADDRESS"
	    
'nonterm' ComposedType (-> COMPOSITETYPE)

     'rule' ComposedType (-> typename (ApplKey)) :
	  IdAppl (-> ApplKey)
     
     'rule' ComposedType (-> Enum) :
	  EnumType (-> Enum)
	  
     'rule' ComposedType ( -> Record) :
	  RecordType (-> Record)
	  
     'rule' ComposedType (-> Union) :
	  UnionType (-> Union)
	  
     'rule' ComposedType (-> Array) :
	  ArrayType (-> Array)
	  
     'rule' ComposedType (-> Ref) :
	  RefType (-> Ref)
	  
     'rule' ComposedType (-> Function) :
	  ProcType (-> Function)
     
     'rule' ComposedType (-> Method) :
	  MethodType (-> Method)

'nonterm' GenericType (-> GENERICTYPE)
	  	  
     'rule' GenericType (-> genericinst (Type, Params)) :
	  IdAppl (-> Type)
	  "["
	  AGenParamList (-> Params)
	  "]" 

'nonterm' AGenParamList (-> TYPEINDEXLIST)

     'rule' AGenParamList (-> typeindexlist (Type, TypeList)) :
	  Type (-> Type)
	  ","
	  AGenParamList (-> TypeList)
 
     'rule' AGenParamList (-> typeindexlist (Type, nil)) :
	  Type (-> Type)
	 
-------------------------------------------------------------------------------
-- Enum Type
-------------------------------------------------------------------------------

'nonterm' EnumType (-> COMPOSITETYPE)
     
     'rule' EnumType (-> enum (EnumeratorList)) :
	  "ENUM" 
	  EnumeratorList (-> EnumeratorList)
	  "END" "ENUM"

'nonterm' EnumeratorList (-> ENUMERATORLIST)
     
     'rule' EnumeratorList (-> L) :
	  EnumeratorList_h (-> L)
	  
     'rule' EnumeratorList (-> nil) :
	  
'nonterm' EnumeratorList_h (-> ENUMERATORLIST)
	  
     'rule' EnumeratorList_h (-> enumeratorlist (Enumerator, Tail)) :
	  Enumerator (-> Enumerator)
	  ","
	  EnumeratorList_h (-> Tail)
	  
     'rule' EnumeratorList_h (-> enumeratorlist (Enumerator, nil)) :
	  Enumerator (-> Enumerator)
     
'nonterm' Enumerator (-> ENUMERATOR)
     
     'rule' Enumerator (-> enumerator (Pos, Id)) :
	  IDENTIFIER (-> I) @ (-> Pos)
	  NewId (Pos, I, actual -> Id)

-------------------------------------------------------------------------------
-- Record Type
-------------------------------------------------------------------------------

'nonterm' RecordType (-> COMPOSITETYPE)

     'rule' RecordType ( -> record (Fields) ) :
	  "RECORD"
	  FieldList (-> Fields)
	  "END" "RECORD"

'nonterm' FieldList (-> FIELD)
	  
     'rule' FieldList (-> seq (Field, Tail)) :
	  Field (-> Field)
	  ";"
	  FieldList (-> Tail)
	  
     'rule' FieldList (-> nil) :
	  
'nonterm' Field (-> FIELD)
     
     'rule' Field (-> FieldDecls) :
	  IdentifierList (-> IdList) @ (-> Pos)
	  ":"
	  Type (-> Type)
	  ExpandFieldDecls (Pos, IdList, Type -> FieldDecls)
	  
'action' ExpandFieldDecls (POS, IDLIST, TYPEINDEX -> FIELD)
	  
     'rule' ExpandFieldDecls (Pos, idlist (Id, nil), Type ->
			      field (Pos, Id, Type)) :
     
     'rule' ExpandFieldDecls (Pos, idlist (Id, IdList), Type ->
			      seq (field (Pos, Id, Type),
				   FieldDecls)) :
	  ExpandFieldDecls (Pos, IdList, Type -> FieldDecls)
	  
-------------------------------------------------------------------------------
-- UnionType
-------------------------------------------------------------------------------
	  
'nonterm' UnionType (-> COMPOSITETYPE)

     'rule' UnionType (-> union (Fields)) :
	  "UNION" 
	  FieldList (-> Fields)
	  "END" "UNION"

-------------------------------------------------------------------------------
-- Array Type
-------------------------------------------------------------------------------

'nonterm' ArrayType (-> COMPOSITETYPE)

     'rule' ArrayType (-> Array) :
	  "ARRAY" @ (-> Pos)
	  DimensionList (-> DimensionList)
	  "OF"
	  Type (-> ElemType)
	  ExpandMultiArray (Pos, DimensionList, ElemType -> Array)
	  
     'rule' ArrayType (-> openarray (ElemType)) :
	  "ARRAY"
	  "OF"
	  Type (-> ElemType)

'nonterm' DimensionList (-> EXPRLIST)

     'rule' DimensionList (-> exprlist (Range, RangeList)) :
	  Dimension (-> Range)
	  DimensionList (-> RangeList)
	  
     'rule' DimensionList (-> exprlist (Range, nil)) :
	  Dimension (-> Range)
	  
'nonterm' Dimension (-> EXPR)

     'rule' Dimension (-> range (Pos, LowerBound, UpperBound)) :
	  "[" @ (-> Pos)
	  Expr (-> LowerBound)
	  ".."
	  Expr (-> UpperBound)
	  "]"
	  
     'rule' Dimension (-> range (Pos, posintliteral (Pos, 1),
				 UpperBound)) :
	  "[" @ (-> Pos)
	  Expr (-> UpperBound)
	  "]"
	  
'action' ExpandMultiArray (POS, EXPRLIST, TYPEINDEX -> COMPOSITETYPE)
     
     'rule' ExpandMultiArray (Pos, exprlist (Dimension, nil), ElemType -> 
			      array (Dimension, ElemType)) :
     
     'rule' ExpandMultiArray (Pos, exprlist (Dimension,
					     DimensionList), ElemType
			      -> array (Dimension, Type)) : 
	  ExpandMultiArray (Pos, DimensionList, ElemType -> ArrayType)
	  NewTypeIndex (Pos, composite (ArrayType) -> Type)

-------------------------------------------------------------------------------
-- Ref Type
-------------------------------------------------------------------------------
	  
'nonterm' RefType (-> COMPOSITETYPE)

     'rule' RefType (-> ref (RefedType)) :
	  "REF"
	  Type (-> RefedType)

-------------------------------------------------------------------------------
-- Procedure and method Type
-------------------------------------------------------------------------------

'nonterm' ProcType (-> COMPOSITETYPE)
	  
     'rule' ProcType (-> procedure (Params, ResType)) :
	  "PROCEDURE"
	  ProcTypeFParamSpec (-> Params)
	  ResultType (-> ResType)

'nonterm' MethodType (-> COMPOSITETYPE)
	  
     'rule' MethodType (-> method (Params, ResType)) :
	  "METHOD"
	  ProcTypeFParamSpec (-> Params)
	  ResultType (-> ResType)

'nonterm' ProcTypeFParamSpec (-> FPARAMLIST)
     
     'rule' ProcTypeFParamSpec (-> FParams) :
	  "(" ProcTypeFParamList (-> FParams) ")"
	  
     'rule' ProcTypeFParamSpec (-> nil) :
	  
'nonterm' ProcTypeFParamList (-> FPARAMLIST)
     
     'rule' ProcTypeFParamList (-> L) :
	  ProcTypeFParamList_h (-> L)
	  
     'rule' ProcTypeFParamList (-> nil) :
	  
'nonterm' ProcTypeFParamList_h (-> FPARAMLIST)
     
     'rule' ProcTypeFParamList_h (-> fparamlist (FParam, Tail)) :
	  ProcTypeFParam (-> FParam)  @ (-> Pos)
	  ","
	  ProcTypeFParamList_h (-> Tail)
	  
     'rule' ProcTypeFParamList_h (-> fparamlist (FParam, nil)) :
	  ProcTypeFParam (-> FParam) @ (-> Pos)
	   
     'rule' ProcTypeFParamList_h (-> ellipsis (Pos)) :
	  ".." @ (-> Pos)
	  
'nonterm' ProcTypeFParam (-> FPARAM)
     
     'rule' ProcTypeFParam (-> fparam (Pos, Id, Mode, Type)) :
	  Mode (-> Mode) @ (-> Pos)
	  Type (-> Type)
	  DummyId -> Id
	  
-------------------------------------------------------------------------------
-- Statements
-------------------------------------------------------------------------------

'nonterm' StmtList (-> STMT)
     
     'rule' StmtList (-> seq (Pos, S1, S2)) :
	  Stmt (-> S1) @ (-> Pos)
	  ";"
	  StmtList (-> S2)
	  
     'rule' StmtList (-> nil (Pos)) :
	  @ (-> Pos)
	  
'nonterm' Stmt (-> STMT)
     
     'rule' Stmt (-> assign (Pos, DestType, Lhs, Rhs)) :
	  Expr (-> Lhs) @ (-> Pos) ":=" Expr (-> Rhs)
	  UnspecTypeIndex -> DestType
	  
     'rule' Stmt (-> Call) :
	  CallStmt (-> Call)
	  
     'rule' Stmt (-> Return) :
	  ReturnStmt (-> Return)
	  
     'rule' Stmt (-> If) :
	  IfStmt (-> If)
	  
     'rule' Stmt (-> Select) : 
	  SelectStmt (-> Select)
	  
     'rule' Stmt (-> TypeSelect) :
	  TypeSelectStmt (-> TypeSelect)
	  
     'rule' Stmt (-> Loop) :
	  LoopStmt (-> Loop)
	  
     'rule' Stmt (-> WhileLoop) :
	  WhileStmt (-> WhileLoop)
	  
     'rule' Stmt (-> ForLoop) :
	  ForStmt (-> ForLoop)
	  
     'rule' Stmt (-> exit (Pos)) :
	  "EXIT" @ (-> Pos)
	  
     'rule' Stmt (-> Raise) :
	  RaiseStmt (-> Raise)
	  
     'rule' Stmt (-> Try) :
	  TryStmt (-> Try)
	  
     'rule' Stmt (-> retry (Pos)) :
	  "RETRY" @ (-> Pos)
	  
     'rule' Stmt (-> reraise (Pos)) :
	  "RERAISE" @ (-> Pos)
	  
     'rule' Stmt (-> delete (Pos, Object)) :
	  "DELETE" @ (-> Pos)
	  Expr (-> Object)
	  
     'rule' Stmt (-> nil (Pos)) :
	  @ (-> Pos)
	  
-------------------------------------------------------------------------------
-- Call und Return Statement
-------------------------------------------------------------------------------
	  
'nonterm' CallStmt (-> STMT)

     'rule' CallStmt (-> StmtCall) :
	  Expr (-> Call) @ (-> Pos)
	  NormaliseCall (Pos, Call -> StmtCall)
	  
'action' NormaliseCall (POS, EXPR -> STMT).

     'rule' NormaliseCall (Pos, CallExpr:callexpr (_, _, _) -> 
			   stmtcall(Pos, CallExpr)) :
	  
     'rule' NormaliseCall (Pos, Routine -> 
			   stmtcall (Pos, callexpr (Pos, Routine, nil))):

'nonterm' ReturnStmt (-> STMT)

     'rule' ReturnStmt (-> returnvalue (Pos, Value)) :
	  "RETURN" @ (-> Pos)
	  Expr (-> Value)
	  
     'rule' ReturnStmt (-> return (Pos)) :
	  "RETURN" @ (-> Pos)
	  
-------------------------------------------------------------------------------
-- If Statement
-------------------------------------------------------------------------------

'nonterm' IfStmt (-> STMT)

     'rule' IfStmt (-> ifelse (Pos, Cond, ThenPart, ElsePart)) :
	  "IF" @ (-> Pos)
	  Expr (-> Cond)
	  "THEN" 
	  StmtList (-> ThenPart)
	  ElsePart (-> ElsePart)
	  "END" "IF"

'nonterm' ElsePart (-> STMT)
     
     'rule' ElsePart (-> ifelse (Pos, Cond, ThenPart, ElsePart)) :
	  "ELSIF" @ (-> Pos)
	  Expr (-> Cond)
	  "THEN"
	  StmtList (-> ThenPart)
	  ElsePart (-> ElsePart)
	  
     'rule' ElsePart (-> ElsePart) :
	  "ELSE"
	  StmtList (-> ElsePart)
	  
     'rule' ElsePart (-> nil (Pos)) :
	  @ (-> Pos)
	  
-------------------------------------------------------------------------------
-- Select Statement
-------------------------------------------------------------------------------
	  
'nonterm' SelectStmt (-> STMT)

     'rule' SelectStmt (-> select (Pos, Cond, CaseParts, OtherPart)) :
	  "SELECT" @ (-> Pos)
	  Expr (-> Cond)
	  "OF"
	  CasePartList (-> CaseParts)
	  OtherwisePart (-> OtherPart)
	  "END" "SELECT"

'nonterm' CasePartList (-> CASE)
     
     'rule' CasePartList (-> seq (CasePart, CaseParts)) :
	  CasePart (-> CasePart)
	  CasePartList (-> CaseParts)
	  
     'rule' CasePartList (-> nil) :
	  
'nonterm' CasePart (-> CASE)

     'rule' CasePart (-> case (Pos, Labels, Body)) :
	  "CASE" @ (-> Pos)
	  Labellist (-> Labels)
	  ":"
	  StmtList (-> Body)
	  
'nonterm' Labellist (-> EXPRLIST)

     'rule' Labellist (-> exprlist (Label, Labellist)) :
	  Label (-> Label) @ (-> Pos)
	  ","
	  Labellist (-> Labellist)
	  
     'rule' Labellist (-> exprlist (Label, nil)) :
	  Label (-> Label)
	  
'nonterm' Label  (-> EXPR)

     'rule' Label (-> range (Pos, Lwb, Upb)) :
	  Expr (-> Lwb) 
	  ".." @ (-> Pos)
	  Expr (-> Upb)
	  
     'rule' Label (-> Label) :
	  Expr (-> Label)
	  
'nonterm' OtherwisePart (-> STMT)

     'rule' OtherwisePart (-> StmtList) :
	  "OTHERWISE"
	  StmtList (-> StmtList)
	  
     'rule' OtherwisePart (-> nil (Pos)) :
	  @ (-> Pos)
	  
-------------------------------------------------------------------------------
-- TypeSelect Statement
-------------------------------------------------------------------------------

'nonterm' TypeSelectStmt (-> STMT)

     'rule' TypeSelectStmt (-> typeselect (Pos, ObjDecl, Init, CaseParts, 
					   OtherPart)) :
	  "TYPESELECT" @ (-> Pos)
	  TSVarDecl (-> ObjDecl, Init)
	  "OF"
	  TypecasePartList (-> CaseParts)
	  OtherwisePart (-> OtherPart)
	  "END" "TYPESELECT"

'nonterm' TSVarDecl (-> DECL, EXPR)
     
     'rule' TSVarDecl (-> implicitvar (Pos, Id), Init) :
	  IDENTIFIER (-> I) @ (-> Pos)
	  ":=" 
	  Expr (-> Init)
	  NewId (Pos, I, actual -> Id)
		       
'nonterm' TypecasePartList (-> TYPECASE)
     
     'rule' TypecasePartList (-> seq (CasePart, CaseParts)) :
	  TypecasePart (-> CasePart)
	  TypecasePartList (-> CaseParts)
	  
     'rule' TypecasePartList (-> nil) :
	  
'nonterm' TypecasePart (-> TYPECASE)

     'rule' TypecasePart (-> typecase (Pos, Label, Body)) :
	  "CASE" @ (-> Pos)
	  IdAppl (-> Label)
	  ":"
	  StmtList (-> Body)
	  
-------------------------------------------------------------------------------
-- Loop Statements
-------------------------------------------------------------------------------
	  
'nonterm' LoopStmt (-> STMT)

     'rule' LoopStmt (-> loop (Pos, Body)) :
	  Loop (-> Body) @ (-> Pos)
	  
'nonterm' WhileStmt (-> STMT) 
     
     'rule' WhileStmt (-> while (Pos, Cond, Body)) :
	  "WHILE" Expr (-> Cond) @ (-> Pos)
	  Loop (-> Body)
	  
'nonterm' ForStmt (-> STMT)

     'rule' ForStmt (-> for (Pos, implicitvar (IPos, Id),
			     Range, Step, Body)) :
	  "FOR" @ (-> Pos) 
	  IDENTIFIER (-> I) @ (-> IPos)
	  "IN"
	  Range (-> Range)
	  Step (-> Step)
	  Loop (-> Body)
	  NewId (IPos, I, actual -> Id)
	  
'nonterm' Range (-> EXPR)

     'rule' Range (-> range (Pos, Lwb, Upb)):
	  Expr (-> Lwb) @ (-> Pos)
	  ".."
	  Expr (-> Upb)
		   
'nonterm' Step (-> STEP)
     
     'rule' Step (-> incr (Value)) :
	  "INCR" Expr (-> Value)
	  
     'rule' Step (-> decr (Value)) :
	  "DECR" Expr (-> Value)

     'rule' Step (-> incr (posintliteral (Pos, 1))) :
	  @ (-> Pos)
     
'nonterm' Loop (-> BLOCK)

     'rule' Loop (-> block (BeginPos, EndPos, Body))
	  "LOOP" @ (-> BeginPos)
	  StmtList (-> Body)
	  "END" @ (-> EndPos) "LOOP"
	  
-------------------------------------------------------------------------------
-- Try und Raise Statement
-------------------------------------------------------------------------------

'nonterm' TryStmt (-> STMT)

     'rule' TryStmt (-> try (Pos, Body, Handlers, Otherwise)) :
	  "TRY" @ (-> Pos)
	  StmtList (-> Body)
	  "EXCEPT"
	  HandlerList (-> Handlers)
	  OtherwisePart (-> Otherwise)
	  "END" "TRY"

'nonterm' HandlerList (-> HANDLER)
     
     'rule' HandlerList (-> seq (Handler, HandlerList)) :
	  Handler (-> Handler)
	  HandlerList (-> HandlerList)
	  
     'rule' HandlerList (-> nil) :
	  
'nonterm' Handler (-> HANDLER)

     'rule' Handler (-> handler (Pos, Name, FParams, Body)) :
	  "CASE" @ (-> Pos)
	  IdAppl (-> Name)
	  FParamSpec (-> FParams)
	  ":"
	  StmtList (-> Body)
	  
'nonterm' RaiseStmt (-> STMT)

     'rule' RaiseStmt (-> raise (Pos, Exception, Params)) :
	  "RAISE" @ (-> Pos)
	  IdAppl (-> Exception)
	  "("
	  ActParamList (-> Params)
	  ")"
	  
     'rule' RaiseStmt (-> raise (Pos, Exception, nil)) :
	  "RAISE" @ (-> Pos)
	  IdAppl (-> Exception)
	  
-------------------------------------------------------------------------------
-- Expressions
-------------------------------------------------------------------------------

'nonterm' Expr (-> EXPR)
	  
     'rule' Expr (-> Value) :
	  Prio6Expr (-> Value)
	  
-------------------------------------------------------------------------------
-- Prio 6 Expression
-------------------------------------------------------------------------------
	  
'nonterm' Prio6Expr (-> EXPR)

     'rule' Prio6Expr (-> Value) :
	  Prio5Expr (-> Value)
	  
     'rule' Prio6Expr (-> dyop (Pos, OpType, Operator, LeftOp, RightOp)) :
	  Prio6Expr (-> LeftOp)
	  Prio6Operator (-> Operator) @ (-> Pos)
	  Prio5Expr (-> RightOp)
	  UnspecTypeIndex -> OpType
	  
'nonterm' Prio6Operator (-> DOP)

     'rule' Prio6Operator (-> or) :
	  "OR"
	  
-------------------------------------------------------------------------------
-- Prio 5 Expression
-------------------------------------------------------------------------------
	  
'nonterm' Prio5Expr (-> EXPR)

     'rule' Prio5Expr (-> Value) :
	  Prio4Expr (-> Value)
	  
     'rule' Prio5Expr (-> dyop (Pos, OpType, Operator, LeftOp, RightOp)) :
	  Prio5Expr (-> LeftOp)
	  Prio5Operator (-> Operator) @ (-> Pos)
	  Prio4Expr (-> RightOp)
	  UnspecTypeIndex -> OpType

'nonterm' Prio5Operator (-> DOP)

     'rule' Prio5Operator (-> and) :
	  "AND"
	  
-------------------------------------------------------------------------------
-- Prio 4 Expression
-------------------------------------------------------------------------------
	  
'nonterm' Prio4Expr (-> EXPR)

     'rule' Prio4Expr (-> Value) :
	  Prio3Expr (-> Value)
	  
     'rule' Prio4Expr (-> dyop (Pos, OpType, Operator, LeftOp, RightOp)) :
	  Prio4Expr (-> LeftOp)
	  Prio4Operator (-> Operator) @ (-> Pos)
	  Prio3Expr (-> RightOp)
	  UnspecTypeIndex -> OpType

'nonterm' Prio4Operator (-> DOP)

     'rule' Prio4Operator (-> lt) :
	  "<"
	  
     'rule' Prio4Operator (-> le) :
	  "<="
	  
     'rule' Prio4Operator (-> eq) :
	  "="
	  
     'rule' Prio4Operator (-> ne) :
	  "<>"
	  
     'rule' Prio4Operator (-> ge) :
	  ">="
	  
     'rule' Prio4Operator (-> gt) :
	  ">"
	  
-------------------------------------------------------------------------------
-- Prio 3 Expressions
-------------------------------------------------------------------------------
	  
'nonterm' Prio3Expr (-> EXPR)

     'rule' Prio3Expr (-> Value) :
	  Prio2Expr (-> Value)
	  
     'rule' Prio3Expr (-> dyop (Pos, OpType, Operator, LeftOp, RightOp)) :
	  Prio3Expr (-> LeftOp)
	  Prio3Operator (-> Operator) @ (-> Pos)
	  Prio2Expr (-> RightOp)
	  UnspecTypeIndex -> OpType

'nonterm' Prio3Operator (-> DOP)

     'rule' Prio3Operator (-> plus) :
	  "+"
	  
     'rule' Prio3Operator (-> minus) :
	  "-"
	  
-------------------------------------------------------------------------------
-- Prio 2 Expressions
-------------------------------------------------------------------------------
	  
'nonterm' Prio2Expr (-> EXPR)

     'rule' Prio2Expr (-> Value) :
	  Prio1Expr (-> Value)
	  
     'rule' Prio2Expr (-> dyop (Pos, OpType, Operator, LeftOp, RightOp)) :
	  Prio2Expr (-> LeftOp)
	  Prio2Operator (-> Operator) @ (-> Pos)
	  Prio1Expr (-> RightOp)
	  UnspecTypeIndex -> OpType

'nonterm' Prio2Operator (-> DOP)

     'rule' Prio2Operator (-> times) :
	  "*"
	  
     'rule' Prio2Operator (-> div) :
	  "/"
	  
     'rule' Prio2Operator (-> idiv) :
	  "DIV"
	  
     'rule' Prio2Operator (-> mod) :
	  "MOD"
	  
-------------------------------------------------------------------------------
-- Prio 1  Expressions
-------------------------------------------------------------------------------
	  
'nonterm' Prio1Expr (-> EXPR)

     'rule' Prio1Expr (-> Value) :
	  PrimaryExpr (-> Value)
	  
     'rule' Prio1Expr (-> monop (Pos, OpType, Operator, Operand)) :
	  Prio1Operator (-> Operator) @ (-> Pos)
	  PrimaryExpr (-> Operand)
	  UnspecTypeIndex -> OpType

     'rule' Prio1Expr (-> adr (Pos, Designator)) :
	  "ADR" @ (-> Pos)
	  PrimaryExpr (-> Designator)
	  
'nonterm' Prio1Operator (-> MOP)

     'rule' Prio1Operator (-> minus) :
	  "-"
	  
     'rule' Prio1Operator (-> plus) :
	  "+"
	  
     'rule' Prio1Operator (-> not) :
	  "NOT"
	  
-------------------------------------------------------------------------------
-- Primary Expressions
-------------------------------------------------------------------------------
	  
'nonterm' PrimaryExpr (-> EXPR)
	  
     'rule' PrimaryExpr (-> expr_nil (Pos)) :
	  "NIL" @ (-> Pos) 
	  
     'rule' PrimaryExpr (-> Value) :
	  ConstVal (-> Value) @(-> Pos)
	  
     'rule' PrimaryExpr (-> sizeof (Pos, Type)) :
	  "SIZEOF" @ (-> Pos)
	  "("
	  Type (-> Type)
	  ")"
	  
     'rule' PrimaryExpr (-> dot (Pos, Object, CompId)) :
	  PrimaryExpr (-> Object)
	  "." @ (-> Pos)
	  IDENTIFIER (-> I) @ (-> IPos)
	  NewId (IPos, I, actual -> CompId)
	  
     'rule' PrimaryExpr (-> Subscr) :
	  Subscription (-> Subscr)
	  
     'rule' PrimaryExpr (-> deref (Pos, RefType, Reference)) :
	  PrimaryExpr (-> Reference) @ (-> Pos)
	  "^"
	  UnspecTypeIndex -> RefType

     'rule' PrimaryExpr (-> applied (Pos, ApplKey)) :
	  IdAppl (-> ApplKey) @ (-> Pos)
     
     'rule' PrimaryExpr (-> Call) :
	  CallExpr (-> Call)
	  
     'rule' PrimaryExpr (-> Expr) :
	  "("
	  Expr (-> Expr)
	  ")"
	  
     'rule' PrimaryExpr (-> NewObj) :
	  ObjectCreation (-> NewObj)
	  
     'rule' PrimaryExpr (-> current (Pos)) :
	  "CURRENT" @ (-> Pos)
	   
     'rule' PrimaryExpr (-> super (Pos)) :
	  "SUPER" @ (-> Pos)
	  
-------------------------------------------------------------------------------
-- Literals
-------------------------------------------------------------------------------
	  
'nonterm' ConstVal (-> EXPR)
     
     'rule' ConstVal (-> false (Pos)) :
	  "FALSE" @ (-> Pos)
     
     'rule' ConstVal (-> true (Pos)) :
	  "TRUE" @ (-> Pos)
	  
     'rule' ConstVal (-> posintliteral (Pos, N)) :
	  INTCONST (-> N) @ (-> Pos)

     'rule' ConstVal (-> doubleliteral (Pos, D)) :
	  DOUBLECONST (-> D) @ (-> Pos)

     'rule' ConstVal (-> charliteral (Pos, C)) :
	  CHARCONST (-> C) @ (-> Pos)

     'rule' ConstVal (-> stringliteral (Pos, S)) :
	  STRINGCONST (-> S) @ (-> Pos)
	  
-------------------------------------------------------------------------------
-- Subscription
-------------------------------------------------------------------------------

'nonterm' Subscription (-> EXPR)

     'rule' Subscription (-> Subscription) :
	  PrimaryExpr (-> Array)
	  "[" @ (-> Pos)
	  IndexList (-> IndexList)
	  "]"
	  NormalizeSubscripton (Pos, Array, IndexList -> Subscription)
     
     'rule' Subscription (-> substring (Pos, String, range (Pos, Lwb, Upb))) :
	  PrimaryExpr (-> String)
	  "[" @ (-> Pos)
	  Expr (-> Lwb) 
	  ".."
	  Expr (-> Upb)
	  "]"

'nonterm' IndexList (-> EXPRLIST)
     
     'rule' IndexList (-> exprlist (Index, IndexList)) :
	  Expr (-> Index)
	  "," @ (-> Pos)
	  IndexList (-> IndexList)
     
     'rule' IndexList (-> exprlist (Index, nil)) :
	  Expr (-> Index)
	  
'action' NormalizeSubscripton (POS, Array : EXPR, IndexList : EXPRLIST -> 
			       Subscription : EXPR)
     
     'rule' NormalizeSubscripton (Pos, Array, exprlist (Head, Tail) -> 
				  Subscription) :
	  NormalizeSubscripton (Pos, subscr (Pos, Array, Head), 
				Tail -> Subscription) 
	  
     'rule' NormalizeSubscripton (_, Array, nil -> Array) :
	  
-------------------------------------------------------------------------------
-- Function Call 
-------------------------------------------------------------------------------
	  
'nonterm' CallExpr (-> EXPR)

     'rule' CallExpr (-> callexpr (Pos, Procedure, Params)) :
	  PrimaryExpr (-> Procedure) @ (-> Pos)
	  "("
	  ActParamList (-> Params)
	  ")"

'nonterm' ActParamList (-> APARAMLIST)
     
     'rule' ActParamList (-> L) :
	  ActParamList_h (-> L)
	  
     'rule' ActParamList (-> nil) :
	  
'nonterm' ActParamList_h (-> APARAMLIST)

     'rule' ActParamList_h (-> aparamlist (Param, Tail)) :
	  ActParam (-> Param)
	  "," 
	  ActParamList_h (-> Tail)
	  
     'rule' ActParamList_h (-> aparamlist (Param, nil)) :
	  ActParam (-> Param)
	  
'nonterm' ActParam (-> APARAM)
     
     'rule' ActParam (-> unspec (Expr)) :
	  Expr (-> Expr)
	  
-------------------------------------------------------------------------------
-- NEW
-------------------------------------------------------------------------------

'nonterm' ObjectCreation (-> EXPR)
     
     'rule' ObjectCreation (-> new (Pos, Type)) :
	  "NEW" @ (-> Pos)
	  IdAppl (-> Id) @ (-> TPos)
	  NewTypeIndex (TPos, composite (typename (Id)) -> Type)
     
     'rule' ObjectCreation (-> new (Pos, Type)) :
	  "NEW" @ (-> Pos)
	  IdAppl (-> Id) @(-> TPos)
	  "["
	  AGenParamList (-> GenParams)
	  "]"
	  NewTypeIndex (TPos, generic (genericinst (Id,	GenParams)) -> Type)

-------------------------------------------------------------------------------
-- Identifier Liste
-------------------------------------------------------------------------------

'nonterm' IdentifierList (-> IDLIST)
	  
     'rule' IdentifierList (-> idlist (Id, Tail)) :
	  IDENTIFIER (-> I) @ (-> IPos)
	  ","
	  IdentifierList (-> Tail)
	  NewId (IPos, I, actual -> Id)
     
     'rule' IdentifierList (-> idlist (Id, nil)) :
          IDENTIFIER (-> I) @ (-> IPos)
	  NewId (IPos, I, actual -> Id)
	  
-------------------------------------------------------------------------------
-- Identifiers
-------------------------------------------------------------------------------

'nonterm' IdAppl (-> ID)
	  
     'rule' IdAppl (-> ApplKey) :
	  IDENTIFIER (-> M) @ (-> MPos)
	  "::"
	  IDENTIFIER (-> I) @ (-> IPos)
	  NewId (MPos, M, actual -> ModId)
	  NewId (IPos, I, module (ModId) -> ApplKey)
	  
     'rule' IdAppl (-> ApplKey) :
	  IDENTIFIER (-> I) @ (-> Pos)
	  NewId (Pos, I, actual -> ApplKey)
     
-------------------------------------------------------------------------------
-- Structured token
-------------------------------------------------------------------------------

'token' IDENTIFIER (-> Ident)
'token' INTCONST (-> INT)
'token' DOUBLECONST (-> DOUBLE)
'token' CHARCONST (-> CHAR)
'token' STRINGCONST (-> STRING)

'token' EOFILE
'token' NEXTUNIT

-------------------------------------------------------------------------------
-- Postlude features: PrintModules, CheckErrorMessages, ...
-------------------------------------------------------------------------------

'action' SyntaxPostlude (SpecModule : MODULE, ImplModule : MODULE)
     
     'rule' SyntaxPostlude (SpecModule, ImplModule) :
	  [|
	       IsPrintSyntaxMode
	       print (SpecModule)
	       print (ImplModule)
	  |]
	  CheckErrorMessages
	  [|
	       IsPrintDependenciesMode
	       PrintDependencies
	  |]   
	  
-------------------------------------------------------------------------------

'action' ContextPostlude (SpecModule : MODULE, ImplModule : MODULE)
     
     'rule' ContextPostlude (SpecModule, ImplModule) :
	  [|
	       IsPrintContextMode
	       print (SpecModule)
	       print (ImplModule)
	  |]
	  CheckErrorMessages
	  FlushMessages
	  
-------------------------------------------------------------------------------

'end'


