/* HEADER */
/* @(#) bcd_acmp.c 4.1.1.2 */
/* @(#)Description: C-Programme und Include-Dateien */
/* @(#)Author: xsde */
/* @(#)Location: PB01 */
/* @(#)Type: cprog */
/* @(#)Created with X/SDE-REL:  */
/* @(#)Date, time modify: 93/03/19, 11:03:08 */
/* @(#)Date, time return: 92/08/25, 08:13:33 */
/* Copyright: @(#) X/SDE-BCD V3.1A90 19930319 */
/* Copyright: @(#) Siemens Nixdorf Informationssysteme AG, 1993 */
/* END HEADER */
/****************************************************************************/
/**                                                                        **/
/**     BCD-Arithmetic for X/SDE     copyright (C) by SIEMENS/NIXDORF      **/
/**                                                                        **/
/****************************************************************************/
/**                                                                        **/
/**     File name: bcd_acmp.c           (arithmetik and compare)           **/
/**                                                                        **/
/**     Date     : 05.04.1991                                              **/
/**                                                                        **/
/**     Release  : V1.0A00                                                 **/
/**                                                                        **/
/**                                                                        **/
/****************************************************************************/
/****************************************************************************/
/**                                                                        **/
/**     Changes:                                                           **/
/**                                                                        **/
/****************************************************************************/

#include "bcd_def.h"
#include <limits.h>

#include "bcd_rel.h"

int b_status = 0;

	      /* BCD Internal constants, must be initialized by b_init call */

void b_init();
void b_ac_init();


static BCD b_zero,b_one,b_infi,b_l_min, b_l_max,
	   b_l_minw, b_s_min, b_s_max,
	   b_s_minw, b_i_min, b_i_max,
	   b_i_minw, b_n_one ;



/***************************************************************************/
/*                                                                         */
/*      b_add  - adds       two numbers in internal format                 */
/*                                                                         */
/*      input:                                                             */
/*              BCD *px,*py,*pz                                            */
/*              *pz  =   b_add(pz,px,py)                                   */
/*                                                                         */
/*      output:                                                            */
/*              (*pz) = (*px) + (*py)                                      */
/*                                                                         */
/*      returns:                                                           */
/*              B_OK                                                       */
/*              B_OVERFLOW_ERROR                                           */
/*                                                                         */
/***************************************************************************/

BCD *b_add(pc,pa,pb)

register BCD *pc,*pa,*pb;
{
   BCD a,b;

   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }
   b_init();

   /* Get operand a and b */
   a = *pa ;
   b = *pb ;

   /* If operand a is zero, operand b is the result */
   if (ZERO(a))
   {
      a = b ;
   }
   else
   {
      /* Add only of operand b is not zero */
      if (!ZERO(b))
      {
	 /* exponents are the same, can add without shifting */
	 if (a.mant.chart == b.mant.chart)
	 {
	    addmant(&a,&b) ;
	 }
	 else
	 {
	    /* operand a's exponent is greater */
	    if (a.mant.chart > b.mant.chart)
	    {
	       /* if exponents in range of each other, shift b and add */
	       if (a.mant.chart - b.mant.chart < NBCD)
	       {
		  shift(&b,a.mant.chart-b.mant.chart) ;
		  b.mant.chart = a.mant.chart ;
		  addmant(&a,&b) ;
	       }
	    }
	    else
	    {
	       /* if exponents in range of each other, shift a and add */
	       if (b.mant.chart - a.mant.chart < NBCD)
	       {
		  shift(&a,b.mant.chart-a.mant.chart) ;
		  a.mant.chart = b.mant.chart ;
		  addmant(&a,&b) ;
	       }
	       else
	       {
		  a = b ;
	       }
	    }
	 }
      }
   }

   /* normalize and check for overflow, underflow */
   if (!NORMALIZED(a))
   {
      norm(&a) ;
   }

   a.b_byte.DIG9 = 0x00;
   if (a.mant.chart > MAX_CHART)                   /* Overflow */
   {
      (*pc) = b_infi ;
      (*pc).mant.sign = a.mant.sign;
      b_status = B_OVERFLOW_ERROR ;
   }
   else if ((int)(a.mant.chart) < (MIN_CHART))     /* Underflow */
   {
      (*pc) = b_zero ;
   }
   else                                            /* Dataflow */
   {
      (*pc) = a ;
   }

   return(pc);
}

/***************************************************************************/
/*                                                                         */
/*      b_sub  - subtracts  two numbers in internal format                 */
/*                                                                         */
/*      input:                                                             */
/*              BCD *px,*py,*pz                                            */
/*              *pz  =  b_sub(pz,px,py);                                   */
/*                                                                         */
/*      output:                                                            */
/*              (*pz) = (*px) - (*py)                                      */
/*                                                                         */
/*      returns:                                                           */
/*              B_OK                                                       */
/*              B_OVERFLOW_ERROR                                           */
/*                                                                         */
/***************************************************************************/
BCD *b_sub(pc,pa,pb)

register BCD *pc,*pa,*pb;
{
   BCD b;

   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }
   b_init();
   b = *pb;
   b.mant.sign ^= NEGATIV;                      /* exchange sign    */
   pc = b_add(pc, pa, &b);                      /* add the negative */

   return (pc);
}

/***************************************************************************/
/*                                                                         */
/*      b_mul - multiplies two numbers in internal format                  */
/*                                                                         */
/*      input:                                                             */
/*              BCD *px,*py,*pz                                            */
/*              *pz  =  b_sub(pz,px,py);                                   */
/*                                                                         */
/*      output:                                                            */
/*              (*pz) = (*px) * (*py)                                      */
/*                                                                         */
/*      returns:                                                           */
/*              B_OK                                                       */
/*              B_OVERFLOW_ERROR                                           */
/*                                                                         */
/***************************************************************************/
BCD *b_mul(pc,pa,pb)

register BCD *pc,*pa,*pb;
{
   register int i,j ;
   BCD  multiple[10] , a,c;
   USHORT flag,nullzahl;
   UCHAR hilf ;

   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }
   b_init();

   /* Determine sign and characteristic of the result */
   a = *pa ;

   a.mant.chart += (*pb).mant.chart ;
   a.mant.sign = a.mant.sign ^ (*pb).mant.sign;

   /* Determine the first non-null digit in operand B */
   flag = 1;
   nullzahl = 0 ;
   i = NBYTE - 2;

   while (flag && i>=0)
   {
      hilf = (*pb).b_byte.dig[I(i)]; /* I(i) swaps bytes if necessary */

      if (hilf == 0)
      {
	 nullzahl = nullzahl + 2;
	 i = i - 1;
      }
      else
      {
	 flag = 0;
	 if (hilf<<4 == 0)
	 {
	    nullzahl = nullzahl + 1;
	 }
      }
   }

   /* Shift to avoid overflow when adding mantissas */
   shift_right(&a) ;

   /* Fill a table with the multiples of operand a */
   c = a ;

   multiple[1] = a;
   for (i=2;i<=9;i++)
   {
      (void)add(&a,&c);
      multiple[i] = a;
   }

   /* Compute mantissa by adding the multiple into appropriate place */
   ZEROMANT(a);
   for (i=NBCD-3-nullzahl;i>=0;i--)
   {
      shift_right(&a) ;
      j = GETDIG(*pb,i);
      if (j>0)
      {
	(void)add(&a,&multiple[j]);
      }
   }

   if (!NORMALIZED(a))
   {
      norm(&a) ;
   }

   a.b_byte.DIG9 = 0x00;

   /* Check for overflow and underflow */
   if (a.mant.chart > MAX_CHART)
   {
      (*pc) = b_infi ;
      (*pc).mant.sign = a.mant.sign;
      b_status = B_OVERFLOW_ERROR ;
   }
   else if ((int)(a.mant.chart) < (MIN_CHART))
   {
      (*pc) = b_zero ;
   }
   else
   {
      (*pc) = a ;
   }

   return(pc);
}

/***************************************************************************/
/*                                                                         */
/*      b_div divide two internal formatted numbers                        */
/*                                                                         */
/*      input:                                                             */
/*              BCD *px,*py,*pz                                            */
/*              *pz  =  b_div(pz,px,py);                                   */
/*                                                                         */
/*      output:                                                            */
/*              (*pz) = (*px) / (*py)                                      */
/*                                                                         */
/*      returns:                                                           */
/*              B_OK                                                       */
/*              B_OVERFLOW_ERROR                                           */
/*              B_ZERO_DIVIDE_ERROR                                        */
/*                                                                         */
/***************************************************************************/
BCD *b_div(pc,pa,pb)

register BCD *pc,*pa,*pb;
{
   register int i,tdig;
   register UCHAR dig ;
   BCD multiple[10],a,b,c ;
   register USHORT leading ;

   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }
   b_init();

   /* copy operand b into temporary memory */
   a = *pa ;
   b = *pb ;

   /* divide by zero */
   if (ZERO(b))
   {
      (*pc) = b_infi;
      (*pc).mant.sign = a.mant.sign != b.mant.sign ? NEGATIV : POSITIV;
      b_status = B_ZERO_DIVIDE_ERROR ;
      return (pc);
   }
   else
   {
      shift_right(&a) ;
      shift_right(&b) ;

      /* Fill a table with the multiples of operand b */
      multiple[0] = b_zero;
      c = b;

      multiple[1] = b;
      for (i=2;i<=9;i++)
      {
	 (void)add(&c,&b);
	 multiple[i] = c;
      }

      /* Determine sign and chart of result */
      c.mant.chart = a.mant.chart - b.mant.chart + 1;
      c.mant.sign = a.mant.sign ^ b.mant.sign;

      /* Loop subtracting multiples from mantissa */
      leading = GETDIG(b,1)*10+GETDIG(b,2);
      for (i=0;i<NBCD-1;i++)
      {
	 tdig = ((GETDIG(a,0)*10+GETDIG(a,1))*10
		 +GETDIG(a,2)) / leading ;

	 dig = tdig ;
	 if ((int)dig>0)
	 {
	    if (dig>9)
	    {
	       dig = 9;
	    }

	    while (comp_d(&a,&multiple[dig]) < 0)
	    {
	       dig-- ;
	    }

	    sub(&a,&multiple[dig]);
	 }

	 PUTDIG(dig,c,i) ;
	 shift_left(&a);
      }
   }

   if (!NORMALIZED(c))
   {
      norm(&c) ;
   }
   c.b_byte.DIG9 = 0x00;

   /* check for overflow and underflow */
   if (c.mant.chart > MAX_CHART)
   {
      (*pc) = b_infi;
      (*pc).mant.sign = c.mant.sign;
      b_status = B_OVERFLOW_ERROR ;
   }
   else if ((int)(c.mant.chart) < (MIN_CHART))
   {
      (*pc) = b_zero ;
   }
   else
   {
      (*pc) = c ;
   }

   return(pc);
}

/***************************************************************************/
/*                                                                         */
/*      b_idiv perform integer division on two internal formatted numbers  */
/*                                                                         */
/*      input:                                                             */
/*              BCD *px,*py,*pz                                            */
/*              *pz  =  b_idiv(pz,px,py);                                   */
/*                                                                         */
/*      output:                                                            */
/*              (*pz) = (*px) / (*py)                                      */
/*                                                                         */
/*      returns:                                                           */
/*              B_OK                                                       */
/*              B_OVERFLOW_ERROR                                           */
/*              B_ZERO_DIVIDE_ERROR                                        */
/*                                                                         */
/***************************************************************************/
BCD *b_idiv(pc,pa,pb)

register BCD *pc,*pa,*pb;
{
   register int i,tdig;
   register UCHAR dig ;
   BCD multiple[10],a,b,c,f1,f2 ;
   register USHORT leading ;
   short chart_ptr;
   extern BCD *b_abs();

   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }
   b_init();

   if ((int)(*pb).mant.chart <= FRACT_CHART)     /* divide by zero or fraction */
   {
      (*pc) = b_infi;
      (*pc).mant.sign = (*pa).mant.sign != (*pb).mant.sign ? NEGATIV : POSITIV;
      b_status = B_ZERO_DIVIDE_ERROR ;
      return (pc);
   }
   else
   {
      (void) b_abs(&a,pa);              /* nm 5.9.90 */
      (void) b_abs(&b,pb);              /* nm 5.9.90 */
      if (b_cmp(&a,&b) < 0)           /* dividend is less than divisor */
      {
	 (*pc) = b_zero;
	 return (pc);
      }
   }
/* else */
   {
      /* Get integer parts of operand a & b */
      f1 = *pa;
      f2 = *pb;

      dec_brk(&f1,&a);
      dec_brk(&f2,&b);

      shift_right(&a) ;
      shift_right(&b) ;

      /* Fill a table with the multiples of operand b */
      multiple[0] = b_zero;
      c = b;

      multiple[1] = b;
      for (i=2;i<=9;i++)
      {
	 (void)add(&c,&b);
	 multiple[i] = c;
      }

      /* Determine sign and chart of result */
      c = b_zero;
      c.mant.chart = a.mant.chart - b.mant.chart + 1;
      c.mant.sign = a.mant.sign ^ b.mant.sign;

      /* Loop subtracting multiples from mantissa     */
      /* Looping ends when all 18 digits are filled   */
      /* or after the last integer digits is computed */
      i = 0;
      chart_ptr = 0 ;
      leading = GETDIG(b,1)*10+GETDIG(b,2);

      while ((i<NBCD-1) && (c.mant.chart > chart_ptr))
      {
	 tdig = ((GETDIG(a,0)*10+GETDIG(a,1))*10
		 +GETDIG(a,2)) / leading ;

	 dig = tdig ;
	 if ((int)dig>0)
	 {
	    if (dig>9)
	    {
	       dig = 9;
	    }

	    while (comp_d(&a,&multiple[dig]) < 0)
	    {
	       dig-- ;
	    }

	    sub(&a,&multiple[dig]);
	 }

	 PUTDIG(dig,c,i) ;
	 shift_left(&a);
	 chart_ptr++;
	 i++;
      }

      if (!NORMALIZED(c))
      {
	 norm(&c) ;
      }
      c.b_byte.DIG9 = 0x00;

      /* check for overflow and underflow */
      if (c.mant.chart > MAX_CHART)
      {
	 (*pc) = b_infi;
	 (*pc).mant.sign = c.mant.sign;
	 b_status = B_OVERFLOW_ERROR ;
      }
      else if ((int)(c.mant.chart) < (MIN_CHART))
      {
	 (*pc) = b_zero ;
      }
      else
      {
	 (*pc) = c ;
      }

      return(pc);
   }
}

/***************************************************************************/
/*                                                                         */
/*      b_mod  perform the modulus function on two bcd values              */
/*                                                                         */
/*      input:                                                             */
/*              BCD *px,*py,*pz                                            */
/*              *pz  =  b_mod(pz,px,py);                                   */
/*                                                                         */
/*      output:                                                            */
/*              (*pz) = (*px) % (*py)                                      */
/*                                                                         */
/*      returns:                                                           */
/*              B_OK                                                       */
/*              B_ZERO_DIVIDE_ERROR                                        */
/*                                                                         */
/***************************************************************************/
BCD *b_mod(pc,pa,pb)

register BCD *pc,*pa,*pb;
{
   register int i,tdig;
   register UCHAR dig ;
   BCD multiple[10],a,b,c,f1,f2;
   register USHORT leading ;
   short chart_ptr;
   extern BCD *b_abs();

   if (b_status < 0)                               /* Error, can't continue */
   {
      return 0;
   }
   b_init();

   if ((int)(*pb).mant.chart <= FRACT_CHART)  /* divide by zero or fraction */
   {
      (*pc) = b_infi;
      (*pc).mant.sign = (*pa).mant.sign != (*pb).mant.sign ? NEGATIV : POSITIV;
      b_status = B_ZERO_DIVIDE_ERROR ;
      return (pc);
   }
   else
   {
      (void) b_abs(&a,pa);                       /* nm 5.9.90 */
      (void) b_abs(&b,pb);                       /* nm 5.9.90 */
      if (b_cmp(&a,&b) < 0)               /* dividend is less than divisor */
      {
	 f1 = *pa;
	 dec_brk(&f1,&a);
	 a.mant.sign = (*pa).mant.sign;
	 (*pc) = a;
	 return (pc);
      }
   }
				      /* Get integer parts of operand a & b */
   f1 = *pa;
   f2 = *pb;

   dec_brk(&f1,&a);
   dec_brk(&f2,&b);

   shift_right(&a) ;
   shift_right(&b) ;

			    /* Fill a table with the multiples of operand b */
   multiple[0] = b_zero;
   c = b;

   multiple[1] = b;
   for (i=2;i<=9;i++)
   {
      (void)add(&c,&b);
      multiple[i] = c;
   }
			    /* Loop subtracting multiples from mantissa     */
			    /* Looping ends when all 18 digits are filled   */
			    /* or after the last integer digits is computed */
   c = b_zero;
   c.mant.chart = a.mant.chart - b.mant.chart + 1;
   i = 0;
   chart_ptr = 0 ;
   leading = GETDIG(b,1)*10+GETDIG(b,2);

   while ((i<NBCD-1) && (c.mant.chart > chart_ptr))
   {
      tdig = ((GETDIG(a,0)*10+GETDIG(a,1))*10
	      +GETDIG(a,2)) / leading ;

      dig = tdig ;
      if ((int)dig>0)
      {
	 if (dig>9)
	 {
	    dig = 9;
	 }

	 while (comp_d(&a,&multiple[dig]) < 0)
	 {
	    dig-- ;
	 }

	 sub(&a,&multiple[dig]);
      }

      PUTDIG(dig,c,i) ;
      shift_left(&a);
      chart_ptr++;
      i++;
   }

   a.mant.chart = b.mant.chart;
   c = a;

   if (!NORMALIZED(c))
   {
      norm(&c) ;
   }
   c.b_byte.DIG9 = 0x00;
   (*pc) = c ;

   return(pc);
}

/***************************************************************************/
/*                                                                         */
/*      b_inc   increment integer BCD value                                */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_op1 ;                                               */
/*              b_dec(b_op1) ;                                             */
/*                                                                         */
/*      returns:                                                           */
/*              B_OK                                                       */
/*                                                                         */
/***************************************************************************/
BCD *b_inc(b_op)

BCD *b_op;
{
   BCD b;
   int mant, pos, endpos;
   USHORT digit;

   if (b_status < 0)                              /* Error, can't continue */
   {
      return 0;
   }
   b_init();

   b = *b_op;
   if (ZERO(b))
   {
      (*b_op) = b_one;
      return (b_op);
   }
   mant = (*b_op).mant.chart;

   if (mant > B_DIGITS)
   {
      b_status = B_LEADING_DIGITS_ERROR;
      return(b_op);
   }

   if ((endpos = mant % 4))
   {
      endpos = mant + 4 - endpos;
      for (pos = mant; pos < endpos; pos++)
      {
	 if (GETDIG(b, pos))
	 {
	    b_status = B_FIRST_OPERAND_ERROR;
	    return(b_op);
	 }
      }
   }
   else
   {
      endpos = mant;
   }
   for (pos = endpos / 4; pos <= B_DIGITS / 4; pos++)
   {
      if ( b.mant.word[pos])
      {
	 b_status = B_FIRST_OPERAND_ERROR;
	 return(b_op);
      }
   }

   pos = mant-1;
   digit = GETDIG(b, pos);
   if ( !b.mant.sign )                          /* sign positive ????      */
   {
      if ( digit < 9)
      {
	 digit++;
	 PUTDIG(digit, b, pos);
      }
      else
      {
	  (void)b_add(&b, b_op, &b_one);
      }
   }
   else
   {
      if ( digit > 0 )
      {
	 digit--;
	 PUTDIG(digit, b, pos);
	 if ( (!digit) && (!pos) )
	 {
	    b.mant.sign = POSITIV;
	    b.mant.chart = 0;
	 }
      }
      else
      {
	  (void)b_add(&b, b_op, &b_one);
      }
   }
   (*b_op) = b;
   return (b_op);
}


/***************************************************************************/
/*                                                                         */
/*      b_dec   decrement integer BCD value                                */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_op1 ;                                               */
/*              b_dec(b_op1) ;                                             */
/*                                                                         */
/*      returns:                                                           */
/*              B_OK                                                       */
/*                                                                         */
/***************************************************************************/
BCD *b_dec(b_op)

BCD *b_op;
{
   BCD b;
   int mant, pos, endpos;
   USHORT digit;

   if (b_status < 0)                              /* Error, can't continue */
   {
      return 0;
   }
   b_init();

   b = *b_op;
   if (ZERO(b))
   {
      (*b_op) = b_n_one;
      return (b_op);
   }
   mant = (*b_op).mant.chart;

   if (mant > B_DIGITS)
   {
      b_status = B_LEADING_DIGITS_ERROR;
      return(b_op);
   }

   if ((endpos = mant % 4))
   {
      endpos = mant + 4 - endpos;
      for (pos = mant; pos < endpos; pos++)
      {
	 if (GETDIG(b, pos))
	 {
	    b_status = B_FIRST_OPERAND_ERROR;
	    return(b_op);
	 }
      }
   }
   else
   {
      endpos = mant;
   }
   for (pos = endpos / 4; pos <= B_DIGITS / 4; pos++)
   {
      if ( b.mant.word[pos])
      {
	 b_status = B_FIRST_OPERAND_ERROR;
	 return(b_op);
      }
   }

   pos = mant-1;
   digit = GETDIG(b, pos);
   if ( !b.mant.sign )                          /* sign positive ????      */
   {
      if ( digit > 0 )
      {
	 digit--;
	 PUTDIG(digit, b, pos);
	 if ( (!digit) && (!pos) )
	 {
	    b.mant.sign = POSITIV;
	    b.mant.chart = 0;
	 }
      }
      else
      {
	  (void)b_add(&b, b_op, &b_n_one);
      }
   }
   else
   {
      if ( digit < 9)
      {
	 digit++;
	 PUTDIG(digit, b, pos);
      }
      else
      {
	  (void)b_add(&b, b_op, &b_n_one);
      }
   }
   (*b_op) = b;
   return (b_op);
}



/****************************************************************************/
/**                                                                        **/
/**     BCD-Arithmetic for X/SDE     copyright (C) by SIEMENS/NIXDORF      **/
/**                                                                        **/
/****************************************************************************/
/**                                                                        **/
/**     C O N V E R S I O N                                                **/
/**                                                                        **/
/**                                                                        **/
/****************************************************************************/



/***************************************************************************/
/*                                                                         */
/*      b_lt  compare two BCD values: return less                          */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_op1, *b_op2 ;                                       */
/*              b_lt(b_op1,b_op2) ;                                        */
/*                                                                         */
/*      returns:                                                           */
/*              B_TRUE:  *b_op1 < *b_op2                                   */
/*              B_FALSE: *b_op1 >= *b_op2                                  */
/*                                                                         */
/***************************************************************************/
int b_lt(b_op1,b_op2)

BCD *b_op1, *b_op2;
{
   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }

   if (b_cmp(b_op1,b_op2) < 0)
   {
      return (B_TRUE);
   }
   else
   {
      return (B_FALSE);
   }
}

/***************************************************************************/
/*                                                                         */
/*      b_le  compare two BCD values: return less than or equal            */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_op1, *b_op2 ;                                       */
/*              b_le(b_op1,b_op2) ;                                        */
/*                                                                         */
/*      returns:                                                           */
/*              B_TRUE:  *b_op1 <= *b_op2                                  */
/*              B_FALSE: *b_op1 > *b_op2                                   */
/*                                                                         */
/***************************************************************************/
int b_le(b_op1,b_op2)

BCD *b_op1, *b_op2;
{
   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }

   if (b_cmp(b_op1,b_op2) <= 0)
   {
      return (B_TRUE);
   }
   else
   {
      return (B_FALSE);
   }
}

/***************************************************************************/
/*                                                                         */
/*      b_eq  compare two BCD values: return equal                         */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_op1, *b_op2 ;                                       */
/*              b_eq(b_op1,b_op2) ;                                        */
/*                                                                         */
/*      returns:                                                           */
/*              B_TRUE:  *b_op1 = *b_op2                                   */
/*              B_FALSE: *b_op1 <> *b_op2                                  */
/*                                                                         */
/***************************************************************************/
int b_eq(b_op1,b_op2)

BCD *b_op1, *b_op2;
{
   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }

   if (b_cmp(b_op1,b_op2) == 0)
   {
      return (B_TRUE);
   }
   else
   {
      return (B_FALSE);
   }
}

/***************************************************************************/
/*                                                                         */
/*      b_ne  compare two BCD values: return not equal                     */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_op1, *b_op2 ;                                       */
/*              b_ne(b_op1,b_op2) ;                                        */
/*                                                                         */
/*      returns:                                                           */
/*              B_TRUE:  *b_op1 != *b_op2                                  */
/*              B_FALSE: *b_op1 == *b_op2                                  */
/*                                                                         */
/***************************************************************************/
int b_ne(b_op1,b_op2)

BCD *b_op1, *b_op2;
{
   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }

   if (b_cmp(b_op1,b_op2) != 0)
   {
      return (B_TRUE);
   }
   else
   {
      return (B_FALSE);
   }
}

/***************************************************************************/
/*                                                                         */
/*      b_gt  compare two BCD values: return greater                       */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_op1, *b_op2 ;                                       */
/*              b_gt(b_op1,b_op2) ;                                        */
/*                                                                         */
/*      returns:                                                           */
/*              B_TRUE:  *b_op1 > *b_op2                                   */
/*              B_FALSE: *b_op1 <= *b_op2                                  */
/*                                                                         */
/***************************************************************************/
int b_gt(b_op1,b_op2)

BCD *b_op1, *b_op2;
{
   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }

   if (b_cmp(b_op1,b_op2) > 0)
   {
      return (B_TRUE);
   }
   else
   {
      return (B_FALSE);
   }
}

/***************************************************************************/
/*                                                                         */
/*      b_ge compare two BCD values: return greater or equal               */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_op1, *b_op2 ;                                       */
/*              b_ge(b_op1,b_op2) ;                                        */
/*                                                                         */
/*      returns:                                                           */
/*              B_TRUE:  *b_op1 >= *b_op2                                  */
/*              B_FALSE: *b_op1 < *b_op2                                   */
/*                                                                         */
/***************************************************************************/
int b_ge(b_op1,b_op2)

BCD *b_op1, *b_op2;
{
   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }

   if (b_cmp(b_op1,b_op2) >= 0)
   {
      return (B_TRUE);
   }
   else
   {
      return (B_FALSE);
   }
}

/***************************************************************************/
/*                                                                         */
/*      b_cmp   compare two BCD values: return releation                   */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_op1, *b_op2 ;                                       */
/*              b_cmp(b_op1,b_op2) ;                                       */
/*                                                                         */
/*      returns:                                                           */
/*             -1: *b_op1 < *b_op2      ( < 0 )                            */
/*              0: *b_op1 = *b_op2      ( = 0 )                            */
/*              1: *b_op1 > *b_op2      ( > 0 )                            */
/*                                                                         */
/***************************************************************************/
int b_cmp(px, py)

BCD *px, *py ;
{
   register int h1, h2;

   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }
   b_init();

   /* word0 plus: h1 = 1,  word0 zero: h1 = 0,  word0 minus: h1 = -1 */
   if (h1 = (*px).mant.word[0])
   {
      h1 = ((*px).mant.sign == POSITIV) ? 1 : -1 ;
   }

   /* word0 plus: h2 = 1,  word0 zero: h2 = 0,  word0 minus: h2 = -1 */
   if (h2 = (*py).mant.word[0])
   {
      h2 = ((*py).mant.sign == POSITIV) ? 1 : -1 ;
   }

   if (h1 != h2)
   {
      return ((h1 > h2) ? (1) : (-1)); /* Signs unequal */
   }
   else if ((*px).mant.chart != (*py).mant.chart)
   {
      return ((((*px).mant.chart * h1 )) > ((*py).mant.chart * h2)
	     ? (1) : (-1)); /* Characteristics unequal */
   }
   else
   {
      return (h1 * comp_d(px,py)); /* Byte by byte compare necessary */
   }
}

/***************************************************************************/
/*                                                                         */
/*      b_sgn   Return sign of BCD value                                   */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_op;                                                 */
/*              sgn  =  b_sgn(b_op);                                       */
/*                                                                         */
/*      output:                                                            */
/*              sgn  =  -1: b_op < 0                                       */
/*                       0: b_op = 0                                       */
/*                      +1: b_op > 0                                       */
/*                                                                         */
/***************************************************************************/
int b_sgn(b_op)

BCD *b_op;
{
   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }
   b_init();

   if (GETSIGN(*b_op) == NEGATIV)
   {
      return (-1); /* Sign is Negative */
   }
   else if (b_cmp(b_op,&b_zero) == 0)
   {
      return (0);  /* Operand is Zero */
   }
   else
   {
      return (1);  /* Sign is Positive */
   }
}

/***************************************************************************/
/*                                                                         */
/*      b_isint Is BCD operand an integer value?                           */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_op;                                                 */
/*              bool =  b_isint(b_op);                                     */
/*                                                                         */
/*      output:                                                            */
/*              bool =  B_TRUE:  operand is an integer value               */
/*                      B_FALSE: operand is a non-integer value            */
/*                                                                         */
/***************************************************************************/
int b_isint(b_op)

BCD *b_op;
{
   BCD b_f,b_i;

   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }
   b_init();

   /* Break into integer and fraction part */
   b_f = *b_op;

   dec_brk(&b_f,&b_i);

   /* Look at fraction part */
   if (b_cmp(&b_f,&b_zero) == 0)
   {
      return B_TRUE;  /* operand is integer */
   }
   else
   {
      return B_FALSE; /* operand is non-integer */
   }
}

/***************************************************************************/
/*                                                                         */
/*      b_isfract  Is BCD operand a fractional value?                      */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_op;                                                 */
/*              bool =  b_isfract(b_op);                                   */
/*                                                                         */
/*      output:                                                            */
/*              bool =  B_TRUE:  operand is a fractional value             */
/*                      B_FALSE: operand is a non-fractional value         */
/*                                                                         */
/***************************************************************************/
int b_isfract(b_op)

BCD *b_op;
{
   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }
   b_init();

   if (((int)(*b_op).mant.chart <= FRACT_CHART)  &&
      (b_cmp(b_op,&b_zero) != 0))
   {
      return B_TRUE;  /* fraction */
   }
   else
   {
      return B_FALSE; /* non-fraction or zero */
   }
}

/***************************************************************************/
/*                                                                         */
/*      b_isnatural  Is BCD operand a natural value?                       */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_op;                                                 */
/*              bool =  b_isnatural(b_op);                                 */
/*                                                                         */
/*      output:                                                            */
/*              bool =  B_TRUE:  operand is a natural value                */
/*                      B_FALSE: operand is a non-natural value            */
/*                                                                         */
/***************************************************************************/
int b_isnatural(b_op)

BCD *b_op;
{
   BCD b_f,b_i;

   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }
   b_init();

   if ((GETSIGN(*b_op) == NEGATIV) ||
      (((int)(*b_op).mant.chart <= FRACT_CHART) &&
      (b_cmp(b_op,&b_zero) != 0)) )
   {
      /* Not natural: operand is negative or a fraction */
      return (B_FALSE);
   }
   else
   {
      /* Break into integer and fraction part */
      b_f = *b_op;

      dec_brk(&b_f,&b_i);

      if (b_cmp(&b_f,&b_zero) != 0)
      {
	 return (B_FALSE);   /* not natural: operand has a fraction part */
      }
      else
      {
	 return (B_TRUE);    /* yawn, it's a positive integer */
      }
   }
}

/***************************************************************************/
/*                                                                         */
/*      b_isodd Is BCD operand odd?                                        */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_op;                                                 */
/*              bool =  b_isodd(b_op);                                     */
/*                                                                         */
/*      output:                                                            */
/*              bool =  B_TRUE:  operand is odd                            */
/*                      B_FALSE: operand is not odd                        */
/*                                                                         */
/***************************************************************************/
int b_isodd(b_op)

BCD *b_op;
{
   BCD b_f,b_i;

   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }
   b_init();

   /* Break into integer and fraction part */
   b_f = *b_op;

   dec_brk(&b_f,&b_i);

   /* Rule out all the cases in which the operand is not odd */

   if (b_cmp(&b_f,&b_zero) != 0)
   {
      return (B_FALSE);   /* not odd: operand has a fraction part */
   }
   else if (b_cmp(&b_i,&b_zero) == 0)
   {
      return (B_FALSE);   /* not odd: operand is zero */
   }
   else if (b_i.mant.chart > NBCD-2)
   {
      return (B_FALSE);   /* not odd: one's place is implied zero */
   }
   else if ((GETDIG(b_i,(b_i.mant.chart-1)) & 1) == 0)
   {
      return (B_FALSE);   /* not odd: low order digit is even */
   }
   else
   {
      return (B_TRUE);    /* odd!! */
   }
}

/***************************************************************************/
/*                                                                         */
/*      b_iszero  Is BCD operand zero?                                     */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_op;                                                 */
/*              bool =  b_iszero(b_op);                                    */
/*                                                                         */
/*      output:                                                            */
/*              bool =  B_TRUE:  operand is zero                           */
/*                      B_FALSE: operand is not zero                       */
/*                                                                         */
/***************************************************************************/
int b_iszero(b_op)

BCD *b_op;
{
   BCD b;

   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }
   b_init();
   b = *b_op;
   if (ZERO(b) )
   {
      return (B_TRUE);                          /* operand is zero     */
   }
   else
   {
      return (B_FALSE);                         /* operand is not zero */
   }
}
/***************************************************************************/
/*                                                                         */
/*      b_iseven  Is BCD operand even?                                     */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_op;                                                 */
/*              bool =  b_iseven(b_op);                                    */
/*                                                                         */
/*      output:                                                            */
/*              bool =  B_TRUE:  operand is even                           */
/*                      B_FALSE: operand is not even                       */
/*                                                                         */
/***************************************************************************/
int b_iseven(b_op)

BCD *b_op;
{
   BCD b_f,b_i;

   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }
   b_init();

   /* Break into integer and fraction part */
   b_f = *b_op;

   dec_brk(&b_f,&b_i);

   /* determine Geradesein of the operand */

   if (b_cmp(&b_f,&b_zero) != 0)
   {
      return (B_FALSE);   /* not even: operand has a fraction part */
   }
   else if (b_cmp(&b_i,&b_zero) == 0)
   {
      return (B_TRUE);    /* even: operand is zero */
   }
   else if (b_i.mant.chart > NBCD-2)
   {
      return (B_TRUE);    /* even: because one's place is implied zero */
   }
   else if ((GETDIG(b_i,(b_i.mant.chart-1)) & 1) == 0)
   {
      return (B_TRUE);    /* even: because low order digit is even */
   }
   else
   {
      return (B_FALSE);   /* not even: because it's probably odd */
   }
}
/****************************************************************************/
/**                                                                        **/
/**     BCD-Arithmetic for X/SDE     copyright (C) by SIEMENS/NIXDORF      **/
/**                                                                        **/
/****************************************************************************/
/**                                                                        **/
/**     M I S C E L L A N E O U S E                                        **/
/**                                                                        **/
/****************************************************************************/



/***************************************************************************/
/*                                                                         */
/*      b_max   return the greater of two BCD values                       */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_rs, *b_op1, *b_op2 ;                                */
/*              b_max(b_rs,b_op1,b_op2) ;                                  */
/*                                                                         */
/*      returns:                                                           */
/*              B_OK                                                       */
/*                                                                         */
/***************************************************************************/
BCD *b_max(b_rs,b_op1,b_op2)

BCD *b_rs,*b_op1,*b_op2;
{
   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }

   if (b_cmp(b_op1,b_op2) > 0)
   {
      /* operand 1 is the maximum */
      *b_rs = *b_op1;
   }
   else
   {
      /* operand 2 is the maximum */
      *b_rs = *b_op2;
   }
   return (b_rs);
}

/***************************************************************************/
/*                                                                         */
/*      b_min   return the lesser of two BCD values                        */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_rs, *b_op1, *b_op2 ;                                */
/*              b_min(b_rs,b_op1,b_op2) ;                                  */
/*                                                                         */
/*      returns:                                                           */
/*              B_OK                                                       */
/*                                                                         */
/***************************************************************************/
BCD *b_min(b_rs,b_op1,b_op2)

BCD *b_rs,*b_op1,*b_op2;
{
   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }

   if (b_cmp(b_op1,b_op2) < 0)
   {
      /* operand 1 is the minimum */
      *b_rs = *b_op1;
   }
   else
   {
      /* operand 2 is the minimum */
      *b_rs = *b_op2;
   }
   return (b_rs);
}



/***************************************************************************/
/*                                                                         */
/*      b_round       Round BCD value to inputted position                 */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_rs, *b_op1;                                         */
/*              int i_op2;                                                 */
/*              b_rs = b_round(b_rs,b_op1,i_op2);                          */
/*                                                                         */
/*      output:                                                            */
/*              b_op1 rounded to position given by i_op2                   */
/*                                                                         */
/***************************************************************************/
BCD *b_round(b_rs,b_op1,i_op2)

BCD *b_rs,*b_op1;
int i_op2;
{
   BCD b_f,b_i,b_z;

   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }
   b_init();

   if (i_op2 > (int)(*b_op1).mant.chart)           /* position too big */
   {
      *b_rs = b_zero ;
   }
   else if ((int)(*b_op1).mant.chart - i_op2 > NBCD-3)  /* too small */
   {
      *b_rs = *b_op1 ;
   }
   else                                               /* just right */
   {
      /* Set up input operand for the dec_brk routine. Adjust the */
      /* exponent so that the operand has the decimal point at    */
      /* the position to adjust.                                  */
      b_f = *b_op1 ;

      b_f.mant.chart = b_f.mant.chart - i_op2;
      dec_brk(&b_f,&b_i);

      if ((b_f.mant.chart == 0) && (GETDIG(b_f,0) >= 5))
      {
	 /* Magnitude is to be incremented. */
	 b_z = b_one;

	 if (GETSIGN(b_f) == NEGATIV)   /* Adjust for negative rounding */
	 {
	    SETSIGN(b_z,NEGATIV);
	 }

	 b_z.mant.chart = i_op2 + 1;
	 b_i.mant.chart = (*b_op1).mant.chart ;
	 b_rs = b_add(b_rs,&b_z,&b_i);
      }
      else
      {
	 /* floor of a positive value or negative with no fractional    */
	 /* part, result is integer part                                */
	 if (ZERO(b_i))
	 {
	    (*b_rs) = b_zero;
	 }
	 else
	 {
	    (*b_rs) = b_i ;
	    (*b_rs).mant.chart = (*b_rs).mant.chart + i_op2 ;
	 }
      }
   }
   return (b_rs);
}

/***************************************************************************/
/*                                                                         */
/*      b_trunc       Truncate BCD value to inputted position              */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_rs, *b_op1;                                         */
/*              int i_op2;                                                 */
/*              b_rs = b_trunc(b_rs,b_op1,i_op2);                          */
/*                                                                         */
/*      output:                                                            */
/*              b_op1 truncated to position given by i_op2                 */
/*                                                                         */
/***************************************************************************/
BCD *b_trunc(b_rs,b_op1,i_op2)

BCD *b_rs,*b_op1;
int i_op2;
{
   BCD b_f,b_i;

   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }
   b_init();

   if (i_op2 > (int)(*b_op1).mant.chart)          /* position too big */
   {
      *b_rs = b_zero ;
   }
   else if ((int)(*b_op1).mant.chart - i_op2 > NBCD-3)  /* too small */
   {
      *b_rs = *b_op1 ;
   }
   else                                               /* just right */
   {
      /* Set up input operand for the dec_brk routine. Adjust the */
      /* exponent so that the operand has the decimal point at    */
      /* the position to truncate                                 */
      b_f = *b_op1 ;

      b_f.mant.chart = b_f.mant.chart - i_op2;
      dec_brk(&b_f,&b_i);

      /* return the "integer" part, adjusting back the exponent      */
      (*b_rs) = b_i ;
      (*b_rs).mant.chart = (*b_rs).mant.chart + i_op2 ;
   }
   return (b_rs);
}

/***************************************************************************/
/*                                                                         */
/*      b_floor       Return next lowest BCD value from position given     */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_rs, *b_op1;                                         */
/*              int i_op2;                                                 */
/*              b_rs = b_floor(b_rs,b_op1,i_op2);                          */
/*                                                                         */
/*      output:                                                            */
/*              first bcd value less than the input, at inputted position  */
/*                                                                         */
/***************************************************************************/
BCD *b_floor(b_rs,b_op1,i_op2)

BCD *b_rs,*b_op1;
int i_op2;
{
   BCD b_f,b_i,b_z;

   if (b_status < 0)                              /* Error, can't continue */
   {
      return 0;
   }
   b_init();

/* i_op2 = i_op2 + 1; */
   if (i_op2 > (int)(*b_op1).mant.chart)          /* position too big */
   {
      *b_rs = b_zero ;
   }
   else if ((int)(*b_op1).mant.chart - i_op2 > NBCD-3)  /* too small */
   {
      *b_rs = *b_op1 ;
   }
   else                                               /* just right */
   {
      /* Set up input operand for the dec_brk routine. Adjust the */
      /* exponent so that the operand has the decimal point at    */
      /* the position to adjust.                                  */
      b_f = *b_op1 ;

      b_f.mant.chart = b_f.mant.chart - i_op2;
      dec_brk(&b_f,&b_i);

      if ((b_f.mant.word[0] != 0) && (GETSIGN(*b_op1) == NEGATIV))
      {
	 /* floor of a negative value: Magnitude of the mantissa is     */
	 /* incremented by 1.                                           */
	 b_z = b_one;
	 SETSIGN(b_z,NEGATIV);
	 b_z.mant.chart = i_op2 + 1;
	 b_i.mant.chart = (*b_op1).mant.chart ;
	 b_rs = b_add(b_rs,&b_z,&b_i);
      }
      else
      {
	 /* floor of a positive value or negative with no fractional    */
	 /* part, result is integer part                                */
	 if (b_i.mant.word[0] == 0) /* No integer part */
	 {
	    (*b_rs) = b_zero;
	 }
	 else
	 {
	    (*b_rs) = b_i ;
	    (*b_rs).mant.chart = (*b_rs).mant.chart + i_op2 ;
	 }
      }
   }
   return (b_rs);
}

/***************************************************************************/
/*                                                                         */
/*      b_ceil  Return next highest BCD value from position given          */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_rs, *b_op1;                                         */
/*              int i_op2;                                                 */
/*              b_rs = b_ceil(b_rs,b_op1,i_op2);                           */
/*                                                                         */
/*      output:                                                            */
/*              first bcd value > than the input, at inputted position     */
/*                                                                         */
/***************************************************************************/
BCD *b_ceil(b_rs,b_op1,i_op2)

BCD *b_rs,*b_op1;
int i_op2;
{
   BCD b_f,b_i,b_z;

   if (b_status < 0)                              /* Error, can't continue */
   {
      return 0;
   }
   b_init();

/* i_op2 = i_op2 + 1; */
   if (i_op2 > (int)(*b_op1).mant.chart)          /* position too big */
   {
      if (GETSIGN(*b_op1) == POSITIV)   /* Positives go to 1 followed by */
      {                                         /* some zeros */
	 *b_rs = b_zero ;
	 (*b_rs).mant.word[0] = 0x1000;
	 (*b_rs).mant.chart = i_op2 + 1;
	 SETSIGN(*b_rs,POSITIV);
      }
      else
      {
	 *b_rs = b_zero; /* Negatives go to zero */
      }
   }
   else if ((int)(*b_op1).mant.chart - i_op2 > NBCD-3)  /* too small */
   {
      *b_rs = *b_op1 ;
   }
   else                                               /* just right */
   {
      /* Set up input operand for the dec_brk routine. Adjust the */
      /* exponent so that the operand has the decimal point at    */
      /* the position to adjust.                                  */
      b_f = *b_op1 ;

      b_f.mant.chart = b_f.mant.chart - i_op2 ;
      dec_brk(&b_f,&b_i);

      if ((b_f.mant.word[0] != 0) && (GETSIGN(*b_op1) == POSITIV))
      {
	 /* ceiling of a positive value: Magnitude of the mantissa is   */
	 /* incremented by 1.                                           */
	 b_z = b_one;                              /* case of -, no int */
	 SETSIGN(b_z,POSITIV);
	 b_z.mant.chart = i_op2 + 1;
	 b_i.mant.chart = (*b_op1).mant.chart ;
	 b_rs = b_add(b_rs,&b_z,&b_i);
      }
      else
      {
	 /* cieling of a negative value or positive with no fractional  */
	 /* part, result is integer part                                */

	 if (b_i.mant.word[0] == 0) /* No integer part */
	 {
	    (*b_rs) = b_zero;
	 }
	 else
	 {
	    (*b_rs) = b_i ;
	    (*b_rs).mant.chart = (*b_rs).mant.chart + i_op2 ;
	 }
      }
   }
   return (b_rs);
}


/***************************************************************************/
/*                                                                         */
/*      b_int   Return integer part of operand                             */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_rs,*b_op;                                           */
/*              b_rs = *b_int(b_rs,b_op);                                  */
/*                                                                         */
/*      output:                                                            */
/*              b_rs =  integer part of b_op                               */
/*                                                                         */
/***************************************************************************/
BCD *b_int(b_rs,b_op)

BCD *b_rs,*b_op;
{
   BCD b_f;

   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }
   b_init();

   /* Break into integer and fraction part */
   b_f = *b_op;

   dec_brk(&b_f,b_rs);

   return b_rs; /* Return integer part */
}

/***************************************************************************/
/*                                                                         */
/*      b_fract Return fraction part of operand                            */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_rs,*b_op;                                           */
/*              b_rs = *b_fract(b_rs,b_op);                                */
/*                                                                         */
/*      output:                                                            */
/*              b_rs =  fraction part of b_op                              */
/*                                                                         */
/***************************************************************************/
BCD *b_fract(b_rs,b_op)

BCD *b_rs,*b_op;
{
   BCD b_i;

   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }
   b_init();

   /* Break into integer and fraction part */
   *b_rs = *b_op;
   dec_brk(b_rs,&b_i);

   return b_rs; /* Return fraction part */
}

/***************************************************************************/
/*                                                                         */
/*      b_abs   Absolut value                                              */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_rs, *b_op                                           */
/*              *pos = *b_abs(b_rs,b_op);                                  */
/*                                                                         */
/*      output:                                                            */
/*              *b_rs = absolute(*b_op);                                   */
/*                                                                         */
/***************************************************************************/
BCD *b_abs(b_rs,b_op)

BCD *b_rs,*b_op;
{
   if (b_status < 0) /* Error, can't continue */
   {
      return (BCD *)0;
   }
   b_init();

   *b_rs = *b_op ;
   SETSIGN(*b_rs,POSITIV);
   return (b_rs);
}

/***************************************************************************/
/*                                                                         */
/*      b_stpos Absolut value                                              */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_rs, *b_op                                           */
/*              *pos = *b_stpos(b_rs,b_op);                                */
/*                                                                         */
/*      output:                                                            */
/*              *b_rs = positive(*b_op);                                   */
/*                                                                         */
/***************************************************************************/
BCD *b_stpos(b_rs,b_op)

BCD *b_rs,*b_op;
{
   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }
   b_init();

   *b_rs = *b_op ;
   SETSIGN(*b_rs,POSITIV);
   return (b_rs);
}

/***************************************************************************/
/*                                                                         */
/*      b_stneg Set sign to negative                                       */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_rs, *b_op                                           */
/*              *neg = *b_stneg(b_rs,b_op);                                */
/*                                                                         */
/*      output:                                                            */
/*              *b_rs = negative(*b_op);                                   */
/*                                                                         */
/***************************************************************************/
BCD *b_stneg(b_rs,b_op)

BCD *b_rs,*b_op;
{
   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }

   *b_rs = *b_op ;
   if (b_cmp(b_rs, &b_zero) == 0)                       /* ba2610  */
   {
      return (b_rs);
   }
   SETSIGN(*b_rs,NEGATIV);
   return (b_rs);
}

/***************************************************************************/
/*                                                                         */
/*      b_inv   Invert sign                                                */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_rs, *b_op                                           */
/*              *inv = *b_inv(b_rs,b_op);                                  */
/*                                                                         */
/*      output:                                                            */
/*              *b_rs = -(*b_op);                                          */
/*                                                                         */
/***************************************************************************/
BCD *b_inv(b_rs,b_op)

BCD *b_rs,*b_op;
{
   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }

   *b_rs = *b_op ;

   if (b_cmp(b_rs, &b_zero) == 0)                       /* ba2610  */
   {
      return (b_rs);
   }

   (*b_rs).mant.sign ^= NEGATIV;

   return (b_rs);
}

/***************************************************************************/
/*                                                                         */
/*      b_shift  - exponentiate a BCD value                                */
/*                                                                         */
/*      input:                                                             */
/*              BCD *b_rs,*b_op ;                                          */
/*              *b_rs =   b_shift(b_rs,b_op1,i_op2)                        */
/*                                                                         */
/*      output:                                                            */
/*              *b_rs = *b_op1 * 10**i_op2 ;                               */
/*                                                                         */
/*      returns:                                                           */
/*              B_OVERFLOW_ERROR                                           */
/*              B_UNDERFLOW_ERROR                                          */
/*                                                                         */
/***************************************************************************/
BCD *b_shift(b_rs,b_op1,i_op2)

BCD *b_rs, *b_op1;
int i_op2;
{
   int hilf;

   if (b_status < 0) /* Error, can't continue */
   {
      return 0;
   }
   b_init();

   hilf = (*b_op1).mant.chart + i_op2 ;       /* Compute new exponent */

   if (hilf < MIN_CHART)           /* Underflow */
   {
      b_status = B_UNDERFLOW_ERROR ;
      *b_rs = b_zero ;
   }
   else if (hilf > MAX_CHART)      /* Overflow */
   {
      b_status = B_OVERFLOW_ERROR ;
      *b_rs = b_infi ;
      (*b_rs).mant.sign = (*b_op1).mant.sign;
   }
   else                            /* Dataflow */
   {
      *b_rs = *b_op1 ;
      (*b_rs).mant.chart = hilf ;
   }

   return (b_rs);
}



/***************************************************************************/
/***************************************************************************/

/***************************************************************************/
/***************************************************************************/
/*                                                                         */
/*      b_ac_init     Initialize the bcd constants zero, one, and          */
/*                      infinity,etc.                                      */
/*                                                                         */
/*                      Requires no input parameters                       */
/*                                                                         */
/***************************************************************************/
/***************************************************************************/
void b_ac_init()
{

   /*********** bcd 0 ***********/
   b_zero.mant.chart = 0x00;
   b_zero.mant.sign  = POSITIV;
   b_zero.mant.word[0] = 0x0000;
   b_zero.mant.word[1] = 0x0000;
   b_zero.mant.word[2] = 0x0000;
   b_zero.mant.word[3] = 0x0000;
   b_zero.mant.word[4] = 0x0000;

   /*********** bcd 1 ***********/
   b_one.mant.chart = 0x01;
   b_one.mant.sign  = POSITIV;
   b_one.mant.word[0] = 0x1000;
   b_one.mant.word[1] = 0x0000;
   b_one.mant.word[2] = 0x0000;
   b_one.mant.word[3] = 0x0000;
   b_one.mant.word[4] = 0x0000;

   /********** bcd -1 **********/
   b_n_one.mant.chart = 0x01;
   b_n_one.mant.sign  = NEGATIV;
   b_n_one.mant.word[0] = 0x1000;
   b_n_one.mant.word[1] = 0x0000;
   b_n_one.mant.word[2] = 0x0000;
   b_n_one.mant.word[3] = 0x0000;
   b_n_one.mant.word[4] = 0x0000;

   /* This code assumes that int long is always 32 bits and that */
   /* int short is always 16 bits. The only C type which varies  */
   /* accross systems is int, which is either 32 bits or 16 bits */
   /* The int limits are conditionally set based on this info.   */

   /****** bcd -2147483648   ******/
   b_l_minw.mant.chart = 0xA;
   b_l_minw.mant.sign  = NEGATIV;
   b_l_minw.mant.word[0] = 0x2147;
   b_l_minw.mant.word[1] = 0x4836;
   b_l_minw.mant.word[2] = 0x4800;
   b_l_minw.mant.word[3] = 0x0000;
   b_l_minw.mant.word[4] = 0x0000;

   /**** bcd -2147483648.99999 ****/
   b_l_min.mant.chart = 0xA;
   b_l_min.mant.sign  = NEGATIV;
   b_l_min.mant.word[0] = 0x2147;
   b_l_min.mant.word[1] = 0x4836;
   b_l_min.mant.word[2] = 0x4899;
   b_l_min.mant.word[3] = 0x9999;
   b_l_min.mant.word[4] = 0x9900;

   /***** bcd 2147483647.99999 *****/
   b_l_max.mant.chart = 0xA;
   b_l_max.mant.sign  = 0x00;
   b_l_max.mant.word[0] = 0x2147;
   b_l_max.mant.word[1] = 0x4836;
   b_l_max.mant.word[2] = 0x4799;
   b_l_max.mant.word[3] = 0x9999;
   b_l_max.mant.word[4] = 0x9900;

   /**** bcd -32768.9999999999 ****/
   b_s_min.mant.chart = 0x5;
   b_s_min.mant.sign  = NEGATIV;
   b_s_min.mant.word[0] = 0x3276;
   b_s_min.mant.word[1] = 0x8999;
   b_s_min.mant.word[2] = 0x9999;
   b_s_min.mant.word[3] = 0x9999;
   b_s_min.mant.word[4] = 0x9900;

   /*********** bcd -32768 ***********/
   b_s_minw.mant.chart = 0x5;
   b_s_minw.mant.sign  = NEGATIV;
   b_s_minw.mant.word[0] = 0x3276;
   b_s_minw.mant.word[1] = 0x8000;
   b_s_minw.mant.word[2] = 0x0000;
   b_s_minw.mant.word[3] = 0x0000;
   b_s_minw.mant.word[4] = 0x0000;

   /***** bcd 32767.9999999999 *****/
   b_s_max.mant.chart = 0x5;
   b_s_max.mant.sign  = POSITIV;
   b_s_max.mant.word[0] = 0x3276;
   b_s_max.mant.word[1] = 0x7999;
   b_s_max.mant.word[2] = 0x9999;
   b_s_max.mant.word[3] = 0x9999;
   b_s_max.mant.word[4] = 0x9900;

   if (INT_MAX == LONG_MAX)
   {
      b_i_min = b_l_min ;
      b_i_max = b_l_max ;
      b_i_minw = b_l_minw ;
   }
   else /* if (INT_MAX == SHRT_MAX) */
   {
      b_i_min = b_s_min ;
      b_i_max = b_s_max ;
      b_i_minw = b_s_minw ;
   }

   /** bcd 9.9999999999999E62 **/
   b_infi.mant.chart = 0x3F;
   b_infi.mant.sign  = POSITIV;
   b_infi.mant.word[0] = 0x9999;
   b_infi.mant.word[1] = 0x9999;
   b_infi.mant.word[2] = 0x9999;
   b_infi.mant.word[3] = 0x9999;
   b_infi.mant.word[4] = 0x9900;

}
