-- ***************************************************************
-- CoLibri
-- ---------------------------------------------------------------
--
-- Set.cs --
-- 
--   Copyright (C) sietec Systemtechnik GmbH & Co OHG 1994
--   All rights reserved
-- 
-- Author          : Alessandro Graffigna
-- 
-- PURPOSE 
--      This object type realizes the frame for the generic
--      foundation class Set.
--
-- REFER
--      |>Documents with client documentation<|
--
-- ---------------------------------------------------------------
--
-- Created On      : Wed Jan 12 14:57:15 1994
-- Last Modified By: Alessandro Graffigna
-- Last Modified On: Tue Feb 03 00:00:00 1994
-- Update Count    : 6
-- 
-- HISTORY
-- 03-Feb-1994          Alessandro Graffigna
--    All strings "element" changed in "entry".
-- 
-- ***************************************************************

SPECIFICATION Set

IMPORT List FROM List;

TYPE Set [EntryType] = OBJECT

-- ---------------------------------------------------------------
-- General
-- ---------------------------------------------------------------
-- The generic object type Set represents a container consisting
-- of an arbitrary number of entries in an unordered homogeneous
-- collection in which no two entries are identical.
-- The range of types of these entries is defined by the range of
-- types allowed for actual generic parameters of generic object
-- types.
-- The type Set offers methods to include and exclude entries as 
-- well as algebraical methods and methods for testing the relation-
-- ship between sets.
--
-- For the retrieval of entries two different ways are possible:
-- 1) Calling the method "ChooseEntry" one will always get an
--    arbitrary entry out of the set. As a consequence of this
--    it could happen that one will get the same entry again
--    calling this method twice.
-- 2) Additional the type Set will have an internal cursor to iterate
--    on all entries of an object of this type. Calling the sequence
--    "ResetCursor", "GetEntry", "GetEntry", ... one will get all
--    entries of the set exactly once but in an arbitrary and
--    accidental order. Calling this sequence again one will get the
--    entries in another order.
-- ---------------------------------------------------------------
  
PUBLIC
  
-- -----------------------------------------------------------------
-- Basic Functionality:
-- Creation, deletion and copying of sets as well as insertion and
-- deletion of entries.
-- -----------------------------------------------------------------

   -- --------------------------------------------------------------
   -- Creation of a set (done internally by INITIALLY)
   -- Deletion of a set (done internally by FINALLY)
   -- --------------------------------------------------------------

   METHOD Copy () : Set [EntryType];
   -- --------------------------------------------------------------
   -- This method creates a new object of this class containing the
   -- state of the current set.
   -- Exceptions: None.
   -- --------------------------------------------------------------
    
   METHOD InsertEntry (IN entry : EntryType);
   -- --------------------------------------------------------------
   -- This method inserts entry into the current Set.
   -- If entry is already existing in the Set, no action is done.
   -- Exceptions: None.
   -- --------------------------------------------------------------

   METHOD RemoveEntry (IN entry : EntryType);
   -- --------------------------------------------------------------
   -- This method removes entry from the current Set.
   -- If entry is already removed from the Set, no action is done.
   -- Exceptions: None.
   -- --------------------------------------------------------------

   METHOD GetNumberOfEntries () : INT;
   -- --------------------------------------------------------------
   -- This method returns the number of entries of the current Set.
   -- Exceptions: None.
   -- --------------------------------------------------------------
    
   METHOD Apply (IN routine : PROCEDURE (INOUT EntryType));
   -- --------------------------------------------------------------
   -- This method applies routine to all entries of the current Set.
   -- The entry values can be modified by routine.
   -- Exceptions: None.
   -- --------------------------------------------------------------
    
-- -----------------------------------------------------------------
-- Predicates:
-- Tests for emptyness of a set and existence of entries.
-- -----------------------------------------------------------------

   METHOD IsEmpty () : BOOL;
   -- --------------------------------------------------------------
   -- This method tests whether the current Set is empty.
   -- Exceptions: None.
   -- --------------------------------------------------------------

   METHOD Contains (IN entry : EntryType) : BOOL;
   -- --------------------------------------------------------------
   -- This method tests whether the current Set contains entry.
   -- Exceptions: None.
   -- --------------------------------------------------------------
    
-- -----------------------------------------------------------------
-- Retrieval of Entries:
-- -----------------------------------------------------------------

   METHOD ResetCursor ();
   -- --------------------------------------------------------------
   -- This method initializes the internal cursor of the current set.
   -- If the set is empty, no action is done.
   -- Calling this method is always necessary before iterating on the
   -- entries of the current Set.
   -- Exceptions: None.
   -- --------------------------------------------------------------

   METHOD IsCursorEmpty () : BOOL;
   -- --------------------------------------------------------------
   -- This method tests whether there are still entries which are
   -- not yet delivered by calling the method GetEntry.
   -- --------------------------------------------------------------

   METHOD GetEntry () : EntryType;
   -- --------------------------------------------------------------
   -- This method returns an arbitrary and accidental entry of the
   -- current set. No entry will be delivered twice. The delivered
   -- entry remains in the set also.
   -- Calling this method is only possible after initializing the
   -- internal cursor of the current set using the method ResetCursor.
   -- After initialization of the internal cursor the method
   -- InsertEntry inserts all entries into this part of the Set
   -- which has to be still processed by the method GetEntry.
   -- If all entries of the current set are delivered, the internal
   -- cursor is no longer initialized.
   -- Exceptions: CursorUndefined.
   -- --------------------------------------------------------------

   METHOD ChooseEntry () : EntryType;
   -- --------------------------------------------------------------
   -- This method returns an arbitrary and accidental entry of the
   -- current Set. The delivered entry remains in the set also.
   -- Exceptions: ContainerEmpty.
   -- --------------------------------------------------------------
    
-- -----------------------------------------------------------------
-- Algebraical functions on sets:
-- -----------------------------------------------------------------

   METHOD Union (IN second_set : Set [EntryType]);
   -- --------------------------------------------------------------
   -- This method delivers all entries which are in the current set
   -- or in the second_set.
   -- Exceptions: None.
   -- --------------------------------------------------------------

   METHOD Intersection (IN second_set : Set [EntryType]);
   -- --------------------------------------------------------------
   -- This method delivers all entries which are in both, in the
   -- current set and in the second_set.
   -- Exceptions: None.
   -- --------------------------------------------------------------

   METHOD Difference (IN second_set : Set [EntryType]);
   -- --------------------------------------------------------------
   -- This method delivers all entries which are in the current set
   -- but not in the second_set.
   -- Exceptions: None.
   -- --------------------------------------------------------------

   METHOD SymmetricDifference (IN second_set : Set [EntryType]);
   -- --------------------------------------------------------------
   -- This method delivers all entries which are in the current set
   -- or in the second_set but not in both.
   -- Exceptions: None.
   -- --------------------------------------------------------------
    
-- -----------------------------------------------------------------
-- Relations between Sets:
-- -----------------------------------------------------------------
    
   METHOD IsEqual (IN second_set : Set [EntryType]) : BOOL;
   -- --------------------------------------------------------------
   -- This method tests whether the current set is equal to second_set.
   -- Two sets are equal, if they contain exactly the same entries.
   -- Exceptions: None.
   -- --------------------------------------------------------------

   METHOD IsSubSet (IN second_set : Set [EntryType]) : BOOL;
   -- --------------------------------------------------------------
   -- This method tests whether the current set is a real sub-set of
   -- second_set. The current set is a real sub-set of second_set,
   -- if all entries of the current set are also contained in
   -- second_set and, in addition, second_set has other entries.
   -- Exceptions: None.
   -- --------------------------------------------------------------

   METHOD IsSuperSet (IN second_set : Set [EntryType]) : BOOL;
   -- --------------------------------------------------------------
   -- This method tests whether the current set is a real super-set
   -- of second_set. The current set is a real super-set of second_set,
   -- if all entries of second_set are also contained in the current
   -- set and, in addition, the current set has other entries.
   -- Exceptions: None.
   -- --------------------------------------------------------------
    
PROTECTED
      
   METHOD clone (IN state : List [EntryType]);
   -- --------------------------------------------------------------
   -- This internal method copies the main state variable of a set.
   -- Exceptions: None.
   -- --------------------------------------------------------------

STATE

   cursorIsEmpty : BOOL;              -- Indcator if cursor is reseted.
   theSet        : List [EntryType];  -- The current set itself.
   shadowSet     : Set [EntryType];   -- Second set for the retrival of
				      -- entries.
  
END OBJECT (* Set *);

EXCEPTION CursorUndefined;
EXCEPTION ContainerEmpty;

-- ***************************************************************
CONST spec_vid : STRING = 
   "@(#) $Header: Set.cs[1.2] Wed Jan 12 18:11:05 1994 kalle@gianna proposed $" ;

END SPECIFICATION (* Set *);
