(*
-------------------------------------------------------------------------------
-- Name of the source		: Calendar.t
-- Author(s)			: Jose Luis Albero
-- Date of creation		: 13 May 1993
-- Defined Objects		: Calendar
-- Description of the contents	: This module has definitions for a non
--                                Persistent Calendar object. The Calendar
--                                object specifies the set of non-working days,
--                                working hours and non-working hours for the 
--                                environment.
-- Document reference		: 
-- Copyright 1993, TAO S.A., Barcelona, Spain
-- This software was produced as part of the ESPRIT project ITHACA-IT.
-------------------------------------------------------------------------------
-- Modification log	:
-- Version	Date	Author	Description
-- ----------- -------- ------- -----------------------------------------------
-------------------------------------------------------------------------------
*)

SPECIFICATION Calendar 

----------------------------------------------------------------------
-- Calendar type is an object type that implements a calendar with the
-- main features needed. It is used to deal with working durations.
----------------------------------------------------------------------

IMPORT Date FROM Date;  --Date object
IMPORT Time FROM Time;  --Time object
IMPORT TimeStamp FROM TimeSt;   --TimeStamp object
IMPORT MonthDay FROM MonthDay;     --MontDay object
IMPORT TimeInterval FROM TimeInterval; --TimeInterval object
IMPORT Interval FROM Interval;
IMPORT Stream FROM Stream;
IMPORT MAX_NB_TIMETABLES, timeTable, MAX_NB_NORMAL_NW_DAYS,
       MAX_NB_VARIABLE_NW_DAYS, MAX_NB_LOCAL_NW_DAYS, TimeAdjustment
   FROM CalendarPR;  --Calendar dedicated procedures

TYPE Calendar = OBJECT (IN inNominator : STRING)

PUBLIC
  METHOD setNominator (IN inNom : STRING);
    -- sets Calendar nominator.
  METHOD setValidPeriod (IN inInterv : Interval);
    -- sets the validity time interval of the calendar.
    -- timetables should have been set in advance covering the interval
    -- specified here.
    -- When you are working with working days and you have any date out
    -- of the validity period an exception is raised.
  METHOD setWDayStart (IN inAtTime : INT, IN inFixedTime : Time);
    -- sets working day start borderline. inAtTime parameter can get any
    -- of the following values:
    -- 1) START_TIME_CURRENT_DAY :
    --   A working day begins at the start working time of that day. It can
    --   be found in the corresponding timetable. inFixedTime parameter has
    --   no effect.
    -- 2) END_TIME_PREVIOUS_DAY :
    --   A working day begins at the end working time of the previous working
    --   day. It can be found in the corresponding timetable. inFixedTime
    --   parameter has no effect.
    -- 3) FIXED_TIME_CURRENT_DAY :
    --   A working day begins at a fixed time of the current working day.
    --   inFixedTime parameter is the set fixed time.
    -- 4) FIXED_TIME_PREVIOUS_DAY :
    --   A working day begins at a fixed time of the previous working day.
    --   inFixedTime parameter is the set fixed time.
  METHOD setTimeAddWDuration (IN inAtTime : INT, IN inFixedTime : Time);
    -- needed for timeStamp.Add.
    -- sets time rounding used when adding a WDuration to a TimeStamp.
    -- inAtTime parameter can get any of the following values:
    -- 1) START_TIME_CURRENT_DAY :
    --   Set the Time of the Timestamp to the start working time of the result
    --   date. The start working time can be found in the corresponding 
    --   timetable. inFixedTime parameter has no effect.
    -- 2) END_TIME_CURRENT_DAY :
    --   Set the Time of the Timestamp to the end working time of the result
    --   date. The end working time can be found in the corresponding 
    --   timetable. inFixedTime parameter has no effect.
    -- 3) FIXED_TIME_CURRENT_DAY :
    --   Set the Time of the Timestamp to a fixed time in the result
    --   date. inFixedTime parameter has no effect.
  METHOD setTimeSubWDuration (IN inAtTime : INT, IN inFixedTime : Time);
    -- needed for timeStamp.Subtract.
    -- sets time rounding used when subtracting a WDuration to a TimeStamp.
    -- inAtTime parameter can get any of the following values:
    -- 1) START_TIME_CURRENT_DAY :
    --   Set the Time of the Timestamp to the start working time of the result
    --   date. The start working time can be found in the corresponding 
    --   timetable. inFixedTime parameter has no effect.
    -- 2) END_TIME_CURRENT_DAY :
    --   Set the Time of the Timestamp to the end working time of the result
    --   date. The end working time can be found in the corresponding 
    --   timetable. inFixedTime parameter has no effect.
    -- 3) FIXED_TIME_CURRENT_DAY :
    --   Set the Time of the Timestamp to a fixed time in the result
    --   date. inFixedTime parameter has no effect.
  METHOD addWHour (IN inTT : timeTable);
    -- adds a timetable to the corresponding set.
    -- Its validity period should not overlap other timetables in the set.
    -- Type timeTable is defined in the file <CalendarPR.t>.
  METHOD addNormalNWDay (IN inMonthDay : MonthDay);
    -- adds a normal non-working day to the corresponding set.
  METHOD addVariableNWDay (IN inDate : Date);
    -- adds a variable non-working day to the corresponding set.
  METHOD addLocalNWDay (IN inDate : Date);
    -- adds a local non-working day to the corresponding set.
  METHOD getNominator () : STRING;
  METHOD GetType () : STRING;
  METHOD getValidPeriod () : Interval;
  METHOD getWDayStart () : TimeAdjustment;
  METHOD getTimeAddWDuration () : TimeAdjustment;
  METHOD getTimeSubWDuration () : TimeAdjustment;
  METHOD getWHours () : ARRAY [MAX_NB_TIMETABLES] OF timeTable;
  METHOD getNbTimeTables () : INT;
  METHOD getNormalNWDays () : ARRAY [MAX_NB_NORMAL_NW_DAYS] OF MonthDay;
  METHOD getNbNormalNWDays () : INT;
  METHOD getVariableNWDays () : ARRAY [MAX_NB_VARIABLE_NW_DAYS] OF Date;
  METHOD getNbVariableNWDays () : INT;
  METHOD getLocalNWDays () : ARRAY [MAX_NB_LOCAL_NW_DAYS] OF Date;
  METHOD getNbLocalNWDays () : INT;
    -- get the specific information of the calendar .
    -- Type TimeAdjustment is defined in the file <CalendarPR.t>.
  METHOD remWHour (IN inInterv : Interval);
    -- removes all timetables whose validity period is contained by inInterv
    -- from the corresponding set and modifies the validity period of all
    -- timetables that intersect with, but are not contained by, inInterv.
  METHOD remNormalNWDay (IN inMonthDay : MonthDay);
    -- removes a normal non-working day from the corresponding set.
  METHOD remVariableNWDay (IN inDate : Date);
    -- removes a variable non-working day from the corresponding set.
  METHOD remLocalNWDay (IN inDate : Date);
    -- removes a local non-working day from the corresponding set.
  METHOD getStartWHour (IN inDate : Date) : Time;
    -- gets the start working hour for a given day.
  METHOD getEndWHour (IN inDate : Date) : Time;
    -- gets the end working hour for a given day.
  METHOD isNWWeekDay (IN inDate : Date) : BOOL;
    -- is the specified day a non-working week day ?
  METHOD isNormalNWDay (IN inDate : Date) : BOOL;
    -- is the specified day a normal non-working day ?
  METHOD isVariableNWDay (IN inDate : Date) : BOOL;
    -- is the specified day a variable non-working day ?
  METHOD isLocalNWDay (IN inDate : Date) : BOOL;
    -- is the specified day a local non-working day ?
  METHOD isWorkingDay (IN inDate : Date) : BOOL;
    -- is the specified day a working day ? 
  METHOD nextWDay (IN inDate : Date) : Date;
    -- gets the next working day.
  METHOD prevWDay (IN inDate : Date) : Date;
    -- gets the previous working day.
  METHOD isTSInWorkingTime (IN inTS : TimeStamp) : BOOL;
    -- is the specified TimeStamp in working hours ?
  METHOD isIntervalInWorkingDay (IN inInterv : Interval) : BOOL;
    -- is the specified Interval in a working day ?
  METHOD isIntervalInWorkingTime (IN inInterv : Interval) : BOOL;
    -- is the specified Interval in working hours ?
  METHOD isIntervalInWorkingWeek (IN inInterv : Interval) : BOOL;
    -- is the specified Interval in a working week ?
  METHOD nextStartWT (IN inTS : TimeStamp) : TimeStamp;
    -- gets the next working time start.
  METHOD nextEndWT (IN inTS : TimeStamp) : TimeStamp;
    -- gets the next working time end.
  METHOD nextStartWW (IN inTS : TimeStamp) : TimeStamp;
    -- gets next working week start.
  METHOD nextEndWW (IN inTS : TimeStamp) : TimeStamp;
    -- gets next working week end.
  METHOD toWDate (IN inDate : Date) : Date;
  -- gets the working day that corresponds to the input date. It is
  -- used before adding or subtracting a working duration to a date.
  METHOD timeAfterAdd (IN inDate : Date) : Time;
  -- gets the time to be set in TimeStamp after adding a working duration
  -- to it. The input date is the resultant date after addition.
  METHOD timeAfterSubtract (IN inDate : Date) : Time;
  -- gets the time to be set in TimeStamp after subtracting a working duration
  -- to it. The input date is the resultant date after subtraction.
  METHOD copy () : Calendar;
    -- gets a copy of this Calendar.
  METHOD isEqual (IN inCalen : Calendar) : BOOL;
    -- is the current calendar equal to the input calendar ?
  METHOD Serialize (INOUT ioStream : Stream);
    -- Method used by calendar persistence. Calendar state value is stored into
    -- ioStream.
  METHOD Deserialize (IN inStream : Stream);
    -- Method used by calendar persistence. Calendar state value is loaded from
    -- inStream.

PROTECTED
STATE -- Calendar
stNominator : STRING;
  -- A string that represents the name of the calendar.
  -- It must not contain white spaces (blanks, tabs, carry returns, ...).

stValidPeriod : Interval;
  -- An interval that sets the validity period of the calendar.

stWHours : ARRAY [MAX_NB_TIMETABLES] OF timeTable;
  -- This is a sorted set of timetables. Each timetable is made of
  -- a validity period (the interval of time in which the timetable
  -- is defined), the working days in a week and the real timetable
  -- (that gives the working hours for each working day of the week).

stNbTimeTables : INT;
  -- Is the number of timetables in the set.

stNormalNWDays : ARRAY [MAX_NB_NORMAL_NW_DAYS] OF MonthDay;
  -- Another sorted set with normal non-working days in every year.

stNbNormal : INT;
  -- Number of elements in the normal non-working days set.

stVariableNWDays : ARRAY [MAX_NB_VARIABLE_NW_DAYS] OF Date;
  -- Non-working days that are not in a fixed day every year. They are sorted.

stNbVariable : INT;
  -- Number of variable non-working days in the set.

stLocalNWDays : ARRAY [MAX_NB_LOCAL_NW_DAYS] OF Date;
  -- Local non-working days (defined for a region, a city, an enterprise, ...).
  -- They are sorted.

stNbLocal : INT;
  -- Number of local non-working days in the set.

stWDayStart : TimeAdjustment;
  -- This is a state variable that determines, for transaction purposes,
  -- the beginning of a working day. It is used when adding or subtracting
  -- working durations.
  -- Type TimeAdjustment is defined in the file <CalendarPR.t>.

stTimeAddWDuration : TimeAdjustment;
  -- This is a state variable that determines
  -- the time of a TimeStamp after adding to it a working duration.
  -- Type TimeAdjustment is defined in the file <CalendarPR.t>.

stTimeSubWDuration : TimeAdjustment;
  -- This is a state variable that determines the
  -- time of a TimeStamp after subtracting from it a working duration.
  -- Type TimeAdjustment is defined in the file <CalendarPR.t>.


END OBJECT; -- Calendar


END SPECIFICATION;
