/* parse.c - universal lexical scanner for all file formats SCEND uses */

/* Written 1994,1995 by Werner Almesberger */


#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <errno.h>

#include "scend.h"
#include "common.h"
#include "descr.h"
#include "warn.h"
#include "parse.h"


int ch = 0,quoted_string = 0;
char token[MAX_TOKEN+1];

static FILE *file;
static const char *file_name = NULL;
static int line;
static int back_flag = 0;


void parse_error(const char *msg,...)
{
    va_list ap;

    va_start(ap,msg);
    vfprintf(stderr,msg,ap);
    va_end(ap);
    fprintf(stderr,"\n  in %s at line %d\n",file_name,line);
    exit(1);
}


int next(void)
{
    char *here;

    if (back_flag) {
	back_flag = 0;
	return ch;
    }
    quoted_string = 0;
    while ((ch = getc(file)) != EOF)
	if (!isspace(ch))
	    if (ch != '#') break;
	    else {
		while ((ch = getc(file)), ch != '\n' && ch != EOF);
		line++;
	    }
	else if (ch == '\n') line++;
    if (ch == EOF) return ch;
    if (isalnum(ch)) {
	here = token;
	*here++ = ch;
	while ((ch = getc(file)) != EOF)
	    if (!isalnum(ch) && ch != '_')
		if (ch != '\\') break;
		else {
		    while ((ch = getc(file)) != EOF)
			if (!isspace(ch)) break;
		    if (ch == EOF || ch == '\\') break;
		    (void) ungetc(ch,file);
		}
	    else if (here-token < MAX_TOKEN) *here++ = ch;
	        else parse_error("token too long");
	if (ch != EOF) (void) ungetc(ch,file);
	*here = 0;
	return ch = 0;
    }
    if (ch != '"') {
	*token = 0;
	return ch;
    }
    quoted_string = 1;
    here = token;
    while ((ch = getc(file)) != EOF)
	switch (ch) {
	    case '"':
		goto done;
	    case '\\':
		ch = getc(file);
		if (ch != '"' && ch != '\\') {
		    while (ch != EOF)
			if (!isspace(ch)) break;
			else ch = getc(file);
		    if (ch != EOF) (void) ungetc(ch,file);
		    break;
		}
		/* fall through */
	    default:
	    	if (here-token < MAX_TOKEN) *here++ = ch;
		else parse_error("token too long");
	}
    done:
    if (ch == EOF) parse_error("EOF in quoted string");
    *here = 0;
    return ch = 0;
}


void back(void)
{
    back_flag = 1;
}


int non_eof(void)
{
    if (next() == EOF) parse_error("unexpected EOF");
    return ch;
}


const char *string(void)
{
    if (non_eof()) parse_error("string expected instead of %c",ch);
    return token;
}


int number(int *base)
{
    int sign,b,curr,value;
    char *scan;

    if (next() == EOF) parse_error("number expected instead of EOF");
    if (sign = ch == '-')
	if (next() == EOF) parse_error("EOF inside number");
    if (ch || !isdigit(*token))
	parse_error("%c is not a digit",ch ? ch : *token);
    scan = token;
    if (*scan != '0') b = 10;
    else {
	if (!scan[1]) {
	    if (base && !*base) *base = 10;
	    return 0;
	}
	if (scan[1] != 'x' || scan[1] == 'X') b = 8;
	else {
	    b = 16;
	    scan += 2;
	}
    }
    if (base)
	if (!*base) *base = b;
	else if (!b) b = *base;
    if (!b) b = 10;
    value = 0;
    while (*scan) {
	if (isdigit(*scan)) curr = *scan-'0';
	else curr = toupper(*scan)-'A'+10;
	if (curr < 0 || curr >= b) parse_error("%c is not a valid digit",*scan);
	if ((INT_MAX-value)/b < value || ((INT_MAX-value)/b == value &&
	  (INT_MAX-value) % b < curr)) parse_error("number too big");
	value = value*b+curr;
	scan++;
    }
    return sign ? -value : value;
}


static void parse_name(NAME *name)
{
    name->long_name = stralloc(string());
    if (next() == ':') name->short_name = stralloc(string());
    else {
	back();
	name->short_name = name->long_name;
    }
}


static void flush(char **s,char *buffer,char **p)
{
    char *tmp,*here;

    if (*p == buffer) return;
    if (!(tmp = realloc(*s,(size_t) (*p-buffer+1+(*s ? strlen(*s) : 0)))))
	die("out of memory");
    here = *s ? strchr(tmp,0) : tmp;
    *s = tmp;
    memcpy(here,buffer,(size_t) (*p-buffer));
    here[*p-buffer] = 0;
    *p = buffer;
}


const char *parse_help(void)
{
#define MAX_CHUNK 100
    char buffer[MAX_CHUNK],last;
    char *s,*curr;
    int nls;

    if (next() != '[') {
	back();
	return NULL;
    }
    s = NULL;
    curr = buffer;
    nls = last = 0;
    while ((ch = getc(file)) != ']') {
	if (ch == '#') {
	    while ((ch = getc(file)) != '\n')
		if (ch == EOF) parse_error("unexpected EOF");
	    continue;
	}
	if (ch == '\\') ch = getc(file);
	if (ch == EOF) parse_error("unexpected EOF");
	if (ch != '\n') nls = 0;
	else {
	    line++;
	    if (nls++ != 1) ch = ' ';
	}
	if (ch == '\t') ch = ' ';
	if (ch != ' ' || last != ' ') {
	    if (curr-buffer == MAX_CHUNK) flush(&s,buffer,&curr);
	    *curr++ = ch;
	    last = ch == '\n' ? ' ' : ch;
	}
    }
    flush(&s,buffer,&curr);
    return s;
}


static VALUE *value_list(ITEM *parent,int *length,const VALUE *prev,int index,
  void (*fn)(ITEM *this,int i))
{
    VALUE value,*list;
    const VALUE *walk;
    int i;

    parse_name(&value.name);
    value.prev = prev;
    value.visible = !demo;
    if (non_eof() == '=') value.value = stralloc(string());
    else {
	back();
	value.value = value.name.long_name;
    }
    value.like = -1;
    if (non_eof() == '*') fn(parent,index);
    else if (ch || (strcmp(token,"0") && strcmp(token,"1"))) back();
	else if (atoi(token)) fn(parent,index);
    if (non_eof() == '{') value.section = section(parent);
    else {
	value.section = NULL;
	if (ch || strcmp(token,"like")) back();
	else {
	    string();
	    value.like = index-1;
	    for (walk = prev; walk; walk = walk->prev)
		if (!strcmp(walk->name.long_name,token)) break;
		else value.like--;
	    if (!walk) parse_error("no such value \"%s\"",token);
	}
    }
    if (non_eof() != '}') {
	if (ch != ',') back();
	list = value_list(parent,length,&value,index+1,fn);
    }
    else {
	*length = index+1;
	list = alloc(sizeof(VALUE)*(index+1));
    }
    list[index] = value;
    for (i = index+1; i < *length; i++)
	if (list[i].like == index) list[i].section = list[index].section;
    return list;
}


VALUE *values(ITEM *parent,int *length,void (*fn)(ITEM *this,int i))
{
    if (non_eof() != '{') parse_error("{ expected at beginning of values");
    return value_list(parent,length,NULL,0,fn);
}


ITEM *section(ITEM *parent)
{
    char class[MAX_TOKEN+1];
    ITEM *item;

    if (next() == EOF || ch == '}') {
	if (ch != '}') back();
	return NULL;
    }
    back();
    strcpy(class,string());
    if (!strcmp(class,"title")) {
	if (subtitle) parse_error("only one title item allowed");
	subtitle = stralloc(string());
	return section(parent);
    }
    if (!strcmp(class,"warn")) {
	warn_parse();
	return section(parent);
    }
    item = alloc_t(ITEM);
    item->flags = (edit_group ? FLAG_ABSENT : 0) |
      (demo || seen ? FLAG_SEEN : 0);
    item->parent = parent;
    if (!strcmp(class,"if"))
	item->name.long_name = item->name.short_name = item->var = NULL;
    else {
	parse_name(&item->name);
	if (!strcmp(class,"comment")) item->var = NULL;
	else item->var = stralloc(string());
    }
    initialize(class,item);
    if (item->next = section(parent)) item->next->prev = item;
    return item;
}


void parse_open(const char *name)
{
    if (!(file = fopen(name,"r"))) die("open %s: %s",name,strerror(errno));
    file_name = stralloc(name);
    line = 1;
    back_flag = 0;
}


void parse_close(void)
{
    (void) fclose(file);
    free((char *) file_name);
    file_name = NULL;
}
