/*======================================================================

    A shared-memory PCMCIA ethernet driver for linux

    This driver supports the IBM Credit Card Adapter, the NE4100,
    and the Thomas Conrad ethernet card.

    Written by David Hinds, dhinds@allegro.stanford.edu

    This is based in part on Keith Moore's IBM CCAE driver code,
    which in turn was based on Donald Becker's ne2000 code.
    
======================================================================*/

#include <linux/config.h>

#ifdef MODULE
#include <linux/module.h>
#include <linux/version.h>
#endif

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/ptrace.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/delay.h>
#include <asm/io.h>
#include <asm/system.h>

#include <linux/netdevice.h>
#include <drivers/net/8390.h>

#include "version.h"
#include "cs_types.h"
#include "cs.h"
#include "cistpl.h"
#include "ds.h"

#define IBMCC_BASE	(dev->base_addr)
#define IBMCC_CMD	0x00
#define IBMCC_DATAPORT	0x10	/* NatSemi-defined port window offset. */
#define IBMCC_RESET	0x1f	/* Issue a read to reset, a write to clear. */

#define IBMCC_MISC	0x18	/* Misc register for IBM CCAE cards */

#define START_PG	0x40	/* First page of TX buffer */
#define STOP_PG		0x80	/* Last page +1 of RX ring */

#ifdef PCMCIA_DEBUG
static int pc_debug = PCMCIA_DEBUG;
static char *version =
    "ibmcc_cs.c 1.33 1995/05/15 04:46:01 (David Hinds)\n";
#endif

static char *if_names[] = {"10baseT", "undefined", "undefined", "BNC"};

/*====================================================================*/

/* Parameters that can be set with 'insmod' */

/* Bit map of interrupts to choose from */
static u_long irq_mask = 0xdeb8;

/* Transceiver type */
static int if_port = 3;

/* Shared memory speed, in ns */
static int mem_speed = 0;

/*====================================================================*/

int ibmcc_probe(struct device *dev);

static void ibmcc_config(dev_link_t *link);
static void ibmcc_release(u_long arg);
static int ibmcc_event(event_t event, int priority,
		       event_callback_args_t *args);

static int ibmcc_open(struct device *dev);
static int ibmcc_close(struct device *dev);

static void ibmcc_reset_8390(struct device *dev);
static int ibmcc_block_input(struct device *dev, int count,
			     char *buf, int ring_offset);
static void ibmcc_block_output(struct device *dev, const int count,
			       const unsigned char *buf,
			       const int start_page);

static dev_info_t dev_info = "ibmcc_cs";

static dev_link_t *ibmcc_attach(void);
static void ibmcc_detach(dev_link_t *);

static dev_link_t *dev_list = NULL;

/*====================================================================*/

static void cs_error(int func, int ret)
{
    CardServices(ReportError, dev_info, (void *)func, (void *)ret);
}

/*======================================================================

    If we're linked into the kernel, this gets called from the net
    driver initialization code in Space.c.  It returns failure,
    since all ibmcc devices are dynamically allocated.
    
======================================================================*/

#ifndef MODULE
int ibmcc_probe(struct device *dev)
{
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(version);
#endif
    register_pcmcia_driver(&dev_info, &ibmcc_attach, &ibmcc_detach);
    return -1;
}
#endif

/*======================================================================

    We never need to do anything when a ibmcc device is "initialized"
    by the net software, because we only register already-found cards.
    
======================================================================*/

static int ibmcc_init(struct device *dev)
{
    return 0;
}

/*======================================================================

    ibmcc_attach() creates an "instance" of the driver, allocating
    local data structures for one device.  The device is registered
    with Card Services.

======================================================================*/

static dev_link_t *ibmcc_attach(void)
{
    client_reg_t client_reg;
    dev_link_t *link;
    struct device *dev;
    int ret;
    
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("ibmcc_attach()\n");
#endif

    /* Create new ethernet device */
    link = kmalloc(sizeof(struct dev_link_t), GFP_KERNEL);
    memset(link, 0, sizeof(struct dev_link_t));
    link->release.function = &ibmcc_release;
    link->release.data = (u_long)link;
    link->io.Attributes1 = IO_DATA_PATH_WIDTH_8;
    link->io.NumPorts1 = 32;
    link->io.IOAddrLines = 5;
    link->irq.Attributes = IRQ_TYPE_EXCLUSIVE;
    link->irq.IRQInfo1 = IRQ_INFO2_VALID|IRQ_LEVEL_ID;
    link->irq.IRQInfo2 = irq_mask;
    link->conf.Attributes = CONF_ENABLE_IRQ;
    link->conf.Vcc = 50;
    link->conf.IntType = INT_MEMORY_AND_IO;
    link->conf.ConfigIndex = 1;
    link->conf.Present = PRESENT_OPTION;
    
    dev = kmalloc(sizeof(struct device), GFP_KERNEL);
    memset(dev, 0, sizeof(struct device));
    ethdev_init(dev);
    dev->name = link->dev_name;
    dev->init = &ibmcc_init;
    dev->open = &ibmcc_open;
    dev->stop = &ibmcc_close;
    dev->tbusy = 1;
    link->priv = dev;
    
    /* Register with Card Services */
    link->next = dev_list;
    dev_list = link;
    client_reg.dev_info = &dev_info;
    client_reg.Attributes = INFO_IO_CLIENT | INFO_CARD_SHARE;
    client_reg.EventMask =
	CS_EVENT_CARD_INSERTION | CS_EVENT_CARD_REMOVAL |
	CS_EVENT_RESET_PHYSICAL | CS_EVENT_CARD_RESET |
	CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
    client_reg.event_handler = &ibmcc_event;
    client_reg.Version = 0x0210;
    client_reg.event_callback_args.client_data = link;
    ret = CardServices(RegisterClient, &link->handle, &client_reg);
    if (ret != 0) {
	cs_error(RegisterClient, ret);
	ibmcc_detach(link);
	return NULL;
    }

    return link;
} /* ibmcc_attach */

/*======================================================================

    This deletes a driver "instance".  The device is de-registered
    with Card Services.  If it has been released, all local data
    structures are freed.  Otherwise, the structures will be freed
    when the device is released.

======================================================================*/

static void ibmcc_detach(dev_link_t *link)
{
    dev_link_t **linkp;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("ibmcc_detach(0x%p)\n", link);
#endif
    
    /* Locate device structure */
    for (linkp = &dev_list; *linkp; linkp = &(*linkp)->next)
	if (*linkp == link) break;
    if (*linkp == NULL)
	return;

    if (link->state & DEV_CONFIG) {
	ibmcc_release((u_long)link);
	if (link->state & DEV_STALE_CONFIG) {
	    link->state |= DEV_STALE_LINK;
	    return;
	}
    }
    
    if (link->handle)
	CardServices(DeregisterClient, link->handle);
    
    /* Unlink device structure, free bits */
    *linkp = link->next;
    if (link->priv) {
	struct device *dev = link->priv;
	if (dev->priv)
	    kfree_s(dev->priv, sizeof(struct ei_device));
	kfree_s(dev, sizeof(struct device));
    }
    kfree_s(link, sizeof(struct dev_link_t));
    
} /* ibmcc_detach */

/*======================================================================

    ibmcc_config() is scheduled to run after a CARD_INSERTION event
    is received, to configure the PCMCIA socket, and to make the
    ethernet device available to the system.
    
======================================================================*/

static void ibmcc_config(dev_link_t *link)
{
    client_handle_t handle;
    tuple_t tuple;
    cisparse_t parse;
    struct device *dev;
    int i, j;
    u_char buf[64];
    char *id;
    win_req_t req;
    modwin_t mod;
    memreq_t mem;
    
    handle = link->handle;
    dev = link->priv;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("ibmcc_config(0x%p)\n", link);
#endif

    do {
	tuple.Attributes = 0;
	tuple.DesiredTuple = CISTPL_CONFIG;
	i = CardServices(GetFirstTuple, handle, &tuple);
	if (i != CS_SUCCESS) break;
	tuple.TupleData = buf;
	tuple.TupleDataMax = 64;
	tuple.TupleOffset = 0;
	i = CardServices(GetTupleData, handle, &tuple);
	if (i != CS_SUCCESS) break;
	i = CardServices(ParseTuple, handle, &tuple, &parse);
	if (i != CS_SUCCESS) break;
	link->conf.ConfigBase = (caddr_t)parse.config.base;
    } while (0);
    if (i != CS_SUCCESS) {
	cs_error(ParseTuple, i);
	link->state &= ~DEV_CONFIG_PENDING;
	return;
    }
    
    id = "IBM CreditCard Ethernet Adapter";

    /* Configure card */
    link->state |= DEV_CONFIG;

    do {
	for (j = 0x300; j < 0x400; j += 0x20) {
	    link->io.BasePort1 = j;
	    link->io.BasePort2 = j+0x10;
	    i = CardServices(RequestIO, link->handle, &link->io);
	    if (i == CS_SUCCESS) break;
	}
	if (i != CS_SUCCESS) {
	    cs_error(RequestIO, i);
	    break;
	}
	i = CardServices(RequestIRQ, link->handle, &link->irq);
	if (i != CS_SUCCESS) {
	    cs_error(RequestIRQ, i);
	    break;
	}
	i = CardServices(RequestConfiguration, link->handle, &link->conf);
	if (i != CS_SUCCESS) {
	    cs_error(RequestConfiguration, i);
	    break;
	}

	/* Allocate a 16K memory window */
	req.Attributes = WIN_DATA_WIDTH_8|WIN_MEMORY_TYPE_AM|WIN_ENABLE;
	req.Base = NULL;
	req.Size = (STOP_PG-START_PG)<<8;
	req.AccessSpeed = mem_speed;
	link->win = (window_handle_t)link->handle;
	i = CardServices(RequestWindow, &link->win, &req);
	if (i != 0) {
	    cs_error(RequestWindow, i);
	    break;
	}
	dev->mem_start = (u_long)req.Base;
	dev->rmem_start = dev->mem_start + (TX_PAGES<<8);
	dev->mem_end = dev->rmem_end = dev->mem_start + req.Size;

	/* First, use the window to get the hardware address */
	mem.CardOffset = 0; mem.Page = 0;
	i = CardServices(MapMemPage, link->win, &mem);
	if (i != 0) {
	    cs_error(MapMemPage, i);
	    break;
	    }
	for (j = 0; j < 6; j++)
	    dev->dev_addr[j] = req.Base[0xff0 + (j<<1)];

	/* Now set up the window to map the packet buffer */
	mod.Attributes = WIN_DATA_WIDTH_16|WIN_MEMORY_TYPE_CM|WIN_ENABLE;
	mod.AccessSpeed = mem_speed;
	i = CardServices(ModifyWindow, link->win, &mod);
	if (i != 0) {
	    cs_error(ModifyWindow, i);
	    break;
	    }
	mem.CardOffset = START_PG<<8;
	i = CardServices(MapMemPage, link->win, &mem);
	if (i != 0) {
	    cs_error(MapMemPage, i);
	    break;
	    }
	
	i = request_irq(link->irq.AssignedIRQ, &ei_interrupt,
			0, "ibmcc_cs");
	if (i != CS_SUCCESS) {
	    printk("ibmcc_cs: unable to allocate irq %ld\n",
		   link->irq.AssignedIRQ);
	    break;
	}
	dev->irq = link->irq.AssignedIRQ;
	dev->base_addr = link->io.BasePort1;
	dev->if_port = if_port;
	dev->tbusy = 0;
	i = register_netdev(dev);
	if (i != 0) {
	    printk("ibmcc_cs: register_netdev() failed\n");
	    break;
	}
    } while (0);
    
    link->state &= ~DEV_CONFIG_PENDING;
    if (i != 0) {
	ibmcc_release((u_long)link);
	return;
    }
    
    ei_status.name = id;
    ei_status.word16 = 1;
    ei_status.tx_start_page = START_PG;
    ei_status.rx_start_page = START_PG + TX_PAGES;
    ei_status.stop_page = STOP_PG;
    ei_status.reset_8390 = &ibmcc_reset_8390;
    ei_status.block_input = &ibmcc_block_input;
    ei_status.block_output = &ibmcc_block_output;
    
    printk("%s: %s, port %#3lx, mem %#5lx, irq %d,",
	   dev->name, id, dev->base_addr, dev->mem_start, dev->irq);
    printk(" %s port,", if_names[dev->if_port]);
    for (i = 0; i < ETHER_ADDR_LEN; i++)
	printk(" %02X", dev->dev_addr[i]);
    printk("\n");
    
} /* ibmcc_config */

/*======================================================================

    After a card is removed, ibmcc_release() will unregister the net
    device, and release the PCMCIA configuration.  If the device is
    still open, this will be postponed until it is closed.
    
======================================================================*/

static void ibmcc_release(u_long arg)
{
    dev_link_t *link = (dev_link_t *)arg;
    struct device *dev = link->priv;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("ibmcc_release(0x%p)\n", link);
#endif
    
    if (link->open) {
	printk("ibmcc_cs: release postponed, '%s' still open\n",
	       link->dev_name);
	link->state |= DEV_STALE_CONFIG;
	return;
    }
    
    if (link->dev_name[0] != '\0')
	unregister_netdev(dev);
    
    CardServices(ReleaseWindow, link->win);
    CardServices(ReleaseConfiguration, link->handle);
    CardServices(ReleaseIO, link->handle, &link->io);
    CardServices(ReleaseIRQ, link->handle, &link->irq);
    if (dev->irq != 0) {
	free_irq(dev->irq);
	irq2dev_map[dev->irq] = NULL;
    }
    
    link->state &= ~DEV_CONFIG;
    if (link->state & DEV_STALE_LINK)
	ibmcc_detach(link);
    
} /* ibmcc_release */

/*======================================================================

    The card status event handler.  Mostly, this schedules other
    stuff to run after an event is received.  A CARD_REMOVAL event
    also sets some flags to discourage the net drivers from trying
    to talk to the card any more.
    
======================================================================*/

static int ibmcc_event(event_t event, int priority,
		       event_callback_args_t *args)
{
    dev_link_t *link = args->client_data;
    struct device *dev = link->priv;
    
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("ibmcc_event()\n");
#endif
    
    switch (event) {
#ifdef PCMCIA_DEBUG
    case CS_EVENT_REGISTRATION_COMPLETE:
	if (pc_debug)
	    printk("ibmcc_cs: registration complete\n");
	break;
#endif
    case CS_EVENT_CARD_REMOVAL:
	link->state &= ~DEV_PRESENT;
	if (link->state & DEV_CONFIG) {
	    dev->tbusy = 1; dev->start = 0;
	    link->release.expires = 5;
	    add_timer(&link->release);
	}
	break;
    case CS_EVENT_CARD_INSERTION:
	link->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
	ibmcc_config(link);
	break;
    case CS_EVENT_PM_SUSPEND:
	link->state |= DEV_SUSPEND;
	/* Fall through... */
    case CS_EVENT_RESET_PHYSICAL:
	if (link->state & DEV_CONFIG) {
	    if (link->open) {
		dev->tbusy = 1; dev->start = 0;
	    }
	    CardServices(ReleaseConfiguration, link->handle);
	}
	break;
    case CS_EVENT_PM_RESUME:
	link->state &= ~DEV_SUSPEND;
	/* Fall through... */
    case CS_EVENT_CARD_RESET:
	if (link->state & DEV_CONFIG) {
	    CardServices(RequestConfiguration, link->handle, &link->conf);
	    if (link->open) {
		ibmcc_reset_8390(dev);
		NS8390_init(dev, 1);
		dev->tbusy = 0; dev->start = 1;
	    }
	}
	break;
    }
    return 0;
} /* ibmcc_event */

/*====================================================================*/

static int ibmcc_open(struct device *dev)
{
    dev_link_t *link;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("ibmcc_open('%s')\n", dev->name);
#endif
    
    for (link = dev_list; link; link = link->next)
	if (link->priv == dev) break;
    if (!DEV_OK(link))
	return -ENODEV;

    link->open++;
#ifdef MODULE
    MOD_INC_USE_COUNT;
#endif
    
    return ei_open(dev);
} /* ibmcc_open */

/*====================================================================*/

static int ibmcc_close(struct device *dev)
{
    dev_link_t *link;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("ibmcc_close('%s')\n", dev->name);
#endif
    
    for (link = dev_list; link; link = link->next)
	if (link->priv == dev) break;
    if (link == NULL)
	return -ENODEV;
    
    link->open--; dev->start = 0;
    if (link->state & DEV_STALE_CONFIG)
	ibmcc_release((u_long)link);
    
#ifdef MODULE
    MOD_DEC_USE_COUNT;
#endif
    
    return 0;
} /* ibmcc_close */

/*======================================================================
  
    Hard reset the card.  This used to pause for the same period that
    a 8390 reset command required, but that shouldn't be necessary.
  
======================================================================*/
  
static void ibmcc_reset_8390(struct device *dev)
{
    int tmp, i;
    
    if (ei_debug > 1) printk("resetting the 8390 t=%ld...", jiffies);
    ei_status.txing = 0;
     
    tmp = inb_p(IBMCC_BASE + IBMCC_RESET);
    outb_p(tmp, IBMCC_BASE + IBMCC_RESET);
    for (i = 0; i < 100; i++) {
	if ((inb_p(IBMCC_BASE+EN0_ISR) & ENISR_RESET) != 0)
	    break;
	udelay(100L);
    }
    if (i == 100)
	printk("%s: ibmcc_reset_8390() did not complete.\n", dev->name);

    tmp = inb_p(IBMCC_BASE + IBMCC_MISC) & ~1;
    if (dev->if_port == 3)
	tmp |= 1;
    outb_p(tmp | 8, IBMCC_BASE + IBMCC_MISC);

} /* ibmcc_reset_8390 */

/*======================================================================

    Shared memory block input and output for IBM CCAE's

======================================================================*/

static void
ibmccae_copyin (unsigned char *dest, unsigned char *src, int c)
{
    unsigned short *d = (unsigned short *) dest;
    unsigned short *s = (unsigned short *) src;
    int odd;
    
    if (c <= 0)
	return;
    odd = (c & 01); c >>= 1;

    if (c) {
	do { *d++ = *s++; } while (--c);
    }
    /* get last byte by fetching a word and masking */
    if (odd)
	*((unsigned char *) d) = *s & 0xff;
}

static void
ibmccae_copyout (unsigned char *dest, const unsigned char *src, int c)
{
    unsigned short *d = (unsigned short *) dest;
    unsigned short *s = (unsigned short *) src;
    int odd;
    
    if (c <= 0)
	return;
    odd = (c & 01);
    c >>= 1;

    if (c) {
	do { *d++ = *s++; } while (--c);
    }
    /* copy last byte doing a read-modify-write */
    if (odd)
	*d = (*d & 0xff00) | *((unsigned char *) s);
}

/*====================================================================*/

static int ibmcc_block_input(struct device *dev, int count,
			     char *buf, int ring_offset)
{
    void *xfer_start = (void *)(dev->mem_start + ring_offset
				- (START_PG<<8));
    
    if (ei_debug > 4)
	printk ("[bi=%d @ %x]\n", count, ring_offset);

    if (xfer_start + count > (void*) dev->rmem_end) {
	/* We must wrap the input move. */
	int semi_count = (void*)dev->rmem_end - xfer_start;
	ibmccae_copyin (buf, xfer_start, semi_count);
	count -= semi_count;
	ibmccae_copyin (buf + semi_count, (char *) dev->rmem_start, count);
	return dev->rmem_start + count;
    }
    ibmccae_copyin (buf, xfer_start, count);
    return ring_offset + count;
}

static void ibmcc_block_output(struct device *dev, int count,
			       const unsigned char *buf,
			       const int start_page)
{
    void *shmem = (void *)dev->mem_start + ((start_page - START_PG)<<8);

    if (ei_debug > 4)
	printk ("[bo=%d @ %x]\n", count, start_page);

    ibmccae_copyout (shmem, buf, count);
}

/*====================================================================*/

#ifdef MODULE
char kernel_version[] = UTS_RELEASE;

int init_module(void)
{
    servinfo_t serv;
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(version);
#endif
    CardServices(GetCardServicesInfo, &serv);
    if (serv.Revision != CS_RELEASE_CODE) {
	printk("ibmcc_cs: Card Services release does not match!\n");
	return -1;
    }
    register_pcmcia_driver(&dev_info, &ibmcc_attach, &ibmcc_detach);
    return 0;
}

void cleanup_module(void)
{
    printk("ibmcc_cs: unloading\n");
    unregister_pcmcia_driver(&dev_info);
    while (dev_list != NULL)
	ibmcc_detach(dev_list);
}
#endif /* MODULE */
