#ifndef __RS_VEC_H__
#define __RS_VEC_H__

////////////////////////////////////////////////////////////////////////////////
//                      Universal vector class.                               //   
//  LAST EDIT: Fri Aug  5 08:55:07 1994 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //
//              yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#ifndef RTD_CPP_INCLUDES
extern "C" {
#endif

#include <math.h>

#ifndef RTD_CPP_INCLUDES
}
#endif

#include "rs_mvgen.h"
#include "rs_io.h"

class RT_RS_Matrix;    //forward to avoid cycles

//a general vector for several use
class RT_RS_Vector : public RT_RS_MVGeneral, public RT_RS_IO
{
protected:
  float *v;    // pointer to the vector itself
  void freeh(); // free heap memory
  void alloch(int anr, int anc, char atemp); //allocate heap memory
public:
  char *get_class() { return "RS_Vector"; }

  //constructors
  RT_RS_Vector(char *filename, FILE *fp = NULL);
  RT_RS_Vector(int anr, int anc = 1, char atemp = 'n')
    { alloch(anr, anc, atemp); }
  RT_RS_Vector( RT_RS_Vector& a);
  //destructor
  ~RT_RS_Vector() { freeh(); }

  void set(float f = .0); // Set the elements with value v
  
  RT_RS_Vector& operator = ( RT_RS_Vector& a);
  RT_RS_Vector  operator ~ ();        // Transposition
  float&        operator () (int i);  // range checked element
  RT_RS_Vector  operator +  ( RT_RS_Vector& a);
  RT_RS_Vector  operator -  ( RT_RS_Vector& a);
  RT_RS_Vector  operator *  (float f);
  RT_RS_Vector  operator /  (float f);

  friend RT_RS_Matrix  operator *  (RT_RS_Vector& a, RT_RS_Vector& b);
  friend RT_RS_Vector  operator *  (RT_RS_Matrix& a, RT_RS_Vector& b);
  friend RT_RS_Vector  operator *  (RT_RS_Vector& a, RT_RS_Matrix& b);
  friend int MatEqSys(RT_RS_Matrix& A, RT_RS_Vector& X, RT_RS_Vector& B, int flag );
  friend void TriMatEqSys(int n, RT_RS_Vector& lower, RT_RS_Vector& diag, RT_RS_Vector& upper,
                          RT_RS_Vector& XB, int flag); //simultaneous linear equations with a tridiagonal matrix
  friend void nspline(RT_RS_Vector& x, RT_RS_Vector& y,
                      RT_RS_Vector& b, RT_RS_Vector& c, RT_RS_Vector& d); //compute the coefficients of a cubic spline
  friend float spval(float x0, RT_RS_Vector& a, RT_RS_Vector& b, RT_RS_Vector& c, RT_RS_Vector& d,
                     RT_RS_Vector& x); // evaluate a cubic spline at x0;
  //in/out functions 
  boolean read(char *filename, FILE *fp = NULL);  // Read ASCII file.
  boolean write(char *filename, FILE *fp = NULL); // Write ASCII file
  void print(FILE *f, char *n = NULL, int width = 8, int decimalPlaces = 2);
};


class RT_RS_3DVector : public RT_RS_IO {
  public:
    float x, y, z;

  char* get_class() { return "RS_3DVector"; }

  RT_RS_3DVector(char *filename, FILE *fp = NULL) { read(filename, fp); }
  RT_RS_3DVector() { x = y = z = .0; }
  RT_RS_3DVector(float a, float b, float c): x(a), y(b), z(c) {}

  RT_RS_3DVector operator = (const RT_RS_3DVector& v) 
    { x=v.x; y=v.y; z=v.z; return *this; }
  RT_RS_3DVector operator + (const RT_RS_3DVector& v) const
    { RT_RS_3DVector c(x+v.x, y+v.y, z+v.z); return c; }
  RT_RS_3DVector operator - (const RT_RS_3DVector& v) const
    { RT_RS_3DVector c(x-v.x, y-v.y, z-v.z); return c; }
  RT_RS_3DVector operator * (const float v) const
    { RT_RS_3DVector c(v*x, v*y, v*z); return c; }
  float vabs() const { return sqrt(x*x + y*y + z*z); }
  float dot(const RT_RS_3DVector& v) const {  return x*v.x + y*v.y + z*v.z; }
  RT_RS_3DVector cross(const RT_RS_3DVector& v) const
    { RT_RS_3DVector c(y*v.z-z*v.y, z*v.x-x*v.z, x*v.y-y*v.x); return c; }
  RT_RS_3DVector uni() const
    { float r = vabs(); RT_RS_3DVector c(x/r, y/r, z/r); return c; }
  RT_RS_3DVector neg() const { RT_RS_3DVector c(-x, -y, -z); return c; }

  boolean read(char *filename, FILE *fp = NULL);  // Read ASCII from a file.
  boolean write(char *filename, FILE *fp = NULL); // Write ASCII file
  void print(FILE *f, char *n = NULL, int width = 12, int decimalPlaces = 6);
};

#endif
 

