////////////////////////////////////////////////////////////////////////////////
//                      Universal vector class.                               //   
//  LAST EDIT: Fri Aug  5 08:55:04 1994 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //
//              yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include <stdio.h>  
#include <stdlib.h>  
#include <math.h>     
#include <string.h>
#include <values.h>
#include "rs_vec.h"
#include "rs_io.h"
#include "../global.h"
#include "../error.h"

int MatEqSys(RT_RS_Matrix& A, RT_RS_Vector& X,
                      RT_RS_Vector& B, int flag = 0 ); 

///////////////// The RT_RS_Vector member functions  /////////////////////////

void RT_RS_Vector::alloch(int anr, int anc, char atemp)
{
  nr = anr;
  nc = anc;
  temp = atemp;
  nelem = nr + nc - 1;
#ifdef RS_DEBUG
  if(anr != 1 && anc != 1)
    rt_Output->fatalVar(get_class(),
      ":Either the number of rows or the number of columns must be 1", NULL);
  if(nelem < 0)
    rt_Output->fatalVar(get_class(), ":Number of elements is < 0", NULL);
#endif
  if(nelem == 0)
    v = NULL;
  else
    v = new float[nelem];
}

void RT_RS_Vector::freeh()
{
  if(v == NULL) return;
  delete v;
  v = NULL;
}

RT_RS_Vector::RT_RS_Vector(char *filename, FILE *fp)
{
  v = NULL;
  if(!read(filename, fp)) freeh();
}

RT_RS_Vector::RT_RS_Vector( RT_RS_Vector& a) {
#ifdef RS_DEBUG
  if(a.v == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in copy-constructor", NULL);
#endif
  nr = a.nr;
  nc = a.nc;
  temp = a.temp;
  nelem = a.nelem;

  if(a.temp == 'y') {
    v = a.v;
    a.v = NULL;
  }
  else {
    v = new float[nelem];
#ifdef RS_USE_MEM_FUNCS
    memcpy(v, a.v, nelem * sizeof(float));
#else
     for(int i = 0; i < nelem; i++) v[i] = a.v[i];
#endif
  }
}

void RT_RS_Vector::set(float f)
{
  for(int i = 0; i < nelem; i++) v[i] = f;
}

RT_RS_Vector& RT_RS_Vector::operator = ( RT_RS_Vector& a) {
#ifdef RS_DEBUG
  if(v == NULL || a.v == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator =", NULL);
#endif
  int n = (nelem < a.nelem) ? nelem : a.nelem;
#ifdef RS_USE_MEM_FUNCS
  memcpy(v, a.v, n * sizeof(float));
#else
  for(int i = 0; i < n; i++) v[i] = a.v[i];
#endif
  a.freet();
  return *this;
}

RT_RS_Vector RT_RS_Vector::operator ~ () // Transposition
{
#ifdef RS_DEBUG
  if(v == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator ~", NULL);
#endif
  RT_RS_Vector Temp(nc, nr, 'y');      //nc <-> nr
#ifdef RS_USE_MEM_FUNCS
  memcpy(Temp.v, v, nelem * sizeof(float));
#else
  for(int i = 0; i < nelem; i++) Temp.v[i] = v[i];
#endif
  return Temp;
}

float& RT_RS_Vector::operator () (int i)
{
#ifdef RS_DEBUG
  if(v == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator ()", NULL);
  if(i < 0 || (nc == 1 && i >= nr) || (nr == 1 && i >= nc))
    rt_Output->fatalVar(get_class(), ": Illegal index in operator ()", NULL);
#endif
  return v[i];
}

RT_RS_Vector RT_RS_Vector::operator + ( RT_RS_Vector& a) {
#ifdef RS_DEBUG
  if(v == NULL || a.v == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator +", NULL);
  if(nelem != a.nelem)
    rt_Output->fatalVar(get_class(), ":Dimensions do not match in operator +", NULL);
#endif
  RT_RS_Vector Temp(nr, nc, 'y');
  for(int i = 0; i < nelem; i++)
    Temp.v[i] = v[i] + a.v[i];
  freet();
  a.freet();
  return Temp;
}

RT_RS_Vector RT_RS_Vector::operator - ( RT_RS_Vector& a) {
#ifdef RS_DEBUG
  if(v == NULL || a.v == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator -", NULL);
  if(nelem != a.nelem)
    rt_Output->fatalVar(get_class(), ":Dimensions do not match in operator -", NULL);
#endif
  RT_RS_Vector Temp(nr, nc, 'y');
  for(int i = 0; i < nelem; i++)
    Temp.v[i] = v[i] - a.v[i];
  freet();
  a.freet();
  return Temp;
}

RT_RS_Vector RT_RS_Vector::operator * (float f)
{
#ifdef RS_DEBUG
  if(v == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator * (float)", NULL);
#endif
  RT_RS_Vector Temp(nr, nc, 'y');
  for(int i = 0; i < nelem; i++)
    Temp.v[i] = f * v[i];
  freet();
  return Temp;
}

RT_RS_Vector RT_RS_Vector::operator / (float f)
{
#ifdef RS_DEBUG
  if(v == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator /", NULL);
#endif
  if(f < epsilon)
    rt_Output->fatalVar(get_class(), ":Division by zero in operator /", NULL);
  float recip = 1./f;
  RT_RS_Vector Temp(nr,nc,'y');
  for(int i = 0; i < nelem; i++)
    Temp.v[i] = v[i] * recip;
  freet();
  return Temp;
}

boolean RT_RS_Vector::read(char *filename, FILE *fp)
{
  int _nr, _nc, i;
  boolean aln = fp == NULL;
  boolean Ok = TRUE;
  int clmn = 0;

  if(aln) Ok = openfile(filename, fp, "rt");
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) Ok = isclass(filename, RS_s1);
  if(Ok) Ok = getint(filename, fp, _nr, RS_s, clmn);
  if(Ok) Ok = getint(filename, fp, _nc, RS_s, clmn);
  if(Ok) {
    freeh();
    alloch(_nr, _nc, 'n');
    for(i = 0; i < nelem && Ok; i++)
      Ok = getfloat(filename, fp, v[i], RS_s, clmn);
  }
  if(aln) fclose(fp);
  return Ok;
}

boolean RT_RS_Vector::write(char *filename, FILE *fp)
{
#ifdef RS_DEBUG
  if(v == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), NULL);
#endif
  int i;
  int aln = fp == NULL;
  boolean Ok = TRUE;

  if(aln) Ok = openfile(filename, fp, "wt");
  if(Ok) {
    fprintf(fp, "%s %d %d\n", get_class(), nr, nc);
    Ok = !ferror(fp);
  }
  for(i = 0; i < nelem && Ok; i++) {
    fprintf(fp, "% .6f\n", v[i]);
    Ok = !ferror(fp);
  }
  if(!Ok) {
    rt_Output->errorVar(get_class(), ":Cannot write ", filename, NULL);
    return FALSE;
  }
  if(aln) fclose(fp);
  return TRUE;
}

void RT_RS_Vector::print(FILE *f, char *n, int width, int decimalPlaces)
{
  if(v == NULL) return;
  if(n) fprintf(f, "%s\n", n);
  int nperline = 79 / width;
  sprintf(fmt, "%s%d.%df", "%", width, decimalPlaces);
  for(int i = 0; i < nelem; i++) {
    fprintf(f, fmt, v[i]);
    if(i % nperline == nperline - 1) fprintf(f, "\n");
  }
}

///////////////// The RT_RS_3DVector member functions  /////////////////////////
boolean RT_RS_3DVector::read(char *filename, FILE *fp)
{
  boolean aln = fp == NULL;
  boolean Ok = TRUE;
  int clmn = 0;

  if(aln) Ok = openfile(filename, fp, "rt");
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) Ok = isclass(filename, RS_s1);
  if(Ok) Ok = getfloat(filename, fp, x, RS_s, clmn);
  if(Ok) Ok = getfloat(filename, fp, y, RS_s, clmn);
  if(Ok) Ok = getfloat(filename, fp, z, RS_s, clmn);
  if(aln) fclose(fp);
  return Ok;
}

boolean RT_RS_3DVector::write(char *filename, FILE *fp)
{
  boolean aln = fp == NULL;
  boolean Ok = TRUE;

  if(aln) Ok = openfile(filename, fp, "wt");
  if(Ok) {
    fprintf(fp,"%s\n%.6f %.6f %.6f \n", get_class(), x, y, z);
    if(ferror(fp)) {
      rt_Output->errorVar(get_class(), ": Cannot write ", filename, 0 );
      return FALSE;
    }
  }
  if(aln) fclose(fp);
  return Ok;
}

void RT_RS_3DVector::print(FILE *f, char *n, int width, int decimalPlaces)
{
  if(n) fprintf(f, "%s\n", n);
  sprintf(fmt,"%s%d.%df", "%", width, decimalPlaces);
   fprintf(f, fmt, x); fprintf(f, fmt, y); fprintf(f, fmt, z);
  fprintf(f, "\n");
}

