////////////////////////////////////////////////////////////////////////////////
//                      Point related classes.                                //   
//  LAST EDIT: Sat AUG 20 18:12:05 1994 by H. Fischer
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //
//              ekki@prakinf.tu-ilmenau.de                                    //
//                                                                            //
// (C) Copyright 1993 YART team                                               //
////////////////////////////////////////////////////////////////////////////////
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <values.h>
#include "rs_pnts.h"
#include "rs_io.h"
#include "rs_areas.h"
#include "../global.h"
#include "../error.h"

int MatEqSys(RT_RS_Matrix& A, RT_RS_Vector& X,
                      RT_RS_Vector& B, int flag = 0 ); 

//////////////////////  THE RT_RS_Vtx MEMBER FUNCTIONS /////////////////////////

RT_RS_Vtx::RT_RS_Vtx(const RT_RS_3DVector& apnt, const RT_RS_3DVector& anrm,
                     vertex_type avtx_type, RT_RS_Scene* ascene)
{
#ifdef RS_DEBUG
  if(anrm.vabs() < epsilon)
    rt_Output->fatalVar(get_class(), ": Normal is zero.", NULL);
#endif
  vtx_type = avtx_type;
  scene = ascene;
  pnt = apnt;
  nrm = anrm.uni();
  ar = 0.;
}

float RT_RS_Vtx::comp_ctheta(RT_RS_3DVector& _pnt)
{
  RT_RS_3DVector udir(_pnt - pnt);
  float len = udir.vabs();
  if(len > epsilon4)
    udir = udir.uni();
  else
    return 0.;           // points are the same
  return nrm.dot(udir);
}


void RT_RS_Vtx::print(FILE *f, char *n, int width, int decimalPlaces)
{
  if(n) fprintf(f, "%s\n", n);
  sprintf(fmt,"area: %s%d.%df vertex_type: %c\n", "%", width, decimalPlaces,
          vtx_type);
   fprintf(f, fmt, ar);
  pnt.print(f, "point:", width, decimalPlaces);
  nrm.print(f, "normal:", width, decimalPlaces);
}

//////////////////////  THE RT_RS_DiffRGB MEMBER FUNCTIONS /////////////////////////
boolean RT_RS_DiffRGB::read(char *filename, FILE *fp)
{
  boolean aln = fp == NULL;
  boolean Ok = TRUE;
  int clmn = 0;

  if(aln) Ok = openfile(filename, fp, "rt");
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) Ok = isclass(filename, RS_s1);
  if(Ok) Ok = getfloat(filename, fp, r, RS_s, clmn);
  if(Ok) Ok = getfloat(filename, fp, g, RS_s, clmn);
  if(Ok) Ok = getfloat(filename, fp, b, RS_s, clmn);

  if(aln) fclose(fp);
  return Ok;
}

boolean RT_RS_DiffRGB::write(char *filename, FILE *fp)
{
  boolean Ok = TRUE;
  boolean aln = fp == NULL;

  if(aln) Ok = openfile(filename, fp, "wt");
  if(Ok) {
     fprintf(fp,"%s\n", get_class());
    Ok = !ferror(fp);
  }
  if(Ok) {
    fprintf(fp, "% .6f % .6f % .6f\n", r, g, b);
    Ok = !ferror(fp);
  }
  if(!Ok) {
    rt_Output->errorVar(get_class(), ":Cannot write ", filename, NULL);
    return FALSE;
  }
  if(aln) fclose(fp);
  return TRUE;
}

void RT_RS_DiffRGB::print(FILE *f, char *n, int width, int decimalPlaces)
{
  if(n) fprintf(f, "%s\n", n);
  sprintf(fmt,"red %s%d.%df\n", "%", width,decimalPlaces);
  fprintf(f, fmt, r);
  sprintf(fmt,"green %s%d.%df\n", "%", width,decimalPlaces);
  fprintf(f, fmt, g);
  sprintf(fmt,"blue %s%d.%df\n", "%", width,decimalPlaces);
  fprintf(f, fmt, b);
}

boolean RT_RS_Vtx::read(char *filename, FILE *fp) {
  boolean Ok = TRUE;
  boolean aln = fp == NULL;
  int vt, clmn = 0;

  if(aln) Ok = openfile(filename, fp, "rt");
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) Ok = isclass(filename, RS_s1);
  if(Ok) Ok = getfloat(filename, fp, ar, RS_s, clmn);
  if(Ok) Ok = getint(filename, fp, vt, RS_s, clmn);
  vtx_type = (vertex_type)vt;
  if(Ok) Ok = pnt.read(filename, fp);
  if(Ok) Ok = nrm.read(filename, fp);
  if(aln) fclose(fp);
  return Ok;
}

boolean RT_RS_Vtx::write(char *filename, FILE *fp) {
  boolean Ok = TRUE;
  boolean aln = fp == NULL;

  if(aln) Ok = openfile(filename, fp, "wt");
  if(Ok) {
     fprintf(fp,"%s\n% .6f %d\n", get_class(), ar, (int)vtx_type);
    Ok = !ferror(fp);
  }
  if(Ok) Ok = pnt.write(filename,fp);
  if(Ok) Ok = nrm.write(filename,fp);
  if(!Ok) {
    rt_Output->errorVar(get_class(), ":Cannot write ", filename, NULL);
    return FALSE;
  }
  if(aln) fclose(fp);
  return TRUE;
}

//////////////////////  THE RT_RS_HyDiffVtx MEMBER FUNCTIONS /////////////////////////
boolean RT_RS_HyDiffVtx::write(char *filename, FILE *fp) {
  boolean Ok = TRUE;
  boolean aln = fp == NULL;

  if(aln) Ok = openfile(filename, fp, "wt");
  if(Ok) {
     fprintf(fp,"%s\n", get_class());
    Ok = !ferror(fp);
  }
  if(Ok) Ok = RT_RS_Vtx::write(filename,fp);
  if(Ok) Ok = d_ush_ins.write(filename,fp);
  if(!Ok) {
    rt_Output->errorVar(get_class(), ":Cannot write ", filename, NULL);
    return FALSE;
  }
  if(aln) fclose(fp);
  return TRUE;
}

boolean RT_RS_HyDiffVtx::read(char *filename, FILE *fp) {
  boolean aln = fp == NULL;
  boolean Ok = TRUE;
  int clmn = 0;

  if(aln) Ok = openfile(filename, fp, "rt");
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) Ok = isclass(filename, RS_s1);
  if(Ok) Ok = RT_RS_Vtx::read(filename,fp);
  if(Ok) Ok = d_ush_ins.read(filename,fp);

  if(aln) fclose(fp);
  return Ok;
}

void RT_RS_HyDiffVtx::print(FILE *f, char *n, int width, int decimalPlaces) {
  if(n) fprintf(f, "%s\n", n);
  if(scene == NULL) fprintf(f, "scene = NULL\n");
  RT_RS_Vtx::print(f, NULL, width, decimalPlaces);
  d_ush_ins.print(f, "unshoot diffuse intensity", width, decimalPlaces);
}


//////////////////////  THE RT_RS_DiffVtx MEMBER FUNCTIONS /////////////////////////

boolean RT_RS_DiffVtx::get_view_val(RT_RS_DiffRGB& val, RT_RS_Vtx& view_dir, boolean) {
  if(vis != unknown) {
    val = view_val; return vis == visible;   //compute only once
  }
  view_val = d_acc_ins;
  val = view_val;
  if((pnt - view_dir.pnt).dot(view_dir.nrm) < 0) {
    vis = hide; return FALSE; //vertex is behind the point of view
  }
  if(comp_ctheta(view_dir.pnt) < epsilon) {  
    vis = hide; return FALSE; //vertex is back-facing
  } 
  vis = visible;
  
  return TRUE;
}

boolean RT_RS_DiffVtx::write(char *filename, FILE *fp) {
  boolean Ok = TRUE;
  boolean aln = fp == NULL;

  if(aln) Ok = openfile(filename, fp, "wt");
  if(Ok) {
     fprintf(fp,"%s\n", get_class());
    Ok = !ferror(fp);
  }
  if(Ok) Ok = RT_RS_Vtx::write(filename,fp);
  if(Ok) Ok = d_p.write(filename,fp);
  if(Ok) Ok = d_acc_ins.write(filename,fp);
  if(Ok) Ok = d_ush_ins.write(filename,fp);
  if(!Ok) {
    rt_Output->errorVar(get_class(), ":Cannot write ", filename, NULL);
    return FALSE;
  }
  if(aln) fclose(fp);
  return TRUE;
}

boolean RT_RS_DiffVtx::read(char *filename, FILE *fp) {
  boolean aln = fp == NULL;
  boolean Ok = TRUE;
  int clmn = 0;

  if(aln) Ok = openfile(filename, fp, "rt");
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) Ok = isclass(filename, RS_s1);
  if(Ok) Ok = RT_RS_Vtx::read(filename,fp);
  if(Ok) Ok = d_p.read(filename,fp);
  if(Ok) Ok = d_acc_ins.read(filename,fp);
  if(Ok) Ok = d_ush_ins.read(filename,fp);

  if(aln) fclose(fp);
  return Ok;
}

void RT_RS_DiffVtx::print(FILE *f, char *n, int width, int decimalPlaces) {
  if(n) fprintf(f, "%s\n", n);
  if(scene == NULL) fprintf(f, "scene = NULL\n");
  RT_RS_Vtx::print(f, NULL, width, decimalPlaces);
  d_p.print(f, "diffuse reflectivity", width, decimalPlaces);
  d_acc_ins.print(f, "accumulated diffuse intensity", width, decimalPlaces);
  d_ush_ins.print(f, "unshoot diffuse intensity", width, decimalPlaces);
}

//////////////////////  THE RT_RS_brdfRGB MEMBER FUNCTIONS /////////////////////////
boolean RT_RS_brdfRGB::read(char *filename, FILE *fp)
{
  boolean aln = fp == NULL;
  boolean Ok = TRUE;
  int clmn = 0;

  if(aln) Ok = openfile(filename, fp, "rt");
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) Ok = isclass(filename, RS_s1);
  if(Ok) Ok = getint(filename, fp, r, RS_s, clmn);
  if(Ok) Ok = getint(filename, fp, g, RS_s, clmn);
  if(Ok) Ok = getint(filename, fp, b, RS_s, clmn);

  if(aln) fclose(fp);
  return Ok;
}

boolean RT_RS_brdfRGB::write(char *filename, FILE *fp)
{
  boolean Ok = TRUE;
  boolean aln = fp == NULL;

  if(aln) Ok = openfile(filename, fp, "wt");
  if(Ok) {
    fprintf(fp,"%s\n", get_class());
    Ok = !ferror(fp);
  }
  if(Ok) {
    fprintf(fp, "%d %d %d\n", r, g, b);
    Ok = !ferror(fp);
  }
  if(!Ok) {
    rt_Output->errorVar(get_class(), ":Cannot write ", filename, NULL);
    return FALSE;
  }
  if(aln) fclose(fp);
  return TRUE;
}

void RT_RS_brdfRGB::print(FILE *f, char *n, int, int) {
  if(n) fprintf(f, "%s\n", n);
  fprintf(f, "red index: %d   green index: %d   blue index: %d", r, g, b);
}

//////////////////////  THE RT_RS_InsRGB MEMBER FUNCTIONS /////////////////////////
void RT_RS_InsRGB::alloch(int r_nr, int g_nr, int b_nr)
{
  r = new RT_RS_InsMatrix(r_nr);
  g = new RT_RS_InsMatrix(g_nr);
  b = new RT_RS_InsMatrix(b_nr);
}

void RT_RS_InsRGB::freeh()
{
  if(r != NULL) { delete r; r = NULL; }
  if(g != NULL) { delete g; g = NULL; }
  if(b != NULL) { delete b; b = NULL; }
}


RT_RS_InsRGB::RT_RS_InsRGB(RT_RS_InsMatrix &ar, RT_RS_InsMatrix &ag,
                           RT_RS_InsMatrix &ab)
{
  r = new RT_RS_InsMatrix(ar);
  g = new RT_RS_InsMatrix(ag);
  b = new RT_RS_InsMatrix(ab);
}

RT_RS_InsRGB::RT_RS_InsRGB(const RT_RS_InsRGB& v)
{
  r = new RT_RS_InsMatrix(*v.r);
  g = new RT_RS_InsMatrix(*v.g);
  b = new RT_RS_InsMatrix(*v.b);
}

RT_RS_DiffRGB RT_RS_InsRGB::eval(float theta_out, float phi_out)
{
  RT_RS_DiffRGB I(r->eval(theta_out, phi_out),
                  g->eval(theta_out, phi_out),
                  b->eval(theta_out, phi_out));
  return I;
}


boolean RT_RS_InsRGB::read(char *filename, FILE *fp)
{
  boolean aln = fp == NULL;
  boolean Ok = TRUE;
  int clmn = 0;

  if(aln) Ok = openfile(filename, fp, "rt");
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) Ok = isclass(filename, RS_s1);
  if(Ok) Ok = r->read(filename, fp);
  if(Ok) Ok = g->read(filename, fp);
  if(Ok) Ok = b->read(filename, fp);

  if(aln) fclose(fp);
  return Ok;
}

RT_RS_InsRGB RT_RS_InsRGB::operator = (const RT_RS_InsRGB& v) {
  *r = *v.r; *g = *v.g; *b = *v.b; return *this;
}

RT_RS_InsRGB RT_RS_InsRGB::operator + (const RT_RS_InsRGB& v) {
  RT_RS_InsMatrix rr = *r + *v.r;
  RT_RS_InsMatrix gg = *g + *v.g;
  RT_RS_InsMatrix bb = *b + *v.b;
  RT_RS_InsRGB c( rr, gg, bb ); return c;
}

RT_RS_InsRGB RT_RS_InsRGB::operator - (const RT_RS_InsRGB& v) {
  RT_RS_InsMatrix rr = *r - *v.r;
  RT_RS_InsMatrix gg = *g - *v.g;
  RT_RS_InsMatrix bb = *b - *v.b;
  RT_RS_InsRGB c( rr, gg, bb ); return c;
}

RT_RS_InsRGB RT_RS_InsRGB::operator * (float f) {
  RT_RS_InsMatrix rr = *r * f;
  RT_RS_InsMatrix gg = *g * f;
  RT_RS_InsMatrix bb = *b * f;
  RT_RS_InsRGB c( rr, gg, bb ); return c;
}

RT_RS_InsRGB RT_RS_InsRGB::operator * (const RT_RS_DiffRGB& v) {
  RT_RS_InsMatrix rr = *r * v.r;
  RT_RS_InsMatrix gg = *g * v.g;
  RT_RS_InsMatrix bb = *b * v.b;
  RT_RS_InsRGB c( rr, gg, bb ); return c;
}

boolean RT_RS_InsRGB::write(char *filename, FILE *fp) {
  boolean Ok = TRUE;
  boolean aln = fp == NULL;

  if(aln) Ok = openfile(filename, fp, "wt");
  if(Ok) {
     fprintf(fp,"%s\n", get_class());
    Ok = !ferror(fp);
  }
  if(Ok) Ok = r->write(filename,fp);
  if(Ok) Ok = g->write(filename,fp);
  if(Ok) Ok = b->write(filename,fp);
  if(!Ok) {
    rt_Output->errorVar(get_class(), ":Cannot write ", filename, NULL);
    return FALSE;
  }
  if(aln) fclose(fp);
  return TRUE;
}

void RT_RS_InsRGB::print(FILE *f, char *n, int width, int decimalPlaces)
{
  if(n) fprintf(f, "%s\n", n);
  sprintf(fmt,"red %s%d.%df\n", "%", width, decimalPlaces);
  fprintf(f, fmt, r->avg);
  sprintf(fmt,"green %s%d.%df\n", "%", width, decimalPlaces);
  fprintf(f, fmt, g->avg);
  sprintf(fmt,"blue %s%d.%df\n", "%", width, decimalPlaces);
  fprintf(f, fmt, b->avg);
}


//////////////////////  THE RT_RS_HyInsVtx MEMBER FUNCTIONS /////////////////////////

boolean RT_RS_HyInsVtx::read(char *filename, FILE *fp)
{
  boolean aln = fp == NULL;
  boolean Ok = TRUE;
  int clmn = 0;

  if(scene == NULL) {
    rt_Output->errorVar(get_class(), ":scene is NULL", NULL);
    Ok = FALSE;
  }
  if(aln && Ok) Ok = openfile(filename, fp, "rt");
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) Ok = isclass(filename, RS_s1);

  if(Ok) Ok = RT_RS_HyDiffVtx::read(filename, fp);
  if(Ok) Ok = tgn.read(filename,fp);

  if(Ok) {
    freeh();
    alloch();
    Ok = i_ush_ins.read(filename,fp);
  }

  if(aln) fclose(fp);
  return Ok;
}

boolean RT_RS_HyInsVtx::write(char *filename, FILE *fp)
{
  boolean Ok = TRUE;
  boolean aln = fp == NULL;

  if(scene == NULL) {
    rt_Output->errorVar(get_class(), ":scene is NULL", NULL);
    Ok = FALSE;
  }
  if(aln && Ok) Ok = openfile(filename, fp, "wt");
  if(Ok) {
     fprintf(fp,"%s\n", get_class());
    Ok = !ferror(fp);
  }
  if(Ok) Ok = RT_RS_HyDiffVtx::write(filename,fp);
  if(Ok) Ok = tgn.write(filename,fp);
  if(Ok) Ok = i_ush_ins.write(filename,fp);

  if(!Ok) {
    rt_Output->errorVar(get_class(), ":Cannot write ", filename, NULL);
    return FALSE;
  }
  if(aln) fclose(fp);
  return TRUE;
}

void RT_RS_HyInsVtx::print(FILE *f, char *n, int width, int decimalPlaces)
{
  if(n) fprintf(f, "%s\n", n);
  if(scene == NULL) fprintf(f, "scene = NULL\n");
  RT_RS_Vtx::print(f, NULL, width, decimalPlaces);
  tgn.print(f, "tanget", width, decimalPlaces);
  i_ush_ins.print(f, "unshoot non-diffuse average intensity");
}

//////////////////////  THE RT_RS_InsVtx MEMBER FUNCTIONS /////////////////////////
void RT_RS_InsVtx::re_alloch()
{
  RT_RS_BRDFMatrix *r, *g, *b;

  get_brdf(r, g, b);
  int r_nr = r->get_rows();
  int g_nr = g->get_rows();
  int b_nr = b->get_rows();
  i_acc_ins.re_alloch(r_nr, g_nr, b_nr);
  i_ush_ins.re_alloch(r_nr, g_nr, b_nr);
}


RT_RS_InsVtx::RT_RS_InsVtx(RT_RS_3DVector& apnt, RT_RS_3DVector& anrm,
  RT_RS_3DVector& atgn, RT_RS_DiffRGB& ad_p, RT_RS_DiffRGB& ai_p,
  RT_RS_brdfRGB& abrdf, vertex_type avtx_type, RT_RS_Scene* aScene)
  : RT_RS_DiffVtx(apnt, anrm, ad_p, avtx_type, aScene), i_p(ai_p), brdf(abrdf)
{
  static RT_RS_3DVector vx(1.,0.,0.);
  static RT_RS_3DVector vy(0.,1.,0.);

  if(tgn.vabs() > epsilon)
    tgn = atgn;
  else {
    //compute a tanget
    tgn = nrm.cross(vx);
    if(tgn.vabs() < epsilon5)
      tgn = nrm.cross(vy);
  }
  tgn = tgn.uni();
#ifdef RS_DEBUG
  if(fabs(nrm.dot(tgn)) > epsilon5)
    rt_Output->fatalVar(get_class(), ":Tangent is not perpendicular to normal", NULL);
#endif
  if(scene != NULL)
    re_alloch();
  else
    alloch();
}

float RT_RS_InsVtx::comp_phi(RT_RS_3DVector& _pnt)
{
  RT_RS_3DVector udir(_pnt - pnt);
  float len = udir.vabs();
  if(len > epsilon5)
    udir = udir.uni();
  else
    return .0;              // points are the same
  RT_RS_3DVector v = nrm * nrm.dot(udir);
  udir = (udir - v);
  len = udir.vabs();
  if(len > epsilon5)
    udir = udir * (1./len);    //(uni);
  else
    return .0;              // if theta == 0, phi is not defined
  float cos_phi = tgn.dot(udir);
  if(cos_phi > 1.) cos_phi = 1.;
  if(cos_phi < -1.) cos_phi = -1.;
  v = tgn.cross(udir);
  len = v.vabs();
  if(len > epsilon5)
    v = v * (1./len);          //(uni);
  else
    return acos(cos_phi);      // phi is 0 or Pi
  return (nrm.dot(v) < 0) ? M_2PI - acos(cos_phi) : acos(cos_phi);
}

void RT_RS_InsVtx::get_brdf(RT_RS_BRDFMatrix*& r, RT_RS_BRDFMatrix*& g,
                            RT_RS_BRDFMatrix*& b)
{
  if(scene == NULL)
    r = g = b = NULL;
  r = scene->BRDFs->get_at(brdf.r);
  g = scene->BRDFs->get_at(brdf.g);
  b = scene->BRDFs->get_at(brdf.b);
}

RT_RS_InsRGB RT_RS_InsVtx::comp_dI(float theta_in, float phi_in,
                                   RT_RS_DiffRGB& PHI)
{
  RT_RS_BRDFMatrix *r, *g, *b;
  get_brdf(r, g, b);
  RT_RS_InsMatrix rr = r->eval(theta_in, phi_in, PHI.r);
  RT_RS_InsMatrix gg = g->eval(theta_in, phi_in, PHI.g);
  RT_RS_InsMatrix bb = b->eval(theta_in, phi_in, PHI.b);
  RT_RS_InsRGB dI( rr, gg, bb );
  dI.r->temp = dI.g->temp = dI.b->temp = 'n';
  return dI;
}

boolean RT_RS_InsVtx::get_view_val(RT_RS_DiffRGB& val, RT_RS_Vtx& view_dir, boolean eval_ins)
{
  if(vis == unknown) {  //compute only once
    vis = hide;
    view_val = d_acc_ins;  //diffuse component
    if((pnt - view_dir.pnt).dot(view_dir.nrm) > epsilon) { //vertex behind point of view ?
      float ctheta = comp_ctheta(view_dir.pnt);
      if(ctheta > epsilon) {  //back-facing ?
        if(eval_ins) {  //evaluate the non-diffuse part
          float phi = comp_phi(view_dir.pnt);
          view_val = view_val + i_acc_ins.eval(acos(ctheta), phi) * (1. / ctheta);
        }
        vis = visible;
      }
    }
  }
  val = view_val; return vis == visible;   //compute only once
}

boolean RT_RS_InsVtx::read(char *filename, FILE *fp)
{
  boolean aln = fp == NULL;
  boolean Ok = TRUE;
  int clmn = 0;
  RT_RS_BRDFMatrix *r, *g, *b;

  if(scene == NULL) {
    rt_Output->errorVar(get_class(), ":scene is NULL", NULL);
    Ok = FALSE;
  }
  if(aln && Ok) Ok = openfile(filename, fp, "rt");
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) Ok = isclass(filename, RS_s1);

  if(Ok) Ok = RT_RS_DiffVtx::read(filename, fp);
  if(Ok) Ok = tgn.read(filename,fp);
  if(Ok) Ok = i_p.read(filename,fp);
  if(Ok) Ok = brdf.read(filename,fp);
  get_brdf(r, g, b);

  clmn = 0;
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) {
    if(strcmp(r->get_name(), RS_s1)) {
      sprintf(RS_s, ":Red BRDF %s from file %s not found at index %d!",
              RS_s1, filename, brdf.r);
      rt_Output->errorVar(get_class(), RS_s, NULL);
      Ok = FALSE;
    }
  }
  clmn = 0;
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) {
    if(strcmp(g->get_name(), RS_s1)) {
      sprintf(RS_s, ":Green BRDF %s from file %s not found at index %d!",
              RS_s1, filename, brdf.g);
      rt_Output->errorVar(get_class(), RS_s, NULL);
      Ok = FALSE;
    }
  }
  clmn = 0;
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) {
    if(strcmp(b->get_name(), RS_s1)) {
      sprintf(RS_s, ":Blue BRDF %s from file %s not found at index %d!",
              RS_s1, filename, brdf.b);
      rt_Output->errorVar(get_class(), RS_s, NULL);
      Ok = FALSE;
    }
  }

  if(Ok) {
    freeh();
    alloch();
    Ok = i_acc_ins.read(filename,fp);
  }
  if(Ok) Ok = i_ush_ins.read(filename,fp);

  if(aln) fclose(fp);
  return Ok;
}

boolean RT_RS_InsVtx::write(char *filename, FILE *fp)
{
  boolean Ok = TRUE;
  boolean aln = fp == NULL;
  RT_RS_BRDFMatrix *r, *g, *b;

  if(scene == NULL) {
    rt_Output->errorVar(get_class(), ":scene is NULL", NULL);
    Ok = FALSE;
  }
  if(aln && Ok) Ok = openfile(filename, fp, "wt");
  if(Ok) {
     fprintf(fp,"%s\n", get_class());
    Ok = !ferror(fp);
  }
  if(Ok) Ok = RT_RS_DiffVtx::write(filename,fp);
  if(Ok) Ok = tgn.write(filename,fp);
  if(Ok) Ok = i_p.write(filename,fp);
  if(Ok) Ok = brdf.write(filename,fp);
  if(Ok) {
    get_brdf(r, g, b);
     fprintf(fp,"%s\n%s\n%s\n", r->get_name(), g->get_name(), b->get_name());
    Ok = !ferror(fp);
  }
  if(Ok) Ok = i_acc_ins.write(filename,fp);
  if(Ok) Ok = i_ush_ins.write(filename,fp);

  if(!Ok) {
    rt_Output->errorVar(get_class(), ":Cannot write ", filename, NULL);
    return FALSE;
  }
  if(aln) fclose(fp);
  return TRUE;
}

void RT_RS_InsVtx::print(FILE *f, char *n, int width, int decimalPlaces)
{
  if(n) fprintf(f, "%s\n", n);
  if(scene == NULL) fprintf(f, "scene = NULL\n");
  RT_RS_Vtx::print(f, NULL, width, decimalPlaces);
  i_p.print(f, "i_p");
  brdf.print(f, "brdf");
  tgn.print(f, "tanget", width, decimalPlaces);
  i_acc_ins.print(f, "accumulated non-diffuse average intensity");
  i_ush_ins.print(f, "unshoot non-diffuse average intensity");
}

////////////////////// THE RT_RS_Voxel MEMBER FUNCTIONS /////////////////////////
void RT_RS_Voxel::freeh() {
    if (next != NULL) {
        next->freeh();
  delete next;
  next = NULL;
    }
}

RT_RS_Voxel::RT_RS_Voxel(long* avidx)
{
  for(int i = 0; i < 4; i++)
    vidx[i] = avidx[i];
  next = NULL;
}

void RT_RS_Voxel::append(long* avidx) {
  if(avidx[0] == vidx[0] && avidx[1] == vidx[1]  &&
     avidx[2] == vidx[2] && avidx[3] == vidx[3]) return;
  if(next != NULL)
    next->append(avidx);
  else
    next = new RT_RS_Voxel(avidx);
}

void RT_RS_Voxel::print(FILE *f, char *n, int, int) {
  static long n_cnt = 0;
  static long top = 0;

  if(next) {
    top--;
    next->print(f);
    n_cnt++;
    top++;
  }
  else
    n_cnt = 1;

  if(top == 0) {
    if(n) fprintf(f, "%s\n", n);
    fprintf(f, "Contains %ld object(s).\n", n_cnt);
  }
}

////////////////////// THE RT_RS_SpatPar MEMBER FUNCTIONS /////////////////////////
RT_RS_SpatPar::RT_RS_SpatPar(int adim)
{
  int x, y, z;

#ifdef RS_DEBUG
  if(adim < 0)
    rt_Output->fatalVar(get_class(), ":Dimension < 0!", NULL);
#endif
  dim = adim;
  Space = new RT_RS_Voxel*** [dim];  //allocate first dimension of the 3D-array
  for(x = 0; x < dim; x++)
    Space[x] = new RT_RS_Voxel** [dim];  //allocate second dimension
  for(x = 0; x < dim; x++)
    for(y = 0; y < dim; y++)
      Space[x][y] = new RT_RS_Voxel* [dim]; //allocate third dimension
  for(x = 0; x < dim; x++)
    for(y = 0; y < dim; y++)
      for(z = 0; z < dim; z++)
        Space[x][y][z] = NULL;     //initialize the pointers
}

RT_RS_SpatPar::~RT_RS_SpatPar()
{
  int x, y ,z;

  for(x = 0; x < dim; x++)
    for(y = 0; y < dim; y++)
      for(z = 0; z < dim; z++)
        if(Space[x][y][z] != NULL)
          delete Space[x][y][z]; //delete Voxel-Lists
  for(x = 0; x < dim; x++)
    for(y = 0; y < dim; y++)
      delete Space[x][y];        //delete third dimension of the 3D-array
  for(x = 0; x < dim; x++)
    delete Space[x];             //delete second dimension of the 3D-array
  delete Space;                  //delete first dimension
}

double RT_RS_SpatPar::get_volume()
{
  double vol = (bnd_p2.x - bnd_p1.x) *
               (bnd_p2.y - bnd_p1.y) *
               (bnd_p2.z - bnd_p1.z);
  return vol;
}

RT_RS_Voxel* RT_RS_SpatPar::get(int* vox)
{
  return Space[vox[0]][vox[1]][vox[2]];
}

void RT_RS_SpatPar::get_voxidx(const RT_RS_3DVector& v, int* vox)
{
  RT_RS_3DVector vx(v - bnd_p1);
  vox[0] = (int)floor(vx.x * rd_bnd.x);
  vox[1] = (int)floor(vx.y * rd_bnd.y);
  vox[2] = (int)floor(vx.z * rd_bnd.z);

#ifdef RS_DEBUG
  if(vox[0] < 0 || vox[1] < 0 || vox[2] < 0)
    rt_Output->fatalVar(get_class(), ":Index < 0 in function get_voxidx()!", NULL);
  if(vox[0] >= dim || vox[1] >= dim || vox[2] >= dim) {
    rt_Output->fatalVar(get_class(), ":Index >= Dimension in function get_voxidx()!", NULL);
  }
#endif
}

void RT_RS_SpatPar::insert(long* vidx, RT_RS_3DVector* pnt)
{
  int i, x, y, z, max_x, max_y, max_z, min_x, min_y, min_z;
  int vox[24][3];
  RT_RS_3DVector dp;

  int j = 0;
  for(i = 0; i < 4; i++) {
    dp.x =  epsilon1; get_voxidx(pnt[i] + dp, vox[j]); j++;
    dp.x = -epsilon1; get_voxidx(pnt[i] + dp, vox[j]); j++;
    dp.x = 0.;
    dp.y =  epsilon1; get_voxidx(pnt[i] + dp, vox[j]); j++;
    dp.y = -epsilon1; get_voxidx(pnt[i] + dp, vox[j]); j++;
    dp.y = 0.;
    dp.z =  epsilon1; get_voxidx(pnt[i] + dp, vox[j]); j++;
    dp.z = -epsilon1; get_voxidx(pnt[i] + dp, vox[j]); j++;
    dp.z = 0.;
  }
  min_x = min_y = min_z = MAXINT;
  max_x = max_y = max_z = -MAXINT;
  for(i = 0; i < 24; i++) {
    if(vox[i][0] > max_x) max_x = (int)vox[i][0];
    if(vox[i][1] > max_y) max_y = (int)vox[i][1];
    if(vox[i][2] > max_z) max_z = (int)vox[i][2];
    if(vox[i][0] < min_x) min_x = (int)vox[i][0];
    if(vox[i][1] < min_y) min_y = (int)vox[i][1];
    if(vox[i][2] < min_z) min_z = (int)vox[i][2];
  }

  for(x = min_x; x <= max_x; x++)
    for(y = min_y; y <= max_y; y++)
      for(z = min_z; z <= max_z; z++)
        if(Space[x][y][z] != NULL)
          Space[x][y][z]->append(vidx);
        else
          Space[x][y][z] = new RT_RS_Voxel(vidx);
}

void RT_RS_SpatPar::print(FILE *f, char *n, int, int) {
  int x, y, z;
  long ne_cnt = 0;

  if(n) fprintf(f, "%s\n", n);
  for(x = 0; x < dim; x++)
    for(y = 0; y < dim; y++)
      for(z = 0; z < dim; z++)
        if(Space[x][y][z] != NULL) {
          ne_cnt++;
          sprintf(fmt, "Volume element x:%d y:%d z:%d ", x, y, z);
          Space[x][y][z]->print(f, fmt);
        }
  long n_cnt = x * y * z;
  sprintf(RS_s, "%ld total volume elements, %ld are not empty.\n", n_cnt, ne_cnt);
  fprintf(f, RS_s);
}

///////////////// The RT_RS_Points member functions  /////////////////////////
void RT_RS_Points::dallc()
{
  if(dspns >= 1024) 
    rt_Output->fatalVar(get_class(), ":Too many points in dallc()", NULL);
  dPnts[dspns++] = new RT_RS_DiffVtx* [1024];
}

void RT_RS_Points::Historydallc()
{
  if(Historydspns >= 1024)
    rt_Output->fatalVar(get_class(), ":Too many History points in Historydallc()", NULL);
  HistorydPnts[Historydspns++] = new RT_RS_HyDiffVtx* [1024];
}

void RT_RS_Points::iallc()
{
  if(ispns >= 1024)
    rt_Output->fatalVar(get_class(), ":Too many points in iallc()", NULL);
  iPnts[ispns++] = new RT_RS_InsVtx* [1024];
}

void RT_RS_Points::Historyiallc()
{
  if(Historyispns >= 1024)
    rt_Output->fatalVar(get_class(), ":Too many History points in Historyiallc()", NULL);
  HistoryiPnts[Historyispns++] = new RT_RS_HyInsVtx* [1024];
}

void RT_RS_Points::freeh()
{
  int spns;
  long cnt;

  for(cnt = 0; cnt < dcnt; cnt++)  delete get_diff(cnt);
  for(spns = 0; spns < dspns; spns++) delete dPnts[spns];

  for(cnt = 0; cnt < Historydcnt; cnt++)  delete get_Hydiff(cnt);
  for(spns = 0; spns < Historydspns; spns++) delete  HistorydPnts[spns];

  for(cnt = 0; cnt < icnt; cnt++)  delete get_ins(cnt);
  for(spns = 0; spns < ispns; spns++) delete iPnts[spns];

  for(cnt = 0; cnt < Historyicnt; cnt++)  delete get_Hyins(cnt);
  for(spns = 0; spns < Historyispns; spns++) delete  HistoryiPnts[spns];

  dcnt = icnt = Historydcnt = Historyicnt = 0; 
  dspns = ispns = Historydspns = Historyispns = 0; 
  delete SpatPar; SpatPar = NULL;
}

RT_RS_Points::RT_RS_Points(RT_RS_Scene* ascene)
{
  scene = ascene;
  dcnt = icnt = Historydcnt = Historyicnt = 0; 
  dspns = ispns = Historydspns = Historyispns = shoots = 0;
  cur_eval_ins = fast = TRUE;
  fast_max_area = 0;   //skip
  max_unshoot = RS_MAX_UNSHOOT;
  fast_max_unshoot = RS_FAST_MAX_UNSHOOT;
  max_scale = view_scale = 0;  //auto mode
  spar_dim = RS_SPPAR_DIM;
  nr_shoots = RS_NR_SHOOTS;
  SpatPar = NULL;
  cur_src_dvtx = NULL; cur_src_ivtx = NULL;
#ifdef RS_SHOW
  cur_src_didx = cur_src_iidx = -1;
#endif
}

void RT_RS_Points::insert(RT_RS_DiffVtx* a, RT_RS_DiffRGB* a_diff)
{
  a->set_scene(scene);
  if(a_diff != NULL)
    a->d_acc_ins = a->d_ush_ins = *a_diff;

  int sn = (int)(dcnt >> 10);
  if(sn >= dspns) dallc();
  dPnts[sn][(int)(dcnt - (sn << 10))] = a;
  dcnt++;
}

void RT_RS_Points::insertHy(RT_RS_HyDiffVtx* a)
{
  a->set_scene(scene);

  int sn = (int)(Historydcnt >> 10);
  if(sn >= Historydspns) Historydallc();
  HistorydPnts[sn][(int)(Historydcnt - (sn << 10))] = a;
  Historydcnt++;
}

void RT_RS_Points::insert(RT_RS_InsVtx* a, RT_RS_brdfRGB* abrdf,
                          RT_RS_DiffRGB* a_diff, RT_RS_DiffRGB* a_ins)
{
  a->set_scene(scene);
  if(abrdf != NULL) {
    a->brdf = *abrdf;
    a->re_alloch();

    if(a_diff != NULL)
      a->d_acc_ins = a->d_ush_ins = *a_diff;

    if(a_ins != NULL)        //start value
      a->i_acc_ins = a->i_ush_ins = a->comp_dI(0., 0., *a_ins);
  }
  int sn = (int)(icnt >> 10);
  if(sn >= ispns) iallc();
  iPnts[sn][(int)(icnt - (sn << 10))] = a;
  icnt++;
}


void RT_RS_Points::insertHy(RT_RS_HyInsVtx* a)
{
  a->set_scene(scene);

  int sn = (int)(Historyicnt >> 10);
  if(sn >= Historyispns) Historyiallc();
  HistoryiPnts[sn][(int)(Historyicnt - (sn << 10))] = a;
  Historyicnt++;
}

void RT_RS_Points::insert_ip(int n, long* vtxidx,
                             RT_RS_DiffVtx*& new_vtx, long& new_vtxidx)
{
  int i;
  long l;
  RT_RS_DiffRGB d_acc_ins, d_ush_ins;
  float recip_n = 1. / (float)n;

  RT_RS_DiffVtx *vtx[4], *cur_dtemp, src_dvtx;
  RT_RS_InsVtx *cur_itemp;
  RT_RS_HyDiffVtx* HyDiffVtx;
  RT_RS_HyInsVtx* HyInsVtx;
  RT_RS_3DVector new_pnt, new_nrm;

  for(i = 0; i < n; i++) {
    vtx[i] = get_diff(vtxidx[i]);
    new_pnt = new_pnt + vtx[i]->pnt;
    new_nrm = new_nrm + vtx[i]->nrm;
  }
  new_pnt = new_pnt * recip_n; 
  new_nrm = new_nrm * recip_n;

  new_vtxidx = get_diffcnt();
  if (rt_RSYMsg) {
      sprintf(RS_s, "Creating a new diffuse vertex nr.: %ld\n at: x = %f y = %f z = %f ...",
	      new_vtxidx, new_pnt.x, new_pnt.y, new_pnt.z);
      rt_Output->message(RS_s);
  }
  new_vtx = new RT_RS_DiffVtx(new_pnt, new_nrm, vtx[0]->d_p);
  insert(new_vtx);
  
  if(cur_src_dvtx != NULL || cur_src_ivtx != NULL) {
    if(Historydcnt > 0) {
      if (rt_RSYMsg) {
	  sprintf(RS_s, "Repeating %ld diffuse shooting operations for the new vertex ...", 
		  Historydcnt);
	  rt_Output->message(RS_s);
      }
  }
    cur_dtemp = cur_src_dvtx;
    cur_src_dvtx = &src_dvtx;
    for(l = 0; l < Historydcnt; l++) {
      getHy(l, HyDiffVtx);
      src_dvtx.pnt = HyDiffVtx->pnt; src_dvtx.nrm = HyDiffVtx->nrm; 
      src_dvtx.set_ar(HyDiffVtx->get_ar());
      src_dvtx.d_ush_ins = HyDiffVtx->d_ush_ins;
      shoot_diff(new_vtxidx, new_vtxidx, 0, -1);
    }
    cur_src_dvtx = cur_dtemp;

    if (rt_RSYMsg) {
	if(Historyicnt > 0) {
	    sprintf(RS_s, "Repeating %ld non-diffuse shooting operations for the new vertex ...", 
		    Historyicnt);
	    rt_Output->message(RS_s);
	}
    }
    cur_itemp = cur_src_ivtx;
    for(l = 0; l < Historyicnt; l++) {
      getHy(l, HyInsVtx);
      cur_src_ivtx = new RT_RS_InsVtx(HyInsVtx->pnt, HyInsVtx->nrm, 
				      HyInsVtx->tgn); 
      cur_src_ivtx->set_ar(HyInsVtx->get_ar());
      cur_src_ivtx->d_ush_ins = HyInsVtx->d_ush_ins;
      cur_src_ivtx->i_ush_ins.re_alloch(HyInsVtx->i_ush_ins.r->get_rows(), 
                                        HyInsVtx->i_ush_ins.g->get_rows(),
				        HyInsVtx->i_ush_ins.b->get_rows());
      cur_src_ivtx->i_ush_ins = HyInsVtx->i_ush_ins;
      shoot_ins(new_vtxidx, new_vtxidx, 0, -1);
      delete cur_src_ivtx;
     }
    cur_src_ivtx = cur_itemp;
  } else {  // interpolate
      if (rt_RSYMsg) {
	  sprintf(RS_s, "Interpolating intensities for the new vertex from %d neighbour vertices ...", n);
	  rt_Output->message(RS_s);
      }
      for(i = 0; i < n; i++) {
	  d_acc_ins = d_acc_ins + vtx[i]->d_acc_ins;
	  d_ush_ins = d_ush_ins + vtx[i]->d_ush_ins;
      }
    d_acc_ins = d_acc_ins * recip_n; d_ush_ins = d_ush_ins * recip_n;
    new_vtx->d_acc_ins = d_acc_ins;
    new_vtx->d_ush_ins = d_ush_ins;
  }
}

void RT_RS_Points::insert_ip(int n, long* vtxidx,
                             RT_RS_InsVtx*& new_vtx, long& new_vtxidx)
{
  int i;
  long l;
  float recip_n = 1. / (float)n;

  RT_RS_DiffVtx *cur_dtemp, src_dvtx;
  RT_RS_InsVtx *vtx[4], *cur_itemp;
  RT_RS_HyDiffVtx* HyDiffVtx;
  RT_RS_HyInsVtx* HyInsVtx;
  RT_RS_3DVector new_pnt, new_nrm, new_tgn;


  for(i = 0; i < n; i++) {
    vtx[i] = get_ins(vtxidx[i]);
    new_pnt = new_pnt + vtx[i]->pnt;
    new_nrm = new_nrm + vtx[i]->nrm;
    new_tgn = new_tgn + vtx[i]->tgn;
  }
  new_pnt = new_pnt * recip_n;
  new_nrm = new_nrm * recip_n;
  new_tgn = new_tgn * recip_n;

  new_vtxidx = get_inscnt();
  if (rt_RSYMsg) {
      sprintf(RS_s, "Creating a new non-diffuse vertex nr.: %ld\n at: x = %f y = %f z = %f ...",
	      new_vtxidx, new_pnt.x, new_pnt.y, new_pnt.z);
      rt_Output->message(RS_s);
  }
  new_vtx = new RT_RS_InsVtx(new_pnt, new_nrm, new_tgn, vtx[0]->d_p, vtx[0]->i_p);
  insert(new_vtx, &vtx[0]->brdf);
  
  if(cur_src_dvtx != NULL || cur_src_ivtx != NULL) {
      if (rt_RSYMsg) {
	  if(Historydcnt > 0) {
	      sprintf(RS_s, "Repeating %ld diffuse shooting operations for the new vertex ...", 
		      Historydcnt);
	      rt_Output->message(RS_s);
	  }
      }
      cur_dtemp = cur_src_dvtx;
      cur_src_dvtx = &src_dvtx;
    for(l = 0; l < Historydcnt; l++) {
      getHy(l, HyDiffVtx);
      src_dvtx.pnt = HyDiffVtx->pnt; src_dvtx.nrm = HyDiffVtx->nrm; 
      src_dvtx.set_ar(HyDiffVtx->get_ar());
      src_dvtx.d_ush_ins = HyDiffVtx->d_ush_ins;
      shoot_diff(0, -1, new_vtxidx, new_vtxidx);
    }
    cur_src_dvtx = cur_dtemp;
    
      if (rt_RSYMsg) {
	  if (Historyicnt > 0) {
	      sprintf(RS_s, "Repeating %ld non-diffuse shooting operations for the new vertex ...", 
		      Historyicnt);
	      rt_Output->message(RS_s);
	  }
    }
    cur_itemp = cur_src_ivtx;
    for(l = 0; l < Historyicnt; l++) {
      getHy(l, HyInsVtx);
      cur_src_ivtx = new RT_RS_InsVtx(HyInsVtx->pnt, HyInsVtx->nrm, 
				      HyInsVtx->tgn); 
      cur_src_ivtx->set_ar(HyInsVtx->get_ar());
      cur_src_ivtx->d_ush_ins = HyInsVtx->d_ush_ins;
      cur_src_ivtx->i_ush_ins.re_alloch(HyInsVtx->i_ush_ins.r->get_rows(), 
                                        HyInsVtx->i_ush_ins.g->get_rows(),
				        HyInsVtx->i_ush_ins.b->get_rows());
      cur_src_ivtx->i_ush_ins = HyInsVtx->i_ush_ins;
      shoot_ins(0, -1, new_vtxidx, new_vtxidx);
      delete cur_src_ivtx;
     }
    cur_src_ivtx = cur_itemp;
  } else {  // interpolate
      if (rt_RSYMsg) {
	  sprintf(RS_s, "Interpolating intensities for the new vertex from %d neighbour vertices ...", n);
	  rt_Output->message(RS_s);
      }
    for(i = 0; i < n; i++) {
      new_vtx->d_acc_ins = new_vtx->d_acc_ins + vtx[i]->d_acc_ins;
      new_vtx->i_acc_ins = new_vtx->i_acc_ins + vtx[i]->i_acc_ins;
      new_vtx->d_ush_ins = new_vtx->d_ush_ins + vtx[i]->d_ush_ins;
      new_vtx->i_ush_ins = new_vtx->i_ush_ins + vtx[i]->i_ush_ins;
    }
    new_vtx->d_acc_ins = new_vtx->d_acc_ins * recip_n;
    new_vtx->i_acc_ins = new_vtx->i_acc_ins * recip_n;
    new_vtx->d_ush_ins = new_vtx->d_ush_ins * recip_n;
    new_vtx->i_ush_ins = new_vtx->i_ush_ins * recip_n;
  }
}

long RT_RS_Points::get_idx(RT_RS_DiffVtx* dvtx) //get index of diffuse vertex dvtx
{
  for(int i = 0; i < dcnt; i++)
    if(get_diff(i) == dvtx)
      return(i);
  return -1;
}

long RT_RS_Points::get_idx(RT_RS_InsVtx* ivtx) //get index of non-diffuse vertex ivtx
{
  for(int i = 0; i < icnt; i++)
    if(get_ins(i) == ivtx)
      return(i);
  return -1;
}

void RT_RS_Points::get(long n, RT_RS_DiffVtx*& dvtx)    //get the diffuse vertex at index n
{
#ifdef RS_DEBUG
  if(n >= dcnt)
    rt_Output->fatalVar(get_class(), ":dcnt <= n in function get_diff()!", NULL);
  if(n < 0)
    rt_Output->fatalVar(get_class(), ":n < 0 in function get_diff()!", NULL);
#endif
  int sn = (int)(n >> 10);
  dvtx = dPnts[sn][(int)(n - (sn << 10))];
}

void RT_RS_Points::getHy(long n, RT_RS_HyDiffVtx*& Historydvtx)    //get the diffuse history vertex at index n
{
#ifdef RS_DEBUG
  if(n >= Historydcnt)
    rt_Output->fatalVar(get_class(), ":Historydcnt <= n in function getHy(long, RT_RS_HyDiffVtx*&)!", NULL);
  if(n < 0)
    rt_Output->fatalVar(get_class(), ":n < 0 in function getHy((long, RT_RS_HyDiffVtx*&)!", NULL);
#endif
  int sn = (int)(n >> 10);
  Historydvtx = HistorydPnts[sn][(int)(n - (sn << 10))];
}

void RT_RS_Points::get(long n, RT_RS_InsVtx*& ivtx)    //get the non-diffuse vertex at index n
{
#ifdef RS_DEBUG
  if(n >= icnt)
    rt_Output->fatalVar(get_class(), ":icnt <= n in function get_ins()!", NULL);
  if(n < 0)
    rt_Output->fatalVar(get_class(), ":n < 0 in function get_ins()!", NULL);
#endif
  int sn = (int)(n >> 10);
  ivtx = iPnts[sn][(int)(n - (sn << 10))];
}

void RT_RS_Points::getHy(long n, RT_RS_HyInsVtx*& Historyivtx)    //get the diffuse history vertex at index n
{
#ifdef RS_DEBUG
  if(n >= Historyicnt)
    rt_Output->fatalVar(get_class(), ":Historyicnt <= n in function getHy(long, RT_RS_HyInsVtx*&)!", NULL);
  if(n < 0)
    rt_Output->fatalVar(get_class(), ":n < 0 in function getHy((long, RT_RS_HyInsVtx*&)!", NULL);
#endif
  int sn = (int)(n >> 10);
  Historyivtx = HistoryiPnts[sn][(int)(n - (sn << 10))];
}

int RT_RS_Points::get_neighbors_of_vtx(long vn, RT_RS_DiffVtx** neighbors)
{
  RT_RS_DiffArea *a, *nb_a[20];
  int j, k, l, n, nb_acnt = 0, nb_vcnt = 0;
  long i, nb_vidx[20], cnt = scene->Areas->get_diffcnt();

  for(i = 0; i < cnt; i++) {
    scene->Areas->get(i, a);
    if(a->vidx[0] == vn || a->vidx[1] == vn || a->vidx[2] == vn || a->vidx[3] == vn)
      nb_a[nb_acnt++] = a;
  }
#ifdef RS_DEBUG
  if(nb_acnt == 0 || nb_acnt > 13) {
    sprintf(RS_s, ":Diffuse vertex %ld belongs to %d areas!", vn, nb_acnt);
    rt_Output->fatalVar(get_class(), RS_s, NULL);
  }
#endif

  for(l = 0; l < nb_acnt; l++) {
    k = nb_a[l]->vidx[2] == nb_a[l]->vidx[3] ? 3 : 4;
    for(j = 0; j < k; j++) {
      for(n = 0; n < nb_vcnt; n++)
        if(nb_vidx[n] == nb_a[l]->vidx[j])
          break;
      if(n >= nb_vcnt)
        nb_vidx[nb_vcnt++] = nb_a[l]->vidx[j];
    }
  }
#ifdef RS_DEBUG
  if(nb_vcnt > 13) {
    sprintf(RS_s, ":Diffuse vertex %ld has %d neighbor vertices!", vn, nb_vcnt);
    rt_Output->fatalVar(get_class(), RS_s, NULL);
  }
#endif
  for(l = 0; l < nb_vcnt; l++)
    get(nb_vidx[l], neighbors[l]);

  return nb_vcnt;
}

int RT_RS_Points::get_neighbors_of_vtx(long vn, RT_RS_InsVtx** neighbors)
{
  RT_RS_InsArea *a, *nb_a[20];
  int j, k, l, n, nb_acnt = 0, nb_vcnt = 0;
  long i, nb_vidx[20], cnt = scene->Areas->get_inscnt();

  for(i = 0; i < cnt; i++) {
    scene->Areas->get(i, a);
    if(a->vidx[0] == vn || a->vidx[1] == vn || a->vidx[2] == vn || a->vidx[3] == vn)
      nb_a[nb_acnt++] = a;
  }
#ifdef RS_DEBUG
  if(nb_acnt == 0 || nb_acnt > 13) {
    sprintf(RS_s, ":Non-diffuse vertex %ld belongs to %d areas!", vn, nb_acnt);
    rt_Output->fatalVar(get_class(), RS_s, NULL);
  }
#endif

  for(l = 0; l < nb_acnt; l++) {
    k = nb_a[l]->vidx[2] == nb_a[l]->vidx[3] ? 3 : 4;
    for(j = 0; j < k; j++) {
      for(n = 0; n < nb_vcnt; n++)
        if(nb_vidx[n] == nb_a[l]->vidx[j])
          break;
      if(n >= nb_vcnt)
        nb_vidx[nb_vcnt++] = nb_a[l]->vidx[j];
    }
  }
#ifdef RS_DEBUG
  if(nb_vcnt > 13) {
    sprintf(RS_s, ":Non-diffuse vertex %ld has %d neighbor vertices!", vn, nb_vcnt);
    rt_Output->fatalVar(get_class(), RS_s, NULL);
  }
#endif
  for(l = 0; l < nb_vcnt; l++)
    get(nb_vidx[l], neighbors[l]);

  return nb_vcnt;
}

//propagate the accumulated energy of a diffuse patch into the environment
void RT_RS_Points::shoot_diff(long start_didx, long end_didx,
                              long start_iidx, long end_iidx, boolean forward)
{
  long idx;
  float dformfac, ctheta_in, theta_in, ctheta_out, phi_in;

  //for each receiving diffuse vertex
  for(idx = start_didx; idx <= end_didx; idx++) {
    //get the receiving vertex
    RT_RS_DiffVtx* rec_dvtx = get_diff(idx);
    //evaluate occlusion
    if(is_visible(cur_src_dvtx, rec_dvtx, ctheta_out, ctheta_in)) {
      //compute delta form-factor
      dformfac = comp_dff(cur_src_dvtx, rec_dvtx);
      //include cos(theta_out)*cos(theta_in) factor
      dformfac *= ctheta_out * ctheta_in;

      //compute incident energy flux
      RT_RS_DiffRGB PHI(cur_src_dvtx->d_ush_ins * dformfac);
      //compute the delta intensity
      RT_RS_DiffRGB dI(PHI * rec_dvtx->d_p * M_1_PI);
      //update accumulated and unshot intensity at the receiver
      if(forward) {
        rec_dvtx->d_acc_ins = rec_dvtx->d_acc_ins + dI;
        rec_dvtx->d_ush_ins = rec_dvtx->d_ush_ins + dI;
      } else {
        rec_dvtx->d_acc_ins = rec_dvtx->d_acc_ins - dI;
        rec_dvtx->d_ush_ins = rec_dvtx->d_ush_ins - dI;
      }
    }
  }

  //for each receiving non-diffuse vertex
  for(idx = start_iidx; idx <= end_iidx; idx++) {
    //get the receiving vertex
    RT_RS_InsVtx* rec_ivtx = get_ins(idx);
    //evaluate occlusion
    if(is_visible(cur_src_dvtx, rec_ivtx, ctheta_out, ctheta_in)) {
      //compute delta form-factor
      dformfac = comp_dff(cur_src_dvtx, rec_ivtx);
      //include cos(theta_out)*cos(theta_in) factor
      dformfac *= ctheta_out * ctheta_in;
      //compute incident energy flux
      RT_RS_DiffRGB PHI(cur_src_dvtx->d_ush_ins * dformfac);
      if(cur_eval_ins) {   //evaluate the whole function
        //compute incident angle theta
        theta_in = acos(ctheta_in);
        //compute incident angle phi in local axes at the receiving vertex
        phi_in  = rec_ivtx->comp_phi(cur_src_dvtx->pnt);
        //compute the delta intensity function
        RT_RS_InsRGB dI(rec_ivtx->comp_dI(theta_in, phi_in, PHI) * rec_ivtx->i_p);
        dI.set_temp('n');
        //update accumulated and unshot intensity at the receiver
        if(forward) {
          rec_ivtx->i_acc_ins = rec_ivtx->i_acc_ins + dI;
          rec_ivtx->i_ush_ins = rec_ivtx->i_ush_ins + dI;
        } else {
          rec_ivtx->i_acc_ins = rec_ivtx->i_acc_ins - dI;
          rec_ivtx->i_ush_ins = rec_ivtx->i_ush_ins - dI;
        }
        rec_ivtx->i_acc_ins.set_temp('n');
        rec_ivtx->i_ush_ins.set_temp('n');
      }
      //evaluate the diffuse part:
      //include cos(theta_in) factor
      PHI = PHI * ctheta_in;
      //compute the delta intensity
      RT_RS_DiffRGB dI(PHI * rec_ivtx->d_p * M_1_PI);
      //update accumulated and unshot intensity at the receiver
      if(forward) {
        rec_ivtx->d_acc_ins = rec_ivtx->d_acc_ins + dI;
        rec_ivtx->d_ush_ins = rec_ivtx->d_ush_ins + dI;
      } else {
        rec_ivtx->d_acc_ins = rec_ivtx->d_acc_ins - dI;
        rec_ivtx->d_ush_ins = rec_ivtx->d_ush_ins - dI;
      }
    }
  }
}

//propagate the accumulated energy of a non-diffuse patch into the environment
void RT_RS_Points::shoot_ins(long start_didx, long end_didx,
                             long start_iidx, long end_iidx, boolean forward)
{
  long idx;
  float dformfac, ctheta_in, theta_in, ctheta_out, theta_out,
        phi_in, phi_out;

  //for each receiving diffuse vertex
  for(idx = start_didx; idx <= end_didx; idx++) {
    //get the receiving vertex
    RT_RS_DiffVtx* rec_dvtx = get_diff(idx);
    //evaluate occlusion
    if(is_visible(cur_src_ivtx, rec_dvtx, ctheta_out, ctheta_in)) {
      //compute delta form-factor
      dformfac = comp_dff(cur_src_ivtx, rec_dvtx);
      //include cos(theta_in) factor
      dformfac *= ctheta_in;
      //evaluate the diffuse part
      RT_RS_DiffRGB PHI(cur_src_ivtx->d_ush_ins * ctheta_out);
      if(cur_eval_ins) {  //evaluate the whole function
        //compute the outgoing angle theta
        theta_out = acos(ctheta_out);
        //compute the outgoing angle phi
        phi_out   = cur_src_ivtx->comp_phi(rec_dvtx->pnt);
        //evaluate the non-diffuse part
        PHI = PHI + cur_src_ivtx->i_ush_ins.eval(theta_out, phi_out);
      }
      //evaluate the diffuse part:
      //compute incident energy flux
      PHI = PHI * dformfac;
      //compute the delta intensity
      RT_RS_DiffRGB dI(PHI * rec_dvtx->d_p * M_1_PI);
      //update accumulated and unshot intensity at the receiver
      if(forward) {
        rec_dvtx->d_acc_ins = rec_dvtx->d_acc_ins + dI;
        rec_dvtx->d_ush_ins = rec_dvtx->d_ush_ins + dI;
      } else {
        rec_dvtx->d_acc_ins = rec_dvtx->d_acc_ins - dI;
        rec_dvtx->d_ush_ins = rec_dvtx->d_ush_ins - dI;
      }
    }
  }

  //for each receiving non-diffuse vertex
  for(idx = start_iidx; idx <= end_iidx; idx++) {
    //get the receiving vertex
    RT_RS_InsVtx* rec_ivtx = get_ins(idx);
    //evaluate occlusion
    if(is_visible(cur_src_ivtx, rec_ivtx, ctheta_out, ctheta_in)) {
      //compute delta form-factor
      dformfac = comp_dff(cur_src_ivtx, rec_ivtx);
      //include cos(theta_in) factor
      dformfac *= ctheta_in;
      //compute outgoing angle theta
      theta_out = acos(ctheta_out);
      //compute incident angle theta
      theta_in = acos(ctheta_in);
      //evaluate the diffuse part
      RT_RS_DiffRGB PHI(cur_src_ivtx->d_ush_ins * ctheta_out);
      if(cur_eval_ins) {  //evaluate the whole function
        //compute the outgoing angle phi
        phi_out = cur_src_ivtx->comp_phi(rec_ivtx->pnt);
        //evaluate the non-diffuse part
        PHI = PHI + cur_src_ivtx->i_ush_ins.eval(theta_out, phi_out);
      }

      //compute incident energy flux
      PHI = PHI * dformfac;
      if(cur_eval_ins) {   //evaluate the whole function
        //compute incident angle phi in local axes at the receiving vertex
        phi_in  = rec_ivtx->comp_phi(cur_src_ivtx->pnt);
        //compute the delta intensity function
        RT_RS_InsRGB dI(rec_ivtx->comp_dI(theta_in, phi_in, PHI) * rec_ivtx->i_p);
        dI.set_temp('n');
        //update accumulated and unshot intensity at the receiver
        if(forward) {
          rec_ivtx->i_acc_ins = rec_ivtx->i_acc_ins + dI;
          rec_ivtx->i_ush_ins = rec_ivtx->i_ush_ins + dI;
        } else {
          rec_ivtx->i_acc_ins = rec_ivtx->i_acc_ins - dI;
          rec_ivtx->i_ush_ins = rec_ivtx->i_ush_ins - dI;
        }
        rec_ivtx->i_acc_ins.set_temp('n');
        rec_ivtx->i_ush_ins.set_temp('n');
      }
      //evaluate the diffuse part:
      //compute the delta intensity
      RT_RS_DiffRGB dI(PHI * rec_ivtx->d_p * M_1_PI);
      //update accumulated and unshot intensity at the receiver
      if(forward) {
        rec_ivtx->d_acc_ins = rec_ivtx->d_acc_ins + dI;
        rec_ivtx->d_ush_ins = rec_ivtx->d_ush_ins + dI;
      } else {
        rec_ivtx->d_acc_ins = rec_ivtx->d_acc_ins - dI;
        rec_ivtx->d_ush_ins = rec_ivtx->d_ush_ins - dI;
      }
    }
  }
}

//propagate the accumulated energy of the patch into the environment
void RT_RS_Points::shoot() {
  int j, n;
  float d_ins, i_ins;
  long  d_idx, i_idx;
  boolean perform_fast;
  RT_RS_DiffVtx *sdvtx, *dnb[13];
  RT_RS_InsVtx  *sivtx, *inb[13];
  RT_RS_HyDiffVtx *Hydvtx;
  RT_RS_HyInsVtx  *Hyivtx;

  comp_max_scale();

  if(scene->Areas->auto_mesh)
    fast_max_area = (float)(pow(SpatPar->get_volume(), 0.33) / 20.);

  int nr = nr_shoots;
  do {
    RT_RS_DiffVtx* dvtx = get_maxd(d_ins, d_idx);
    RT_RS_InsVtx*  ivtx = get_maxi(i_ins, i_idx);
    perform_fast = FALSE;
    
    if(d_ins > i_ins) {

      if(d_ins < max_unshoot) return;
      sprintf(RS_s,
        "Shoot nr. %d: max. unshoot energy is %f at diffuse vertex nr. %ld.\n",
         ++shoots, d_ins, d_idx);
#ifdef RS_SHOW
      cur_src_didx = d_idx;
      cur_src_iidx = -1;
#endif
      if(fast && d_ins < fast_max_unshoot && dvtx->ar < fast_max_area) {
        n = get_neighbors_of_vtx(d_idx, dnb);
        perform_fast = n > 8; // single shoots at edges
      } 
      if(perform_fast) {
        RT_RS_DiffRGB d_ush_avg;
        float sum_ar = .0;
        for(j = 0; j < n; j++) {
          sum_ar += dnb[j]->ar;
          d_ush_avg = d_ush_avg + dnb[j]->d_ush_ins;
        }
        float recip_n = 1./(float)n;
        sdvtx = new RT_RS_DiffVtx(*dvtx);
        sdvtx->ar = sum_ar;
        sdvtx->d_ush_ins = d_ush_avg * recip_n;
        cur_src_dvtx = sdvtx;
        sprintf(RS_s1, "  (Shooting from %d vertices)", n);
        strcat(RS_s, RS_s1);
      } else
        cur_src_dvtx = dvtx;
      cur_src_ivtx = NULL;
      if (rt_RSYMsg) {
	  sprintf(RS_s1, "  (crd:  x = %f y = %f z = %f)     (nrm:  x = %f y = %f z = %f)  ...",
		  dvtx->pnt.x, dvtx->pnt.y, dvtx->pnt.z,  dvtx->nrm.x, dvtx->nrm.y, dvtx->nrm.z);
	  rt_Output->messageVar(RS_s, RS_s1, NULL);
      }

      shoot_diff(0, dcnt-1, 0, icnt-1);

      Hydvtx = new RT_RS_HyDiffVtx(*cur_src_dvtx);
      insertHy(Hydvtx);   	

      scene->Areas->subdivide_grad();
 
      // set unshot intensity of the patch = 0
      if(perform_fast) {
        for(j = 0; j < n; j++)
          dnb[j]->d_ush_ins.set(0.);
        delete sdvtx;
      } else 
        dvtx->d_ush_ins.set(0);

    } else { // if(d_ins > i_ins)

      if(i_ins < max_unshoot) return;
      sprintf(RS_s,
        "Shoot nr. %d: max. unshoot energy is %f at non-diffuse vertex nr. %ld.\n",
         ++shoots, i_ins, i_idx);
#ifdef RS_SHOW
      cur_src_iidx = i_idx;
      cur_src_didx = -1;
#endif
      if(fast && i_ins < fast_max_unshoot && ivtx->ar < fast_max_area) {
        n = get_neighbors_of_vtx(i_idx, inb);
        perform_fast = n > 8; // single shoots at edges
      } 
      if(perform_fast) {
        RT_RS_DiffRGB d_ush_avg;
        RT_RS_InsRGB i_ush_avg(ivtx->i_ush_ins.r->get_rows(), 
                               ivtx->i_ush_ins.g->get_rows(),
                               ivtx->i_ush_ins.b->get_rows());
        float sum_ar = .0;
        for(j = 0; j < n; j++) {
          sum_ar += inb[j]->ar;
          d_ush_avg = d_ush_avg + inb[j]->d_ush_ins;
          i_ush_avg = i_ush_avg + inb[j]->i_ush_ins;
        }
        float recip_n = 1./(float)n;
        sivtx = new RT_RS_InsVtx(*ivtx);
        sivtx->ar = sum_ar;
        sivtx->d_ush_ins = d_ush_avg * recip_n;
        sivtx->i_ush_ins = i_ush_avg * recip_n;
        cur_src_ivtx = sivtx;
        sprintf(RS_s1, "  (Shooting from %d vertices)", n);
        strcat(RS_s, RS_s1);
      } else
        cur_src_ivtx = ivtx;
      cur_src_dvtx = NULL;
      if (rt_RSYMsg) {
	  sprintf(RS_s1, "  (crd:  x = %f y = %f z = %f)     (nrm:  x = %f y = %f z = %f)  ...",
		  ivtx->pnt.x, ivtx->pnt.y, ivtx->pnt.z,  ivtx->nrm.x, ivtx->nrm.y, ivtx->nrm.z);
	  rt_Output->messageVar(RS_s, RS_s1, NULL);
      }

      shoot_ins(0, dcnt-1, 0, icnt-1);

      Hyivtx = new RT_RS_HyInsVtx(*cur_src_ivtx);
      insertHy(Hyivtx);
 
      scene->Areas->subdivide_grad();

      // set unshot intensity of the patch = 0
      if(perform_fast) {
        for(j = 0; j < n; j++) {
          inb[j]->d_ush_ins.set(0.);
          inb[j]->i_ush_ins.set(0.);
        }
        delete sivtx;
      } else {
        ivtx->i_ush_ins.set(0.);
        ivtx->d_ush_ins.set(0.);
      }
    }

    cur_src_dvtx = NULL; cur_src_ivtx = NULL;

  } while (--nr);
}

void RT_RS_Points::init_spatpar() {
  RT_RS_3DVector v[4];
  RT_RS_Vtx* vtx;
  RT_RS_Area* area;
  long vidx[4], cnt;
  int i, j, m, vcnt = 0;
  float min_x, min_y, min_z;
  float max_x, max_y, max_z;

  if(SpatPar != NULL) delete SpatPar;
  SpatPar = new RT_RS_SpatPar(spar_dim);
  //compute the bounding box of the scene
  min_x = min_y = min_z = BIGFLOAT;
  max_x = max_y = max_z = -BIGFLOAT;
  cnt = dcnt;
  for(m = 0; m < 2; m++) {
    for(i = 0; i < cnt; i++) {
      if(m) vtx = get_ins(i); else vtx = get_diff(i);
      if(vtx->pnt.x > max_x) max_x = vtx->pnt.x;
      if(vtx->pnt.y > max_y) max_y = vtx->pnt.y;
      if(vtx->pnt.z > max_z) max_z = vtx->pnt.z;
      if(vtx->pnt.x < min_x) min_x = vtx->pnt.x;
      if(vtx->pnt.y < min_y) min_y = vtx->pnt.y;
      if(vtx->pnt.z < min_z) min_z = vtx->pnt.z;
    }
    cnt = icnt;
  }
  float rdim = 3. / (float)spar_dim;
  float bx = (max_x - min_x) * rdim;
  float by = (max_y - min_y) * rdim;
  float bz = (max_z - min_z) * rdim;
  SpatPar->bnd_p1.x = min_x - bx;
  SpatPar->bnd_p1.y = min_y - by;
  SpatPar->bnd_p1.z = min_z - bz;
  SpatPar->bnd_p2.x = max_x + bx;
  SpatPar->bnd_p2.y = max_y + by;
  SpatPar->bnd_p2.z = max_z + bz;
  SpatPar->d_bnd = (SpatPar->bnd_p2 - SpatPar->bnd_p1) * (1/(float)spar_dim);
#ifdef RS_DEBUG
  if(SpatPar->d_bnd.x < epsilon || SpatPar->d_bnd.y < epsilon ||
     SpatPar->d_bnd.z < epsilon)
    rt_Output->fatalVar(get_class(), ":Bounding box has no volume", NULL);
#endif
  SpatPar->rd_bnd.x = 1 / SpatPar->d_bnd.x;
  SpatPar->rd_bnd.y = 1 / SpatPar->d_bnd.y;
  SpatPar->rd_bnd.z = 1 / SpatPar->d_bnd.z;

  //insert the bounding vertices of the planar surfaces
  cnt = dcnt;
  for(m = 0; m < 2; m++) {
    for(i = 0; i < cnt; i++) {
      if(m) vtx = get_ins(i); else vtx = get_diff(i);
      if(vtx->vtx_type >= (int)corner0 && vtx->vtx_type <= (int)corner3) {//corner vertex of a planar surface ?
        j = (int)vtx->vtx_type - (int)corner0;
        v[j] = vtx->pnt;
        vidx[j] = m ? -i - 1 : i;
        vcnt++;
      }
      if(vcnt < 4) continue;
      vcnt = 0;
      SpatPar->insert(vidx, v);
    }
    cnt = icnt;
  }
  //insert the vertices of the curved surfaces
  cnt = scene->Areas->get_diffcnt();
  for(m = 0; m < 2; m++) {
    for(i = 0; i < cnt; i++) {
      if(m) area = scene->Areas->get_ins(i); else area = scene->Areas->get_diff(i);
      vtx = area->get_vtx(area->vidx[0]);
      if(vtx->vtx_type != curved) continue;           //a curved surface vertex ?
      for(j = 0; j < 4; j++) {
        vidx[j] = m ? -area->vidx[j] - 1 : area->vidx[j];
        v[j] = area->get_vtx(area->vidx[j])->pnt;
      }
      SpatPar->insert(vidx, v);
    }
    cnt = scene->Areas->get_inscnt();
  }
}


void RT_RS_Points::set_ar0()
{
  long i;
  for(i = 0; i < dcnt; i++) get_diff(i)->ar = 0.;
  for(i = 0; i < icnt; i++) get_ins(i)->ar = 0.;
}

void RT_RS_Points::reset_vis()
{
  long i;
  for(i = 0; i < dcnt; i++) get_diff(i)->vis = unknown;
  for(i = 0; i < icnt; i++) get_ins(i)->vis = unknown;
}

void RT_RS_Points::scale_view_vals() {
  if (rt_RSYMsg) {
      sprintf(RS_s, "  Scaling view-values: %f", view_scale);
      rt_Output->message(RS_s);
  }

  for(int m = 0; m < 2; m++) {
    long cnt = m ? icnt : dcnt;
    for(long i = 0; i < cnt; i++) {
      RT_RS_DiffRGB* pvw = m ? &get_ins(i)->view_val : &get_diff(i)->view_val;
#ifdef RS_SHOW
      if(m && i == cur_src_iidx || !m && i == cur_src_didx) {
        pvw->r = 255.0; pvw->g = pvw->b = 0.0;    // draw the current source vertex in red
      } else
#endif
      if(pvw->r < max_scale && pvw->g < max_scale && pvw->b < max_scale) //check to avoid overflow
        *pvw = *pvw * view_scale;
    }
  }
}

void RT_RS_Points::comp_max_scale()
{
  if(max_scale > epsilon) return;
  long i; float f;
  RT_RS_DiffVtx *dvtx;
  RT_RS_InsVtx *ivtx;
  float min = BIGFLOAT;
  for(i = 0; i < dcnt; i++) {
    dvtx = get_diff(i);
    f = dvtx->d_acc_ins.r; if(f > epsilon && f < min) min = f;
    f = dvtx->d_acc_ins.g; if(f > epsilon && f < min) min = f;
    f = dvtx->d_acc_ins.b; if(f > epsilon && f < min) min = f;
  }
  for(i = 0; i < icnt; i++) {
    ivtx = get_ins(i);
    f = ivtx->d_acc_ins.r + ivtx->i_acc_ins.r->avg; if(f > epsilon && f < min) min = f;
    f = ivtx->d_acc_ins.g + ivtx->i_acc_ins.g->avg; if(f > epsilon && f < min) min = f;
    f = ivtx->d_acc_ins.b + ivtx->i_acc_ins.b->avg; if(f > epsilon && f < min) min = f;
  }
  max_scale = min - epsilon;
}

RT_RS_DiffVtx* RT_RS_Points::get_maxd(float& ins_max, long& idx)
{
  float f;
  RT_RS_DiffVtx* dvtx;
  RT_RS_DiffVtx* dvtx_max;

  ins_max = -BIGFLOAT;
  for(long i = 0; i < dcnt; i++) {
    dvtx = get_diff(i);
    f = dvtx->d_ush_ins.comp_sum();
    if(f > ins_max) {
      ins_max = f;
      dvtx_max = dvtx;
      idx = i;
    }
  }
  return dvtx_max;
}


RT_RS_InsVtx* RT_RS_Points::get_maxi(float& ins_max, long& idx)
{
  float f;
  RT_RS_InsVtx* ivtx;
  RT_RS_InsVtx* ivtx_max;

  ins_max = -BIGFLOAT;
  for(long i = 0; i < icnt; i++) {
    ivtx = get_ins(i);
    f = ivtx->i_ush_ins.comp_sum() + ivtx->d_ush_ins.comp_sum();
    if(f > ins_max) {
      ins_max = f;
      ivtx_max = ivtx;
      idx = i;
    }
  }
  return ivtx_max;
}

float RT_RS_Points::comp_dff(RT_RS_Vtx* vtx1, RT_RS_Vtx* vtx2)
{
  RT_RS_3DVector dir(vtx1->pnt - vtx2->pnt);
  float r = dir.vabs();
  if(r < epsilon) return .0;    //dFii = 0;
  return M_PI * vtx1->ar / (vtx1->ar + M_PI * r * r);
}

boolean RT_RS_Points::is_occluded(RT_RS_Vtx* vtx1, RT_RS_Vtx* vtx2)
{
  static RT_RS_Matrix A(3, 3);
  static RT_RS_Vector X(3), B(3);
  RT_RS_3DVector v[4];
  RT_RS_3DVector d_dir, bd_dir;
  RT_RS_Voxel* Voxel;
  int i, j, k, l, jp1, res, vox1[3], vox2[3], voxdir[3], d_vox[3], pd_vox[3];
  float t, f, den, xi, yi, xt, yt, x1, y1, x2, y2;

  RT_RS_3DVector dir(vtx2->pnt - vtx1->pnt);
  float dir_abs = dir.vabs();
  if(dir_abs < epsilon3) return FALSE;

  RT_RS_3DVector d_pnt(vtx1->pnt);

#ifdef RS_DEBUG
  if(SpatPar == NULL)
    rt_Output->fatalVar(get_class(), ":Spatial partitioning is not initialized!", NULL);
#endif
  SpatPar->get_voxidx(vtx1->pnt, vox1);
  SpatPar->get_voxidx(vtx2->pnt, vox2);
  voxdir[0] = vox2[0] - vox1[0];
  voxdir[1] = vox2[1] - vox1[1];
  voxdir[2] = vox2[2] - vox1[2];
  l = _max(_max(abs(voxdir[0]), abs(voxdir[1])), abs(voxdir[2]));
  if(l > 0) d_dir = dir * (1./(float)l);
  d_vox[0] = vox1[0]; d_vox[1] = vox1[1]; d_vox[2] = vox1[2];
  B.set(1.);
  for(;;) {
    Voxel = SpatPar->get(d_vox);
    while(Voxel != NULL) {
      if(Voxel->vidx[0] < 0)
        for(j = 0; j < 4; j++)
          v[j] = get_ins(-(Voxel->vidx[j] + 1))->pnt;
      else
        for(j = 0; j < 4; j++)
          v[j] = get_diff(Voxel->vidx[j])->pnt;

      //compute the plane coefficients
      k = res = 0;
      do {
        if(res != 0) {   //plane goes through zero point
          switch(k % 3) {
          case 0: f = v[0].x;
                  v[0].x = fabs(f) > 1. ? f + f * epsilon5 : f + epsilon5;
                  f = v[1].x;
                  v[1].x = fabs(f) > 1. ? f - f * epsilon5 : f - epsilon5;
                  f = v[2].x;
                  v[2].x = fabs(f) > 1. ? f - f * epsilon5 : f - epsilon5;
                  break;
          case 1: f = v[0].y;
                  v[0].y = fabs(f) > 1. ? f + f * epsilon5 : f + epsilon5;
                  f = v[1].y;
                  v[1].y = fabs(f) > 1. ? f - f * epsilon5 : f - epsilon5;
                  f = v[2].y;
                  v[2].y = fabs(f) > 1. ? f - f * epsilon5 : f - epsilon5;
                  break;
          case 2: f = v[0].z;
                  v[0].z = fabs(f) > 1. ? f + f * epsilon5 : f + epsilon5;
                  f = v[1].z;
                  v[1].z = fabs(f) > 1. ? f - f * epsilon5 : f - epsilon5;
                  f = v[2].z;
                  v[2].z = fabs(f) > 1. ? f - f * epsilon5 : f - epsilon5;
          }
          k++;
        }
        // check if this patch is a triangle
        A(0,0) = v[0].x; A(0,1) = v[0].y; A(0,2) = v[0].z;
        j = (v[1].x - v[0].x > epsilon5 || v[1].y - v[0].y > epsilon5 ||
             v[1].z - v[0].z > epsilon5) ? 1 : 2;
        A(1,0) = v[j].x; A(1,1) = v[j].y; A(1,2) = v[j].z;
        jp1 = j + 1;
        if(!(v[jp1].x - v[j].x > epsilon5  || v[jp1].y - v[j].y > epsilon5 ||
             v[jp1].z - v[j].z > epsilon5) && jp1 < 3) jp1++;
        A(2,0) = v[jp1].x; A(2,1) = v[jp1].y; A(2,2) = v[jp1].z;

        res = MatEqSys(A, X, B);

        if(k > 12)
          rt_Output->fatalVar(get_class(),
          ":Cannot compute plane coefficients in function is_occluded()!", NULL);

      } while(res != 0);
      float a = X(0), b = X(1), c = X(2);
      //compute plane - ray intersection point
      den = a * dir.x + b * dir.y + c * dir.z;
      if(fabs(den) < epsilon4) {
        Voxel = Voxel->next;
        continue; // plane and ray are parallel
      }
      t = (1 - (a * vtx1->pnt.x + b * vtx1->pnt.y + c * vtx1->pnt.z)) / den;
      if(t < epsilon3 || t > 1.0 - epsilon3) {
        Voxel = Voxel->next;
        continue;
      }
      RT_RS_3DVector inter_pnt(vtx1->pnt + dir * t);

      //determine whether the intersection point lies within the patch
      a = fabs(X(0)); b = fabs(X(1)); c = fabs(X(2));
      float abmax = _max(a, b);
      int mx = a > b ? 0 : 1;
      if(c > abmax) mx = 2;

      //scale the projected polygon
      switch(mx) {
      case 0: xt = (v[0].y + v[1].y + v[2].y + v[3].y) * .25;
              yt = (v[0].z + v[1].z + v[2].z + v[3].z) * .25;
              for(j=0; j<3; j++) {
                v[j].y -= xt;
                v[j].z -= yt;
              }
              for(j=0; j<3; j++) {
                v[j].y += v[j].y * epsilon3;
                v[j].z += v[j].z * epsilon3;
              }
              for(j=0; j<3; j++) {
                v[j].y += xt;
                v[j].z += yt;
              }
              break;
      case 1: xt = (v[0].x + v[1].x + v[2].x + v[3].x) * .25;
              yt = (v[0].z + v[1].z + v[2].z + v[3].z) * .25;
              for(j=0; j<3; j++) {
                v[j].x -= xt;
                v[j].z -= yt;
              }
              for(j=0; j<3; j++) {
                v[j].x += v[j].x * epsilon3;
                v[j].z += v[j].z * epsilon3;
              }
              for(j=0; j<3; j++) {
                v[j].x += xt;
                v[j].z += yt;
              }
              break;
      case 2: xt = (v[0].x + v[1].x + v[2].x + v[3].x) * .25;
              yt = (v[0].y + v[1].y + v[2].y + v[3].y) * .25;
              for(j=0; j<3; j++) {
                v[j].x -= xt;
                v[j].y -= yt;
              }
              for(j=0; j<3; j++) {
                v[j].x += v[j].x * epsilon3;
                v[j].y += v[j].y * epsilon3;
              }
              for(j=0; j<3; j++) {
                v[j].x += xt;
                v[j].y += yt;
              }
              break;
      }

      switch(mx) {
      case 0: xi = inter_pnt.y; yi = inter_pnt.z;
              xt = (v[3].y + v[0].y) * .5;
              yt = (v[3].z + v[0].z) * .5;
              break;
      case 1: xi = inter_pnt.x; yi = inter_pnt.z;
              xt = (v[3].x + v[0].x) * .5;
              yt = (v[3].z + v[0].z) * .5;
              break;
      case 2: xi = inter_pnt.x; yi = inter_pnt.y;
              xt = (v[3].x + v[0].x) * .5;
              yt = (v[3].y + v[0].y) * .5;
      }
      float ytmyi = yt - yi, xtmxi = xt - xi;

      if(fabs(ytmyi) < epsilon3 && fabs(xtmxi) < epsilon3)
        return TRUE;
      boolean outside = FALSE;
      for(j = 0; j < 3; j++) {
        jp1 = j + 1;
        switch(mx) {
        case 0: x1 = v[j].y; y1 = v[j].z;
                x2 = v[jp1].y; y2 = v[jp1].z;
                break;
        case 1: x1 = v[j].x; y1 = v[j].z;
                x2 = v[jp1].x; y2 = v[jp1].z;
                break;
        case 2: x1 = v[j].x; y1 = v[j].y;
                x2 = v[jp1].x; y2 = v[jp1].y;
        }
        float y2my1 = y2 - y1, x2mx1 = x2 - x1,
              xtmxi_y2my1 = xtmxi * y2my1, ytmyi_x2mx1 = ytmyi * x2mx1;

        den = xtmxi_y2my1 - ytmyi_x2mx1;
        if(fabs(den) < epsilon5)  continue; // parallel
        t = ((x1 - xi) * y2my1 + (yi - y1) * x2mx1) / den; // line parameter value
        if(t < .0) continue; // intersection is "behind"

        den = ytmyi_x2mx1 - xtmxi_y2my1;
        t = ((xi - x1) * ytmyi + (y1 - yi) * xtmxi) / den; // edge parameter value
        if(t < .0 || t > 1.) continue;
        outside = TRUE;
      }
      if(!outside)
        return TRUE;
      Voxel = Voxel->next;
    }
    if(abs(d_vox[0] - vox2[0]) <= 2 && abs(d_vox[1] - vox2[1]) <= 2 &&
       abs(d_vox[2] - vox2[2]) <= 2) 
      if((d_pnt - vtx1->pnt).vabs() >= dir_abs) 
        break;
    if(d_vox[0] != vox2[0] || d_vox[1] != vox2[1] || d_vox[2] != vox2[2]) {
      d_pnt = d_pnt + d_dir;
      bd_dir = d_dir;
      pd_vox[0] = d_vox[0]; pd_vox[1] = d_vox[1]; pd_vox[2] = d_vox[2];
      i = 0;
      for(;;) {
        SpatPar->get_voxidx(d_pnt, d_vox);
        if((pd_vox[0] != d_vox[0] && pd_vox[1] != d_vox[1]) ||
           (pd_vox[1] != d_vox[1] && pd_vox[2] != d_vox[2]) ||
           (pd_vox[2] != d_vox[2] && pd_vox[0] != d_vox[0])) {
          if(++i > 8) break;
          bd_dir = bd_dir * .5;
          d_pnt = d_pnt - bd_dir;
        } else
          if(pd_vox[0] == d_vox[0] && pd_vox[1] == d_vox[1] && pd_vox[2] == d_vox[2]) {
            if(++i > 16) break;
            bd_dir = bd_dir * .5;
            d_pnt = d_pnt + bd_dir;
          } else break;
      }
      continue;
    }
    break;
  }
  return FALSE;
}

boolean RT_RS_Points::is_visible(RT_RS_Vtx* src_vtx, RT_RS_Vtx* rec_vtx,
                                 float& ctheta_out, float& ctheta_in,
                                 boolean do_occ_test)
{
  ctheta_out = src_vtx->comp_ctheta(rec_vtx->pnt);
  if(ctheta_out < epsilon3) return FALSE;

  ctheta_in  = rec_vtx->comp_ctheta(src_vtx->pnt);
  if(ctheta_in  < epsilon3) return FALSE;

  if(do_occ_test)
    if(is_occluded(src_vtx, rec_vtx))
      return FALSE;

  return TRUE;
}


boolean RT_RS_Points::read(char *filename, FILE *fp)
{
  long _dcnt, _icnt, _Historydcnt, _Historyicnt, i;
  RT_RS_DiffVtx* diff_vtx;
  RT_RS_InsVtx* ins_vtx;
  RT_RS_HyDiffVtx* hydiff_vtx;
  RT_RS_HyInsVtx* hyins_vtx;
  boolean aln = fp == NULL;
  boolean Ok = TRUE;
  int clmn = 0;

  if(aln) Ok = openfile(filename, fp, "rt");
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) Ok = isclass(filename, RS_s1);
  if(Ok) Ok = getlong(filename, fp, _dcnt, RS_s, clmn);
  if(Ok) Ok = getlong(filename, fp, _icnt, RS_s, clmn);
  if(Ok) Ok = getlong(filename, fp, _Historydcnt, RS_s, clmn);
  if(Ok) Ok = getlong(filename, fp, _Historyicnt, RS_s, clmn);
  if(Ok) Ok = getint(filename, fp, shoots, RS_s, clmn);
  if(Ok) Ok = getint(filename, fp, cur_eval_ins, RS_s, clmn);
  if(Ok) Ok = getint(filename, fp, fast, RS_s, clmn);
  if(Ok) Ok = getint(filename, fp, spar_dim, RS_s, clmn);
  if(Ok) Ok = getint(filename, fp, nr_shoots, RS_s, clmn);
  if(Ok) Ok = getfloat(filename, fp, max_unshoot, RS_s, clmn);
  if(Ok) Ok = getfloat(filename, fp, fast_max_unshoot, RS_s, clmn);
  if(Ok) Ok = getfloat(filename, fp, fast_max_area, RS_s, clmn);
  if(Ok) Ok = getfloat(filename, fp, view_scale, RS_s, clmn);
  if(Ok) Ok = getfloat(filename, fp, max_scale, RS_s, clmn);
  if(Ok) {
    freeh();
    for(i = 0; i < _dcnt; i++) {
      diff_vtx = new RT_RS_DiffVtx(scene);
      insert(diff_vtx);
      Ok = diff_vtx->read(filename, fp);
      if(!Ok) break;
    }
  }
  if(Ok) {
    for(i = 0; i < _icnt; i++) {
      ins_vtx = new RT_RS_InsVtx(scene);
      insert(ins_vtx);
      Ok = ins_vtx->read(filename, fp);
      if(!Ok) break;
    }
  }
  if(Ok) {
    for(i = 0; i < _Historydcnt; i++) {
      hydiff_vtx = new RT_RS_HyDiffVtx(scene);
      insertHy(hydiff_vtx);
      Ok = hydiff_vtx->read(filename, fp);
      if(!Ok) break;
    }
  }
  if(Ok) {
    for(i = 0; i < _Historyicnt; i++) {
      hyins_vtx = new RT_RS_HyInsVtx(scene);
      insertHy(hyins_vtx);
      Ok = hyins_vtx->read(filename, fp);
      if(!Ok) break;
    }
  }
  if(aln) fclose(fp);
  return Ok;
}

boolean RT_RS_Points::write(char *filename, FILE *fp)
{
  long i;
  boolean Ok = TRUE;
  boolean aln = fp == NULL;

  if(aln) Ok = openfile(filename, fp, "wt");
  if(Ok) {
     fprintf(fp,"%s %ld %ld %ld %ld\n%d %d %d %d %d %f %f %f\n%f %f\n", 
	     get_class(), dcnt, icnt, Historydcnt, Historyicnt,
             shoots, cur_eval_ins, fast, spar_dim, nr_shoots, 
             max_unshoot, fast_max_unshoot, fast_max_area,
             view_scale, max_scale);
    Ok = !ferror(fp);
  }
  for(i = 0; i < dcnt && Ok; i++)
    Ok = get_diff(i)->write(filename,fp);
  for(i = 0; i < icnt && Ok; i++) 
    Ok = get_ins(i)->write(filename,fp);
  for(i = 0; i < Historydcnt && Ok; i++) 
    Ok = get_Hydiff(i)->write(filename,fp);
  for(i = 0; i < Historyicnt && Ok; i++)
    Ok = get_Hyins(i)->write(filename,fp);

  if(!Ok) {
    rt_Output->errorVar(get_class(), ":Cannot write ", filename, NULL);
    return FALSE;
  }
  if(aln) fclose(fp);
  return TRUE;
}

void RT_RS_Points::print(FILE *f, char *n, int width, int decimalPlaces)
{

  if(n) fprintf(f, "%s\n", n);

  fprintf(f, "statistics:\n");
  fprintf(f, "  diffuse vertices: %ld\n", dcnt);
  fprintf(f, "  non-diffuse vertices: %ld\n", icnt);
  fprintf(f, "  diffuse history-vertices: %ld\n", Historydcnt);
  fprintf(f, "  non-diffuse history-vertices: %ld\n", Historyicnt);
  fprintf(f, "  number of iterations so far: %d\n", shoots);

  fprintf(f, "mode:\n");
  fprintf(f, "  evaluate non-diffuse part: %d\n", cur_eval_ins);
  fprintf(f, "  fast convergence mode: %d\n", fast);

  fprintf(f, "parameters:\n");
  fprintf(f, "  spatial partitioning solution in each dimension: %d\n", spar_dim);
  fprintf(f, "  specified number of iterations: %d\n", nr_shoots);
  sprintf(fmt,"  %s: %s%d.%df\n", "%s", "%", width, decimalPlaces);
  fprintf(f, fmt, "maximum of unshoot values", max_unshoot);
  fprintf(f, fmt, "fast convergence mode - unshoot value", fast_max_unshoot);
  fprintf(f, fmt, "fast convergence mode - area value", fast_max_area);
  fprintf(f, fmt, "scale intensity in viewing direction", view_scale);
  fprintf(f, fmt, "maximum intensity for viewscale", max_scale);
#ifdef RS_PRINT_ALL
  long i;
  char stitle[30];

  for(i = 0; i < dcnt; i++) {
    sprintf(stitle, "diffuse vertex %d:", i);
    get_diff(i)->print(f, stitle, width, decimalPlaces);
  }
  for(i = 0; i < icnt; i++) {
    sprintf(stitle,"non-diffuse vertex %d:", i);
    get_ins(i)->print(f, stitle, width, decimalPlaces);
  }
#endif
}

RT_RS_DiffRGB rt_DefaultRGB = RT_RS_DiffRGB();
RT_RS_brdfRGB rt_BrdfRGB = RT_RS_brdfRGB();
RT_RS_3DVector rt_Tangent(0,0,0);
RT_RS_3DVector rt_PointsPRP; 
RT_RS_Vtx rt_PointsVtx;

