////////////////////////////////////////////////////////////////////////////////
//  Implementation of class RT_Primitive.                                     //  
//  LAST EDIT: Wed Mar  8 14:16:00 1995 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include "primitiv.h"
#include "attrobj.h"
#include "intrsect.h"
#include "scene.h"
#include "high/analytic.h" // needed for bounding box (Quader)

const char *RTN_PRIMITIVE = "Primitive";

int RT_Primitive::rotF, RT_Primitive::scaF, RT_Primitive::traF;
int RT_Primitive::matF, RT_Primitive::matG, RT_Primitive::bndG;
int RT_Primitive::fatF, RT_Primitive::fatG, RT_Primitive::chiG;
int RT_Primitive::wmatG;
RT_Vector RT_Primitive::rotV, RT_Primitive::scaV, RT_Primitive::traV;
RT_Matrix RT_Primitive::matV;
char *RT_Primitive::fatV; int RT_Primitive::hbdG;
int RT_Primitive::clAtF; char *RT_Primitive::clAtV; int RT_Primitive::updF;
int RT_Primitive::atLiG;
int RT_Primitive::visF, RT_Primitive::invF, RT_Primitive::visG;
int RT_Primitive::bxF, RT_Primitive::bxV, RT_Primitive::bxG;
int RT_Primitive::hdF, RT_Primitive::unF, RT_Primitive::hdG;
int RT_Primitive::pmF, RT_Primitive::pmV, RT_Primitive::pmG;
int RT_Primitive::cpF; char *RT_Primitive::cpV;

RT_ParseEntry RT_Primitive::table[] = {
    { "-father", RTP_STRING, (char*)&fatV, &fatF, "Set the {ARG 1 Father} object. If {ARG 1 Father} is an empty string then remove the current father.", "Object" },
    { "-get_father", RTP_NONE, 0, &fatG, "Get the Father object.", RTPS_NONE },
    { "-get_children", RTP_NONE, 0, &chiG, "Get the children.", RTPS_NONE },
    { "-update", RTP_NONE, 0, &updF, "Update the geometry, properties and attributes of the primitive and its children.", RTPS_NONE },
    { "-rotate", RTP_VECTOR, (char*)&rotV, &rotF, "{ARG 1 Rotate} the object in the area {RANGE 1 \"-2 2\"}. First the rotation around the x axis will be executed, then y and at last z. The orientation of a positive rotate arc is determined by the right-hand-rule.", RTPS_VECTOR },
    { "-scale", RTP_VECTOR, (char*)&scaV, &scaF, "{ARG 1 Scale} the object.", RTPS_VECTOR },
    { "-translate", RTP_VECTOR, (char*)&traV, &traF, "{ARG 1 Translate} the object.", RTPS_VECTOR },

    { "-matrix", RTP_MATRIX, (char*)&matV, &matF, "Overload the modelling {ARG 1 Matrix} of the object.", RTPS_MATRIX },
    { "-get_matrix", RTP_NONE, 0, &matG, "Return the modelling matrix of the object.", RTPS_NONE },
    { "-get_worldMatrix", RTP_NONE, 0, &wmatG, "Return the world modelling matrix of the object.", RTPS_NONE },

    { "-visible", RTP_NONE, 0, &visF, "Make the object visible.", RTPS_NONE },
    { "-invisible", RTP_NONE, 0, &invF, "Hide the object.", RTPS_NONE },
    { "-get_visible", RTP_NONE, 0, &visG, "Get the visibility of the object.", RTPS_NONE },

    { "-copy", RTP_STRING, (char*)&cpV , &cpF, "Copy the geometry into another {ARG 1 Primitive}. No attributes will be copied. However sometimes, the generated vertex-based primitive reflects the attributes of this primitive, e.g. a resolution attribute. Some kinds of primitives may be copied to various kinds of primitives. Generally, there is a loss of information when copying.", "Primitive" },

    { "-hide", RTP_NONE, 0, &hdF, "Hide the object that it will not be visible in hierarchical displays.", RTPS_NONE },
    { "-unhide", RTP_NONE, 0, &unF, "Unhide the object.", RTPS_NONE },
    { "-get_hide", RTP_NONE, 0, &hdG, "Get the hide-state of the object.", RTPS_NONE },

    { "-pickmode", RTP_INTEGER, (char*)&pmV, &pmF, "Set the pick {ARG 1 Mode}. The values are (0 - 2): {ENUM 1 \"NotPickable Pickable PickFather\"}.", RTPS_INTEGER },
    { "-get_pickmode", RTP_NONE, 0, &pmG, "Return pickmode.", RTPS_NONE },

    { "-get_bounds", RTP_NONE, 0, &bndG, "Get the boundary box of the object in Modeling (local) Coordinates.", RTPS_NONE },
    { "-get_hierarchBounds", RTP_NONE, 0, &hbdG, "Get the boundary box inclusive children also in Modeling Coordinates.", RTPS_NONE },

    { "-box", RTP_INTEGER, (char*)&bxV, &bxF, "Set bounding box {ARG 1 Mode}. Possible values are (from zero): {ENUM 1 \"NoBox ObjectBox HierarchicalBox FilledBox FilledHierarchicalBox\"}. These modes do not affect the raytraced image.", RTPS_INTEGER },
    { "-get_box", RTP_NONE, 0, &bxG, "Return current bounding box mode.", RTPS_NONE },

    { "-attribute", RTP_SPECIAL, 0, 0, "Set an attribute of specified {ARG 1 Type} with an initial {ARG 2 Value}.", "Attribute String" },
    { "-get_attribute", RTP_SPECIAL, 0, 0, "Inquire the value of an attribute of specified {ARG 1 Type}.", "Attribute" },
    { "-get_attrList", RTP_NONE, 0, &atLiG, "Returns a list of all attributes the object owns.", RTPS_NONE },

    { "-attrNames", RTP_SPECIAL, 0, 0, "Set a {ARG 2 Names} for an attribute of the specified {ARG 1 Attribute} class. You can specify a list of class and object names the specified attribute should be related to. The object itself should match at least one term of the list. An exception could be made if it is of class Top.", "Attribute String_List" },
    { "-attrClear", RTP_STRING, (char*)&clAtV, &clAtF, "Clear an {ARG 1 Attribute} from the attribute list of the primitive.", "Attribute" },

    { 0, RTP_END, 0, 0, 0, 0 }
};

class RT_PrintAttributeFunc: public RT_GeneralListFunctoid {
    RT_String &str;
  public:
    RT_PrintAttributeFunc(RT_String &_str):str(_str) {}
    void exec(RT_GeneralListEntry *e, void * = 0 ) {
	// assume, there are only attributes in an attribute list:
	RT_Attribute *a = (RT_Attribute *)e;
	str += "{";
	str += a->getClass(); str += " {";
	// using curly brakes because <primClass> could be empty
	str += a->getNamesList(); str += "} ";
	str += a->get(); 
	str += "} ";
    }
};

void RT_Primitive::father(RT_Primitive *f) {
    if ( (long)f == -1) f = 0;
    else if (xfather) xfather->removeChild( this );
    xfather = f;
    if (xfather) {
	xfather->addChild( this );
	if (hasModel) { 
	    if (isPart && get_father()) get_father()->partModelling = 1; 
	    // force the father to transfer down changes of his matrix
	    // on default partModelling is turned off - that saves
	    // a lot of time for e.g. OFF objects
	    model->concat( get_father()->model );
	    matrixChanged();
	} 
	else setModel( get_father()->model );
    }
    else { 
	// reinitialize the reference
	if (!hasModel) setModel( (RT_Modelling*)&rt_Modelling );
    }
    // force a message to the father:
    attrsChangedFlag = 0;
    attrsChanged();
    primChanged();
}

#ifdef RTD_RSY

class RT_toRSYChildFunc: public RT_GeneralListFunctoid {
  public:
    void exec(RT_GeneralListEntry *e, void * rsc) {
	((RT_Primitive*)e)->toRSYHierarchical((RT_RSYScene*) rsc); 
    }
};

void RT_Primitive::toRSYHierarchical(RT_RSYScene *rsc) {
    if ( /*rsyChangedFlag &&*/ get_visible()) {
	toRSY(rsc);
	RT_toRSYChildFunc func;
	children.doWithElements( &func, rsc );
    }
    rsyChangedFlag = 0;
}

void RT_Primitive::toRSY(RT_RSYScene *rsc) {
    RT_toRSYChildFunc func;
    parts.doWithElements( &func, rsc );
}

#endif

int RT_Primitive::objectCMD(char *argv[]) {
    int r = 0;
    RT_parseTable( argv, table );
    if (fatF) {
	if (!strcmp( fatV, "" )) father( 0 );
	else {
	    RT_Object *o = RT_Object::getObject( fatV );
	    if (!o || !o->isA( RTN_PRIMITIVE )) {
		rt_Output->errorVar( get_name(), ": No such primitive: ", fatV, "! ", 0 );
		fatF = 0;
	    }
	    else father( (RT_Primitive*)o);
	}
    }
    if (fatG ) RT_Object::result( xfather ? xfather->get_name() : "" );
    if (atLiG) {
	r++;
	RT_String tmp(300); 
	tmp += '{';
	RT_PrintAttributeFunc func( tmp );
	get_attrList()->doWithElements( &func );
	tmp += '}';
	RT_Object::result( (char*)tmp );
    }
    if (chiG) {
	RT_String tmp( 100 );
	RT_PrintObjectNameFunc func( &tmp );
	tmp += '{';
	children.doWithElements( &func );
	tmp += '}';
	RT_Object::result( (char*)tmp );
    }
    if (rotF) rotate( rotV );
    if (scaF) scale( scaV );
    if (traF) translate( traV );
    if (matF) matrix( matV );
    if (clAtF) attrClear( clAtV );
    {  	int start, diff1 = 0, diff2 = 0;
	char *at, *cl;
	if ((start = RT_findString( argv, "-attrNames" )) >= 0) { 
	    if ( RT_getString( &argv[ start + 1 ], at, diff1) ) 
		if ( RT_getString( &argv[ start +2 ], cl, diff2) ) 
		    { attrNames( at, cl ); r++; }
	    if ( diff1 + diff2 < 2) rt_Output->errorVar( get_name(), ": Bad arguments for -attrNames.", 0 );
	    RT_clearArgv( argv, start, diff1 + diff2 );
	}
    }
    {  	int start, diff1 = 0, diff2 = 0;
	char *nm, *vl;
	if ((start = RT_findString( argv, "-attribute" )) >= 0) { 
	    if ( RT_getString( &argv[ start + 1 ], nm, diff1) ) 
		if ( RT_getString( &argv[ start +2 ], vl, diff2) ) {
		    RT_Attribute *at;		
		    if (at = rt_AttributeObject->create( nm, vl )) {
			attribute( *at ); r++; 			
		    } 
		    else rt_Output->errorVar( get_name(), ": Bad values: Attribute type: ", nm, ", and value: ", vl, ".", 0 );
		}
	    if ( diff1 + diff2 < 2) rt_Output->errorVar( get_name(), ": Bad arguments for -attribute.", 0 );
	    RT_clearArgv( argv, start, diff1 + diff2 );
	}
    }
    {  	int start, diff = 0;
	char *nm;
	if ((start = RT_findString( argv, "-get_attribute" )) >= 0) { 
	    if ( RT_getString( &argv[ start + 1 ], nm, diff) ) {

		// look for that attribute type to catch an exit on fatal error:
		RT_Attribute *at = rt_AttributeObject->create( nm, 0 );
		if (at) {
		    delete at; // 'cos it was dynamically created
		    // its a valid attribute type:
		    const RT_Attribute &t = get_attribute( nm );
		    RT_Object::result( t.get() ); r++;
		}
		else rt_Output->errorVar( get_name(), ": No such attribute type ", nm, "!", 0 );
	    }
	}
    }
    r +=  fatF + fatG + chiG + rotF + scaF + traF + matF + clAtF;
    if (matG) {
	static char tmp[360];
	RT_matrix2string( get_matrix(), tmp );
	RT_Object::result( tmp );
	r++;
    }
    if (wmatG) {
	static char tmp[360];
	RT_matrix2string( get_worldMatrix(), tmp );
	RT_Object::result( tmp );
	r++;
    }
    if (updF) update( 1 );
    if (visF) visible();
    if (invF) invisible();
    if (visG) RT_Object::result( get_visible() ? "1" : "0" ); 
    r += visF + invF + visG + updF;

    if (bndG) {
	static char tmp[60];
	RT_bounds2string( get_bounds(), tmp );
	RT_Object::result( tmp );
	r++;
    }
    if (hbdG) {
	static char tmp[60];
	RT_bounds2string( get_hierarchBounds(), tmp );
	RT_Object::result( tmp );
	r++;
    }
    if (bxF) {
	if ( bxV >= RTE_NO_BOX && bxV <= RTE_FILLED_HIERARCHICAL_BOX ) {
	    box( (RT_BoxMode)bxV ) ;
	    r++;
	}
	else rt_Output->errorVar( get_name(), ": Bad value for boxmode.", 0);
    }
    if (bxG) {
	static char tmp[5];
	RT_int2string( get_box(), tmp );
	RT_Object::result( tmp );
	r++ ; 
    }
    r += hdF + unF + hdG;
    if (hdF) hide();
    if (unF) unhide();
    if (hdG) RT_Object::result( get_hide() ? "1" : "0" ); 

    if (cpF) {
	RT_Object *obj = RT_Object::getObject( cpV );
	if (!obj || !(obj->isA( RTN_PRIMITIVE ))) 
	    rt_Output->errorVar( get_name(), ": No such primitive: ", cpV, "!", 0 );
	else { RT_Object::result( copy( (RT_Primitive*)obj ) ? "1" : "0" ); r++; }
    }

    if (pmF) {
	if ( pmV >= RTE_NOT_PICKABLE && pmV <= RTE_PICK_FATHER ) {
	    pickmode( (RT_Pickability)pmV ) ;
	    r++;
	}
	else {
	    rt_Output->errorVar( get_name(), ": Bad value for pick mode.", 0);
	}
    }
    if (pmG) {
	static char tmp[5];
	RT_int2string( get_pickmode(), tmp );
	RT_Object::result( tmp );
	r++ ; 
    }
    r += parseAttributes( argv );
    return r;
}

RT_Primitive::~RT_Primitive() {
    // clear modelling matrices if there are any
    if (hasModel) delete model;
  
    if (!isPart) rt_Objects->specialRemove( RTN_SCENE, this );
    attributes->clear();
    delete attributes; 
    // remove child from father (if there is one):
    if (xfather) xfather->removeChild( this );

    // destroy the kids:
    class LFunctoid: public RT_GeneralListFunctoid {
      public:
	void exec(RT_GeneralListEntry *e, void * = 0) { delete e; }
    } func;
    parts.doWithElements( &func );
    children.doWithElements( &func );

    // delete the related display list if there is one:
    RT_dlDelete( (long)this );
}

class RT_ReferenceFunc: public RT_GeneralListFunctoid {
    RT_AttributeList *list;
  public:
    RT_ReferenceFunc(  RT_AttributeList *_list) { list = _list; }
    void exec(RT_GeneralListEntry *e, void * = 0) {
	if ( (e->isA( RTN_PRIMITIVE ))) ((RT_Primitive*)e)->referencing( list );
    }
};

void RT_Primitive::referencing( const RT_AttributeList *list) {
    if ( !list && !attrsChangedFlag ) return;
    // referencing only object trees that are changed
    // if list is zero it is a top (as seen from the scene)
#ifdef RTD_UPDATE_DEBUG 
    fprintf( stderr, "Referencing object %s (%i)\n", get_name(), this );
#endif
    RT_AttributeList alist( this );
    // merge the overgiven attribute list into the temporary list:
    if (list) alist.merge( list );
    else {
	// else (the referencing starts at this primitive)
	// write the default values into the list:
	alist.insert( &RT_ResolutionAttribute::Default );
	alist.insert( &RT_SurfaceAttribute::Default );
	alist.insert( &RT_FillstyleAttribute::Default );
	alist.insert( &RT_MappingAttribute::Default );
    }
    // merge the private attribute list into the temporary list:
    alist.merge( attributes );
    createReferences( alist );
    // referencing the parts and childs, too:
    RT_ReferenceFunc func( &alist );
    parts.doWithElements( &func );
    children.doWithElements( &func );
}

void RT_Primitive::primChanged() {
    dlChangedFlag = 1;
#ifdef RTD_RSY
    rsyChangedFlag = 1;
#endif
    // recursive up transfer
    if ( isPart && get_father()) get_father()->primChanged();
}

void RT_Primitive::update(int cr) {
    if (cr) {
	createReferences( *attributes );
	// referencing the parts and childs, too:
	RT_ReferenceFunc func( attributes );
	parts.doWithElements( &func );
	children.doWithElements( &func );
    }
    
    // ckeck the new attributes:
    checkAttributes();
    attrsChangedFlag = 0;

    if (matrixChangedFlag) {

	newMatrix();

	class LFunctoid: public RT_GeneralListFunctoid {
	    RT_Primitive &xfather;
	  public:
	    LFunctoid( RT_Primitive &f): xfather( f ) {}
	    void exec(RT_GeneralListEntry *e, void * = 0) {
		((RT_Primitive*)e)->changedFatherModelling( xfather.getModel() );
	    }
	} func( *this );
	
	children.doWithElements( &func );
	if (partModelling) parts.doWithElements( &func );
	
	matrixChangedFlag = 0;
    }

    // recreate the prim if its geometry has changed:
    if (geomChangedFlag) create();
    geomChangedFlag = 0;

    class LFunctoid1: public RT_GeneralListFunctoid {
      public:
	void exec(RT_GeneralListEntry *e, void * = 0) {
	    ((RT_Primitive*)e)->update(); 
	}
    } func1;

    parts.doWithElements( &func1 );
    children.doWithElements( &func1 );

    // reset the state of the changed attributes:
    class LFunctoid2: public RT_GeneralListFunctoid {
	void exec(RT_GeneralListEntry *e, void * = 0) {
	    ((RT_Attribute*)e)->reset(); 
	}
    } func2;
    attributes->doWithElements( &func2 );
}

class RT_IntersectFunc: public RT_GeneralListFunctoid {
    int res;
    const RT_Ray &ray;
    RT_InterSectionList &list;
  public:
    RT_IntersectFunc( const RT_Ray &_ray, RT_InterSectionList &_list)
	: ray( _ray ), list( _list) { res = 0;}
    void exec(RT_GeneralListEntry *e, void * = 0) 
	{ res += ((RT_Primitive*)e)->intersectHierarchical( ray, list ); }
    int get() { return res; }
};

int RT_Primitive::intersectHierarchical(const RT_Ray &ray, RT_InterSectionList &list ) {
    if (!get_visible() || !rayInBounds( ray )) return 0;
    int n = intersect( ray, list);
    RT_IntersectFunc func( ray, list );
    children.doWithElements( &func );
    n += func.get();
    return n;
}

int RT_Primitive::intersect(const RT_Ray &ray, RT_InterSectionList &list) {
    RT_IntersectFunc func( ray, list );
    parts.doWithElements( &func );
    return func.get();
}

class RT_RenderHierFunc: public RT_GeneralListFunctoid {
    void exec(RT_GeneralListEntry *e, void * = 0) 
	{ ((RT_Primitive*)e)->renderHierarchical(); }
};

void RT_Primitive::render() {
    RT_RenderHierFunc func;
    parts.doWithElements( &func );
}

static RT_Quader *boundingBox = 0; 

void RT_Primitive::renderHierarchical() {
    if ( isPart && get_father()) { 
	//  render the object into the display list of the father:
	if (get_visible() && (xbox < RTE_FILLED_BOX )) {
	    // do not draw the object if it has a FILLED bounding box 
	    
	    if (hasModel) RT_dlMatrix( model->get_matrix() );
	    // if the local modelling matrix is equal to 1
	    // dont write the matrix element
	    render();
	    if (hasModel) RT_dlUnMatrix( );
	}
    }
    else {
	// the object owns a private display list 
	
	// create the display list if changed:
	if ( dlChangedFlag ) {
#ifdef RTD_UPDATE_DEBUG 
	    fprintf( stderr, "Create display list of object %s.\n", get_name() );
#endif
	    RT_dlOpen((long)this);
	    render();
	    RT_dlClose((long)this);
	}
	
	// call the display list:
	if ( get_visible() && (xbox < RTE_FILLED_BOX )) {
	    RT_dlMatrix( model->get_concatMatrix() );
	    RT_dlSurface( get_surface() );
	    RT_dlCall( (long)this );
	    RT_dlUnMatrix();
	}
	
	// draw the children:
	if (!isPart && get_visible() && (xbox < RTE_FILLED_HIERARCHICAL_BOX )) {
	    RT_RenderHierFunc func;
	    children.doWithElements( &func );
	}
	
	// draw the bounding box:
	if (xbox != RTE_NO_BOX) {
	    // recompute bounding box:
	    RT_Bounds b;
	    if ((xbox == RTE_OBJECT_BOX ) || (xbox == RTE_FILLED_BOX ))
		b = get_bounds();
	    else
		b = get_hierarchBounds();

	    RT_Vector pt((b.max.x + b.min.x) * 0.5,
			 (b.max.y + b.min.y) * 0.5,
			 (b.max.z + b.min.z) * 0.5 );  
	    
	    boundingBox->xdim( b.max.x - b.min.x );
	    boundingBox->ydim( b.max.y - b.min.y );
	    boundingBox->zdim( b.max.z - b.min.z );  

	    boundingBox->changedFatherModelling( model );
	    boundingBox->matrix( RT_IDENTITY );
	    boundingBox->translate( pt );
	    boundingBox->attribute( RT_FillstyleAttribute( (xbox < RTE_FILLED_BOX) ? 0 : 1) );
	    boundingBox->surface( get_surface() );
	    boundingBox->update( 1 );
	    boundingBox->renderHierarchical();
	}
    }
    dlChangedFlag = 0;
}

void RT_Primitive::printHierarchical(FILE *f) {
    print( f );

    class LFunctoid: public RT_GeneralListFunctoid {
	FILE *f;
      public:
	LFunctoid( FILE *_f) { f = _f; }
	void exec(RT_GeneralListEntry *e, void * = 0) 
	    { ((RT_Primitive*)e)->printHierarchical( f ); }
    } func( f );

    children.doWithElements( &func );
}

void RT_Primitive::attrsChanged() {
    if (attrsChangedFlag) return;
    attrsChangedFlag = 1;
    if (xfather) xfather->attrsChanged();
}

void RT_Primitive::matrixChanged() { 
    if (matrixChangedFlag) return;
    matrixChangedFlag = 1; 
    if (isPart && get_father()) get_father()->primChanged(); 
#ifdef RTD_RSY
    rsyChangedFlag = 1;
#endif
}

void RT_Primitive::geomChanged() { 
    if (geomChangedFlag) return;
    if (isPart && get_father()) get_father()->geomChanged(); 
    geomChangedFlag = 1; 
    primChanged(); 
}

void RT_Primitive::attribute( const RT_Attribute &a) {
    RT_Attribute *t = attributes->get( a.getClass() );
    if (!t) attributes->append( t = rt_AttributeObject->create( a.getClass(), 0 ));
    *t = a; // call overloaded operator
    attrsChanged();
}

const RT_Attribute &RT_Primitive::get_attribute(const char *cls) const {
    const RT_Attribute *at = attributes->retraverse( cls, this );
    if (!at) rt_Output->fatalVar( get_name(), ": No attribute type ", cls, " installed!", 0 );
    return *at;
}

void RT_Primitive::attrNames(const char *at, const  char *cl) {
    RT_Attribute *t = attributes->get( at );
    if (t) {
	t->setNamesList( cl );
	attrsChanged();
    }
    else rt_Output->warningVar( get_name(), " hasn't an attribute of type ", at, ".", 0 );
}

void RT_Primitive::attrClear( const char *at ) { 
    RT_Attribute *attr = attributes->get( at );
    if (attr) {
	attributes->remove( attr ); 
	delete attr;
	attrsChanged();
    }
    else rt_Output->warningVar( get_name(), " hasn't an attribute of type ", at, ".", 0 );
}

void RT_Primitive::createReferences(const RT_AttributeList &list) {
    RT_SurfaceAttribute *tmp = xsurface;
    xsurface = (RT_SurfaceAttribute*)list.retraverse( RTN_SURFACE_ATTR, this );
    if ( (tmp != xsurface) || ( *tmp != *xsurface)) primChanged(); 
    // create mapping reference:
    RT_MappingAttribute *tmp2 = xmapping;
    xmapping = (RT_MappingAttribute*)list.retraverse( RTN_MAPPING_ATTR, this );
    if ( (tmp2 != xmapping) || ( *tmp2 != *xmapping)) primChanged(); 
}

void RT_Primitive::checkAttributes() { if (xsurface->isChanged()) primChanged(); } 

RT_Primitive::RT_Primitive(char *_name ): RT_Object(_name) { 
    // initialize the flags:
    xhide = 0;
    xvis = 1;
    isPart = _name ? 0 : 1;
    
    xpick = isPart ? RTE_PICK_FATHER : RTE_PICKABLE;
    
    if (!boundingBox) {
	// avoid never ending recursion:
	boundingBox = (RT_Quader*)1;
	boundingBox = new RT_Quader( 0, 1, 1, 1);
	boundingBox->createModel();
    }
    
    dlChangedFlag = 1;
#ifdef RTD_RSY
    rsyChangedFlag = 1;
#endif
    matrixChangedFlag = 0;
    attrsChangedFlag = 1;
    geomChangedFlag = 1;
    partModelling = 0;
    
    xfather = 0;
    attributes = new RT_AttributeList( this );
    xsurface = &RT_SurfaceAttribute::Default;
    xmapping = &RT_MappingAttribute::Default;
    xbox = RTE_NO_BOX;
    
    model = (RT_Modelling*)&rt_Modelling;
    hasModel = 0;
}

class RT_GetBoundingBoxFunc:public RT_GeneralListFunctoid {
    RT_Bounds bnds;
    int frst;
    int hier;
    // equal to 1 if Hierarchical bounding box is requested
    void exec(RT_GeneralListEntry *, void * );
  public:
    RT_GetBoundingBoxFunc(RT_Bounds *b = 0) {
	if (b) {
	    frst = 0;
	    bnds = *b;
	    hier = 1;
	}
	else {
	    frst = 1; 
	    hier = 0;
	}
    }
    RT_Bounds get() { return bnds; }
};

void  RT_GetBoundingBoxFunc::exec(RT_GeneralListEntry *e, void * ) {
    if ( !(e->isA( RTN_PRIMITIVE ))) return;
    RT_Primitive *p = (RT_Primitive*)e;
    if (frst) {
	frst = 0;
	bnds = p->mobox();
    }
    else {
	RT_Bounds tbnds = p->mobox();
	if (tbnds.min.x < bnds.min.x) bnds.min.x = tbnds.min.x;
	if (tbnds.min.y < bnds.min.y) bnds.min.y = tbnds.min.y;
	if (tbnds.min.z < bnds.min.z) bnds.min.z = tbnds.min.z;
	if (tbnds.max.x > bnds.max.x) bnds.max.x = tbnds.max.x;
	if (tbnds.max.y > bnds.max.y) bnds.max.y = tbnds.max.y;
	if (tbnds.max.z > bnds.max.z) bnds.max.z = tbnds.max.z;
    }
}

RT_Bounds RT_Primitive::get_bounds() {
    RT_GetBoundingBoxFunc func;
    parts.doWithElements( &func );
    return func.get();
} 

RT_Bounds RT_Primitive::get_hierarchBounds() {
#define RTM_boundingBox(xx,yy,zz) \
    wcbp = mc2wc( RT_Vector(xx, yy, zz ));\
    if (wcbp.x < wcbounds.min.x) wcbounds.min.x = wcbp.x;\
    if (wcbp.x > wcbounds.max.x) wcbounds.max.x = wcbp.x;\
    if (wcbp.y < wcbounds.min.y) wcbounds.min.y = wcbp.y;\
    if (wcbp.y > wcbounds.max.y) wcbounds.max.y = wcbp.y;\
    if (wcbp.z < wcbounds.min.z) wcbounds.min.z = wcbp.z;\
    if (wcbp.z > wcbounds.max.z) wcbounds.max.z = wcbp.z

    RT_Bounds b = get_bounds();
    RT_GetBoundingBoxFunc func( &b );
    children.doWithElements( &func );
    RT_Bounds bnds = func.get();
    // convert the hierarchical bbox into a WC box used for raytracer optimization:
    RT_Vector wcbp;
    wcbp = mc2wc( RT_Vector(bnds.min.x, bnds.min.y, bnds.min.z ));
    wcbounds.min = wcbp; wcbounds.max = wcbp; 
    RTM_boundingBox(bnds.min.x, bnds.min.y, bnds.max.z );
    RTM_boundingBox(bnds.min.x, bnds.max.y, bnds.min.z );
    RTM_boundingBox(bnds.min.x, bnds.max.y, bnds.max.z );
    RTM_boundingBox(bnds.max.x, bnds.min.y, bnds.min.z );
    RTM_boundingBox(bnds.max.x, bnds.min.y, bnds.max.z );
    RTM_boundingBox(bnds.max.x, bnds.max.y, bnds.min.z );
    RTM_boundingBox(bnds.max.x, bnds.max.y, bnds.max.z );
    return bnds;
}

RT_Bounds RT_Primitive::mobox() {
    RT_Bounds b;
    b = get_hierarchBounds();
    RT_Vector va[8];
    va[0] = RT_Vector( b.min.x, b.min.y, b.min.z );
    va[1] = RT_Vector( b.min.x, b.min.y, b.max.z );
    va[2] = RT_Vector( b.min.x, b.max.y, b.min.z );
    va[3] = RT_Vector( b.min.x, b.max.y, b.max.z );
    va[4] = RT_Vector( b.max.x, b.min.y, b.min.z );
    va[5] = RT_Vector( b.max.x, b.min.y, b.max.z );
    va[6] = RT_Vector( b.max.x, b.max.y, b.min.z );
    va[7] = RT_Vector( b.max.x, b.max.y, b.max.z );
    if (hasModel) {
	// compute modelling offset:
	const RT_Matrix &mat = model->get_matrix();
	for (int i = 0; i < 8; i++) va[i] = mat * va[i];
    }
    b.min.x =  b.max.x = va[0].x; b.min.y = b.max.y = va[0].y; b.min.z = b.max.z = va[0].z;
    for ( int i = 1; i < 8; i++) {
	if (va[i].x < b.min.x) b.min.x = va[i].x;
	if (va[i].x > b.max.x) b.max.x = va[i].x;
	if (va[i].y < b.min.y) b.min.y = va[i].y;
	if (va[i].y > b.max.y) b.max.y = va[i].y;
	if (va[i].z < b.min.z) b.min.z = va[i].z;
	if (va[i].z > b.max.z) b.max.z = va[i].z;
    }
    return b;
}

void RT_Primitive::mappedSurface(const RT_Vector &p, RT_Surface &s) {
    ((RT_Mapping*)*xmapping)->smapping( this, p, s );
}

void RT_Primitive::mappedNormal(const RT_Vector &p, RT_Vector &n) {
    normal( p, n );
    ((RT_Mapping*)*xmapping)->nmapping( this, p, n );
}

int RT_Primitive::rayInBounds(const RT_Ray &ray) {
    double t[6];
    double tt, u;
    int tcnt = 0;
    
    if ( ray.dir.x != 0. ) {
	tt = ( wcbounds.min.x - ray.pt.x ) / ray.dir.x;
	u = ray.pt.y + ray.dir.y * tt;
	if (!( wcbounds.min.y > u || wcbounds.max.y < u)) {
	    u = ray.pt.z + ray.dir.z * tt;
	    if (!( wcbounds.min.z > u || wcbounds.max.z < u)) 
		t[tcnt++] = tt;
	}
	tt = ( wcbounds.max.x - ray.pt.x ) / ray.dir.x;
	u = ray.pt.y + ray.dir.y * tt;
	if (!( wcbounds.min.y > u || wcbounds.max.y < u)) {
	    u = ray.pt.z + ray.dir.z * tt;
	    if (!( wcbounds.min.z > u || wcbounds.max.z < u)) 
		t[tcnt++] = tt;
	}
    }	

    if ( ray.dir.y != 0. ) {
	tt = ( wcbounds.min.y - ray.pt.y ) / ray.dir.y;
	u = ray.pt.x + ray.dir.x * tt;
	if (!( wcbounds.min.x > u || wcbounds.max.x < u)) {
	    u = ray.pt.z + ray.dir.z * tt;
	    if (!( wcbounds.min.z > u || wcbounds.max.z < u)) 
		t[tcnt++] = tt;
	}
	tt = ( wcbounds.max.y - ray.pt.y ) / ray.dir.y;
	u = ray.pt.x + ray.dir.x * tt;
	if (!( wcbounds.min.x > u || wcbounds.max.x < u)) {
	    u = ray.pt.z + ray.dir.z * tt;
	    if (!( wcbounds.min.z > u || wcbounds.max.z < u)) 
		t[tcnt++] = tt;
	}
    }	
    
    if ( ray.dir.z != 0. ) {
	tt = ( wcbounds.min.z - ray.pt.z ) / ray.dir.z;
	u = ray.pt.x + ray.dir.x * tt;
	if (!( wcbounds.min.x > u || wcbounds.max.x < u)) {
	    u = ray.pt.y + ray.dir.y * tt;
	    if (!( wcbounds.min.y > u || wcbounds.max.y < u)) 
		t[tcnt++] = tt;
	}
	tt = ( wcbounds.max.z - ray.pt.z ) / ray.dir.z;
	u = ray.pt.x + ray.dir.x * tt;
	if (!( wcbounds.min.x > u || wcbounds.max.x < u)) {
	    u = ray.pt.y + ray.dir.y * tt;
	    if (!( wcbounds.min.y > u || wcbounds.max.y < u)) 
		t[tcnt++] = tt;
	}
    }

    if (!tcnt) return 0;
    
    double t1 = t[0];
    double t2 = t[0];
    
    for (int i = 1; i < tcnt; i++) { if (fabs(t[i] - t2) > rt_RayEps) { t2 = t[i]; break; } }
    return ray.valid( t1, t2 );
}

void RT_Primitive::createSurface() {
    primChanged();
    xsurface = (RT_SurfaceAttribute*)attributes->get( RTN_SURFACE_ATTR );
    if (xsurface) return;
    // create a private surface object if that doesnt exist already:
    attributes->insert( xsurface = new RT_SurfaceAttribute); 
}

class RT_TransferModelFunc: public RT_GeneralListFunctoid {
    RT_Modelling *model;
  public:
    RT_TransferModelFunc( RT_Modelling *_model) { model = _model; }
    void exec(RT_GeneralListEntry *e, void * = 0) {
	((RT_Primitive*)e)->replacedFatherModelling( model );
    }
};

void RT_Primitive::replacedFatherModelling( RT_Modelling *m) {
    if (!hasModel) {
	model = m;
	RT_TransferModelFunc func( model );
	children.doWithElements( &func );
	parts.doWithElements( &func );
    }
}

void RT_Primitive::setModel( RT_Modelling *m) {
    // clear the own model if there is one:
    if (hasModel) { delete model; hasModel = 0; }
    model = m;
    RT_TransferModelFunc func( model );
    children.doWithElements( &func );
    parts.doWithElements( &func );
    matrixChanged();
}

int RT_Primitive::parseAttributes(char *argv[]) {
    return rt_AttributeObject->parse( argv, this );
}

void RT_Primitive::print(FILE *f) const {
    printCon( f );
    if (xfather) fprintf( f, "\n%s -father %s", get_name(), xfather->get_name() );
    fprintf( f, "\n%s ", get_name() );
    if (!xvis) fprintf( f, "-invisible " );
    if (xhide) fprintf( f, "-hide " );
    if (xbox != RTE_NO_BOX ) fprintf( f, "-box %i ", xbox );
    if (xpick != RTE_PICKABLE) fprintf( f, "-pickmode %i ", xpick );
    if (hasModel) {
	const RT_Matrix &m = model->get_matrix();
	if (m.changed) { fprintf( f, "-matrix " ); m.print( f ); }
    }
    attributes->print( f);
}

