////////////////////////////////////////////////////////////////////////////////
//  Implementations of concrete image classes.                                //  
//  LAST EDIT: Fri Aug  5 08:55:01 1994 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRGHT which should be distributed with this  //
//  file. If COPYRGHT is not available or for more info please contact:       //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include "usefimag.h"

void rt_initImageCommands(Tcl_Interp *) {
    RTM_command( RTN_SGI_IMAGE, RT_SGIImage::classCMD );
    RTM_command( RTN_TARGA_IMAGE, RT_TargaImage::classCMD );
    RTM_command( RTN_RGBA_IMAGE, RT_RGBAImage::classCMD );
}

#ifndef RTD_CPP_INCLUDES
extern "C" {
#endif

#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>

#ifndef RTD_CPP_INCLUDES
}
#endif

// class SGI Image:

const char *RTN_SGI_IMAGE = "SGIImage";

#ifdef NEVER
// to test the read routine on a SGI use following stuff (inside the read method):
(#include "gl.h")
prefsize(xsize,ysize);
winopen( (char*)fname);
RGBmode();
gconfig();
reshapeviewport();
lrectwrite(0,0,xsize-1,ysize-1,base);
#endif

int RT_SGIImage::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) {
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String File} {Creates a new image that reads/writes an SGI RGB {ARG 2 File}. The created object is named {ARG 1 Image}.}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	if ( argc != 3 ) {
	    Tcl_AppendResult( ip, "Bad syntax. Must be: ", argv[0], " <name> <file>! ", NULL );
	    return TCL_ERROR;
	}
	new RT_SGIImage( argv[1], argv[2]);
	RTM_classReturn;
    } 
    return res;
}

RT_SGIImage::RT_SGIImage(char *_name, char *_filename): RT_Image( _name, _filename ) {
    base = 0; zsize = 3;
    // in following, a virtual method will be called
    // thus, a subclass of SGIImage shouldn't overload the read() method
    rt_Output->messageVar( get_name(), ": Trying to autoload image ", get_filename(), ".", 0 );
    read();
}

void RT_SGIImage::copyBw(long *lptr,int n) {
    while(n>=8) {
	lptr[0] = 0xff000000+(0x010101*(lptr[0]&0xff));
	lptr[1] = 0xff000000+(0x010101*(lptr[1]&0xff));
	lptr[2] = 0xff000000+(0x010101*(lptr[2]&0xff));
	lptr[3] = 0xff000000+(0x010101*(lptr[3]&0xff));
	lptr[4] = 0xff000000+(0x010101*(lptr[4]&0xff));
	lptr[5] = 0xff000000+(0x010101*(lptr[5]&0xff));
	lptr[6] = 0xff000000+(0x010101*(lptr[6]&0xff));
	lptr[7] = 0xff000000+(0x010101*(lptr[7]&0xff));
	lptr += 8;
	n-=8;
    }
    while(n--) {
	*lptr = 0xff000000+(0x010101*(*lptr&0xff));
	lptr++;
    }
}

void RT_SGIImage::setAlpha(unsigned char *lptr,int n) {
    while(n>=8) {
	lptr[0*4] = 0xff;
	lptr[1*4] = 0xff;
	lptr[2*4] = 0xff;
	lptr[3*4] = 0xff;
	lptr[4*4] = 0xff;
	lptr[5*4] = 0xff;
	lptr[6*4] = 0xff;
	lptr[7*4] = 0xff;
	lptr += 4*8;
	n -= 8;
    }
    while(n--) {
	*lptr = 0xff;
	lptr += 4;
    }
}

void RT_SGIImage::expandRow(unsigned char *optr,unsigned char *iptr,int z) {
    unsigned char pixel, count;
    optr += z;
    while(1) {
	pixel = *iptr++;
	if ( !(count = (pixel & 0x7f)) )
	    return;
	if(pixel & 0x80) {
	    while(count>=8) {
		optr[0*4] = iptr[0];
		optr[1*4] = iptr[1];
		optr[2*4] = iptr[2];
		optr[3*4] = iptr[3];
		optr[4*4] = iptr[4];
		optr[5*4] = iptr[5];
		optr[6*4] = iptr[6];
		optr[7*4] = iptr[7];
		optr += 8*4;
		iptr += 8;
		count -= 8;
	    }
	    while(count--) {
		*optr = *iptr++;
		optr+=4;
	    }
	} else {
	    pixel = *iptr++;
	    while(count>=8) {
		optr[0*4] = pixel;
		optr[1*4] = pixel;
		optr[2*4] = pixel;
		optr[3*4] = pixel;
		optr[4*4] = pixel;
		optr[5*4] = pixel;
		optr[6*4] = pixel;
		optr[7*4] = pixel;
		optr += 8*4;
		count -= 8;
	    }
	    while(count--) {
		*optr = pixel;
		optr+=4;
	    }
	}
    }
}

void RT_SGIImage::addLongImgTag(unsigned long *dptr, int xsize, int ysize) {
    dptr = dptr+(xsize*ysize);
    dptr[0] = 0x12345678;
    dptr[1] = 0x59493333;
    dptr[2] = 0x69434222;
    dptr[3] = xsize;
    dptr[4] = ysize;
}

unsigned short RT_SGIImage::getShort(FILE *inf) {
    unsigned char buf[2];
    fread(buf,2,1,inf);
    return (buf[0]<<8)+(buf[1]<<0);
}

unsigned long RT_SGIImage::getLong( FILE *inf) {
    unsigned char buf[4];
    fread(buf,4,1,inf);
    return (buf[0]<<24)+(buf[1]<<16)+(buf[2]<<8)+(buf[3]<<0);
}

void RT_SGIImage::putShort(FILE *outf, unsigned short val) {
    unsigned char buf[2];
    buf[0] = (val>>8);
    buf[1] = (val>>0);
    fwrite(buf,2,1,outf);
}

int RT_SGIImage::putLong(FILE *outf,unsigned long val ) {
    unsigned char buf[4];
    buf[0] = (unsigned char)(val>>24);
    buf[1] = (unsigned char)(val>>16);
    buf[2] = (unsigned char)(val>>8);
    buf[3] = (unsigned char)(val>>0);
    return fwrite(buf,4,1,outf);
}

void RT_SGIImage::readHeader(FILE *inf) {
    bzero(&image,sizeof( RT_SGIImageStruct ));
    image.imagic = getShort(inf);
    image.type = getShort(inf);
    image.dim = getShort(inf);
    image.xsize = getShort(inf);
    image.ysize = getShort(inf);
    image.zsize = getShort(inf);
}

int RT_SGIImage::writeHeader(FILE *outf) {
    RT_SGIImageStruct t;
    bzero(&t,sizeof( RT_SGIImageStruct ));
    // write out a zero field:
    fwrite(&t,sizeof( RT_SGIImageStruct ),1,outf);
    fseek(outf,0, SEEK_SET);
    putShort(outf,image.imagic);
    putShort(outf,image.type);
    putShort(outf,image.dim);
    putShort(outf,image.xsize);
    putShort(outf,image.ysize);
    putShort(outf,image.zsize); 
    putLong(outf,(unsigned char)image.min);
    putLong(outf,(unsigned char)image.max);
    putLong(outf,0);
    return fwrite("no name",8,1,outf);
}

int RT_SGIImage::writeTab(FILE *outf,unsigned long *tab,int len) {
    int r;
    while(len) {
	r = putLong(outf,*tab++);
	len -= 4;
    }
    return r;
}

void RT_SGIImage::readTab(FILE *inf,unsigned long *tab,int len) {
    while(len) {
	*tab++ = getLong(inf);
	len -= 4;
    }
}

void RT_SGIImage::lumRow(unsigned char *rgbptr,unsigned char *lumptr,int n) {
    lumptr += RT_SGI_CHANOFFSET(0);
    while(n--) {
	*lumptr = RT_SGI_ILUM(rgbptr[RT_SGI_OFFSET_R],rgbptr[RT_SGI_OFFSET_G],rgbptr[RT_SGI_OFFSET_B]);
	lumptr += 4;
	rgbptr += 4;
    }
}

int RT_SGIImage::compressRow(unsigned char *lbuf,unsigned char *rlebuf,int z,int cnt) {
    unsigned char *iptr, *ibufend, *sptr, *optr;
    short todo, cc;							
    long count;

    lbuf += z;
    iptr = lbuf;
    ibufend = iptr+cnt*4;
    optr = rlebuf;

    while(iptr<ibufend) {
	sptr = iptr;
	iptr += 8;
	while((iptr<ibufend)&& ((iptr[-8]!=iptr[-4])||(iptr[-4]!=iptr[0])))
	    iptr+=4;
	iptr -= 8;
	count = (iptr-sptr)/4;
	while(count) {
	    todo = count > 126 ? 126 : (short)count;
	    count -= todo;
	    *optr++ = 0x80|todo;
	    while(todo>8) {
		optr[0] = sptr[0*4];
		optr[1] = sptr[1*4];
		optr[2] = sptr[2*4];
		optr[3] = sptr[3*4];
		optr[4] = sptr[4*4];
		optr[5] = sptr[5*4];
		optr[6] = sptr[6*4];
		optr[7] = sptr[7*4];
		optr += 8;
		sptr += 8*4;
		todo -= 8;
	    }
	    while(todo--) {
		*optr++ = *sptr;
		sptr += 4;
	    }
	}
	sptr = iptr;
	cc = *iptr;
	iptr += 4;
	while( (iptr<ibufend) && (*iptr == cc) )
	    iptr += 4;
	count = (iptr-sptr)/4;
	while(count) {
	    todo = count > 126 ? 126 : (short)count;
	    count -= todo;
	    *optr++ = todo;
	    *optr++ = cc;
	}
    }
    *optr++ = 0;
    return optr - (unsigned char *)rlebuf;
}

int RT_SGIImage::sizeOfImage() {
    FILE *inf = fopen((char*)fname,"r");
    if(!inf) {
	rt_Output->warningVar( get_name(), ": Can't open image file ", (char*)fname, ".", 0 );
	return 0;
    }
    readHeader( inf );
    fclose(inf);
    if(image.imagic != RT_SGI_IMAGIC) {
	rt_Output->errorVar( get_name(), ": Bad magic number in image file ", (char*)fname, ".", 0 );
	return 0;
    }
    xsize = image.xsize;
    ysize = image.ysize;
    return 1;
}

int RT_SGIImage::read() {
    if (!sizeOfImage()) return 0;

    unsigned long *lptr;
    // the temporary pointer to the long array
    // will be initialized with this->base 
    
    FILE *inf = fopen((char*)fname,"r");
    if(!inf) {
	rt_Output->warningVar( get_name(), ": Can't open image file ", (char*)fname, ".", 0 );
	return 0;
    }
    readHeader(inf);
    if(image.imagic != RT_SGI_IMAGIC) {
	rt_Output->errorVar( get_name(), ": Bad magic number in image file ", (char*)fname, ".", 0 );
	return 0;
    }
    int rle = RT_SGI_ISRLE(image.type);
    int bpp = RT_SGI_BPP(image.type);
    if(bpp != 1 ) {
	rt_Output->errorVar( get_name(), ": Image must have 1 byte per pix chan.", 0 );
	return 0;
    }
    xsize = image.xsize;
    ysize = image.ysize;
    zsize = image.zsize;
    if(rle) {
	int tablen = ysize*zsize*sizeof(long);
	long *starttab = new long[tablen];
	long *lengthtab = new long[tablen];
	int rlebuflen = (int)1.05*xsize+10;
	unsigned char *rledat = new unsigned char [rlebuflen];
	fseek(inf,512,SEEK_SET);
 	readTab(inf,(unsigned long*)starttab,tablen);
	readTab(inf,(unsigned long*)lengthtab,tablen);

	// check data order:
	int cur = 0;
	int badorder = 0;
	for(int y = 0; y<ysize; y++) {
	    for(int z=0; z<zsize; z++) {
		if(starttab[y+z*ysize]<cur) {
		    badorder = 1;
		    break;
		}
		cur = (int)starttab[y+z*ysize];
	    }
	    if (badorder) break;
	}

	fseek(inf,512+2*tablen,SEEK_SET);
	cur = 512+2*tablen;

	if (base) delete base;
	base = new unsigned long [xsize*ysize+RT_SGI_TAGLEN];

	addLongImgTag(base,xsize,ysize);
  	if (badorder) {
	    for(int z=0; z<zsize; z++) {
		lptr = base;
		for(y = 0; y < ysize; y++) {
		    if(cur != starttab[y+z*ysize]) {
			fseek(inf,starttab[y+z*ysize],SEEK_SET);
			cur = (int)starttab[y+z*ysize];
		    }
		    if (lengthtab[y+z*ysize] > rlebuflen) {
			rt_Output->errorVar( get_name(), ": Rlebuf is too small - bad poop.", 0);
			fclose(inf);
			delete starttab;
			delete lengthtab;
			delete rledat;
			return 0;
		    }
		    fread( rledat,(unsigned int)lengthtab[y+z*ysize],1,inf);
		    cur += (int)lengthtab[y+z*ysize];
		    expandRow((unsigned char*)lptr,rledat,3-z);
		    lptr += xsize;
		}
	    }
	} else {
	    lptr = base;
	    for(y = 0; y < ysize; y++) {
		for(int z = 0; z < zsize; z++) {
		    if(cur != starttab[y+z*ysize]) {
			fseek(inf,starttab[y+z*ysize],SEEK_SET);
			cur = (int)starttab[y+z*ysize];
		    }
		    fread(rledat,(unsigned int)lengthtab[y+z*ysize],1,inf);
		    cur += (int)lengthtab[y+z*ysize];
		    expandRow((unsigned char*)lptr,rledat,3-z);
		}
		lptr += xsize;
	    }
    	}

	if (zsize == 3) setAlpha((unsigned char*)base,xsize*ysize);
	else if(zsize<3) copyBw((long*)base,xsize*ysize);

	fclose(inf);
	delete starttab;
	delete lengthtab;
	delete rledat;
	return 1;
    } else {
	if (base) delete base;
	base = new unsigned long[xsize*ysize+RT_SGI_TAGLEN];

	addLongImgTag(base,xsize,ysize);
	unsigned char *verdat = new unsigned char [xsize];
	fseek(inf,512,SEEK_SET);
	for(int z = 0; z < zsize; z++) {
	    lptr = base;
	    for(int y = 0; y < ysize; y++) {
		fread(verdat,xsize,1,inf);
		interleaveRow((unsigned char*)lptr,verdat,3-z,xsize);
		lptr += xsize;
	    }
	}
	if(zsize == 3) setAlpha((unsigned char*)base,xsize*ysize);
	else if(zsize<3) copyBw((long*)base,xsize*ysize);
	fclose(inf);
	delete verdat;
	return 1;
    }
}

int RT_SGIImage::write() {
    FILE *outf;
    int tablen, y, z, pos, len;
    long *starttab, *lengthtab;
    unsigned char *rlebuf;
    unsigned long *lptr, *lumbuf;
    int rlebuflen, goodwrite;

    lptr = base;
    goodwrite = 1;
    outf = fopen((char*)fname,"w");
    if(!outf) {
	rt_Output->errorVar( get_name(), ": Can't open file ", (char*)fname, " for writing.", 0 );
	return 0;
    }
    tablen = ysize*zsize*sizeof(long);

    starttab = new long [tablen];
    lengthtab = new long [tablen];
    rlebuflen = (int)1.05*xsize+10;
    rlebuf = new unsigned char [rlebuflen];
    lumbuf = new unsigned long [xsize];
    bzero(&image, sizeof( RT_SGIImageStruct ));
    image.imagic = RT_SGI_IMAGIC; 

    image.type = RT_SGI_RLE(1); 
    if(zsize>1) image.dim = 3;
    else image.dim = 2;
    image.xsize = xsize;
    image.ysize = ysize;
    image.zsize = zsize;
    image.min = 0;
    image.max = 255;
    goodwrite *= writeHeader(outf);

    fseek(outf,512+2*tablen,SEEK_SET);
    
    pos = 512+2*tablen;
    for(y=0; y<ysize; y++) {
	for(z=0; z<zsize; z++) {
	    if(zsize == 1) {
		lumRow((unsigned char*)lptr,(unsigned char*)lumbuf,xsize);
		len = compressRow((unsigned char*)lumbuf,rlebuf,RT_SGI_CHANOFFSET(z),xsize);
	    } else {
		len = compressRow((unsigned char*)lptr,rlebuf,RT_SGI_CHANOFFSET(z),xsize);
	    }
	    if(len>rlebuflen) {
		rt_Output->errorVar( get_name(), ": Rlebuf is too small - bad poop.", 0);
		delete starttab;
		delete lengthtab;
		delete rlebuf;
		delete lumbuf;
		return 0;
	    }
	    goodwrite *= fwrite(rlebuf,len,1,outf);
	    starttab[y+z*ysize] = pos;
	    lengthtab[y+z*ysize] = len;
	    pos += len;
	}
	lptr += xsize;
    }

    fseek(outf,512,SEEK_SET);
    goodwrite *= writeTab(outf,(unsigned long*)starttab,tablen);
    goodwrite *= writeTab(outf,(unsigned long*)lengthtab,tablen);
    delete starttab;
    delete lengthtab;
    delete rlebuf;
    delete lumbuf;
    fclose(outf);
    if (goodwrite) return 1;
    else {
	rt_Output->errorVar( get_name(), ": Not enough space for image!", 0);
	return 0;
    }
}

void RT_SGIImage::color(int x, int y, const RT_Color &c) {
    if (!checkIndices( x, y )) return;
    long l;
    unsigned char *cc = (unsigned char*)&l;
    cc[3] = (char) (c.r * 255);
    cc[2] = (char) (c.g * 255);
    cc[1] = (char) (c.b * 255);
    cc[0] = 255;
    base[ y * xsize + x] = l;
}

RT_Color RT_SGIImage::get_color(int x, int y) const { 
    RT_Color c;
    if (!checkIndices( x, y )) return c;
    unsigned char *cc = (unsigned char*)&base[ y * xsize + x];
    c.r = ((double)cc[3])/255.0;
    c.g = ((double)cc[2])/255.0;
    c.b = ((double)cc[1])/255.0;
    return c; 
}

// TARGA IMAGE:

const char *RTN_TARGA_IMAGE = "TargaImage";

RT_TargaImage::RT_TargaImage(char *a, char *b) : RT_Image( a, b ) { 
  base = 0;
  unsigned int test = 1;
  unsigned char *ctest = (unsigned char*)&test;
  lh = *ctest;  //short integer : low-byte - high-byte order ?  
  
  // in following, a virtual method will be called
  // thus, a subclass of TargaImage shouldn't overload the read() method
  rt_Output->messageVar( get_name(), ": Trying to autoload image ", get_filename(), ".", 0 );
  read();
}

int RT_TargaImage::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) {
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String File} {Creates a new image that reads (un- & compressed) / writes (compressed) Targa RGB {ARG 2 File}. The created object is named {ARG 1 Image}.}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	if ( argc != 3 ) {
	    Tcl_AppendResult( ip, "Bad syntax. Must be: ", argv[0], " <name> <file>! ", NULL );
	    return TCL_ERROR;
	}
	new RT_TargaImage( argv[1], argv[2]);
	RTM_classReturn;
    } 
    return res;
}

unsigned short RT_TargaImage::getShort(FILE *inf) {
    unsigned char *cval, buf[2];
    unsigned int val;
    cval = (unsigned char*)&val;
    fread(buf, 2, 1, inf);
    if(lh) { 
       cval[0] = buf[0]; cval[1] = buf[1];
    } else {
       cval[1] = buf[0]; cval[0] = buf[1]; // swap low / high
    }
    return val;   
}

void RT_TargaImage::putShort(FILE *outf, unsigned short val) {
  unsigned char *cval, buf[2];
  cval = (unsigned char*)&val;
  if(lh) { 
    buf[0] = cval[0]; buf[1] = cval[1];
  } else {
    buf[0] = cval[1]; buf[1] = cval[0]; // swap low / high
  }
  fwrite(buf, 2, 1, outf);
}

void RT_TargaImage::readHeader(FILE *inf) {
  fread(&Header, 1, 3, inf);
  Header.cmap_origin = getShort(inf);
  Header.cmap_len = getShort(inf);
  fread(&Header.centry_size, 1, 1, inf);              
  Header.image_xorg = getShort(inf);
  Header.image_yorg = getShort(inf); 
  Header.image_width = getShort(inf); 
  Header.image_height = getShort(inf); 
  fread(&Header.pixel_size, 2, 1, inf);      
}

int RT_TargaImage::writeHeader(FILE *outf) {
  Header.ident_len    = 8;   
  Header.cmap_type    = 0;   // no color-table
  Header.image_type   = 10;  // compressed RGB
  Header.cmap_origin  = 0;
  Header.cmap_len     = 0;
  Header.centry_size  = 0;
  Header.image_xorg   = 0;
  Header.image_yorg   = 0;
  Header.image_width  = xsize;
  Header.image_height = ysize;
  Header.pixel_size   = 24;  // 24 bit / pixel
  Header.image_discr  = 0;   // left -> right ; bottom -> up

  fwrite(&Header, 12, 1, outf); 
  putShort(outf, Header.image_width);
  putShort(outf, Header.image_height);
  fwrite(&Header.pixel_size, 2, 1, outf);        
  return fwrite("no name", 8, 1, outf);
}

int RT_TargaImage::sizeOfImage() {
    FILE *inf = fopen((char*)fname,"rb");
    if(!inf) {
	rt_Output->errorVar( get_name(), ": Can't open image file ", (char*)fname, ".", 0 );
	return 0;
    }
    readHeader(inf);
    fclose(inf);
    xsize = Header.image_width;
    ysize = Header.image_height;
    return 1;
}

int RT_TargaImage::read() {
    if (!sizeOfImage()) return 0;

    FILE *inf = fopen((char*)fname,"r");
    if(!inf) {
      rt_Output->errorVar( get_name(), ": Can't open image file ", (char*)fname, ".", 0 );
      return 0;
    }
    readHeader(inf);
    if(Header.cmap_type != 0 || (Header.image_type != 2 && Header.image_type != 10) ||
       Header.pixel_size != 24 || (Header.image_discr & 0xc0) != 0) {
       rt_Output->errorVar( get_name(), ": Can't read ", (char*)fname, ". (Today the TARGA image class is limited to RGB images (no color-table). This will be changed in later versions).", 0);    
       return 0;
    } 
    xsize = Header.image_width;
    ysize = Header.image_height;
	
    if (base) delete base;
    base = new unsigned long [xsize * ysize];
    
    if(Header.ident_len) {
      char ident[256];
      fread(ident, 1, Header.ident_len, inf); 
      ident[Header.ident_len] = '\0';
      rt_Output->messageVar( get_name(), ": ", ident, 0);
    } 
    
    long l; unsigned char *cc = (unsigned char*)&l;
    int top_down = Header.image_discr & 0x20;  
    int h, x, y = top_down ? ysize - 1 : 0;

    if(Header.image_type == 2) {            // uncompressed RGB
      RT_TargaRGB *pcnt, *pix_row = new RT_TargaRGB [xsize];
      
      for(h = 0; h < ysize; h++) {
        pcnt = pix_row; 
        fread(pcnt, sizeof(RT_TargaRGB), xsize, inf); 
        for(x = 0; x < xsize; x++) {
          cc[3] = pcnt->r; cc[2] = pcnt->g; cc[1] = pcnt->b; cc[0] = 255; 
          pcnt++;
          base[y * xsize + x] = l;
        }
        if(top_down) y--; else y++;
      }
      delete pix_row;
    }

    if(Header.image_type == 10) {            // compressed RGB
      RT_TargaRGB pix; unsigned char cnt; int mode = 0;
      for(h = 0; h < ysize; h++) {
        x = 0;
        while(x < xsize) { 
          if(!mode) {   
            fread(&cnt, 1, 1, inf); //read length       
            if(cnt & 0x80) {     //runlength coded
              fread(&pix, sizeof(RT_TargaRGB), 1, inf);
              cc[3] = pix.r; cc[2] = pix.g; cc[1] = pix.b; cc[0] = 255;
              cnt &= 0x7f; mode = 2;
            } else mode = 1;
          } else {
            if(mode == 1) {
              fread(&pix, sizeof(RT_TargaRGB), 1, inf);
              cc[3] = pix.r; cc[2] = pix.g; cc[1] = pix.b; cc[0] = 255;
            }
            base[y * xsize + x] = l;
            if(!cnt) mode = 0; else cnt--;
            x++;
          }
        }
        if(top_down) y--; else y++;   
      }
    }
    return 1;
}

int RT_TargaImage::compressRow(RT_TargaRGB *pix_row, unsigned char *rlebuf) {
  int off = 0, offp1 = 1, rleoff = 0; 

  while(off < xsize) {
    if(offp1 < xsize) {
      unsigned char cnt = 0;
      if(pix_row[off].r == pix_row[offp1].r && 
         pix_row[off].g == pix_row[offp1].g &&
         pix_row[off].b == pix_row[offp1].b) {
        while (pix_row[off].r == pix_row[offp1].r && 
               pix_row[off].g == pix_row[offp1].g &&
               pix_row[off].b == pix_row[offp1].b && cnt < 127) {
          off++; offp1++; cnt++; if(offp1 >= xsize) break; 
        } 
        cnt |= 0x80;
        rlebuf[rleoff++] = cnt;
        rlebuf[rleoff++] = pix_row[off].b;
        rlebuf[rleoff++] = pix_row[off].g;
        rlebuf[rleoff++] = pix_row[off].r;
        off++; offp1++;   
      } else {
        int rleoff1 = rleoff++;
        while ((pix_row[off].r != pix_row[offp1].r ||
                pix_row[off].g != pix_row[offp1].g ||
                pix_row[off].b != pix_row[offp1].b) && cnt < 127) {
          rlebuf[rleoff++] = pix_row[off].b;
          rlebuf[rleoff++] = pix_row[off].g;
          rlebuf[rleoff++] = pix_row[off].r;
          off++; offp1++; cnt++; if(offp1 >= xsize) break;
        }     
        rlebuf[rleoff1] = --cnt;
      }
    } else {
      rlebuf[rleoff++] = 0;
      rlebuf[rleoff++] = pix_row[off].b;
      rlebuf[rleoff++] = pix_row[off].g;
      rlebuf[rleoff++] = pix_row[off].r;
      off++; offp1++;
    }
  }
  return rleoff;   
}

int RT_TargaImage::write() {
    FILE *outf;
    unsigned long *lptr = base;
    int goodwrite = 1;

    outf = fopen((char*)fname, "wb");
    if(!outf) {
      rt_Output->errorVar( get_name(), ": Can't open file ", (char*)fname, " for writing.", 0 );
      return 0;
    }

    goodwrite *= writeHeader(outf);

    RT_TargaRGB *pcnt, *pix_row = new RT_TargaRGB [xsize];
    unsigned char *rlebuf = new unsigned char [xsize * sizeof(RT_TargaRGB) + xsize];      
    for(int y = 0; y < ysize; y++) {
      pcnt = pix_row;
      for(int x = 0; x < xsize; x++) {         
        unsigned char *cc = (unsigned char*)lptr++;
        pcnt->r = cc[3]; pcnt->g = cc[2]; pcnt->b = cc[1];
        pcnt++;
      } 
      int len = compressRow(pix_row, rlebuf);
      goodwrite *= fwrite(rlebuf, len, 1, outf);
    }     
    delete rlebuf; delete pix_row;
    fclose(outf);
 
    if (goodwrite) return 1;
    else {
      rt_Output->errorVar( get_name(), ": Not enough space for image!", 0);
      return 0;
    }
}

void RT_TargaImage::color(int x, int y, const RT_Color &c) {
    if (!checkIndices( x, y )) return;
    long l;
    unsigned char *cc = (unsigned char*)&l;
    cc[3] = (char) (c.r * 255);
    cc[2] = (char) (c.g * 255);
    cc[1] = (char) (c.b * 255);
    cc[0] = 255;
    base[ y * xsize + x] = l;
}

RT_Color RT_TargaImage::get_color(int x, int y) const { 
    RT_Color c;
    if (!checkIndices( x, y )) return c;
    unsigned char *cc = (unsigned char*)&base[ y * xsize + x];
    c.r = ((double)cc[3])/255.0;
    c.g = ((double)cc[2])/255.0;
    c.b = ((double)cc[1])/255.0;
    return c; 
}

// class RGAB Image:

const char *RTN_RGBA_IMAGE = "RGBAImage";

RT_RGBAImage::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) {
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String File} {Creates a new image that reads/writes an RGBA {ARG 2 File}. The created object is named {ARG 1 Image}.}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	if ( argc != 3 ) {
	    Tcl_AppendResult( ip, "Bad syntax. Must be: ", argv[0], " <name> <file>! ", NULL );
	    return TCL_ERROR;
	}
	new RT_RGBAImage( argv[1], argv[2]);
	RTM_classReturn;
    } 
    return res; 
}

unsigned short RT_RGBAImage::getShort( int fd ) {
    unsigned char buf[2];
    ::read( fd, buf, 2 );
    return (buf[0]<<8)+(buf[1]<<0);
}

unsigned long RT_RGBAImage::getLong( int fd ) {
    unsigned char buf[4];
    ::read( fd, buf, 4 );
    return (buf[0]<<24)+(buf[1]<<16)+(buf[2]<<8)+(buf[3]<<0);
}
 
int RT_RGBAImage::putShort( int fd, unsigned short val ) {
    unsigned char buf[2];
    buf[0] = (val>>8);
    buf[1] = (val>>0);
    return ::write( fd, buf, 2 );
}

int RT_RGBAImage::putLong(int fd,unsigned long val ) {
    unsigned char buf[4];
    buf[0] = (unsigned char)(val>>24);
    buf[1] = (unsigned char)(val>>16);
    buf[2] = (unsigned char)(val>>8);
    buf[3] = (unsigned char)(val>>0);
    return ::write( fd, buf, 4 );
}

void RT_RGBAImage::writeHeader( int fd ) {
    putShort( fd, image.imagic );
    putShort( fd, image.xsize );
    putShort( fd, image.ysize );
}

void RT_RGBAImage::readHeader( int fd ) {
    bzero( &image, sizeof( RT_RGBAImageHeader ) );
    image.imagic = getShort( fd );
    image.xsize = getShort( fd );
    image.ysize = getShort( fd );
}

int RT_RGBAImage::read() {
    int fd, size, ret;

    if ( base ) delete (base);
    if ( ( fd = open( (char*)fname, O_RDONLY, 0666 ) ) < 0 ) {
	rt_Output->errorVar( "Can't open file: ", get_filename(), "." );
	return 0;
    }
    
    // read header
    readHeader( fd );
    if( image.imagic != RT_RGBA_IMAGIC ) {
	rt_Output->errorVar( "Bad magic number in image file ", (char*)fname, ".", 0 );
	return 0;
    }
    
    // allocate data memory and read data
    size = xsize * ysize * 4;
    base = (unsigned long*) new char [size];
    assert( base != 0 );
    ret = ::read( fd, (char*)base, size );
    close( fd );
    if ( ret < size ) {
	rt_Output->errorVar( "Error while reading file: ", (char*)fname, "." );
	return 0;
    }
    return 1;
}

int RT_RGBAImage::write() {
    int fd;
    int size;
    int ret;

    if ( !base ) return 0;
    if (( fd = open( (char*)fname, O_WRONLY|O_CREAT | O_TRUNC, 0666 )) < 0 ) {
	rt_Output->errorVar( "Can't create file: ", get_filename(), "." );
	return 0;
    }

    // set the header info and write header
    bzero( &image, sizeof( RT_RGBAImageHeader ) );
    image.imagic = RT_RGBA_IMAGIC;
    image.xsize = xsize;
    image.ysize = ysize;
    writeHeader( fd );

    // dump image data
    size = xsize * ysize * 4;
    ret = ::write( fd, (char*)base, size );
    close ( fd );
    if ( ret < size ) {
	rt_Output->errorVar( "Error while writing file: ", (char*)fname, "." );
	return 0;
    }
    return 1;
}

void RT_RGBAImage::color( int x, int y, const RT_Color &c ) {
    int offset;
    char *src;
    
    y = ysize - y;
    if ( !checkIndices( x, y ) ) return;
    offset = (y * xsize + x) * 4;
    src = &((char*)base)[offset];
    ++src; *src = (char) (c.b * 255);
    ++src; *src = (char) (c.g * 255);
    ++src; *src = (char) (c.r * 255);
    return;
}

RT_Color RT_RGBAImage::get_color( int x, int y ) const {
    RT_Color c;
    char *src;
    int offset;

    y = ysize - y;
    if ( !checkIndices( x, y ) ) return c;
    offset = (y * xsize + x) * 4;
    src = &((char*)base)[offset];
    ++src; c.b = ((double)*src)/255.0; 
    ++src; c.g = ((double)*src)/255.0; 
    ++src; c.r = ((double)*src)/255.0; 
    return c;
}

