////////////////////////////////////////////////////////////////////////////////
//  implementation of text primitive.                                         //
//  LAST EDIT: Tue Nov 15 15:44:58 1994 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include "text.h"

const char *RTN_TEXT = "Text";

RT_FontServer rt_FontServer;

void RT_Text::font( char *_newFont ) {
    RT_Font *newFnt = rt_FontServer.getFont( _newFont );  
    // don't destroy the old font:
    if (!newFnt) return; 
    fnt = newFnt;
    if ( fontCreated ) {
	if (pnt) delete pnt; // if the text doesn't have a valid font -> break
	if (pvx) {
	    for (int i=0; i<maxlines; i++) if ( pvx[i] ) delete pvx[i]; 
	    delete pvx;
	}
    }
    maxpoints = fnt->get_maxPoints();
    maxlines = 0;
    int len = textstring.length();
    for (int n = 0; n < len; n++) {
	maxlines += fnt->get_lines(((char*)textstring)[n]); }
    pnt = new RT_Vector[maxpoints];
    pvx = new RT_Polyvertex*[maxlines];
    geomChanged();
    fontCreated = 0;          //  disable calls to get_width etc.
}

void RT_Text::text( char *_t ) {
    if ( fontCreated ) {
	if (pnt) delete pnt;        // delete the old text
	if (pvx) {
	    for (int i=0; i<maxlines; i++) if (pvx[i]) delete pvx[i]; 
	    delete pvx;
	}
    }
    textstring = _t;
    maxpoints = fnt->get_maxPoints();
    maxlines = 0;
    int len = textstring.length();
    for (int n = 0; n < len; n++) { maxlines += fnt->get_lines(((char*)textstring)[n]); }
    pnt = new RT_Vector[maxpoints];
    pvx = new RT_Polyvertex*[maxlines];
    geomChanged();
    fontCreated = 0;          //  disable calls to get_width etc.
}

void RT_Text::deletePoly() {
    if (pnt) delete pnt; 
    if (pvx) {
	for (int i=0; i<maxlines; i++) delete pvx[i]; 
	delete pvx;
    }
    maxpoints = fnt->get_maxPoints();
    maxlines = 0;
    int len = textstring.length();
    for (int n = 0; n < len; n++) {
	maxlines += fnt->get_lines(((char*)textstring)[n]); }
    pnt = new RT_Vector[maxpoints];
    pvx = new RT_Polyvertex*[maxlines];
    fontCreated = 0;
}

void RT_Text::create() {
    if (fnG) return;
    if (!fnt) return; // if the text doesn't have a valid font -> break
    int lnr = 0;
    float stringwidth = 0.0;
    int ft = fnt->get_type();
    int len = textstring.length();
    for (int n = 0; n < len; n++) {
	int ascii = ((char*)textstring)[n];
	for (int i = 0; i<fnt->get_lines(ascii); i++) {
	    int nrp = fnt->get_numberOfPoints( ascii, i );
	    for ( int k=0; k<nrp; k++ ) {
		pnt[k] = fnt->get_point(ascii, i, k );
		pnt[k].x += ( tpos.x + stringwidth  );
		pnt[k].y = tpos.y + (pnt[k].y - fnt->get_baseLineDist() ) * tHeight;
		pnt[k].z += tpos.z;
	    }
	    switch ( ft ) {
	      case 1: 
		pvx[lnr] = new RT_Polyline( 0 , nrp, pnt );
		break;
	      case 2:
		pvx[lnr] = new RT_Polygon( 0, nrp, pnt );
		((RT_Polygon*)pvx[lnr])->fillstyle( 1 );
		break;
	    }
     	    pvx[lnr]->gnormal( RT_Vector(0, 0, 1) );
	    pvx[lnr]->father(this);
	    pvx[lnr]->referencing( attributes );
	    lnr++;
	}
	stringwidth += ( fnt->get_charWidth(ascii) + fnt->get_spacing() );
    }
    fontCreated = 1;
    double wf = tWidth / get_width();
    if ( tWidth > 0.0 ) for ( int i=0; i<maxlines; i++) {
	int nr = pvx[i]->get_number();
	for ( int j=0; j<nr; j++ ) {
	    RT_Vector p = pvx[i]->get_vtPoint(j);
	    p.x = tpos.x + (p.x - tpos.x) * wf;
	    pvx[i]->vtPoint(j, p);
	}
    }
    double xshift, yshift;
    switch ( xAlign ) {
      case -1: xshift = 0.0;
	 break;
      case 0:
	 if ( tWidth > 0.0 ) xshift = -tWidth / 2.0;
	 else xshift = -get_width() /2.0;
	 break;
      case 1:
	 xshift = -get_width();
	 break;
    }
    switch ( yAlign ) {
      case -2:
	yshift = fnt->get_baseLineDist() * tHeight;
	break;
      case -1:
	yshift = 0.0;
	break;
      case 0:
	yshift = -fnt->get_capHeight() * tHeight / 2.0;
	break;
      case 1:
	yshift = -fnt->get_capHeight() * tHeight;
	break;
      case 2:
	yshift = -tHeight;
	break;
    }
    for ( int k=0; k<maxlines; k++ ) {
	int nr = pvx[k]->get_number();
	for ( int j=0; j<nr; j++ ) {
	    RT_Vector p = pvx[k]->get_vtPoint(j);
	    p.x += xshift;
	    p.y += yshift;
	    pvx[k]->vtPoint(j, p);
	}
    }
}

void RT_Text::xalign( int _align ) {
    if (( _align < (-1)) || ( _align > 1 )) {
	rt_Output->errorVar( " Sorry, xalign must be in range -1...1 !", 0);
	return;
    }
    if ( fontCreated ) deletePoly();
    xAlign = _align;
    geomChanged();
}

void RT_Text::yalign( int _align ) {
    if (( _align < (-2)) || ( _align > 2 )) {
	rt_Output->errorVar( " Sorry, yalign must be in range -2...2 !", 0);
	return;
    }
    if ( fontCreated ) deletePoly();
    yAlign = _align;
    geomChanged();
}

void RT_Text::zalign( int ) {
// reserved for further implementation of 3D-text
}

double RT_Text::get_width() const {
    if ( !fontCreated ) return 0.0;
    double xmin, xmax;
    RT_Vector p = pvx[0]->get_vtPoint(0);
    xmin = xmax = p.x;
    for (int i=0; i<maxlines; i++) {
	int nr = pvx[i]->get_number();
	for ( int j=0; j<nr; j++ ) {
	    p = pvx[i]->get_vtPoint(j);
	    xmin = p.x < xmin ? p.x : xmin;
	    xmax = p.x > xmax ? p.x : xmax;
	}
    }
    return (xmax - xmin);
}

void RT_Text::height( double _h ) {
    if ( _h <= 0.0 ) {
	rt_Output->errorVar(" Sorry, height must be greater than 0. Value ignored", 0);
	return;
    }
    if ( fontCreated ) deletePoly();
    tHeight = _h;
    geomChanged();
}

void RT_Text::width( double _w ) {
    if ( _w < 0.0 ) {
	rt_Output->errorVar(" Sorry, width must be not lower than 0. Value ignored", 0);
	return;
    }
    if ( fontCreated ) deletePoly();
    tWidth = _w;
    geomChanged();
}
    

void RT_Text::depth( double ) {
// reserved for further implementation of 3D-text
}

void RT_Text::pos( RT_Vector _pos ) {
    tpos = _pos;
    if ( fontCreated ) deletePoly();
    geomChanged();
}
 
char *RT_Text::fnV, *RT_Text::txtV;
int RT_Text::fnF, RT_Text::fnG, RT_Text::wG;
int RT_Text::xalF, RT_Text::xalV, RT_Text::yalV, RT_Text::yalF;
int RT_Text::thF, RT_Text::thG, RT_Text::twF, RT_Text::posG, RT_Text::posF;
int RT_Text::zalF, RT_Text::zalV, RT_Text::tdF, RT_Text::tdG, RT_Text::txtG;
int RT_Text::xalG, RT_Text::yalG, RT_Text::zalG, RT_Text::txtF;
double RT_Text::thV, RT_Text::twV, RT_Text::tdV;
RT_Vector RT_Text::posV;

RT_ParseEntry RT_Text::table[] = {
    { "-font", RTP_STRING, (char*)&fnV, &fnF, "Specify a new {ARG 1 Font} for the text.", RTPS_FONT },
    { "-get_font", RTP_NONE, 0, &fnG, "Get the font.", RTPS_NONE },
    { "-text", RTP_STRING, (char*)&txtV, &txtF, "Specify a new {ARG 1 Textstring} for the text.", RTPS_STRING },
    { "-height", RTP_DOUBLE, (char*)&thV, &thF, "Specify a new {ARG 1 Height} for the text.", RTPS_DOUBLE },
    { "-width", RTP_DOUBLE, (char*)&twV, &twF, "Specify a new {ARG 1 Width} for the text.", RTPS_DOUBLE },
    { "-depth", RTP_DOUBLE, (char*)&tdV, &tdF, "Specify a new {ARG 1 Depth} for the text.", RTPS_DOUBLE },
    { "-pos", RTP_VECTOR, (char*)&posV, &posF, "Specify a new {ARG 1 Startpoint} for the text.", RTPS_VECTOR },
    { "-get_pos", RTP_NONE, 0, &posG, "Get the text startpoint.", RTPS_NONE },
    { "-get_height", RTP_NONE, 0, &thG, "Get the text height.", RTPS_NONE },
    { "-get_width", RTP_NONE, 0, &wG, "Get the string width.", RTPS_NONE },
    { "-get_depth", RTP_NONE, 0, &tdG, "Get the string depth.", RTPS_NONE },
    { "-xalign", RTP_INTEGER, (char*)&xalV, &xalF, "Specify the {ARG 1 X-alignment} for the text.", RTPS_INTEGER},
    { "-yalign", RTP_INTEGER, (char*)&yalV, &yalF, "Specify the {ARG 1 Y-alignment} for the text.", RTPS_INTEGER },
    { "-zalign", RTP_INTEGER, (char*)&zalV, &zalF, "Specify the {ARG 1 Z-alignment} for the text.",RTPS_INTEGER },
    { "-get_xalign", RTP_NONE, 0, &xalG, "Get the X-alignment.", RTPS_NONE },
    { "-get_yalign", RTP_NONE, 0, &yalG, "Get the Y-alignment.", RTPS_NONE },
    { "-get_zalign", RTP_NONE, 0, &zalG, "Get the Z-alignment.", RTPS_NONE },
    { "-get_text", RTP_NONE, 0, &txtG, "Get the current textstring.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_Text::objectCMD(char *argv[]) {
    int t = RT_Primitive::objectCMD( argv );
    RT_parseTable( argv, table );
    if (fnF) { font(fnV); t++; }
    if (fnG) { RT_Object::result( get_font() ); t++; }
    if (txtF) { text(txtV); t++; }
    int ft;
    ft = fnt->get_type();
    if (posF) { pos( posV ); t++; }
    if (posG) {
	t++;
	char tmp[30];
	RT_vector2string( get_pos(), tmp );
	RT_Object::result( tmp );
    }
    if (thG) {
	t++; 
	char tmp[20];
	RT_double2string( get_height(), tmp );
	RT_Object::result( tmp );
    }
    if (wG) {
	t++; 
	char tmp[20];
	RT_double2string( get_width(), tmp );
	RT_Object::result( tmp );
    }
    if (tdG) {
	t++; 
	char tmp[20];
	RT_double2string( get_depth(), tmp );
	RT_Object::result( tmp );
    }
    if (xalG) {
	t++; 
	char tmp[20];
	RT_int2string( get_xalign(), tmp );
	RT_Object::result( tmp );
    }
    if (yalG) {
	t++; 
	char tmp[20];
	RT_int2string( get_yalign(), tmp );
	RT_Object::result( tmp );
    }
    if (zalG) {
	t++; 
	char tmp[20];
	RT_int2string( get_zalign(), tmp );
	RT_Object::result( tmp );
    }
    if (txtG) {
	t++; 
	RT_Object::result( get_text() );
    }
    if (xalF) { xalign( xalV ); t++; }
    if (yalF) { yalign( yalV ); t++; }
    if (zalF) { zalign( zalV ); t++; }
    if (thF)  { height( thV );  t++; }
    if (twF)  { width( twV );   t++; }
    if (tdF)  { depth( tdV );   t++; }
    return t;
}

int RT_Text::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res = _classCMD( cd, ip, argc, argv );
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String String Vector} {  Creates a new text, arguments are {ARG 1 Name}, {ARG 2 Text}, and {ARG 3 Startpoint}.}}", 0 );
	return TCL_OK;
    }
    if (res == TCL_OK) { 
	if (argc != 4) { 
	    Tcl_AppendResult( ip, "Bad number of arguments. Try \"", RTN_TEXT, " ?\".", 0 ); 
	    return TCL_ERROR;
	}
	RT_Vector pos;
	int dummy;
	int ret = TCL_ERROR;
	if (!strlen(argv[2])) {
	    rt_Output->errorVar( argv[0], ": No text string specified. Could not create one !", 0);
	    return ret;
	}
	if (!RT_getVector( &argv[3], pos, dummy)) {
	    rt_Output->errorVar( argv[0], ": Don't know where to create the text! Please specify the start point !", 0);
	    return ret;
	}
	new RT_Text( argv[1], argv[2], pos );
	RTM_classReturn;
    }
    return res;
}

void RT_Text::print(FILE *f) const { 
    RT_Primitive::print( f );
    fprintf( f, "%s ", get_name() );
    if ( xAlign != (-1) ) fprintf( f, "-xalign %i ", xAlign );
    if ( yAlign != (-1) ) fprintf( f, "-yalign %i ", yAlign );
    if ( zAlign != 0 ) fprintf( f, "-zalign %i ", zAlign );
    if ( tHeight != 1.0 ) fprintf( f, "-height %lf ", tHeight );
    if ( tDepth != 0.0 ) fprintf( f, "-depth %lf ", tDepth );
    if ( strcmp(get_font(), RTD_DEFAULT_FONT) )
	fprintf( f, "-font %s ", get_font() );
    if (tWidth > 0.0) fprintf( f, "-width %lf ", get_width() );
    fprintf( f, "\n");
}
