////////////////////////////////////////////////////////////////////////////////
// Height field implementation.                                               //  
// LAST EDIT: Wed Mar  8 08:52:47 1995 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRGHT which should be distributed with this  //
//  file. If COPYRGHT is not available or for more info please contact:       //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include "height.h"

const char *RTN_HEIGHT_FIELD = "HeightField";

RT_HeightField::RT_HeightField( char *na, int x, int y, const double *values, const RT_Surface *s, const RT_Vector *n):  RT_Primitive( na ) {

    if ( (x < 2) || ( y < 2) ) {
	rt_Output->errorVar( get_name(), ": Bad dimensions!", 0 );
	return;
    }

    // create the quadmesh as a part object:
    qm = new RT_Quadmesh( 0, x, y, 0, s, n );
    qm->father( this );

    // write the height values:
    RT_Vector v(0, 0, (1.0 - y) / 2);
    for (int j = 0; j < y; j++) {
	v.x = (1.0 - x) / 2;
	for (int i = 0; i < x; i++) { 
	    v.y = (values) ? *values++ : 0;
	    qm->vtPoint( i, j, v );
	    v.x++;
	}
	v.z++;
    }
}

int RT_HeightField::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res = _classCMD( cd, ip, argc, argv );
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " { String Integer Integer Double_List Surface_List Vector_List} {  Creates a new height field. Parameters are: {ARG 1 Name} of the new object, number of {ARG 2 Rows}, number of {ARG 3 Columns} of columns, list of heigt {ARG 4 Values}, list of {ARG 5 Surfaces}, list of {ARG 6 Normals}.}}", 0 );
	return TCL_OK;
    }
    if (res == TCL_OK) { 
	if (argc != 7) { Tcl_AppendResult( ip, "Bad number of arguments. Try \"", RTN_HEIGHT_FIELD, " ?\".", 0 ); return TCL_ERROR; }
	int x; 
	if ( !RT_string2int( argv[2], x)) { Tcl_AppendResult( ip, "Arg 2 should be an integer.", 0 ); return TCL_ERROR; }
	if ( x <= 1 ) { Tcl_AppendResult( ip, "X dimension should be greater than 1!.", 0 ); return TCL_ERROR; }
	int y;
	if ( !RT_string2int( argv[3], y)) { Tcl_AppendResult( ip, "Arg 3 should be an integer.", 0 ); return TCL_ERROR; }
	if ( y <= 1 ) { Tcl_AppendResult( ip, "Y dimension should be greater than 1!.", 0 ); return TCL_ERROR; }

	int nr = x * y;
	double  *vals = new double[nr];
	RT_Surface *surfs = 0;
	RT_Vector *norms = 0;

	// initialize points
	for (int i = 0; i < nr; i++) vals[i] = 0.0; 

	int xargc; char **xargv; int dum, cnt;
	int ret = TCL_ERROR;
	if (Tcl_SplitList( ip, argv[4], &xargc, &xargv) == TCL_OK) {
	    if (xargc < nr) rt_Output->warningVar( argv[0], ": not enough height values. Using defaults!", 0);
	    cnt = xargc < nr ? xargc : nr;
	    for ( i = 0; i < cnt; i++) 
		if (!RT_getDouble( &xargv[i], vals[i], dum)) 
		    rt_Output->warningVar( argv[0], ": found a bad height value. Using default value!", 0);
	    free((char*)xargv);

	    if (Tcl_SplitList( ip, argv[5], &xargc, &xargv) == TCL_OK) {
		surfs = new RT_Surface[nr];	
		// initialize surfaces:
		for ( i = 0; i < nr; i++) surfs[i] = RT_NULL_SURFACE;
		if (xargc < nr && xargc) rt_Output->warningVar( argv[0], ": not enough surfaces specified. Using defaults!", 0);
		cnt = xargc < nr ? xargc : nr;
		for ( i = 0; i < cnt; i++) 
		    if (strlen( xargv[i]) && !RT_getSurface( &xargv[i], surfs[i], dum)) 
			rt_Output->warningVar( argv[0], ": found a bad surface value. Using default value!", 0);
		free((char*)xargv);
	    }

	    if (Tcl_SplitList( ip, argv[6], &xargc, &xargv) == TCL_OK) {
		norms = new RT_Vector[nr];
		// initialize normals:
		for ( i = 0; i < nr; i++) norms[i] = RT_NULL_NORMAL;
		if (xargc < nr && xargc) rt_Output->warningVar( argv[0], ": not enough normals specified. Using defaults!", 0);
		cnt = xargc < nr ? xargc : nr;
		for ( i = 0; i < cnt; i++) if (strlen( xargv[i]) && !RT_getVector( &xargv[i], norms[i], dum)) 
		    rt_Output->warningVar( argv[0], ": found a bad normal value. Using default value!", 0);
		free((char*)xargv);
	    }
	    new RT_HeightField( argv[1], x, y, vals, surfs, norms );
	    RTM_classReturn;
	}
	else rt_Output->errorVar( argv[0], ": Could not create a quadmesh! No vertices found.", 0);
	delete [] vals; if (surfs) delete [] surfs; if (norms) delete [] norms;
	return ret;
    }
    return res;
}

int RT_HeightField::normF, RT_HeightField::normG, RT_HeightField::getXF, RT_HeightField::getYF, RT_HeightField::cnF;
RT_Vector RT_HeightField::normV;

RT_ParseEntry RT_HeightField::table[] = {
    { "-get_x", RTP_NONE, 0, &getXF, "Get the number of vertexes in x dimension.", RTPS_NONE },
    { "-get_y", RTP_NONE, 0, &getYF, "Get the number of vertexes in y dimension.", RTPS_NONE },

    { "-vtValue", RTP_SPECIAL, 0, 0, "Set a new {ARG 3 Value} to vertex: {ARG 1 Row}, {ARG 2 Column}.", "Integer Integer Double" },
    { "-get_vtValue", RTP_SPECIAL, 0, 0, "Get value of vertex: {ARG 1 Row}, {ARG 2 Column}.", "Integer Integer" },

    { "-vtSurface", RTP_SPECIAL, 0, 0, "Set {ARG 3 Surface} for specified vertex: {ARG 1 Row}, {ARG 2 Column}.", "Integer Integer Surface" },
    { "-get_vtSurface", RTP_SPECIAL, 0, 0, "Get Surface of specified vertex: {ARG 1 Row}, {ARG 2 Column}. If there isnt a surface at the vertex return the object default surface.", "Integer Integer" },
    { "-vtDeleteSurface", RTP_SPECIAL, 0, 0, "Clear surface at {ARG 1 Row}, {ARG 2 Column}.", "Integer Integer" },

    { "-vtNormal", RTP_SPECIAL, 0, 0, "Set {ARG 3 Normal} for specified vertex: {ARG 1 Row}, {ARG 2 Column}.", "Integer Integer Vector" },
    { "-get_vtNormal", RTP_SPECIAL, 0, 0, "Get normal of specified vertex: {ARG 1 Row}, {ARG 2 Column}. If there isnt a normal at the vertex return the object default normal.", "Integer Integer" },
    { "-vtDeleteNormal", RTP_SPECIAL, 0, 0, "Clear normal at {ARG 1 Row}, {ARG 2 Column}.", "Integer Integer" },

    { "-gnormal", RTP_VECTOR, (char*)&normV, &normF, "Set the global {ARG 1 Normal} vector.", RTPS_VECTOR },
    { "-get_gnormal", RTP_NONE, 0, &normG, "Get the global normal vector.", RTPS_NONE },
    { "-computeNormals", RTP_NONE, 0, &cnF, "For each vertex a normal vector will be computed. Existing normals will be overwritten.", RTPS_NONE },

    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_HeightField::objectCMD(char *argv[])  { 
    int ret =  RT_Primitive::objectCMD( argv );
    RT_parseTable( argv, table );

    if (getXF) {
	char tmp[10]; ret++;
	RT_int2string( get_x(), tmp );
	RT_Object::result( tmp );
    }
    if (getYF) {
	char tmp[10]; ret++;
	RT_int2string( get_y(), tmp );
	RT_Object::result( tmp );
    }

    { 	int start, diff1 = 0, diff2 = 0, diff3 = 0;
	int posx, posy; double p;
	if ((start = RT_findString( argv, "-vtValue" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) 
		    if ( RT_getDouble( &argv[ start + 3 ], p, diff3) ) {
			vtValue( posx, posy, p ); ret++;
		    }
	RT_clearArgv( argv, start, diff1 + diff2 + diff3 );
    }
    { 	int start, diff1 = 0, diff2 = 0;
	int posx, posy;
	if ((start = RT_findString( argv, "-get_vtValue" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) {
		    char tmp[60]; ret++;
		    RT_double2string( get_vtValue(posx, posy), tmp );
		    RT_Object::result( tmp );
		}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    { 	int start, diff1 = 0, diff2 = 0, diff3 = 0;
	int posx, posy; RT_Surface s;
	if ((start = RT_findString( argv, "-vtSurface" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) 
		    if ( RT_getSurface( &argv[ start + 3 ], s, diff3) )	{ 
			vtSurface( posx, posy, s ); ret++; 
		    }
	RT_clearArgv( argv, start, diff1 + diff2 + diff3 );
    }
    { 	int start, diff1 = 0, diff2 = 0;
	int posx, posy;
	if ((start = RT_findString( argv, "-get_vtSurface" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) {
		    char tmp[200]; ret++;
		    RT_surface2string( get_vtSurface(posx, posy), tmp );
		    RT_Object::result( tmp );
		}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }
    { 	int start, diff1 = 0, diff2 = 0;
	int posx, posy;
	if ((start = RT_findString( argv, "-vtDeleteSurface" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) 
		    vtDeleteSurface(posx, posy);
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    { 	int start, diff1 = 0, diff2 = 0, diff3 = 0;
	int posx, posy; RT_Vector n;
	if ((start = RT_findString( argv, "-vtNormal" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) 
		    if ( RT_getVector( &argv[ start + 3 ], n, diff3) ) {
			vtNormal( posx, posy, n ); ret++;
		    }
	RT_clearArgv( argv, start, diff1 + diff2 + diff3 );
    }
    { 	int start, diff1 = 0, diff2 = 0;
	int posx, posy;
	if ((start = RT_findString( argv, "-get_vtNormal" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) {
		    char tmp[60]; ret++;
		    RT_vector2string( get_vtNormal(posx, posy), tmp );
		    RT_Object::result( tmp );
		}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }
    { 	int start, diff1 = 0, diff2 = 0;
	int posx, posy;
	if ((start = RT_findString( argv, "-vtDeleteNormal" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) {
		    vtDeleteNormal(posx, posy); ret++;
		}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    if (normF) { gnormal( normV ); ret++; }
    if (normG) {
	char tmp[30]; ret++;
	RT_vector2string( get_gnormal(), tmp );
	RT_Object::result( tmp );
    }

    if (cnF) { qm->computeNormals(); ret++; }

    return ret;
}

void RT_HeightField::printCon(FILE *f) const {
    RT_Object::printCon( f );
    fprintf( f, "%i %i { ", get_x(),get_y() );

    int i,j;
    int x = get_x();
    int y = get_y();

    // save height values:
    for (j = 0; j < y; j++)
	for ( i = 0; i < x; i++)
	    fprintf( f, "%lf ", get_vtValue( i, j ));

    fprintf( f, "} {" );

    // save the surfaces:
    for (j = 0; j < y; j++)
	for ( i = 0; i < x; i++) {
	    const RT_Surface *su = qm->get( i, j)->getSurface();
	    if (su) su->print( f );
	    else fprintf( f, "{} " );
	}

    fprintf( f, "} {" );

    // save the normals:
    for (j = 0; j < y; j++)
	for ( i = 0; i < x; i++) {
	    const RT_Vector *no = qm->get( i, j)->getNormal();
	    if (no) no->print( f );
	    else fprintf( f, "{} " );
	}
    
    fprintf( f, "} " );
}

void RT_HeightField::print(FILE *f) const {
    RT_Primitive::print( f ); 

    if (!(get_gnormal() == RT_NULL_NORMAL )) {
	fprintf( f, "%s -gnormal ", get_name() );
	get_gnormal().print( f );
	fprintf( f, "\n" );
    }
}

inline double lamda_to_X(int X, const RT_Vector& PT, const RT_Vector& DIR) {
    return (DIR.x == 0) ? DBL_MAX : (X - PT.x) / DIR.x;
}

inline double lamda_to_Y(int Y, const RT_Vector& PT, const RT_Vector& DIR) {
    return (DIR.z == 0) ? DBL_MAX : (Y - PT.z) / DIR.z;
}

inline double lamda_at_X(int X, int Y,const RT_Vector& PT,const RT_Vector& DIR) {
    return DIR.z / DIR.x * (X - PT.x) + PT.z - Y;
}


int RT_HeightField::intersect(const RT_Ray& ray, RT_InterSectionList& il) {
    RT_Ray mray;
    RTM_wcRay2mcRay(ray,mray);
    RT_Vector pt_2d = mray.pt, dir_2d = mray.dir;
    pt_2d.y = dir_2d.y = 0; dir_2d = dir_2d.UNITIZE();
    // move camera to centre of base plane
    pt_2d.x -= (1.0 - get_x()) / 2; pt_2d.z -= (1.0 - get_y()) / 2;
    int ret = 0;
    int x_bnd = get_x()-1, y_bnd = get_y()-1;
    int x,y;
    qm->GetICache()->restart();
    if (dir_2d.x != 0 && dir_2d.z != 0) {
	// try to find 2 intersections with the edges of base plane of the heightfield
	int inters = 0;
	double l,lamda[2];
	register double tmp;
	// test if intersect (0,0) , (0,y_bnd)
	if ((l = lamda_to_X(0,pt_2d,dir_2d)) >= 0 && (tmp = l * dir_2d.z + pt_2d.z) >= 0 && tmp <= y_bnd) {
	    lamda[inters] = l;
	    inters++;
	}
	// test if intersect (0,0) , (x_bnd,0)
	if ((l = lamda_to_Y(0,pt_2d,dir_2d)) >= 0 && (tmp = l * dir_2d.x + pt_2d.x) >= 0 && tmp <= x_bnd) {
	    lamda[inters] = l;
	    inters++;
	}
	// test if intersect (x_bnd,0) , (x_bnd,y_bnd)
	if (inters < 2 && (l = lamda_to_X(x_bnd,pt_2d,dir_2d)) >= 0 && (tmp = l * dir_2d.z + pt_2d.z) >= 0 && tmp <= y_bnd) {
	    lamda[inters] = l;
	    inters++;
	}
	// test if intersect (0,y_bnd) , (x_bnd,y_bnd)
	if (inters < 2 && (l = lamda_to_Y(y_bnd,pt_2d,dir_2d)) >= 0 && (tmp = l * dir_2d.x + pt_2d.x) >= 0 && tmp <= x_bnd) {
	    lamda[inters] = l;
	    inters++;
	}
	if (!inters) return 0;
	if (inters == 1) {lamda[1] = 0; inters++;}
	if (lamda[0] > lamda[1]) {tmp = lamda[0]; lamda[0] = lamda[1]; lamda[1]= tmp;}

	int type = (dir_2d.x > 0) + (dir_2d.z > 0) * 2;
	// corresponding to the type only 2 edges of a cell are checked
	x = (int)(pt_2d.x + dir_2d.x * lamda[0]);
	y = (int)(pt_2d.z + dir_2d.z * lamda[0]);
	if (x >= x_bnd) x = x_bnd - 1; else if (x < 0) x = 0;
	if (y >= y_bnd) y = y_bnd - 1; else if (y < 0) y = 0;
	while (x >= 0 && x < x_bnd && y >= 0 && y < y_bnd) {
	    ret += qm->_intersect(x,y,mray,ray,il);
	    switch (type) {
	      case 0:
		if (lamda_at_X(x,y,pt_2d,dir_2d) >= 0) x--; else y--;
		break;
	      case 1:
		if (lamda_at_X(x+1,y,pt_2d,dir_2d) >= 0) x++; else y--;
		break;
	      case 2:
		if (lamda_at_X(x,y,pt_2d,dir_2d) <= 1) x--; else y++;
		break;
	      case 3:
		if (lamda_at_X(x+1,y,pt_2d,dir_2d) <= 1) x++; else y++;
	    }
	}
    } else {
	// handle special cases here
	x = (int)pt_2d.x; y = (int)pt_2d.z;
	if (dir_2d.x == 0.0 && dir_2d.z != 0.0) {
	    // up or down
	    if (x >= 0 && x < x_bnd) {
		// in the x range
		if (dir_2d.z > 0.0 && y < y_bnd) {
		    // there can be intersect some cells
		    if (y < 0) y = 0; // setting to start cell
		    while (y < y_bnd) {
			ret += qm->_intersect(x,y,mray,ray,il);
			y++;
		    }
		} else
		    if (dir_2d.z < 0.0 && y >= 0) {
			if (y >= y_bnd) y = y_bnd - 1;
			while (y >= 0) {
			    ret += qm->_intersect(x,y,mray,ray,il);
			    y--;
			}
		    } else
			return 0;
	    }
	} else
	    if (dir_2d.x != 0.0 && dir_2d.z == 0.0) {
		// left or right
		if (y >= 0 && y < y_bnd) {
		    // in the y range
		    if (dir_2d.x > 0.0 && x < x_bnd) {
			if (x < 0) x = 0;
			while (x < x_bnd) {
			    ret += qm->_intersect(x,y,mray,ray,il);
			    x++;
			}
		    } else
			if (dir_2d.z < 0.0 && x >= 0) {
			    if (x >= x_bnd) x = x_bnd - 1;
			    while (x >= 0) {
				ret += qm->_intersect(x,y,mray,ray,il);
				x--;
			    }
			} else
			    return 0;
		}
	    } else
		// both directions must be zero
		if (x >= 0 && x < x_bnd && y >= 0 && y < y_bnd)
		    ret = qm->_intersect(x,y,mray,ray,il);
	
    } // end of special cases
    return( ret );
}


int RT_HeightField::copy(RT_Primitive *p) const {
    if (p->isA( RTN_QUADMESH )) return( qm->copy( p ));
    if (p->isA( RTN_HEIGHT_FIELD )) {
	RT_HeightField *h = (RT_HeightField*)p;
	qm->copy( h->qm );
	return( 1 );
    }
    return( RT_Primitive::copy(p) );
}

