////////////////////////////////////////////////////////////////////////////////
//  implementation of a generic vector class                                  //  
//  LAST EDIT: Fri Feb 10 15:39:56 1995 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRGHT which should be distributed with this  //
//  file. If COPYRGHT is not available or for more info please contact:       //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1993 - 1995 YART team                                        //
////////////////////////////////////////////////////////////////////////////////

#include "genvector.h"
#include <math.h>

void _Vector::print(FILE* f) const { 
    for( int i = 0; i < d; i++) printf("%.3f ", v[i] );
    fprintf(f,"\n");
}

void _Vector::check_dimensions(const _Vector& vec) const { 
    if (d!=vec.d) printf("_Vector arguments have different dimensions.\n");
}

_Vector::_Vector() {
    d = 0; 
    v = 0;
}

_Vector::_Vector(int n) { 
    if (n<0) printf("_Vector: negative dimension.\n"); 
    d = n; 
    v = 0;
    if (d > 0) { 
	v = new double[d]; 
	while (n--) v[n] = 0.0;
    }
}

_Vector::~_Vector() { 
    if (v) delete[] v;
}

_Vector::_Vector(const _Vector& p) {
    d = p.d; 
    v = 0;
    if (d > 0) {
	v = new double[d];
	for(int i=0; i<d; i++) v[i] = p.v[i];
    }
}

_Vector::_Vector(double x, double y, double z) { 
    v = new double [3];
    d = 3;
    v[0] = x;
    v[1] = y;
    v[2] = z;
}

void _Vector::newGeometry(int i) {  
    if (v) delete[] v;
    v = 0;

    d = i;
    if (d > 0) { 
	v = new double[i];
	while (i--) v[i] = 0.0;
    }
}

double _Vector::operator[](int i) const { 
    if (i<0 || i>=d)  printf("_Vector: index out of range \n");
    return v[i]; 
}

double& _Vector::operator[](int i) { 
    if (i<0 || i>=d) printf("_Vector: index out of range \n");
    return v[i]; 
}

_Vector& _Vector::operator+=(const _Vector& vec) { 
    check_dimensions(vec);
    register int n = d;
    while (n--) v[n] += vec.v[n];
    return *this;
}

_Vector& _Vector::operator-=(const _Vector& vec) {
    check_dimensions(vec);
    register int n = d;
    while (n--) v[n] -= vec.v[n];
    return *this;
}

_Vector _Vector::operator+(const _Vector& vec) const {
    check_dimensions(vec);
    register int n = d;
    _Vector result(n);
    while (n--) result.v[n] = v[n]+vec.v[n];
    return result;
}

_Vector _Vector::operator-(const _Vector& vec) const {
    check_dimensions(vec);
    register int n = d;
    _Vector result(n);
    while (n--) result.v[n] = v[n]-vec.v[n];
    return result;
}

_Vector _Vector::operator-() const { 
    register int n = d;
    _Vector result(n);
    while (n--) result.v[n] = -v[n];
    return result;
}

_Vector _Vector::operator*(double x) const {
    int n = d;
    _Vector result(n);
    while (n--) result.v[n] = v[n] * x;
    return result;
}

_Vector _Vector::operator/(double x) const { 
    int n = d;
    _Vector result(n);
    while (n--) result.v[n] = v[n] / x;
    return result;
}

//friend
_Vector operator*(double f, const _Vector& v) { return v*f; } 

double _Vector::operator*(const _Vector& vec) const { 
    check_dimensions(vec);
    double result=0;
    register int n = d;
    while (n--) result = result+v[n]*vec.v[n];
    return result;
}

_Vector& _Vector::operator=(const _Vector& vec) { 
    register int n = vec.d;

    if (n != d) { 
	if (v) { delete[] v; }
	v = new double[n];	  
	d = n;
    }
  
    while (n--) v[n] = vec.v[n];

    return *this;
}

int _Vector::operator==(const _Vector& vec)  const {
    if (vec.d != d) return 0;
    int i = 0;
    while ((i<d) && (v[i]==vec.v[i])) i++;
    return (i==d) ? 1 : 0;
}

double _Vector::length() const { return sqrt(*this * *this); }

double _Vector::angle(const _Vector& y)  const {
    double l = length();
    double yl = y.length();

    if ( l==0 || yl==0)	printf("angle: zero argument\n");

    return  acos((*this)*y/(l*yl));  
}

int compare(const _Vector& v1, const _Vector& v2) {
    register int i;

    if (v1.dim() != v2.dim())
	printf("compare(_Vector,_Vector): different dimensions\n");

    for(i=0; i < v1.dim() && v1[i]==v2[i]; i++);
    
    if (i == v1.dim()) return 0;
    
    return (v1[i] < v2[i]) ?  -1 : 1;
}
