#include <deltabl/deltabl.h>

#include <sys/time.h>

///// utility procedures and variables:

long startTime;

long millisecondClock() {
    struct timeval v;
    struct timezone z;
    
    gettimeofday(&v,&z);
    return (v.tv_sec * 1000) + (v.tv_usec / 1000);
}

void start() { startTime = millisecondClock(); }

long finish() { return millisecondClock() - startTime; }

///// a class especially for testing:

class TS_Variable: public DB_Variable {
  public:
    TS_Variable(char *c, long l, DB_Boolean b=FALSE):
    DB_Variable(c, b) {	value = l; }
    long value;
    void print(FILE *) const;
    void assign(long);
    void change(long);
};

void TS_Variable::print(FILE *f) const {
    fprintf( f, "%s: %s, value: %ld, strength: %s\n", con ? "Constant" : DBN_VARIABLE, get_name(), value, DB_strength2string(walkStrength));
}

void TS_Variable::assign(long newValue) {
    DB_Constraint *editC = new DB_EditC(this, DBS_REQUIRED );
    if (editC->satisfied()) {
	value = newValue;
	DB_List	*plan = editC->extractPlanFromConstraint();
	plan->executePlan();
	delete plan;
    }
    delete editC;
}

void TS_Variable::change(long newValue) {
    printf("Changing %s...\n", get_name());

    start();
    DB_Constraint *editC = new DB_EditC(this, DBS_STRONG_DEFAULT );
    long msecs = finish();
    printf("Adding Constraint: %ld msecs.\n", msecs);
   
    start();
    DB_List *plan = editC->extractPlanFromConstraint();
    msecs = finish();
    printf("Making Plan (length: %d): %ld msecs.\n", plan->listSize(), msecs);
    
    start();
    value = newValue;
    for (int i = 0; i < 100; i++) plan->executePlan();
    msecs = finish();
    printf("Executing Plan: %.3f msecs.\n", msecs / 100.0);
    delete plan;

    start();
    delete editC;
    msecs = finish();
    printf("Removing Constraint: %ld msecs\n", msecs);
}

///// test constraints:

class TS_EqualsC: public DB_Constraint {
    // a == b 
    void execute();
  public:
    TS_EqualsC(DB_Variable *a, DB_Variable *b, DB_Strength strength):DB_Constraint(2, strength) {
	variables[0] = a;
	variables[1] = b;
	methodCount = 2;
	methodOuts[0] = 0;
	methodOuts[1] = 1;
	add();
    }
};

void TS_EqualsC::execute() {
    switch (whichMethod) {
      case 0:
	((TS_Variable*)variables[0])->value
	    = ((TS_Variable*)variables[1])->value;
	break;
      case 1:
	((TS_Variable*)variables[1])->value
	    = ((TS_Variable*)variables[0])->value;
	break;
    }
}

class TS_AddC: public DB_Constraint {
    // a + b = sum 
    void execute();
  public:
    TS_AddC(DB_Variable *a, DB_Variable *b, DB_Variable *sum, DB_Strength strength):DB_Constraint(3, strength) {
	variables[0] = a;
	variables[1] = b;
	variables[2] = sum;
	methodCount = 3;
	methodOuts[0] = 2;
	methodOuts[1] = 1;
	methodOuts[2] = 0;
	add();
    }
};

void TS_AddC::execute() {
    switch (whichMethod) {
      case 0:
	((TS_Variable*)variables[2])->value
	    = ((TS_Variable*)variables[0])->value
		+ ((TS_Variable*)variables[1])->value;
	break;
      case 1:
	((TS_Variable*)variables[1])->value
	    = ((TS_Variable*)variables[2])->value
		- ((TS_Variable*)variables[0])->value;
	break;
      case 2:
	((TS_Variable*)variables[0])->value
	    = ((TS_Variable*)variables[2])->value
		- ((TS_Variable*)variables[1])->value;
	break;
    }
}

class TS_MultiplyC: public DB_Constraint {
    // a * b = prod 
    void execute();
  public:
    TS_MultiplyC(DB_Variable *a, DB_Variable *b, DB_Variable *prod, DB_Strength strength):DB_Constraint(3, strength) {
	variables[0] = a;
	variables[1] = b;
	variables[2] = prod;
	methodCount = 3;
	methodOuts[0] = 2;
	methodOuts[1] = 1;
	methodOuts[2] = 0;
	add();
    }
};

void TS_MultiplyC::execute() {
    switch (whichMethod) {
      case 0:
	((TS_Variable*)variables[2])->value
	    = ((TS_Variable*)variables[0])->value 
		* ((TS_Variable*)variables[1])->value;
	break;
      case 1:
	((TS_Variable*)variables[1])->value
	    = ((TS_Variable*)variables[2])->value
		/ ((TS_Variable*)variables[0])->value;
	break;
      case 2:
	((TS_Variable*)variables[0])->value 
	    = ((TS_Variable*)variables[2])->value 
		/ ((TS_Variable*)variables[1])->value;
	break;
    }
}

class TS_ScaleOffsetC: public DB_Constraint {
    // (src * scale ) + offset = dest 
    void execute();
  public:
    TS_ScaleOffsetC(DB_Variable *src, DB_Variable *scale, DB_Variable *offset, DB_Variable *dest, DB_Strength strength):DB_Constraint(4, strength) {
	variables[0] = src;
	variables[1] = scale;
	variables[2] = offset;
	variables[3] = dest;
	methodCount = 2;
	methodOuts[0] = 3;
	methodOuts[1] = 0;
	add();
    }
};

void TS_ScaleOffsetC::execute() {
    switch (whichMethod) {
      case 0:
	((TS_Variable*)variables[3])->value 
	    = (((TS_Variable*)variables[0])->value 
	       * ((TS_Variable*)variables[1])->value) 
		+ ((TS_Variable*)variables[2])->value;
	break;
      case 1:
	((TS_Variable*)variables[0])->value
	    = (((TS_Variable*)variables[3])->value
	       - ((TS_Variable*)variables[2])->value)
		/ ((TS_Variable*)variables[1])->value;
	break;
    }
}

///// a Fahrenheit to Celcius temperature converter:

void TS_temperatureConverter() {
    DB_init();
    TS_Variable *celcius = new TS_Variable("C", 0);
    TS_Variable *fahrenheit = new TS_Variable("F", 0);
    TS_Variable *t1 = new TS_Variable("t1", 1);
    TS_Variable *t2 = new TS_Variable("t2", 1);
    TS_Variable *nine = new TS_Variable("*const*", 9, TRUE);
    TS_Variable *five = new TS_Variable("*const*", 5, TRUE);
    TS_Variable *thirtyTwo = new TS_Variable("*const*", 32, TRUE);

    new TS_MultiplyC(celcius, nine, t1, DBS_REQUIRED );
    new TS_MultiplyC(t2, five, t1, DBS_REQUIRED );
    new TS_AddC(t2, thirtyTwo, fahrenheit, DBS_REQUIRED ); 

    printf("***** Changing celcius to 0 *****\n");
    celcius->assign( 0);
    celcius->print(stdout);
    fahrenheit->print(stdout);

    printf("***** Changing fahrenheit to 212 *****\n");
    fahrenheit->assign( 212 );
    celcius->print(stdout);
    fahrenheit->print(stdout);

    printf("***** Changing celcius to -40 *****\n");
    celcius->assign( -40);
    celcius->print(stdout);
    fahrenheit->print(stdout);

    printf("***** Changing fahrenheit to 70 *****\n");
    fahrenheit->assign( 70);
    celcius->print(stdout);
    fahrenheit->print(stdout);
    DB_exit();
}

///// This is the standard DeltaBlue benchmark. A long chain of equality
///// constraints is constructed with a stay constraint on one end. An edit
///// constraint is then added to the opposite end and the time is measured for
///// adding and removing this constraint, and extracting and executing a
///// constraint satisfaction plan. There are two cases. In case 1, the added
///// constraint is stronger than the stay constraint and values must propagate
///// down the entire length of the chain. In case 2, the added constraint is
///// weaker than the stay constraint so it cannot be accomodated. The cost in
///// this case is, of course, very low. Typical situations lie somewhere between
///// these two extremes.

void TS_benchmark(int n) {
    DB_init();
    TS_Variable *prev, *v, *first, *last;
    prev = first = last = NULL;

    start();
    for (int i = 0; i < n; i++) {
	char name[20];
	sprintf(name, "v%ld", i);
	v = new TS_Variable(name, 0);
	if (prev) new TS_EqualsC(prev, v, DBS_REQUIRED );
	if (i == 0) first = v;
	if (i == (n-1)) last = v;
	prev = v;
    }
    long msecs = finish();
    printf("\n%ld msecs to add %d constraints.\n", msecs, n);
    new DB_StayC(last, DBS_DEFAULT);

    start();
    DB_Constraint *editC = new DB_EditC(first, DBS_STRONG_DEFAULT);
    msecs = finish();
    printf("Add Constraint (case 1): %ld msecs.\n", msecs);

    start();
    DB_List *plan = editC->extractPlanFromConstraint();
    msecs = finish();
    printf( "Make Plan (case 1): %ld msecs (plan is length %d).\n",
	   msecs, plan->listSize());

    start();
    for (i = 0; i < 100; i++) plan->executePlan();
    msecs = finish();
    printf("Execute Plan (case 1): %.3f msecs.\n", msecs / 100.0);
    delete plan;

    start();
    delete editC;
    msecs = finish();
    printf("Remove Constraint (case 1): %ld msecs\n", msecs);
    
    start();
    editC = new DB_EditC(first, DBS_WEAK_DEFAULT );
    msecs = finish();
    printf("Add Constraint (case 2): %ld msecs.\n", msecs);
    
    start();
    plan = editC->extractPlanFromConstraint();
    msecs = finish();
    printf( "Make Plan (case 2): %ld msecs (plan is length %d).\n",
	   msecs, plan->listSize());

    start();
    for (i = 0; i < 100; i++) plan->executePlan();
    msecs = finish();
    printf("Execute Plan (case 2): %.3f msecs.\n", msecs / 100.0);
    delete plan;

    start();
    delete editC;
    msecs = finish();
    printf("Remove Constraint (case 2): %ld msecs\n", msecs);

    start();
    editC = new DB_EditC(last, DBS_STRONG_DEFAULT );
    msecs = finish();
    printf("Add Constraint (case 3): %ld msecs.\n", msecs);
    
    start();
    plan = editC->extractPlanFromConstraint();
    msecs = finish();
    printf( "Make Plan (case 3): %ld msecs (plan is length %d).\n",
	   msecs, plan->listSize());

    start();
    for (i = 0; i < 100; i++) plan->executePlan();
    msecs = finish();
    printf("Execute Plan (case 3): %.3f msecs.\n", msecs / 100.0);
    delete plan;

    start();
    delete editC;
    msecs = finish();
    printf("Remove Constraint (case 3): %ld msecs\n\n", msecs);
}

///// This test constructs a two sets of variables related to each other by a
///// simple linear transformation (scale and offset). The time is measured to
///// change a variable on either side of the mapping and to change the scale or
///// offset factors. It has been tested for up to 2000 variable pairs.

void TS_projectionTest(int n) {
    TS_Variable *src, *scale, *offset, *dest;
    long msecs;
    char name[20];
    DB_init();
    start();
    scale = new TS_Variable("scale", 10);
    offset = new TS_Variable("offset", 1000);
    
    for (int i = 1; i <= n; i++) {
	/* make src and dest variables */
	sprintf(name, "src%ld", i);
	src = new TS_Variable(name, i);
	sprintf(name, "dest%ld", i);
	dest = new TS_Variable(name, i);
	
	/* add stay on src */
	new DB_StayC(src, DBS_DEFAULT);
	
	/* add scale/offset constraint */
	new TS_ScaleOffsetC(src, scale, offset, dest, DBS_REQUIRED );
    }
    msecs = finish();
    printf("Setup time for %d points: %ld msecs.\n", n, msecs);
    
    src->change( 17);
    dest->change( 1050);
    scale->change( 5);
    offset->change( 2000);
    DB_exit();
}

///// This test constructs a full binary tree of add constraints of the given
///// depth and then measures the time to change the root variable. Log(depth)
///// constraints must be traversed. DeltaBlue chooses an arbitrary path to a
///// leaf of the tree.

// returns the root variable of an adder tree of the given depth 
TS_Variable *TS_makeTree(int depth) {
    TS_Variable *root, *left, *right;
    if (depth <= 0) {
	root = new TS_Variable("leaf", 1);
	new DB_StayC(root, DBS_DEFAULT);
    } else {
	root = new TS_Variable("nonleaf", 1);
	left = TS_makeTree(depth - 1);
	right = TS_makeTree(depth - 1);
	new TS_AddC(left, right, root, DBS_REQUIRED );
    }
    return root;
}

void TS_treeTest(int depth) {
    DB_init();
    printf("Adder tree of depth %d\n", depth);
    start();

    TS_Variable *root = TS_makeTree(depth);
    
    long msecs = finish();
    // Note: with stays, there is one constraint per variable 
    printf( "%d constraints added in %ld msecs.\n", db_allVariables->listSize(), msecs);
    root->change( 17);
    DB_exit();
}

main() {
    TS_temperatureConverter();
    TS_treeTest( 5 );
    TS_treeTest(10);
    TS_benchmark(50);
    TS_benchmark(100);
    TS_benchmark(200);
    TS_benchmark(400);
    TS_benchmark(800);
    TS_projectionTest(250);
    TS_projectionTest(500);
    TS_projectionTest(1000);
    TS_projectionTest(2000);
}
