/*
 * x11window.c
 * 
 * Kevin P. Smith  6/11/89 Much modified by Jerry Frain and Joe Young
 */

#include <stdio.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <sys/types.h>
#include <sys/time.h>
#include <ctype.h>
#include <string.h>
#ifdef SETTCPNODELAY
#include <netinet/in.h>
#include <netinet/tcp.h>
#endif
#include <assert.h>
#include "Wlib.h"
#include "defs.h"
#include "struct.h"
#include "data.h"

#define MAX_TEXT_WIDTH		90

#ifdef XSIG
extern int Xpending;
extern int sigio_handler();
#   define	XCR(v, p)	(critical=1,v=p,critical=0,(Xpending ? \
				  sigio_handler():0),v)
#   define	XC(p)		(critical=1,p,critical=0,(Xpending ? \
				  sigio_handler():0))
#else
#   define	XCR(v, p)	p
#   define	XC(p)		p
#endif

/* XFIX speedup */
#define MAXCACHE	128

/* changes too good to risk leaving out, by Richard Caley (rjc@cstr.ed.ac.uk) */
#define RJC
#define FOURPLANEFIX

#define NORMAL_FONT     "6x10"
#define BOLD_FONT       "-*-clean-bold-r-normal--10-100-75-75-c-60-*"
#define ITALIC_FONT     "-*-clean-bold-r-normal--10-100-75-75-c-60-*"
#define BIG_FONT        "-*-lucidatypewriter-*-*-*-*-40-*-*-*-*-*-*-*"
#define IND_FONT        "-*-clean-bold-r-normal--10-100-75-75-c-60-*"

static char    *_nfonts[] =
{
   NORMAL_FONT,
   "-*-clean-medium-r-normal--10-100-75-75-c-60-*",
   "6x10",
   NULL,
};
static char    *_bfonts[] =
{
   BOLD_FONT,
   "-*-clean-bold-r-normal--10-100-75-75-c-60-*",
   "6x10",
   NULL,
};
static char    *_ifonts[] =
{
   ITALIC_FONT,
   "-*-clean-bold-r-normal--10-100-75-75-c-60-*",
   "6x10",
   NULL,
};
static char    *_bgfonts[] =
{
   BIG_FONT,
   "6x10",
   "6x10",
   NULL,
};

XFontStruct    *find_font();

#define FONTS 4
#define BITGC 4

#define WHITE   0
#define BLACK   1
#define RED     2
#define GREEN   3
#define YELLOW  4
#define CYAN    5
#define GREY	6

static int      zero = 0;
static int      one = 1;
static int      two = 2;
static int      three = 3;

int             W_FastClear = 0;
Display        *W_Display;
Window          W_Root;
Colormap        W_Colormap;
int             W_Screen;
#ifdef FOURPLANEFIX
Visual         *W_Visual;
#endif
W_Font          W_BigFont = (W_Font) & zero, W_RegularFont = (W_Font) & one;
W_Font          W_HighlightFont = (W_Font) & two, W_UnderlineFont = (W_Font) & three;
W_Color         W_White = WHITE, W_Black = BLACK, W_Red = RED, W_Green = GREEN;
W_Color         W_Yellow = YELLOW, W_Cyan = CYAN, W_Grey = GREY;
int             W_Textwidth, W_Textheight;
char           *getdefault();
char           *strdup();

int             W_in_message = 0;	/* jfy -- for Jerry's warp message
					 * hack */

#ifdef RJC
extern W_Window baseWin;
static XClassHint class_hint =
{
   "netrek", "Netrek",
};

static XWMHints wm_hint =
{
   InputHint | StateHint,
   True,
   NormalState,
   None,
   None,
   0, 0,
   None,
   None,
};

static XSizeHints wm_size_hint;
#endif				/* RJC */

static W_Event  W_myevent;
static int      W_isEvent = 0;

struct fontInfo {
   int             baseline;
};

struct colors {
   char           *name;
   GC              contexts[FONTS + 1];
   Pixmap          pixmap;
   int             pixelValue;
};

struct icon {
   Window          window;
   Pixmap          bitmap;
   int             width, height;
   Pixmap          pixmap;
};

#define WIN_GRAPH	1
#define WIN_TEXT	2
#define WIN_MENU	3
#define WIN_SCROLL	4

struct window {
   Window          window;
   int             type;
   char           *data;
   int             mapped;
   int             width, height;
   char           *name;
    W_Callback handle_keydown;
    W_Callback handle_keyup;
    W_Callback handle_button;
    W_Callback handle_expose;
};

struct scrollingWindow {
   int		   	lines;
   int			updated;
   struct stringList	*head;
   struct stringList	*tail;
};

struct stringList {
   char           	string[MAX_TEXT_WIDTH];
   W_Color         	color;
   struct stringList 	*next, *prev;
};

struct menuItem {
   char           *string;
   W_Color         color;
};

struct colors   colortable[] =
{
   {"white"},
   {"black"},
   {"red"},
   {"green"},
   {"yellow"},
   {"cyan"},
   {"light grey"},
   {"dummy"}			/* extensions */
};

struct windowlist {
   struct window  *window;
   struct windowlist *next;
};

#define HASHSIZE 29
#define hash(x) (((int) (x)) % HASHSIZE)

struct windowlist *hashtable[HASHSIZE];
struct fontInfo fonts[FONTS];

struct window  *newWindow();
struct window  *findWindow();
short          *x11tox10bits();

struct window   myroot;

/* Last entry reserved for extensions */
#define NCOLORS (sizeof(colortable)/sizeof(colortable[0]))-1
#define W_Void2Window(win) ((win) ? (struct window *) (win) : &myroot)
#define W_Window2Void(window) ((W_Window) (window))
#define W_Void2Icon(bit) ((struct icon *) (bit))
#define W_Icon2Void(bit) ((W_Icon) (bit))
#define fontNum(font) (*((int *) font))
#define TILESIDE 16

#define WIN_EDGE 5		/* border on l/r edges of text windows */
#define MENU_PAD 4		/* border on t/b edges of text windows */
#define MENU_BAR 4		/* width of menu bar */

static char     gray[] =
{
   0xaa, 0xaa, 0x55, 0x55, 0xaa, 0xaa, 0x55, 0x55,
   0xaa, 0xaa, 0x55, 0x55, 0xaa, 0xaa, 0x55, 0x55,
   0xaa, 0xaa, 0x55, 0x55, 0xaa, 0xaa, 0x55, 0x55,
   0xaa, 0xaa, 0x55, 0x55, 0xaa, 0xaa, 0x55, 0x55
};

static char     striped[] =
{
   0x00, 0xff, 0x00, 0xff, 0x00, 0xff, 0x00, 0xff,
   0xf0, 0x0f, 0xf0, 0x0f, 0xf0, 0x0f, 0xf0, 0x0f,
   0xff, 0x00, 0xff, 0x00, 0xff, 0x00, 0xff, 0x00,
   0x0f, 0xf0, 0x0f, 0xf0, 0x0f, 0xf0, 0x0f, 0xf0
};

static char     solid[] =
{
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff
};

#ifdef nodef
/* X debugging */
int 
_myerror(d, e)
    Display        *d;
    XErrorEvent    *e;
{
   abort();
}
#endif


static long 
WMXYHintMode_default()		/* BRM2.0pl18 */
{
   static int fetched = 0;
   static long WMXYHM_default;
   char *hm_default_string;

   if(!fetched)
   {  hm_default_string = getdefault("WMXYHintMode");
      if(!hm_default_string || strcmp(hm_default_string,"USPosition") == 0)
         WMXYHM_default = USPosition;
      else WMXYHM_default = PPosition;
      fetched = 1;
   }
   return WMXYHM_default;
}

/* BRM2.00pl18 */
static void 
configureScrolling(win, x, y, width, height)

   struct window *win;
   int		 x, y;	/* TODO */
   int		 width, height;

{   
   int 			new_text_width, new_text_height;
   int 			new_real_width,new_real_height;
   XWindowAttributes	wa;

   /* XXX: can't shrink window */

   if(width <= win->width*W_Textwidth+ WIN_EDGE*2 &&
      height <= win->height*W_Textheight + MENU_PAD*2)
      return;

   XGetWindowAttributes(W_Display, win->window, &wa);

   new_text_width = (wa.width - WIN_EDGE*2)/W_Textwidth;
   new_text_height = (wa.height - MENU_PAD*2)/W_Textheight;

   if(new_text_width <= 0) new_text_width = 1;
   if(new_text_height <= 0) new_text_height = 1;
   if(new_text_width >= MAX_TEXT_WIDTH) new_text_width = MAX_TEXT_WIDTH-1;

   new_real_width = new_text_width*W_Textwidth + WIN_EDGE*2;
   new_real_height = new_text_height*W_Textheight + MENU_PAD*2;

   if(new_real_height != wa.height || new_real_width != wa.width)
      XResizeWindow(W_Display, win->window,new_real_width,new_real_height);

   win->width = new_text_width;
   win->height = new_text_height;

   /* an expose event will follow a resize request, triggering 
      redrawScrolling */
}

/* BRM2.00pl18 */
/*****************************************************************************/
/*   Looks up any default geometry specified in the defaults file and        */
/*   returns the values found there.  Geometry should be of the form         */
/*        [=][<width>x<height>][{+-}<xoffset>{+-}<yoffset>]                  */
/*                                                                           */
/*   The result returned indicates which values were set.                    */
/*    XValue, YValue, WidthValue, HeightValue				     */
/*                                                                           */
/*****************************************************************************/

static int 
checkGeometry(name, x, y, width, height)
   char *name;
   int *x, *y, *width, *height;
{  
   char	buf[80], *geom_default;
   int                mask, i;

   sprintf(buf,"%s.geometry",name);
   geom_default = getdefault(buf);
#ifdef SHOW_DEFAULTS
   {
      char	df[160], desc[80];
      sprintf(df, "%dx%d%s%d%s%d", *width, *height,(*x<0?"":"+"),*x,
					(*y<0?"":"+"),*y);
      show_defaults("windows", buf, df, 
	 "Default window geometry (use 'xwininfo -tree' on the netrek window \n\
to get names.)");
   }
#endif
   if(!geom_default) return 0;                                /* nothing set */

    strcpy (buf, geom_default);

    /* Eat trailing spaces */
    for (i = strlen (buf) - 1;i > 0;i--)
      if (isspace (buf [i]))
        buf [i] = '\0';

    mask = XParseGeometry(buf, x, y, (unsigned int *)width, 
                                             (unsigned int *)height);
    if(!mask) {
       fprintf(stderr, "netrek: geometry parse error on \"%s.geometry: %s\"\n",
        name, geom_default);
    }
    return mask;
}

void 
W_Initialize(str)
    char           *str;
{
   int             i;

#ifdef SETTCPNODELAY
   int		   mi = 1;
#endif

#ifdef DEBUG
   printf("Initializing...\n");
#endif
   for (i = 0; i < HASHSIZE; i++) {
      hashtable[i] = NULL;
   }
   if ((W_Display = XOpenDisplay(str)) == NULL) {
      extern char *getenv();
      if(str)
	 fprintf(stderr, "netrek: Can't open display %s.\n", str);
      else
	 fprintf(stderr, "netrek: Can't open display %s.\n", 
	    (getenv("DISPLAY")!=NULL)?getenv("DISPLAY"):"(none)");
      exit(1);
   }
#ifdef SETTCPNODELAY
   if(setsockopt(ConnectionNumber(W_Display), IPPROTO_TCP,
              TCP_NODELAY, (char *)&mi, sizeof(int)) < 0){
      /* harmless error
      perror("X socket tcp_nodelay failed.");
      */
   }
#endif

   /* tmp */
   /* XSynchronize(W_Display, True); */
   /* XSetErrorHandler(_myerror); */

   W_Root = DefaultRootWindow(W_Display);
#ifdef FOURPLANEFIX
   W_Visual = DefaultVisual(W_Display, DefaultScreen(W_Display));
#endif
   W_Screen = DefaultScreen(W_Display);
   W_Colormap = DefaultColormap(W_Display, W_Screen);
   myroot.window = W_Root;
   myroot.type = WIN_GRAPH;
   GetFonts();
   GetColors();
}

/*
 * Make sure the font will work, ie: that it fits in the 6x10 character
 * cell that we expect. (BRM)
 */
checkFont (fontinfo, fontname)
     XFontStruct *fontinfo;
     char *fontname;
{
  if (fontinfo->max_bounds.width != 6 ||
      fontinfo->min_bounds.width != 6 ||
      fontinfo->descent + fontinfo->ascent != 10 ||
      fontinfo->min_bounds.lbearing < 0 ||
      fontinfo->max_bounds.rbearing > 6 ||
      fontinfo->max_bounds.ascent > 8 ||
      fontinfo->max_bounds.descent > 2) {
      fprintf (stderr, 
	 "Warning: font '%s'\ndoes not conform to 6x10 character cell rules.\n",
	 fontname);
   }
}

GetFonts()
{
   Font            regular, italic, bold, big;
   int             i, j;
   XGCValues       values;
   XFontStruct    *fontinfo;
   char           *fontname;

   fontname = getdefault("font");
   if (fontname == NULL)
      fontname = NORMAL_FONT;
#ifdef SHOW_DEFAULTS
   show_defaults("fonts", "font", fontname,
      "Default font.");
#endif
   fontinfo = XLoadQueryFont(W_Display, fontname);
   if (fontinfo == NULL) {
      fontinfo = find_font(fontname, _nfonts);
   }
   if (fontinfo == NULL) {
      printf("netrek: Can't find any fonts!\n");
      exit(1);
   }
   checkFont(fontinfo, fontname);
   regular = fontinfo->fid;
   W_Textwidth = fontinfo->max_bounds.width;
   W_Textheight = fontinfo->max_bounds.descent + fontinfo->max_bounds.ascent;
   fonts[1].baseline = fontinfo->max_bounds.ascent;

   fontname = getdefault("boldfont");
   if (fontname == NULL)
      fontname = BOLD_FONT;
#ifdef SHOW_DEFAULTS
   show_defaults("fonts", "boldfont", fontname,
      "Bold font -- used for highlighting.");
#endif
   fontinfo = XLoadQueryFont(W_Display, fontname);
   if (fontinfo == NULL) {
      fontinfo = find_font(fontname, _bfonts);
   }
   if (fontinfo == NULL) {
      bold = regular;
      fonts[2].baseline = fonts[1].baseline;
   } else {
      checkFont(fontinfo, fontname);
      bold = fontinfo->fid;
      fonts[2].baseline = fontinfo->max_bounds.ascent;
      if (fontinfo->max_bounds.width > W_Textwidth)
	 W_Textwidth = fontinfo->max_bounds.width;
      if (fontinfo->max_bounds.descent + fontinfo->max_bounds.ascent > W_Textheight)
	 W_Textheight = fontinfo->max_bounds.descent + fontinfo->max_bounds.ascent;
   }

   fontname = getdefault("italicfont");
   if (fontname == NULL)
      fontname = ITALIC_FONT;
#ifdef SHOW_DEFAULTS
   show_defaults("fonts", "italicfont", fontname,
      "Italic font.");
#endif
   fontinfo = XLoadQueryFont(W_Display, fontname);
   if (fontinfo == NULL) {
      fontinfo = find_font(fontname, _ifonts);
   }
   if (fontinfo == NULL) {
      italic = regular;
      fonts[3].baseline = fonts[1].baseline;
   } else {
      checkFont(fontinfo, fontname);
      italic = fontinfo->fid;
      fonts[3].baseline = fontinfo->max_bounds.ascent;
      if (fontinfo->max_bounds.width > W_Textwidth)
	 W_Textwidth = fontinfo->max_bounds.width;
      if (fontinfo->max_bounds.descent + fontinfo->max_bounds.ascent > W_Textheight)
	 W_Textheight = fontinfo->max_bounds.descent + fontinfo->max_bounds.ascent;
   }

   fontname = getdefault("bigfont");
   if (fontname == NULL)
      fontname = BIG_FONT;
#ifdef SHOW_DEFAULTS
   show_defaults("fonts", "bigfont", fontname,
      "Large font -- used to display number of players per team.");
#endif
   fontinfo = XLoadQueryFont(W_Display, fontname);
   if (fontinfo == NULL) {
      fontinfo = find_font(fontname, _bgfonts);
   }
   if (fontinfo == NULL) {
      big = regular;
      fonts[0].baseline = fonts[1].baseline;
   } else {
      big = fontinfo->fid;
      fonts[0].baseline = fontinfo->max_bounds.ascent;
   }
   for (i = 0; i < NCOLORS; i++) {
      values.font = big;
      colortable[i].contexts[0] = XCreateGC(W_Display, W_Root, GCFont, &values);
      XSetGraphicsExposures(W_Display, colortable[i].contexts[0], False);
      values.font = regular;
      colortable[i].contexts[1] = XCreateGC(W_Display, W_Root, GCFont, &values);
      XSetGraphicsExposures(W_Display, colortable[i].contexts[1], False);

      values.font = bold;
      colortable[i].contexts[2] = XCreateGC(W_Display, W_Root, GCFont, &values);
      XSetGraphicsExposures(W_Display, colortable[i].contexts[2], False);
      values.font = italic;
      colortable[i].contexts[3] = XCreateGC(W_Display, W_Root, GCFont, &values);
      XSetGraphicsExposures(W_Display, colortable[i].contexts[3], False);
      values.function = GXor;
      colortable[i].contexts[BITGC] = XCreateGC(W_Display, W_Root, GCFunction,
						&values);
      XSetGraphicsExposures(W_Display, colortable[i].contexts[BITGC], False);
      {
	 static char     dl[] =
	 {1, 8};
	 /* was 3 */
	 XSetLineAttributes(W_Display, colortable[i].contexts[BITGC],
			    0, LineOnOffDash, CapButt, JoinMiter);
	 XSetDashes(W_Display, colortable[i].contexts[BITGC], 0, dl, 2);
      }
   }
}

XFontStruct    *
find_font(oldf, fonts)
    char           *oldf, **fonts;
{
   XFontStruct    *fi;
   char          **f;
   fprintf(stderr, "netrek: Can't find font %s.  Trying others...\n",
	   oldf);
   for (f = fonts; *f; f++) {
      if (strcmp(*f, oldf) != 0) {
	 if ((fi = XLoadQueryFont(W_Display, *f)))
	    return fi;
      }
   }

  return (NULL);
}

#ifdef FOURPLANEFIX
static unsigned short extrared[8] =
{0x00, 0x20, 0x40, 0x60, 0x80, 0xa0, 0xb0, 0xc0};
static unsigned short extragreen[8] =
{0x40, 0x60, 0x80, 0xa0, 0xb0, 0xc0, 0x00, 0x20};
static unsigned short extrablue[8] =
{0x80, 0xa0, 0xb0, 0xc0, 0x00, 0x20, 0x40, 0x60};
#endif

int bw = 0;
int forceMono = 0;

GetColors()
{
   int             i, j;
   XColor          foo;
   int             white, black;
   unsigned long   pixel;
   unsigned long   planes[3];
   char            defaultstring[100];
   char           *defaults;
#ifdef FOURPLANEFIX
   char           *colorname;
   unsigned long   extracolors[8];
   XColor          colordef;
#endif
   XGCValues	   values;

#ifdef SHOW_DEFAULTS
   show_defaults("display", "forceMono", forceMono?"on":"off",
      "Force black&white even if color supported.");
#endif
   forceMono = booleanDefault("forcemono", forceMono);	/* 11/14/91 TC */

   if (DisplayCells(W_Display, W_Screen) <= 2 || forceMono) {
      bw = 1;
      white = WhitePixel(W_Display, W_Screen);
      black = BlackPixel(W_Display, W_Screen);
      for (i = 0; i < NCOLORS; i++) {
	 if (i != W_Black) {
	    colortable[i].pixelValue = white;
	 } else {
	    colortable[i].pixelValue = black;
	 }
	 if (i == W_Red) {
	    colortable[i].pixmap = XCreatePixmapFromBitmapData(W_Display,
			  W_Root, striped, TILESIDE, TILESIDE, white, black,
					 DefaultDepth(W_Display, W_Screen));
	 } else if (i == W_Yellow) {
	    colortable[i].pixmap = XCreatePixmapFromBitmapData(W_Display,
			     W_Root, gray, TILESIDE, TILESIDE, white, black,
					 DefaultDepth(W_Display, W_Screen));
	 } else if ( i == W_Green ) {
	    colortable[i].pixmap = XCreatePixmapFromBitmapData(W_Display,
					  W_Root, solid, TILESIDE, TILESIDE,
						   colortable[i].pixelValue,
						   colortable[i].pixelValue,
					 DefaultDepth(W_Display, W_Screen));
	 }
	 /* Do other colors need pixmaps?  I don't think so */
	 else
	    colortable[i].pixmap = 0;
	 /*
	  * We assume white is 0 or 1, and black is 0 or 1. We adjust
	  * graphics function based upon who is who.
	  */
	 if (white == 0) {	/* Black is 1 */
	    XSetFunction(W_Display, colortable[i].contexts[BITGC],
			 GXand);
	 }
      }
    } else if (DefaultVisual(W_Display, W_Screen)->class == TrueColor) {
/* Stuff added by sheldon@iastate.edu 5/28/93
 * This is supposed to detect a TrueColor display, and then do a lookup of
 * the colors in default colormap, instead of creating new colormap
 */
      for (i=0; i<NCOLORS; i++) {
	 sprintf(defaultstring, "color.%s", colortable[i].name);

	 defaults=getdefault(defaultstring);
	 if (defaults==NULL) defaults=colortable[i].name;
#ifdef SHOW_DEFAULTS
	 show_defaults("display", defaultstring, defaults,
	    "Color remapping");
#endif
	 XParseColor(W_Display, W_Colormap, defaults, &foo);
	 XAllocColor(W_Display, W_Colormap, &foo);
	 colortable[i].pixelValue = foo.pixel;
	 switch(i){
	    case RED:
	    case GREEN:
	    case YELLOW:
	       colortable[i].pixmap = XCreatePixmapFromBitmapData(W_Display,
		  W_Root, solid, TILESIDE, TILESIDE, foo.pixel, foo.pixel,
		  DefaultDepth(W_Display, W_Screen));
	    default:
	       colortable[i].pixmap = 0;
	 }
      }
   }  else {
#ifdef FOURPLANEFIX
      if (!XAllocColorCells(W_Display, W_Colormap, False, planes, 3,
			    &pixel, 1)) {
	 /* couldn't allocate 3 planes, make a new colormap */
	 W_Colormap = XCreateColormap(W_Display, W_Root, W_Visual, AllocNone);
	 if (!XAllocColorCells(W_Display, W_Colormap, False, planes, 3,
			       &pixel, 1)) {
	    fprintf(stderr, "Cannot create new colormap\n");
	    exit(1);
	 }
	 /*
	  * and fill it with at least 8 more colors so when mouse is inside
	  * netrek windows, use might be able to see his other windows
	  */
	 if (XAllocColorCells(W_Display, W_Colormap, False, NULL, 0,
			      extracolors, 8)) {
	    colordef.flags = DoRed | DoGreen | DoBlue;
	    for (i = 0; i < 8; i++) {
	       colordef.pixel = extracolors[i];
	       colordef.red = extrared[i] << 8;
	       colordef.green = extragreen[i] << 8;
	       colordef.blue = extrablue[i] << 8;
	       XStoreColor(W_Display, W_Colormap, &colordef);
	    }
	 }
      }
#else
      XAllocColorCells(W_Display, W_Colormap, False, planes, 3, &pixel, 1);
#endif

      for (i = 0; i < NCOLORS; i++) {
	 sprintf(defaultstring, "color.%s", colortable[i].name);

	 defaults = getdefault(defaultstring);
	 if (defaults == NULL)
	    defaults = colortable[i].name;
#ifdef SHOW_DEFAULTS
	 show_defaults("display", defaultstring, defaults,
	    "Color remapping.");
#endif
	 XParseColor(W_Display, W_Colormap, defaults, &foo);
	 /*
	  * Black must be the color with all the planes off. That is the only
	  * restriction I concerned myself with in the following case
	  * statement.
	  */
	 switch (i) {
	 case WHITE:
	    foo.pixel = pixel | planes[0] | planes[1] | planes[2];
	    break;
	 case BLACK:
	    foo.pixel = pixel;
	    break;
	 case RED:
	    foo.pixel = pixel | planes[0];
	    break;
	 case CYAN:
	    foo.pixel = pixel | planes[1];
	    break;
	 case YELLOW:
	    foo.pixel = pixel | planes[2];
	    break;
	 case GREY:
	    foo.pixel = pixel | planes[0] | planes[1];
	    break;
	 case GREEN:
	    foo.pixel = pixel | planes[1] | planes[2];
	    break;
	 }
	 XStoreColor(W_Display, W_Colormap, &foo);
	 colortable[i].pixelValue = foo.pixel;
	 switch(i){
	    case RED:
	    case GREEN:
	    case YELLOW:
	       colortable[i].pixmap = XCreatePixmapFromBitmapData(W_Display,
		    W_Root, solid, TILESIDE, TILESIDE, foo.pixel, foo.pixel,
					 DefaultDepth(W_Display, W_Screen));
	       break;
	    default:
	       colortable[i].pixmap = 0;
	 }
      }
   }
   for (i = 0; i < NCOLORS; i++) {
      for (j = 0; j < FONTS + 1; j++) {
	 XSetForeground(W_Display, colortable[i].contexts[j],
			colortable[i].pixelValue);
	 XSetBackground(W_Display, colortable[i].contexts[j],
			colortable[W_Black].pixelValue);
      }
   }

#ifdef nodef
   /* extra stuff */

   /* grey bitmap */
   values.fill_style = FillTiled;
   values.tile = XCreatePixmapFromBitmapData(W_Display, W_Root, lines_bits,
		       lines_width, lines_height, 
		       WhitePixel(W_Display, W_Screen),
		       BlackPixel(W_Display, W_Screen),
		       DefaultDepth(W_Display, W_Screen));
						
   colortable[NCOLORS].contexts[0] = XCreateGC(W_Display, W_Root, 
      GCFillStyle|GCTile,
      &values);
#endif
}

W_Window 
W_MakeWindow(name, x, y, width, height, parent, border, color)
    char           *name;
    int             x, y, width, height;
    W_Window        parent;
    int             border;
    W_Color         color;
{
   struct window  *newwin;
   Window          wparent;
   XSetWindowAttributes attrs;
   Window	   nw;
   int		   gr;

#ifdef DEBUG
   printf("New window...\n");
#endif

   gr = checkGeometry(name, &x, &y, &width, &height);
   checkParent(name, &parent);
   wparent = W_Void2Window(parent)->window;
   attrs.border_pixel = colortable[color].pixelValue;
   attrs.event_mask = KeyPressMask | ButtonPressMask | ExposureMask;

#ifdef MOTION_MOUSE
      attrs.event_mask |= ButtonMotionMask;
#endif

   if(strcmp(name, "message") == 0)		/* TSH */
      attrs.event_mask |= LeaveWindowMask;

#ifdef RJC
   if (strcmp(name, "netrek_icon") == 0)	/* hack, icon should not
						 * select for inpu t */
      attrs.event_mask = ExposureMask;
#endif				/* RJC */

   attrs.background_pixel = colortable[W_Black].pixelValue;
#ifdef RJC
   attrs.do_not_propagate_mask = KeyPressMask | ButtonPressMask | ExposureMask;
#endif				/* RJC */
   newwin = newWindow(
	      XCR(nw, XCreateWindow(W_Display, wparent, x, y, width, height, border,
			    CopyFromParent, InputOutput, CopyFromParent,
			    CWBackPixel | CWEventMask |
			    CWBorderPixel,
			    &attrs)),
			WIN_GRAPH);
   /* top window */
   if (strcmp(name, "netrek") == 0 || strcmp(name, "wait") == 0) {
      char           *s;
      char            buf[80];

#ifdef nodef
      if ((s = getdefault("title")) == NULL)
	 s = name;
#endif				/* nodef */
      /* from -h arg */

      if (strcmp(name, "wait") == 0)
	 strcpy(buf, serverName);
      else
	 sprintf(buf, "Netrek  @  %s", serverName);
      s = buf;

      /* but title on command line will override */
      if (title)
	 s = title;
      XC(XStoreName(W_Display, newwin->window, s));
   } else
      XC(XStoreName(W_Display, newwin->window, name));
#ifdef RJC
   wm_size_hint.min_width =
      wm_size_hint.max_width = wm_size_hint.base_width = width;
   wm_size_hint.min_height =
      wm_size_hint.max_height = wm_size_hint.base_height = height;
   wm_size_hint.flags = USSize | PMinSize | PMaxSize | PBaseSize;
   if(gr & XValue || gr & YValue){
      wm_size_hint.flags |= USPosition;
      wm_size_hint.x = x;
      wm_size_hint.y = y;
   }
   XC(XSetWMNormalHints(W_Display, newwin->window, &wm_size_hint));

   class_hint.res_name = name;
   XC(XSetClassHint(W_Display, newwin->window, &class_hint));
   XC(XSetWMHints(W_Display, newwin->window, &wm_hint));
   if (wparent == W_Root && baseWin){
      /* XXX */
      if(strcmp(name, "wait") != 0 &&
         strcmp(name, "Wait and Read the Motd") != 0)
	 XC(XSetTransientForHint(W_Display, newwin->window,
			      W_Void2Window(baseWin)->window));
   }
#endif				/* RJC */
   newwin->name = strdup(name);
   newwin->width = width;
   newwin->height = height;
   if (wparent != W_Root)
      if (checkMapped(name))
	 W_MapWindow(W_Window2Void(newwin));

#ifdef DEBUG
   printf("New graphics window %d, child of %d\n", newwin, parent);
#endif

#ifdef FOURPLANEFIX
   XC(XSetWindowColormap(W_Display, newwin->window, W_Colormap));
#endif

   return (W_Window2Void(newwin));
}

void 
W_ChangeBorder(window, color)
    W_Window        window;
    int             color;
{
#ifdef DEBUG
   printf("Changing border of %d\n", window);
#endif

   /* fix inexplicable color bug */
   if (bw){
      XC(XSetWindowBorderPixmap(W_Display, W_Void2Window(window)->window,
			     colortable[color].pixmap));
   }
   else
      XC(XSetWindowBorder(W_Display, W_Void2Window(window)->window,
		       colortable[color].pixelValue));


}

void 
W_MapWindow(window)
    W_Window        window;
{
   struct window  *win;

#ifdef DEBUG
   printf("Mapping %d\n", window);
#endif
   win = W_Void2Window(window);
   if (win->mapped)
      return;
   win->mapped = 1;
   XC(XMapRaised(W_Display, win->window));
}

void 
W_UnmapWindow(window)
    W_Window        window;
{
   struct window  *win;

#ifdef DEBUG
   printf("UnMapping %d\n", window);
#endif
   win = W_Void2Window(window);
   if (win->mapped == 0)
      return;
   win->mapped = 0;
   XC(XUnmapWindow(W_Display, win->window));
}

int 
W_IsMapped(window)
    W_Window        window;
{
   struct window  *win;

   win = W_Void2Window(window);
   if (win == NULL)
      return (0);
   return (win->mapped);
}

void 
W_FillArea(window, x, y, width, height, color)
    W_Window        window;
    int             x, y, width, height;
    W_Color         color;
{
   struct window  *win;

#ifdef DEBUG
   printf("Clearing (%d %d) x (%d %d) with %d on %d\n", x, y, width, height,
	  color, window);
#endif
   win = W_Void2Window(window);
   switch (win->type) {
   case WIN_GRAPH:
      XC(XFillRectangle(W_Display, win->window, colortable[color].contexts[0],
		     x, y, width, height));
      break;
   default:
      XC(XFillRectangle(W_Display, win->window, colortable[color].contexts[0],
		     WIN_EDGE + x * W_Textwidth, MENU_PAD + y * W_Textheight,
		     width * W_Textwidth, height * W_Textheight));
   }
}

/* XFIX */

/* one cache for w, one for mapw */
static XRectangle _rcache[2][MAXCACHE];
static int      _rcache_index[2];

static void 
FlushClearAreaCache(win, in)
    Window          win;
    int		    in;
{
   XC(XFillRectangles(W_Display, win, colortable[backColor].contexts[0],
		   _rcache[in], _rcache_index[in]));
   _rcache_index[in] = 0;
}

/* caches provided for local and mapw only */
void 
W_CacheClearArea(window, x, y, width, height)
    W_Window        window;
    int             x, y, width, height;
{
   Window          win = W_Void2Window(window)->window;
   register XRectangle *r;
   register int	   i = (window == w ? 0 : /* mapw */1);

   if (_rcache_index[i] == MAXCACHE)
      FlushClearAreaCache(win, i);

   r = &_rcache[i][_rcache_index[i]++];
   r->x = (short) x;
   r->y = (short) y;
   r->width = (unsigned short) width;
   r->height = (unsigned short) height;
}

void 
W_FlushClearAreaCache(window)
    W_Window        window;
{
   Window          win = W_Void2Window(window)->window;
   register int	   i = (window == w ? 0 : /* mapw */1);

   if (_rcache_index[i])
      FlushClearAreaCache(win, i);
}

/* XFIX: clears now instead of filling. */
void 
W_ClearArea(window, x, y, width, height)
    W_Window        window;
    int             x, y, width, height;
{
   struct window  *win;

#ifdef DEBUG
   printf("Clearing (%d %d) x (%d %d) on %d\n", x, y, width, height,
	  window);
#endif
   win = W_Void2Window(window);
   switch (win->type) {
   case WIN_GRAPH:
      /* XFIX: changed */
      XC(XClearArea(W_Display, win->window, x, y, width, height, False));
      break;
   default:
      /* XFIX: changed */
      XC(XClearArea(W_Display, win->window, WIN_EDGE + x * W_Textwidth,
		 MENU_PAD + y * W_Textheight, width * W_Textwidth, height * W_Textheight, False));
      break;
   }
}

void 
W_ClearWindow(window)
    W_Window        window;
{
#ifdef DEBUG
   printf("Clearing %d\n", window);
#endif
   if (window == NULL)
      return;
   XC(XClearWindow(W_Display, W_Void2Window(window)->window));
}

int 
W_EventsPending()
{
   if (W_isEvent)
      return (1);
   while (XPending(W_Display)) {
      if (W_SpNextEvent(&W_myevent)) {
	 W_isEvent = 1;
	 return (1);
      }
   }
   return (0);
}

int
W_EventsQueued()
{
   int	v;
   return XCR(v, XEventsQueued(W_Display, QueuedAlready));
}

static Bool 
CheckIfEvent(dpy, ev, proc)

   Display	*dpy;
   XEvent	*ev;
   int		(*proc)();
{
   XKeyEvent		*key = (XKeyEvent *)ev;
   XButtonEvent		*button = (XButtonEvent *)ev;
   char			ch;
   W_Event		wevent;
   struct window  	*win;
#ifdef CONTROL_KEY
  int control_key = 0;
#endif

   if(key->send_event == True)
      return False;
   
   switch(ev->type){
      case KeyPress:
#ifdef CONTROL_KEY
	  if (key->state & ControlMask && use_control_key)
            {
	      control_key = 1;
	      key->state = key->state & ~ControlMask;
            }
	  else
	    control_key = 0;
#endif
	 win = findWindow(key->window);
	 if(win != (struct window *)w && win != (struct window *)mapw)
	    return False;
	 if(XLookupString(key, &ch, 1, NULL, NULL) > 0){
	    wevent.type = W_EV_KEY;
	    wevent.Window = W_Window2Void(win);
	    wevent.x = key->x;
	    wevent.y = key->y;
#ifdef CONTROL_KEY
	      if (control_key)
	        wevent.key = (char) ((int) ch + 96);
	      else
	        wevent.key = ch;
#else
	      wevent.key = ch;
#endif
	    return (*proc)(&wevent);
	 }
	 return False;

      case ButtonPress:
         win = findWindow(key->window);
         if(win != (struct window *)w && win != (struct window *)mapw)
            return False;
         wevent.type = W_EV_BUTTON;
         wevent.Window = W_Window2Void(win);
         wevent.x = button->x;
         wevent.y = button->y;
         switch(button->button & 0xf){
            case Button3:
               wevent.key = W_RBUTTON;
               break;
            case Button1:
               wevent.key = W_LBUTTON;
               break;
            case Button2:
               wevent.key = W_MBUTTON;
               break;
         }
	 return (*proc)(&wevent);
      default:
         return False;
   }
}

int
W_SigSafeEvent(proc)

   int		(*proc)();
{
   XEvent	dummy;
   return XCheckIfEvent(W_Display, &dummy, CheckIfEvent, (void *)proc);
}

W_ReadEvents()
{
   XEvent		event;
static
   struct timeval  timeout = { 0, 0 };
   fd_set          readfds;

   FD_ZERO(&readfds);
   FD_SET(ConnectionNumber(W_Display), &readfds);
   if(select(max_fd, &readfds, 0, 0, &timeout) == 1){
      XC(XPeekEvent(W_Display, &event));
      return 1;
   }
   return 0;
}

void 
W_NextEvent(wevent)
    W_Event        *wevent;
{
   if (W_isEvent) {
      *wevent = W_myevent;
      W_isEvent = 0;
      return;
   }
   while (W_SpNextEvent(wevent) == 0);
}

int 
W_SpNextEvent(wevent)
    W_Event        *wevent;
{
   XEvent		event;
   XKeyEvent		*key;
   XButtonEvent 	*button;
   XExposeEvent   	*expose;
#ifdef MOTION_MOUSE
  XMotionEvent *motion;
  static int prev_x, prev_y;
  int thresh;
#endif
   char			ch;
   int			nchars;
   struct window	*win;
#ifdef CONTROL_KEY
  int control_key = 0;
#endif

#ifdef DEBUG
   printf("Getting an event...\n");
#endif
   key = (XKeyEvent *) & event;
   button = (XButtonEvent *) & event;
   expose = (XExposeEvent *) & event;
#ifdef MOTION_MOUSE
  motion = (XMotionEvent *) & event;
#endif

   for (;;) {
      XC(XNextEvent(W_Display, &event));

#ifdef SPTEST
      spt_dupevent(&event);
#else
      if (key->send_event == True)	/* event sent by another client */
	 return 0;
#endif
      win = findWindow(key->window);
      if (win == NULL){
	 /*
	 printf("no window found (%d, local:%d)\n", key->window, 
						    ((struct window *)
						    w)->window);
	 */
	 return (0);
      }
      if ((event.type == KeyPress || event.type == ButtonPress) &&
	  win->type == WIN_MENU) {
	 if (key->y % (W_Textheight + MENU_PAD * 2 + MENU_BAR) >=
	     W_Textheight + MENU_PAD * 2)
	    return (0);
	 key->y = key->y / (W_Textheight + MENU_PAD * 2 + MENU_BAR);
      }
      switch ((int) event.type) {
      case LeaveNotify:	/* for message window -- jfy */
	 if (win == (struct window *) messagew) {
	    W_in_message = 0;
	 }
	 return (0);
	 break;
      case KeyPress:
#ifdef CONTROL_KEY
	  if (key->state & ControlMask && use_control_key)
            {
	      control_key = 1;
	      key->state = key->state & ~ControlMask;
            }
	  else
	    control_key = 0;
#endif
	 if (XLookupString(key, &ch, 1, NULL, NULL) > 0){
	    wevent->type = W_EV_KEY;
	    wevent->Window = W_Window2Void(win);
	    wevent->x = key->x;
	    wevent->y = key->y;
#ifdef CONTROL_KEY
	      if (control_key)
	        wevent->key = (char) ((int) ch + 96);
	      else
	        wevent->key = ch;
#else
	      wevent->key = ch;
#endif
	    return (1);
	 }
	 return (0);
	 break;
      case ButtonPress:
	 wevent->type = W_EV_BUTTON;
	 wevent->Window = W_Window2Void(win);
#ifdef MOTION_MOUSE
          prev_x = wevent->x = button->x;
          prev_y = wevent->y = button->y;
#else
          wevent->x = button->x;
          wevent->y = button->y;
#endif

#ifdef SHIFTED_MOUSE
            if (extended_mouse)
      {
	  if (button->state & ControlMask && button->state & ShiftMask)
	    {
	      switch (button->button & 0xf)
		{
		case Button3:
		  wevent->key = W_RBUTTON4;
		  break;
		case Button1:
		  wevent->key = W_LBUTTON4;
		  break;
		case Button2:
		  wevent->key = W_MBUTTON4;
		  break;
		}
	      return (1);
	    }

	  if (button->state & ShiftMask)
	    {
	      switch (button->button & 0xf)
		{
		case Button3:
		  wevent->key = W_RBUTTON2;
		  break;
		case Button1:
		  wevent->key = W_LBUTTON2;
		  break;
		case Button2:
		  wevent->key = W_MBUTTON2;
		  break;
		}
	      return (1);
	    }

	  if (button->state & ControlMask)
	    {
	      switch (button->button & 0xf)
		{
		case Button3:
		  wevent->key = W_RBUTTON3;
		  break;
		case Button1:
		  wevent->key = W_LBUTTON3;
		  break;
		case Button2:
		  wevent->key = W_MBUTTON3;
		  break;
		}
	      return (1);
	    }
      }
#endif
	  switch (button->button & 0xf)
	    {
	    case Button3:
	      wevent->key = W_RBUTTON;
	      break;
	    case Button1:
	      wevent->key = W_LBUTTON;
	      break;
	    case Button2:
	      wevent->key = W_MBUTTON;
	      break;
	    }
	  return (1);
#ifdef MOTION_MOUSE
	case MotionNotify:
	  if (!steering_mouse)
	      return(0);

	  wevent->type = W_EV_CM_BUTTON;
	  wevent->Window = W_Window2Void (win);
 
          thresh = abs (prev_x - motion->x) + abs (prev_y - motion->y);
 
          if (thresh < user_motion_thresh)
             return (0);
 
          prev_x = wevent->x = motion->x;
          prev_y = wevent->y = motion->y;
#ifdef SHIFTED_MOUSE
	 if (extended_mouse)
	 {
	   if (button->state & ControlMask && button->state & ShiftMask)
	   { 
	     if (motion->state & Button1Mask)
	       wevent->key = W_LBUTTON4;
	     else if (motion->state & Button2Mask)
	       wevent->key = W_MBUTTON4;
	     else if (motion->state & Button3Mask)
	       wevent->key = W_RBUTTON4;

	     return (1);
	   }

	   if (button->state & ShiftMask)
	   {
	     if (motion->state & Button1Mask)
	       wevent->key = W_LBUTTON2;
	     else if (motion->state & Button2Mask)
	       wevent->key = W_MBUTTON2;
	     else if (motion->state & Button3Mask)
	       wevent->key = W_RBUTTON2;
 
	     return (1);
	   }

	   if (button->state & ControlMask)
	   {
	     if (motion->state & Button1Mask)
	       wevent->key = W_LBUTTON3;
	     else if (motion->state & Button2Mask)
	       wevent->key = W_MBUTTON3;
	     else if (motion->state & Button3Mask)
	       wevent->key = W_RBUTTON3;

	     return (1);
	   }
	 }
#endif
	  if (motion->state & Button1Mask)
	    wevent->key = W_LBUTTON;
	  else if (motion->state & Button2Mask)
	    wevent->key = W_MBUTTON;
	  else if (motion->state & Button3Mask)
	    wevent->key = W_RBUTTON;
	 
	  return (1);
#endif
      case Expose:
	 if (expose->count != 0 /* xx */ && W_Window2Void(win) != mapw)
	    return (0);
	 if (win->type == WIN_SCROLL) {
	    configureScrolling(win, expose->x, expose->y, 
				    expose->width, expose->height);
	    redrawScrolling(win);
	    return (0);
	 }
	 if (win->type == WIN_MENU) {
	    redrawMenu(win);
	    return (0);
	 }
	 wevent->x = expose->x;
	 wevent->y = expose->y;
	 wevent->width = expose->width;		/* new -- tsh */
	 wevent->height = expose->height;	/* new -- tsh */
	 wevent->type = W_EV_EXPOSE;
	 wevent->Window = W_Window2Void(win);
	 return (1);

      default:
	 return (0);
	 break;
      }
   }
}

void 
W_MakeLine(window, x0, y0, x1, y1, color)
    W_Window        window;
    int             x0, y0, x1, y1;
    W_Color         color;
{
   Window          win;

#ifdef DEBUG
   printf("Line on %d\n", window);
#endif
   win = W_Void2Window(window)->window;
   XC(XDrawLine(W_Display, win, colortable[color].contexts[0], x0, y0, x1, y1));
}

/* XFIX */

static XSegment _lcache[NCOLORS][MAXCACHE];
static int      _lcache_index[NCOLORS];

static void 
FlushLineCache(win, color)
    Window          win;
    int             color;
{
   XC(XDrawSegments(W_Display, win, colortable[color].contexts[0],
		 _lcache[color], _lcache_index[color]));
   _lcache_index[color] = 0;
}

/* for local window only */
void 
W_CacheLine(window, x0, y0, x1, y1, color)
    W_Window        window;
    int             x0, y0, x1, y1, color;
{
   Window          win = W_Void2Window(window)->window;
   register XSegment *s;

   if (_lcache_index[color] == MAXCACHE)
      FlushLineCache(win, color);

   s = &_lcache[color][_lcache_index[color]++];
   s->x1 = (short) x0;
   s->y1 = (short) y0;
   s->x2 = (short) x1;
   s->y2 = (short) y1;
}

void 
W_FlushLineCaches(window)
    W_Window        window;
{
   Window          win = W_Void2Window(window)->window;
   register        i;
   for (i = 0; i < NCOLORS; i++) {
      if (_lcache_index[i])
	 FlushLineCache(win, i);
   }
}

void 
W_MakeTractLine(window, x0, y0, x1, y1, color)
    W_Window        window;
    int             x0, y0, x1, y1;
    W_Color         color;
{
   Window          win;

#ifdef DEBUG
   printf("Line on %d\n", window);
#endif
   win = W_Void2Window(window)->window;
   XC(XDrawLine(W_Display, win, colortable[color].contexts[BITGC], x0, y0, 
      x1, y1));
}

void 
W_WriteTriangle(window, x, y, s, t, color)
    W_Window        window;
    int             x, y, s;
    int             t;
    W_Color         color;
{
   struct window  *win = W_Void2Window(window);
   XPoint          points[4];

   if (t == 0) {
      points[0].x = x;
      points[0].y = y;
      points[1].x = x + s;
      points[1].y = y - s;
      points[2].x = x - s;
      points[2].y = y - s;
   } else {
      points[0].x = x;
      points[0].y = y;
      points[1].x = x + s;
      points[1].y = y + s;
      points[2].x = x - s;
      points[2].y = y + s;
   }

   /* for XDrawLines */
   points[3].x = points[0].x;
   points[3].y = points[0].y;


   if (fillTriangle)
      XC(XFillPolygon(W_Display, win->window, colortable[color].contexts[0],
		   points, 3, Convex, CoordModeOrigin));
   else
      XC(XDrawLines(W_Display, win->window, colortable[color].contexts[0],
		 points, 4, CoordModeOrigin));
}

void 
W_WriteText(window, x, y, color, str, len, font)
    W_Window        window;
    int             x, y, len;
    W_Color         color;
    W_Font          font;
    char           *str;
{
   struct window  *win;
   int             addr;

#ifdef DEBUG
   printf("Text for %d @ (%d, %d) in %d: [%s]\n", window, x, y, color, str);
#endif
   win = W_Void2Window(window);
   switch (win->type) {
   case WIN_GRAPH:
      addr = fonts[fontNum(font)].baseline;
      XC(XDrawImageString(W_Display, win->window,
		       colortable[color].contexts[fontNum(font)],
		       x, y + addr, str, len));
      break;
   case WIN_SCROLL:
#ifdef nodef
      if(win->mapped){
	 XC(XCopyArea(W_Display, win->window, win->window,
	    colortable[W_White].contexts[0], WIN_EDGE, MENU_PAD + W_Textheight,
		   win->width * W_Textwidth, (win->height - 1) * W_Textheight,
		   WIN_EDGE, MENU_PAD));
	 XC(XClearArea(W_Display, win->window,
		    WIN_EDGE, MENU_PAD + W_Textheight * (win->height - 1),
		    W_Textwidth * win->width, W_Textheight, False));
	 /* was XDrawImageString -- if we clear it, why bother? -tsh */
	 XC(XDrawString(W_Display, win->window,
			  colortable[color].contexts[1],
			  WIN_EDGE, MENU_PAD + W_Textheight * (win->height - 1)
				    + fonts[1].baseline,
			  str, len));
      }
#endif
      AddToScrolling(win, color, str, len);
      
      break;
   case WIN_MENU:
      changeMenuItem(win, y, str, len, color);
      break;
   default:
      addr = fonts[fontNum(font)].baseline;
      XC(XDrawImageString(W_Display, win->window,
		       colortable[color].contexts[fontNum(font)],
	     x * W_Textwidth + WIN_EDGE, MENU_PAD + y * W_Textheight + addr,
		       str, len));
      break;
   }
}

void 
W_MaskText(window, x, y, color, str, len, font)
    W_Window        window;
    int             x, y, len;
    W_Color         color;
    W_Font          font;
    char           *str;
{
   struct window  *win;
   int             addr;

   addr = fonts[fontNum(font)].baseline;
#ifdef DEBUG
   printf("TextMask for %d @ (%d, %d) in %d: [%s]\n", window, x, y, color, str);
#endif
   win = W_Void2Window(window);
   XC(XDrawString(W_Display, win->window,
	  colortable[color].contexts[fontNum(font)], x, y + addr, str, len));
}

W_Icon 
W_StoreBitmap(width, height, data, window)
    int             width, height;
    W_Window        window;
    char           *data;
{
   struct icon    *newicon;
   struct window  *win;
   int             black, white;

#ifdef DEBUG
   printf("Storing bitmap for %d (%d x %d)\n", window, width, height);
   fflush(stdout);
#endif
   win = W_Void2Window(window);
   newicon = (struct icon *) malloc(sizeof(struct icon));
   newicon->width = width;
   newicon->height = height;
   newicon->bitmap = XCR(newicon->bitmap, 
			 XCreateBitmapFromData(W_Display, win->window,
					   data, width, height));
#ifdef nodef
   /* XFIX: changed to Pixmap */
   white = WhitePixel(W_Display, W_Screen);
   black = BlackPixel(W_Display, W_Screen);
   newicon->bitmap = XCR(newicon->bitmap, 
			 XCreatePixmapFromBitmapData(W_Display, W_Root, data,
						 width, height, white, black,
						 DefaultDepth(W_Display,
							      W_Screen)));
#endif	/* nodef */

   newicon->window = win->window;
   newicon->pixmap = 0;
   return (W_Icon2Void(newicon));
}

void 
W_WriteBitmap(x, y, bit, color)
    int             x, y;
    W_Icon          bit;
    W_Color         color;
{
   struct icon    *icon;

   icon = W_Void2Icon(bit);
#ifdef DEBUG
   printf("Writing bitmap to %d\n", icon->window);
#endif
   XC(XCopyPlane(W_Display, icon->bitmap, icon->window,
	 colortable[color].contexts[BITGC], 0, 0, icon->width, icon->height,
	      x, y, 1));

#ifdef nodef
   /* XFIX : copyarea */
   XC(XCopyArea(W_Display, icon->bitmap, icon->window,
	 colortable[color].contexts[BITGC], 0, 0, icon->width, icon->height,
	     x, y));
#endif
}


void 
W_TileWindow(window, bit)
    W_Window        window;
    W_Icon          bit;
{
   Window          win;
   struct icon    *icon;

#ifdef DEBUG
   printf("Tiling window %d\n", window);
#endif
   icon = W_Void2Icon(bit);
   win = W_Void2Window(window)->window;

   if (icon->pixmap == 0) {
      icon->pixmap = XCR(icon->pixmap, XCreatePixmap(W_Display, W_Root,
	      icon->width, icon->height, DefaultDepth(W_Display, W_Screen)));
      XC(XCopyPlane(W_Display, icon->bitmap, icon->pixmap,
	   colortable[W_White].contexts[0], 0, 0, icon->width, icon->height,
		 0, 0, 1));
   }
   XC(XSetWindowBackgroundPixmap(W_Display, win, icon->pixmap));
   XC(XClearWindow(W_Display, win));

   /*
    * if (icon->pixmap==0) { icon->pixmap=XMakePixmap(icon->bitmap,
    * colortable[W_White].pixelValue, colortable[W_Black].pixelValue); }
    * XChangeBackground(win, icon->pixmap); XClear(win);
    */
}

void 
W_UnTileWindow(window)
    W_Window        window;
{
   Window          win;

#ifdef DEBUG
   printf("Untiling window %d\n", window);
#endif
   win = W_Void2Window(window)->window;

   XC(XSetWindowBackground(W_Display, win, colortable[W_Black].pixelValue));
   XC(XClearWindow(W_Display, win));
}

W_Window 
W_MakeTextWindow(name, x, y, width, height, parent, border)
    char           *name;
    int             x, y, width, height;
    W_Window        parent;
    int             border;
{
   struct window  *newwin;
   Window          wparent;
   XSetWindowAttributes attrs;
   Window	   nw;
   int		   gr;

#ifdef DEBUG
   printf("New window...\n");
#endif
   gr = checkGeometry(name, &x, &y, &width, &height);
   checkParent(name, &parent);
   attrs.border_pixel = colortable[W_White].pixelValue;
   attrs.event_mask = ExposureMask;
   attrs.background_pixel = colortable[W_Black].pixelValue;
#ifdef RJC
   attrs.do_not_propagate_mask = ExposureMask;
#endif				/* RJC */
   wparent = W_Void2Window(parent)->window;
   newwin = newWindow(
			XCR(nw, XCreateWindow(W_Display, wparent, x, y,
   width * W_Textwidth + WIN_EDGE * 2, MENU_PAD * 2 + height * W_Textheight,
			border, CopyFromParent, InputOutput, CopyFromParent,
				      CWBackPixel | CWEventMask |
				      CWBorderPixel,
				      &attrs)),
			WIN_TEXT);
#ifdef RJC
   class_hint.res_name = name;
   XC(XSetClassHint(W_Display, newwin->window, &class_hint));
   XC(XSetWMHints(W_Display, newwin->window, &wm_hint));
   if (wparent == W_Root && baseWin != NULL) {
      XC(XSetTransientForHint(W_Display, newwin->window,
			   W_Void2Window(baseWin)->window));
   }
#endif				/* RJC */
   XC(XStoreName(W_Display, newwin->window, name));

   if(gr & XValue || gr & YValue){
      wm_size_hint.flags = USPosition;
      wm_size_hint.x = x;
      wm_size_hint.y = y;
      XC(XSetWMNormalHints(W_Display, newwin->window, &wm_size_hint));
   }

   newwin->name = strdup(name);
   newwin->width = width;
   newwin->height = height;
   if (wparent != W_Root)
      if (checkMapped(name))
	 W_MapWindow(W_Window2Void(newwin));
#ifdef DEBUG
   printf("New text window %d, child of %d\n", newwin, parent);
#endif
#ifdef FOURPLANEFIX
   XC(XSetWindowColormap(W_Display, newwin->window, W_Colormap));
#endif
   return (W_Window2Void(newwin));
}

struct window  *
newWindow(window, type)
    Window          window;
    int             type;
{
   struct window  *newwin;

   newwin = (struct window *) malloc(sizeof(struct window));
   newwin->window = window;
   newwin->type = type;
   newwin->mapped = 0;
  newwin->handle_keydown = 0;
  newwin->handle_keyup = 0;
  newwin->handle_button = 0;
  newwin->handle_expose = 0;
   addToHash(newwin);
   return (newwin);
}

struct window  *
findWindow(window)
    Window          window;
{
   struct windowlist *entry;

   entry = hashtable[hash(window)];
   while (entry != NULL) {
      if (entry->window->window == window)
	 return (entry->window);
      entry = entry->next;
   }
   return (NULL);
}

addToHash(win)
    struct window  *win;
{
   struct windowlist **new;

#ifdef DEBUG
   printf("Adding to %d\n", hash(win->window));
#endif
   new = &hashtable[hash(win->window)];
   while (*new != NULL) {
      new = &((*new)->next);
   }
   *new = (struct windowlist *) malloc(sizeof(struct windowlist));
   (*new)->next = NULL;
   (*new)->window = win;
}

W_Window 
W_MakeScrollingWindow(name, x, y, width, height, parent, border)
    char           *name;
    int             x, y, width, height;
    W_Window        parent;
    int             border;
{
   struct window  *newwin;
   Window          wparent;
   XSetWindowAttributes attrs;
   XSizeHints     *sz_hints;
#ifdef RJC
   int             gx, gy;
#endif				/* RJC */
   Window	   nw;
   int		   gcheck_result;
   struct scrollingWindow	*sw;

#ifdef DEBUG
   printf("New window...\n");
#endif
   gcheck_result = checkGeometry(name, &x, &y, &width, &height);
   checkParent(name, &parent);
   wparent = W_Void2Window(parent)->window;
   attrs.border_pixel = colortable[W_White].pixelValue;
   attrs.event_mask = ExposureMask;
   attrs.background_pixel = colortable[W_Black].pixelValue;
#ifdef RJC
   attrs.do_not_propagate_mask = ExposureMask;
#endif				/* RJC */
   newwin = newWindow(
			XCR(nw, XCreateWindow(W_Display, wparent, x, y,
   width * W_Textwidth + WIN_EDGE * 2, MENU_PAD * 2 + height * W_Textheight,
			border, CopyFromParent, InputOutput, CopyFromParent,
				      CWBackPixel | CWEventMask |
				      CWBorderPixel,
				      &attrs)),
			WIN_SCROLL);
   class_hint.res_name=name;
   sz_hints = XAllocSizeHints();
   sz_hints->width_inc = W_Textwidth;
   sz_hints->height_inc = W_Textheight;
   sz_hints->min_width = WIN_EDGE*2 + W_Textwidth;
   sz_hints->min_height = MENU_PAD*2 + W_Textheight;
   sz_hints->base_width = WIN_EDGE*2;
   sz_hints->base_height = MENU_PAD*2;
   sz_hints->flags = PResizeInc | PMinSize | PBaseSize;
   if(gcheck_result & XValue || gcheck_result & YValue)
      sz_hints->flags |= WMXYHintMode_default();
   XC(XStoreName(W_Display,newwin->window,name));
   XC(XSetWMNormalHints(W_Display,newwin->window,sz_hints));
   XC(XSetClassHint(W_Display, newwin->window, &class_hint));
   XC(XSetWMHints(W_Display, newwin->window, &wm_hint));
   if (wparent==W_Root && baseWin != NULL)
      XC(XSetTransientForHint(W_Display, newwin->window,W_Void2Window(baseWin)->window));

   newwin->name = strdup(name);
   sw = (struct scrollingWindow *) malloc(sizeof(struct scrollingWindow));
   sw->lines = 0;
   sw->updated = 0;
   sw->head = sw->tail = NULL;
   newwin->data = (char *) sw;
   newwin->width = width;
   newwin->height = height;
   if (wparent != W_Root)
      if (checkMapped(name))
	 W_MapWindow(W_Window2Void(newwin));
#ifdef DEBUG
   printf("New scroll window %d, child of %d\n", newwin, parent);
#endif
#ifdef FOURPLANEFIX
   XC(XSetWindowColormap(W_Display, newwin->window, W_Colormap));
#endif
   return (W_Window2Void(newwin));
}

/*
 * Add a string to the string list of the scrolling window.
 */
AddToScrolling(win, color, str, len)
    struct window  *win;
    W_Color         color;
    char           *str;
    int             len;
{
   struct scrollingWindow	*sw;
   struct stringList 		*new;

   /* simple, fast */

   sw = (struct scrollingWindow *) win->data;
   if(sw->lines > 100 /* some large number */){
      /* resuse tail */
      new = sw->tail;
      sw->tail = new->prev;
      new->prev->next = NULL;
      new->prev = NULL;
      new->next = sw->head;
      sw->head->prev = new;
      sw->head = new;
   }
   else {
      new = (struct stringList *) malloc(sizeof(struct stringList));
      new->next = sw->head;
      new->prev = NULL;
      if(sw->head)
	 sw->head->prev = new;
      sw->head = new;
      if(!sw->tail)
	 sw->tail = new;
      sw->lines ++;
      /*
      printf("adding one line \"%s\".\n", str);
      */
   }

   sw->updated ++;	/* mark for W_FlushScrollingWindow */

   strncpy(new->string, str, MAX_TEXT_WIDTH-1);
   new->color = color;

   if(len >= MAX_TEXT_WIDTH){
      new->string[MAX_TEXT_WIDTH-1] = 0;
   }
   else{
      /* we pad out the string with spaces so we don't have to clear
         the window */
      memset(&new->string[len], ' ', MAX_TEXT_WIDTH-len-1);
      new->string[MAX_TEXT_WIDTH-1] = 0;
   }
}

W_FlushScrollingWindow(window)

   W_Window	window;
{
   struct window		*win = W_Void2Window(window);
   struct scrollingWindow	*sw;
   if(!win->mapped)
      return;
   if(win->type != WIN_SCROLL){
      fprintf(stderr, "bad window type in W_FlushScrollingWindow.\n");
      return;
   }
   sw = (struct scrollingWindow *) win->data;
   if(!sw->updated)
      return;
#ifndef NO_COPYAREA
   else {
      register struct stringList	*item;
      register			y;
   
      if(win->height > sw->updated){
	 XC(XCopyArea(W_Display, win->window, win->window, 
	    colortable[W_White].contexts[0], 
	    WIN_EDGE, MENU_PAD + sw->updated * W_Textheight,
	    win->width * W_Textwidth, (win->height - sw->updated)*W_Textheight,
	       WIN_EDGE, MENU_PAD));
      }


      y = (win->height -1) * W_Textheight + fonts[1].baseline;

      for(item = sw->head; item && y > 0 && sw->updated; item=item->next, 
						      y -= W_Textheight,
						      sw->updated -- ){
	 XC(XDrawImageString(W_Display, win->window,
			     colortable[item->color].contexts[1],
			     WIN_EDGE, MENU_PAD + y, item->string, 
			     win->width));
      }
      sw->updated = 0;
   }
#else
   redrawScrolling(win);
#endif
}

redrawScrolling(win)
    struct window  *win;
{
   int             		y;
   struct scrollingWindow	*sw;
   register struct stringList	*item;
   register			l;

   if(!win->mapped)
      return;

   /* simple, fast */

   sw = (struct scrollingWindow *)win->data;
   if(!sw->lines) return;
   sw->updated = 0;

   y = (win->height -1) * W_Textheight + fonts[1].baseline;
   for(item = sw->head; item && y > 0; item = item->next, y -= W_Textheight){
      XC(XDrawImageString(W_Display, win->window,
			  colortable[item->color].contexts[1],
			  WIN_EDGE, MENU_PAD + y, item->string, 
			  win->width));
   }
}

W_Window 
W_MakeMenu(name, x, y, width, height, parent, border)
    char           *name;
    int             x, y, width, height;
    W_Window        parent;
    int             border;
{
   struct window  *newwin;
   struct menuItem *items;
   Window          wparent;
   int             i;
   XSetWindowAttributes attrs;
   Window	   nw;
   int		   gr;

#ifdef DEBUG
   printf("New window...\n");
#endif
   gr = checkGeometry(name, &x, &y, &width, &height);
   checkParent(name, &parent);
   wparent = W_Void2Window(parent)->window;
   attrs.border_pixel = colortable[W_White].pixelValue;
   attrs.event_mask = KeyPressMask | ButtonPressMask | ExposureMask;
   attrs.background_pixel = colortable[W_Black].pixelValue;
#ifdef RJC
   attrs.do_not_propagate_mask = KeyPressMask | ButtonPressMask | ExposureMask;
#endif				/* RJC */
   newwin = newWindow(
			XCR(nw, XCreateWindow(W_Display, wparent, x, y,
				      width * W_Textwidth + WIN_EDGE * 2,
   height * (W_Textheight + MENU_PAD * 2) + (height - 1) * MENU_BAR, border,
				CopyFromParent, InputOutput, CopyFromParent,
				      CWBackPixel | CWEventMask |
				      CWBorderPixel,
				      &attrs)),
			WIN_MENU);
#ifdef RJC
   class_hint.res_name = name;
   XC(XSetClassHint(W_Display, newwin->window, &class_hint));
   XC(XSetWMHints(W_Display, newwin->window, &wm_hint));
   if (wparent == W_Root && baseWin != NULL) {
      XC(XSetTransientForHint(W_Display, newwin->window,
			   W_Void2Window(baseWin)->window));
   }
#endif				/* RJC */
   XC(XStoreName(W_Display, newwin->window, name));
   if(gr & XValue || gr & YValue){
      wm_size_hint.flags = USPosition;
      wm_size_hint.x = x;
      wm_size_hint.y = y;
      XC(XSetWMNormalHints(W_Display, newwin->window, &wm_size_hint));
   }
   newwin->name = strdup(name);
   items = (struct menuItem *) malloc(height * sizeof(struct menuItem));
   for (i = 0; i < height; i++) {
      /* new: allocate once and reuse -tsh */
      items[i].string = (char *)malloc(MAX_TEXT_WIDTH);	
      items[i].color = W_White;
   }
   newwin->data = (char *) items;
   newwin->width = width;
   newwin->height = height;
   if (wparent != W_Root)
      if (checkMapped(name))
	 W_MapWindow(W_Window2Void(newwin));
#ifdef DEBUG
   printf("New menu window %d, child of %d\n", newwin, parent);
#endif
#ifdef FOURPLANEFIX
   XC(XSetWindowColormap(W_Display, newwin->window, W_Colormap));
#endif
   return (W_Window2Void(newwin));
}

redrawMenu(win)
    struct window  *win;
{
   int             count;

   for (count = 1; count < win->height; count++) {
      XC(XFillRectangle(W_Display, win->window,
		     colortable[W_White].contexts[0],
	  0, count * (W_Textheight + MENU_PAD * 2) + (count - 1) * MENU_BAR,
		     win->width * W_Textwidth + WIN_EDGE * 2, MENU_BAR));
   }
   for (count = 0; count < win->height; count++) {
      redrawMenuItem(win, count);
   }
}

redrawMenuItem(win, n)
    struct window  *win;
    int             n;
{
   struct menuItem *items;

   items = (struct menuItem *) win->data;
   XC(XFillRectangle(W_Display, win->window,
		  colortable[W_Black].contexts[0],
	  WIN_EDGE, n * (W_Textheight + MENU_PAD * 2 + MENU_BAR) + MENU_PAD,
		  win->width * W_Textwidth, W_Textheight));
   if (items[n].string) {
      XC(XDrawImageString(W_Display, win->window,
		       colortable[items[n].color].contexts[1],
		       WIN_EDGE,
		       n * (W_Textheight + MENU_PAD * 2 + MENU_BAR) + MENU_PAD + fonts[1].baseline,
		       items[n].string, strlen(items[n].string)));
   }
}

changeMenuItem(win, n, str, len, color)
    struct window  *win;
    int             n;
    char           *str;
    int             len;
    W_Color         color;
{
   struct menuItem *items;
   char           *news;

   items = (struct menuItem *) win->data;

#ifdef nodef
   if (items[n].string) {
      free(items[n].string);
   }
   news = malloc(len + 1);
   strncpy(news, str, len);
   news[len] = 0;
   items[n].string = news;
   items[n].color = color;
#endif
   
   strncpy(items[n].string, str, MAX_TEXT_WIDTH-1);
   items[n].string[MAX_TEXT_WIDTH-1] = 0;
   items[n].color = color;
   redrawMenuItem(win, n);
}

#ifdef nodef

void 
W_DefineCursorFromBitmap(window, mapbits, width, height, maskbits, maskwidth, maskheight)
    W_Window        window;
    unsigned char  *mapbits;
    int             width, height;
    unsigned char  *maskbits;
    int             maskwidth, maskheight;
{
   Cursor          new;
   Pixmap          Cursormaskbit;
   Pixmap          Cursorbit;
   struct window  *win = W_Void2Window(window);
   static XColor   f, b;

   f.pixel = colortable[W_White].pixelValue;
   b.pixel = colortable[W_Black].pixelValue;

   XC(XQueryColor(W_Display, W_Colormap, &f));
   XC(XQueryColor(W_Display, W_Colormap, &b));

   Cursorbit = XCR(Cursorbit, XCreateBitmapFromData(W_Display, win->window, 
				     mapbits, width, height));
   Cursormaskbit = XCR(Cursormaskbit, XCreateBitmapFromData(W_Display, 
				      win->window, maskbits, 
				      maskwidth, maskheight));
   new = XCR(new, XCreatePixmapCursor(W_Display, Cursorbit, Cursormaskbit,
			     &b, &f, 5, 5));
   XC(XRecolorCursor(W_Display, new, &f, &b));
   XC(XDefineCursor(W_Display, win->window, new));
}

void 
W_DefineTCrossCursor(window)
    W_Window        window;
{
static
   Cursor          new;
   struct window  *win = W_Void2Window(window);
   static
   XColor          f, b;

   if(new){
      XC(XDefineCursor(W_Display, win->window, new));
      return;
   }

   f.pixel = colortable[W_White].pixelValue;
   b.pixel = colortable[W_Black].pixelValue;

   XC(XQueryColor(W_Display, W_Colormap, &f));
   XC(XQueryColor(W_Display, W_Colormap, &b));

   new = XCR(new, XCreateFontCursor(W_Display, XC_tcross));
   XC(XRecolorCursor(W_Display, new, &f, &b));
   XC(XDefineCursor(W_Display, win->window, new));
}

void 
W_DefineTrekCursor(window)
    W_Window        window;
{
static
   Cursor          new;
   struct window  *win = W_Void2Window(window);
   XColor          f, b;

   if(new){
      XC(XDefineCursor(W_Display, win->window, new));
      return;
   }

   f.pixel = colortable[W_Yellow].pixelValue;
   b.pixel = colortable[W_Black].pixelValue;

   XC(XQueryColor(W_Display, W_Colormap, &f));
   XC(XQueryColor(W_Display, W_Colormap, &b));

   new = XCR(new, XCreateFontCursor(W_Display, XC_trek));
   XC(XRecolorCursor(W_Display, new, &f, &b));
   XC(XDefineCursor(W_Display, win->window, new));
}

void 
W_DefineWarningCursor(window)
    W_Window        window;
{
static
   Cursor          new;
   struct window  *win = W_Void2Window(window);
   XColor          f, b;

   if(new){
      XC(XDefineCursor(W_Display, win->window, new));
      return;
   }

   f.pixel = colortable[W_Red].pixelValue;
   b.pixel = colortable[W_Black].pixelValue;

   XC(XQueryColor(W_Display, W_Colormap, &f));
   XC(XQueryColor(W_Display, W_Colormap, &b));

   new = XCR(new, XCreateFontCursor(W_Display, XC_pirate));
   XC(XRecolorCursor(W_Display, new, &f, &b));
   XC(XDefineCursor(W_Display, win->window, new));
}

void 
W_DefineArrowCursor(window)
    W_Window        window;
{
static
   Cursor          new;
   struct window  *win = W_Void2Window(window);
   XColor          f, b;

   if(new){
      XC(XDefineCursor(W_Display, win->window, new));
      return;
   }

   f.pixel = colortable[W_Black].pixelValue;
   b.pixel = colortable[W_White].pixelValue;

   XC(XQueryColor(W_Display, W_Colormap, &f));
   XC(XQueryColor(W_Display, W_Colormap, &b));

   new = XCR(new, XCreateFontCursor(W_Display, XC_left_ptr));
   XC(XRecolorCursor(W_Display, new, &f, &b));
   XC(XDefineCursor(W_Display, win->window, new));
}

void 
W_DefineTextCursor(window)
    W_Window        window;
{
static
   Cursor          new;
   struct window  *win = W_Void2Window(window);
   XColor          f, b;

   if(new){
      XC(XDefineCursor(W_Display, win->window, new));
      return;
   }

   f.pixel = colortable[W_Yellow].pixelValue;
   b.pixel = colortable[W_Black].pixelValue;

   XC(XQueryColor(W_Display, W_Colormap, &f));
   XC(XQueryColor(W_Display, W_Colormap, &b));

   new = XCR(new, XCreateFontCursor(W_Display, XC_xterm));
   XC(XRecolorCursor(W_Display, new, &f, &b));
   XC(XDefineCursor(W_Display, win->window, new));
}
#endif

/*
 * ACK!  This only works if you define the same cursor in the * windows, and
 * is really hosed if you do different cursors * in the various windows. I
 * think....  -ERic
 */
void 
W_DefineCursor(window, width, height, bits, mask, xhot, yhot)
    W_Window        window;
    int             width, height, xhot, yhot;
    char           *bits, *mask;
{
   static char    *oldbits = NULL;
   static Cursor   curs;
   Pixmap          cursbits;
   Pixmap          cursmask;
   short          *curdata;
   short          *maskdata;
   struct window  *win;
   XColor          whiteCol, blackCol;

#ifdef DEBUG
   printf("Defining cursor for %d\n", window);
#endif
   win = W_Void2Window(window);
   whiteCol.pixel = colortable[W_White].pixelValue;
   XC(XQueryColor(W_Display, W_Colormap, &whiteCol));
   blackCol.pixel = colortable[W_Black].pixelValue;
   XC(XQueryColor(W_Display, W_Colormap, &blackCol));

   if (!oldbits || oldbits != bits) {
      cursbits = XCR(cursbits, XCreateBitmapFromData(W_Display, win->window,
				       bits, width, height));
      cursmask = XCR(cursmask, XCreateBitmapFromData(W_Display, win->window,
				       mask, width, height));
      oldbits = bits;
      curs = XCR(curs, XCreatePixmapCursor(W_Display, cursbits, cursmask,
				 &whiteCol, &blackCol, xhot, yhot));
      XC(XFreePixmap(W_Display, cursbits));
      XC(XFreePixmap(W_Display, cursmask));
   }
   XC(XDefineCursor(W_Display, win->window, curs));
}

void 
W_Beep()
{
   XC(XBell(W_Display, 0));
}

int 
W_WindowWidth(window)
    W_Window        window;
{
   return (W_Void2Window(window)->width);
}

int 
W_WindowHeight(window)
    W_Window        window;
{
   return (W_Void2Window(window)->height);
}

int 
W_Socket()
{
   return (ConnectionNumber(W_Display));
}

void 
W_DestroyWindow(window)
    W_Window        window;
{
   struct window  *win;

#ifdef DEBUG
   printf("Destroying %d\n", window);
#endif
   win = W_Void2Window(window);
   deleteWindow(win);
   XC(XDestroyWindow(W_Display, win->window));
   free((char *) win);
}

#ifdef nodef
/* NOT USED */
void 
W_SetTransientForHint(w, pw)
    W_Window        w, pw;
{
   XC(XSetTransientForHint(W_Display, W_Void2Window(w)->window,
			pw?(W_Void2Window(pw)->window):0));
}

#endif


deleteWindow(window)
    struct window  *window;
{
   struct windowlist **rm;
   struct windowlist *temp;

   rm = &hashtable[hash(window->window)];
   while (*rm != NULL && (*rm)->window != window) {
      rm = &((*rm)->next);
   }
   if (*rm == NULL) {
      printf("Attempt to delete non-existent window!\n");
      return;
   }
   temp = *rm;
   *rm = temp->next;
   free((char *) temp);
}

void 
W_SetIconWindow(main, icon)
    W_Window        main;
    W_Window        icon;
{
   XWMHints        hints;

   XC(XSetIconName(W_Display, W_Void2Window(icon)->window, W_Void2Window(main)->name));

   hints.flags = IconWindowHint;
   hints.icon_window = W_Void2Window(icon)->window;
   XC(XSetWMHints(W_Display, W_Void2Window(main)->window, &hints));
}

checkParent(name, parent)
    char           *name;
    W_Window       *parent;
{
   char           *adefault;
   char            buf[100];
   int             i;
   struct windowlist *windows;

   sprintf(buf, "%s.parent", name);
   adefault = getdefault(buf);
#ifdef SHOW_DEFAULTS
   show_defaults("windows", buf, adefault?adefault:
	  (*parent?W_Void2Window(*parent)->name:"root"),
      "Window hierarchy parent.");
#endif
   if (adefault == NULL)
      return;
   /* parent must be name of other window or "root" */
   if (strcmpi(adefault, "root") == 0) {
      *parent = W_Window2Void(&myroot);
      return;
   }
   for (i = 0; i < HASHSIZE; i++) {
      windows = hashtable[i];
      while (windows != NULL) {
	 if (strcmpi(adefault, windows->window->name) == 0) {
	    *parent = W_Window2Void(windows->window);
	    return;
	 }
	 windows = windows->next;
      }
   }
}

checkMapped(name)
    char           *name;
{
   char           *adefault;
   char            buf[100];

   sprintf(buf, "%s.mapped", name);
#ifdef SHOW_DEFAULTS
   show_defaults("windows", buf, "off",
      "Map window by default.");
#endif
   return (booleanDefault(buf, 0));
}

void 
W_WarpPointer(window)
    W_Window        window;
{
   static int      warped_from_x = 0, warped_from_y = 0;

   if (window == NULL) {
      if (W_in_message) {
	 XC(XWarpPointer(W_Display, None, W_Root, 0, 0, 0, 0, warped_from_x, warped_from_y));
	 W_in_message = 0;
      }
   } else {
      findMouse(&warped_from_x, &warped_from_y);
      XC(XWarpPointer(W_Display, None, W_Void2Window(window)->window, 0, 0, 0, 0, 0, 0));
      W_in_message = 1;
   }
}

findMouse(x, y)
    int            *x, *y;
{
   Window          theRoot, theChild;
   int             wX, wY, rootX, rootY, status;
   unsigned int    wButtons;

   status = XCR(status, XQueryPointer(W_Display, W_Root, &theRoot, &theChild, &rootX, &rootY, &wX, &wY, &wButtons));
   if (status == True) {
      *x = wX;
      *y = wY;
   } else {
      *x = 0;
      *y = 0;
   }
}

int 
findMouseInWin(x, y, w)
    int            *x, *y;
    W_Window        w;
{
   Window          theRoot, theChild;
   int             wX, wY, rootX, rootY, status;
   unsigned int    wButtons;
   struct window  *win = W_Void2Window(w);
   Window          thisWin = win->window;

   status = XCR(status, XQueryPointer(W_Display, thisWin, &theRoot, &theChild,
			  &rootX, &rootY, &wX, &wY, &wButtons));
   if (status == True) {
      /*
       * if it's in the window we specified then the values returned should
       * be within the with and height of the window
       */
      if (wX <= win->width && wY <= win->height) {
	 *x = wX;
	 *y = wY;
	 return 1;
      }
   }
   *x = 0;
   *y = 0;

   return 0;
}

W_Flush()
{
   XC(XFlush(W_Display));
}

void
W_ResizeWindow(window, neww, newh)              /* TSH 2/93 */
 
   W_Window     window;
   int          neww, newh;
{
   Window       win = W_Void2Window(window)->window;
   XSizeHints *sz_hints;
 
   sz_hints = XAllocSizeHints ();
   sz_hints->min_width = (unsigned int) neww;
   sz_hints->max_width = (unsigned int) neww;
   sz_hints->min_height = (unsigned int) newh;
   sz_hints->max_height = (unsigned int) newh;
   sz_hints->flags = PMinSize | PMaxSize;
   XSetWMNormalHints (W_Display, win, sz_hints);
   XResizeWindow(W_Display, win, (unsigned int) neww, (unsigned int) newh);
}

void
W_ResizeMenu(window, neww, newh)                /* TSH 2/93 */

   W_Window     window;
   int          neww, newh;
{
   W_ResizeWindow(window, neww*W_Textwidth+WIN_EDGE*2,
            newh*(W_Textheight+MENU_PAD*2)+(newh-1)*MENU_BAR);
}

void
W_ResizeTextWindow(window, neww, newh)                /* TSH 2/93 */

   W_Window     window;
   int          neww, newh;
{
   W_ResizeWindow(window, neww*W_Textwidth+WIN_EDGE*2, 
     newh*W_Textheight + WIN_EDGE*2 );
}

int
W_Mono()
{
   return DisplayCells(W_Display, W_Screen) <= 2;
}

void 
W_MakePhaserLine (window, x0, y0, x1, y1, color)
     W_Window window;
     int x0, y0, x1, y1;
     W_Color color;
{
  Window win;

#ifdef DEBUG
  printf ("Line on %d\n", window);
#endif
  win = W_Void2Window (window)->window;
  XC(XDrawLine (W_Display, win, colortable[color].contexts[1], x0, y0, x1, y1));
}

#ifdef TCURSORS
#include "cursors.h"
void 
W_DefineWarningCursor(window)
    W_Window        window;
{
static
   Cursor          new;
   struct window  *win = W_Void2Window(window);
   XColor          f, b;

   if(new){
      XC(XDefineCursor(W_Display, win->window, new));
      return;
   }

   f.pixel = colortable[W_Red].pixelValue;
   b.pixel = colortable[W_Black].pixelValue;

   XC(XQueryColor(W_Display, W_Colormap, &f));
   XC(XQueryColor(W_Display, W_Colormap, &b));

   new = XCR(new, XCreateFontCursor(W_Display, XC_pirate));
   XC(XRecolorCursor(W_Display, new, &f, &b));
   XC(XDefineCursor(W_Display, win->window, new));
}

void 
W_DefineMapcursor (window)

     W_Window window;
{
  Cursor new;
  Pixmap mapcursmaskbit;
  Pixmap mapcursbit;
  struct window *win = W_Void2Window (window);
  char *path;
  static
  XColor f, b;
  int w, h, xh, yh;

  xh = yh = 5;
  f.pixel = colortable[W_White].pixelValue;
  b.pixel = colortable[W_Black].pixelValue;

  XC(XQueryColor (W_Display, W_Colormap, &f));
  XC(XQueryColor (W_Display, W_Colormap, &b));

  mapcursbit = XCreateBitmapFromData (W_Display, win->window, mapcursor_bits,
				      mapcursor_width, mapcursor_height);

  if ((path = getdefault ("mapCursorDef")))
    {

      if (W_LoadBitmap (window, path, &mapcursmaskbit, &w, &h, &xh, &yh) != 1)
        {
           new = XCR(new, XCreateFontCursor(W_Display, XC_tcross));
           XRecolorCursor (W_Display, new, &f, &b);
        }
       else
        {
          mapcursbit = XCreatePixmap (W_Display, win->window, w, h, 1);
          XFillRectangle (W_Display, mapcursbit, 
                      colortable[W_White].contexts[0], 0, 0, w, h);
        }

      new = XCreatePixmapCursor (W_Display, mapcursbit, mapcursmaskbit,
			     &b, &f, xh, yh);
      XRecolorCursor (W_Display, new, &b, &f);
    }
  else
    {
      new = XCR(new, XCreateFontCursor(W_Display, XC_tcross));
      XRecolorCursor (W_Display, new, &f, &b);
    }

  XC(XDefineCursor (W_Display, win->window, new));
}

void 
W_DefineLocalcursor (window)

     W_Window window;
{
  Cursor new;
  Pixmap localcursmaskbit;
  Pixmap localcursbit;
  struct window *win = W_Void2Window (window);
  char *path;
  static
  XColor f, b;
  int w, h, xh, yh;

  xh = yh = 5;
  f.pixel = colortable[W_White].pixelValue;
  b.pixel = colortable[W_Black].pixelValue;

  XC(XQueryColor (W_Display, W_Colormap, &f));
  XC(XQueryColor (W_Display, W_Colormap, &b));

  localcursbit = XCreateBitmapFromData (W_Display, win->window, 
		localcursor_bits, localcursor_width, localcursor_height);

  if ((path = getdefault ("localCursorDef")))
    {
      if (W_LoadBitmap (window, path, &localcursmaskbit, &w, &h, &xh, &yh) != 1)
        {
           new = XCR(new, XCreateFontCursor(W_Display, XC_tcross));
           XRecolorCursor (W_Display, new, &f, &b);
        }
       else
        {
          localcursbit = XCreatePixmap (W_Display, win->window, w, h, 1);
          XFillRectangle (W_Display, localcursbit, 
                      colortable[W_White].contexts[0], 0, 0, w, h);
        }

      new = XCreatePixmapCursor (W_Display, localcursbit, localcursmaskbit,
			     &b, &f, xh, yh);
      XRecolorCursor (W_Display, new, &b, &f);
    }
  else
    {
      new = XCR(new, XCreateFontCursor(W_Display, XC_tcross));
      XRecolorCursor (W_Display, new, &f, &b);
    }

  XC(XDefineCursor (W_Display, win->window, new));
}
void 
W_DefineTrekCursor (window)

     W_Window window;
{
  Cursor new;
  struct window *win = W_Void2Window (window);
  XColor f, b;
  char *path;
  int w, h, xh, yh, mw, mh, mxh, myh;

  f.pixel = colortable[W_Yellow].pixelValue;
  b.pixel = colortable[W_Black].pixelValue;

  XC(XQueryColor (W_Display, W_Colormap, &f));
  XC(XQueryColor (W_Display, W_Colormap, &b));


  if ((path = getdefault ("infoCursorDef")))
    {
      Pixmap pm, mpm;

      if (W_LoadBitmap (window, path, &pm, &w, &h, &xh, &yh) != 1)
        new = XCreateFontCursor (W_Display, XC_trek);
      else
        {
          char mask_path [512];
          
          strcpy (mask_path, path);
          strcat (mask_path, ".mask");

          if (W_LoadBitmap (window, mask_path, &mpm, &mw, &mh, &mxh, &myh) != 1)
            {
              mw = w;
              mh = h;
              mpm = XCreatePixmap (W_Display, win->window, w, h, 
		1);
              XFillRectangle (W_Display, mpm, 
                      colortable[W_White].contexts[0], 0, 0, w, h);
            }

          if ((w != mw) || (h != mh))
            {
              printf ("Cursor and mask are not the same size. %s\n", path);
              new = XCreateFontCursor (W_Display, XC_trek);
            }
          else
            new = XCreatePixmapCursor (W_Display, pm, mpm,
	 		     &b, &f, xh, yh);
        }
    }
  else
    new = XCreateFontCursor (W_Display, XC_trek);

  XC(XRecolorCursor (W_Display, new, &f, &b));
  XC(XDefineCursor (W_Display, win->window, new));
}

void 
W_DefineArrowCursor (window)

     W_Window window;
{
  Cursor new;
  struct window *win = W_Void2Window (window);
  XColor f, b;
  char *path;
  int w, h, xh, yh, mw, mh, mxh, myh;

  f.pixel = colortable[W_Black].pixelValue;
  b.pixel = colortable[W_White].pixelValue;

  XC(XQueryColor (W_Display, W_Colormap, &f));
  XC(XQueryColor (W_Display, W_Colormap, &b));


  if ((path = getdefault ("arrowCursorDef")))
    {
      Pixmap pm, mpm;

      if (W_LoadBitmap (window, path, &pm, &w, &h, &xh, &yh) != 1)
        new = XCreateFontCursor (W_Display, XC_left_ptr);
      else
        {
          char mask_path [512];
          
          strcpy (mask_path, path);
          strcat (mask_path, ".mask");

          if (W_LoadBitmap (window, mask_path, &mpm, &mw, &mh, &mxh, &myh) != 1)
            {
              mw = w;
              mh = h;
              mpm = XCreatePixmap (W_Display, win->window, w, h, 
		1);
              XFillRectangle (W_Display, mpm, 
                      colortable[W_White].contexts[0], 0, 0, w, h);
            }

          if ((w != mw) || (h != mh))
            {
              printf ("Cursor and mask are not the same size. %s\n", path);
              new = XCreateFontCursor (W_Display, XC_left_ptr);
            }
          else
            new = XCreatePixmapCursor (W_Display, pm, mpm,
	 		     &b, &f, xh, yh);
        }
    }
  else
    new = XCreateFontCursor (W_Display, XC_left_ptr);

  XC(XRecolorCursor (W_Display, new, &f, &b));
  XC(XDefineCursor (W_Display, win->window, new));
}

void 
W_DefineTextCursor (window)

     W_Window window;
{
  Cursor new;
  struct window *win = W_Void2Window (window);
  XColor f, b;
  char *path;
  int w, h, xh, yh, mw, mh, mxh, myh;

  f.pixel = colortable[W_Yellow].pixelValue;
  b.pixel = colortable[W_Black].pixelValue;

  XC(XQueryColor (W_Display, W_Colormap, &f));
  XC(XQueryColor (W_Display, W_Colormap, &b));


  if ((path = getdefault ("textCursorDef")))
    {
      Pixmap pm, mpm;

      if (W_LoadBitmap (window, path, &pm, &w, &h, &xh, &yh) != 1)
        new = XCreateFontCursor (W_Display, XC_xterm);
      else
        {
          char mask_path [512];
          
          strcpy (mask_path, path);
          strcat (mask_path, ".mask");

          if (W_LoadBitmap (window, mask_path, &mpm, &mw, &mh, &mxh, &myh) != 1)
            {
              mw = w;
              mh = h;
              mpm = XCreatePixmap (W_Display, win->window, w, h, 
		1);
              XFillRectangle (W_Display, mpm, 
                      colortable[W_White].contexts[0], 0, 0, w, h);
            }

          if ((w != mw) || (h != mh))
            {
              printf ("Cursor and mask are not the same size. %s\n", path);
              new = XCreateFontCursor (W_Display, XC_xterm);
            }
          else
            new = XCreatePixmapCursor (W_Display, pm, mpm,
	 		     &b, &f, xh, yh);
        }
    }
  else
    new = XCreateFontCursor (W_Display, XC_xterm);

  XC(XRecolorCursor (W_Display, new, &f, &b));
  XC(XDefineCursor (W_Display, win->window, new));
}

int W_LoadBitmap (window, path, pixmap, width, height, x_hot, y_hot)
  W_Window window;
  char *path;
  Pixmap *pixmap;
  int *width, *height, *x_hot, *y_hot;

  {
    int status;
    struct window *win;
    win = W_Void2Window (window);

    status = XReadBitmapFile (W_Display, win->window, path, width,
                              height, pixmap, x_hot, y_hot);

    if (status == BitmapSuccess)
      {
        if (*x_hot < 0)
          {
            *x_hot = *width/2;
            *y_hot = *height/2;
          }

        return (1);
      }
    else
      return (0);
  }

void 
W_DefineCursorFromBitmap(window, mapbits, width, height, maskbits, maskwidth, maskheight)
    W_Window        window;
    unsigned char  *mapbits;
    int             width, height;
    unsigned char  *maskbits;
    int             maskwidth, maskheight;
{
   Cursor          new;
   Pixmap          Cursormaskbit;
   Pixmap          Cursorbit;
   struct window  *win = W_Void2Window(window);
   static XColor   f, b;

   f.pixel = colortable[W_White].pixelValue;
   b.pixel = colortable[W_Black].pixelValue;

   XC(XQueryColor(W_Display, W_Colormap, &f));
   XC(XQueryColor(W_Display, W_Colormap, &b));

   Cursorbit = XCR(Cursorbit, XCreateBitmapFromData(W_Display, win->window, 
				     mapbits, width, height));
   Cursormaskbit = XCR(Cursormaskbit, XCreateBitmapFromData(W_Display, 
				      win->window, maskbits, 
				      maskwidth, maskheight));
   new = XCR(new, XCreatePixmapCursor(W_Display, Cursorbit, Cursormaskbit,
			     &b, &f, 5, 5));
   XC(XRecolorCursor(W_Display, new, &f, &b));
   XC(XDefineCursor(W_Display, win->window, new));
}

void 
W_DefineTCrossCursor(window)
    W_Window        window;
{
static
   Cursor          new;
   struct window  *win = W_Void2Window(window);
   static
   XColor          f, b;

   if(new){
      XC(XDefineCursor(W_Display, win->window, new));
      return;
   }

   f.pixel = colortable[W_White].pixelValue;
   b.pixel = colortable[W_Black].pixelValue;

   XC(XQueryColor(W_Display, W_Colormap, &f));
   XC(XQueryColor(W_Display, W_Colormap, &b));

   new = XCR(new, XCreateFontCursor(W_Display, XC_tcross));
   XC(XRecolorCursor(W_Display, new, &f, &b));
   XC(XDefineCursor(W_Display, win->window, new));
}
#endif

W_Window 
W_RenameWindow (window, str)
     struct window *window;
     char *str;
{
  XStoreName (W_Display, window->window, str);
}

#define MAKE_WINDOW_GETTER(name, part) \
  W_Callback name(w) \
     W_Window w; \
  { \
    return W_Void2Window(w)->part; \
  }
 
#define MAKE_WINDOW_SETTER(name, part) \
  W_Callback name(w, c) \
     W_Window w; \
     W_Callback c; \
  { \
    W_Void2Window(w)->part = c; \
  }

MAKE_WINDOW_GETTER (W_GetWindowKeyDownHandler, handle_keydown);
MAKE_WINDOW_SETTER (W_SetWindowKeyDownHandler, handle_keydown);
 
MAKE_WINDOW_GETTER (W_GetWindowKeyUpHandler, handle_keyup);
MAKE_WINDOW_SETTER (W_SetWindowKeyUpHandler, handle_keyup);
 
MAKE_WINDOW_GETTER (W_GetWindowButtonHandler, handle_button);
MAKE_WINDOW_SETTER (W_SetWindowButtonHandler, handle_button);
 
MAKE_WINDOW_GETTER (W_GetWindowExposeHandler, handle_expose);
MAKE_WINDOW_SETTER (W_SetWindowExposeHandler, handle_expose);

#ifdef BEEPLITE
void W_OverlayBitmap (x, y, bit, color)
     int x,y;
     W_Icon bit;
     W_Color color;
{
  struct icon *icon = W_Void2Icon (bit);
#ifdef DEBUG
  printf("Overlaying bitmap to %d\n", icon->window);
#endif
  XCopyPlane (W_Display, icon->bitmap, icon->window,
	      colortable[color].contexts[0], 0, 0, icon->width, icon->height,
	      x, y, 1);
}
#endif
