#include <stdio.h>
#include <string.h>

#ifndef NO_STDLIB_H
#include <stdlib.h>
#endif

#include "config.h"
#include "slang.h"
#include "slwindow.h"

SLwin_Base_Window_Type *SLwin_Root_Window;
SLwin_Base_Window_Type *SLwin_Focus_Window;

/* redraw children in z order */
void slwin_redraw_children (SLwin_Base_Window_Type *win)
{
   win = win->child;
   while (win != NULL)
     {
	(*win->draw)(win);
	win = win->znext;
     }
}


/* Note that the final result should be invariant under interchange 
 * of the indices 0 <--> 1.
 */
int SLwin_compute_overlap (SLwin_Rect_Type *r0, SLwin_Rect_Type *r1,
			   SLwin_Rect_Type *r)
{
   int x, y;
   int x0 = r0->x, y0 = r0->y;
   int x1 = r1->x, y1 = r1->y;
   
   /* Rule out non-overlaping cases */
   
   if ((x1 >= x0 + r0->dx)
       || (x0 >= x1 + r1->dx)
       || (y1 >= y0 + r0->dy)
       || (y0 >= y1 + r1->dy))
     return 0;
   
   /* all other cases indicate an overlap. */

   if (x1 >= x0) x = x1; else x = x0;    r->x = x;
   if (y1 >= y0) y = y1; else y = y0;    r->y = y;
   
   y1 += r1->dy; y0 += r0->dy;
   x1 += r1->dx; x0 += r0->dx;

   if (x1 <= x0) r->dx = x1 - x; else r->dx = x0 - x;
   if (y1 <= y0) r->dy = y1 - y; else r->dy = y0 - y;
   
   return 1;
}

/* The assumption here is that the parents
 * xabs and yabs values are correct.
 */
void SLwin_get_absolute_coords (SLwin_Base_Window_Type *w, SLwin_Rect_Type *r)
{
   int xabs, yabs;
   if (w->parent == NULL)
     {
	xabs = yabs = 0;
     }
   else
     {
	xabs = w->parent->xabs;
	yabs = w->parent->yabs;
     }
	
   r->x = xabs + w->rect.x;
   r->y = yabs + w->rect.y;
   r->dx = w->rect.dx;
   r->dy = w->rect.dy;
}

   
int SLwin_get_clipping_region (SLwin_Base_Window_Type *win, SLwin_Rect_Type *r)
{
   SLwin_Rect_Type r1, r2;
   
   /* Check width/height and fix them if necessary */
   if ((win->rect.dx) < 0) win->rect.dx = 0;
   if ((win->rect.dy) < 0) win->rect.dy = 0;
   
   /* Convert to absolute coordinates */
			      
   if (win->parent == NULL)
     {
	r->x = win->rect.x;
	r->y = win->rect.y;
	r->dx = win->rect.dx;
	r->dy = win->rect.dy;
     }
   else 
     {
	SLwin_get_absolute_coords (win->parent, &r1);
	SLwin_get_absolute_coords (win, &r2);
	if (0 == SLwin_compute_overlap (&r1, &r2, r)) return 0;
     }
   return 1;
}

static void default_redraw_window (SLwin_Base_Window_Type *win)
{
   SLwin_Rect_Type r;
   int len;
   
   if (SLwin_get_clipping_region (win, &r))
     {
	SLsmg_set_color (win->color);
   
	if (win->style_flags & SLWIN_FILLED)
	  SLsmg_fill_region (r.y, r.x, r.dy, r.dx, ' ');
	
	if (win->style_flags & SLWIN_BORDER) 
	  SLsmg_draw_box (r.y, r.x, r.dy, r.dx);
	
	if ((win->text != NULL) && (win->style_flags & SLWIN_TITLED))
	  {
	     SLsmg_gotorc (r.y, r.x + 2);
	     len = strlen (win->text);
	     if (len + 4 >= r.dx) len = r.dx - 4;
	     SLsmg_write_nchars (win->text, len);
	  }
     }
   
   if (win->child) slwin_redraw_children (win);
}

/* destroy children, unlink self and destroy self */
static void default_destroy_window (SLwin_Base_Window_Type *win)
{
   SLwin_Base_Window_Type *child = win->child;
   
   /* Each child destroyed will fixup the links such that the next child
    * will be promoted to oldest if the child is the oldest.  This is 
    * what the code does below.
    */
   while ((child = win->child) != NULL)
     (*child->destroy)(child); 
   
   /* Now do unlink.  If first child, fixup parents link to next child */
   if (win->zprev == NULL)
     {
	if (win->parent != NULL)
	  {
	     win->parent->child = win->znext;
	  }
	/* else win is the root window */
     }
   else win->zprev->znext = win->znext;
   
   if (win->znext != NULL) win->znext->zprev = win->zprev;
   
   if (win->text != NULL) SLFREE (win->text);
   SLFREE (win);
}

SLwin_Base_Window_Type *SLwin_create_window (int dx, int dy, 
					     unsigned int style_flags,
					     int color)
{
   SLwin_Base_Window_Type *w;
   
   if (NULL == (w = (SLwin_Base_Window_Type *) SLMALLOC (sizeof (SLwin_Base_Window_Type))))
     return NULL;
   
   w->rect.x = 0;
   w->rect.y = 0;
   w->rect.dx = dx;
   w->rect.dy = dy;
   
   w->event_mask = 0;
   w->windproc = NULL;
   w->destroy = default_destroy_window;
   w->draw = default_redraw_window;
   
   w->parent = w->child = NULL;
   w->znext = w->zprev = NULL;
   
   w->type = 0;
   w->flags = 0;
   
   w->style_flags = style_flags;
   w->color = color;
   
   w->text = NULL;
   return w;
}

SLwin_Base_Window_Type *SLwin_create_group_window (int dx, int dy, 
						   unsigned int style_flags, 
						   int color)
{
   SLwin_Base_Window_Type *w;

   if ((w = SLwin_create_window (dx, dy, style_flags, color)) == NULL) return NULL;
   w->type = SLWIN_GROUP_TYPE;
   w->xabs = w->yabs = 0;
   return w;
}

int SLwin_insert_window (SLwin_Base_Window_Type *g, SLwin_Base_Window_Type *w, 
			 int x, int y)
{
   SLwin_Base_Window_Type *child;
   
   if ((w == NULL) || (g == NULL)) return 0;
   
   if (g->type != SLWIN_GROUP_TYPE) return 0;
   
   w->parent = g;
   child = g->child;
   
   /* Only children of the root window can have titles. */
   if ((g != SLwin_Root_Window) && (w->style_flags & SLWIN_TITLED))
     {
	w->style_flags &= ~SLWIN_TITLED;
     }
   
   if (child == NULL) g->child = w;
   else
     {
	while (child->znext != NULL) child = child->znext;
	child->znext = w;
	w->zprev = child;
     }
   
   w->rect.x = x;
   w->rect.y = y;
   w->xabs = g->xabs + x;
   w->yabs = g->yabs + y;
   return 1;
}




SLwin_Base_Window_Type *SLwin_init (void)
{
   
   SLwin_Base_Window_Type *w = SLwin_create_group_window (SLtt_Screen_Cols, 
							  SLtt_Screen_Rows,
							  SLWIN_FILLED, 0);
   SLwin_Root_Window = w;
   return SLwin_Root_Window;
}

SLwin_Base_Window_Type *SLwin_create_frame_box (int dx, int dy)
{
   SLwin_Base_Window_Type *w;

   if ((w = SLwin_create_window (dx, dy, 0, 0)) == NULL) return NULL;
   
   return w;
}



static void redraw_label (SLwin_Base_Window_Type *win)
{
   int dx;
   int dborder = 0;
   SLwin_Rect_Type r;
   int len;
   
   if (win->text != NULL) len = strlen (win->text); else len = 0;
   default_redraw_window (win);
   
   if (0 == SLwin_get_clipping_region (win, &r)) return;
   
   /* For now, center label */
   
   if (win->style_flags & SLWIN_BORDER) dborder = 2;
   
   if (r.dy < dborder + 1) return;
   r.y += r.dy / 2;
   
   dx = (r.dx - len) / 2;
   if (dx < dborder / 2) dx = dborder / 2;
   
   r.x += dx;
   if (len > r.dx - dborder) len = r.dx - dborder;
   
   SLsmg_gotorc (r.y, r.x);
   SLsmg_set_color (win->color);
   SLsmg_write_nchars (win->text, len);
}

void SLwin_set_window_text (SLwin_Base_Window_Type *w, char *text)
{
   int len = strlen (text) + 1;
   
   if (w->text != NULL) SLFREE (w->text);
   if (NULL != (w->text = (char *) SLMALLOC (len + 1)))
     {
	strcpy (w->text, text);
     }
}


   
SLwin_Base_Window_Type *SLwin_create_label (int dx, int dy, 
					    unsigned int style_flags,
					    int color,
					    char *text)
{
   SLwin_Base_Window_Type *w;
   int len = strlen (text);
   
   if ((w = SLwin_create_window (dx, dy, style_flags, color)) == NULL) return NULL;
   w->type = SLWIN_LABEL_TYPE;
   
   SLwin_set_window_text (w, text);
   
   w->draw = redraw_label;

   /* Now determine the correct size */
   if (dx < 1)
     {
	w->rect.dx = len;
	if (style_flags & SLWIN_BORDER) w->rect.dx += 2;
     }
   else w->rect.dx = dx;
   
   if (dy < 1)
     {
	w->rect.dy = 1;
	if (style_flags & SLWIN_BORDER) w->rect.dy += 2;
     }
   else w->rect.dy = dy;
   
   return w;
}

void SLwin_move_relative (SLwin_Base_Window_Type *win, int dx, int dy)
{
   win->rect.x += dx;
   win->xabs += dx;
   win->yabs += dy;
   win->rect.y += dy;
}

void SLwin_enlarge_window (SLwin_Base_Window_Type *win, int dx, int dy)
{
   win->rect.dx += dx;
   win->rect.dy += dy;
   if (win->rect.dx < 0) win->rect.dx = 0;
   if (win->rect.dy < 0) win->rect.dy = 0;
}


void SLwin_reset (void)
{
   if (SLwin_Root_Window != NULL)
     {
	(*SLwin_Root_Window->destroy) (SLwin_Root_Window);
	SLwin_Root_Window = NULL;
     }
}

void SLwin_realize_windows (void)
{
   if (SLwin_Root_Window != NULL)
     {
	(*SLwin_Root_Window->draw)(SLwin_Root_Window);
     }
}

SLwin_Base_Window_Type *SLwin_find_focus_window (void)
{
   SLwin_Base_Window_Type *focus, *child;
   
   if ((focus = SLwin_Root_Window) == NULL) return NULL;
   
   
   while (focus->child != NULL) 
     {
	focus = focus->child;
	while (focus->znext != NULL) focus = child->znext;
     }
   return focus;
}

static int is_point_in_window (SLwin_Base_Window_Type *win, int x, int y)
{
   SLwin_Rect_Type r;
   
   SLwin_get_absolute_coords (win, &r);
   return ((x >= r.x) && (x < r.x + r.dx) && (y >= r.y) && (y < r.y + r.dy));
}

/* return top level window at point */
SLwin_Base_Window_Type *find_top_window (SLwin_Base_Window_Type *win, 
					 int x, int y)
{
   SLwin_Base_Window_Type *best = NULL, *trial;
   
   while (win != NULL)
     {
	if ((win->flags & SLWIN_STATIC) == 0)
	  {
	     trial = NULL;
	     if (win->child != NULL)
	       {
		  trial = find_top_window (win->child, x, y);
		  if (trial != NULL) best = trial;
	       }
	     
	     if ((trial == NULL) && is_point_in_window (win, x, y))
	       {
		  best = win;
	       }
	  }
	win = win->znext;
     }
   return best;
}

SLwin_Base_Window_Type *SLwin_find_top_window (int x, int y)
{
   return find_top_window (SLwin_Root_Window, x, y);
}

/* make window the focus window */
void SLwin_set_focus (SLwin_Base_Window_Type *win)
{
   SLwin_Base_Window_Type *prev, *next, *focus;
   
   SLwin_Focus_Window = win;
   do
     {
	/* First, make win the oldest sibling */
	if (win->znext != NULL)
	  {
	     prev = win->zprev;
	     next = focus = win->znext;
	     while (focus->znext != NULL) focus = focus->znext;
	     /* Now focus is the oldest sibling.  Swap them */
	     
	     /* Remove win from the chain */
	     if (prev != NULL) prev->znext = next;
	     next->zprev = prev;
	     
	     /* put it at the head */
	     focus->znext = win;
	     win->znext = NULL;
	     win->zprev = focus;
	     
	     /* Make sure parent is still pointing at smallest child */
	     if ((win->parent != NULL) && (win->parent->child == win))
	       win->parent->child = next;
	  }
   
	/* Now walk back the chain to the root making this branch the oldest. */
	win = win->parent;
     }
   while (win != NULL);
}
#if 0
void SLwin_next_window (void)
{
   SLwin_Base_Window_Type *focus, *prev;
   
   if (SLwin_Focus_Window == NULL) 
     SLwin_Focus_Window = SLwin_find_focus_window ();
   
   if ((focus = SLwin_Focus_Window) == NULL) return;

   while (1)
     {
	/* if there are no siblings, climb the tree until there are */
	while (focus->zprev == NULL)
	  {
	     focus = focus->parent;
	     if (focus == NULL) return;
	  }
	
	prev = focus->zprev;		       /* potential candidate */
	
	/* Make sure this sibling is not simply a static label */
	while ((prev != NULL) && (prev->flags & SLWIN_STATIC))
	  prev = prev->zprev;
	
	if (prev != NULL) break;
	focus = focus->parent;
	if (focus == NULL) return;
     }
   /* Not finished !!! */
}
#endif


int main ()
{
   SLwin_Base_Window_Type *top, *run, *key, *group, *label, *app;
   SLwin_Base_Window_Type *run1, *key1, *group1, *label1, *app1;
   int i;
   
   SLang_init_tty (7, 0, 0);
   SLtt_init_video ();
   SLtt_get_terminfo (); SLsmg_init_smg ();

   fputs ("\033[?9h", stdout); fflush (stdout);
   SLtt_Use_Ansi_Colors = 1;
   
   top = SLwin_init ();
   
   if (top == NULL)
     {
	fprintf (stderr, "Failed to create top.\n");
	exit (-1);
     }
   
   app = SLwin_create_group_window (60, 20, SLWIN_FILLED|SLWIN_BORDER|SLWIN_TITLED, 3);
   SLwin_set_window_text (app, "This window is titled.");
   
   SLwin_insert_window (top, app, 3, 3);
   
   run = SLwin_create_window (3, 14, SLWIN_FILLED | SLWIN_BORDER, 1),
   SLwin_insert_window (app, run, 0, 6);
   
   SLtt_set_color (2, NULL, "yellow", "blue");
   
   label = SLwin_create_label (60, 5, SLWIN_FILLED  | SLWIN_BORDER, 2, 
		       "Press any key to exit."),
   
   SLwin_insert_window (app, label, 5, 11);
   
   group = SLwin_create_group_window (20, 40, SLWIN_FILLED|SLWIN_BORDER, 4);
   SLwin_insert_window (app, group, 5, 5);
   

   key = SLwin_create_label (0, 6, SLWIN_FILLED  | SLWIN_BORDER, 12,
			     "This is a window. And it should be clipped.");
   SLwin_insert_window (group, key, 0, 0);

   /* Second group  */
   app1 = SLwin_create_group_window (60, 20, SLWIN_FILLED|SLWIN_BORDER|SLWIN_TITLED, 14);
   SLwin_set_window_text (app1, "This window is titled.");
   
   SLwin_insert_window (top, app1, 15, 6);
   
   run1 = SLwin_create_window (3, 14, SLWIN_FILLED | SLWIN_BORDER, 1);
   SLwin_insert_window (app1, run1, 0, 6);
   
   label1 = SLwin_create_label (60, 5, SLWIN_FILLED  | SLWIN_BORDER, 2, 
				"Press any key to exit."),
   
   SLwin_insert_window (app1, label1, 5, 11);
   
   group1 = SLwin_create_group_window (20, 40, SLWIN_FILLED|SLWIN_BORDER, 4);
   SLwin_insert_window (app1, group1, 5, 5);
   

   key1 = SLwin_create_label (0, 6, SLWIN_FILLED  | SLWIN_BORDER, 12,
			      "This is a window. And it should be clipped.");
   SLwin_insert_window (group1, key1, 0, 0);

   i = 0;
   while (1)
     {
	SLwin_realize_windows ();
	/* if (SLang_input_pending (5) == 0) SLsmg_refresh (); */
	SLsmg_refresh ();
	
	while (0 == SLang_input_pending (60));
	
	/* SLwin_move_relative (run, 1 - 2 * ((i / 70) % 2) , 0); */
	i++;
	
	if (SLang_input_pending (0))
	  {
	     switch (SLang_getkey ())
	       {
		case 27:
		  SLang_getkey ();
		  switch (SLang_getkey ())
		    {
		     case 'A':
		       SLwin_move_relative (key, 0, -1);
		       break;
		     case 'B':
		       SLwin_move_relative (key, 0, 1);
		       break;
		     case 'C':
		       SLwin_enlarge_window (group, 1, 0);
		       break;
		     case 'D':
		       SLwin_enlarge_window (group, -1, 0);
		       break;
		       
		     case 'M':
			 {
			    int x, y;
			    SLang_getkey ();
			    x = SLang_getkey () - 33;
			    y = SLang_getkey () - 33;
			    SLwin_set_focus (SLwin_find_top_window (x, y));
			    break;
			 }
		     default:
		       goto done;
		    }
		  break;
		  
		case '1':
		  SLwin_set_focus (label);
		  break;
		case '2':
		  SLwin_set_focus (run);
		  break;
		case '3':
		  SLwin_set_focus (key);
		  break;
		case '4':
		  SLwin_set_focus (group);
		  break;
		default:
		  goto done;
	       }
	  }
     }
   
   done:
   SLwin_reset ();
   
   SLsmg_reset_smg ();
   SLtt_reset_video ();
   SLang_reset_tty ();
   fputs ("\033[?9l", stdout); fflush (stdout);
   return 0;
}

   
