/*
 *  linux/ibcs/timod.c
 *
 *  Copyright 1995  Mike Jagdis
 *
 * $Id: timod.c,v 1.7 1995/07/18 08:03:46 mike Exp $
 * $Source: /u3/CVS/ibcs/iBCSemul/timod.c,v $
 */

#ifdef EMU_XTI

#include <linux/config.h>

#include <linux/module.h>
#include <linux/version.h>

#include <asm/segment.h>
#ifndef KERNEL_DS
#include <linux/segment.h>
#endif

#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/ptrace.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/malloc.h>
#include <linux/mm.h>
#include <linux/in.h>

#include <ibcs/ibcs.h>
#include <ibcs/tli.h>

#ifdef IBCS_TRACE
#include <ibcs/trace.h>
#endif


#define timod_mkpri(len) kmalloc(sizeof(struct T_primsg)-sizeof(long)+len, \
					GFP_KERNEL)


static void
timod_ok(int fd, int prim)
{
	struct T_primsg *it;

#ifdef IBCS_TRACE
	if (ibcs_trace & TRACE_STREAMS)
		printk(KERN_DEBUG "timod: [%d] %lx ok ack prim=%d\n",
			current->pid, (unsigned long)current->FD[fd],
			prim);
#endif
	it = timod_mkpri(sizeof(struct T_ok_ack));
	if (it) {
		struct T_ok_ack *ok = (struct T_ok_ack *)&it->type;
		ok->PRIM_type = T_OK_ACK;
		ok->CORRECT_prim = prim;
		it->length = sizeof(struct T_ok_ack);
		it->next = NULL;
		if (!Priv(fd)->plast) {
			Priv(fd)->pfirst = Priv(fd)->plast = it;
			return;
		}
		Priv(fd)->plast->next = it;
		Priv(fd)->plast = it;
	}
}


static void
timod_error(int fd, int prim, int terr, int uerr)
{
	struct T_primsg *it;

#ifdef IBCS_TRACE
	if (ibcs_trace & TRACE_STREAMS)
		printk(KERN_DEBUG "timod: [%d] %lx error prim=%d, TLI=%d, UNIX=%d\n",
			current->pid, (unsigned long)current->FD[fd],
			prim, terr, uerr);
#endif
	it = timod_mkpri(sizeof(struct T_error_ack));
	if (it) {
		struct T_error_ack *err = (struct T_error_ack *)&it->type;
		err->PRIM_type = T_ERROR_ACK;
		err->ERROR_prim = prim;
		err->TLI_error = terr;
		err->UNIX_error = iABI_errors(uerr);
		it->length = sizeof(struct T_error_ack);
		it->next = NULL;
		if (!Priv(fd)->plast) {
			Priv(fd)->pfirst = Priv(fd)->plast = it;
			return;
		}
		Priv(fd)->plast->next = it;
		Priv(fd)->plast = it;
	}
}


static int
do_getmsg(int fd, unsigned long esp,
	char *ctl_buf, int ctl_maxlen, int *ctl_len,
	char *dat_buf, int dat_maxlen, int *dat_len,
	int *flags_p)
{
	int error;
	unsigned long *tsp;
	struct T_unitdata_ind udi;

#ifdef IBCS_TRACE
	if ((ibcs_trace & TRACE_STREAMS) || ibcs_func_p->trace) {
		printk(KERN_DEBUG "iBCS: getmsg %d, 0x%lx[%d], 0x%lx[%d], %x\n",
			fd,
			(unsigned long)ctl_buf, ctl_maxlen,
			(unsigned long)dat_buf, dat_maxlen,
			*flags_p);
	}
#endif

	if (ctl_maxlen >= 0 && Priv(fd)->pfirst) {
		int l = ctl_maxlen <= Priv(fd)->pfirst->length
				? ctl_maxlen : Priv(fd)->pfirst->length;
		error = verify_area(VERIFY_WRITE, ctl_buf, l);
		if (error)
			return error;
#ifdef IBCS_TRACE
		if ((ibcs_trace & TRACE_STREAMS) || ibcs_func_p->trace) {
			printk(KERN_DEBUG "iBCS: priority message %ld\n",
				Priv(fd)->pfirst->type);
		}
#endif
		memcpy_tofs(ctl_buf, ((char *)&Priv(fd)->pfirst->type)
					+ Priv(fd)->offset, l);
		put_fs_long(l, ctl_len);
		if (dat_maxlen >= 0)
			put_fs_long(0, dat_len);
		*flags_p = RS_HIPRI;
		Priv(fd)->pfirst->length -= l;
		if (Priv(fd)->pfirst->length) {
			Priv(fd)->offset += l;
			return MORECTL;
		} else {
			struct T_primsg *it = Priv(fd)->pfirst;
			Priv(fd)->pfirst = it->next;
			if (!Priv(fd)->pfirst)
				Priv(fd)->plast = NULL;
			kfree(it);
			Priv(fd)->offset = 0;
			return 0;
		}
	}

	if (*flags_p & RS_HIPRI)
		return -EAGAIN;

	if (dat_maxlen <= 0) {
		if (dat_maxlen == 0)
			put_fs_long(0, dat_len);
		if (ctl_maxlen >= 0)
			put_fs_long(0, ctl_len);
		return 0;
	}

	tsp = (unsigned long *)(esp - 4*PAGE_SIZE);
	error = verify_area(VERIFY_WRITE, tsp, 6*sizeof(long));
	if (error)
		return error;
	put_fs_long(fd, tsp);
	put_fs_long((unsigned long)dat_buf, tsp+1);
	put_fs_long((dat_maxlen < 0 ? 0 : dat_maxlen), tsp+2);
	put_fs_long(0, tsp+3);
	if (ctl_maxlen > (int)sizeof(udi)) {
		put_fs_long((unsigned long)ctl_buf+sizeof(udi), tsp+4);
		put_fs_long(ctl_maxlen-sizeof(udi), ctl_len);
		put_fs_long((int)ctl_len, tsp+5);
	} else {
		put_fs_long(0, tsp+4);
		put_fs_long(0, ctl_len);
		put_fs_long((int)ctl_len, tsp+5);
	}
	error = SYS(socketcall)(SYS_RECVFROM, tsp);
	if (error < 0)
		return error;
	if (ctl_maxlen > (int)sizeof(udi)) {
		udi.PRIM_type = T_UNITDATA_IND;
		udi.SRC_length = get_fs_long(ctl_len);
		udi.SRC_offset = sizeof(udi);
		udi.OPT_length = udi.OPT_offset = 0;
		memcpy_tofs(ctl_buf, &udi, (int)sizeof(udi));
		put_fs_long(sizeof(udi)+udi.SRC_length, ctl_len);
	} else {
		put_fs_long(0, ctl_len);
	}
	put_fs_long(error, dat_len);

	return 0;
}


int
timod_getmsg(int fd, struct inode *ino, struct pt_regs *regs)
{
	struct strbuf *ctlptr, *datptr;
	int *flags_p, flags;
	int error;
	struct strbuf ctl, dat;

	ctlptr = (struct strbuf *)get_fs_long(((unsigned long *)regs->esp) + 2);
	datptr = (struct strbuf *)get_fs_long(((unsigned long *)regs->esp) + 3);
	flags_p = (int *)get_fs_long(((unsigned long *)regs->esp) + 4);

	error = verify_area(VERIFY_WRITE, flags_p, sizeof(int));
	if (error)
		return error;

	if (ctlptr) {
		error = verify_area(VERIFY_WRITE, ctlptr, sizeof(ctl));
		if (error)
			return error;
		memcpy_fromfs(&ctl, ctlptr, sizeof(ctl));
	} else {
		ctl.maxlen = -1;
	}

	if (datptr) {
		error = verify_area(VERIFY_WRITE, datptr, sizeof(dat));
		if (error)
			return error;
		memcpy_fromfs(&dat, datptr, sizeof(dat));
	} else {
		dat.maxlen = -1;
	}

	error = verify_area(VERIFY_WRITE, flags_p, sizeof(int));
	if (error)
		return error;
	flags = (int)get_fs_long(flags_p);

	error = do_getmsg(fd, regs->esp,
			ctl.buf, ctl.maxlen, &ctlptr->len,
			dat.buf, dat.maxlen, &datptr->len,
			&flags);
	if (!error) {
		put_fs_long(flags, flags_p);
	}
	return error;
}


static int
do_putmsg(int fd, unsigned long esp, char *ctl_buf, int ctl_len,
	char *dat_buf, int dat_len, int flags)
{
	int error, terror;

#ifdef IBCS_TRACE
	if ((ibcs_trace & TRACE_STREAMS) || ibcs_func_p->trace) {
		printk(KERN_DEBUG "iBCS: putmsg %d, 0x%lx[%d], 0x%lx[%d], %x\n",
			fd,
			(unsigned long)ctl_buf, ctl_len,
			(unsigned long)dat_buf, dat_len,
			flags);
	}
#if 0
{ int i;
for (i=0; i<ctl_len; i+=4)
printk(KERN_ERR "ctl: 0x%08lx\n", get_fs_long(ctl_buf+i));
}
if (dat.len > 0) { int i;
for (i=0; i<dat_len; i+=4)
printk(KERN_ERR "dat: 0x%08lx\n", get_fs_long(dat_buf+i));
}
#endif
#endif

	switch (get_fs_long(ctl_buf)) {
		case T_BIND_REQ: {
			struct T_bind_req req;
			unsigned long *tsp;

#ifdef IBCS_TRACE
			if (ibcs_trace & TRACE_STREAMS)
				printk(KERN_DEBUG "timod: [%d] %lx bind req\n",
					current->pid,
					(unsigned long)current->FD[fd]);
#endif
			error = verify_area(VERIFY_READ, ctl_buf, sizeof(req));
			if (error)
				return error;

			if (Priv(fd)->state != T_UNBND) {
				timod_error(fd, T_BIND_REQ, TOUTSTATE, 0);
				return 0;
			}

			tsp = (unsigned long *)(esp - 4*PAGE_SIZE);
			error = verify_area(VERIFY_WRITE, tsp, 3*sizeof(long));
			if (error) {
				timod_error(fd, T_BIND_REQ, TSYSERR, -error);
				return 0;
			}
			memcpy_fromfs(&req, ctl_buf, sizeof(req));
			if (req.ADDR_offset && req.ADDR_length) {
				put_fs_long(fd, tsp);
				put_fs_long((unsigned long)ctl_buf
						+ req.ADDR_offset, tsp+1);
				put_fs_long(req.ADDR_length, tsp+2);
				error = SYS(socketcall)(SYS_BIND, tsp);
			} else {
				error = 0;
			}
			if (!error) {
				struct T_primsg *it;
				it = timod_mkpri(ctl_len);
				if (it) {
					struct T_bind_ack *ack = (struct T_bind_ack *)&it->type;
					memcpy_fromfs(ack, ctl_buf, ctl_len);
					ack->PRIM_type = T_BIND_ACK;
					it->length = ctl_len;
					it->next = NULL;
					Priv(fd)->state = T_IDLE;
					timod_ok(fd, T_BIND_ACK);
					Priv(fd)->plast->next = it;
					Priv(fd)->plast = it;
					return 0;
				}
			}
			switch (error) {
				case -EINVAL:
					terror = TOUTSTATE;
					error = 0;
					break;
				case -EACCES:
					terror = TACCES;
					error = 0;
					break;
				case -EADDRNOTAVAIL:
				case -EADDRINUSE:
					terror = TNOADDR;
					error = 0;
					break;
				default:
					terror = TSYSERR;
					break;
			}
			timod_error(fd, T_BIND_REQ, terror, -error);
			return 0;
		}
		case T_CONN_REQ: {
			struct T_conn_req req;
			unsigned long *tsp;

#ifdef IBCS_TRACE
			if (ibcs_trace & TRACE_STREAMS)
				printk(KERN_DEBUG "timod: [%d] %lx connect req\n",
					current->pid,
					(unsigned long)current->FD[fd]);
#endif
			error = verify_area(VERIFY_READ, ctl_buf, sizeof(req));
			if (error)
				return error;

			if (Priv(fd)->state != T_IDLE) {
				timod_error(fd, T_CONN_REQ, TOUTSTATE, 0);
				return 0;
			}

			tsp = (unsigned long *)(esp - 4*PAGE_SIZE);
			error = verify_area(VERIFY_WRITE, tsp, 3*sizeof(long));
			if (error) {
				timod_error(fd, T_CONN_REQ, TSYSERR, -error);
				return 0;
			}
			memcpy_fromfs(&req, ctl_buf, sizeof(req));
			put_fs_long(fd, tsp);
			put_fs_long((unsigned long)ctl_buf + req.DEST_offset, tsp+1);
			put_fs_long(req.DEST_length, tsp+2);
			error = SYS(socketcall)(SYS_CONNECT, tsp);
			if (!error) {
				struct T_primsg *it;
				it = timod_mkpri(ctl_len);
				if (it) {
					struct T_conn_con *con = (struct T_conn_con *)&it->type;
					memcpy_fromfs(con, ctl_buf, ctl_len);
					con->PRIM_type = T_CONN_CON;
					it->length = ctl_len;
					it->next = NULL;
					Priv(fd)->state = T_DATAXFER;
					timod_ok(fd, T_CONN_REQ);
					Priv(fd)->plast->next = it;
					Priv(fd)->plast = it;
					return 0;
				}
			}
			timod_error(fd, T_CONN_REQ, TSYSERR, -error);
			return 0;
		}
		case T_UNITDATA_REQ: {
			struct T_unitdata_req req;
			unsigned long *tsp;

#ifdef IBCS_TRACE
			if (ibcs_trace & TRACE_STREAMS)
				printk(KERN_DEBUG "timod: [%d] %lx unitdata req\n",
					current->pid,
					(unsigned long)current->FD[fd]);
#endif
			error = verify_area(VERIFY_READ, ctl_buf, sizeof(req));
			if (error)
				return error;

			if (Priv(fd)->state != T_IDLE) {
				timod_error(fd, T_UNITDATA_REQ, TOUTSTATE, 0);
				return 0;
			}

			tsp = (unsigned long *)(esp - 4*PAGE_SIZE);
			error = verify_area(VERIFY_WRITE, tsp, 6*sizeof(long));
			if (error) {
				timod_error(fd, T_UNITDATA_REQ, TSYSERR, -error);
				return 0;
			}
			put_fs_long(fd, tsp);
			put_fs_long((unsigned long)dat_buf, tsp+1);
			put_fs_long(dat_len, tsp+2);
			put_fs_long(0, tsp+3);
			memcpy_fromfs(&req, ctl_buf, sizeof(req));
			if (req.DEST_length > 0) {
				put_fs_long((unsigned long)(ctl_buf+req.DEST_offset), tsp+4);
				put_fs_long(req.DEST_length, tsp+5);
				return SYS(socketcall)(SYS_SENDTO, tsp);
			}
			return SYS(socketcall)(SYS_SEND, tsp);
		}

		case T_UNBIND_REQ:
			timod_ok(fd, T_CONN_REQ);
			break;
	}
	return -EINVAL;
}


int
timod_putmsg(int fd, struct inode *ino, struct pt_regs *regs)
{
	struct strbuf *ctlptr, *datptr;
	int flags, error;
	struct strbuf ctl, dat;

	ctlptr = (struct strbuf *)get_fs_long(((unsigned long *)regs->esp) + 2);
	datptr = (struct strbuf *)get_fs_long(((unsigned long *)regs->esp) + 3);
	flags = (int)get_fs_long(((unsigned long *)regs->esp) + 4);

	error = verify_area(VERIFY_READ, ctlptr, sizeof(ctl));
	if (error)
		return error;
	memcpy_fromfs(&ctl, ctlptr, sizeof(ctl));
	if (ctl.len < 0 && flags)
		return -EINVAL;

	error = verify_area(VERIFY_READ, datptr, sizeof(dat));
	if (error)
		return error;
	memcpy_fromfs(&dat, datptr, sizeof(dat));

	return do_putmsg(fd, regs->esp, ctl.buf, ctl.len,
			dat.buf, dat.len, flags);
}


int
timod_ioctl(struct pt_regs *regs,
	int fd, unsigned int func, void *arg, int len, int *len_p)
{
	struct inode *ino;
	int error;

	if (!current->FD[fd]
	|| !(ino = current->FD[fd]->f_inode)
	|| !ino->i_sock)
		return TBADF;

	error = verify_area(VERIFY_WRITE, len_p, sizeof(int));
	if (error)
		return (-error << 8) | TSYSERR;

	/* SCO/SVR3 starts at 100, ISC/SVR4 starts at 140. */
	switch (func >= 140 ? func-140 : func-100) {
		case 0: /* TI_GETINFO */
		{
			struct T_info_ack it;

#ifdef IBCS_TRACE
			if (ibcs_trace & TRACE_STREAMS)
				printk(KERN_DEBUG "timod: [%d] %lx getinfo\n",
					current->pid,
					(unsigned long)current->FD[fd]);
#endif
			/* The pre-SVR4 T_info_ack structure didn't have
			 * the PROVIDER_flag on the end.
			 */
			error = verify_area(VERIFY_WRITE, arg,
				func == 140
				? sizeof(struct T_info_ack)
				: sizeof(struct T_info_ack)-sizeof(long));
			if (error)
				return (-error << 8) | TSYSERR;

			if (get_fs_long(&((struct T_info_req *)arg)->PRIM_type) != T_INFO_REQ)
				return (EINVAL << 8) | TSYSERR;

			it.PRIM_type = T_INFO_ACK;
			it.CURRENT_state = Priv(fd)->state;
			it.CDATA_size = -2;
			it.DDATA_size = -2;
			it.ADDR_size = sizeof(struct sockaddr);
			it.OPT_size = -1;
			it.TIDU_size = 16384;
			switch ((MINOR(ino->i_rdev)>>4) & 0x0f) {
				case AF_INET:
					it.ADDR_size = sizeof(struct sockaddr);
					break;
				default:
					/* Uh... dunno... */
					it.ADDR_size = 0;
					break;
			}
			switch (ino->u.socket_i.type) {
				case SOCK_STREAM:
					it.ETSDU_size = 1;
					it.TSDU_size = 0;
					it.SERV_type = 2;
					break;
				default:
					it.ETSDU_size = -2;
					it.TSDU_size = 16384;
					it.SERV_type = 3;
					break;
			}

			/* The pre-SVR4 T_info_ack structure didn't have
			 * the PROVIDER_flag on the end.
			 */
			if (func == 140) {
				it.PROVIDER_flag = 0;
				memcpy_tofs(arg, &it, sizeof(it));
				put_fs_long(sizeof(it), len_p);
				return 0;
			}
			memcpy_tofs(arg, &it, sizeof(it)-sizeof(long));
			put_fs_long(sizeof(it)-sizeof(long), len_p);
			return 0;
		}

		case 2: /* TI_BIND */
		{
			int i;
			long prim;

			error = do_putmsg(fd, regs->esp, arg, len,
					NULL, -1, 0);
			if (error)
				return (-error << 8) | TSYSERR;

			/* Get the response. This should be either
			 * T_OK_ACK or T_ERROR_ACK.
			 */
			i = RS_HIPRI;
			error = do_getmsg(fd, regs->esp,
					arg, len, len_p,
					NULL, -1, NULL,
					&i);
			if (error)
				return (-error << 8) | TSYSERR;

			prim = get_fs_long((unsigned long *)arg);
			if (prim == T_ERROR_ACK)
				return (get_fs_long(((unsigned long *)arg)+3) << 8)
					| get_fs_long(((unsigned long *)arg)+2);
			if (prim != T_OK_ACK)
				return TBADSEQ;

			/* Get the response to the bind request. */
			i = RS_HIPRI;
			error = do_getmsg(fd, regs->esp,
					arg, len, len_p,
					NULL, -1, NULL,
					&i);
			if (error)
				return (-error << 8) | TSYSERR;

			return 0;
		}

		case 1: /* TI_OPTMGMT */
		case 3: /* TI_UNBIND */

#ifdef EMU_SVR4
		case 4: /* TI_GETMYNAME */
		case 5: /* TI_SETPEERNAME */
		case 6: /* TI_GETMYNAME */
		case 7: /* TI_SETPEERNAME */
#endif /* EMU_SVR4 */
	}

#ifdef IBCS_TRACE
	if (ibcs_trace & TRACE_STREAMS)
		printk(KERN_ERR "iBCS: STREAMS timod op %d not supported\n",
			func);
#endif
	return TNOTSUPPORT;
}


#ifdef EMU_SVR4
int
ibcs_ioctl_sockmod(int fd, unsigned int func, void *arg)
{
	int error;
	struct inode *ino;

	if (!current->FD[fd]
	|| !(ino = current->FD[fd]->f_inode)
	|| !ino->i_sock)
		return TBADF;

	switch (func) {
		case 101: { /* SI_GETUDATA */
			struct {
				int tidusize, addrsize, optsize, etsdusize;
				int servtype, so_state, so_options;
			} *it = arg;

#ifdef IBCS_TRACE
			if (ibcs_trace & TRACE_STREAMS)
				printk(KERN_DEBUG "sockmod: [%d] %lx getudata\n",
					current->pid,
					(unsigned long)current->FD[fd]);
#endif
			error = verify_area(VERIFY_WRITE, it, sizeof(*it));
			if (error)
				return (-error << 8) | TSYSERR;

			put_fs_long(16384, &it->tidusize);
			put_fs_long(sizeof(struct sockaddr), &it->addrsize);
			put_fs_long(-1, &it->optsize);
			put_fs_long(0, &it->so_state);
			put_fs_long(0, &it->so_options);

			switch (ino->u.socket_i.type) {
				case SOCK_STREAM:
					put_fs_long(1, &it->etsdusize);
					put_fs_long(2, &it->servtype);
					break;
				default:
					put_fs_long(-2, &it->etsdusize);
					put_fs_long(3, &it->servtype);
					break;
			}
			return 0;
		}

		case 102: /* SI_SHUTDOWN */
		case 103: /* SI_LISTEN */
		case 104: /* SI_SETMYNAME */
		case 105: /* SI_SETPEERNAME */
		case 106: /* SI_GETINTRANSIT */
		case 107: /* SI_TCL_LINK */
		case 108: /* SI_TCL_UNLINK */
	}

#ifdef IBCS_TRACE
	if (ibcs_trace & TRACE_STREAMS)
		printk(KERN_ERR "iBCS: STREAMS sockmod op %d not supported\n",
			func);
#endif
	return TNOTSUPPORT;
}
#endif /* EMU_SVR4 */
#endif /* EMU_XTI */
