------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                               C S T A N D                                --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.150 $                            --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Csets;    use Csets;
with Einfo;    use Einfo;
with Gnatvsn;  use Gnatvsn;
with Namet;    use Namet;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Opt;      use Opt;
with Output;   use Output;
with Tbuild;   use Tbuild;
with Ttypes;   use Ttypes;
with Ttypef;   use Ttypef;
with Sem_Util; use Sem_Util;
with Sinfo;    use Sinfo;
with Snames;   use Snames;
with Stand;    use Stand;
with Types;    use Types;
with Uintp;    use Uintp;
with Urealp;   use Urealp;

package body CStand is

   Stloc  : constant Source_Ptr := Standard_Location;
   Staloc : constant Source_Ptr := Standard_Ascii_Location;
   --  Standard abbreviations used throughout this package

   ---------------------------------------
   -- Format of Standard_Version String --
   ---------------------------------------

   --  The purpose of the 16-character string Gnatvsn.Standard_Version is to
   --  make sure that an attempt to bind a program containing units compiled
   --  with incompatible versions of standard does not succeed. In some GCC
   --  ports, the target dependent values in Ttypes may depend on the setting
   --  of command line switches, and we have to be sure that these switches
   --  are set in a compatible manner for all units in a program.

   --  At the moment, we record the sizes of the predefined integer and float
   --  types, and also type Address using the following encoding scheme:

   --     '1'   8 bits
   --     '2'   16 bits
   --     '3'   32 bits
   --     '4'   64 bits
   --     '5'   128 bits
   --     '6'   other

   --  The following declare character positions in the Standard_Version
   --  string used for the predefined types:

   SV_Short_Short_Integer : constant := 1;
   SV_Short_Integer       : constant := 2;
   SV_Integer             : constant := 3;
   SV_Long_Integer        : constant := 4;
   SV_Long_Long_Integer   : constant := 5;
   SV_Short_Float         : constant := 6;
   SV_Float               : constant := 7;
   SV_Long_Float          : constant := 8;
   SV_Long_Long_Float     : constant := 9;
   SV_Address             : constant := 10;

   -----------------------
   -- Local Subprograms --
   -----------------------

   procedure Build_Float_Type (E : Entity_Id; Siz : Int; Digs : Int);
   --  Procedure to build standard predefined float base type. The first
   --  parameter is the entity for the type, and the second parameter
   --  is the size in bits. The third parameter is the digits value.

   procedure Build_Signed_Integer_Type (E : Entity_Id; Siz : Int);
   --  Procedure to build standard predefined signed integer base type. The
   --  first parameter is the entity for the type, and the second parameter
   --  is the size in bits.

   procedure Create_Operators;
   --  Make entries for each of the predefined operators in Standard

   function Encode_Size (Size : Pos) return Character;
   --  Encodes a Size value, using the encoding described in the previous
   --  section on the format of the Standard_Version string.

   function Identifier_For (S : Standard_Entity_Type) return Node_Id;
   --  Returns an identifier node with the same name as the defining
   --  identifier corresponding to the given Standard_Entity_Type value

   function Make_Formal
     (Typ         : Entity_Id;
      Formal_Name : String)
      return        Entity_Id;
   --  Construct entity for subprogram formal with given name and type

   function Make_Integer (V : Uint) return Node_Id;
   --  Builds integer literal with given value

   procedure Make_Name (Id : Entity_Id; Nam : String);
   --  Make an entry in the names table for Nam, and set as Chars field of Id

   function New_Operator (Op : Name_Id) return Entity_Id;
   --  Build entity for standard operator with given name

   function New_Standard_Entity
     (New_Node_Kind : Node_Kind := N_Defining_Identifier)
      return          Entity_Id;
   --  Builds a new entity for Standard, with the Is_Pure flag set and
   --  a source location of Standard_Location

   procedure Set_Integer_Bounds
     (Id  : Entity_Id;
      Typ : Entity_Id;
      Lb  : Uint;
      Hb  : Uint);
   --  Procedure to set bounds for integer type or subtype. Id is the entity
   --  whose bounds and type are to be set. The Typ parameter is the Etype
   --  value for the entity (which will be the same as Id for all predefined
   --  integer base types. The third and fourth parameters are the bounds.

   procedure Set_Float_Bounds
     (Id  : Entity_Id;
      Typ : Entity_Id);
   --  Procedure to set bounds for float type or subtype. Id is the entity
   --  whose bounds and type are to be set. The Typ parameter is the Etype
   --  value for the entity (which will be the same as Id for all predefined
   --  float base types).

   ----------------------
   -- Build_Float_Type --
   ----------------------

   procedure Build_Float_Type (E : Entity_Id; Siz : Int; Digs : Int) is
   begin
      Set_Type_Definition (Parent (E),
        Make_Floating_Point_Definition (Stloc,
          Digits_Expression => Make_Integer (UI_From_Int (Digs))));
      Set_Ekind                      (E, E_Floating_Point_Type);
      Set_Etype                      (E, E);
      Set_Esize                      (E, UI_From_Int (Siz));
      Set_Digits_Value               (E, UI_From_Int (Digs));
      Set_Float_Bounds               (E, E);
      Set_Is_Frozen                  (E);
      Set_Is_Public                  (E);
      Set_Size_Known_At_Compile_Time (E);
   end Build_Float_Type;

   -------------------------------
   -- Build_Signed_Integer_Type --
   -------------------------------

   procedure Build_Signed_Integer_Type (E : Entity_Id; Siz : Int) is
      U2Siz1 : constant Uint := 2 ** (Siz - 1);
      Lbound : constant Uint := -U2Siz1;
      Ubound : constant Uint := U2Siz1 - 1;

   begin
      Set_Type_Definition (Parent (E),
        Make_Signed_Integer_Type_Definition (Stloc,
          Low_Bound  => Make_Integer (Lbound),
          High_Bound => Make_Integer (Ubound)));

      Set_Ekind                      (E, E_Signed_Integer_Type);
      Set_Etype                      (E, E);
      Set_Esize                      (E, UI_From_Int (Siz));
      Set_Integer_Bounds             (E, E, Lbound, Ubound);
      Set_Is_Frozen                  (E);
      Set_Is_Public                  (E);
      Set_Size_Known_At_Compile_Time (E);
   end Build_Signed_Integer_Type;

   ----------------------
   --  Create_Standard --
   ----------------------

   --  The tree for the package Standard is prefixed to all compilations.
   --  Several entities required by semantic analysis are denoted by global
   --  variables that are initialized to point to the corresponding
   --  occurences in STANDARD. The visible entities of STANDARD are
   --  created here. The private entities defined in STANDARD are created
   --  by Initialize_Standard in the semantics module.

   procedure Create_Standard is
      Decl_S : List_Id;
      --  List of declarations in Standard

      Decl_A : List_Id;
      --  List of declarations in Ascii

      Decl       : Node_Id;
      Pspec      : Node_Id;
      Tdef_Node  : Node_Id;
      Ident_Node : Node_Id;
      Ccode      : Char_Code;
      E_Id       : Entity_Id;
      R_Node     : Node_Id;
      B_Node     : Node_Id;

      procedure Build_Exception (S : Standard_Entity_Type);
      --  Procedure to declare given entity as an exception

      procedure Build_Exception (S : Standard_Entity_Type) is
      begin
         Set_Ekind (Standard_Entity (S), E_Exception);
         Set_Etype (Standard_Entity (S), Standard_Exception_Type);
         Set_Is_Public (Standard_Entity (S));
         Decl :=
           Make_Exception_Declaration (Stloc,
             Defining_Identifier => Standard_Entity (S));
         Append (Decl, Decl_S);
      end Build_Exception;

   --  Start of processing for Create_Standard

   begin
      Decl_S := New_List;

      --  First step is to create defining identifiers for each entity

      for S in Standard_Entity_Type loop
         declare
            S_Name : constant String := Standard_Entity_Type'Image (S);
            --  Name of entity (note we skip S_ at the start)

            Ident_Node : Node_Id;
            --  Defining identifier node

         begin
            Ident_Node := New_Standard_Entity;
            Make_Name (Ident_Node, S_Name (3 .. S_Name'Length));
            Standard_Entity (S) := Ident_Node;
         end;
      end loop;

      --  Create package declaration node for package Standard

      Standard_Package_Node := New_Node (N_Package_Declaration, Stloc);

      Pspec := New_Node (N_Package_Specification, Stloc);
      Set_Specification (Standard_Package_Node, Pspec);

      Set_Defining_Unit_Name (Pspec, Standard_Standard);
      Set_Visible_Declarations (Pspec, Decl_S);

      Set_Ekind (Standard_Standard, E_Package);
      Set_Is_Pure (Standard_Standard);

      --  Create type declaration nodes for standard types

      for S in S_Types loop
         Decl := New_Node (N_Full_Type_Declaration, Stloc);
         Set_Defining_Identifier (Decl, Standard_Entity (S));
         Set_Is_Frozen (Standard_Entity (S));
         Set_Is_Public (Standard_Entity (S));
         Append (Decl, Decl_S);
      end loop;

      --  Create type definition node for type Boolean. The Size is set to
      --  1 as required by Ada 95 and current ARG interpretations for Ada/83.

      Tdef_Node := New_Node (N_Enumeration_Type_Definition, Stloc);
      Set_Literals (Tdef_Node, New_List);
      Append (Standard_False, Literals (Tdef_Node));
      Append (Standard_True, Literals (Tdef_Node));
      Set_Type_Definition (Parent (Standard_Boolean), Tdef_Node);

      Set_Ekind         (Standard_Boolean, E_Enumeration_Type);
      Set_First_Literal (Standard_Boolean, Standard_False);
      Set_Etype         (Standard_Boolean, Standard_Boolean);
      Set_Esize         (Standard_Boolean, Uint_1);
      Set_Size_Known_At_Compile_Time
                        (Standard_Boolean);

      Set_Ekind           (Standard_True, E_Enumeration_Literal);
      Set_Etype           (Standard_True, Standard_Boolean);
      Set_Enumeration_Pos (Standard_True, Uint_1);
      Set_Enumeration_Rep (Standard_True, Uint_1);

      Set_Ekind           (Standard_False, E_Enumeration_Literal);
      Set_Etype           (Standard_False, Standard_Boolean);
      Set_Enumeration_Pos (Standard_False, Uint_0);
      Set_Enumeration_Rep (Standard_False, Uint_0);

      --  For the bounds of Boolean, we create a range node corresponding to

      --    range False .. True

      --  where the occurrences of the literals must point to the
      --  corresponding  definition.

      R_Node := New_Node (N_Range, Stloc);
      B_Node := New_Node (N_Identifier, Stloc);
      Set_Chars  (B_Node, Chars (Standard_False));
      Set_Entity (B_Node,  Standard_False);
      Set_Etype  (B_Node, Standard_Boolean);
      Set_Is_Static_Expression (B_Node);
      Set_Low_Bound  (R_Node, B_Node);

      B_Node := New_Node (N_Identifier, Stloc);
      Set_Chars  (B_Node, Chars (Standard_True));
      Set_Entity (B_Node,  Standard_True);
      Set_Etype  (B_Node, Standard_Boolean);
      Set_Is_Static_Expression (B_Node);
      Set_High_Bound (R_Node, B_Node);

      Set_Scalar_Range (Standard_Boolean, R_Node);

      --  Create type definition nodes for predefined integer types

      Build_Signed_Integer_Type
        (Standard_Short_Short_Integer, Standard_Short_Short_Integer_Size);

      Build_Signed_Integer_Type
        (Standard_Short_Integer, Standard_Short_Integer_Size);

      Build_Signed_Integer_Type
        (Standard_Integer, Standard_Integer_Size);

      Build_Signed_Integer_Type
        (Standard_Long_Integer, Standard_Long_Integer_Size);

      Build_Signed_Integer_Type
        (Standard_Long_Long_Integer, Standard_Long_Long_Integer_Size);

      --  Create type definition nodes for predefined float types

      Build_Float_Type
        (Standard_Short_Float,
         Standard_Short_Float_Size,
         Standard_Short_Float_Digits);

      Build_Float_Type
        (Standard_Float,
         Standard_Float_Size,
         Standard_Float_Digits);

      Build_Float_Type
        (Standard_Long_Float,
         Standard_Long_Float_Size,
         Standard_Long_Float_Digits);

      Build_Float_Type
        (Standard_Long_Long_Float,
         Standard_Long_Long_Float_Size,
         Standard_Long_Long_Float_Digits);

      --  Create type definition node for type Character. Note that we do not
      --  set the Literals field, since type Character is handled with special
      --  routine that do not need a literal list.

      Tdef_Node := New_Node (N_Enumeration_Type_Definition, Stloc);
      Set_Type_Definition (Parent (Standard_Character), Tdef_Node);

      Set_Ekind (Standard_Character, E_Enumeration_Type);
      Set_Etype (Standard_Character, Standard_Character);
      Set_Esize (Standard_Character, UI_From_Int (Standard_Character_Size));

      Set_Is_Character_Type (Standard_Character, True);
      Set_Size_Known_At_Compile_Time (Standard_Character);

      --  Create the bounds for type Character.

      R_Node := New_Node (N_Range, Stloc);

      --  Low bound for type Character (Standard.Nul)

      B_Node := New_Node (N_Character_Literal, Stloc);
      Set_Is_Static_Expression (B_Node);
      Set_Chars                (B_Node, No_Name);
      Set_Char_Literal_Value   (B_Node, 16#00#);
      Set_Entity               (B_Node,  Empty);
      Set_Etype                (B_Node, Standard_Character);
      Set_Low_Bound (R_Node, B_Node);

      --  High bound for type Character

      B_Node := New_Node (N_Character_Literal, Stloc);
      Set_Is_Static_Expression (B_Node);
      Set_Chars                (B_Node, No_Name);
      Set_Char_Literal_Value   (B_Node, 16#FF#);
      Set_Entity               (B_Node,  Empty);
      Set_Etype                (B_Node, Standard_Character);
      Set_High_Bound (R_Node, B_Node);

      Set_Scalar_Range (Standard_Character, R_Node);

      --  Create type definition for type Wide_Character. Note that we do not
      --  set the Literals field, since type Wide_Character is handled with
      --  special routines that do not need a literal list.

      Tdef_Node := New_Node (N_Enumeration_Type_Definition, Stloc);
      Set_Type_Definition (Parent (Standard_Wide_Character), Tdef_Node);

      Set_Ekind             (Standard_Wide_Character, E_Enumeration_Type);
      Set_Etype             (Standard_Wide_Character, Standard_Wide_Character);
      Set_Esize             (Standard_Wide_Character, Uint_16);
      Set_Is_Character_Type (Standard_Wide_Character, True);
      Set_Size_Known_At_Compile_Time
                            (Standard_Wide_Character);

      --  Create the bounds for type Wide_Character.

      R_Node := New_Node (N_Range, Stloc);

      --  Low bound for type Wide_Character

      B_Node := New_Node (N_Character_Literal, Stloc);
      Set_Is_Static_Expression (B_Node);
      Set_Chars                (B_Node, No_Name);    --  ???
      Set_Char_Literal_Value   (B_Node, 16#0000#);
      Set_Entity               (B_Node,  Empty);
      Set_Etype                (B_Node, Standard_Wide_Character);
      Set_Low_Bound (R_Node, B_Node);

      --  High bound for type Wide_Character

      B_Node := New_Node (N_Character_Literal, Stloc);
      Set_Is_Static_Expression (B_Node);
      Set_Chars                (B_Node, No_Name);    --  ???
      Set_Char_Literal_Value   (B_Node, 16#FFFF#);
      Set_Entity               (B_Node,  Empty);
      Set_Etype                (B_Node, Standard_Wide_Character);
      Set_High_Bound           (R_Node, B_Node);

      Set_Scalar_Range (Standard_Wide_Character, R_Node);

      --  Create type definition node for type String

      Tdef_Node := New_Node (N_Unconstrained_Array_Definition, Stloc);
      Set_Subtype_Indication (Tdef_Node, Identifier_For (S_Character));
      Set_Subtype_Marks      (Tdef_Node, New_List);
      Append (Identifier_For (S_Positive), Subtype_Marks (Tdef_Node));
      Set_Type_Definition (Parent (Standard_String), Tdef_Node);

      Set_Ekind          (Standard_String, E_String_Type);
      Set_Etype          (Standard_String, Standard_String);
      Set_Component_Type (Standard_String, Standard_Character);
      Set_Esize          (Standard_String, Uint_0);

      --  Set index type of String

      E_Id := First
        (Subtype_Marks (Type_Definition (Parent (Standard_String))));
      Set_First_Index (Standard_String, E_Id);
      Set_Entity (E_Id, Standard_Positive);
      Set_Etype (E_Id, Standard_Positive);

      --  Create type definition node for type Wide_String

      Tdef_Node := New_Node (N_Unconstrained_Array_Definition, Stloc);
      Set_Subtype_Indication (Tdef_Node, Identifier_For (S_Wide_Character));
      Set_Subtype_Marks (Tdef_Node, New_List);
      Append (Identifier_For (S_Positive), Subtype_Marks (Tdef_Node));
      Set_Type_Definition (Parent (Standard_Wide_String), Tdef_Node);

      Set_Ekind          (Standard_Wide_String, E_String_Type);
      Set_Etype          (Standard_Wide_String, Standard_Wide_String);
      Set_Component_Type (Standard_Wide_String, Standard_Wide_Character);
      Set_Esize          (Standard_Wide_String, Uint_0);

      --  Set index type of Wide_String

      E_Id := First
        (Subtype_Marks (Type_Definition (Parent (Standard_Wide_String))));
      Set_First_Index (Standard_Wide_String, E_Id);
      Set_Entity (E_Id, Standard_Positive);
      Set_Etype (E_Id, Standard_Positive);

      --  Create subtype declaration for Natural

      Decl := New_Node (N_Subtype_Declaration, Stloc);
      Set_Defining_Identifier (Decl, Standard_Natural);
      Set_Subtype_Indication (Decl,
        New_Occurrence_Of (Standard_Integer, Stloc));
      Append (Decl, Decl_S);

      Set_Ekind     (Standard_Natural, E_Signed_Integer_Subtype);
      Set_Etype     (Standard_Natural, Standard_Integer);
      Set_Esize     (Standard_Natural, Esize (Standard_Integer));
      Set_Size_Known_At_Compile_Time
                    (Standard_Natural);
      Set_Integer_Bounds
        (Id  => Standard_Natural,
         Typ => Standard_Integer,
         Lb  => Uint_0,
         Hb  => Intval (High_Bound (Scalar_Range (Standard_Integer))));
      Set_Is_Frozen (Standard_Natural);
      Set_Is_Public (Standard_Natural);

      --  Create subtype declaration for Positive

      Decl := New_Node (N_Subtype_Declaration, Stloc);
      Set_Defining_Identifier (Decl, Standard_Positive);
      Set_Subtype_Indication (Decl,
        New_Occurrence_Of (Standard_Integer, Stloc));
      Append (Decl, Decl_S);

      Set_Ekind     (Standard_Positive, E_Signed_Integer_Subtype);
      Set_Etype     (Standard_Positive, Standard_Integer);
      Set_Esize     (Standard_Positive, Esize (Standard_Integer));
      Set_Size_Known_At_Compile_Time
                    (Standard_Positive);

      Set_Integer_Bounds
        (Id  => Standard_Positive,
         Typ => Standard_Integer,
         Lb  => Uint_1,
         Hb  => Intval (High_Bound (Scalar_Range (Standard_Integer))));
      Set_Is_Frozen (Standard_Positive);
      Set_Is_Public (Standard_Positive);

      --  Create subtype declaration for Duration. For the moment, this is
      --  represented as a Long_Float value, eventually it should be a 64-bit
      --  fixed-point type.

      Decl := New_Node (N_Subtype_Declaration, Stloc);
      Set_Defining_Identifier (Decl, Standard_Duration);
      Set_Subtype_Indication (Decl,
        New_Occurrence_Of (Standard_Long_Float, Stloc));
      Append (Decl, Decl_S);

      Set_Ekind        (Standard_Duration, E_Floating_Point_Subtype);
      Set_Etype        (Standard_Duration, Standard_Long_Float);
      Set_Esize        (Standard_Duration, Esize (Standard_Long_Float));
      Set_Scalar_Range (Standard_Duration,
                         Scalar_Range (Standard_Long_Float));
      Set_Digits_Value (Standard_Duration,
                         UI_From_Int (Standard_Long_Float_Digits));

      Set_Size_Known_At_Compile_Time (Standard_Duration);

      --  Create declaration for package Ascii

      Decl := New_Node (N_Package_Declaration, Stloc);
      Append (Decl, Decl_S);

      Pspec := New_Node (N_Package_Specification, Stloc);
      Set_Specification (Decl, Pspec);

      Set_Defining_Unit_Name (Pspec, Standard_Entity (S_Ascii));
      Set_Ekind (Standard_Entity (S_Ascii), E_Package);
      Decl_A := New_List; -- for ASCII declarations
      Set_Visible_Declarations (Pspec, Decl_A);

      --  Create control character definitions in package ASCII. Note that
      --  the character literal entries created here correspond to literal
      --  values that are impossible in the source, but can be represented
      --  internally with no difficulties.

      Ccode := 16#00#;

      for S in S_Ascii_Names loop
         Decl := New_Node (N_Object_Declaration, Staloc);
         Set_Constant_Present (Decl, True);

         declare
            A_Char    : Entity_Id := Standard_Entity (S);
            Expr_Decl : Node_Id;

         begin
            Set_Sloc  (A_Char, Staloc);
            Set_Ekind (A_Char, E_Constant);
            Set_Etype (A_Char, Standard_Character);
            Set_Scope (A_Char, Standard_Entity (S_Ascii));
            Set_Is_Immediately_Visible (A_Char, False);
            Set_Is_Public (A_Char);
            Append_Entity (A_Char, Standard_Entity (S_Ascii));
            Set_Defining_Identifier (Decl, A_Char);

            Set_Object_Definition (Decl, Identifier_For (S_Character));
            Expr_Decl := New_Node (N_Character_Literal, Staloc);
            Set_Expression (Decl, Expr_Decl);

            Set_Is_Static_Expression (Expr_Decl);
            Set_Chars                (Expr_Decl, No_Name);
            Set_Etype                (Expr_Decl, Standard_Character);
            Set_Char_Literal_Value   (Expr_Decl, Ccode);
         end;

         Append (Decl, Decl_A);

         --  Increment character code, dealing with non-contiguities

         Ccode := Ccode + 1;

         if Ccode = 16#20# then
            Ccode := 16#21#;
         elsif Ccode = 16#27# then
            Ccode := 16#3A#;
         elsif Ccode = 16#3C# then
            Ccode := 16#3F#;
         elsif Ccode = 16#41# then
            Ccode := 16#5B#;
         end if;
      end loop;

      --  Create semantic phase entities

      Standard_Void_Type := New_Standard_Entity;
      Set_Ekind (Standard_Void_Type, E_Void);
      Set_Etype (Standard_Void_Type, Standard_Void_Type);
      Set_Esize (Standard_Void_Type, Uint_0);
      Set_Scope (Standard_Void_Type, Standard_Standard);
      Make_Name (Standard_Void_Type, "_void_type");

      --  The type field of packages is set to void

      Set_Etype (Standard_Standard, Standard_Void_Type);
      Set_Etype (Standard_Ascii, Standard_Void_Type);

      --  Standard_A_String is actually used in generated code, so it has a
      --  type name that is reasonable, but does not overlap any Ada name.

      Standard_A_String := New_Standard_Entity;
      Set_Ekind     (Standard_A_String, E_Access_Type);
      Set_Scope     (Standard_A_String, Standard_Standard);
      Set_Etype     (Standard_A_String, Standard_A_String);
      Set_Esize     (Standard_A_String, UI_From_Int (System_Address_Size));

      Set_Directly_Designated_Type (Standard_A_String, Standard_String);
      Make_Name     (Standard_A_String, "access_string");

      --  Note on type names. The type names for the following special types
      --  are constructed so that they will look reasonable should they ever
      --  appear in error messages etc, although in practice the use of the
      --  special insertion character } for types results in special handling
      --  of these type names in any case. The blanks in these names would
      --  trouble in Gigi, but that's OK here, since none of these types
      --  should ever get through to Gigi! Attributes of these types are
      --  filled out to minimize problems with cascaded errors (for example,
      --  Any_Integer is given reasonable and consistent type and size values)

      Any_Type := New_Standard_Entity;
      Set_Ekind (Any_Type, E_Signed_Integer_Type);
      Set_Scope (Any_Type, Standard_Standard);
      Set_Etype (Any_Type, Any_Type);
      Set_Esize (Any_Type, Uint_0);
      Make_Name (Any_Type, "any type");

      Any_Id := New_Standard_Entity;
      Set_Ekind (Any_Id, E_Variable);
      Set_Scope (Any_Id, Standard_Standard);
      Set_Etype (Any_Id, Any_Type);
      Set_Esize (Any_Id, Uint_0);
      Make_Name (Any_Id, "any id");

      Any_Access := New_Standard_Entity;
      Set_Ekind (Any_Access, E_Access_Type);
      Set_Scope (Any_Access, Standard_Standard);
      Set_Etype (Any_Access, Any_Access);
      Set_Esize (Any_Access, UI_From_Int (System_Address_Size));
      Make_Name (Any_Access, "an access type");

      Any_Array := New_Standard_Entity;
      Set_Ekind (Any_Array, E_String_Type);
      Set_Scope (Any_Array, Standard_Standard);
      Set_Etype (Any_Array, Any_Array);
      Set_Component_Type (Any_Array, Any_Character);
      Set_Esize (Any_Array, Uint_0);
      Make_Name (Any_Array, "an array type");

      Any_Boolean := New_Standard_Entity;
      Set_Ekind (Any_Boolean, E_Enumeration_Type);
      Set_Scope (Any_Boolean, Standard_Standard);
      Set_Etype (Any_Boolean, Standard_Boolean);
      Set_Esize (Any_Boolean, UI_From_Int (1));
      Set_Scalar_Range (Any_Boolean, Scalar_Range (Standard_Boolean));
      Make_Name (Any_Boolean, "a boolean type");

      Any_Character := New_Standard_Entity;
      Set_Ekind (Any_Character, E_Enumeration_Type);
      Set_Scope (Any_Character, Standard_Standard);
      Set_Etype (Any_Character, Any_Character);
      Set_Is_Character_Type (Any_Character);
      Set_Esize (Any_Character, UI_From_Int (Standard_Character_Size));
      Set_Scalar_Range (Any_Character, Scalar_Range (Standard_Character));
      Make_Name (Any_Character, "a character type");

      Any_Composite := New_Standard_Entity;
      Set_Ekind (Any_Composite, E_Array_Type);
      Set_Scope (Any_Composite, Standard_Standard);
      Set_Etype (Any_Composite, Any_Composite);
      Set_Component_Type (Any_Composite, Standard_Integer);
      Set_Esize (Any_Composite, Uint_0);
      Make_Name (Any_Composite, "a composite type");

      Any_Discrete := New_Standard_Entity;
      Set_Ekind (Any_Discrete, E_Signed_Integer_Type);
      Set_Scope (Any_Discrete, Standard_Standard);
      Set_Etype (Any_Discrete, Any_Discrete);
      Set_Esize (Any_Discrete, UI_From_Int (Standard_Integer_Size));
      Make_Name (Any_Discrete, "a discrete type");

      Any_Fixed := New_Standard_Entity;
      Set_Ekind (Any_Fixed, E_Ordinary_Fixed_Point_Type);
      Set_Scope (Any_Fixed, Standard_Standard);
      Set_Etype (Any_Fixed, Any_Fixed);
      Make_Name (Any_Fixed, "a fixed-point type");

      Any_Integer := New_Standard_Entity;
      Set_Ekind (Any_Integer, E_Signed_Integer_Type);
      Set_Scope (Any_Integer, Standard_Standard);
      Set_Etype (Any_Integer, Standard_Long_Long_Integer);
      Make_Name (Any_Integer, "an integer type");
      Set_Esize (Any_Integer, Esize (Standard_Long_Long_Integer));

      Any_Numeric := New_Standard_Entity;
      Set_Ekind (Any_Numeric, E_Signed_Integer_Type);
      Set_Scope (Any_Numeric, Standard_Standard);
      Set_Etype (Any_Numeric, Standard_Long_Long_Integer);
      Make_Name (Any_Numeric, "a numeric type");
      Set_Esize (Any_Numeric, Esize (Standard_Long_Long_Integer));

      Any_Real := New_Standard_Entity;
      Set_Ekind (Any_Real, E_Floating_Point_Type);
      Set_Scope (Any_Real, Standard_Standard);
      Set_Etype (Any_Real, Standard_Long_Long_Float);
      Make_Name (Any_Real, "a real type");
      Set_Esize (Any_Real, Esize (Standard_Long_Long_Float));

      Any_Scalar := New_Standard_Entity;
      Set_Ekind (Any_Scalar, E_Signed_Integer_Type);
      Set_Scope (Any_Scalar, Standard_Standard);
      Set_Etype (Any_Scalar, Any_Scalar);
      Set_Esize (Any_Scalar, UI_From_Int (Standard_Integer_Size));
      Make_Name (Any_Scalar, "a scalar type");

      Any_String := New_Standard_Entity;
      Set_Ekind (Any_String, E_String_Type);
      Set_Scope (Any_String, Standard_Standard);
      Set_Etype (Any_String, Any_String);
      Set_Component_Type (Any_String, Any_Character);
      Set_Esize (Any_String, Uint_0);
      Make_Name (Any_String, "a string type");

      Standard_Integer_8 := New_Standard_Entity;
      Decl := New_Node (N_Full_Type_Declaration, Stloc);
      Set_Defining_Identifier (Decl, Standard_Integer_8);
      Make_Name (Standard_Integer_8, "integer_8");
      Set_Scope (Standard_Integer_8, Standard_Standard);
      Build_Signed_Integer_Type (Standard_Integer_8, 8);

      Standard_Integer_16 := New_Standard_Entity;
      Decl := New_Node (N_Full_Type_Declaration, Stloc);
      Set_Defining_Identifier (Decl, Standard_Integer_16);
      Make_Name (Standard_Integer_16, "integer_16");
      Set_Scope (Standard_Integer_16, Standard_Standard);
      Build_Signed_Integer_Type (Standard_Integer_16, 16);

      Standard_Integer_32 := New_Standard_Entity;
      Decl := New_Node (N_Full_Type_Declaration, Stloc);
      Set_Defining_Identifier (Decl, Standard_Integer_32);
      Make_Name (Standard_Integer_32, "integer_32");
      Set_Scope (Standard_Integer_32, Standard_Standard);
      Build_Signed_Integer_Type (Standard_Integer_32, 32);

      Standard_Integer_64 := New_Standard_Entity;
      Decl := New_Node (N_Full_Type_Declaration, Stloc);
      Set_Defining_Identifier (Decl, Standard_Integer_64);
      Make_Name (Standard_Integer_64, "integer_64");
      Set_Scope (Standard_Integer_64, Standard_Standard);
      Build_Signed_Integer_Type (Standard_Integer_64, 64);

      --  Note: universal integer and universal real are constructed as fully
      --  formed signed numeric types, with parameters corresponding to the
      --  longest runtime types (Long_Long_Integer and Long_Long_Float). This
      --  allows Gigi to properly process references to universal types that
      --  are not folded at compile time.

      Universal_Integer := New_Standard_Entity;
      Decl := New_Node (N_Full_Type_Declaration, Stloc);
      Set_Defining_Identifier (Decl, Universal_Integer);
      Make_Name (Universal_Integer, "universal_integer");
      Set_Scope (Universal_Integer, Standard_Standard);
      Build_Signed_Integer_Type
        (Universal_Integer, Standard_Long_Long_Integer_Size);

      Universal_Real := New_Standard_Entity;
      Decl := New_Node (N_Full_Type_Declaration, Stloc);
      Set_Defining_Identifier (Decl, Universal_Real);
      Make_Name (Universal_Real, "universal_real");
      Set_Scope (Universal_Real, Standard_Standard);
      Build_Float_Type
        (Universal_Real,
         Standard_Long_Long_Float_Size,
         Standard_Long_Long_Float_Digits);

      --  Note: universal fixed, unlike universal integer and universal real,
      --  is never used at runtime, so it does not need to have bounds set.

      Universal_Fixed := New_Standard_Entity;
      Decl := New_Node (N_Full_Type_Declaration, Stloc);
      Set_Defining_Identifier (Decl, Universal_Fixed);
      Make_Name (Universal_Fixed, "universal_fixed");
      Set_Ekind (Universal_Fixed, E_Ordinary_Fixed_Point_Type);
      Set_Etype (Universal_Fixed, Universal_Fixed);
      Set_Scope (Universal_Fixed, Standard_Standard);
      Set_Esize
        (Universal_Fixed, UI_From_Int (Standard_Long_Long_Integer_Size));
      Set_Size_Known_At_Compile_Time (Universal_Fixed);

      --  Build standard exception type. Note that the type name here is
      --  actually used in the generated code, so it must be set correctly

      Standard_Exception_Type := New_Standard_Entity;
      Set_Ekind (Standard_Exception_Type, E_Exception_Type);
      Set_Etype (Standard_Exception_Type, Standard_Exception_Type);
      Set_Scope (Standard_Exception_Type, Standard_Standard);
      Set_Esize (Standard_Exception_Type, Uint_0);
      Set_Size_Known_At_Compile_Time
                (Standard_Exception_Type);
      Make_Name (Standard_Exception_Type, "exception");

      --  Create declarations of standard exceptions

      Build_Exception (S_Constraint_Error);
      Build_Exception (S_Program_Error);
      Build_Exception (S_Storage_Error);
      Build_Exception (S_Tasking_Error);

      --  Numeric_Error is a normal exception in Ada 83, but in Ada 95
      --  it is a renaming of Constraint_Error

      if Ada_83 then
         Build_Exception (S_Numeric_Error);

      else
         Decl := New_Node (N_Exception_Renaming_Declaration, Stloc);
         E_Id := Standard_Entity (S_Numeric_Error);

         Set_Ekind     (E_Id, E_Exception);
         Set_Etype     (E_Id, Standard_Exception_Type);
         Set_Is_Public (E_Id);

         Set_Renamed_Object (E_Id, Standard_Entity (S_Constraint_Error));

         Set_Defining_Identifier (Decl, E_Id);
         Append (Decl, Decl_S);

         Ident_Node := New_Node (N_Identifier, Stloc);
         Set_Chars  (Ident_Node, Chars (Standard_Entity (S_Constraint_Error)));
         Set_Entity (Ident_Node, Standard_Entity (S_Constraint_Error));
         Set_Name   (Decl, Ident_Node);
      end if;

      --  Abort_Signal is an entity that does not get made visible

      Abort_Signal := New_Standard_Entity;
      Set_Chars     (Abort_Signal, Name_uAbort_Signal);
      Set_Ekind     (Abort_Signal, E_Exception);
      Set_Etype     (Abort_Signal, Standard_Exception_Type);
      Set_Is_Public (Abort_Signal, True);
      Decl :=
        Make_Exception_Declaration (Stloc,
          Defining_Identifier => Abort_Signal);

      --  Create standard operator declarations

      Create_Operators;

      --  Initialize visibility table with entities in Standard

      for E in Standard_Entity_Type loop
         if Ekind (Standard_Entity (E)) /= E_Operator then
            Set_Name_Entity_Id
              (Chars (Standard_Entity (E)), Standard_Entity (E));
            Set_Homonym (Standard_Entity (E), Empty);
         end if;

         if E not in S_Ascii_Names then
            Set_Scope (Standard_Entity (E), Standard_Standard);
            Set_Is_Immediately_Visible (Standard_Entity (E));
         end if;
      end loop;

      --  The predefined package Standard itself does not have a scope;
      --  it is the only entity in the system not to have one, and this
      --  is what identifies the package to Gigi.

      Set_Scope (Standard_Standard, Empty);

      --  Set global variables indicating last Id values and version

      Last_Standard_Node_Id := Last_Node_Id;
      Last_Standard_List_Id := Last_List_Id;

      --  Initialize Standard_Version string

      Standard_Version (SV_Short_Short_Integer) :=
         Encode_Size (Standard_Short_Short_Integer_Size);
      Standard_Version (SV_Short_Integer)       :=
         Encode_Size (Standard_Short_Integer_Size);
      Standard_Version (SV_Integer)             :=
         Encode_Size (Standard_Integer_Size);
      Standard_Version (SV_Long_Integer)        :=
         Encode_Size (Standard_Long_Integer_Size);
      Standard_Version (SV_Long_Long_Integer)   :=
         Encode_Size (Standard_Long_Long_Integer_Size);
      Standard_Version (SV_Short_Float)         :=
         Encode_Size (Standard_Short_Float_Size);
      Standard_Version (SV_Float)               :=
         Encode_Size (Standard_Float_Size);
      Standard_Version (SV_Long_Float)          :=
         Encode_Size (Standard_Long_Float_Size);
      Standard_Version (SV_Long_Long_Float)     :=
         Encode_Size (Standard_Long_Long_Float_Size);
      Standard_Version (SV_Address)             :=
         Encode_Size (System_Address_Size);

   end Create_Standard;

   ----------------------
   -- Create_Operators --
   ----------------------

   --  Each operator has an abbreviated signature. The formals have the names
   --  LEFT and RIGHT. Their types are not actually used for resolution.

   procedure Create_Operators is
      Op_Node : Entity_Id;

      type Binary_Names is array (S_Binary_Ops) of Name_Id;

      --  Following list has two entries for concatenation, to include
      --  explicitly the operation on wide strings.

      Binary_Ops : constant array (S_Binary_Ops) of Name_Id :=
        (Name_Op_Add,      Name_Op_And,   Name_Op_Concat,   Name_Op_Concat,
         Name_Op_Divide,   Name_Op_Eq,    Name_Op_Expon,    Name_Op_Ge,
         Name_Op_Gt,       Name_Op_Le,    Name_Op_Lt,       Name_Op_Mod,
         Name_Op_Multiply, Name_Op_Ne,    Name_Op_Or,       Name_Op_Rem,
         Name_Op_Subtract, Name_Op_Xor);

      Unary_Ops : constant array (S_Unary_Ops) of Name_Id :=
        (Name_Op_Abs, Name_Op_Subtract, Name_Op_Not, Name_Op_Add);

      --  Corresponding to Abs, Minus, Not, and Plus.

   begin
      for J in S_Binary_Ops loop
         Op_Node := New_Operator (Binary_Ops (J));
         SE (J)  := Op_Node;
         Append_Entity (Make_Formal (Any_Type, "LEFT"),  Op_Node);
         Append_Entity (Make_Formal (Any_Type, "RIGHT"), Op_Node);
      end loop;

      for J in S_Unary_Ops loop
         Op_Node := New_Operator (Unary_Ops (J));
         SE (J)  := Op_Node;
         Append_Entity (Make_Formal (Any_Type, "RIGHT"), Op_Node);
      end loop;

      --  For concatenation, we create a separate operator for each
      --  array type. This simplifies the resolution of the component-
      --  component concatenation operation. In Standard, we set the types
      --  of the formals for string and wide string concatenation.

      Set_Etype (First_Entity (Standard_Op_Concat),  Standard_String);
      Set_Etype (Last_Entity  (Standard_Op_Concat),  Standard_String);
      Set_Etype (Standard_Op_Concat,                 Standard_String);

      Set_Etype (First_Entity (Standard_Op_Concatw), Standard_Wide_String);
      Set_Etype (Last_Entity  (Standard_Op_Concatw), Standard_Wide_String);
      Set_Etype (Standard_Op_Concatw,                Standard_Wide_String);

   end Create_Operators;

   -----------------
   -- Encode_Size --
   -----------------

   function Encode_Size (Size : Pos) return Character is
   begin
      if Size = 8 then
         return '1';
      elsif Size = 16 then
         return '2';
      elsif Size = 32 then
         return '3';
      elsif Size = 64 then
         return '4';
      elsif Size = 128 then
         return '5';
      else
         return '6';
      end if;
   end Encode_Size;

   --------------------
   -- Identifier_For --
   --------------------

   function Identifier_For (S : Standard_Entity_Type) return Node_Id is
      Ident_Node : Node_Id;

   begin
      Ident_Node := New_Node (N_Identifier, Stloc);
      Set_Chars (Ident_Node, Chars (Standard_Entity (S)));
      return Ident_Node;
   end Identifier_For;

   -----------------
   -- Make_Formal --
   -----------------

   function Make_Formal
     (Typ         : Entity_Id;
      Formal_Name : String)
      return        Entity_Id
   is
      Formal : Entity_Id;

   begin
      Formal := New_Standard_Entity;
      Set_Ekind (Formal, E_In_Parameter);
      Set_Scope (Formal, Standard_Standard);
      Set_Etype (Formal, Typ);
      Make_Name (Formal, Formal_Name);
      return Formal;
   end Make_Formal;

   ------------------
   -- Make_Integer --
   ------------------

   function Make_Integer (V : Uint) return Node_Id is
      N : constant Node_Id := Make_Integer_Literal (Stloc, V);

   begin
      Set_Is_Static_Expression (N);
      return N;
   end Make_Integer;

   ---------------
   -- Make_Name --
   ---------------

   procedure Make_Name (Id : Entity_Id; Nam : String) is
   begin
      for J in 1 .. Nam'Length loop
         Name_Buffer (J) := Fold_Lower (Nam (Nam'First + (J - 1)));
      end loop;

      Name_Len := Nam'Length;
      Set_Chars (Id, Name_Find);
   end Make_Name;

   ------------------
   -- New_Operator --
   ------------------

   function New_Operator (Op : Name_Id) return Entity_Id is
      Ident_Node : Entity_Id;

   begin
      Ident_Node := Make_Defining_Identifier (Stloc, Op);

      Set_Is_Pure    (Ident_Node, True);
      Set_Ekind      (Ident_Node, E_Operator);
      Set_Etype      (Ident_Node, Universal_Integer);
      Set_Scope      (Ident_Node, Standard_Standard);
      Set_Homonym    (Ident_Node, Get_Name_Entity_Id (Op));
      Set_Convention (Ident_Node, Convention_Intrinsic);

      Set_Is_Immediately_Visible   (Ident_Node, True);
      Set_Is_Intrinsic_Subprogram  (Ident_Node, True);

      Set_Name_Entity_Id (Op, Ident_Node);
      Append_Entity (Ident_Node, Standard_Standard);
      return Ident_Node;
   end New_Operator;

   -------------------------
   -- New_Standard_Entity --
   -------------------------

   function New_Standard_Entity
     (New_Node_Kind : Node_Kind := N_Defining_Identifier)
      return          Entity_Id
   is
      E : constant Entity_Id := New_Entity (New_Node_Kind, Stloc);

   begin
      Set_Is_Pure (E);
      Set_Is_Frozen (E);
      Set_Is_Public (E);
      return E;
   end New_Standard_Entity;

   ----------------------
   -- Set_Float_Bounds --
   ----------------------

   procedure Set_Float_Bounds
     (Id  : Entity_Id;
      Typ : Entity_Id)
   is
      L   : Node_Id;     --  Low bound of literal value
      H   : Node_Id;     --  High bound of literal value
      R   : Node_Id;     --  Range specification

   begin
      if Typ = Standard_Short_Float then
         L := Real_Convert
                (Short_Float_Attr_First'Universal_Literal_String);
         H := Real_Convert
                (Short_Float_Attr_Last'Universal_Literal_String);

      elsif Typ = Standard_Float then
         L := Real_Convert
                (Float_Attr_First'Universal_Literal_String);
         H := Real_Convert
                (Float_Attr_Last'Universal_Literal_String);

      elsif Typ = Standard_Long_Float then
         L := Real_Convert
                (Long_Float_Attr_First'Universal_Literal_String);
         H := Real_Convert
                (Long_Float_Attr_Last'Universal_Literal_String);

      elsif Typ = Standard_Long_Long_Float
        or else Typ = Universal_Real
      then
         L := Real_Convert
                (Long_Long_Float_Attr_First'Universal_Literal_String);
         H := Real_Convert
                (Long_Long_Float_Attr_Last'Universal_Literal_String);

      else
         pragma Assert (False); null;
      end if;

      Set_Etype                (L, Typ);
      Set_Is_Static_Expression (L);

      Set_Etype                (H, Typ);
      Set_Is_Static_Expression (H);

      R := New_Node (N_Range, Stloc);
      Set_Low_Bound  (R, L);
      Set_High_Bound (R, H);
      Set_Scalar_Range (Id, R);
   end Set_Float_Bounds;

   ------------------------
   -- Set_Integer_Bounds --
   ------------------------

   procedure Set_Integer_Bounds
     (Id  : Entity_Id;
      Typ : Entity_Id;
      Lb  : Uint;
      Hb  : Uint)
   is
      L : Node_Id;     -- Low bound of literal value
      H : Node_Id;     -- High bound of literal value
      R : Node_Id;     -- Range specification

   begin
      L := Make_Integer (Lb);
      H := Make_Integer (Hb);

      Set_Etype (L, Typ);
      Set_Etype (H, Typ);

      R := New_Node (N_Range, Stloc);
      Set_Low_Bound  (R, L);
      Set_High_Bound (R, H);
      Set_Scalar_Range (Id, R);
   end Set_Integer_Bounds;

end CStand;
