/*
 * ldconfig - update shared library symlinks
 *
 * usage: ldconfig [-DvnNX] dir ...
 *        ldconfig -l [-Dv] lib ...
 *        ldconfig -p
 *        -D: debug mode, don't update links
 *        -v: verbose mode, print things as we go
 *        -n: don't process standard directories
 *        -N: don't update the library cache
 *        -X: don't update the library links
 *        -l: library mode, manually link libraries
 *        -p: print the current library cache
 *        dir ...: directories to process
 *        lib ...: libraries to link
 *
 * Copyright 1994 David Engel and Mitch D'Souza
 *
 * This program may be used for any purpose as long as this
 * copyright notice is kept.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <asm/string.h>
#include <ctype.h>
#include <getopt.h>
#include <dirent.h>
#include <unistd.h>
#include <a.out.h>
#include <linux/elf.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <errno.h>

#include "../config.h"

char *___strtok = NULL;

/* For SunOS */
#ifndef PATH_MAX
#include <limits.h>
#define PATH_MAX _POSIX_PATH_MAX
#endif

/* For SunOS */
#ifndef N_MAGIC
#define N_MAGIC(exec) ((exec).a_magic & 0xffff)
#endif

#define EXIT_OK    0
#define EXIT_FATAL 128

char *prog = NULL;
int debug = 0;			/* debug mode */
int verbose = 0;		/* verbose mode */
int libmode = 0;		/* library mode */
int nocache = 0;		/* don't build cache */
int nolinks = 0;		/* don't update links */

#define LEN_SORT	0x1
#define NAME_SORT	0x2
#define CTIME_SORT	0x3

char sort_type = NAME_SORT;	/* default to sort by name */
char *cachefile = LDSO_CACHE;	/* default cache file */
signed char invert = 1;		/* Invert any of the sort options */

void cache_print(void);
void cache_dodir(char *, short);
void cache_dolib(char *, short, char);
void cache_write(void);

void warn(char *fmt, ...)
{
    va_list ap;

    fprintf(stderr, "%s: warning: ", prog);

    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    va_end(ap);

    fprintf(stderr, "\n");

    return;
}

void error(char *fmt, ...)
{
    va_list ap;

    fprintf(stderr, "%s: ", prog);

    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    va_end(ap);

    fprintf(stderr, "\n");

    exit(EXIT_FATAL);
}

void *xmalloc(size_t size)
{
    void *ptr;
    if ((ptr = malloc(size)) == NULL)
	error("out of memory");
    return ptr;
}

char *xstrdup(char *str)
{
    char *ptr;
    char *strdup(const char *);
    if ((ptr = strdup(str)) == NULL)
	error("out of memory");
    return ptr;
}

/* if shared library, return length of base name, else return -1 */
int is_shlib(char *dir, char *name, char *type)
{
    int good = 0;
    char *cp = name, *cp2;
    FILE *file;
    struct exec exec;
    struct elfhdr *elf_hdr;
    struct stat statbuf;
    char buff[1024];

    /* see if name is of the form libZ.so.M[.N[.P]] */
    if (strncmp(name, "lib", 3) == 0 &&
	(cp = strstr(name, ".so.")) && cp[4])
    {
	/* find the start of the M.P part, if any */
	if ((cp2 = strchr(cp + 4, '.')))
	    cp = cp2;
	else
	    cp = cp + strlen(cp);

	/* construct the full path name */
	sprintf(buff, "%s%s%s", dir, (*dir && strcmp(dir, "/")) ?
		"/" : "", name);

	/* first, make sure it's a regular file */
	if (lstat(buff, &statbuf))
	    warn("can't lstat %s (%s), skipping", buff, strerror(errno));
	else if (S_ISLNK(statbuf.st_mode))
	{
	    if (stat(buff, &statbuf))
	    {
		warn("%s is a stale symlink, removing", buff);
		if (remove(buff))
		    warn("can't remove %s (%s), skipping", buff,
			 strerror(errno));
	    }
	}
	else if (!S_ISREG(statbuf.st_mode))
	    warn("%s is not a regular file, skipping", buff, strerror(errno));
	else
	{
	    /* then try opening it */
	    if (!(file = fopen(buff, "rb")))
		warn("can't open %s (%s), skipping", buff, strerror(errno));
	    else
	    {
		/* now make sure it's [ZQ]MAGIC with a non-zero address
                   or ELF */
		if (fread(&exec, sizeof exec, 1, file) < 1)
		    warn("can't read header from %s, skipping", buff);
		else if (N_MAGIC(exec) != ZMAGIC && N_MAGIC(exec) != QMAGIC)
		{
		    elf_hdr = (struct elfhdr *) &exec;
		    if (elf_hdr->e_ident[0] != 0x7f ||
			strncmp(&elf_hdr->e_ident[1], "ELF",3) != 0)
			warn("%s is not a [ZQ]MAGIC or ELF file, skipping", buff);
		    else
		    {
			good = 1;
			*type = LIB_ELF;
		    }
		}
		else if (!exec.a_entry)
		    warn("%s has a zero load address, skipping", buff);
		else
		{
		    good = 1;	/* looks good */
		    *type = LIB_DLL;
		}
		fclose(file);
	    }
	}
    }

    return good ? cp - name : -1;
}

/* update the symlink to new library */
void link_shlib(char *dir, char *file, int len)
{
    int change = -1;
    char libname[1024];
    char linkname[1024];
    struct stat libstat;
    struct stat linkstat;

    /* construct the full path names */
    sprintf(libname, "%s%s%s", dir, (*dir && strcmp(dir, "/")) ?
	    "/" : "", file);
    sprintf(linkname, "%s%s%.*s", dir, (*dir && strcmp(dir, "/"))
	    ? "/" : "", len, file);

    /* see if a link already exists */
    if (!stat(linkname, &linkstat))
    {
	/* now see if it's the one we want */
	if (stat(libname, &libstat))
	    warn("can't stat %s (%s)", libname, strerror(errno));
	else if (libstat.st_dev == linkstat.st_dev &&
	    libstat.st_ino == linkstat.st_ino)
	    change = 0;
    }

    /* then update the link, if required */
    if (change && !nolinks)
    {
	if (!lstat(linkname, &linkstat) && remove(linkname))
	    warn("can't unlink %s (%s)", linkname, strerror(errno));
	else if (symlink(file, linkname))
	    warn("can't link %s to %s (%s)", linkname, file, strerror(errno));
	else
	    change = 1;
    }

    /* some people like to know what we're doing */
    if (verbose)
	printf("\t%.*s => %s%s\n", len, file, file,
	       change < 0 ? " (SKIPPED)" :
	       (change > 0 ? " (changed)" : ""));

    return;
}

/* figure out which library is greater */
int libcmp(char *p1, char *p2)
{
    while (*p1)
    {
	if (isdigit(*p1) && isdigit(*p2))
	{
	    /* must compare this numerically */
	    int v1, v2;
	    v1 = strtoul(p1, &p1, 10);
	    v2 = strtoul(p2, &p2, 10);
	    if (v1 != v2)
		return v1 - v2;
	}
	else if (*p1 != *p2)
	    return *p1 - *p2;
	else
	    p1++, p2++;
    }

    return *p1 - *p2;
}

struct lib
{
    char *name;			/* name of a library */
    int len;			/* length of base name */
    struct lib *next;		/* next library in list */
};

/* update all shared library links in a directory */
void scan_dir(char *name)
{
    DIR *dir;
    struct dirent *ent;
    int len;
    struct lib *lp, *libs = NULL;
    char libtype;

    /* let 'em know what's going on */
    if (verbose)
	printf("%s:\n", name);

    if (!nocache)
	cache_dodir(name, strlen(name));

    /* if we can't open it, we can't do anything */
    if ((dir = opendir(name)) == NULL)
    {
	warn("can't open %s (%s), skipping", name, strerror(errno));
	return;
    }

    /* yes, we have to look at every single file */
    while ((ent = readdir(dir)) != NULL)
    {
	/* if it's not a shared library, don't bother */
	if ((len = is_shlib(name, ent->d_name, &libtype)) < 0)
	    continue;

	if (!nocache)
	    cache_dolib(ent->d_name, len, libtype);

	/* have we already seen one with the same base name? */
	for (lp = libs; lp; lp = lp->next)
	{
	    if (len == lp->len && 
		strncmp(ent->d_name, lp->name, len) == 0)
	    {
		/* we have, which one do we want to use? */
		if (libcmp(&ent->d_name[len+1], &lp->name[len+1]) > 0)
		{
		    /* let's use the new one */
		    free(lp->name);
		    lp->name = xstrdup(ent->d_name);
		} 
		break;
	    }
	}

	/* congratulations, you're the first one we've seen */
	if (!lp)
	{
	    lp = xmalloc(sizeof *lp);
	    lp->name = xstrdup(ent->d_name);
	    lp->len = len;
	    lp->next = libs;
	    libs = lp;
	}
    }

    /* don't need this any more */
    closedir(dir);

    /* now we have all the latest libs, update the links */
    for (lp = libs; lp; lp = lp->next)
	link_shlib(name, lp->name, lp->len);

    /* always try to clean up after ourselves */
    while (libs)
    {
	lp = libs->next;
	free(libs->name);
	free(libs);
	libs = lp;
    }

    return;
}

/* return the list of system-specific directories */
char *get_extpath(void)
{
    char *cp = NULL;
    FILE *file;
    struct stat stat;

    if ((file = fopen(LDSO_CONF, "r")) != NULL)
    {
	fstat(fileno(file), &stat);
	cp = xmalloc(stat.st_size + 1);
	fread(cp, 1, stat.st_size, file);
	fclose(file);
	cp[stat.st_size] = '\0';
    }

    return cp;
}

int main(int argc, char **argv)
{
    int i, c, len;
    int nodefault = 0;
    int printcache = 0;
    char *cp, *dir;
    char *extpath, libtype;
    static char defpath[] = DEFAULT_PATH;

    prog = argv[0];
    opterr = 0;

    while ((c = getopt(argc, argv, "DvnNXlpf:S:")) != EOF)
	switch (c)
	{
	case 'D':
	    debug = 1;		/* debug mode */
	    nocache = 1;
	    nolinks = 1;
	    verbose = 1;
	    break;
	case 'v':
	    verbose = 1;	/* verbose mode */
	    break;
	case 'n':
	    nodefault = 1;	/* no default dirs */
	    nocache = 1;
	    break;
	case 'N':
	    nocache = 1;	/* don't build cache */
	    break;
	case 'X':
	    nolinks = 1;	/* don't update links */
	    break;
	case 'l':
	    libmode = 1;	/* library mode */
	    break;
	case 'p':
	    printcache = 1;	/* print cache */
	    break;
	default:
	    fprintf(stderr, "usage: %s [-DvnNX] dir ...\n", prog);
	    fprintf(stderr, "       %s -l [-Dv] lib ...\n", prog);
	    fprintf(stderr, "       %s -p\n", prog);
	    exit(EXIT_FATAL);
	    break;

	/* THE REST OF THESE ARE UNDOCUMENTED AND MAY BE REMOVED
	   IN FUTURE VERSIONS. */
	case 'f':
	    cachefile = optarg;	/* alternate cache file */
	    break;
	case 'S':
	    for (cp = optarg; *cp; cp++)
	    {
		switch (*cp)
		{
		case 'C':
		    sort_type = CTIME_SORT; /* sort by lib creation time */
		    break;
		case 'L':
		    sort_type = LEN_SORT; /* sort by lib length */
		    break;
		case 'N':
		    sort_type = NAME_SORT; /* sort by lib name */
		    break;
		case 'i':
		    invert = -1; /* invert sort order */
		    break;
		default:
		    error("invalid sort option -S%s", optarg);
		    break;
		}
	    }
	}

    /* allow me to introduce myself, hi, my name is ... */
    if (verbose)
	printf("%s: version %s\n", argv[0], VERSION);

    if (printcache)
    {
	/* print the cache -- don't you trust me? */
	cache_print();
	exit(EXIT_OK);
    }
    else if (libmode)
    {
	/* so you want to do things manually, eh? */

	/* ok, if you're so smart, which libraries do we link? */
	for (i = optind; i < argc; i++)
	{
	    /* split into directory and file parts */
	    if (!(cp = strrchr(argv[i], '/')))
	    {
		dir = "";	/* no dir, only a filename */
		cp = argv[i];
	    }
	    else
	    {
		if (cp == argv[i])
		    dir = "/";	/* file in root directory */
		else
		    dir = argv[i];
		*cp++ = '\0';	/* neither of the above */
	    }

	    /* we'd better do a little bit of checking */
	    if ((len = is_shlib(dir, cp, &libtype)) < 0)
		error("%s%s%s is not a shared library", dir,
		      (*dir && strcmp(dir, "/")) ? "/" : "", cp);

	    /* so far, so good, maybe he knows what he's doing */
	    link_shlib(dir, cp, len);
	}
    }
    else
    {
	/* the lazy bum want's us to do all the work for him */

	/* don't cache dirs on the command line */
	int nocache_save = nocache;
	nocache = 1;

	/* OK, which directories should we do? */
	for (i = optind; i < argc; i++)
	    scan_dir(argv[i]);

	/* restore the desired caching state */
	nocache = nocache_save;

	/* look ma, no defaults */
	if (!nodefault)
	{
	    /* I guess the defaults aren't good enough */
	    if ((extpath = get_extpath()))
	    {
		for (cp = strtok(extpath, DIR_SEP); cp;
		     cp = strtok(NULL, DIR_SEP))
		    scan_dir(cp);
		free(extpath);
	    }

	    /* everybody needs these, don't they? */
	    for (cp = strtok(defpath, DIR_SEP); cp;
		 cp = strtok(NULL, DIR_SEP))
		scan_dir(cp);
	}

	if (!nocache)
	    cache_write();
    }

    exit(EXIT_OK);
}

typedef struct dirlist
{
    short dirnamelen;
    long offset;
    char *entry;
    struct dirlist *q_forw;
    struct dirlist *q_back;
} dirlist_t;

typedef struct liblist
{
    char flags;
    short libnamelen;
    short dirnamelen;
    long liboffset;
    long diroffset;
    char *entry;
    struct stat st;
    struct liblist *q_forw;
    struct liblist *q_back;
} liblist_t;

#define LINK_MASK 0xff

static header_t magic;
static dirlist_t *cur_dir = NULL;
static liblist_t *cur_lib = NULL;

void cache_dodir(char *entry, short len)
{
    magic.ndirs++;
    if (cur_dir == NULL)
    {
	cur_dir = xmalloc(sizeof (dirlist_t));
	cur_dir->dirnamelen = len;
	cur_dir->q_forw = NULL;
	cur_dir->q_back = NULL;
	cur_dir->offset = 0;
	cur_dir->entry = xstrdup(entry);
    }
    else
    {
	cur_dir->q_forw = xmalloc(sizeof (dirlist_t));
	cur_dir->q_forw->dirnamelen = len;
	cur_dir->q_forw->q_forw = NULL;
	cur_dir->q_forw->q_back = cur_dir;
	cur_dir->q_forw->offset = cur_dir->offset
	    + cur_dir->dirnamelen + 1;
	cur_dir->q_forw->entry = xstrdup(entry);
	cur_dir = cur_dir->q_forw;
    }
}

void cache_dolib(char *entry, short len, char libtype)
{
    char fullpath[PATH_MAX];

    magic.nlibs++;
    if (cur_lib == NULL)
    {
	sprintf(fullpath, "%s/%s", cur_dir->entry, entry);
	cur_lib = (liblist_t *) xmalloc(sizeof (liblist_t));
	cur_lib->flags = libtype;
	cur_lib->libnamelen = strlen(entry)+(len<<8);
	cur_lib->dirnamelen = cur_dir->dirnamelen;
	cur_lib->diroffset = cur_dir->offset;
	cur_lib->liboffset = 0;
	stat(fullpath,&cur_lib->st);
	cur_lib->q_forw = NULL;
	cur_lib->q_back = NULL;
	cur_lib->entry = xstrdup(entry);
    }
    else
    {
	sprintf(fullpath, "%s/%s", cur_dir->entry, entry);
	cur_lib->q_forw = (liblist_t *) xmalloc(sizeof (liblist_t));
	cur_lib->q_forw->flags = libtype;
	cur_lib->q_forw->libnamelen = strlen(entry)+(len<<8);
	cur_lib->q_forw->dirnamelen = cur_dir->dirnamelen;
	cur_lib->q_forw->diroffset = cur_dir->offset;
	cur_lib->q_forw->liboffset = cur_lib->liboffset +
	    (cur_lib->libnamelen & LINK_MASK) + 1;
	stat(fullpath,&cur_lib->q_forw->st);
	cur_lib->q_forw->q_forw = NULL;
	cur_lib->q_forw->q_back = cur_lib;
	cur_lib->q_forw->entry = xstrdup(entry);
	cur_lib = cur_lib->q_forw;
    }
}

void cache_write(void)
{
    int i, cachefd;
    char tempfile[1024];
    void sort_cache(liblist_t *, int);

    if (!magic.nlibs)
	return;

    sprintf(tempfile, "%s~", cachefile);

    if ((cachefd = open(tempfile, O_RDWR|O_CREAT|O_TRUNC, 0666)) < 0)
	error("can't open %s (%s)", tempfile, strerror(errno));

    lseek(cachefd, sizeof (header_t), SEEK_SET);

    sort_cache(cur_lib, magic.nlibs);

    while (cur_lib->q_back)
	cur_lib = cur_lib->q_back;

    for (i = 0; i < magic.nlibs; i++)
    {
	write(cachefd, cur_lib, sizeof (libentry_t));
	if (cur_lib->q_forw == NULL)
	    break;
	else
	    cur_lib = cur_lib->q_forw;
    }

    while (cur_lib->q_back)
	cur_lib = cur_lib->q_back;

    for (i = 0; i < magic.nlibs; i++)
    {
	write(cachefd, cur_lib->entry, (cur_lib->libnamelen & LINK_MASK) + 1);

	if (cur_lib->q_forw == NULL)
	    break;
	else
	    cur_lib = cur_lib->q_forw;

	free (cur_lib->q_back->entry);
	free (cur_lib->q_back);
    }

    magic.diroffset = lseek(cachefd, 0L, SEEK_CUR);

    while (cur_dir->q_back)
	cur_dir = cur_dir->q_back;

    while (cur_dir)
    {
	write(cachefd, cur_dir->entry, cur_dir->dirnamelen + 1);

	if (cur_dir->q_forw == NULL)
	    break;
	else
	    cur_dir = cur_dir->q_forw;

	free (cur_dir->q_back->entry);
	free (cur_dir->q_back);
    }

    lseek(cachefd, 0L, SEEK_SET);
    memcpy(&magic, LDSO_CACHE_MAGIC LDSO_CACHE_VER,
	   sizeof magic.magic + sizeof magic.version);
    magic.liboffset = sizeof (header_t)
	+ magic.nlibs * sizeof (libentry_t);
    write(cachefd, &magic, sizeof (header_t));

    close(cachefd);

    if (rename(tempfile, cachefile))
	error("can't rename %s to %s (%s)", tempfile, cachefile, strerror(errno));
}

void cache_print(void)
{
    caddr_t c;
    struct stat st;
    int fd = 0;
    char nlibs;
    char *liboffset, *diroffset;
    libentry_t libent;

    if (stat(cachefile, &st) || (fd = open(cachefile, O_RDONLY))<0)
	error("can't read %s (%s)", cachefile, strerror(errno));
    if ((c = mmap(0,st.st_size, PROT_READ, MAP_SHARED ,fd, 0)) == (caddr_t)-1)
	error("can't map %s (%s)", cachefile, strerror(errno));
    close(fd);

    if (memcmp(((header_t *)c)->magic, LDSO_CACHE_MAGIC, LDSO_CACHE_MAGIC_LEN))
	error("%s cache corrupt", cachefile);

    if (memcmp(((header_t *)c)->version, LDSO_CACHE_VER, LDSO_CACHE_VER_LEN))
	error("wrong cache version - expected %s", LDSO_CACHE_VER);

    nlibs = ((header_t *)c)->nlibs;
    liboffset = c+((header_t *)c)->liboffset;
    diroffset = c+((header_t *)c)->diroffset;

    fprintf(stdout,"%d libs found in cache `%s' (version %s)\n", nlibs, cachefile, LDSO_CACHE_VER);

    for (fd = 0; fd<nlibs; fd++)
    {
	memcpy(&libent, c+sizeof(header_t)+fd * sizeof(libentry_t),
	       sizeof(libentry_t));
	fprintf(stdout, "\t%2d - ",fd+1);
	fflush(stdout);
	write(1,diroffset+libent.diroffset,libent.dirnamelen);
	write(1,"/",1);
	write(1,liboffset+libent.liboffset,libent.libnamelen & LINK_MASK);
	write(1, libent.flags == LIB_DLL ? " (DLL)":" (ELF)",6);
	write(1,"\n",1);
    }
    munmap (c,st.st_size);
}

static int libnamecomp(const void *x, const void *y)
{
#if 1
    return invert * libcmp((*(liblist_t **)y)->entry,
			   (*(liblist_t **)x)->entry);
#else
    return invert * strncmp((*(liblist_t **)y)->entry,
			    (*(liblist_t **)x)->entry,
			    (*(liblist_t **)y)->libnamelen & LINK_MASK);
#endif
}

static int liblencomp(const void *x, const void *y)
{
    return invert * (((*(liblist_t **)y)->libnamelen & LINK_MASK) -
		     ((*(liblist_t **)x)->libnamelen & LINK_MASK));
}

static int libctimecomp(const void *x, const void *y)
{
    return invert * (((*(liblist_t **)y)->st.st_ctime) -
		     ((*(liblist_t **)x)->st.st_ctime));
}

void sort_cache(liblist_t *cur_lib, int nlibs)
{
    int i;
    char **array;

    while (cur_lib->q_back)
	cur_lib = cur_lib->q_back;

    array = (char **)malloc(nlibs * sizeof(char *));

    for (i = 0; i < nlibs; i++)
    {
	array[i] = (char *)cur_lib;
	if (!cur_lib->q_forw) break;
	cur_lib = cur_lib->q_forw;
    }

    switch (sort_type)
    {
    case LEN_SORT:
	qsort(array, nlibs, sizeof (char *), liblencomp);
	break;
    case NAME_SORT:
	qsort(array, nlibs, sizeof (char *), libnamecomp);
	break;
    case CTIME_SORT:
	qsort(array, nlibs, sizeof (char *), libctimecomp);
	break;
    }

    while (cur_lib->q_back)
	cur_lib = cur_lib->q_back;
    
    /* Relink the sorted linked list and calculate the new offsets */
    for (i = 0; i < nlibs; i++)
	if (i == 0)
	{
	    ((liblist_t *)array[i])->q_back = NULL;
	    ((liblist_t *)array[i])->q_forw = NULL;
	    ((liblist_t *)array[i])->liboffset = 0;
	}
	else
	{
	    ((liblist_t *)array[i-1])->q_forw = ((liblist_t *)array[i]);
	    ((liblist_t *)array[i])->q_back = ((liblist_t *)array[i-1]);
	    ((liblist_t *)array[i])->q_forw = NULL;
	    ((liblist_t *)array[i])->liboffset = 
		(long)((liblist_t *)array[i])->q_back->liboffset +
		    ((long)((liblist_t *)array[i])->q_back->libnamelen & LINK_MASK) + 1;
	}

    cur_lib = (liblist_t *)array;

    free(array);
}
