/*
 * general purpose mouse support for Linux
 *
 * Copyright 1993        ajh@gec-mrc.co.uk (Andrew Haylett)
 * Copyright 1994,1995   rubini@ipvvis.unipv.it (Alessandro Rubini)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 ********/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>        /* strerror(); ?!?  */
#include <errno.h>
#include <unistd.h>        /* select(); */
#include <signal.h>        /* SIGPIPE */
#include <sys/param.h>
#include <sys/fcntl.h>     /* O_RDONLY */
#include <sys/wait.h>      /* wait()   */
#include <sys/stat.h>      /* mkdir()  */
#include <sys/time.h>      /* timeval */
#include <sys/types.h>     /* socket() */
#include <sys/socket.h>    /* socket() */
#include <sys/un.h>        /* struct sockaddr_un */

#include <linux/vt.h>      /* VT_GETSTATE */
#include <sys/kd.h>        /* KDGETMODE */
#include <termios.h>       /* winsize */

#include "gpmInt.h"

#ifndef max
#define max(a,b) ((a)>(b) ? (a) : (b))
#endif

extern int	errno;

static void gpm_killed(int);

/*-------------------------------------------------------------------*/
char *opt_type=DEF_TYPE;
char *opt_lut=DEF_LUT;
char *opt_dev=DEF_DEV;
char *opt_sequence=DEF_SEQUENCE;
int opt_baud=DEF_BAUD;
int opt_sample=DEF_SAMPLE;
int opt_delta=DEF_DELTA;
int opt_accel=DEF_ACCEL;
int opt_scale=DEF_SCALE;
int opt_time=DEF_TIME;
int opt_cluster=DEF_CLUSTER;
int opt_three=DEF_THREE;
int opt_dirty=DEF_DIRTY;
int opt_test=DEF_TEST;
int opt_ptrdrag=DEF_PTRDRAG;
int opt_kill=0;
int opt_toggle=DEF_TOGGLE;

static int opt_resize=0; /* not really an option */

char *prgname;
Gpm_Type *m_type;
struct winsize win;
int maxx, maxy;

int eventFlag=0;
Gpm_Cinfo *cinfo[MAX_VC+1];
fd_set selSet, readySet, connSet;

/* these are needed to provide a fictious motion to the client */
int mouseX, mouseY;

/*===================================================================*/
/*
 *      first, all the stuff that used to be in gpn.c 
 */
/*-------------------------------------------------------------------*/
static inline int open_console(const int mode)
{
int fd;

  if ((fd=open("/dev/console", mode)) < 0)
    oops("/dev/console");
  return fd;
}

/*-------------------------------------------------------------------*/
static inline int check_mode(int *fdptr)
{
int fd, ch = 0;
long kd_mode;

  while(1)
    {
    fd = open_console(O_RDONLY);
    if (ioctl(fd, KDGETMODE, &kd_mode)<0)
      oops("ioctl(KDGETMODE)");
    close(fd);
    if (kd_mode == KD_TEXT) break;
    /* should I close the mouse descriptor, now? */
    if (!ch) close(*fdptr);
    sleep(2); ch++;
    }

  if (ch) {
    if ((*fdptr=open(opt_dev,O_RDWR))<0)
      oops(opt_dev);
    if (m_type->init)
      m_type=(m_type->init)(*fdptr, m_type->flags, m_type);

  if (opt_toggle)
    {
    unsigned int modem_lines;

    ioctl(fd, TIOCMGET, &modem_lines);
    modem_lines &= ~opt_toggle;
    ioctl(fd, TIOCMSET, &modem_lines);
    }
  }

  return (ch>0);
}

/*-------------------------------------------------------------------*/
static inline void selection_copy(int x1, int y1, int x2, int y2, int mode)
{
/*
 * The approach in "selection" causes a bus error when run under SunOS 4.1
 * due to alignment problems...
 */
unsigned char buf[6*sizeof(short)];
unsigned short *arg = (unsigned short *)buf + 1;
int fd;

  buf[sizeof(short)-1] = 2;  /* set selection */

  arg[0]=(unsigned short)x1;
  arg[1]=(unsigned short)y1;
  arg[2]=(unsigned short)x2;
  arg[3]=(unsigned short)y2;
  arg[4]=(unsigned short)mode;

  if ((fd=open_console(O_WRONLY))<0)
    oops("open_console");
  DEBUGG((stderr,"ctl %i, mode %i\n",(int)*buf,arg[4]));
  if (ioctl(fd, TIOCLINUX, buf+sizeof(short)-1) < 0)
    oops("ioctl(TIOCLINUX)");
  close(fd);
}

/*-------------------------------------------------------------------*/
static inline void selection_paste(void)
{
char c=3;
int fd;

  fd=open_console(O_WRONLY);
  if (ioctl(fd, TIOCLINUX, &c) < 0)
     oops("ioctl(TIOCLINUX)");
  close(fd);
}

/*-------------------------------------------------------------------*/
static  inline int do_selection(Gpm_Event *event)  /* returns 0, always */
{
static int x1=1, y1=1, x2, y2;
#define UNPOINTER() 0

  x2=event->x; y2=event->y;
  switch(GPM_BARE_EVENTS(event->type))
    {
    case GPM_MOVE:
      if (x2<1) x2++; else if (x2>maxx) x2--;
      if (y2<1) y2++; else if (y2>maxx) y2--;
      selection_copy(x2,y2,x2,y2,3); /* just highlight pointer */
      return 0;

    case GPM_DRAG:
      if (event->buttons==GPM_B_LEFT)
	{
	if (event->margin) /* fix margins */
	  switch(event->margin)
	    {
	    case GPM_TOP: x2=1; y2++; break;
	    case GPM_BOT: x2=maxx; y2--; break;
	    case GPM_RGT: x2--; break;
	    case GPM_LFT: y2<=y1 ? x2++ : (x2=maxx, y2--); break;
	    }
        selection_copy(x1,y1,x2,y2,event->clicks);
	if (event->clicks>=opt_ptrdrag && !event->margin) /* pointer */
	  selection_copy(x2,y2,x2,y2,3);
	}
      return 0;

    case GPM_DOWN:
      switch (event->buttons)
	{
	case GPM_B_LEFT:
	  x1=x2; y1=y2;
	  selection_copy(x1,y1,x2,y2,event->clicks);  /*  start selection */
	  return 0;

	case GPM_B_MIDDLE:
	  if (!opt_three) opt_three++;
	  selection_paste();
	  return 0;

	case GPM_B_RIGHT:
	  if (opt_three==1)
	    selection_copy(x1,y1,x2,y2,event->clicks);
	  else
	    selection_paste();
	  return 0;
	}
    }
  return 0;
}

/*-------------------------------------------------------------------*/
/* returns 0 if the event has not been processed, and 1 if it has */
static inline int do_client(Gpm_Cinfo *cinfo, Gpm_Event *event)
{
Gpm_Connect info=cinfo->data;
int fd=cinfo->fd;
/* value to return if event is not used */
int res = !(info.defaultMask & event->type); 
MAGIC_P((static int magic=GPM_MAGIC));

  /* control keys: pass it over */
  if ((info.minMod & event->modifiers) < info.minMod) return 0;
  if ((info.maxMod & event->modifiers) < event->modifiers) return 0;

  /* not managed, use default mask */
  if (!(info.eventMask & event->type))
    return res;
 
  /* WARNING */ /* This can generate a SIGPIPE... I'd better catch it */
  MAGIC_P((write(fd,&magic, sizeof(int))));
  write(fd,event, sizeof(Gpm_Event));

  return info.defaultMask & GPM_HARD ? res : 1; /* HARD forces pass-on */
}


/*-------------------------------------------------------------------*/
/*-------------------------------------------------------------------*/
/*-------------------------------------------------------------------*/
static inline char *getMouseData(int fd, Gpm_Type *m_type)
{
static unsigned char data[32]; /* quite a big margin :) */
char *edata=data+m_type->packetlen;
int howmany=m_type->howmany;
int i,j;
/*....................................... read and identify one byte */

    if (read(fd, data, howmany)!=howmany)
      {
      if (opt_test) exit(0);
      LOG(("Error in read()ing first"));
      return NULL;
      }
    if ((data[0]&(m_type->proto)[0]) != (m_type->proto)[1])
      {
      if (m_type->getextra)
	{
	data[1]=GPM_EXTRA_MAGIC_1; data[2]=GPM_EXTRA_MAGIC_2;
	return data;
	}
      LOG(("Error in protocol"));
      return NULL;
      }

/*....................................... read the rest */

    /*
     * well, this seems to work almost right with ps2 mice. However, I've never
     * tried ps2 with the original selection package, which called usleep()
     */
    
    if ( (i=m_type->packetlen-howmany) ) /* still to get */
      do
	{
	j=read(fd,edata-i,i); /* edata is pointer just after data */
	i-=j;
	}
      while (i&&j);

    if (i)
      {
      LOG(("Error in read()ing rest"));
      return NULL;
      }
    
    if ((data[1]&(m_type->proto)[2]) != (m_type->proto)[3])
      {
      LOG(("Skipping a data packet (?)"));
      return NULL;
      }
  return data;
}

/*-------------------------------------------------------------------*/
static inline int processMouse(int fd, Gpm_Event *event, Gpm_Type *m_type)
{
char *data;
static int fine_dx, fine_dy;
static int i, margin;
static Gpm_Event nEvent;
static struct vt_stat stat;
static int clicks=0;    /* 0,1 or 2 */
static struct timeval tv1={0,0}, tv2; /* tv1=0: force first click as single */
static struct timeval timeout={0,0};
fd_set fdSet;
static int newB=0, oldB=0, oldT=0; /* old buttons and Type to chain events */

#define GET_TIME(tv) (gettimeofday(&tv, (struct timezone *)NULL))
#define DIF_TIME(t1,t2) ((t2.tv_sec -t1.tv_sec) *1000+ \
                         (t2.tv_usec-t1.tv_usec)/1000)


  oldT=event->type;

  if (eventFlag)
    {
    eventFlag=0;
    event->dx=nEvent.dx;
    event->dy=nEvent.dy;
    event->buttons=nEvent.buttons;
    }
  else
    {
    event->dx=event->dy=0;

    FD_ZERO(&fdSet); FD_SET(fd,&fdSet); i=0; do /* cluster loop */
      {
      if (   ((data=getMouseData(fd,m_type))==NULL)
	  || ((*(m_type->fun))(&nEvent,data)==-1) )
	{
	if (!i) return 0;
	else break;
	}

      nEvent.buttons = opt_sequence[nEvent.buttons]&7; /* change the order */
      oldB=newB; newB=nEvent.buttons;
      if (!i) event->buttons=nEvent.buttons;

      if (oldB!=newB)
	{
	eventFlag = (i!=0);
	break;
	}

      if (abs(nEvent.dx)+abs(nEvent.dy) > opt_delta)
	nEvent.dx*=opt_accel, nEvent.dy*=opt_accel;

      /* increment the reported dx,dy */
      event->dx+=nEvent.dx;
      event->dy+=nEvent.dy;

      select(fd+1,&fdSet,(fd_set *)NULL,(fd_set *)NULL,&timeout/* zero */);
      }
    while (i++<opt_cluster && nEvent.buttons==oldB && FD_ISSET(fd,&fdSet));
    
    } /* eventFlag */

  /* use fine values now */
  fine_dx+=event->dx;             fine_dy+=event->dy;
  event->dx=fine_dx/opt_scale;    event->dy=fine_dy/opt_scale;
  fine_dx %= opt_scale;           fine_dy %= opt_scale;
  if (!event->dx && !event->dy && (event->buttons==oldB)) 
    return 0;


/*....................................... fill missing fields */

  event->x+=event->dx, event->y+=event->dy;

  i=open_console(O_RDONLY);
  ioctl(i,VT_GETSTATE,&stat);
  event->modifiers=6; /* code for the ioctl */
  if (ioctl(i,TIOCLINUX,&(event->modifiers))<0)
    oops("get_shift_state");
  close(i);
  event->vc = stat.v_active;

  if (oldB==event->buttons)
    event->type = (event->buttons ? GPM_DRAG : GPM_MOVE);
  else
    event->type = (event->buttons > oldB ? GPM_DOWN : GPM_UP);

  switch(event->type)                    /* now provide the cooked bits */
    {
    case GPM_DOWN:
      GET_TIME(tv2);
      if (tv1.tv_sec && (DIF_TIME(tv1,tv2)<opt_time)) /* check first click */
	clicks++, clicks%=3; /* 0, 1 or 2 */
      else clicks=0;
      break;

    case GPM_UP:
      GET_TIME(tv1);
      event->buttons^=oldB; /* for button-up, tell which one */
      event->type|= (GPM_SINGLE<<clicks);
      event->type|= (oldT&GPM_MFLAG);
      break;

    case GPM_DRAG:
      event->type |= GPM_MFLAG;
      break;

    case GPM_MOVE:
      clicks=0;
    default:
      break;
    }
  event->clicks=clicks;

/* UGLY */
/* The current policy is to force the following behaviour:
 * - At buttons up, must fit inside the screen, though flags are set.
 * - At button down, allow going outside by one single step
 */

  margin=0;
  i = ((event->type&(GPM_DRAG|GPM_UP))!=0); /* i is boolean */

  /* Hmm... seems that selection used 1-based coordinates... */

  if (event->x>win.ws_col)  {margin = GPM_RGT; event->x=win.ws_col+1-!i;}
  else if (event->x<=0)     {margin = GPM_LFT; event->x=1-i;}
  if (event->y>win.ws_row)  {margin = GPM_BOT; event->y=win.ws_row+1-!i;}
  else if (event->y<=0)     {margin = GPM_TOP; event->y=1-i;}

  event->margin=margin;

  LOG(("M: %3i %3i (%3i %3i) - butt=%i vc=%i cl=%i",event->dx,event->dy,
                                      event->x,event->y,
                                      event->buttons, event->vc,
                                      event->clicks));
  /* update the global state */
  mouseX=event->x; mouseY=event->y;

  return 1;
}

/*-------------------------------------------------------------------*/
static inline int get_data(Gpm_Connect *where, int whence)
{
static int i;

#ifdef GPM_USE_MAGIC
  while ((i=read(whence,&check,sizeof(int)))==4 && check!=GPM_MAGIC)
    LOG(("No magic"));

  if (!i) return 0;

  if (check!=GPM_MAGIC)
    {
    LOG(("Nothing more"));
    return -1;
    }
#endif

  if ((i=read(whence, where, sizeof(Gpm_Connect)))!=sizeof(Gpm_Connect))
    {
    LOG(("No data"));
    return i ? -1 : 0;
    }

  return 1;
}

/*-------------------------------------------------------------------*/
                                 /* returns -1 if closing connection */
static inline int processRequest(int fd, int vc) 
{
int i;
Gpm_Cinfo *cinfoPtr, *next;

  LOG(("Request on %i (console %i)",fd,vc));

  /* Hmm... data on inactive client: drop him */
  if (vc==-1)                   
    for (i=0; i<=MAX_VC; i++)          /* loop on consoles */
      for (cinfoPtr=cinfo[i];
	   cinfoPtr && cinfoPtr->next;
	   cinfoPtr=cinfoPtr->next)    /* loop on clients */
	if (cinfoPtr->next->fd==fd)
	  {
	  LOG(("Closing"));
	  close(fd);
	  FD_CLR(fd,&connSet);
	  next=cinfoPtr->next;
	  cinfoPtr->next=next->next;
	  free(next);
	  return(-1);
	  }

  if (vc==-1) oops("Unknown fd selected");

  i=get_data(&(cinfo[vc]->data),fd);

  if (!i)
    {
    LOG(("Closing"));
    close(fd);
    FD_CLR(fd,&connSet);
    cinfoPtr=cinfo[vc];
    cinfo[vc]=cinfoPtr->next; /* pop the stack */
    free(cinfoPtr);
    return(-1);
    }

  return (i ? 0 : -1);
}

/*-------------------------------------------------------------------*/
static inline int processConn(int fd) /* returns newfd or -1 */
{
Gpm_Cinfo *info;
Gpm_Connect *request;
Gpm_Cinfo *next;
int vc, newfd, len;
struct sockaddr_un addr; /* reuse this each time */

/*....................................... Accept */

  bzero((char *)&addr,sizeof(addr));
  addr.sun_family=AF_UNIX;

  len=sizeof(addr);
  if ((newfd=accept(fd,(struct sockaddr *)&addr, &len))<0)
    {
    LOG(("accept() failed: %s",strerror(errno)));
    return -1;
    }

  LOG(("Connecting at fd %i",newfd));

  info=malloc(sizeof(Gpm_Cinfo));
  if (!info) oops("malloc()");
  request=&(info->data);

  if (get_data(request,newfd)==-1)
    {
    free(info);
    return -1;
    }

  if ((vc=request->vc)>MAX_VC)
    {
    LOG(("Request on vc %i > %i",vc,MAX_VC));
    free(info);
    return -1;
    }

/* register the connection information in the right place */

  info->next=next=cinfo[vc];
  info->fd=newfd;
  cinfo[vc]=info;

  LOG(("Pid %i, vc %i, events %02X, default %02X, minmod %02X, maxmod %02X",
       request->pid, request->vc, request->eventMask, request->defaultMask,
       request->minMod, request->maxMod));

/* if the client gets motions, give it the current position */
  if(request->eventMask & GPM_MOVE)
    {
    Gpm_Event event={0,0,vc,0,0,mouseX,mouseY,GPM_MOVE,0,0};
    do_client(info, &event);
    }

  return newfd;
}

/*-------------------------------------------------------------------*/
void get_console_size(Gpm_Event *ePtr)
{
int i;
  i=open_console(O_RDONLY);
  ioctl(i, TIOCGWINSZ, &win);
  close(i);
  if (!win.ws_col || !win.ws_row)
    {
    fprintf(stderr, "%s: zero screen dimension, assuming 80x25.\n",prgname);
    win.ws_col=80; win.ws_row=25;
    }
  maxx=win.ws_col; maxy=win.ws_row;
  DEBUGG((stderr,"%i - %i\n",maxx,maxy));
  ePtr->x=maxx/2; ePtr->y=maxy/2;
}

/*-------------------------------------------------------------------*/
int main(int argc, char **argv)
{
int mousefd, ctlfd, newfd;
struct sockaddr_un ctladdr;
int i, len;
struct   timeval timeout;
int maxfd=-1;
int pending;
Gpm_Event event;

  prgname=argv[0];
/*....................................... parse command line */

  mousefd=cmdline(argc, argv);
  maxfd=max(mousefd,maxfd);

  signal(SIGTERM, gpm_killed);
  signal(SIGINT,  gpm_killed);
  signal(SIGUSR1, gpm_killed); /* usr1 is used by a new gpm killing the old */
  signal(SIGWINCH,gpm_killed); /* winch can be sent if console is resized */

/*....................................... create your nodes */

  /* control node */

  if ((ctlfd=socket(AF_UNIX,SOCK_STREAM,0))==-1)
    oops("socket()");
  
  bzero((char *)&ctladdr,sizeof(ctladdr));
  ctladdr.sun_family=AF_UNIX;
  strcpy(ctladdr.sun_path,GPM_NODE_CTL);
  unlink (GPM_NODE_CTL);

  len=sizeof(ctladdr.sun_family)+strlen(GPM_NODE_CTL);
  if (bind(ctlfd,(struct sockaddr *)(&ctladdr),len)==-1)
      oops(ctladdr.sun_path);
  maxfd=max(maxfd,ctlfd);

/*....................................... get screen dimensions */

  get_console_size(&event);
  
/*....................................... wait for mouse and connections */

  listen(ctlfd, 5);          /* Queue up calls */

#define NULL_SET ((fd_set *)NULL)
#define resetTimeout() (timeout.tv_sec=SELECT_TIME,timeout.tv_usec=0)

  FD_ZERO(&connSet);
  FD_SET(ctlfd,&connSet);


  signal(SIGPIPE,SIG_IGN);  /* WARN */
  while(1)
    {
    check_mode(&mousefd);
    maxfd=max(maxfd,mousefd);

    /*
     * readySet is mouse and ctl, connSet is connections only. I must
     * re-init readySet because mousefd can be changed.
     */
    readySet=connSet;
    FD_SET(mousefd,&readySet);
    selSet=readySet;
    resetTimeout();
    if (opt_test) timeout.tv_sec=0;
    
    if (eventFlag)
      pending=1;
    else
      while ((pending=select(maxfd+1,&selSet, NULL_SET,NULL_SET,&timeout))==0)
	{
	if (check_mode(&mousefd))
	  break;
	LOGG(("Hi"));
	selSet=readySet;
	resetTimeout();
	} /* go on */

    if (opt_resize)
      {
      get_console_size(&event);
      opt_resize--;
      signal(SIGWINCH,gpm_killed); /* reinstall handler */

      for(i=0; i<MAX_VC+1; i++)
	if (cinfo[i])
	  kill(cinfo[i]->data.pid,SIGWINCH);
      }

    if (pending<0)
      {
      LOG(("select(): %s",strerror(errno)));
      selSet=readySet;
      resetTimeout();
      continue;
      }

    LOGG(("selected %i times",pending));

/*....................................... got mouse, process event */

    if (FD_ISSET(mousefd,&selSet) || eventFlag)
      {
      FD_CLR(mousefd,&selSet); pending--;
      LOGG(("Got mouse"));
      if (processMouse(mousefd, &event, m_type))
	/*
	 * pass it to the client, if any
	 * or to the default handler, if any
	 * or to the selection handler
	 */
        (cinfo[event.vc] && do_client(cinfo[event.vc], &event))
        || (cinfo[0]     && do_client(cinfo[0], &event))
        || do_selection(&event);
      }

/*....................................... got connection, process it */

    if (pending && FD_ISSET(ctlfd,&selSet))
      {
      FD_CLR(ctlfd,&selSet); pending--;
      LOG(("Got connection"));
      newfd=processConn(ctlfd);
      if (newfd>=0)
	{
	FD_SET(newfd,&connSet);
	maxfd=max(maxfd,newfd);
        }
      }

/*....................................... got request */

    for (i=0; pending && (i<=MAX_VC); i++)
      if (cinfo[i] && FD_ISSET(cinfo[i]->fd,&selSet))
	{
	int fd=cinfo[i]->fd;
        FD_CLR(fd,&selSet); pending--;
	if ((processRequest(fd,i)==-1) && maxfd==fd)
	  maxfd--; /* WARN */
        }

/*....................................... look for a spare fd */

    for (i=0; pending && i<=maxfd; i++)
      if (FD_ISSET(i,&selSet))
	{
	FD_CLR(i,&selSet);
        pending--;
        processRequest(i,-1); /* WARN *//*maxfd*/
        }
	     
/*....................................... all done. */
    
    if(pending) oops("select()");
    } /* while(1) */
}

/*-------------------------------------------------------------------*/
static void gpm_killed(int signo)
{
  if (signo==SIGWINCH)
    {
    DEBUGG((stderr,"%s pid %i is resizing :-)\n",prgname,getpid()));
    opt_resize++;
    return;
    }
  if (signo==SIGUSR1)
    fprintf(stderr,"%s pid %i killed by a new %s\n",prgname,getpid(),prgname);
  else
    {
    DEBUG((stderr,"Removing files...\n"));
    unlink(GPM_NODE_PID);
    unlink(GPM_NODE_CTL);
    }
  exit(0);
}


/*
 * This is because Linus uses 4-wide tabstops,
 * forcing me to use the same default to manage kernel sources
 */

/* Local Variables: */
/* tab-width:8      */
/* End:             */

