/*
 * Copyright (c) 1995 Danny Gasparovski.
 * 
 * Please read the file COPYRIGHT for the 
 * terms and conditions of the copyright.
 */

#include "h/common.h"
#include "h/ip.h"
#include "h/tcp.h"
#include "h/udp.h"
#include "h/socket.h"
#include "h/main.h"
#include "h/terminal.h"

FILE *dfd = NULL;
int slirp_errors = 0;
int dostats = 0;
int slirp_debug = 0;

extern char *strerror _P((int));

void
debug_init(file, dbg)
	char *file;
	int dbg;
{
	dfd = fopen(file,"w");
	if (dfd != NULL) {
		fprintf(dfd,"SLiRP %s - Debugging Started.\n", VERSION);
		fflush(dfd);
		slirp_debug = dbg;
	} else {
		fprintf(stderr,"ERROR: Debuging file \"%s\" could not be opened: %s\n",
			file, strerror(errno));
	}
}

/*
 * Dump a packet in the same format as tcpdump -x
 */
#ifdef DEBUG
void
dump_packet(dat, n)
	void *dat;
	int n;
{
	u_short *ptr = (u_short *)dat;
	int j,k;
	
	n /= 16;
	n++;
	debug_misc(dfd,"PACKET DUMPED: \n");
	for(j = 0; j < n; j++)   {
		for(k = 0; k < 8; k++)
			debug_misc(dfd,"%04x ", *ptr++);
		debug_misc(dfd,"\n");
		fflush(dfd);
	}
}
#endif

/*
 * Statistic routines
 * 
 * These will print statistics to the screen, the debug file (dfd), or
 * a buffer, depending on "type", so that the stats can be sent over
 * the link as well.
 */

/* 
 * This is common to all the stats functions
 */
#define STAT_INIT int (*print) _P((void *, const char *format, ...)); \
	char *ptr; \
	char *ptr2; \
        char **arg; \
\
	switch (type) { \
	 case PRN_SPRINTF: \
		print = (int (*) _P((void *, const char *, ...)))sprintf; \
		ptr = sb->sb_wptr; \
		arg = (char **)&ptr; \
		break; \
	 case PRN_DFD: \
		print = (int (*) _P((void *, const char *, ...)))fprintf; \
		arg = (char **)&dfd; \
		break; \
	 case PRN_STDERR: \
	 default: \
		print = (int (*) _P((void *, const char *, ...)))fprintf; \
		ptr2 = (char *)stderr; \
		arg = (char **)&ptr2; \
		break; \
	}

int
ipstats(sb, type)
	struct sbuf *sb;
	int type;
{
	STAT_INIT;

	ptr += (*print)(*arg," \r\n");	

	ptr += (*print)(*arg,"IP stats:\r\n");
	ptr += (*print)(*arg,"Total packets received:			%d\r\n", ipstat.ips_total);
	ptr += (*print)(*arg,"    Bad version:			%d\r\n", ipstat.ips_badvers);
	ptr += (*print)(*arg,"    Bad checksum:			%d\r\n", ipstat.ips_badsum);
	ptr += (*print)(*arg,"    Too short:				%d\r\n", ipstat.ips_tooshort);
	ptr += (*print)(*arg,"    Too small:				%d\r\n", ipstat.ips_toosmall);
	ptr += (*print)(*arg,"    Bad header length:			%d\r\n", ipstat.ips_badhlen);
	ptr += (*print)(*arg,"    Bad length:				%d\r\n", ipstat.ips_badlen);
	ptr += (*print)(*arg,"    Fragments:				%d\r\n", ipstat.ips_fragments);
	ptr += (*print)(*arg,"    Fragments dropped:			%d\r\n", ipstat.ips_fragdropped);
	ptr += (*print)(*arg,"    Fragments timed out:		%d\r\n", ipstat.ips_fragtimeout);
	ptr += (*print)(*arg,"    Total fragments reassembled:	%d\r\n", ipstat.ips_reassembled);
	ptr += (*print)(*arg,"    Total outgoing packets fragmented:	%d\r\n", ipstat.ips_fragmented);
	ptr += (*print)(*arg,"    Total outgoing fragments:		%d\r\n", ipstat.ips_ofragments);
	ptr += (*print)(*arg,"    Bad protocol feild:			%d\r\n", ipstat.ips_noproto);
	ptr += (*print)(*arg,"Total packets delivered:		%d\r\n", ipstat.ips_delivered);
	
	if (sb)
	   return ptr - sb->sb_data;
	else
	   return 0;
}

int
tcpstats(sb, type)
	struct sbuf *sb;
	int type;
{
	STAT_INIT;

	ptr += (*print)(*arg," \r\n");

	ptr += (*print)(*arg,"TCP stats:\r\n");
	ptr += (*print)(*arg,"    Connections initiated: 		%d\r\n", tcpstat.tcps_connattempt);
	ptr += (*print)(*arg,"    Connections accepted:		%d\r\n", tcpstat.tcps_accepts);
	ptr += (*print)(*arg,"    Connections established: 		%d\r\n", tcpstat.tcps_connects);
	ptr += (*print)(*arg,"    Connections dropped:		%d\r\n", tcpstat.tcps_drops);
	ptr += (*print)(*arg,"    Embryonic connections dropped: 	%d\r\n", tcpstat.tcps_conndrops);
	ptr += (*print)(*arg,"    Conn. closed (includes drops): 	%d\r\n", tcpstat.tcps_closed);
	ptr += (*print)(*arg,"    Segs where we tried to get rtt: 	%d\r\n", tcpstat.tcps_segstimed);
	ptr += (*print)(*arg,"    Times we succeeded:			%d\r\n", tcpstat.tcps_rttupdated);
	ptr += (*print)(*arg,"    Delayed acks sent: 			%d\r\n", tcpstat.tcps_delack);
	ptr += (*print)(*arg,"    Conn. dropped in rxmt timeout: 	%d\r\n", tcpstat.tcps_timeoutdrop);
	ptr += (*print)(*arg,"    Retransmit timeouts:		%d\r\n", tcpstat.tcps_rexmttimeo);
	ptr += (*print)(*arg,"    Persist timeouts: 			%d\r\n", tcpstat.tcps_persisttimeo);
	ptr += (*print)(*arg,"    Keepalive timeouts:			%d\r\n", tcpstat.tcps_keeptimeo);
	ptr += (*print)(*arg,"    Keepalive probes sent: 		%d\r\n", tcpstat.tcps_keepprobe);
	ptr += (*print)(*arg,"    Connections dropped in keepalive: 	%d\r\n", tcpstat.tcps_keepdrops);
	
	ptr += (*print)(*arg,"Total TCP packets sent:			%d\r\n", tcpstat.tcps_sndtotal);
	ptr += (*print)(*arg,"    Data packets sent: 			%d\r\n", tcpstat.tcps_sndpack);
	ptr += (*print)(*arg,"    Data bytes sent: 			%d\r\n", tcpstat.tcps_sndbyte);
	ptr += (*print)(*arg,"    Data packets retransmitted:		%d\r\n", tcpstat.tcps_sndrexmitpack);
	ptr += (*print)(*arg,"    Data bytes retransmitted: 		%d\r\n", tcpstat.tcps_sndrexmitbyte);
	ptr += (*print)(*arg,"    ACK-only packets sent: 		%d\r\n", tcpstat.tcps_sndacks);
	ptr += (*print)(*arg,"    Window probes sent:			%d\r\n", tcpstat.tcps_sndprobe);
	ptr += (*print)(*arg,"    Packets sent with URG only:		%d\r\n", tcpstat.tcps_sndurg);
	ptr += (*print)(*arg,"    Window update-only packets sent: 	%d\r\n", tcpstat.tcps_sndwinup);
	ptr += (*print)(*arg,"    Control (SYN|FIN|RST) packets sent: %d\r\n", tcpstat.tcps_sndctrl);
	ptr += (*print)(*arg,"    Times tcp_output did nothing:	%d\r\n", tcpstat.tcps_didnuttin);
	
	ptr += (*print)(*arg,"Total TCP packets received:		%d\r\n", tcpstat.tcps_rcvtotal);       
	ptr += (*print)(*arg,"    Packets received in sequence:	%d\r\n", tcpstat.tcps_rcvpack);       
	ptr += (*print)(*arg,"    Bytes received in sequence:		%d\r\n", tcpstat.tcps_rcvbyte);
	ptr += (*print)(*arg,"    Packets received with cksum errs: 	%d\r\n", tcpstat.tcps_rcvbadsum);
	ptr += (*print)(*arg,"    Packets received with bad offset: 	%d\r\n", tcpstat.tcps_rcvbadoff);      
/*	ptr += (*print)(*arg,"    Packets received too short:		%d\r\n", tcpstat.tcps_rcvshort); */
        ptr += (*print)(*arg,"    Duplicate-only packets received: 	%d\r\n", tcpstat.tcps_rcvduppack);
	ptr += (*print)(*arg,"    Duplicate-only bytes received: 	%d\r\n", tcpstat.tcps_rcvdupbyte);
	ptr += (*print)(*arg,"    Packets with some duplicate data: 	%d\r\n", tcpstat.tcps_rcvpartduppack);
	ptr += (*print)(*arg,"    Dup. bytes in part-dup. packets: 	%d\r\n", tcpstat.tcps_rcvpartdupbyte);
	ptr += (*print)(*arg,"    Out-of-order packets received: 	%d\r\n", tcpstat.tcps_rcvoopack);
	ptr += (*print)(*arg,"    Out-of-order bytes received:	%d\r\n", tcpstat.tcps_rcvoobyte);
	ptr += (*print)(*arg,"    Packets with data after window: 	%d\r\n", tcpstat.tcps_rcvpackafterwin);
	ptr += (*print)(*arg,"    Bytes rcvd after window: 		%d\r\n", tcpstat.tcps_rcvbyteafterwin);
	ptr += (*print)(*arg,"    Packets rcvd after \"close\":		%d\r\n", tcpstat.tcps_rcvafterclose);
	ptr += (*print)(*arg,"    Rcvd window probe packets: 		%d\r\n", tcpstat.tcps_rcvwinprobe);
	ptr += (*print)(*arg,"    Rcvd duplicate acks:		%d\r\n", tcpstat.tcps_rcvdupack);
	ptr += (*print)(*arg,"    Rcvd acks for unsent data: 		%d\r\n", tcpstat.tcps_rcvacktoomuch);
	ptr += (*print)(*arg,"    Rcvd ack packets: 			%d\r\n", tcpstat.tcps_rcvackpack);
	ptr += (*print)(*arg,"    Bytes acked by rcvd acks: 		%d\r\n", tcpstat.tcps_rcvackbyte);
	ptr += (*print)(*arg,"    Rcvd window update packets:		%d\r\n", tcpstat.tcps_rcvwinupd);
/*	ptr += (*print)(*arg,"    Segments dropped due to PAWS:	%d\r\n", tcpstat.tcps_pawsdrop); */
	ptr += (*print)(*arg,"    Times hdr predict ok for acks:	%d\r\n", tcpstat.tcps_predack);
	ptr += (*print)(*arg,"    Times hdr predict ok for data pkts:	%d\r\n", tcpstat.tcps_preddat);
	ptr += (*print)(*arg,"    TCP ptr cache misses:		%d\r\n", tcpstat.tcps_socachemiss);
	
	if (sb)
	   return ptr - sb->sb_data;
	else
	   return 0;
}

int
udpstats(sb, type)
	struct sbuf *sb;
	int type;
{
	STAT_INIT;
	
        ptr += (*print)(*arg," \r\n");

	ptr += (*print)(*arg,"UDP stats:\r\n");
	ptr += (*print)(*arg,"Total UDP packets received:		%d\r\n", udpstat.udps_ipackets);
	ptr += (*print)(*arg,"    Packets shorter than header:	%d\r\n", udpstat.udps_hdrops);
	ptr += (*print)(*arg,"    Bad checksum:			%d\r\n", udpstat.udps_badsum);
	ptr += (*print)(*arg,"    Data length larger than packet:	%d\r\n", udpstat.udps_badlen);
	ptr += (*print)(*arg,"    Socket cache misses:		%d\r\n", udpstat.udpps_pcbcachemiss);
	ptr += (*print)(*arg,"Total UDP output packets:		%d\r\n", udpstat.udps_opackets);
	
	if (sb)
	   return ptr - sb->sb_data;
	else
	   return 0;
}

int
mbufstats(sb, type)
	struct sbuf *sb;
	int type;
{
	struct mbuf *m;
	int i;
	STAT_INIT;
	
        ptr += (*print)(*arg," \r\n");

	ptr += (*print)(*arg,"Number of mbufs allocated: %d\r\n", mbuf_alloced);
	ptr += (*print)(*arg,"Max number of mbufs allocated: %d\r\n", mbuf_max);
	
	i = 0;
	for (m = m_freelist.m_next; m != &m_freelist; m = m->m_next)
		i++;
	ptr += (*print)(*arg,"Number of mbufs on free list: %d\r\n",  i);
	
	i = 0;
	for (m = m_usedlist.m_next; m != &m_usedlist; m = m->m_next)
		i++;
	ptr += (*print)(*arg,"Number of mbufs on used list: %d\r\n",  i);
        ptr += (*print)(*arg,"Number of packets queued: %d\r\n\r\n", if_queued);
	
	if (sb) 
	   return ptr - sb->sb_data;
	else
	   return 0;
}

int
tcpsockstats(sb, type)
	struct sbuf *sb;
	int type;
{
	struct socket *so;
	STAT_INIT;

        ptr += (*print)(*arg," \r\n");
	
	ptr += (*print)(*arg,"Active TCP Sockets:\r\n");

	for (so = tcb.so_next; so != &tcb; so = so->so_next) {
		
		/* XXX 1024? */
		if (type == 3 && (ptr - sb->sb_wptr) >= (sb->sb_datalen - 1024)) {
			int deltaw = sb->sb_wptr - sb->sb_data;
			int deltar = sb->sb_rptr - sb->sb_data;
			int deltap = ptr -         sb->sb_data;
			
			sb->sb_data = (char *)realloc(sb->sb_data, sb->sb_datalen + 8096);
			
			/* Adjust all values */
			sb->sb_wptr = sb->sb_data + deltaw;
			sb->sb_rptr = sb->sb_data + deltar;
			ptr = sb->sb_data + deltap;
			
			sb->sb_datalen += 8096;
		}
		
		
		ptr += (*print)(*arg,"    Socket no.: %d\r\n", so->s);
		ptr += (*print)(*arg,"    Local address (your end of the link): %s\r\n",
			inet_ntoa(so->so_laddr));
		ptr += (*print)(*arg,"    Local port: %d\r\n", ntohs(so->so_lport));
		ptr += (*print)(*arg,"    Foreign address (host on internet): %s\r\n",
			inet_ntoa(so->so_faddr));
		ptr += (*print)(*arg,"    Foreign port: %d\r\n", ntohs(so->so_fport));
		ptr += (*print)(*arg,"    Type of service: %d\r\n", so->so_iptos);
		ptr += (*print)(*arg,"    Socket state (see source): %x\r\n", so->so_state);
		if (so->so_tcpcb) {
			ptr += (*print)(*arg,"    TCP state: %s\r\n", tcpstates[so->so_tcpcb->t_state]);
			ptr += (*print)(*arg,"    Bytes left in receive buffer: %d\r\n", so->so_rcv.sb_cc);
			ptr += (*print)(*arg,"    Bytes left in send buffer: %d\r\n", so->so_snd.sb_cc);
		}
		ptr += (*print)(*arg,"\r\n");
	}
	
	if (sb)
	   return ptr - sb->sb_data;
	else
	   return 9;
}

int
udpsockstats(sb, type)
	struct sbuf *sb;
	int type;
{
	struct socket *so;
	STAT_INIT;
	
        ptr += (*print)(*arg," \r\n");

	ptr += (*print)(*arg,"Active UDP Sockets:\r\n");
	
	for (so = udb.so_next; so != &udb; so = so->so_next) {
		
		/* XXX 1024? */
		if (type == 3 && (ptr - sb->sb_wptr) >= (sb->sb_datalen - 1024))  {
			int deltaw = sb->sb_wptr - sb->sb_data;
			int deltar = sb->sb_rptr - sb->sb_data;
			int deltap = ptr -         sb->sb_data;
			
			sb->sb_data = (char *)realloc(sb->sb_data, sb->sb_datalen + 8096);
			
			/* Adjust all values */
			sb->sb_wptr = sb->sb_data + deltaw;
			sb->sb_rptr = sb->sb_data + deltar;
			ptr = sb->sb_data + deltap;
			
			sb->sb_datalen += 8096;
		}
		
		
		ptr += (*print)(*arg,"    Socket fd: %d\r\n", so->s);

		ptr += (*print)(*arg,"    Local address (your end of the link): %s\r\n",
				  inet_ntoa(so->so_laddr));
		ptr += (*print)(*arg,"    Local port: %d\r\n", ntohs(so->so_lport));
		ptr += (*print)(*arg,"    Foreign address (host on internet): %s\r\n",
				  inet_ntoa(so->so_faddr));
		ptr += (*print)(*arg,"    Type of service: %d\r\n", so->so_iptos);
		ptr += (*print)(*arg,"    State (see source): %d\r\n", so->so_state);
		if (so->so_expire)
		   ptr += (*print)(*arg,"    Expires in: %u seconds\r\n", 
				     (so->so_expire - curtime) / 1000);
		ptr += (*print)(*arg,"    Number of packets queued: %d\r\n\r\n", so->so_queued);
	}
	
	if (sb)
	   return ptr - sb->sb_data;
	else
	   return 0;
}

void
slirp_exit(exit_status)
	int exit_status;
{
	int type = PRN_DFD;
	
	term_restore(ttyfd);

	if (dostats)
	   type = PRN_STDERR;

#define SBNULL (struct sbuf *)NULL
	
	if (type == PRN_STDERR || dfd != NULL) {
		ipstats(SBNULL, type);
		tcpstats(SBNULL, type);
		udpstats(SBNULL, type);
		mbufstats(SBNULL, type);
		
		tcpsockstats(SBNULL, type);
		udpsockstats(SBNULL, type);
	}

#undef SBNULL

	/*
	 * Report "regular" errors that occured during the running of slirp,
	 * which might have perplexed the user
	 */
	if (slirp_errors & ERR_REDUCEMTU) {
		fprintf(stderr,"Error:\r\n");
		fprintf(stderr,"    SLiRP received (and discarded) packets that were too large.\n");
		fprintf(stderr,"    Please check your SLIP software and make sure your\n");
		fprintf(stderr,"    MTU matches SLiRP's MTU (currently %d).\n", ifp.ifp_mtu);
		fprintf(stderr,"    You can change SLiRP's MTU via the -m command-line option.\n\n");
		fprintf(stderr,"    If your MTU setting is OK then ignore this message,\n");
		fprintf(stderr,"    it could have been caused by line noise or something.\n\n");
	}
	if (slirp_errors & ERR_RCVICMPECHOREQ) {
		fprintf(stderr,"Warning:\n");
		fprintf(stderr,"    SLiRP received some ICMP ECHO-REQUEST packets.\n");
		fprintf(stderr,"    SLiRP does not yet emulate ICMP ECHO-REQUEST packets, so programs\n");
		fprintf(stderr,"    like \"ping\" will not work.  If you used \"ping\" to test wheather\n");
		fprintf(stderr,"    the link was up, then it will fail, even though the link may have\n");
		fprintf(stderr,"    been up.  Use \"telnet\" to test the link instead.\n");
	}
	exit(exit_status);
}


