------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                       S Y S T E M . W C H _ C O N                        --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.5 $                              --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- The GNAT library is free software; you can redistribute it and/or modify --
-- it under terms of the GNU Library General Public License as published by --
-- the Free Software  Foundation; either version 2, or (at your option) any --
-- later version.  The GNAT library is distributed in the hope that it will --
-- be useful, but WITHOUT ANY WARRANTY;  without even  the implied warranty --
-- of MERCHANTABILITY  or  FITNESS FOR  A PARTICULAR PURPOSE.  See the  GNU --
-- Library  General  Public  License for  more  details.  You  should  have --
-- received  a copy of the GNU  Library  General Public License  along with --
-- the GNAT library;  see the file  COPYING.LIB.  If not, write to the Free --
-- Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.        --
--                                                                          --
------------------------------------------------------------------------------

--  This package defines the codes used to identify the encoding method for
--  wide characters in string and character constants. This is needed both
--  at compile time and at runtime (for the wide character runtime routines)

package System.WCh_Con is
pragma Pure (WCh_Con);

   -------------------------------------
   -- Wide_Character Encoding Methods --
   -------------------------------------

   --  A wide character encoding method is a method for uniquely representing
   --  a Wide_Character value using a one or more Character values. Two types
   --  of encoding method are supported by GNAT:

   --    An escape encoding method uses ESC as the first character of the
   --    sequence, and subsequent characters determine the wide character
   --    value that is represented. Any character other than ESC stands
   --    for itself as a single byte (i.e. any character in Latin-1, other
   --    than ESC itself, is represented as a single character: itself).

   --    An upper half encoding method uses a character in the upper half
   --    range (i.e. in the range 16#80# .. 16#FF#) as the first byte of
   --    a wide character encoding sequence. Subsequent characters are
   --    used to determine the wide character value that is represented.
   --    Any character in the lower half (16#00# .. 16#7F#) represents
   --    itself as a single character.

   --  Note that GNAT does not currently support escape-in, escape-out
   --  encoding methods, where an escape sequence is used to set a mode
   --  used to recognize subsequent characters. All encoding methods use
   --  individual character-by-character encodings, so that a sequence of
   --  wide characters is represented by a sequence of encodings.

   --  To add new encoding methods, the following steps are required:

   --     1.  Define a code for a new value of type WC_Encoding_Method
   --     2.  Adjust the definition of WC_Encoding_Method accordingly
   --     3.  Provide appropriate conversion routines in System.Wch_Cnv
   --     4.  Adjust definition of WC_Longest_Sequence if necessary
   --     5.  Add an entry in WC_Encoding_Letters for the new method

   --  Note that the WC_Encoding_Method values must be kept ordered so that
   --  the definitions of the subtypes WC_Upper_Half_Encoding_Method and
   --  WC_ESC_Encoding_Method are still correct.

   --------------------------------
   -- Compatibility Requirements --
   --------------------------------

   --  When a unit is compiled with a particular option, any units that it
   --  with's are compiled with the same setting. Basically this means that
   --  all units in a program which contain wide character encodings must
   --  use a consistent encoding method. At bind time, the binder checks
   --  that all units in the program are either compiled in WCEM_None mode,
   --  meaning that they have no wide character encodings, or that they are
   --  compiled in a consistent mode. It is not permitted to mix two modes
   --  (other than WCEM_None) in a single program.

   ---------------------------------
   -- Encoding Method Definitions --
   ---------------------------------

   type WC_Encoding_Method is range 0 .. 4;
   --  Type covering the range of values used to represent wide character
   --  encoding methods. An enumeration type might be a little neater, but
   --  more trouble than it's worth, given the need to pass these values
   --  from the compiler to the backend, and to record them in the ALI file.

   WCEM_None : constant WC_Encoding_Method := 0;
   --       No wide character encoding is permitted in the source program.
   --       This is the default. A unit compiled with this option can be
   --       used with units compiled with any other option.

   WCEM_Hex : constant WC_Encoding_Method := 1;
   --       The wide character with code 16#abcd# is represented by the
   --       escape sequence ESC a b c d (five characters, where abcd are
   --       ASCII hex characters, using upper case for letters). This
   --       method is easy to deal with in external environments that do
   --       not support wide characters, and covers the whole BMP. This
   --       is the default encoding method.

   WCEM_Upper : constant WC_Encoding_Method := 2;
   --       The wide character with encoding 16#abcd#, where the upper bit
   --       is on (i.e. a is in the range 8-F) is represented as two bytes
   --       16#ab# and 16#cd#. The second byte may never be a format control
   --       character, but is not required to be in the upper half. This
   --       method can be also used for shift-JIS or EUC where the internal
   --       coding matches the external coding.

   WCEM_Shift_JIS : constant WC_Encoding_Method := 3;
   --       A wide character is represented by a two character sequence
   --       16#ab# and 16#cd#, with the restrictions described for upper
   --       half encoding as described above. The internal character code
   --       is the corresponding JIS character according to the standard
   --       algorithm for Shift-JIS conversion. See the body of package
   --       System.JIS_Conversions for further details.

   WCEM_EUC : constant WC_Encoding_Method := 4;
   --       A wide character is represented by a two character sequence
   --       16#ab# and 16#cd#, with both characters being in the upper half.
   --       The internal character code is the corresponding JIS character
   --       according to the EUC encoding algorithm. See the body of package
   --       System.JIS_Conversions for further details.

   WC_Encoding_Letters : constant array (WC_Encoding_Method) of Character :=
     (WCEM_None      => 'n',
      WCEM_Hex       => 'h',
      WCEM_Upper     => 'u',
      WCEM_Shift_JIS => 's',
      WCEM_EUC       => 'e');
   --  Letters used for selection of wide character encoding method in the
   --  compiler options (-gnatj? switch) and for Wide_Text_IO (wcem parameter
   --  in the form string).

   subtype WC_ESC_Encoding_Method is
     WC_Encoding_Method range WCEM_Hex .. WCEM_Hex;
   --  Encoding methods using an ESC character at the start of the sequence.

   subtype WC_Upper_Half_Encoding_Method is
     WC_Encoding_Method range WCEM_Upper .. WCEM_EUC;
   --  Encoding methods using an upper half character (16#80#..16#FF) at
   --  the start of the sequence.

   WC_Longest_Sequence : constant := 5;
   --  The longest number of characters that can be used for a wide
   --  character sequence for any of the active encoding methods.

end System.WCh_Con;
