/* Hypertext file browser.
   Copyright (C) 1994 Miguel de Icaza.
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   Implements the hypertext file viewer.
   The hypertext file is a file that may have one or more nodes.  Each
   node ends with a ^D character and starts with a bracket, then the
   name of the node and then a closing bracket.

   Links in the hypertext file are specified like this: the text that
   will be highlighted should have a leading ^A, then it comes the
   text, then a ^B indicating that highlighing is done, then the name
   of the node you want to link to and then a ^C.

   The file must contain a ^D at the beggining and at the end of the
   file or the program will not be able to detect the end of file.

*/
#include <ncurses.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <malloc.h>
#include <errno.h>
#include "color.h"
#include "dialog.h"
#include "util.h"
#include "input.h"
#include "win.h"
#include "global.h"
#include "help.h"
#include "mouse.h"

#define MAXLINKNAME 80
#define HISTORY_SIZE 20

static char rcsid [] = "$Id: help.c,v 1.11 1994/10/13 04:27:35 miguel Exp $";
static char *data;		/* Pointer to the loaded data file */
static int help_lines = 18;	/* Lines in help viewer */
WINDOW *whelp;			/* Help Window */
static int  history_ptr;	/* For the history queue */
static char *main;		/* The main node */
static char *last_shown = 0;	/* Last byte shown in a screen */
static char *current, *start, *selected_item;

static struct {
    char *page;			/* Pointer to the selected page */
    char *link;			/* Pointer to the selected link */
} history [HISTORY_SIZE];

typedef struct Link_Area {
    int x1, y1, x2, y2;
    char *link_name;
    struct Link_Area *next;
} Link_Area;
static Link_Area *link_area = NULL;
static int inside_link_area = 0;

/* returns the position where text was found in the start buffer */
/* or 0 if not found */
char *search_string (char *start, char *text)
{
    char *d = text;
    char *e = start;

    /* fmt sometimes replaces a space with a newline in the help file */
    /* Replace the newlines in the link name with spaces to correct the situation */
    while (*d){
	if (*d == '\n')
	    *d = ' ';
	d++;
    }
    /* Do search */
    for (d = text; *e; e++){
	if (*d == *e)
	    d++;
	else
	    d = text;
	if (!*d)
	    return e+1;
    }
    return 0;
}

/* Searches text in the buffer pointed by start.  Search ends */
/* if the CHAR_NODE_END is found in the text.  Returns 0 on failure */
static char *search_string_node (char *start, char *text)
{
    char *d = text;
    char *e = start;

    if (!start)
	return 0;
    
    for (; *e && *e != CHAR_NODE_END; e++){
	if (*d == *e)
	    d++;
	else
	    d = text;
	if (!*d)
	    return e+1;
    }
    return 0;
}

/* Searches the_char in the buffer pointer by start and searches */
/* it can search forward (direction = 1) or backward (direction = -1) */
static char *search_char_node (char *start, char the_char, int direction)
{
    char *e;

    e = start;
    
    for (; *e && (*e != CHAR_NODE_END); e += direction){
	if (*e == the_char)
	    return e;
    }
    return 0;
}

/* Returns the new current pointer when moved lines lines */
static char *move_forward2 (char *current, int lines)
{
    char *p;
    int  line;

    for (line = 0, p = current; *p && *p != CHAR_NODE_END; p++){
	if (*p == '\n')
	    line++;
	if (line == lines)
	    return p;
    }
    return current;
}

static char *move_backward2 (char *current, int lines)
{
    char *p;
    int line;

    for (line = 0, p = current; *p && p >= data; p--){
	if (*p == CHAR_NODE_END)
	{
	    /* We reached the beginning of the node */
	    /* Skip the node headers */
	    while (*p != ']') p++;
	    return p + 1;
	}
	if (*p == '\n')
	    line++;
	if (line == lines)
	    return p;
    }
    return current;
}

static void move_forward (int i)
{
    if (i == 1) i++;
    current = move_forward2 (current, i);
}

static void move_backward (int i)
{
    if (i == 1) i++;
    current = move_backward2 (current, i);
}

static void null_fn (int dummy)
{
    /* Nothing */
}

static char *follow_link (char *start, char *selected_item)
{
    char link_name [MAXLINKNAME];
    char *p;
    int  i = 0;

    if (!selected_item)
	return start;
    
    for (p = selected_item; *p && *p != CHAR_NODE_END && *p != CHAR_LINK_POINTER; p++)
	;
    if (*p == CHAR_LINK_POINTER){
	link_name [0] = '[';
	for (i = 1; *p != CHAR_LINK_END && *p && *p != CHAR_NODE_END && i < MAXLINKNAME-3; )
	    link_name [i++] = *++p;
	link_name [i-1] = ']';
	link_name [i] = 0;
	p = search_string (data, link_name);
	if (p)
	    return p;
    }
    return "Help file format error\n\x4";	/*  */
}

static char *select_next_link (char *start, char *current_link)
{
    char *p;

    if (!current_link)
	return 0;
    
    p = search_string_node (current_link, STRING_LINK_END);
    if (!p)
	p = start;
    p = search_string_node (p, STRING_LINK_START);
    if (p)
	return p - 1;
    p = search_string_node (start, STRING_LINK_START);
    if (p)
	return p - 1;
    return 0;
}

static char *select_prev_link (char *start, char *current_link)
{
    char *p;

    if (!current_link)
	return 0;
    
    p = current_link - 1;
    if (p <= start)
	return 0;
    
    p = search_char_node (p, CHAR_LINK_START, -1);
    return p;
}

static void start_link_area (int x, int y, char *link_name)
{
    Link_Area *new;

    if (inside_link_area)
	message (0, " Warning ", " Internal bug: Double start of link area ");

    /* Allocate memory for a new link area */
    new = (Link_Area*) xmalloc (sizeof (Link_Area), "Help, link_area");
    new->next = link_area;
    link_area = new;

    /* Save the beginning coordinates of the link area */
    link_area->x1 = x;
    link_area->y1 = y;

    /* Save the name of the destination anchor */
    link_area->link_name = link_name;

    inside_link_area = 1;
}

static void end_link_area (int x, int y)
{
    if (inside_link_area){
	/* Save the end coordinates of the link area */
	link_area->x2 = x;
	link_area->y2 = y;

	inside_link_area = 0;
    }
}

static void clear_link_areas ()
{
    Link_Area *current;

    while (link_area){
	current = link_area;
	link_area = current -> next;
	free (current);
    }
    inside_link_area = 0;
}

static void show (char *paint_start)
{
    char *p;
    int  col, line;
    int  painting = 1;
    
    line = col = 0;
    werase (whelp);
    wclr (whelp);
    clear_link_areas ();
    if (selected_item < paint_start)
	selected_item = NULL;
    for (p = paint_start; *p != CHAR_NODE_END && line < help_lines; p++){
	if (*p == CHAR_LINK_START){
	    if (selected_item == NULL)
		selected_item = p;
	    if (p == selected_item)
		wattrset (whelp, MARKED_COLOR | A_BOLD);
	    else
		wattrset (whelp, INPUT_COLOR);
	    start_link_area (col, line, p);
	} else if (*p == CHAR_LINK_POINTER){
	    painting = 0;
	    end_link_area (col - 1, line);
	} else if (*p == CHAR_LINK_END){
	    painting = 1;
	    wattrset (whelp, REVERSE_COLOR);
	} else if (*p == '\n'){
	    line++;
	    col = 0;
	} else {
	    if (!painting)
		continue;
	    mvwaddch (whelp, line, col, *p);
	    col++;
	}
    }
    last_shown = p;
    wattrset (whelp, REVERSE_COLOR);
    if (selected_item >= last_shown){
	if (link_area != NULL){
	    selected_item = link_area->link_name;
	    show (paint_start);
	}
	else
	    selected_item = NULL;
    }
    wrefresh (whelp);
}

static int help_event (Gpm_Event *event)
{
    int hp;
    Link_Area *current_area;

    if (! (event->type & GPM_UP))
	return 0;

    /* Test whether the mouse click is inside one of the link areas */
    current_area = link_area;
    while (current_area)
    {
	/* Test one line link area */
	if (event->y == current_area->y1 && event->x >= current_area->x1 &&
	    event->y == current_area->y2 && event->x <= current_area->x2)
	    break;
	/* Test two line link area */
	if (current_area->y1 + 1 == current_area->y2){
	    /* The first line */
	    if (event->y == current_area->y1 && event->x >= current_area->x1)
		break;
	    /* The second line */
	    if (event->y == current_area->y2 && event->x <= current_area->x2)
		break;
	}
	/* Mouse will not work with link areas of more than two lines */

	current_area = current_area -> next;
    }

    /* Test whether a link area was found */
    if (current_area){
	/* The click was inside a link area -> follow the link */
	hp = history_ptr++ % HISTORY_SIZE;
	history [hp].page = current;
	history [hp].link = current_area->link_name;
	current = start = follow_link (current, current_area->link_name);
	selected_item = NULL;
    }
    else{
	if (event->y < help_lines/2)
	    move_backward (1);
	else
	    move_forward (1);
    }

    /* Show the new node */
    show (current);

    return 0;
}

void interactive_display (char *filename, char *node)
{
    int c;
    int quit = 0;
    char *new_item;
    
    if ((data = load_file (filename)) == 0){
	message (1, " Error ", " Can't open file %s \n %s ",
		 filename, unix_error_string (errno));
	return;
    }
    if (!(main = search_string (data, node))){
	message (1, " Error ", " Can't find node %s in help file ", node);
	return;
    }
    create_dialog (60, help_lines, " Help ", "", 0);
    whelp = get_top_text ();
    
    selected_item = search_string_node (main, STRING_LINK_START) - 1;
    current = start = main;
    
    for (history_ptr = 0; history_ptr < HISTORY_SIZE; history_ptr++){
	history [history_ptr].page = current;
	history [history_ptr].link = selected_item;
    }

    push_event (3, 3, 60 + 2, help_lines + 2, (mouse_h) help_event, 0);

    do {
	show (current);
	c = mi_getch ();
	if (check_movement_keys (c, 1, help_lines, move_backward, move_forward,
				 null_fn, null_fn))
	    /* Nothing */;
	else switch (c){
				/* goto previous node */
	case 'l':
	case KEY_LEFT:
	    {
	    int hp;

	    hp = --history_ptr % HISTORY_SIZE;
	    current = start = history [hp].page;
	    selected_item = history [hp].link;
	    }
	    break;
	    
				/* follow link */
	case '\n':
	case KEY_RIGHT:
	    if (!selected_item){
		/* If there are no links, go backward in history */
		int hp = --history_ptr % HISTORY_SIZE;
		
		current = start = history [hp].page;
		selected_item   = history [hp].link;
	    } else {
		int hp;

		hp = history_ptr++ % HISTORY_SIZE;
		history [hp].page = current;
		history [hp].link = selected_item;
		current = start = follow_link (current, selected_item);
	    }
	    selected_item = NULL;
	    break;

				/* show help */
	case KEY_F(1):
	    {
		int hp;

		hp = history_ptr++ % HISTORY_SIZE;
		history [hp].page = current;
		history [hp].link = selected_item;
		current = start = search_string (data, "[Help]");
	    }
	    selected_item = NULL;
	    break;

				/* select next link */
	case '\t':
	    new_item = select_next_link (start, selected_item);
	    if (new_item){
		selected_item = new_item;
		if (selected_item >= last_shown){
		    selected_item = NULL;
		}
	    }
	    break;

				/* select previous link */
	case 'b':
	    new_item = select_prev_link (start, selected_item);
	    selected_item = new_item;
	    if (selected_item < current || selected_item >= last_shown){
		if (link_area != NULL)
		    selected_item = link_area->link_name;
		else
		    selected_item = NULL;
	    }
	    break;
				/* quit */
	case KEY_F(10):
	case ESC_CHAR:
	    quit = 1;
	}
    } while (!quit);
    pop_event ();
    free (data);
    destroy_dialog ();
    do_refresh ();
}
