#include <sys/types.h>		/* Needed to include local .h files */
#include <sys/stat.h>
#include <sys/param.h>
#include <ncurses.h>

#include "dir.h"
#include "file.h"
#include "global.h"
#include "panel.h"
#include "util.h"
#include "main.h"

extern Panel *current_panel, *other_panel;
extern Panel left_panel, right_panel;

static struct {
    char *key;
    sortfn *sort_type;
} sort_names [] = {
    "name",      (sortfn *) sort_name,
    "extension", (sortfn *) sort_ext,
    "time",      (sortfn *) sort_time,
    "size",      (sortfn *) sort_size,
    "unsorted",  (sortfn *) unsorted,
    0, 0
};

static struct {
    char *key;
    int  view_type;
} view_types [] = {
    "full",  view_full,
    "brief", view_brief,
    "long",  view_long,
    0, 0
}; 	

void panel_save (char *profile, Panel *panel, char *section)
{
    char buffer [6];
    int  i;
    
    sprintf (buffer, "%d", panel->reverse);
    WritePrivateProfileString (section, "reverse", buffer, profile);
    for (i = 0; sort_names [i].key; i++)
	if (sort_names [i].sort_type == (sortfn *) panel->sort_type){
	    WritePrivateProfileString (section, "sort_order",
				       sort_names [i].key, profile);
	    break;
	}
    for (i = 0; view_types [i].key; i++)
	if (view_types [i].view_type == panel->view_type){
	    WritePrivateProfileString (section, "display",
				       view_types [i].key, profile);
	    break;
	}
}

void save_setup ()
{
    char profile [100];

    sprintf (profile, "%s/.mc.ini", home_dir);

    set_int (profile, "show_backups", show_backups);
    set_int (profile, "show_dot_files", show_dot_files);
    set_int (profile, "verbose", verbose);
    set_int (profile, "mark_moves_down", mark_moves_down);
    set_int (profile, "pause_after_run", pause_after_run);
    set_int (profile, "show_mini_info", show_mini_info);
    set_int (profile, "shell_patterns", easy_patterns);
    set_int (profile, "auto_save_setup", auto_save_setup);
    set_int (profile, "align_extensions", align_extensions);
    set_int (profile, "auto_menu", auto_menu);
    set_int (profile, "use_internal_view", use_internal_view);
    set_int (profile, "clear_before_exec", clear_before_exec);
    set_int (profile, "mix_all_files", mix_all_files);
    set_int (profile, "fast_reload", fast_reload);
    set_int (profile, "fast_reload_msg_shown", fast_reload_w);
    set_int (profile, "confirm_delete", confirm_delete);
    set_int (profile, "mouse_repeat_rate", mou_auto_repeat);
    WritePrivateProfileString ("Dirs", "other_dir", opanel->cwd, profile);
    panel_save (profile, &left_panel,  "Left Panel");
    panel_save (profile, &right_panel, "Right Panel");
    sync_profiles ();
}

static void panel_load (char *profile, Panel *panel, char *section)
{
    int i;
    char buffer [40];
    
    panel->reverse = GetPrivateProfileInt (section, "reverse", 0, profile);
    GetPrivateProfileString (section, "sort_order", "name", buffer,
			     sizeof (buffer), profile);
    
    for (i = 0; sort_names [i].key; i++)
	if (strcasecmp (sort_names [i].key, buffer) == 0){
	    panel->sort_type = sort_names [i].sort_type;
	    break;
	}
    GetPrivateProfileString (section, "display", "full", buffer,
			     sizeof (buffer), profile);
    for (i = 0; view_types [i].key; i++)
	if (strcasecmp (view_types [i].key, buffer) == 0){
	    panel->view_type = view_types [i].view_type;
	    break;
	}
}
			  
void load_setup ()
{
    static char buffer [100];
    char   *profile;

    /* Other modules expect load_setup to set this variables */
    left_panel.sort_type = right_panel.sort_type = (sortfn *) sort_name;
    left_panel.view_type = right_panel.view_type = view_full;
    
    sprintf (buffer, "%s/.mc.ini", home_dir);
    
    if (exist_file (buffer)){
	profile = buffer;
    } else if (exist_file (PROFILE)){
	profile = PROFILE;
    } else
	return;
    
    show_backups    = get_int (profile, "show_backups", show_backups);
    show_dot_files  = get_int (profile, "show_dot_files", show_dot_files);
    verbose         = get_int (profile, "verbose", verbose);
    mark_moves_down = get_int (profile, "mark_moves_down", mark_moves_down);
    pause_after_run = get_int (profile, "pause_after_run", pause_after_run);
    show_mini_info  = get_int (profile, "show_mini_info", show_mini_info);
    easy_patterns   = get_int (profile, "shell_patterns", easy_patterns);
    auto_save_setup = get_int (profile, "auto_save_setup", auto_save_setup);
    align_extensions= get_int (profile, "align_extensions", align_extensions);
    auto_menu       = get_int (profile, "auto_menu", auto_menu);
    fast_reload     = get_int (profile, "fast_reload", fast_reload);
    fast_reload_w   = get_int (profile, "fast_reload_msg_shown", 0);
    mix_all_files   = get_int (profile, "mix_all_files", mix_all_files);
    use_internal_view=get_int (profile, "use_internal_view",use_internal_view);
    clear_before_exec=get_int (profile, "clear_before_exec",clear_before_exec);
    confirm_delete   =get_int (profile, "confirm_delete", confirm_delete);
    mou_auto_repeat  =get_int (profile, "mou_auto_repeat", mou_auto_repeat);
    panel_load (profile, &left_panel, "Left Panel");
    panel_load (profile, &right_panel,"Right Panel");

    if (left_panel.view_type == view_long)
	right_panel.view_type |= VIEW_DISABLED;
    
    if (!other_dir){
	GetPrivateProfileString ("Dirs", "other_dir", ".", buffer,
			     sizeof (buffer), profile);
	other_dir = buffer;
    }
}

